//******************************************************************************
//  IBM 7094 Emulator - Tape Drive display form
//  By Rob Storey 2001-2004 intabits@optushome.com.au
//------------------------------------------------------------------------------
//  This unit provides the Tape Viewer window functions.
//  It is activated by clicking on the file name label in a tape drive window.
//  (TFormTape defined in B709TapF).
//------------------------------------------------------------------------------
Unit B709TapV;

Interface

Uses Windows, Messages, SysUtils, Classes, Graphics,
     Controls, Forms, Dialogs, StdCtrls, ExtCtrls,
     B709Strm, // Customized TMemoryStream class
     B709Cmps, // Display Components
     B709TapF, // Tape Drive display subform
     B709Chan, // I/O Channel and Device functions
     B709Misc, // Miscellaneous utility functions
     B709Defs; // General definitions

Type TRecType=(RTNORM,RTRGAP,RTMARK);

Type
  TTapeViewForm = class(TForm)
    PACtrl: TPanel;
    Label1: TLabel;
    Label2: TLabel;
    LACount: TLabel;
    PBClose: TButton;
    EBWordCount: TEdit;
    PBNxtMrk: TButton;
    LBDump: TListBox;
    PBSave: TButton;
    PBDisp: TButton;
    CBLinNums: TCheckBox;
    RBBCD: TRadioButton;
    RBBIN: TRadioButton;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure PBDispClick(Sender: TObject);
    procedure PBNxtMrkClick(Sender: TObject);
    procedure PBCloseClick(Sender: TObject);
    procedure PBSaveClick(Sender: TObject);
    procedure BIModeClick(Sender: TObject);
    procedure EBWordCountChange(Sender: TObject);
    procedure CBLinNumsClick(Sender: TObject);
  private
    TAPDev:     TDevTAP;
    DumpList:   TStringList;
    MarkPtr:    Integer;
    BCDMode:    Boolean;
    RecdOffs:   Integer;
    LineCtr:    Integer;
    procedure Log(SS: String);
    procedure DispBCD(NW: Integer);
    procedure DispBIN(NW: Integer);
    procedure LogBCDData(DS: String);
    procedure LogBINData(DP,WC,BI: Integer; HS,OS,CS: String; RT: TRecType);
  public
    Procedure Initialize;
    Procedure SelectDrive(TD: TDevTAP);
    Procedure SetDispMode(BM: Boolean);
    Procedure SetLineNumb(LN: Integer);
    Procedure SaveToFile(SS: String);
  End;

Var TapeViewForm: TTapeViewForm;

Implementation

{$R *.DFM}

Type Str2=String[2];

Function OCT(Const BB: Byte): Str2;
Begin
  Result:=Chr(((BB Shr 3) And 7)+$30)+Chr((BB And 7)+$30);
End;

Function BCD(Const BB: Byte): Char;
Begin
  Result:=TabBCDtoASC[(BB And $3F)+1];
End;

procedure TTapeViewForm.FormCreate(Sender: TObject);
begin
  FormHandles[FITapeView]:=Handle;
  Caption:=Heading+' - Tape Viewer';
  DumpList:=TStringList.Create;
  SetDispMode(False);
end;

procedure TTapeViewForm.FormDestroy(Sender: TObject);
begin
  DumpList.Free;
end;

procedure TTapeViewForm.Initialize;
begin
end;

procedure TTapeViewForm.PBCloseClick(Sender: TObject);
begin
  Close;
end;

procedure TTapeViewForm.SelectDrive(TD: TDevTAP);
begin
  TAPDev:=TD;
  Caption:=Heading+' - Tape Viewer: '+TAPDev.FileNam;
  Visible:=TD<>NIL;
  Application.ProcessMessages;
end;

procedure TTapeViewForm.SetDispMode(BM: Boolean);
begin
  BCDMode:=BM;
  RBBCD.Checked:=BCDMode;
  If BCDMode then begin
    PACtrl.Height:=28;
    LACount.Caption:='Lines';
  End else Begin
    PACtrl.Height:=55;
    LACount.Caption:='Words';
  End;
  PBNxtMrk.Visible:=Not BCDMode;
  CBLinNums.Visible:=BCDMode;
  PBDispClick(Self);
end;

procedure TTapeViewForm.SetLineNumb(LN: Integer);
begin
  LBDump.TopIndex:=LN-1;
end;

procedure TTapeViewForm.SaveToFile(SS: String);
begin
  LBDump.Items.SaveToFile(SS);
end;

procedure TTapeViewForm.BIModeClick(Sender: TObject);
begin
  SetDispMode(RBBCD.Checked);
end;

procedure TTapeViewForm.EBWordCountChange(Sender: TObject);
begin
  PBDisp.Enabled:=True;
end;

procedure TTapeViewForm.CBLinNumsClick(Sender: TObject);
begin
  PBDispClick(Self);
end;

Procedure TTapeViewForm.Log(SS: String);
Begin
  DumpList.Add(SS);
  With LBDump.Items do
    If Count<32000 then
      Add(SS);
End;

Procedure TTapeViewForm.PBDispClick(Sender: TObject);
Var SP,NW: Integer;
begin
  If TAPDev=NIL then Exit;
  Screen.Cursor:=CRHourGlass;
  SP:=TAPDev.DataPosn;
  PBDisp.Enabled:=False;
  NW:=StrToInt(EBWordCount.Text);
  DumpList.Clear;
  LBDump.Clear;
  LBDump.Hide;
  If TAPDev.DataSize>0 then
    If BCDMode then DispBCD(NW)
               else DispBIN(NW);
  LBDump.Show;
  MarkPtr:=0;
  TAPDev.SetDataPosition(SP);
  Screen.Cursor:=CRDefault;
end;

// Display as lines of BCD characters
Procedure TTapeViewForm.DispBCD(NW: Integer);
Var SP: Longint;
    WC,LL: Integer;
    BB: Byte;
    DS: String;
    RM,FM: Boolean;
begin
  SP:=0; WC:=0; BB:=0; LineCtr:=0;
  Repeat
    RM:=False; FM:=False; DS:='';
    // Read a record, forming BCD string
    Repeat
      If SP>TAPDev.DataSize then Break;
      Try
        BB:=TAPDev.PeekByte(SP); Inc(SP);
      Except Break; End;
      If (BB=$0F) Or (BB=TAPEMARK) then Break;
      If BB=$3A then
        RM:=True;
      If (BB And $80)>0 then RM:=True;
      If RM then Break;
      DS:=DS+BCD(BB);
    Until RM Or FM Or (SP>=TAPDev.DataSize);
    // Dump large records as 96 char lines
    LL:=Length(DS);
    If LL>132 then begin
      While LL>96 do begin
        LogBCDData(Copy(DS,1,96));
        Delete(DS,1,96);
        LL:=Length(DS);
      End;
    End;
    // Dump remainder (or short records)
    If LL>0 then
      LogBCDData(DS);
  Until (WC>=NW) Or (SP>=TAPDev.DataSize);
End;

Procedure TTapeViewForm.LogBCDData(DS: String);
Begin
  Inc(LineCtr);
  If CBLinNums.Checked then DS:=Format('%6.6d ',[LineCtr])+DS;
  Log(DS);
End;

// Display as binary word dump
Procedure TTapeViewForm.DispBIN(NW: Integer);
Var DP,wp: Longint;
    WC,BI: Integer;
    BB: Byte;
    HS,OS,CS: String;
    RT: TRecType;
Begin
  DP:=0; WC:=0; BB:=0; RecdOffs:=0; RT:=RTNORM;       // Reset stuff
  Repeat
    HS:=''; OS:=''; CS:='';                           // Reset word strings
    WP:=DP;                                           // Set word offset
    // Read a word
    For BI:=1 to 6 do begin
      Try                                             // Get next byte
        BB:=TAPDev.PeekByte(DP); Inc(DP);
      Except Break; End;
      If (BB=$0F) Or (BB=TAPEMARK) then begin         // Tape mark?
        RT:=RTMARK;                                   // Set mark state
        LogBINData(DP-1,WC,BI,HS,OS,CS,RT);           // Log prev record
        Break;
      End;
      If ((BB And $80)>0) And (RT<>RTRGAP) then begin // New Record gap?
        RT:=RTRGAP;                                   // Set gap state
        LogBINData(WP,WC,BI,HS,OS,CS,RT);             // Log prev record
        Dec(DP);                                      // Backup for next rec
        Break;
      End;
      // Build display strings, Reset record gap state
      HS:=HS+IntToHex(BB,2); OS:=OS+OCT(BB); CS:=CS+BCD(BB);
      RT:=RTNORM;
    End;
    // Log normal word data
    If RT=RTNORM then begin
      LogBINData(WP,WC,BI,HS,OS,CS,RT);
      Inc(WC);
    End;
  Until (WC>=NW) Or (DP>=TAPDev.DataSize);
End;

Procedure TTapeViewForm.LogBINData(DP,WC,BI: Integer; HS,OS,CS: String; RT: TRecType);
Var SS: String;
    RL: Integer;
Begin
  Case RT Of
    RTNORM: Begin
              SS:=Format('%8.8d %6.6d ',[DP,WC])+
                  Pad(HS,13)+Pad(OS,13)+Pad(CS,07);
              If BI<>7 then
                SS:=SS+' Short Word';
              Log(SS);
            End;
    RTRGAP: Begin
              RL:=DP-RecdOffs;
              SS:=Format('%8.8d',[DP])+StringOfChar(' ',42)+
                  'Record Gap (Prev Rec Len='+IntToStr(RL)+')';
              Log(SS);
              RecdOffs:=DP;
            End;
    RTMARK: Begin
              RL:=DP-RecdOffs;
              SS:=Format('%8.8d',[DP])+StringOfChar(' ',42)+
                  'Tape Mark (Prev Rec Len='+IntToStr(RL)+')';
              Log(SS);
              RecdOffs:=DP+1;
            End;
  End;
End;

procedure TTapeViewForm.PBSaveClick(Sender: TObject);
begin
  Screen.Cursor:=CRHourGlass;
  DumpList.SaveToFile(TAPDev.FileNam+'.DMP');
  Screen.Cursor:=CRDefault;
end;

procedure TTapeViewForm.PBNxtMrkClick(Sender: TObject);
begin
  If MarkPtr>32000 then
    MarkPtr:=0;
  With LBDump do
    While MarkPtr<Items.Count-1 do begin
      Inc(MarkPtr);
      If Length(Items[MarkPtr])>50 then begin
        TopIndex:=MarkPtr-2;
        Exit;
      End;
    End;
  MarkPtr:=0;
End;

End.
