/*
 * Decompiled with CFR 0.152.
 */
package nintaco.files;

import com.github.junrar.Archive;
import com.github.junrar.rarfile.FileHeader;
import java.io.BufferedInputStream;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Paths;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import nintaco.App;
import nintaco.files.ArchiveEntry;
import nintaco.files.FilePath;
import nintaco.gui.archive.EntryRegion;
import nintaco.preferences.AppPrefs;
import nintaco.util.StreamUtil;
import nintaco.util.StringUtil;
import org.apache.commons.compress.archivers.sevenz.SevenZArchiveEntry;
import org.apache.commons.compress.archivers.sevenz.SevenZFile;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.compressors.bzip2.BZip2CompressorInputStream;
import org.apache.commons.compress.compressors.gzip.GzipCompressorInputStream;
import org.apache.commons.compress.compressors.lzma.LZMACompressorInputStream;
import org.apache.commons.compress.compressors.xz.XZCompressorInputStream;
import org.apache.commons.compress.compressors.z.ZCompressorInputStream;

public final class FileUtil {
    private static final int MAX_ARCHIVE_ENTRIES = 10000;
    private static final int MAX_FILE_NUMBER = 10000;
    private static final String FILE_EXTENSION_ZIP = "zip";
    private static final String FILE_EXTENSION_7Z = "7z";
    private static final String FILE_EXTENSION_RAR = "rar";
    private static final String FILE_EXTENSION_TAR = "tar";
    private static final String FILE_EXTENSION_TAR_BZ2 = "tar.bz2";
    private static final String FILE_EXTENSION_TAR_GZ = "tar.gz";
    private static final String FILE_EXTENSION_TAR_LZMA = "tar.lzma";
    private static final String FILE_EXTENSION_TAR_XZ = "tar.xz";
    private static final String FILE_EXTENSION_TAR_Z = "tar.Z";
    private static final String FILE_EXTENSION_TB2 = "tb2";
    private static final String FILE_EXTENSION_TBZ = "tbz";
    private static final String FILE_EXTENSION_TBZ2 = "tbz2";
    private static final String FILE_EXTENSION_TGZ = "tgz";
    private static final String FILE_EXTENSION_TLZ = "tlz";
    private static final String FILE_EXTENSION_TXZ = "txz";
    private static final String FILE_EXTENSION_TZ = "tZ";
    private static final Pattern REVISION_PATTERN = Pattern.compile("\\((prg|v|rev)(\\d+)\\)");
    private static final Set<String> ALL_FILE_EXTENSIONS;
    private static final Set<String> WITHIN_ARCHIVE_EXTENSIONS;
    private static final Set<String> ARCHIVE_EXTENSIONS;
    private static final String[] COMPRESSED_TAR_EXTENSIONS;
    private static final String[] HEADERS;
    private static final int[] FILE_TYPES;
    private static final SimpleDateFormat dateFormat;
    private static final Pattern GoodDumpPattern;
    private static String workingDir;

    private FileUtil() {
    }

    public static void getInputStream(FilePath filePath, InputStreamListener listener) throws Throwable {
        if (filePath.isArchivedEntry()) {
            FileUtil.getArchiveInputStream(filePath, listener);
        } else {
            File file = new File(filePath.getEntryPath());
            listener.handleInputStream(new FileInputStream(file), file.length());
        }
    }

    public static void getArchiveInputStream(FilePath filePath, InputStreamListener listener) throws Throwable {
        FileUtil.getArchiveInputStream(filePath.getArchivePath(), filePath.getEntryPath(), listener);
    }

    public static void getArchiveInputStream(FilePath filePath, InputStreamListener listener, Set<String> validExtensions) throws Throwable {
        FileUtil.getArchiveInputStream(filePath.getArchivePath(), filePath.getEntryPath(), listener, validExtensions);
    }

    public static void getArchiveInputStream(String archiveFileName, String entryFileName, InputStreamListener listener) throws Throwable {
        FileUtil.getArchiveInputStream(archiveFileName, entryFileName, listener, WITHIN_ARCHIVE_EXTENSIONS);
    }

    public static void getArchiveInputStream(String archiveFileName, String entryFileName, InputStreamListener listener, Set<String> validExtensions) throws Throwable {
        String extension = FileUtil.getFileExtension(archiveFileName);
        if (extension == null) {
            throw new IOException("Unknown archive file type.");
        }
        File archiveFile = new File(archiveFileName);
        if (!archiveFile.exists()) {
            FileUtil.throwFileNotFound(archiveFile, entryFileName);
        }
        switch (extension) {
            case "zip": {
                FileUtil.getZipInputStream(archiveFile, entryFileName, listener, validExtensions);
                break;
            }
            case "7z": {
                FileUtil.get7ZipInputStream(archiveFile, entryFileName, listener, validExtensions);
                break;
            }
            case "rar": {
                FileUtil.getRarInputStream(archiveFile, entryFileName, listener, validExtensions);
                break;
            }
            case "tar": {
                FileUtil.getTarInputStream(archiveFile, entryFileName, listener, validExtensions);
                break;
            }
            case "tar.bz2": 
            case "tb2": 
            case "tbz": 
            case "tbz2": {
                FileUtil.getTarBz2InputStream(archiveFile, entryFileName, listener, validExtensions);
                break;
            }
            case "tar.gz": 
            case "tgz": {
                FileUtil.getTarGzInputStream(archiveFile, entryFileName, listener, validExtensions);
                break;
            }
            case "tar.lzma": 
            case "tlz": {
                FileUtil.getTarLzmaInputStream(archiveFile, entryFileName, listener, validExtensions);
                break;
            }
            case "tar.xz": 
            case "txz": {
                FileUtil.getTarXzInputStream(archiveFile, entryFileName, listener, validExtensions);
                break;
            }
            case "tar.Z": 
            case "tZ": {
                FileUtil.getTarZInputStream(archiveFile, entryFileName, listener, validExtensions);
                break;
            }
            default: {
                throw new IOException("Unknown archive file type: " + extension);
            }
        }
    }

    private static void getTarInputStream(InputStream in, File archiveFile, String entryFileName, InputStreamListener listener, Set<String> validExtensions) throws Throwable {
        try (TarArchiveInputStream tarInput = new TarArchiveInputStream(in);){
            TarArchiveEntry entry;
            while ((entry = tarInput.getNextTarEntry()) != null) {
                if (entry.isDirectory() || validExtensions != null && !validExtensions.contains(FileUtil.getFileExtension(entry.getName())) || entry.getSize() <= 0L || !entryFileName.equals(entry.getName())) continue;
                listener.handleInputStream(tarInput, entry.getSize());
                return;
            }
        }
        FileUtil.throwFileNotFound(archiveFile, entryFileName);
    }

    private static void getTarInputStream(File archiveFile, String entryFileName, InputStreamListener listener, Set<String> validExtensions) throws Throwable {
        FileUtil.getTarInputStream(new FileInputStream(archiveFile), archiveFile, entryFileName, listener, validExtensions);
    }

    private static void getTarBz2InputStream(File archiveFile, String entryFileName, InputStreamListener listener, Set<String> validExtensions) throws Throwable {
        FileUtil.getTarInputStream(new BZip2CompressorInputStream(new FileInputStream(archiveFile)), archiveFile, entryFileName, listener, validExtensions);
    }

    private static void getTarGzInputStream(File archiveFile, String entryFileName, InputStreamListener listener, Set<String> validExtensions) throws Throwable {
        FileUtil.getTarInputStream(new GzipCompressorInputStream(new FileInputStream(archiveFile)), archiveFile, entryFileName, listener, validExtensions);
    }

    private static void getTarLzmaInputStream(File archiveFile, String entryFileName, InputStreamListener listener, Set<String> validExtensions) throws Throwable {
        FileUtil.getTarInputStream(new LZMACompressorInputStream(new FileInputStream(archiveFile)), archiveFile, entryFileName, listener, validExtensions);
    }

    private static void getTarXzInputStream(File archiveFile, String entryFileName, InputStreamListener listener, Set<String> validExtensions) throws Throwable {
        FileUtil.getTarInputStream(new XZCompressorInputStream(new FileInputStream(archiveFile)), archiveFile, entryFileName, listener, validExtensions);
    }

    private static void getTarZInputStream(File archiveFile, String entryFileName, InputStreamListener listener, Set<String> validExtensions) throws Throwable {
        FileUtil.getTarInputStream(new ZCompressorInputStream(new FileInputStream(archiveFile)), archiveFile, entryFileName, listener, validExtensions);
    }

    private static void getZipInputStream(File archiveFile, String entryFileName, InputStreamListener listener, Set<String> validExtensions) throws Throwable {
        try (ZipFile zipFile = new ZipFile(archiveFile);){
            ZipEntry entry = zipFile.getEntry(entryFileName);
            if (entry == null || entry.isDirectory() || validExtensions != null && !validExtensions.contains(FileUtil.getFileExtension(entry.getName())) || entry.getSize() == 0L) {
                FileUtil.throwFileNotFound(archiveFile, entryFileName);
            }
            listener.handleInputStream(zipFile.getInputStream(entry), entry.getSize());
        }
    }

    private static void get7ZipInputStream(File archiveFile, String entryFileName, InputStreamListener listener, Set<String> validExtensions) throws Throwable {
        try (SevenZFile sevenZFile = new SevenZFile(archiveFile);){
            SevenZArchiveEntry entry;
            while ((entry = sevenZFile.getNextEntry()) != null) {
                if (entry.isDirectory() || validExtensions != null && !validExtensions.contains(FileUtil.getFileExtension(entry.getName())) || entry.getSize() <= 0L || !entryFileName.equals(entry.getName())) continue;
                listener.handleInputStream(new SevenZInputStream(sevenZFile), entry.getSize());
                return;
            }
        }
        FileUtil.throwFileNotFound(archiveFile, entryFileName);
    }

    private static void getRarInputStream(File archiveFile, String entryFileName, InputStreamListener listener, Set<String> validExtensions) throws Throwable {
        try (Archive archive = new Archive(archiveFile);){
            for (FileHeader fileHeader : archive.getFileHeaders()) {
                if (fileHeader.isDirectory() || validExtensions != null && !validExtensions.contains(FileUtil.getFileExtension(fileHeader.getFileNameString())) || fileHeader.getFullUnpackSize() <= 0L || !entryFileName.equals(fileHeader.getFileNameString())) continue;
                listener.handleInputStream(archive.getInputStream(fileHeader), fileHeader.getFullUnpackSize());
                return;
            }
        }
        FileUtil.throwFileNotFound(archiveFile, entryFileName);
    }

    public static byte[] readArchiveEntry(String archiveFileName, ArchiveEntry entry) throws Throwable {
        byte[] data = new byte[(int)entry.getSize()];
        FileUtil.readArchiveEntry(archiveFileName, entry, data);
        return data;
    }

    public static void readArchiveEntry(String archiveFileName, ArchiveEntry entry, byte[] data) throws Throwable {
        FileUtil.readArchiveEntry(archiveFileName, entry, data, 0, data.length);
    }

    public static void readArchiveEntry(String archiveFileName, ArchiveEntry entry, byte[] data, int offset, int length) throws Throwable {
        FileUtil.getArchiveInputStream(archiveFileName, entry.getName(), (InputStream in, long size) -> {
            try (DataInputStream dis = new DataInputStream(in);){
                dis.readFully(data, offset, length);
            }
        });
    }

    private static void throwFileNotFound(File archiveFile, String entryFileName) throws FileNotFoundException {
        throw new FileNotFoundException(String.format("%s <%s> not found.", archiveFile.getPath(), entryFileName));
    }

    public static List<ArchiveEntry> getArchiveEntries(File archiveFile) throws Throwable {
        return FileUtil.getArchiveEntries(archiveFile.getPath());
    }

    public static List<ArchiveEntry> getArchiveEntries(String archiveFileName) throws Throwable {
        return FileUtil.getArchiveEntries(archiveFileName, WITHIN_ARCHIVE_EXTENSIONS);
    }

    public static List<ArchiveEntry> getArchiveEntries(String archiveFileName, Set<String> validExtensions) throws Throwable {
        String extension = FileUtil.getFileExtension(archiveFileName);
        if (extension == null) {
            return null;
        }
        switch (extension) {
            case "zip": {
                return FileUtil.getZipEntries(archiveFileName, validExtensions);
            }
            case "7z": {
                return FileUtil.get7ZipEntries(archiveFileName, validExtensions);
            }
            case "rar": {
                return FileUtil.getRarEntries(archiveFileName, validExtensions);
            }
            case "tar": {
                return FileUtil.getTarEntries(archiveFileName, validExtensions);
            }
            case "tar.bz2": 
            case "tb2": 
            case "tbz": 
            case "tbz2": {
                return FileUtil.getTarBz2Entries(archiveFileName, validExtensions);
            }
            case "tar.gz": 
            case "tgz": {
                return FileUtil.getTarGzEntries(archiveFileName, validExtensions);
            }
            case "tar.lzma": 
            case "tlz": {
                return FileUtil.getTarLzmaEntries(archiveFileName, validExtensions);
            }
            case "tar.xz": 
            case "txz": {
                return FileUtil.getTarXzEntries(archiveFileName, validExtensions);
            }
            case "tar.Z": 
            case "tZ": {
                return FileUtil.getTarZEntries(archiveFileName, validExtensions);
            }
        }
        return null;
    }

    private static List<ArchiveEntry> getRarEntries(String fileName, Set<String> validExtensions) throws Throwable {
        ArrayList<ArchiveEntry> entries = new ArrayList<ArchiveEntry>();
        try (Archive archive = new Archive(new File(fileName));){
            for (FileHeader fileHeader : archive.getFileHeaders()) {
                if (fileHeader.isDirectory() || validExtensions != null && !validExtensions.contains(FileUtil.getFileExtension(fileHeader.getFileNameString())) || fileHeader.getFullUnpackSize() <= 0L) continue;
                entries.add(new ArchiveEntry(fileHeader.getFileNameString(), fileHeader.getFullUnpackSize()));
            }
        }
        Collections.sort(entries, ArchiveEntry.CASE_INSENSITIVE_ORDER);
        return entries;
    }

    private static List<ArchiveEntry> get7ZipEntries(String fileName, Set<String> validExtensions) throws Throwable {
        ArrayList<ArchiveEntry> entries = new ArrayList<ArchiveEntry>();
        try (SevenZFile sevenZFile = new SevenZFile(new File(fileName));){
            SevenZArchiveEntry entry;
            while ((entry = sevenZFile.getNextEntry()) != null) {
                if (entries.size() >= 10000) {
                    break;
                }
                if (entry.isDirectory() || validExtensions != null && !validExtensions.contains(FileUtil.getFileExtension(entry.getName())) || entry.getSize() <= 0L) continue;
                entries.add(new ArchiveEntry(entry.getName(), entry.getSize()));
            }
        }
        Collections.sort(entries, ArchiveEntry.CASE_INSENSITIVE_ORDER);
        return entries;
    }

    public static List<ArchiveEntry> getZipEntries(String fileName, Set<String> validExtensions) throws Throwable {
        ArrayList<ArchiveEntry> entries = new ArrayList<ArchiveEntry>();
        try (ZipFile zipFile = new ZipFile(new File(fileName));){
            Enumeration<? extends ZipEntry> i = zipFile.entries();
            while (i.hasMoreElements() && entries.size() < 10000) {
                ZipEntry entry = i.nextElement();
                if (entry.isDirectory() || validExtensions != null && !validExtensions.contains(FileUtil.getFileExtension(entry.getName())) || entry.getSize() <= 0L) continue;
                entries.add(new ArchiveEntry(entry.getName(), entry.getSize()));
            }
        }
        Collections.sort(entries, ArchiveEntry.CASE_INSENSITIVE_ORDER);
        return entries;
    }

    public static List<ArchiveEntry> getTarEntries(InputStream in, Set<String> validExtensions) throws Throwable {
        TarArchiveEntry entry;
        TarArchiveInputStream tarInput = new TarArchiveInputStream(in);
        ArrayList<ArchiveEntry> entries = new ArrayList<ArchiveEntry>();
        while ((entry = tarInput.getNextTarEntry()) != null) {
            if (entry.isDirectory() || validExtensions != null && !validExtensions.contains(FileUtil.getFileExtension(entry.getName())) || entry.getSize() <= 0L) continue;
            entries.add(new ArchiveEntry(entry.getName(), entry.getSize()));
        }
        Collections.sort(entries, ArchiveEntry.CASE_INSENSITIVE_ORDER);
        return entries;
    }

    public static List<ArchiveEntry> getTarEntries(String fileName, Set<String> validExtensions) throws Throwable {
        return FileUtil.getTarEntries(new FileInputStream(fileName), validExtensions);
    }

    public static List<ArchiveEntry> getTarBz2Entries(String fileName, Set<String> validExtensions) throws Throwable {
        return FileUtil.getTarEntries(new BZip2CompressorInputStream(new FileInputStream(fileName)), validExtensions);
    }

    public static List<ArchiveEntry> getTarGzEntries(String fileName, Set<String> validExtensions) throws Throwable {
        return FileUtil.getTarEntries(new GzipCompressorInputStream(new FileInputStream(fileName)), validExtensions);
    }

    public static List<ArchiveEntry> getTarLzmaEntries(String fileName, Set<String> validExtensions) throws Throwable {
        return FileUtil.getTarEntries(new LZMACompressorInputStream(new FileInputStream(fileName)), validExtensions);
    }

    public static List<ArchiveEntry> getTarXzEntries(String fileName, Set<String> validExtensions) throws Throwable {
        return FileUtil.getTarEntries(new XZCompressorInputStream(new FileInputStream(fileName)), validExtensions);
    }

    public static List<ArchiveEntry> getTarZEntries(String fileName, Set<String> validExtensions) throws Throwable {
        return FileUtil.getTarEntries(new ZCompressorInputStream(new FileInputStream(fileName)), validExtensions);
    }

    public static boolean isArchiveFile(String fileName) {
        return ARCHIVE_EXTENSIONS.contains(FileUtil.getFileExtension(fileName));
    }

    public static int getFileExtensionIndex(String fileName) {
        int i;
        for (i = COMPRESSED_TAR_EXTENSIONS.length - 1; i >= 0; --i) {
            if (!fileName.endsWith(COMPRESSED_TAR_EXTENSIONS[i])) continue;
            return fileName.length() - COMPRESSED_TAR_EXTENSIONS[i].length();
        }
        for (i = fileName.length() - 1; i >= 0; --i) {
            char c = fileName.charAt(i);
            if (Character.isJavaIdentifierPart(c)) continue;
            if (c == '.') {
                return i + 1;
            }
            return -1;
        }
        return -1;
    }

    public static String getFileExtension(File file) {
        String fileName = file.getName();
        int index = FileUtil.getFileExtensionIndex(fileName);
        if (index < 0) {
            return "";
        }
        return fileName.substring(index).toLowerCase(Locale.ENGLISH);
    }

    public static String getFileExtension(String fileName) {
        return StringUtil.isBlank(fileName) ? null : FileUtil.getFileExtension(new File(fileName));
    }

    public static String getDirectoryPath(String fileName) {
        if (StringUtil.isBlank(fileName)) {
            return null;
        }
        if (FileUtil.isDirectory(fileName)) {
            return fileName;
        }
        return new File(fileName).getParent();
    }

    public static int getFileType(String fileName) {
        return FileUtil.getFileType(new File(fileName));
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static int getFileType(File file) {
        if (!file.exists()) return 0;
        if (!file.isFile()) return 0;
        try (DataInputStream in = new DataInputStream(new BufferedInputStream(new FileInputStream(file)));){
            int n = FileUtil.getFileType(in);
            return n;
        }
        catch (Throwable throwable6) {
            // empty catch block
        }
        return 0;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static int getFileType(DataInputStream in) throws Throwable {
        if (!in.markSupported()) {
            throw new IOException("Stream does not support mark.");
        }
        in.mark(256);
        try {
            int[] header = new int[5];
            StreamUtil.readByteArray(in, header, 0, header.length, true);
            for (int i = 0; i < HEADERS.length; ++i) {
                if (!StringUtil.compareStrings(HEADERS[i], header)) continue;
                int n = FILE_TYPES[i];
                return n;
            }
            in.skipBytes(1);
            if ("*NINTENDO-HVC*".equals(StreamUtil.readString(in, "*NINTENDO-HVC*".length()))) {
                int n = 2;
                return n;
            }
        }
        finally {
            in.reset();
        }
        return 0;
    }

    public static synchronized String getWorkingDirectory() {
        if (StringUtil.isBlank(workingDir)) {
            File file = null;
            try {
                file = new File(FileUtil.class.getProtectionDomain().getCodeSource().getLocation().toURI().getPath());
            }
            catch (Throwable throwable) {
                // empty catch block
            }
            workingDir = file != null && file.exists() && file.isFile() && file.getName().toLowerCase().endsWith(".jar") ? file.getParent() : Paths.get("", new String[0]).toAbsolutePath().toString();
        }
        return workingDir;
    }

    public static String getWorkingDirectory(String ... appends) {
        StringBuilder sb = new StringBuilder(FileUtil.getWorkingDirectory());
        for (String append : appends) {
            sb.append(File.separator);
            sb.append(append);
        }
        return sb.toString();
    }

    public static File getFile(String directory, String fileName) {
        return FileUtil.getFile(new File(directory), fileName);
    }

    public static File getFile(File directory, String fileName) {
        if (directory == null) {
            directory = new File(AppPrefs.getInstance().getPaths().getContentDirectory());
        }
        if (StringUtil.isBlank(fileName)) {
            return new File(directory.getPath());
        }
        return new File(directory.getPath() + File.separator + fileName);
    }

    public static String getFileName(String path) {
        if (path == null) {
            return null;
        }
        return new File(path).getName();
    }

    public static String getFileNameWithoutExtension(String path) {
        return FileUtil.removeExtension(FileUtil.getFileName(path));
    }

    public static String removeExtension(String fileName) {
        if (fileName == null) {
            return null;
        }
        int index = FileUtil.getFileExtensionIndex(fileName);
        if (index <= 0) {
            return "";
        }
        return fileName.substring(0, index - 1);
    }

    public static boolean directoryExists(String directory) {
        if (StringUtil.isBlank(directory)) {
            return false;
        }
        File dir = new File(directory);
        return FileUtil.isDirectory(directory) && dir.exists();
    }

    public static File mkdir(String directory) {
        if (StringUtil.isBlank(directory)) {
            return null;
        }
        File dir = new File(directory);
        if (FileUtil.isDirectory(directory) && !dir.exists()) {
            dir.mkdirs();
        }
        return dir;
    }

    public static File findExistingParent(String path) {
        return FileUtil.findExistingParent(new File(path));
    }

    public static File findExistingParent(File file) {
        while (file != null) {
            if (file.exists()) {
                return file;
            }
            file = file.getParentFile();
        }
        return file;
    }

    public static int getSuggestedStartIndex(String filePrefix, String outputDir) {
        File outDir = new File(outputDir);
        if (!outDir.exists() || !outDir.isDirectory()) {
            return 0;
        }
        int index = -1;
        for (File file : outDir.listFiles()) {
            int i;
            String name;
            if (!file.isFile() || StringUtil.isBlank(name = FileUtil.getFileNameWithoutExtension(file.getPath())) || name.length() < 4 || !name.startsWith(filePrefix) || (i = name.lastIndexOf(45)) < 0 || i == name.length() - 1) continue;
            try {
                index = Math.max(Integer.parseInt(name.substring(i + 1)), index);
            }
            catch (Throwable throwable) {
                // empty catch block
            }
        }
        return index + 1;
    }

    public static int indexOf(List<String> directories, String directory) {
        if (directories == null || StringUtil.isBlank(directory)) {
            return -1;
        }
        File dir = new File(directory);
        for (int i = directories.size() - 1; i >= 0; --i) {
            String dirName = directories.get(i);
            if (dirName == null || !new File(dirName).equals(dir)) continue;
            return i;
        }
        return -1;
    }

    public static String appendSeparator(String path) {
        if (path.endsWith(File.separator) || path.endsWith("/") || path.endsWith("\\")) {
            return path;
        }
        return path + File.separator;
    }

    public static String getCanonicalName(String name) {
        if (StringUtil.isBlank(name)) {
            return null;
        }
        try {
            return new File(name).getCanonicalPath();
        }
        catch (Throwable t) {
            return name.toLowerCase(Locale.ENGLISH);
        }
    }

    public static File createUniqueFile(String filePath, boolean alwaysAppendSuffix, String suffix) {
        return FileUtil.createUniqueFile(new File(filePath), alwaysAppendSuffix, suffix);
    }

    public static File createUniqueFile(File file, boolean alwaysAppendSuffix, String suffix) {
        String fileName = file.getName();
        int index = FileUtil.getFileExtensionIndex(fileName);
        if (index <= 0) {
            return null;
        }
        String extension = fileName.substring(index).toLowerCase(Locale.ENGLISH);
        String prefix = fileName.substring(0, index - 1);
        String directory = file.getParent();
        if (directory == null) {
            directory = "";
        }
        return FileUtil.createUniqueFile(directory, prefix, extension, alwaysAppendSuffix, suffix);
    }

    public static File createUniqueFile(String directory, String prefix, String extension, boolean alwaysAppendSuffix, String suffix) {
        return new File(FileUtil.appendSeparator(directory) + FileUtil.createUniqueFileName(directory, prefix, extension, alwaysAppendSuffix, suffix));
    }

    public static String createUniqueFileName(String directory, String prefix, String extension, boolean alwaysAppendSuffix, String suffix) {
        HashSet<String> nameSet = new HashSet<String>();
        for (File file : new File(directory).listFiles()) {
            String name = file.getName();
            if (!name.startsWith(prefix)) continue;
            nameSet.add(name);
        }
        if (StringUtil.isBlank(suffix)) {
            suffix = "-%03d";
        }
        String fileName = prefix + "." + extension;
        if (alwaysAppendSuffix || nameSet.contains(fileName)) {
            int i;
            int n = i = alwaysAppendSuffix ? 0 : 1;
            while (i < 10000 && nameSet.contains(fileName = prefix + String.format(suffix, i) + "." + extension)) {
                ++i;
            }
        }
        return fileName;
    }

    public static boolean containsSeparators(String path) {
        return path.contains("/") || path.contains("\\") || path.contains(":") || path.contains(";");
    }

    public static boolean isDirectory(String path) {
        File dir = new File(path);
        return dir.isDirectory() || FileUtil.isDirectoryPath(dir, path);
    }

    public static boolean isDirectory(File file) {
        if (file.isDirectory()) {
            return true;
        }
        return FileUtil.isDirectoryPath(file, file.getPath());
    }

    private static boolean isDirectoryPath(File dir, String path) {
        return (path = path.trim()).endsWith("/") || path.endsWith("\\") || path.endsWith(":") || FileUtil.getFileExtensionIndex(dir.getPath()) < 0;
    }

    public static String createSaveFile(String fileName) {
        return FileUtil.appendSeparator(AppPrefs.getInstance().getPaths().getSaveStatesDir()) + FileUtil.getFileNameWithoutExtension(fileName) + ".save";
    }

    public static String createGamePreferencesFile(String fileName) {
        return FileUtil.appendSeparator(AppPrefs.getInstance().getPaths().getGamePreferencesDir()) + FileUtil.getFileNameWithoutExtension(fileName) + ".preferences";
    }

    public static String createLogFile() {
        String entryFileName = App.getEntryFileName();
        return FileUtil.appendSeparator(AppPrefs.getInstance().getPaths().getLogsDir()) + (StringUtil.isBlank(entryFileName) ? "trace" : FileUtil.getFileNameWithoutExtension(entryFileName)) + ".log";
    }

    public static String createCheatFile(String fileName) {
        return FileUtil.appendSeparator(AppPrefs.getInstance().getPaths().getCheatsDir()) + FileUtil.getFileNameWithoutExtension(fileName) + ".cht";
    }

    public static synchronized String getFileTimestamp(int index, Date date) {
        return String.format("%d  %s      ", index, dateFormat.format(date));
    }

    public static String getFileTimestamp(int index, File file) {
        return FileUtil.getFileTimestamp(index, new Date(file.lastModified()));
    }

    public static boolean isFamicomDiskSystemFile(FilePath filePath) {
        return FileUtil.isFamicomDiskSystemFile(filePath.isArchivedEntry() ? filePath.getEntryPath() : filePath.getArchivePath());
    }

    public static boolean isFamicomDiskSystemFile(String fileName) {
        return "fds".equals(FileUtil.getFileExtension(fileName));
    }

    public static int getDefaultArchiveEntry(String archiveFileName, List<String> entries) {
        return FileUtil.getDefaultArchiveEntry(archiveFileName, entries, EntryRegion.getPrioritizedRegions(AppPrefs.getInstance().getArchivePrefs().getArchiveEntryRegion()));
    }

    private static int getDefaultArchiveEntry(String archiveFileName, List<String> entries, String[] regions) {
        if (entries.isEmpty()) {
            return -1;
        }
        if (entries.size() == 1) {
            return 0;
        }
        String fileName = FileUtil.getFileNameWithoutExtension(archiveFileName);
        int index = fileName.indexOf(40);
        if (index >= 0) {
            fileName = fileName.substring(0, index);
        }
        if ((index = fileName.indexOf(91)) >= 0) {
            fileName = fileName.substring(0, index);
        }
        fileName = fileName.trim().toLowerCase(Locale.ENGLISH);
        int bestIndex = -1;
        int bestRank = -1;
        for (int i = entries.size() - 1; i >= 0; --i) {
            int rank = FileUtil.rankArchiveEntry(fileName, entries.get(i), regions);
            if (rank <= bestRank) continue;
            bestRank = rank;
            bestIndex = i;
        }
        return bestIndex;
    }

    private static int rankArchiveEntry(String archiveFileName, String entry, String[] regions) {
        Matcher matcher;
        if ((entry = entry.toLowerCase(Locale.ENGLISH)).endsWith(".nsf") || entry.endsWith(".nsfe")) {
            return -1;
        }
        int rank = 0;
        int len2 = regions.length * regions.length;
        int len3 = len2 * regions.length;
        int len4 = len3 * regions.length;
        int len5 = len4 * regions.length;
        if (GoodDumpPattern.matcher(entry).find()) {
            rank += len5;
        } else if (entry.indexOf(91) < 0 || entry.contains("ntsc")) {
            rank += len4;
        }
        for (int i = regions.length - 1; i >= 0; --i) {
            if (!entry.contains(regions[i])) continue;
            rank += len3 * i;
            break;
        }
        if (entry.contains(archiveFileName)) {
            rank += len2;
        }
        if ((matcher = REVISION_PATTERN.matcher(entry)).find()) {
            rank += regions.length * Math.min(9, Integer.valueOf(matcher.group(2)));
        }
        return rank - entry.length();
    }

    static {
        ARCHIVE_EXTENSIONS = new HashSet<String>();
        ARCHIVE_EXTENSIONS.add(FILE_EXTENSION_ZIP);
        ARCHIVE_EXTENSIONS.add(FILE_EXTENSION_7Z);
        ARCHIVE_EXTENSIONS.add(FILE_EXTENSION_RAR);
        ARCHIVE_EXTENSIONS.add(FILE_EXTENSION_TAR);
        ARCHIVE_EXTENSIONS.add(FILE_EXTENSION_TAR_BZ2);
        ARCHIVE_EXTENSIONS.add(FILE_EXTENSION_TAR_GZ);
        ARCHIVE_EXTENSIONS.add(FILE_EXTENSION_TAR_LZMA);
        ARCHIVE_EXTENSIONS.add(FILE_EXTENSION_TAR_XZ);
        ARCHIVE_EXTENSIONS.add(FILE_EXTENSION_TAR_Z);
        ARCHIVE_EXTENSIONS.add(FILE_EXTENSION_TB2);
        ARCHIVE_EXTENSIONS.add(FILE_EXTENSION_TBZ);
        ARCHIVE_EXTENSIONS.add(FILE_EXTENSION_TBZ2);
        ARCHIVE_EXTENSIONS.add(FILE_EXTENSION_TGZ);
        ARCHIVE_EXTENSIONS.add(FILE_EXTENSION_TLZ);
        ARCHIVE_EXTENSIONS.add(FILE_EXTENSION_TXZ);
        ARCHIVE_EXTENSIONS.add(FILE_EXTENSION_TZ);
        WITHIN_ARCHIVE_EXTENSIONS = new HashSet<String>(Arrays.asList("nes", "unf", "unif", "fds", "nsf", "nez"));
        ALL_FILE_EXTENSIONS = new HashSet<String>(WITHIN_ARCHIVE_EXTENSIONS);
        ALL_FILE_EXTENSIONS.addAll(ARCHIVE_EXTENSIONS);
        COMPRESSED_TAR_EXTENSIONS = new String[]{FILE_EXTENSION_TAR_BZ2, FILE_EXTENSION_TAR_GZ, FILE_EXTENSION_TAR_LZMA, FILE_EXTENSION_TAR_XZ, FILE_EXTENSION_TAR_Z};
        HEADERS = new String[]{"NES\u001a", "FDS\u001a", "UNIF", "NESM\u001a", "NSFE"};
        FILE_TYPES = new int[]{1, 2, 3, 4, 4};
        dateFormat = new SimpleDateFormat("yyyy/MM/dd HH:mm");
        GoodDumpPattern = Pattern.compile("\\[[^\\]]*\\![^\\]]*\\]");
    }

    private static class SevenZInputStream
    extends InputStream {
        final SevenZFile sevenZFile;

        public SevenZInputStream(SevenZFile sevenZFile) {
            this.sevenZFile = sevenZFile;
        }

        @Override
        public int read() throws IOException {
            return this.sevenZFile.read();
        }

        @Override
        public int read(byte[] b) throws IOException {
            return this.sevenZFile.read(b);
        }

        @Override
        public int read(byte[] b, int off, int len) throws IOException {
            return this.sevenZFile.read(b, off, len);
        }
    }

    public static interface InputStreamListener {
        public void handleInputStream(InputStream var1, long var2) throws Throwable;
    }
}

