; Core emulation for 65816 microprocessor
; (C) 1997 Realtime Simulations and Roleplaying games, Inc.
; More of Grog's handywork
;   Mark II:  Last revision 12/3/97

MemRead MACRO
  call SNESFullMemRead
  movzx edx,cl
  IFDEF DEBUGON
    push eax
    mov al,dl
    call DebugRead
    pop eax
  ENDIF
ENDM
MemWrite MACRO
  mov cl,dl
  call SNESFullMemWrite
  IFDEF DEBUGON
    push eax
    mov al,dl
    call DebugWrite
    pop eax
  ENDIF
ENDM
MemReadH MACRO
  call SNESFullMemRead
  mov dh,cl
  IFDEF DEBUGON
    push eax
    mov al,ah
    call DebugRead
    pop eax
  ENDIF
ENDM
MemWriteH MACRO
  mov cl,dh
  call SNESFullMemWrite
  IFDEF DEBUGON
    push eax
    mov al,dh
    call DebugWrite
    pop eax
  ENDIF
ENDM
MemReadS MACRO
  ;Stack only read
  movzx edx,BYTE PTR [ebx+ebp+SNESMEM+MIRRORRAM]   ;How's that for fast?
  IFDEF DEBUGON
    push eax
    mov al,dl
    call DebugRead
    pop eax
  ENDIF
ENDM
MemReadSH MACRO
  ;stack only read high
  mov dh,BYTE PTR [ebx+ebp+SNESMEM+MIRRORRAM]   ;How's that for fast?
  IFDEF DEBUGON
    push eax
    mov al,dh
    call DebugRead
    pop eax
  ENDIF
ENDM
MemWriteS MACRO
  mov BYTE PTR [ebx+ebp+SNESMEM+MIRRORRAM],dl
  IFDEF DEBUGON
    push eax
    mov al,dl
    call DebugWrite
    pop eax
  ENDIF
  ;stack only write
ENDM
MemWriteSH MACRO
  mov BYTE PTR [ebx+ebp+SNESMEM+MIRRORRAM],dh
  IFDEF DEBUGON
    push eax
    mov al,dh
    call DebugWrite
    pop eax
  ENDIF
  ;stack only write high
ENDM
MemReadP MACRO
  ;Program memory access (aka never a memory mapped register)
  ;check for mirrored RAM eventually (just in case)
  movzx eax,BYTE PTR [esi+ebp+SNESMEM]   ;How's that for fast?
  IFDEF DEBUGON
    call DebugRead
  ENDIF
ENDM
MemReadPH MACRO
  ;Program memory access (aka never a memory mapped register)
  ;check for mirrored RAM eventually (just in case)
  ;duplicates MemReadP except data is stored into ah
  mov ah,BYTE PTR [esi+ebp+SNESMEM]   ;How's that for fast?
  IFDEF DEBUGON
    push eax
    mov al,ah
    call DebugRead
    pop eax
  ENDIF
ENDM
MemReadPB MACRO
  ;Program memory access (aka never a memory mapped register)
  ;check for mirrored RAM eventually (just in case)
  ;duplicates MemReadP except data is stored into bits 16-23
  movzx ebx,BYTE PTR [esi+ebp+SNESMEM]   ;How's that for fast?
  IFDEF DEBUGON
    push eax
    mov al,bl
    call DebugRead
    pop eax
  ENDIF
  shl ebx,16
  or eax,ebx
ENDM

even
eAL db 0
eAH db 0,0,0
eDL db 0
eDH db 0,0,0
eDBP db 0,0
eDB db 0,0
eXL db 0
eXH db 0,0,0
eYL db 0
eYH db 0,0,0
ePCL db 0
ePCH db 0
ePB db 0,0
eSL db 0
eSH db 0,0,0
eP db 0,0,0,0
ePe db 1
ePn EQU 128
ePv EQU 64
ePm EQU 32
ePx EQU 16
ePd EQU 8
ePi EQU 4
ePz EQU 2
ePc EQU 1
eWAIT dd 0

debugstring db 0,0,0,0,0,0,0,0

OpEmu PROC   ;Execute a 65816 instruction  (does not preserve registers)
  mov esi,DWORD PTR [ePCL]            ;get the program counter
  MemReadP                            ;get the opcode byte into eax
  inc si                              ;increment the program counter
  IFDEF DEBUGON
    call DebugOpcode
  ENDIF
  shl eax,2                           ;set up for indirect jump
  test [eP],ePm                       ;determine accumulator width
  jz SHORT OpEx16                     ;
  jmp [eax+eJumpTable]                ;execute opcode (8-bit accumulator)
  OpEx16:                             ;
  jmp [eax+nJumpTable]                ;execute opcode (16-bit accumulator)
  EndOp:                              ;return from opcode
  mov DWORD PTR [ePCL],esi            ;preserve the program counter
  ret
ENDP

;ͻ
;                      Memory Addressing Modes                        
;ͼ
;These macros encapsulate the memory addressing modes of the 65816, providing
;for simplified opcode emulation code.  8 and 16 bit flavors are seperated.
;
;edi returns a SNES memory pointer for data access (passable to MemXXXB)

aA MACRO                     ;Absolute Addressing
  MemReadP                     ;Get AAL
  inc si                       ;increment PC
  MemReadPH                    ;Get AAH into ah
  inc si                       ;increment PC
  or eax,DWORD PTR [eDBP]      ;cancate DB:AA
  mov edi,eax                  ;set output [look into eliminating this step]
ENDM
aAXX MACRO                   ;Absolute Indexed,X Addressing
  MemReadP                     ;Get AAL
  inc si                       ;increment PC
  MemReadPH                    ;Get AAH into ah
  inc si                       ;increment PC
  or eax,DWORD PTR [eDBP]      ;cancate DB:AA
  add eax,DWORD PTR [eXL]      ;add index X
  mov edi,eax                  ;set output [""]
ENDM
aAXY MACRO                   ;Absolute Indexed,Y Addressing
  MemReadP                     ;Get AAL
  inc si                       ;increment PC
  MemReadPH                    ;Get AAH into ah
  inc si                       ;increment PC
  or eax,DWORD PTR [eDBP]      ;cancate DB:AA
  add eax,DWORD PTR [eYL]      ;add index Y
  mov edi,eax                  ;set output [""]
ENDM
aAXI MACRO                   ;Absolute Indexed,X Indirect Addressing
  MemReadP                     ;Get IAL
  inc si                       ;increment PC
  MemReadPH                    ;Get IAH into ah
  inc si                       ;increment PC
  add ax,WORD PTR [eXL]        ;add index X
  mov ebx,DWORD PTR [ePCL]     ;
  xor bx,bx                    ;
  or eax,ebx                   ;cancate PB:AA
  mov esi,eax                  ;setup for AA
  MemReadP                     ;Get AAL
  inc si                       ;increment program counter
  MemReadPH                    ;Get AAH
  mov si,ax                    ;Update PC
ENDM
aAI MACRO                    ;Absolute Indirect Addressing
  MemReadP                     ;Get IAL
  inc si                       ;increment PC
  MemReadPH                    ;Get IAH into ah
  inc si                       ;increment PC
  mov esi,eax                  ;setup for AA
  MemReadP                     ;Get AAL
  inc si                       ;increment program counter
  MemReadPH                    ;Get AAH
  mov ebx,DWORD PTR [ePCL]     ;
  xor bx,bx                    ;
  or eax,ebx                   ;cancate PB:AA
  mov edi,eax                  ;set output [""]
ENDM
aAIL MACRO                   ;Absolute Indirect Long Addressing
  MemReadP                     ;Get IAL
  inc si                       ;increment PC
  MemReadPH                    ;Get IAH into ah
  inc si                       ;increment PC
  mov esi,eax                  ;setup for AA
  MemReadP                     ;Get AAL
  inc si                       ;increment indirect counter
  MemReadPH                    ;Get AAH
  inc si                       ;increment indirect counter
  MemReadPB                    ;Get AAB
  mov esi,eax                  ;set new PC
ENDM
aAL MACRO                    ;Absolute Long Addressing
  MemReadP                     ;Get IAL
  inc si                       ;increment PC
  MemReadPH                    ;Get IAH into ah
  inc si                       ;increment PC
  MemReadPB                    ;Get AAB
  inc si                       ;increment PC
  mov edi,eax                  ;set output [""]
ENDM
aALXX MACRO                  ;Absolute Long Indexed,X
  MemReadP                     ;Get IAL
  inc si                       ;increment PC
  MemReadPH                    ;Get IAH into ah
  inc si                       ;increment PC
  MemReadPB                    ;Get AAB
  inc si                       ;increment PC
  add eax,DWORD PTR [eXL]      ;add index X
  mov edi,eax                  ;set output [""]
ENDM
aD MACRO                     ;Direct Page Addressing
  MemReadP                     ;Get byte offset
  inc si                       ;increment PC
  add ax,WORD PTR [eDL]        ;add Direct register and byte offset
  mov edi,eax                  ;set output [""]
ENDM
aDXX MACRO                   ;Direct Page Indexed,X Addressing
  MemReadP                     ;Get byte offset
  inc si                       ;increment PC
  add ax,WORD PTR [eDL]        ;add Direct register and byte offset
  add ax,WORD PTR [eXL]        ;add index X
  mov edi,eax                  ;set output [""]
ENDM
aDXY MACRO                   ;Direct Page Indexed,Y Addressing
  MemReadP                     ;Get byte offset
  inc si                       ;increment PC
  add ax,WORD PTR [eDL]        ;add Direct register and byte offset
  add ax,WORD PTR [eYL]        ;add index X
  mov edi,eax                  ;set output [""]
ENDM
aDXXI MACRO                  ;Direct Indexed,X Indirect Addressing
  MemReadP                     ;Get byte offset
  inc si                       ;increment PC
  add ax,WORD PTR [eDL]        ;add Direct register and byte offset
  add ax,WORD PTR [eXL]        ;add index X
  push esi                     ;preserve PC
  mov esi,eax                  ;setup for AA
  MemReadP                     ;Get AAL
  inc si                       ;increment program counter
  MemReadPH                    ;Get AAH
  pop esi                      ;restore PC
  or eax,DWORD PTR [eDBP]      ;cancate DB:AA
  mov edi,eax                  ;set output [""]
ENDM
aDI MACRO                    ;Direct Indirect Addressing
  MemReadP                     ;Get byte offset
  inc si                       ;increment PC
  add ax,WORD PTR [eDL]        ;add Direct register and byte offset
  push esi                     ;preserve PC
  mov esi,eax                  ;setup for AA
  MemReadP                     ;Get AAL
  inc si                       ;increment program counter
  MemReadPH                    ;Get AAH
  pop esi                      ;restore PC
  or eax,DWORD PTR [eDBP]      ;cancate DB:AA
  mov edi,eax                  ;set output [""]
ENDM
aDIL MACRO                   ;Direct Indirect Long Addressing
  MemReadP                     ;Get byte offset
  inc si                       ;increment PC
  add ax,WORD PTR [eDL]        ;add Direct register and byte offset
  push esi                     ;preserve PC
  mov esi,eax                  ;setup for AA
  MemReadP                     ;Get AAL
  inc si                       ;increment indirect counter
  MemReadPH                    ;Get AAH
  inc si                       ;increment indirect counter
  MemReadPB                    ;Get AAB
  pop esi                      ;restore PC
  mov edi,eax                  ;set output [""]
ENDM
aDIXY MACRO                  ;Direct Indirect Indexed,Y Addressing
  MemReadP                     ;Get byte offset
  inc si                       ;increment PC
  add ax,WORD PTR [eDL]        ;add Direct register and byte offset
  push esi                     ;preserve PC
  mov esi,eax                  ;setup for AA
  MemReadP                     ;Get AAL
  inc si                       ;increment program counter
  MemReadPH                    ;Get AAH
  pop esi                      ;restore PC
  or eax,DWORD PTR [eDBP]      ;cancate DB:AA
  add eax,DWORD PTR [eYL]      ;postindex by Y
  mov edi,eax                  ;set output [""]
ENDM
aDILXY MACRO                 ;Direct Indirect Long Indexed,Y Addressing
  MemReadP                     ;Get byte offset
  inc si                       ;increment PC
  add ax,WORD PTR [eDL]        ;add Direct register and byte offset
  push esi                     ;preserve PC
  mov esi,eax                  ;setup for AA
  MemReadP                     ;Get AAL
  inc si                       ;increment indirect counter
  MemReadPH                    ;Get AAH
  inc si                       ;increment indirect counter
  MemReadPB                    ;Get AAB
  pop esi                      ;restore PC
  add eax,DWORD PTR [eYL]      ;postindex by Y
  mov edi,eax                  ;set output [""]
ENDM
aSR MACRO                    ;Stack Relative Addressing
  MemReadP                     ;Get byte offset
  inc si                       ;increment PC
  add ax,WORD PTR [eSL]        ;add Stack register and byte offset
  mov edi,eax                  ;set output [""]
ENDM
aSRIXY MACRO                 ;Stack Relative Indirect Indexed,Y Addressing
  MemReadP                     ;Get byte offset
  inc si                       ;increment PC
  add ax,WORD PTR [eSL]        ;add Stack register and byte offset
  push esi                     ;preserve PC
  mov esi,eax                  ;setup for AA
  MemReadP                     ;Get AAL
  inc si                       ;increment program counter
  MemReadPH                    ;Get AAH
  pop esi                      ;restore PC
  or eax,DWORD PTR [eDBP]      ;cancate DB:AA
  add eax,DWORD PTR [eYL]      ;postindex by Y register
  mov edi,eax                  ;set output [""]
ENDM
aBRA MACRO                   ;Branch Relative Short Addressing, Branch taken
  MemReadP                     ;Get branch offset
  inc si                       ;increment PC
  movsx eax,al                 ;sign extend offset
  add si,ax                    ;execute branch
ENDM
aNBRA MACRO                  ;Branch Relative Short Addressing, Branch not taken
  inc si                       ;increment PC (branch not taken)
ENDM
aBRL MACRO                   ;Branch Relative Long Addressing, Branch taken
  MemReadP                     ;Get branch offset low
  inc si                       ;increment PC
  MemReadPH                    ;Get branch offset high
  inc si                       ;increment PC
  add si,ax                    ;execute branch
ENDM
aPUSH8 MACRO                 ;Push 8-bit value from dl
  LOCAL aPUSH81,aPUSH82        ;Define local labels
  mov ebx,DWORD PTR [eSL]      ;get stack pointer
  MemWriteS                    ;Write byte to stack
  test [ePe],0                 ;check for emulation mode
  je aPUSH81                   ;
  dec bl                       ;emulation mode stack update
  jmp SHORT aPUSH82            ;
  aPUSH81:                     ;
  dec bx                       ;native mode stack update
  mov DWORD PTR [eSL],ebx      ;store updated stack pointer
  aPUSH82:                      
ENDM         
aPULL8 MACRO                 ;Pull an 8-bit value from the stack into dl
  LOCAL aPULL81,aPULL82        ;Define local variables
  mov ebx,DWORD PTR [eSL]      ;get stack pointer
  test [ePe],0                 ;check for emulation mode
  je aPULL81                   ;
  inc bl                       ;emulation mode stack update
  jmp SHORT aPULL82            ;
  aPULL81:                     ;
  inc bx                       ;native mode stack update
  mov DWORD PTR [eSL],ebx      ;store updated stack pointer
  aPULL82:                      
  MemReadS                     ;Write byte to stack
ENDM
aPUSH16 MACRO                ;Push a 16-bit value onto the stack from dx
  LOCAL aPUSH161,aPUSH162      ;Define local variables
  mov ebx,DWORD PTR [eSL]      ;get stack pointer
  test [ePe],0                 ;check for emulation mode
  je aPUSH161
  MemWriteSH                   ;Write byte to stack
  dec bl                       ;emulation mode stack update
  MemWriteS                    ;Write byte to stack
  dec bl                       ;emulation mode stack update
  jmp aPUSH162
  aPUSH161:
  MemWriteSH                   ;Write byte to stack
  dec bx                       ;native mode stack update
  MemWriteS                    ;Write byte to stack
  dec bx                       ;native mode stack update
  aPUSH162:
  mov DWORD PTR [eSL],ebx      ;store updated stack pointer
ENDM
aPULL16 MACRO
  LOCAL aPULL161,aPULL162      ;Define local variables
  mov ebx,DWORD PTR [eSL]      ;get stack pointer
  test [ePe],0                 ;check for emulation mode
  je aPULL161
  inc bl                       ;emulation mode stack update
  MemReadS                     ;Write byte to stack
  inc bl                       ;emulation mode stack update
  MemReadSH                    ;Write byte to stack
  jmp aPULL162
  aPULL161:
  inc bx                       ;native mode stack update
  MemReadS                     ;Write byte to stack
  inc bx                       ;native mode stack update
  MemReadSH                    ;Write byte to stack
  aPULL162:
  mov DWORD PTR [eSL],ebx      ;store updated stack pointer
ENDM                            

aIMM8 MACRO
  MemReadP
  inc si
  mov edx,eax
ENDM
aIMM16 MACRO
  MemReadP
  inc si
  MemReadPH
  inc si
  mov edx,eax
ENDM
aREAD8 MACRO
  MemRead
ENDM
aREAD16 MACRO
  MemRead
  inc edi
  MemReadH
  dec edi
ENDM
aWRITE8 MACRO
  MemWrite
ENDM
aWRITE16 MACRO
  MemWrite
  inc edi
  MemWriteH
  dec edi
ENDM

;ͻ
;                        Oprand Emulation                             
;ͼ
;These macros standardize the process of loading and storing oprands in 65816
;opcode emulation.
;
;values are always loaded into edx (16 bit always on A, X, Y, D, PC, S)
;values are always stored from edx (operation size (8/16 is flag dependent))
oGA MACRO                             ;Get Accumulator
  mov edx,DWORD PTR [eAL]               ;
ENDM                                    
oGX MACRO                             ;Get X-index
  mov edx,DWORD PTR [eXL]               ;will be used with Group II opcodes
ENDM
oGY MACRO                             ;Get Y-index
  mov edx,DWORD PTR [eYL]               ;will be used with Group II opcodes
ENDM
oGD MACRO                             ;Get Direct register
  mov edx,DWORD PTR [eDL]               ;
ENDM
oGPC MACRO                            ;Get Program counter (24b)
  mov edx,DWORD PTR [ePCL]              ;
ENDM
oGPB MACRO                            ;Get Program counter (24b)
  movzx edx,BYTE PTR [ePB]              ;
ENDM
oGDB MACRO
  movzx edx,BYTE PTR [eDB]
ENDM
oGS MACRO                             ;Get Stack register
  mov edx,DWORD PTR [eSL]               ;
ENDM
oGP MACRO                             ;Get Flags register
  mov edx,DWORD PTR [eP]                ;
ENDM
oPA MACRO
  mov DWORD PTR [eAL],edx
ENDM
oPDB MACRO
  mov BYTE PTR [eDB],dl
ENDM
oPX MACRO
  mov DWORD PTR [eXL],edx
ENDM
oPY MACRO
  mov DWORD PTR [eYL],edx
ENDM
oPP MACRO
  mov BYTE PTR [eP],dl
ENDM
oPD MACRO
  mov WORD PTR [eDL],dx
ENDM
oPS MACRO
  mov DWORD PTR [eSL],edx
ENDM
oPPC MACRO
  mov WORD PTR [ePCL],dx
ENDM
oSTACK MACRO                          ;reset SH to 01h if in emulation mode
  LOCAL NoResetSTACK
  cmp [ePe],0
  je NoResetSTACK
  mov dh,1
  NoResetSTACK:
ENDM

;ͻ
;                         Flag Emulation                              
;ͼ
;Flag checking is a key problem in 65816 emulation-- there arn't many fast
;ways to do it.  My current method is to use [eP] to store the P register
;as a whole.  An alternative method has each flag in a seperate memory
;location, allowing easy modification but slowing REP and SEP considerably.
;al should be initialized to the existing flags (with the modifying ones
;cleared) prior to the operation creating the flags.
;The operation should be done using ebx
fGNVZC MACRO                        ;Get x86 Sign, Overflow, Zero, and Carry (aka for ADC)
  LOCAL fGNVZC1,fGNVZC2               ;Define local jumps
  jno fGNVZC1                         ;test overflow first
  lea eax,[eax+64]                    ;set overflow bit
  fGNVZC1:                            ;
  jnz fGNVZC2                         ;
  lea eax,[eax+2]                     ;set zero bit
  fGNVZC2:                            ;
  lahf                                ;get x86 flags into ah
  and ah,129                          ;x86 bit positions corrispond to 65816
  or al,ah                            ;bits for negative and carry
  mov BYTE PTR [eP],al                ;store flags
ENDM
fGNZ MACRO                          ;Get x86 Negative and Zero bits
  LOCAL fGNZ1,fGNZ2                   ;Define local jumps
  jnz fGNZ1                           ;
  lea eax,[eax+2]                     ;set zero bit
  fGNZ1:                              ;
  jns fGNZ2
  lea eax,[eax+128]
  fGNZ2:
  mov BYTE PTR [eP],al                ;store flags
ENDM
fGNZC MACRO                         ;Get x86 Negative, Zero, and Carry bits
  LOCAL fGNZC1                        ;Define local jumps
  lahf                                ;get x86 flags into ah
  bt eax,14                           ;test zero flag
  jnc fGNZC1                          ;
  or al,2                             ;set zero bit
  fGNZC1:                             ;
  and ah,129                          ;x86 bit positions corrispond to 65816
  or al,ah                            ;bits for negative and carry
  mov BYTE PTR [eP],al                ;store flags
ENDM
fGZ MACRO                           ;Get x86 Zero bit
  LOCAL fGZ1                          ;Define local jumps
  jnz fGZ1                            ;
  lea eax,[eax+2]                     ;set zero bit
  fGZ1:                               ;
  mov BYTE PTR [eP],al                ;store flags
ENDM

;ͻ
;                         Opcode Macros                               
;ͼ
;These macros encapsulate common 65816 instructions, such as ADC and SBC.
;This eliminates some repetitiveness in the source code, and allows faster
;debugging.
;Arithmetic opcodes load the first oprand into ecx, and the second into edx.
;These macros assume ecx and edx are already loaded with the oprands.
;If the first oprand is implicity the accumulator, it is directly modified.
oADC8 MACRO                         ;Add with Carry, 8bit version
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,61                           ;clear flags that will be modified
  btr eax,0                           ;get current 65816 carry and clear
  adc BYTE PTR [eAL],dl               ;perform addition
  fGNVZC                              ;get resulting flags
ENDM
oADC16 MACRO                        ;Add with Carry, 16bit version
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,61                           ;clear flags that will be modified
  btr eax,0                           ;get current 65816 carry and clear
  adc WORD PTR [eAL],dx               ;perform addition
  fGNVZC                              ;get resulting flags
ENDM
oSBC8 MACRO                         ;Subtract with Carry, 8bit version
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,61                           ;clear flags that will be modified
  btr eax,0                           ;get current 65816 carry and clear
  cmc                                 ;65816 carry is backwards in SBC
  sbb BYTE PTR [eAL],dl               ;perform subtraction
  cmc                                 ;65816 carry is backwards in SBC
  fGNVZC                              ;get resulting flags
ENDM
oSBC16 MACRO                        ;Subtract with Carry, 8bit version
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,61                           ;clear flags that will be modified
  btr eax,0                           ;get current 65816 carry and clear
  cmc                                 ;65816 carry is backwards in SBC
  sbb WORD PTR [eAL],dx               ;perform subtraction
  cmc                                 ;65816 carry is backwards in SBC
  fGNVZC                              ;get resulting flags
ENDM
oAND8 MACRO                         ;Bitwise Logical AND, 8bit version
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  and BYTE PTR [eAL],dl               ;perform AND
  fGNZ                                ;get resulting flags
ENDM
oAND16 MACRO                        ;Bitwise Logical AND, 16bit version
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  and WORD PTR [eAL],dx               ;perform AND
  fGNZ                                ;get resulting flags
ENDM
oEOR8 MACRO                         ;Bitwise Logical XOR, 8bit version
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  xor BYTE PTR [eAL],dl               ;perform XOR
  fGNZ                                ;get resulting flags
ENDM
oEOR16 MACRO                        ;Bitwise Logical XOR, 16bit version
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  xor WORD PTR [eAL],dx               ;perform XOR
  fGNZ                                ;get resulting flags
ENDM
oORA8 MACRO                         ;Bitwise Logical OR, 8bit version
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  and BYTE PTR [eAL],dl               ;perform OR
  fGNZ                                ;get resulting flags
ENDM
oORA16 MACRO                        ;Bitwise Logical OR, 16bit version
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  and WORD PTR [eAL],dx               ;perform OR
  fGNZ                                ;get resulting flags
ENDM
oCMP8 MACRO                         ;Compare Oprands, 8bit version
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,124                          ;clear flags that will be modified
  cmp BYTE PTR [eAL],dl               ;perform compare
  cmc                                 ;65816 carry is backwards in CMP
  fGNZC                               ;get resulting flags
ENDM
oCMP16 MACRO                        ;Compare Oprands, 16bit version
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,124                          ;clear flags that will be modified
  cmp WORD PTR [eAL],dx               ;perform compare
  cmc                                 ;65816 carry is backwards in CMP
  fGNZC                               ;get resulting flags
ENDM
oLDA8 MACRO                         ;Load memory-->A, 8bit version
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  mov BYTE PTR [eAL],dl               ;perform load   
  or dl,dl                            ;set flags (x86 mov doesn't modify them)
  fGNZ                                ;get resulting flags
ENDM
oLDA16 MACRO                        ;Load memory-->A, 16bit version
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  mov WORD PTR [eAL],dx               ;perform load   
  or dx,dx                            ;set flags (x86 mov doesn't modify them)
  fGNZ                                ;get resulting flags
ENDM
oSTA8 MACRO                         ;Store A-->memory, 8bit version
  movzx edx,BYTE PTR [eAL]            ;perform store  
ENDM
oSTA16 MACRO                        ;Store A-->memory, 16bit version
  movzx edx,WORD PTR [eAL]            ;perform store  
ENDM
oASL8 MACRO                         ;Shift left, 8bit version
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,124                          ;clear flags that will be modified
  shl dl,1                            ;perform shift
  fGNZC                               ;get resulting flags
ENDM
oASL16 MACRO                        ;Shift left, 16bit version
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,124                          ;clear flags that will be modified
  shl dx,1                            ;perform shift
  fGNZC                               ;get resulting flags
ENDM
oLSR8 MACRO                         ;Logical Shift Right, 8bit version
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,124                          ;clear flags that will be modified
  shr dl,1                            ;perform shift
  fGNZC                               ;get resulting flags
ENDM
oLSR16 MACRO                        ;Logical Shift Right, 16bit version
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,124                          ;clear flags that will be modified
  shr dx,1                            ;perform shift
  fGNZC                               ;get resulting flags
ENDM
oDEC8 MACRO                         ;Decrement, 8bit version
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  dec dl                              ;perform dec
  fGNZ                                ;get resulting flags
ENDM
oDEC16 MACRO                        ;Decrement, 16bit version
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  dec dx                              ;perform dec
  fGNZ                                ;get resulting flags
ENDM
oINC8 MACRO                         ;Increment, 8bit version
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  inc dl                              ;perform inc
  fGNZ                                ;get resulting flags
ENDM
oINC16 MACRO                        ;Increment, 16bit version
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  inc dx                              ;perform inc
  fGNZ                                ;get resulting flags
ENDM
oROL8 MACRO                         ;Rotates left, 8bit version
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  btr eax,0                           ;get current 65816 carry and clear
  rol dl,1                            ;perform rotate
  fGNZC                               ;get resulting flags
ENDM
oROL16 MACRO                        ;Rotates left, 16bit version
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  btr eax,0                           ;get current 65816 carry and clear
  rol dx,1                            ;perform rotate
  fGNZC                               ;get resulting flags
ENDM
oROR8 MACRO                         ;Rotates right, 8bit version
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  btr eax,0                           ;get current 65816 carry and clear
  ror dl,1                            ;perform rotate
  fGNZC                               ;get resulting flags
ENDM
oROR16 MACRO                        ;Rotates right, 16bit version
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  btr eax,0                           ;get current 65816 carry and clear
  ror dx,1                            ;perform rotate
  fGNZC                               ;get resulting flags
ENDM
oSTX MACRO                          ;Store X or Y register (Px controls width)
  LOCAL oSTX1                         ;Define local label
  MemWrite                            ;Write low byte
  test [eP],ePx                       ;check for 16 bit mode
  jnz oSTX1                           ;
  inc edi                             ;
  MemWriteH                           ;Write High byte
  dec edi                             ;
  oSTX1:                              ;
ENDM


;ͻ
;                        Opcode Emulation                             
;ͼ
;These procedures are jumped into by OpEmu, and emulate the 65816 instruction
;set.  Repetitive Opcodes may have macros;  see the Opcode Macros section
;There are two seperate jump tables;  one is for Pm=1 (8-bit accumulator), the
;other is for Pm=0 (16-bit accumulator)
;
;esi contains the current PC (pointing to first oprand byte) and must not be
;modified except to increment the PC as oprands are used.
;Memory addressing macros preserve ecx and edx, but eax and ebx are scratchpad
;registers.  In most cases, an Opcode below should not use MemRead/MemWrite
;directly, but should call a macro for the needed addressing mode

;OpXX -- 8-bit accumulator and non-accumulator opcodes  (see OnXX below)
Op00:       ;BRK s
  call eBRK                           ;call the BRK routine
  jmp EndOp                           
Op01:       ;ORA (d,x)
  aDXXI
  aREAD8
  oORA8
  jmp EndOp
Op02:       ;COP s
  call eCOP                           ;call the COP routine
  jmp EndOp
Op03:       ;ORA d,s
  aSR
  aREAD8
  oORA8
  jmp EndOp
Op04:       ;TSB d
  aD
  aREAD8
  push edx
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,253                          ;clear flags that will be modified
  and dl,BYTE PTR [eAL]
  fGZ
  pop edx
  or dl,BYTE PTR [eAL]
  aWRITE8
  jmp EndOp
Op05:       ;ORA d
  aD
  aREAD8
  oORA8
  jmp EndOp
Op06:       ;ASL d
  aD
  aREAD8
  oASL8
  aWRITE8
  jmp EndOp
Op07:       ;ORA [d]
  aDIL
  aREAD8
  oORA8
  jmp EndOp
Op08:       ;PHP s
  oGP
  aPUSH8
  jmp EndOp
Op09:       ;ORA #
  aIMM8
  oORA8
  jmp EndOp
Op0A:       ;ASL A
  oGA
  oASL8
  oPA
  jmp EndOp
Op0B:       ;PHD s
  oGD
  aPUSH16
  jmp EndOp
Op0C:       ;TSB a
  aA
  aREAD8
  push edx
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,253                          ;clear flags that will be modified
  and dl,BYTE PTR [eAL]
  fGZ
  pop edx
  or dl,BYTE PTR [eAL]
  aWRITE8
  jmp EndOp
Op0D:       ;ORA a
  aA
  aREAD8
  oORA8
  jmp EndOp
Op0E:       ;ASL a
  aA
  aREAD8
  oASL8
  aWRITE8
  jmp EndOp
Op0F:       ;ORA al
  aAL
  aREAD8
  oORA8
  jmp EndOp
Op10:       ;BPL r
  test [eP],ePn
  jnz NoBPL
  aBRA
  jmp EndOp
  NoBPL:
  aNBRA
  jmp EndOp
Op11:       ;ORA [d],y
  aDILXY
  aREAD8
  oORA8
  jmp EndOp
Op12:       ;ORA (d)     
  aDI
  aREAD8
  oORA8
  jmp EndOp
Op13:       ;ORA (d,s),y
  aSRIXY
  aREAD8
  oORA8
  jmp EndOp
Op14:       ;TRB d
  aD
  aREAD8
  push edx
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,253                          ;clear flags that will be modified
  and dl,BYTE PTR [eAL]
  fGZ
  pop edx
  mov al,BYTE PTR [eAL]
  not al
  and dl,al
  aWRITE8
  jmp EndOp
Op15:       ;ORA d,x
  aDXX
  aREAD8
  oORA8
  jmp EndOp
Op16:       ;ASL d,x  
  aDXX
  aREAD8
  oASL8
  aWRITE8
  jmp EndOp
Op17:       ;ORA [d],y
  aDILXY
  aREAD8
  oORA8
  jmp EndOp
Op18:       ;CLC i
  and [eP],254
  jmp EndOp
Op19:       ;ORA a,y
  aAXY
  aREAD8
  oORA8
  jmp EndOp
Op1A:       ;INC A
  oGA
  oINC8
  oPA
  jmp EndOp
Op1B:       ;TCS i
  oGA
  oSTACK
  mov DWORD PTR [eSL],edx
  jmp EndOp
Op1C:       ;TRB a
  aA
  aREAD8
  push edx
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,253                          ;clear flags that will be modified
  and dl,BYTE PTR [eAL]
  fGZ
  pop edx
  mov al,BYTE PTR [eAL]
  not al
  and dl,al
  aWRITE8
  jmp EndOp
Op1D:       ;ORA a,x
  aAXX
  aREAD8
  oORA8
  jmp EndOp
Op1E:       ;ASL a,x
  aAXX
  aREAD8
  oASL8
  aWRITE8
  jmp EndOp
Op1F:       ;ORA al,x
  aALXX
  aREAD8
  oORA8
  jmp EndOp
Op20:       ;JSR a
  MemReadP                     ;Get AAL
  inc si                       ;increment PC
  MemReadPH                    ;Get AAH into ah
  mov edx,esi
  aPUSH16
  mov si,ax
  jmp EndOp
Op21:       ;AND (d,x)
  aDXXI
  aREAD8
  oAND8
  jmp EndOp
Op22:       ;JSL al
  MemReadP                     ;Get AAL
  inc si                       ;increment PC
  MemReadPH                    ;Get AAH into ah
  inc si
  MemReadPB
  mov edx,esi
  shr edx,16
  aPUSH8
  mov edx,esi
  aPUSH16
  mov esi,eax
  jmp EndOp
Op23:       ;AND d,s
  aSR
  aREAD8
  oAND8
  jmp EndOp
Op24:       ;BIT d
  aD
  aREAD8
  push edx
  mov al,BYTE PTR [eP]
  and al,63
  and dl,192
  or al,dl
  pop edx
  and al,253                          ;clear flags that will be modified
  and dl,BYTE PTR [eAL]
  fGZ
  jmp EndOp
Op25:       ;AND d
  aD
  aREAD8
  oAND8
  jmp EndOp
Op26:       ;ROL d
  aD
  aREAD8
  oROL8
  aWRITE8
  jmp EndOp
Op27:       ;AND [d]
  aDIL
  aREAD8
  oAND8
  jmp EndOp
Op28:       ;PLP s
  aPULL8
  oPP
  test [eP],ePx
  jz Op28S
  mov BYTE PTR [eXH],0
  mov BYTE PTR [eYH],0
  Op28S:
  jmp EndOp
Op29:       ;AND #
  aIMM8
  oAND8
  jmp EndOp
Op2A:       ;ROL A
  oGA
  oROL8
  oPA
  jmp EndOp
Op2B:       ;PLD s
  aPULL16
  oPD
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dx,dx
  fGNZ
  jmp EndOp
Op2C:       ;BIT a
  aA
  aREAD8
  push edx
  mov al,BYTE PTR [eP]
  and al,63
  and dl,192
  or al,dl
  pop edx
  and al,253                          ;clear flags that will be modified
  and dl,BYTE PTR [eAL]
  fGZ
  jmp EndOp
Op2D:       ;AND a
  aA
  aREAD8
  oAND8
  jmp EndOp
Op2E:       ;ROL a
  aA
  aREAD8
  oROL8
  aWRITE8
  jmp EndOp
Op2F:       ;AND al
  aAL
  aREAD8
  oAND8
  jmp EndOp
Op30:       ;BMI r
  test [eP],ePn
  jz NoBMI
  aBRA
  jmp EndOp
  NoBMI:
  aNBRA
  jmp EndOp
Op31:       ;AND (d),y
  aDILXY
  aREAD8
  oAND8
  jmp EndOp
Op32:       ;AND (d)
  aDI
  aREAD8
  oAND8
  jmp EndOp
Op33:       ;AND (d,s),y
  aSRIXY
  aREAD8
  oAND8
  jmp EndOp
Op34:       ;BIT d,x
  aDXX
  aREAD8
  push edx
  mov al,BYTE PTR [eP]
  and al,63
  and dl,192
  or al,dl
  pop edx
  and al,253                          ;clear flags that will be modified
  and dl,BYTE PTR [eAL]
  fGZ
  jmp EndOp
Op35:       ;AND d,x
  aDXX
  aREAD8
  oAND8
  jmp EndOp
Op36:       ;ROL d,x
  aDXX
  aREAD8
  oROL8
  aWRITE8
  jmp EndOp
Op37:       ;AND [d],y
  aDILXY
  aREAD8
  oAND8
  jmp EndOp
Op38:       ;SEC i
  or [eP],ePc
  jmp EndOp
Op39:       ;AND a,y
  aAXY
  aREAD8
  oAND8
  jmp EndOp
Op3A:       ;DEC A
  oGA
  oDEC8
  oPA
  jmp EndOp
Op3B:       ;TSC i
  oGS
  oPA
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dx,dx
  fGNZ
  jmp EndOp
Op3C:       ;BIT a,x
  aAXX
  aREAD8
  push edx
  mov al,BYTE PTR [eP]
  and al,63
  and dl,192
  or al,dl
  pop edx
  and al,253                          ;clear flags that will be modified
  and dl,BYTE PTR [eAL]
  fGZ
  jmp EndOp
Op3D:       ;AND a,x
  aAXX
  aREAD8
  oAND8
  jmp EndOp
Op3E:       ;ROL a,x
  aAXX
  aREAD8
  oROL8
  aWRITE8
  jmp EndOp
Op3F:       ;AND al,x
  aALXX
  aREAD8
  oAND8
  jmp EndOp
Op40:       ;RTI s
  aPULL8
  oPP
  test [eP],ePx
  jz Op40S
  mov BYTE PTR [eXH],0
  mov BYTE PTR [eYH],0
  Op40S:
  aPULL16
  mov esi,edx
  cmp [ePe],0
  jne Op40S2
  aPULL8
  shl edx,16
  or esi,edx
  Op40S2:
  jmp EndOp
Op41:       ;EOR (d,x)
  aDXXI
  aREAD8
  oEOR8
  jmp EndOp
Op42:       ;WDM
  jmp EndOp
Op43:       ;EOR d,s
  aSR
  aREAD8
  oEOR8
  jmp EndOp
Op44:       ;MVP xyc
  jmp EndOp
Op45:       ;EOR d
  aD
  aREAD8
  oEOR8
  jmp EndOp
Op46:       ;LSR d
  aD
  aREAD8
  oLSR8
  aWRITE8
  jmp EndOp
Op47:       ;EOR [d]
  aDIL
  aREAD8
  oEOR8
  jmp EndOp
Op48:       ;PHA s
  oGA
  aPUSH8
  jmp EndOp
Op49:       ;EOR #
  aIMM8
  oEOR8
  jmp EndOp    
Op4A:       ;LSR A
  oGA
  oASL8
  oPA
  jmp EndOp
Op4B:       ;PHK s
  oGPB
  aPUSH8
  jmp EndOp
Op4C:       ;JMP a
  MemReadP                     ;Get AAL
  inc si                       ;increment PC
  MemReadPH                    ;Get AAH into ah
  mov si,ax
  jmp EndOp
Op4D:       ;EOR a
  aA
  aREAD8
  oEOR8
  jmp EndOp
Op4E:       ;LSR a
  aA
  aREAD8
  oLSR8
  aWRITE8
  jmp EndOp
Op4F:       ;EOR al
  aAL
  aREAD8
  oEOR8
  jmp EndOp
Op50:       ;BVC r
  test [eP],ePv
  jnz NoBVC
  aBRA
  jmp EndOp
  NoBVC:
  aNBRA
  jmp EndOp
Op51:       ;EOR (d),y
  aDILXY
  aREAD8
  oEOR8
  jmp EndOp
Op52:       ;EOR (d)
  aDI
  aREAD8
  oEOR8
  jmp EndOp
Op53:       ;EOR (d,s),y
  aSRIXY
  aREAD8
  oEOR8
  jmp EndOp
Op54:       ;MVN xyc
  jmp EndOp   
Op55:       ;EOR d,x
  aDXX
  aREAD8
  oEOR8
  jmp EndOp
Op56:       ;LSR d,x
  aDXX
  aREAD8
  oLSR8
  aWRITE8
  jmp EndOp
Op57:       ;EOR [d],y
  aDILXY
  aREAD8
  oEOR8
  jmp EndOp
Op58:       ;CLI i
  and [eP],251
  jmp EndOp
Op59:       ;EOR a,y
  aAXY
  aREAD8
  oEOR8
  jmp EndOp
Op5A:       ;PHY s
  oGY
  test [eP],ePx
  jnz Op5AS
  aPUSH16
  jmp EndOp
  Op5AS:
  aPUSH8
  jmp EndOp
Op5B:       ;TCD i
  oGA
  oPD
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dx,dx
  fGNZ
  jmp EndOp
Op5C:       ;JMP al
  MemReadP                     ;Get AAL
  inc si                       ;increment PC
  MemReadPH                    ;Get AAH into ah
  inc si
  MemReadPB
  mov esi,eax
  jmp EndOp
Op5D:       ;EOR a,x
  aAXX
  aREAD8
  oEOR8
  jmp EndOp
Op5E:       ;LSR a,x
  aAXX
  aREAD8
  oLSR8
  aWRITE8
  jmp EndOp
Op5F:       ;EOR al,x
  aALXX
  aREAD8
  oEOR8
  jmp EndOp
Op60:       ;RTS s
  aPULL16
  mov esi,edx
  inc si
  jmp EndOp
Op61:       ;ADC (d,x)
  aDXXI
  aREAD8     
  oADC8
  jmp EndOp
Op62:       ;PER s
  aIMM16
  oGPC
  add dx,ax
  oPPC
  jmp EndOp
Op63:       ;ADC d,s
  aSR
  aREAD8
  oADC8
  jmp EndOp
Op64:       ;STZ d
  aD
  xor edx,edx
  aWRITE8
  jmp EndOp
Op65:       ;ADC d
  aD
  aREAD8
  oADC8
  jmp EndOp
Op66:       ;ROR d
  aD
  aREAD8
  oROR8
  aWRITE8
  jmp EndOp
Op67:       ;ADC [d]
  aDIL
  aREAD8
  oADC8
  jmp EndOp
Op68:       ;PLA s
  oGA
  aPULL8
  oPA
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dl,dl
  fGNZ
  jmp EndOp
Op69:       ;ADC #
  aIMM8
  oADC8
  jmp EndOp
Op6A:       ;ROR A
  oGA
  oROR8
  oPA
  jmp EndOp
Op6B:       ;RTL s
  aPULL16
  mov esi,edx
  inc si
  aPULL8
  shl edx,16
  or esi,edx
  jmp EndOp
Op6C:       ;JMP (a)
  MemReadP                     ;Get IAL
  inc si                       ;increment PC
  MemReadPH                    ;Get IAH into ah
  mov esi,eax                  ;setup for AA
  MemReadP                     ;Get AAL
  inc si                       ;increment program counter
  MemReadPH                    ;Get AAH
  mov si,ax                    ;update PC
  jmp EndOp
Op6D:       ;ADC a
  aA
  aREAD8
  oADC8
  jmp EndOp
Op6E:       ;ROR a
  aA
  aREAD8
  oROR8
  aWRITE8
  jmp EndOp
Op6F:       ;ADC al,x
  aAL
  aREAD8
  oADC8
  jmp EndOp
Op70:       ;BVS r
  test [eP],ePv
  jz NoBVS
  aBRA
  jmp EndOp
  NoBVS:
  aNBRA
  jmp EndOp
Op71:       ;ADC (d),y
  aDILXY
  aREAD8
  oADC8
  jmp EndOp
Op72:       ;ADC (d)
  aDI
  aREAD8
  oADC8
  jmp EndOp
Op73:       ;ADC (d,s),y
  aSRIXY
  aREAD8
  oADC8
  jmp EndOp
Op74:       ;STZ d,x
  aDXX
  xor edx,edx
  aWRITE8
  jmp EndOp
Op75:       ;ADC d,x
  aDXX
  aREAD8
  oADC8
  jmp EndOp
Op76:       ;ROR d,x
  aDXX
  aREAD8
  oROR8
  aWRITE8
  jmp EndOp
Op77:       ;ADC [d],y
  aDILXY
  aREAD8
  oADC8
  jmp EndOp
Op78:       ;SEI i
  or [eP],ePi
  jmp EndOp
Op79:       ;ADC a,y
  aAXY
  aREAD8
  oADC8
  jmp EndOp
Op7A:       ;PLY s
  oGY
  test [eP],ePx
  jnz Op7AS
  aPULL16
  oPY
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dl,dl
  fGNZ
  jmp EndOp
  Op7AS:
  aPULL8
  oPY
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dx,dx
  fGNZ
  jmp EndOp
Op7B:       ;TDC i
  oGD
  oPA
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dx,dx
  fGNZ
  jmp EndOp
Op7C:       ;JMP (a,x)
  aAXI
  jmp EndOp
Op7D:       ;ADC a,x
  aAXX
  aREAD8
  oADC8
  jmp EndOp
Op7E:       ;ROR a,x
  aAXX
  aREAD8
  oROR8
  aWRITE8
  jmp EndOp
Op7F:       ;ADC al,x
  aALXX
  aREAD8
  oADC8
  jmp EndOp
Op80:       ;BRA r
  aBRA
  jmp EndOp
Op81:       ;STA (d,x)
  aDXXI
  oSTA8
  aWRITE8
  jmp EndOp
Op82:       ;BRL rl
  aBRL
  jmp EndOp
Op83:       ;STA d,s
  aSR
  oSTA8
  aWRITE8
  jmp EndOp
Op84:       ;STY d
  aD
  oGY
  oSTX
  jmp EndOp
Op85:       ;STA d
  aD
  oSTA8
  aWRITE8
  jmp EndOp
Op86:       ;STX d
  aD
  oGX
  oSTX
  jmp EndOp
Op87:       ;STA [d]
  aDIL
  oSTA8
  aWRITE8
  jmp EndOp
Op88:       ;DEY i
  test [eP],ePx
  jnz Op88S
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  dec WORD PTR [eYL]
  fGNZ
  jmp EndOp
  Op88S:
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  dec BYTE PTR [eYL]
  fGNZ
  jmp EndOp
Op89:       ;BIT #
  aIMM8
  push edx
  mov al,BYTE PTR [eP]
  and al,63
  and dl,192
  or al,dl
  pop edx
  and al,253                          ;clear flags that will be modified
  and dl,BYTE PTR [eAL]
  fGZ
  jmp EndOp
Op8A:       ;TXA i
  oGX
  mov BYTE PTR [eAL],dl
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dl,dl
  fGNZ
  jmp EndOp
Op8B:       ;PHB s
  oGDB
  aPUSH8
  jmp EndOp
Op8C:       ;STY a
  aA
  oGY
  oSTX
  jmp EndOp
Op8D:       ;STA a
  aA
  oSTA8
  aWRITE8
  jmp EndOp
Op8E:       ;STX a
  aA
  oGX
  oSTX
  jmp EndOp
Op8F:       ;STA al
  aAL
  oSTA8
  aWRITE8
  jmp EndOp
Op90:       ;BCC r
  test [eP],ePc
  jnz NoBCC
  aBRA
  jmp EndOp
  NoBCC:
  aNBRA
  jmp EndOp
Op91:       ;STA (d),y
  aDILXY
  oSTA8
  aWRITE8
  jmp EndOp
Op92:       ;STA (d)
  aDI
  oSTA8
  aWRITE8
  jmp EndOp
Op93:       ;STA (d,s),y
  aSRIXY
  oSTA8
  aWRITE8
  jmp EndOp
Op94:       ;STY d,x
  aDXX
  oGY
  oSTX
  jmp EndOp
Op95:       ;STA d,x
  aDXX
  oSTA8
  aWRITE8
  jmp EndOp
Op96:       ;STX d,y
  aDXY
  oGX
  oSTX
  jmp EndOp
Op97:       ;STA [d],y
  aDILXY
  oSTA8
  aWRITE8
  jmp EndOp
Op98:       ;TYA i
  oGY
  mov BYTE PTR [eAL],dl
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dl,dl
  fGNZ
  jmp EndOp
Op99:       ;STA a,y
  aAXY
  oSTA8
  aWRITE8
  jmp EndOp
Op9A:       ;TXS i
  cmp [ePe],0
  je Op9AS
  oGX
  mov BYTE PTR [eSL],dl
  jmp EndOp
  Op9AS:
  oGX
  oPS
  jmp EndOp
Op9B:       ;TXY i
  mov eax,DWORD PTR [eXL]
  mov DWORD PTR [eYL],eax
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or eax,eax
  fGNZ
  jmp EndOp
Op9C:       ;STZ a
  aA
  xor edx,edx
  aWRITE8
  jmp EndOp
Op9D:       ;STA a,x
  aAXX
  oSTA8
  aWRITE8
  jmp EndOp
Op9E:       ;STZ a,x
  aAXX
  xor edx,edx
  aWRITE8
  jmp EndOp
Op9F:       ;STA al,x
  aALXX
  oSTA8
  aWRITE8
  jmp EndOp
OpA0:       ;LDY #
  test [eP],ePx
  jz OpA0S
  aIMM8
  oPY
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dl,dl
  fGNZ
  jmp EndOp
  OpA0S:
  aIMM16
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dx,dx
  oPY
  fGNZ
  jmp EndOp
OpA1:       ;LDA (d,x)
  aDXXI
  aREAD8     
  oLDA8
  jmp EndOp
OpA2:       ;LDX #
  test [eP],ePx
  jz OpA2S
  aIMM8
  oPX
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dl,dl
  fGNZ
  jmp EndOp
  OpA2S:
  aIMM16
  oPX
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dx,dx
  fGNZ
  jmp EndOp
OpA3:       ;LDA d,s
  aSR
  aREAD8
  oLDA8
  jmp EndOp
OpA4:       ;LDY d
  aD
  test [eP],ePx
  jz OpA4S
  aREAD8
  oPY
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dl,dl
  fGNZ
  jmp EndOp
  OpA4S:
  aREAD16
  oPY
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dx,dx
  fGNZ
  jmp EndOp
OpA5:       ;LDA d
  aD
  aREAD8
  oLDA8
  jmp EndOp
OpA6:       ;LDX d
  aD
  test [eP],ePx
  jz OpA6S
  aREAD8
  oPX
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dl,dl
  fGNZ
  jmp EndOp
  OpA6S:
  aREAD16
  oPX
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dx,dx
  fGNZ
  jmp EndOp
OpA7:       ;LDA [d]
  aDIL
  aREAD8
  oLDA8
  jmp EndOp
OpA8:       ;TAY i
  test [eP],ePx
  jz OpA8S
  mov dl,BYTE PTR [eAL]
  mov BYTE PTR [eYL],dl
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dl,dl
  fGNZ
  jmp EndOp
  OpA8S:
  oGA
  oPY
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dx,dx
  fGNZ
  jmp EndOp
OpA9:       ;LDA #
  aIMM8
  oLDA8
  jmp EndOp
OpAA:       ;TAX i
  test [eP],ePx
  jz OpAAS
  mov dl,BYTE PTR [eAL]
  mov BYTE PTR [eXL],dl
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dl,dl
  fGNZ
  jmp EndOp
  OpAAS:
  oGA
  oPX
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dx,dx
  fGNZ
  jmp EndOp
OpAB:       ;PLB s
  aPULL8
  oPDB
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dl,dl
  fGNZ
  jmp EndOp
OpAC:       ;LDY a
  aA
  test [eP],ePx
  jz OpACS
  aREAD8
  oPY
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dl,dl
  fGNZ
  jmp EndOp
  OpACS:
  aREAD16
  oPY
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dx,dx
  fGNZ
  jmp EndOp
OpAD:       ;LDA a
  aA
  aREAD8
  oLDA8
  jmp EndOp
OpAE:       ;LDX a
  aA
  test [eP],ePx
  jz OpAES
  aREAD8
  oPX
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dl,dl
  fGNZ
  jmp EndOp
  OpAES:
  aREAD16
  oPX
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dx,dx
  fGNZ
  jmp EndOp
OpAF:       ;LDA al
  aAL
  aREAD8
  oLDA8
  jmp EndOp
OpB0:       ;BCS r
  test [eP],ePc
  jz NoBCS
  aBRA
  jmp EndOp
  NoBCS:
  aNBRA
  jmp EndOp
OpB1:       ;LDA (d),y
  aDILXY
  aREAD8
  oLDA8
  jmp EndOp
OpB2:       ;LDA (d)
  aDI
  aREAD8
  oLDA8
  jmp EndOp
OpB3:       ;LDY (d,s),y
  aSRIXY
  aREAD8
  oLDA8
  jmp EndOp
OpB4:       ;LDY d,x
  aDXX
  test [eP],ePx
  jz OpB4S
  aREAD8
  oPY
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dl,dl
  fGNZ
  jmp EndOp
  OpB4S:
  aREAD16
  oPY
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dx,dx
  fGNZ
  jmp EndOp
OpB5:       ;LDA d,x
  aDXX
  aREAD8
  oLDA8
  jmp EndOp
OpB6:       ;LDX d,y
  aDXY
  test [eP],ePx
  jz OpB6S
  aREAD8
  oPX
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dl,dl
  fGNZ
  jmp EndOp
  OpB6S:
  aREAD16
  oPX
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dx,dx
  fGNZ
  jmp EndOp
OpB7:       ;LDA [d],y
  aDILXY
  aREAD8
  oLDA8
  jmp EndOp
OpB8:       ;CLV i
  and [eP],255-128
  jmp EndOp
OpB9:       ;LDA a,y
  aAXY
  aREAD8
  oLDA8
  jmp EndOp
OpBA:       ;TSX i
  test [eP],ePx
  jz OpBAS
  oGS
  mov BYTE PTR [eXL],dl
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dl,dl
  fGNZ
  jmp EndOp
  OpBAS:
  oGS
  oPX
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dx,dx
  fGNZ
  jmp EndOp
OpBB:       ;TYX i
  oGY
  oPX
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or edx,edx
  fGNZ
  jmp EndOp
OpBC:       ;LDY a,x
  aAXX
  test [eP],ePx
  jz OpBCS
  aREAD8
  oPY
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dl,dl
  fGNZ
  jmp EndOp
  OpBCS:
  aREAD16
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dx,dx
  oPY
  fGNZ
  jmp EndOp
OpBD:       ;LDA a,x
  aAXX
  aREAD8
  oLDA8
  jmp EndOp
OpBE:       ;LDX a,y
  aAXY
  test [eP],ePx
  jz OpBES
  aREAD8
  oPX
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dl,dl
  fGNZ
  jmp EndOp
  OpBES:
  aREAD16
  oPX
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dx,dx
  fGNZ
  jmp EndOp
OpBF:       ;LDA al,x
  aALXX
  aREAD8
  oLDA8
  jmp EndOp
OpC0:       ;CPY #
  test [eP],ePx
  jz OpC0S
  aIMM8
  mov ecx,edx
  oGY
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,124                          ;clear flags that will be modified
  sub cl,dl
  cmc
  fGNZC
  jmp EndOp
  OpC0S:
  aIMM16
  mov ecx,edx
  oGY
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,124                          ;clear flags that will be modified
  sub cx,dx
  cmc
  fGNZC
  jmp EndOp
OpC1:       ;CMP (d,x)
  aDXXI
  aREAD8
  oCMP8
  jmp EndOp
OpC2:       ;REP #
  aIMM8
  cmp [ePe],0
  je OpC2S
  and dl,207
  OpC2S:
  not dl
  and BYTE PTR [eP],dl
  jmp EndOp
OpC3:       ;CMP d,s
  aSR
  aREAD8
  oCMP8
  jmp EndOp
OpC4:       ;CPY d
  aD
  test [eP],ePx
  jz OpC4S
  aREAD8
  mov ecx,edx
  oGY
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,124                          ;clear flags that will be modified
  sub cl,dl
  cmc
  fGNZC
  jmp EndOp
  OpC4S:
  aREAD16
  mov ecx,edx
  oGY
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,124                          ;clear flags that will be modified
  sub cx,dx
  cmc
  fGNZC
  jmp EndOp
OpC5:       ;CMP d
  aD
  aREAD8
  oCMP8
  jmp EndOp
OpC6:       ;DEC d
  aD
  aREAD8
  oDEC8
  aWRITE8
  jmp EndOp
OpC7:       ;CMP [d]
  aDIL
  aREAD8
  oCMP8
  jmp EndOp
OpC8:       ;INY i
  test [eP],ePx
  jnz OpC8S
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  inc WORD PTR [eYL]
  fGNZ
  jmp EndOp
  OpC8S:
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  inc BYTE PTR [eYL]
  fGNZ
  jmp EndOp
OpC9:       ;CMP #
  aIMM8
  oCMP8
  jmp EndOp
OpCA:       ;DEX i
  test [eP],ePx
  jnz OpCAS
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  dec WORD PTR [eXL]
  fGNZ
  jmp EndOp
  OpCAS:
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  dec BYTE PTR [eXL]
  fGNZ
  jmp EndOp
OpCB:       ;WAI i
  mov [eWAIT],1
  dec WORD PTR [ePCL]
  jmp EndOp
OpCC:       ;CPY a
  aA
  test [eP],ePx
  jz OpCCS
  aREAD8
  mov ecx,edx
  oGY
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,124                          ;clear flags that will be modified
  sub cl,dl
  cmc
  fGNZC
  jmp EndOp
  OpCCS:
  aREAD16
  mov ecx,edx
  oGY
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,124                          ;clear flags that will be modified
  sub cx,dx
  cmc
  fGNZC
  jmp EndOp
OpCD:       ;CMP a
  aA
  aREAD8
  oCMP8
  jmp EndOp
OpCE:       ;DEC a
  aA
  aREAD8
  oDEC8
  aWRITE8
  jmp EndOp
OpCF:       ;CMP al
  aAL
  aREAD8
  oCMP8
  jmp EndOp
iOpD0:       ;BNE r       Note renamed to avoid symbol conflict
  test [eP],ePz
  jnz NoBNE
  aBRA
  jmp EndOp
  NoBNE:
  aNBRA
  jmp EndOp
OpD1:       ;CMP (d),y
  aDILXY
  aREAD8
  oCMP8
  jmp EndOp
OpD2:       ;CMP (d)
  aDI
  aREAD8
  oCMP8
  jmp EndOp
OpD3:       ;CMP (d,s),y
  aSRIXY
  aREAD8
  oCMP8
  jmp EndOp
OpD4:       ;PEI s
  aDI
  mov edx,eax
  aPUSH16
  jmp EndOp
OpD5:       ;CMP d,x
  aDXX
  aREAD8
  oCMP8
  jmp EndOp
OpD6:       ;DEC d,x
  aDXX
  aREAD8
  oDEC8
  aWRITE8
  jmp EndOp
OpD7:       ;CMP [d],y
  aDILXY
  aREAD8
  oCMP8
  jmp EndOp
OpD8:       ;CLD i
  and [eP],255-8
  jmp EndOp
OpD9:       ;CMP a,y
  aAXY
  aREAD8
  oCMP8
  jmp EndOp
OpDA:       ;PHX s
  oGX
  test [eP],ePx
  jnz OpDAS
  aPUSH16
  jmp EndOp
  OpDAS:
  aPUSH8
  jmp EndOp
iOpDB:       ;STP i
  dec WORD PTR [ePCL]
  jmp EndOp
OpDC:       ;JML (a)
  MemReadP                     ;Get IAL
  inc si                       ;increment PC
  MemReadPH                    ;Get IAH into ah
  mov esi,eax                  ;setup for AA
  MemReadP                     ;Get AAL
  inc si                       ;increment program counter
  MemReadPH                    ;Get AAH
  inc si
  MemReadPB
  mov esi,eax                  ;update PC
  jmp EndOp
OpDD:       ;CMP a,x
  aAXX
  aREAD8
  oCMP8
  jmp EndOp
OpDE:       ;DEC a,x
  aAXX
  aREAD8
  oDEC8
  aWRITE8
  jmp EndOp
OpDF:       ;CMP al,x
  aALXX
  aREAD8
  oCMP8
  jmp EndOp
OpE0:       ;CPX #
  test [eP],ePx
  jz OpE0S
  aIMM8
  mov ecx,edx
  oGX
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,124                          ;clear flags that will be modified
  sub cl,dl
  cmc
  fGNZC
  jmp EndOp
  OpE0S:
  aIMM16
  mov ecx,edx
  oGX
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,124                          ;clear flags that will be modified
  sub cx,dx
  cmc
  fGNZC
  jmp EndOp
OpE1:       ;SBC (d,x)
  aDXXI
  aREAD8     
  oSBC8
  jmp EndOp
OpE2:       ;SEP #
  aIMM8
  cmp [ePe],0
  je OpE2S
  and dl,207
  OpE2S:
  or BYTE PTR [eP],dl
  test [eP],ePx
  jz OpE2S2
  mov BYTE PTR [eXH],0
  mov BYTE PTR [eYH],0
  OpE2S2:
  jmp EndOp
OpE3:       ;SBC d,s
  aSR
  aREAD8
  oSBC8
  jmp EndOp
OpE4:       ;CPX d
  aD
  test [eP],ePx
  jz OpE4S
  aREAD8
  mov ecx,edx
  oGX
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,124                          ;clear flags that will be modified
  sub cl,dl
  cmc
  fGNZC
  jmp EndOp
  OpE4S:
  aREAD16
  mov ecx,edx
  oGX
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,124                          ;clear flags that will be modified
  sub dx,cx
  cmc
  fGNZC
  jmp EndOp
OpE5:       ;SBC d
  aD
  aREAD8
  oSBC8
  jmp EndOp
OpE6:       ;INC d
  aD
  aREAD8
  oINC8
  aWRITE8
  jmp EndOp
OpE7:       ;SBC [d]
  aDIL
  aREAD8
  oSBC8
  jmp EndOp
OpE8:       ;INX i
  test [eP],ePx
  jnz OpE8S
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  inc WORD PTR [eXL]
  fGNZ
  jmp EndOp
  OpE8S:
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  inc BYTE PTR [eXL]
  fGNZ
  jmp EndOp
OpE9:       ;SBC #
  aIMM8
  oSBC8
  jmp EndOp
OpEA:       ;NOP i
  jmp EndOp
OpEB:       ;XBA i
  mov ax,WORD PTR [eAL]
  xchg al,ah
  mov WORD PTR [eAL],ax
  jmp EndOp
OpEC:       ;CPX a
  aA
  test [eP],ePx
  jz OpECS
  aREAD8
  mov ecx,edx
  oGX
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,124                          ;clear flags that will be modified
  sub cl,dl
  cmc
  fGNZC
  jmp EndOp
  OpECS:
  aREAD16
  mov ecx,edx
  oGX
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,124                          ;clear flags that will be modified
  sub cx,dx
  cmc
  fGNZC
  jmp EndOp
OpED:       ;SBC a
  aA
  aREAD8
  oSBC8
  jmp EndOp
OpEE:       ;INC a
  aA
  aREAD8
  oINC8
  aWRITE8
  jmp EndOp
OpEF:       ;SBC al
  aAL
  aREAD8
  oSBC8
  jmp EndOp
OpF0:       ;BEQ r
  test [eP],ePz
  jz NoBEQ
  aBRA
  jmp EndOp
  NoBEQ:
  aNBRA
  jmp EndOp
OpF1:       ;SBC (d),y
  aDILXY
  aREAD8
  oSBC8
  jmp EndOp
OpF2:       ;SBC (d)
  aDI
  aREAD8
  oSBC8
  jmp EndOp
OpF3:       ;SBC (d,s),y
  aSRIXY
  aREAD8
  oSBC8
  jmp EndOp
OpF4:       ;PEA s
  aIMM16
  aPUSH16
  jmp EndOp
OpF5:       ;SBC d,x
  aDXX
  aREAD8
  oSBC8
  jmp EndOp
OpF6:       ;INC d,x
  aDXX
  aREAD8
  oINC8
  aWRITE8
  jmp EndOp
OpF7:       ;SBC [d],y
  aDILXY
  aREAD8
  oSBC8
  jmp EndOp
OpF8:       ;SED i
  or [eP],ePd
  jmp EndOp
OpF9:       ;SBC a,y
  aAXY
  aREAD8
  oSBC8
  jmp EndOp
OpFA:       ;PLX s
  oGX
  test [eP],ePx
  jnz OpFAS
  aPULL16
  oPX
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dl,dl
  fGNZ
  jmp EndOp
  OpFAS:
  aPULL8
  oPX
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dx,dx
  fGNZ
  jmp EndOp
OpFB:       ;XCE i
  mov al,[eP]
  mov dl,al
  and al,254
  or al,[ePe]
  and dl,1
  mov [ePe],dl
  cmp [ePe],0
  jne OpFBS
  or al,48
  mov BYTE PTR [eXH],0
  mov BYTE PTR [eYH],0
  mov BYTE PTR [eSH],1
  jmp EndOp
  OpFBS:
  or al,48
  jmp EndOp
OpFC:       ;JSR (a,x)
  MemReadP                     ;Get IAL
  inc si                       ;increment PC
  MemReadPH                    ;Get IAH into ah
  mov edx,esi
  aPUSH16
  add eax,DWORD PTR [eXL]      ;add index X
  mov ebx,DWORD PTR [ePCL]     ;
  xor bx,bx                    ;
  or eax,ebx                   ;cancate PB:AA
  mov esi,eax                  ;setup for AA
  MemReadP                     ;Get AAL
  inc si                       ;increment program counter
  MemReadPH                    ;Get AAH
  mov si,ax                    ;Update PC
  jmp EndOp
OpFD:       ;SBC a,x
  aAXX
  aREAD8
  oSBC8
  jmp EndOp
OpFE:       ;INC a,x
  aAXX
  aREAD8
  oINC8
  aWRITE8
  jmp EndOp
OpFF:       ;SBC al,x
  aALXX
  aREAD8
  oSBC8
  jmp EndOp

;OnXX -- 16-bit accumulator opcodes  (see OpXX above)
On01:       ;ORA (d,x)
  aDXXI
  aREAD16
  oORA16
  jmp EndOp
On03:       ;ORA d,s
  aSR
  aREAD16
  oORA16
  jmp EndOp
On04:       ;TSB d
  aD
  aREAD16
  push edx
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,253                          ;clear flags that will be modified
  and dx,WORD PTR [eAL]
  fGZ
  pop edx
  or dx,WORD PTR [eAL]
  aWRITE16
  jmp EndOp
On05:       ;ORA d
  aD
  aREAD16
  oORA16
  jmp EndOp
On06:       ;ASL d
  aD
  aREAD16
  oASL16
  aWRITE16
  jmp EndOp
On07:       ;ORA [d]
  aDIL
  aREAD16
  oORA16
  jmp EndOp
On09:       ;ORA #
  aIMM16
  oORA16
  jmp EndOp
On0A:       ;ASL A
  oGA
  oASL16
  oPA
  jmp EndOp
On0B:       ;PHD s
  oGD
  aPUSH16
  jmp EndOp
On0C:       ;TSB a
  aA
  aREAD16
  push edx
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,253                          ;clear flags that will be modified
  and dx,WORD PTR [eAL]
  fGZ
  pop edx
  or dx,WORD PTR [eAL]
  aWRITE16
  jmp EndOp
On0D:       ;ORA a
  aA
  aREAD16
  oORA16
  jmp EndOp
On0E:       ;ASL a
  aA
  aREAD16
  oASL16
  aWRITE16
  jmp EndOp
On0F:       ;ORA al
  aAL
  aREAD16
  oORA16
  jmp EndOp
On11:       ;ORA [d],y
  aDILXY
  aREAD16
  oORA16
  jmp EndOp
On12:       ;ORA (d)     
  aDI
  aREAD16
  oORA16
  jmp EndOp
On13:       ;ORA (d,s),y
  aSRIXY
  aREAD16
  oORA16
  jmp EndOp
On14:       ;TRB d
  aD
  aREAD16
  push edx
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,253                          ;clear flags that will be modified
  and dx,WORD PTR [eAL]
  fGZ
  pop edx
  mov ax,WORD PTR [eAL]
  not ax
  and dx,ax
  aWRITE16
  jmp EndOp
On15:       ;ORA d,x
  aDXX
  aREAD16
  oORA16
  jmp EndOp
On16:       ;ASL d,x  
  aDXX
  aREAD16
  oASL16
  aWRITE16
  jmp EndOp
On17:       ;ORA [d],y
  aDILXY
  aREAD16
  oORA16
  jmp EndOp
On19:       ;ORA a,y
  aAXY
  aREAD16
  oORA16
  jmp EndOp
On1A:       ;INC A
  oGA
  oINC16
  oPA
  jmp EndOp
On1C:       ;TRB a
  aA
  aREAD16
  push edx
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,253                          ;clear flags that will be modified
  and dx,WORD PTR [eAL]
  fGZ
  pop edx
  mov ax,WORD PTR [eAL]
  not ax
  and dx,ax
  aWRITE16
  jmp EndOp
On1D:       ;ORA a,x
  aAXX
  aREAD16
  oORA16
  jmp EndOp
On1E:       ;ASL a,x
  aAXX
  aREAD16
  oASL16
  aWRITE16
  jmp EndOp
On1F:       ;ORA al,x
  aALXX
  aREAD16
  oORA16
  jmp EndOp
On20:       ;JSR a
  jmp EndOp
On21:       ;AND (d,x)
  aDXXI
  aREAD16
  oAND16
  jmp EndOp
On22:       ;JSL al
  jmp EndOp
On23:       ;AND d,s
  aSR
  aREAD16
  oAND16
  jmp EndOp
On24:       ;BIT d
  aD
  aREAD16
  push edx
  mov al,BYTE PTR [eP]
  and al,63
  and dh,192
  or al,dh
  pop edx
  and al,253                          ;clear flags that will be modified
  and dx,WORD PTR [eAL]
  fGZ
  jmp EndOp
On25:       ;AND d
  aD
  aREAD16
  oAND16
  jmp EndOp
On26:       ;ROL d
  aD
  aREAD16
  oROL16
  aWRITE16
  jmp EndOp
On27:       ;AND [d]
  aDIL
  aREAD16
  oAND16
  jmp EndOp
On28:       ;PLP s
  jmp EndOp
On29:       ;AND #
  aIMM16
  oAND16
  jmp EndOp
On2A:       ;ROL A
  oGA
  oROL16
  oPA
  jmp EndOp
On2B:       ;PLD s
  jmp EndOp
On2C:       ;BIT a
  aA
  aREAD16
  push edx
  mov al,BYTE PTR [eP]
  and al,63
  and dh,192
  or al,dh
  pop edx
  and al,253                          ;clear flags that will be modified
  and dx,WORD PTR [eAL]
  fGZ
  jmp EndOp
On2D:       ;AND a
  aA
  aREAD16
  oAND16
  jmp EndOp
On2E:       ;ROL a
  aA
  aREAD16
  oROL16
  aWRITE16
  jmp EndOp
On2F:       ;AND al
  aAL
  aREAD16
  oAND16
  jmp EndOp
On31:       ;AND (d),y
  aDILXY
  aREAD16
  oAND16
  jmp EndOp
On32:       ;AND (d)
  aDI
  aREAD16
  oAND16
  jmp EndOp
On33:       ;AND (d,s),y
  aSRIXY
  aREAD16
  oAND16
  jmp EndOp
On34:       ;BIT d,x
  aDXX
  aREAD16
  push edx
  mov al,BYTE PTR [eP]
  and al,63
  and dh,192
  or al,dh
  pop edx
  and al,253                          ;clear flags that will be modified
  and dx,WORD PTR [eAL]
  fGZ
  jmp EndOp
On35:       ;AND d,x
  aDXX
  aREAD16
  oAND16
  jmp EndOp
On36:       ;ROL d,x
  aDXX
  aREAD16
  oROL16
  aWRITE16
  jmp EndOp
On37:       ;AND [d],y
  aDILXY
  aREAD16
  oAND16
  jmp EndOp
On38:       ;SEC i
  jmp EndOp
On39:       ;AND a,y
  aAXY
  aREAD16
  oAND16
  jmp EndOp
On3A:       ;DEC A
  oGA
  oDEC16
  oPA
  jmp EndOp
On3B:       ;TSC i
  jmp EndOp
On3C:       ;BIT a,x
  aAXX
  aREAD16
  push edx
  mov al,BYTE PTR [eP]
  and al,63
  and dh,192
  or al,dh
  pop edx
  and al,253                          ;clear flags that will be modified
  and dx,WORD PTR [eAL]
  fGZ
  jmp EndOp
On3D:       ;AND a,x
  aAXX
  aREAD16
  oAND16
  jmp EndOp
On3E:       ;ROL a,x
  aAXX
  aREAD16
  oROL16
  aWRITE16
  jmp EndOp
On3F:       ;AND al,x
  aALXX
  aREAD16
  oAND16
  jmp EndOp
On40:       ;RTI s
  jmp EndOp
On41:       ;EOR (d,x)
  aDXXI
  aREAD16
  oEOR16
  jmp EndOp
On42:       ;WDM
  jmp EndOp
On43:       ;EOR d,s
  aSR
  aREAD16
  oEOR16
  jmp EndOp
On44:       ;MVP xyc
  jmp EndOp
On45:       ;EOR d
  aD
  aREAD16
  oEOR16
  jmp EndOp
On46:       ;LSR d
  aD
  aREAD16
  oLSR16
  aWRITE16
  jmp EndOp
On47:       ;EOR [d]
  aDIL
  aREAD16
  oEOR16
  jmp EndOp
On48:       ;PHA s
  oGA
  aPUSH16
  jmp EndOp
On49:       ;EOR #
  aIMM16
  oEOR16
  jmp EndOp    
On4A:       ;LSR A
  oGA
  oLSR16
  oPA
  jmp EndOp
On4B:       ;PHK s
  jmp EndOp
On4C:       ;JMP a
  jmp EndOp
On4D:       ;EOR a
  aA
  aREAD16
  oEOR16
  jmp EndOp
On4E:       ;LSR a
  aA
  aREAD16
  oLSR16
  aWRITE16
  jmp EndOp
On4F:       ;EOR al
  aAL
  aREAD16
  oEOR16
  jmp EndOp
On51:       ;EOR (d),y
  aDILXY
  aREAD16
  oEOR16
  jmp EndOp
On52:       ;EOR (d)
  aDI
  aREAD16
  oEOR16
  jmp EndOp
On53:       ;EOR (d,s),y
  aSRIXY
  aREAD16
  oEOR16
  jmp EndOp
On54:       ;MVN xyc
  jmp EndOp   
On55:       ;EOR d,x
  aDXX
  aREAD16
  oEOR16
  jmp EndOp
On56:       ;LSR d,x
  aDXX
  aREAD16
  oLSR16
  aWRITE16
  jmp EndOp
On57:       ;EOR [d],y
  aDILXY
  aREAD16
  oEOR16
  jmp EndOp
On58:       ;CLI i
  jmp EndOp
On59:       ;EOR a,y
  aAXY
  aREAD16
  oEOR16
  jmp EndOp
On5A:       ;PHY s
  jmp EndOp
On5B:       ;TCD i
  jmp EndOp
On5C:       ;JMP al
  jmp EndOp
On5D:       ;EOR a,x
  aAXX
  aREAD16
  oEOR16
  jmp EndOp
On5E:       ;LSR a,x
  aAXX
  aREAD16
  oLSR16
  aWRITE16
  jmp EndOp
On5F:       ;EOR al,x
  aALXX
  aREAD16
  oEOR16
  jmp EndOp
On60:       ;RTS s
  jmp EndOp
On61:       ;ADC (d,x)
  aDXXI
  aREAD16     
  oADC16
  jmp EndOp
On62:       ;PER s
  jmp EndOp
On63:       ;ADC d,s
  aSR
  aREAD16
  oADC16
  jmp EndOp
On64:       ;STZ d
  aD
  xor edx,edx
  aWRITE16
  jmp EndOp
On65:       ;ADC d
  aD
  aREAD16
  oADC16
  jmp EndOp
On66:       ;ROR d
  aD
  aREAD16
  oROR16
  aWRITE16
  jmp EndOp
On67:       ;ADC [d]
  aDIL
  aREAD16
  oADC16
  jmp EndOp
On68:       ;PLA s
  oGA
  aPULL16
  oPA
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dx,dx
  fGNZ
  jmp EndOp
On69:       ;ADC #
  aIMM16
  oADC16
  jmp EndOp
On6A:       ;ROR A
  oGA
  oROR16
  oPA
  jmp EndOp
On6B:       ;RTL s
  jmp EndOp
On6C:       ;JMP (a)
  jmp EndOp
On6D:       ;ADC a
  aA
  aREAD16
  oADC16
  jmp EndOp
On6E:       ;ROR a
  aA
  aREAD16
  oROR16
  aWRITE16
  jmp EndOp
On6F:       ;ADC al,x
  aAL
  aREAD16
  oADC16
  jmp EndOp
On71:       ;ADC (d),y
  aDILXY
  aREAD16
  oADC16
  jmp EndOp
On72:       ;ADC (d)
  aDI
  aREAD16
  oADC16
  jmp EndOp
On73:       ;ADC (d,s),y
  aSRIXY
  aREAD16
  oADC16
  jmp EndOp
On74:       ;STZ d,x
  aDXX
  xor edx,edx
  aWRITE16
  jmp EndOp
On75:       ;ADC d,x
  aDXX
  aREAD16
  oADC16
  jmp EndOp
On76:       ;ROR d,x
  aDXX
  aREAD16
  oROR16
  aWRITE16
  jmp EndOp
On77:       ;ADC [d],y
  aDILXY
  aREAD16
  oADC16
  jmp EndOp
On78:       ;SEI i
  jmp EndOp
On79:       ;ADC a,y
  aAXY
  aREAD16
  oADC16
  jmp EndOp
On7B:       ;TDC i
  jmp EndOp
On7C:       ;JMP (a,x)
  jmp EndOp
On7D:       ;ADC a,x
  aAXX
  aREAD16
  oADC16
  jmp EndOp
On7E:       ;ROR a,x
  aAXX
  aREAD16
  oROR16
  aWRITE16
  jmp EndOp
On7F:       ;ADC al,x
  aALXX
  aREAD16
  oADC16
  jmp EndOp
On81:       ;STA (d,x)
  aDXXI
  oSTA16
  aWRITE8
  jmp EndOp
On82:       ;BRL rl
  jmp EndOp
On83:       ;STA d,s
  aSR
  oSTA16
  aWRITE8
  jmp EndOp
On84:       ;STY d
  jmp EndOp
On85:       ;STA d
  aD
  oSTA16
  aWRITE8
  jmp EndOp
On86:       ;STX d
  jmp EndOp
On87:       ;STA [d]
  aDIL
  oSTA16
  aWRITE8
  jmp EndOp
On88:       ;DEY i
  jmp EndOp
On89:       ;BIT #
  aIMM16
  push edx
  mov al,BYTE PTR [eP]
  and al,63
  and dh,192
  or al,dh
  pop edx
  and al,253                          ;clear flags that will be modified
  and dx,WORD PTR [eAL]
  fGZ
  jmp EndOp
On8A:       ;TXA i
  oGX
  oPA
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dx,dx
  fGNZ
  jmp EndOp
On8B:       ;PHB s
  jmp EndOp
On8C:       ;STY a
  jmp EndOp
On8D:       ;STA a
  aA
  oSTA16
  aWRITE8
  jmp EndOp
On8E:       ;STX a
  jmp EndOp
On8F:       ;STA al
  aAL
  oSTA16
  aWRITE8
  jmp EndOp
On91:       ;STA (d),y
  aDILXY
  oSTA16
  aWRITE8
  jmp EndOp
On92:       ;STA (d)
  aDI
  oSTA16
  aWRITE8
  jmp EndOp
On93:       ;STA (d,s),y
  aSRIXY
  oSTA16
  aWRITE8
  jmp EndOp
On94:       ;STY d,x
  jmp EndOp
On95:       ;STA d,x
  aDXX
  oSTA16
  aWRITE8
  jmp EndOp
On96:       ;STX d,y
  jmp EndOp
On97:       ;STA [d],y
  aDILXY
  oSTA16
  aWRITE8
  jmp EndOp
On98:       ;TYA i
  oGY
  oPA
  mov al,BYTE PTR [eP]                ;get 65816 flags into al
  and al,125                          ;clear flags that will be modified
  or dx,dx
  fGNZ
  jmp EndOp
On99:       ;STA a,y
  aAXY
  oSTA16
  aWRITE8
  jmp EndOp
On9A:       ;TXS i
  jmp EndOp
On9B:       ;TXY i
  jmp EndOp
On9C:       ;STZ a
  aA
  xor edx,edx
  aWRITE16
  jmp EndOp
On9D:       ;STA a,x
  aAXX
  oSTA16
  aWRITE8
  jmp EndOp
On9E:       ;STZ a,x
  aAXX
  xor edx,edx
  aWRITE16
  jmp EndOp
On9F:       ;STA al,x
  aALXX
  oSTA16
  aWRITE8
  jmp EndOp
OnA0:       ;LDY #
  jmp EndOp
OnA1:       ;LDA (d,x)
  aDXXI
  aREAD16     
  oLDA16
  jmp EndOp
OnA2:       ;LDX #
  jmp EndOp
OnA3:       ;LDA d,s
  aSR
  aREAD16
  oLDA16
  jmp EndOp
OnA4:       ;LDY d
  jmp EndOp
OnA5:       ;LDA d
  aD
  aREAD16
  oLDA16
  jmp EndOp
OnA6:       ;LDX d
  jmp EndOp
OnA7:       ;LDA [d]
  aDIL
  aREAD16
  oLDA16
  jmp EndOp
OnA8:       ;TAY i
  jmp EndOp
OnA9:       ;LDA #
  aIMM16
  oLDA16
  jmp EndOp
OnAA:       ;TAX i
  jmp EndOp
OnAB:       ;PLB s
  jmp EndOp
OnAC:       ;LDY a
  jmp EndOp
OnAD:       ;LDA a
  aA
  aREAD16
  oLDA16
  jmp EndOp
OnAE:       ;LDX a
  jmp EndOp
OnAF:       ;LDA al
  aAL
  aREAD16
  oLDA16
  jmp EndOp
OnB1:       ;LDA (d),y
  aDILXY
  aREAD16
  oLDA16
  jmp EndOp
OnB2:       ;LDA (d)
  aDI
  aREAD16
  oLDA16
  jmp EndOp
OnB3:       ;LDY (d,s),y
  aSRIXY
  aREAD16
  oLDA16
  jmp EndOp
OnB4:       ;LDY d,x
  jmp EndOp
OnB5:       ;LDA d,x
  aDXX
  aREAD16
  oLDA16
  jmp EndOp
OnB6:       ;LDX d,y
  jmp EndOp
OnB7:       ;LDA [d],y
  aDILXY
  aREAD16
  oLDA16
  jmp EndOp
OnB8:       ;CLV i
  jmp EndOp
OnB9:       ;LDA a,y
  aAXY
  aREAD16
  oLDA16
  jmp EndOp
OnBA:       ;TSX i
  jmp EndOp
OnBB:       ;TYX i
  jmp EndOp
OnBC:       ;LDY a,x
  jmp EndOp
OnBD:       ;LDA a,x
  aAXX
  aREAD16
  oLDA16
  jmp EndOp
OnBE:       ;LDX a,y
  jmp EndOp
OnBF:       ;LDA al,x
  aALXX
  aREAD16
  oLDA16
  jmp EndOp
OnC0:       ;CPY #
  jmp EndOp
OnC1:       ;CMP (d,x)
  aDXXI
  aREAD16
  oCMP16
  jmp EndOp
OnC2:       ;REP #
  jmp EndOp
OnC3:       ;CMP d,s
  aSR
  aREAD16
  oCMP16
  jmp EndOp
OnC4:       ;CPY d
  jmp EndOp
OnC5:       ;CMP d
  aD
  aREAD16
  oCMP16
  jmp EndOp
OnC6:       ;DEC d
  aD
  aREAD16
  oDEC16
  aWRITE16
  jmp EndOp
OnC7:       ;CMP [d]
  aDIL
  aREAD16
  oCMP16
  jmp EndOp
OnC8:       ;INY i
  jmp EndOp
OnC9:       ;CMP #
  aIMM16
  oCMP16
  jmp EndOp
OnCA:       ;DEX i
  jmp EndOp
OnCB:       ;WAI i
  jmp EndOp
OnCC:       ;CPY a
  jmp EndOp
OnCD:       ;CMP a
  aA
  aREAD16
  oCMP16
  jmp EndOp
OnCE:       ;DEC a
  aA
  aREAD16
  oDEC16
  aWRITE16
  jmp EndOp
OnCF:       ;CMP al
  aAL
  aREAD16
  oCMP16
  jmp EndOp
OnD1:       ;CMP (d),y
  aDILXY
  aREAD16
  oCMP16
  jmp EndOp
OnD2:       ;CMP (d)
  aDI
  aREAD16
  oCMP16
  jmp EndOp
OnD3:       ;CMP (d,s),y
  aSRIXY
  aREAD16
  oCMP16
  jmp EndOp
OnD4:       ;PEI s
  jmp EndOp
OnD5:       ;CMP d,x
  aDXX
  aREAD16
  oCMP16
  jmp EndOp
OnD6:       ;DEC d,x
  aDXX
  aREAD16
  oDEC16
  aWRITE16
  jmp EndOp
OnD7:       ;CMP [d],y
  aDILXY
  aREAD16
  oCMP16
  jmp EndOp
OnD8:       ;CLD i
  jmp EndOp
OnD9:       ;CMP a,y
  aAXY
  aREAD16
  oCMP16
  jmp EndOp
OnDA:       ;PHX s
  jmp EndOp
OnDB:       ;STP i
  jmp EndOp
OnDC:       ;JML (a)
  jmp EndOp
OnDD:       ;CMP a,x
  aAXX
  aREAD16
  oCMP16
  jmp EndOp
OnDE:       ;DEC a,x
  aAXX
  aREAD16
  oDEC16
  aWRITE16
  jmp EndOp
OnDF:       ;CMP al,x
  aALXX
  aREAD16
  oCMP16
  jmp EndOp
OnE0:       ;CPX #
  jmp EndOp
OnE1:       ;SBC (d,x)
  aDXXI
  aREAD16     
  oSBC16
  jmp EndOp
OnE2:       ;SEP #
  jmp EndOp
OnE3:       ;SBC d,s
  aSR
  aREAD16
  oSBC16
  jmp EndOp
OnE4:       ;CPX d
  jmp EndOp
OnE5:       ;SBC d
  aD
  aREAD16
  oSBC16
  jmp EndOp
OnE6:       ;INC d
  aD
  aREAD16
  oINC16
  aWRITE16
  jmp EndOp
OnE7:       ;SBC [d]
  aDIL
  aREAD16
  oSBC16
  jmp EndOp
OnE8:       ;INX i
  jmp EndOp
OnE9:       ;SBC #
  aIMM16
  oSBC16
  jmp EndOp
OnEA:       ;NOn i
  jmp EndOp
OnEB:       ;XBA i
  jmp EndOp
OnEC:       ;CPX a
  jmp EndOp
OnED:       ;SBC a
  aA
  aREAD16
  oSBC16
  jmp EndOp
OnEE:       ;INC a
  aA
  aREAD16
  oINC16
  aWRITE16
  jmp EndOp
OnEF:       ;SBC al
  aAL
  aREAD16
  oSBC16
  jmp EndOp
OnF1:       ;SBC (d),y
  aDILXY
  aREAD16
  oSBC16
  jmp EndOp
OnF2:       ;SBC (d)
  aDI
  aREAD16
  oSBC16
  jmp EndOp
OnF3:       ;SBC (d,s),y
  aSRIXY
  aREAD16
  oSBC16
  jmp EndOp
OnF4:       ;PEA s
  jmp EndOp
OnF5:       ;SBC d,x
  aDXX
  aREAD16
  oSBC16
  jmp EndOp
OnF6:       ;INC d,x
  aDXX
  aREAD16
  oINC16
  aWRITE16
  jmp EndOp
OnF7:       ;SBC [d],y
  aDILXY
  aREAD16
  oSBC16
  jmp EndOp
OnF8:       ;SED i
  jmp EndOp
OnF9:       ;SBC a,y
  aAXY
  aREAD16
  oSBC16
  jmp EndOp
OnFA:       ;PLX s
  jmp EndOp
OnFB:       ;XCE i
  jmp EndOp
OnFC:       ;JSR (a,x)
  jmp EndOp
OnFD:       ;SBC a,x
  aAXX
  aREAD16
  oSBC16
  jmp EndOp
OnFE:       ;INC a,x
  aAXX
  aREAD16
  oINC16
  aWRITE16
  jmp EndOp
OnFF:       ;SBC al,x
  aALXX
  aREAD16
  oSBC16
  jmp EndOp



eJumpTable dd Op00,Op01,Op02,Op03,Op04,Op05,Op06,Op07,Op08,Op09,Op0A,Op0B,Op0C,Op0D,Op0E,Op0F
           dd Op10,Op11,Op12,Op13,Op14,Op15,Op16,Op17,Op18,Op19,Op1A,Op1B,Op1C,Op1D,Op1E,Op1F
           dd Op20,Op21,Op22,Op23,Op24,Op25,Op26,Op27,Op28,Op29,Op2A,Op2B,Op2C,Op2D,Op2E,Op2F
           dd Op30,Op31,Op32,Op33,Op34,Op35,Op36,Op37,Op38,Op39,Op3A,Op3B,Op3C,Op3D,Op3E,Op3F
           dd Op40,Op41,Op42,Op43,Op44,Op45,Op46,Op47,Op48,Op49,Op4A,Op4B,Op4C,Op4D,Op4E,Op4F
           dd Op50,Op51,Op52,Op53,Op54,Op55,Op56,Op57,Op58,Op59,Op5A,Op5B,Op5C,Op5D,Op5E,Op5F
           dd Op60,Op61,Op62,Op63,Op64,Op65,Op66,Op67,Op68,Op69,Op6A,Op6B,Op6C,Op6D,Op6E,Op6F
           dd Op70,Op71,Op72,Op73,Op74,Op75,Op76,Op77,Op78,Op79,Op7A,Op7B,Op7C,Op7D,Op7E,Op7F
           dd Op80,Op81,Op82,Op83,Op84,Op85,Op86,Op87,Op88,Op89,Op8A,Op8B,Op8C,Op8D,Op8E,Op8F
           dd Op90,Op91,Op92,Op93,Op94,Op95,Op96,Op97,Op98,Op99,Op9A,Op9B,Op9C,Op9D,Op9E,Op9F
           dd OpA0,OpA1,OpA2,OpA3,OpA4,OpA5,OpA6,OpA7,OpA8,OpA9,OpAA,OpAB,OpAC,OpAD,OpAE,OpAF
           dd OpB0,OpB1,OpB2,OpB3,OpB4,OpB5,OpB6,OpB7,OpB8,OpB9,OpBA,OpBB,OpBC,OpBD,OpBE,OpBF
           dd OpC0,OpC1,OpC2,OpC3,OpC4,OpC5,OpC6,OpC7,OpC8,OpC9,OpCA,OpCB,OpCC,OpCD,OpCE,OpCF
           dd iOpD0,OpD1,OpD2,OpD3,OpD4,OpD5,OpD6,OpD7,OpD8,OpD9,OpDA,iOpDB,OpDC,OpDD,OpDE,OpDF
           dd OpE0,OpE1,OpE2,OpE3,OpE4,OpE5,OpE6,OpE7,OpE8,OpE9,OpEA,OpEB,OpEC,OpED,OpEE,OpEF
           dd OpF0,OpF1,OpF2,OpF3,OpF4,OpF5,OpF6,OpF7,OpF8,OpF9,OpFA,OpFB,OpFC,OpFD,OpFE,OpFF

nJumpTable dd Op00,On01,Op02,On03,On04,On05,On06,On07,Op08,On09,On0A,Op0B,On0C,On0D,On0E,On0F
           dd Op10,On11,On12,On13,On14,On15,On16,On17,Op18,On19,On1A,Op1B,On1C,On1D,On1E,On1F
           dd Op20,On21,Op22,On23,On24,On25,On26,On27,Op28,On29,On2A,Op2B,On2C,On2D,On2E,On2F
           dd Op30,On31,On32,On33,On34,On35,On36,On37,Op38,On39,On3A,Op3B,On3C,On3D,On3E,On3F
           dd Op40,On41,Op42,On43,Op44,On45,On46,On47,On48,On49,On4A,Op4B,Op4C,On4D,On4E,On4F
           dd Op50,On51,On52,On53,Op54,On55,On56,On57,Op58,On59,Op5A,Op5B,Op5C,On5D,On5E,On5F
           dd Op60,On61,Op62,On63,On64,On65,On66,On67,On68,On69,On6A,Op6B,Op6C,On6D,On6E,On6F
           dd Op70,On71,On72,On73,On74,On75,On76,On77,Op78,On79,Op7A,Op7B,Op7C,On7D,On7E,On7F
           dd Op80,On81,Op82,On83,On84,On85,On86,On87,Op88,On89,On8A,Op8B,On8C,On8D,On8E,On8F
           dd Op90,On91,On92,On93,On94,On95,On96,On97,On98,On99,Op9A,Op9B,On9C,On9D,On9E,On9F
           dd OpA0,OnA1,OpA2,OnA3,OpA4,OnA5,OpA6,OnA7,OpA8,OnA9,OpAA,OpAB,OpAC,OnAD,OpAE,OnAF
           dd OpB0,OnB1,OnB2,OnB3,OpB4,OnB5,OpB6,OnB7,OpB8,OnB9,OpBA,OpBB,OpBC,OnBD,OpBE,OnBF
           dd OpC0,OnC1,OpC2,OnC3,OpC4,OnC5,OnC6,OnC7,OpC8,OnC9,OpCA,OpCB,OpCC,OnCD,OnCE,OnCF
           dd iOpD0,OnD1,OnD2,OnD3,OpD4,OnD5,OnD6,OnD7,OpD8,OnD9,OpDA,iOpDB,OpDC,OnDD,OnDE,OnDF
           dd OpE0,OnE1,OpE2,OnE3,OpE4,OnE5,OnE6,OnE7,OpE8,OnE9,OpEA,OpEB,OpEC,OnED,OnEE,OnEF
           dd OpF0,OnF1,OnF2,OnF3,OpF4,OnF5,OnF6,OnF7,OpF8,OnF9,OpFA,OpFB,OpFC,OnFD,OnFE,OnFF

;ͻ
;                      Interrupt Emulation                            
;ͼ
;These subroutines emulate the Reset, NMI, BRK, COP, IRQ, etc.

eCOP PROC                              ;COP interrupt
  inc si                                 ;increment PC (for return address)
  movzx edx,BYTE PTR [ePB]               ;get PB
  mov ecx,0fff4h                         ;get e mode COP vector
  cmp [ePe],0                            ;
  jne eCOPNoPB                           ;
  aPUSH8                                 ;If n mode, push PB
  mov ecx,0ffe4h                         ;get n mode COP vector
  eCOPNoPB:                              ;
  mov dx,si                              ;
  aPUSH16                                ;push PC
  mov dl,[eP]                            ;
  aPUSH8                                 ;push P
  or [eP],4                              ;set IRQ mask flag
  and [eP],247                           ;clear decimal flag
  mov esi,ecx                            ;get reset vector
  MemReadP                               ;
  inc si                                 ;
  MemReadPH                              ;
  mov si,ax                              ;get interrupt handler address
  ret                                    ;done
ENDP
eNMI PROC
  cmp [eWAIT],0                          ;check for WAI instruction active
  je eNMIW                               ;
  inc WORD PTR [ePCL]                    ;if in WAI loop, get out of it
  mov [eWAIT],0                          ;
  eNMIW:                                 ;
  movzx edx,BYTE PTR [ePB]               ;get PB
  mov ecx,0fffah                         ;get e mode IRQ/BRK vector
  cmp [ePe],0                            ;
  jne eNMINoPB                           ;
  aPUSH8                                 ;If n mode, push PB
  mov ecx,0ffeah                         ;get n mode IRQ vector
  eNMINoPB:                              ;
  mov dx,si                              ;
  aPUSH16                                ;push PC
  mov dl,[eP]                            ;
  aPUSH8                                 ;push P
  or [eP],4                              ;set IRQ mask flag
  and [eP],247                           ;clear decimal flag
  mov esi,ecx                            ;get reset vector
  MemReadP                               ;
  inc si                                 ;
  MemReadPH                              ;
  mov si,ax                              ;get interrupt handler address
  mov DWORD PTR [ePCL],esi               ;preserve the program counter
  ret
ENDP
eBRK PROC
  inc si                                 ;increment PC (for return address)
  movzx edx,BYTE PTR [ePB]               ;get PB
  mov ecx,0fffeh                         ;get e mode BRK/IRQ vector
  cmp [ePe],0                            ;
  jne eBRKNoPB                           ;
  aPUSH8                                 ;If n mode, push PB
  mov ecx,0ffe6h                         ;get n mode BRK vector
  eBRKNoPB:                              ;
  mov dx,si                              ;
  aPUSH16                                ;push PC
  mov dl,[eP]                            ;
  aPUSH8                                 ;push P
  or [eP],4                              ;set IRQ mask flag
  cmp [ePe],0                            ;
  je NoBRKFlag                           ;
  or [eP],16                             ;set BRK flag if in e mode
  NoBRKFlag:                             ;
  and [eP],247                           ;clear decimal flag
  mov esi,ecx                            ;get reset vector
  MemReadP                               ;
  inc si                                 ;
  MemReadPH                              ;
  mov si,ax                              ;get interrupt handler address
  ret                                    ;done
  ret
ENDP
eIRQ PROC
  test [eP],ePi                          ;is IRQ masked?
  jz NoIRQMask                           ;
  ret                                    ;if so, don't process IRQ
  NoIRQMask:                             ;
  cmp [eWAIT],0                          ;check for WAI instruction active
  je eIRQW                               ;
  inc WORD PTR [ePCL]                    ;if in WAI loop, get out of it
  mov [eWAIT],0                          ;
  eIRQW:                                 ;
  movzx edx,BYTE PTR [ePB]               ;get PB
  mov ecx,0fffeh                         ;get e mode IRQ/BRK vector
  cmp [ePe],0                            ;
  jne eIRQNoPB                           ;
  aPUSH8                                 ;If n mode, push PB
  mov ecx,0ffeeh                         ;get n mode IRQ vector
  eIRQNoPB:                              ;
  mov dx,si                              ;
  aPUSH16                                ;push PC
  mov dl,[eP]                            ;
  aPUSH8                                 ;push P
  or [eP],4                              ;set IRQ mask flag
  and [eP],247                           ;clear decimal flag
  mov esi,ecx                            ;get reset vector
  MemReadP                               ;
  inc si                                 ;
  MemReadPH                              ;
  mov si,ax                              ;get interrupt handler address
  mov DWORD PTR [ePCL],esi               ;preserve the program counter
  ret                                    ;done
ENDP
eRESET PROC                            ;Reset the 65816
  mov DWORD PTR [eAL],0                  ;clear all registers
  mov DWORD PTR [eXL],0                  ;
  mov DWORD PTR [eYL],0                  ;
  mov DWORD PTR [eDL],0                  ;
  mov DWORD PTR [eDBP],0                 ;
  mov DWORD PTR [eSL],01FFh              ;reset stack
  mov DWORD PTR [eWAIT],0                ;clear any WAI state
  mov [eP],52                            ;initialize flags (m=1 x=1 d=0 i=1)
  mov [ePe],1                            ;set in Emulation mode (e mode)
  mov esi,0fffch                         ;reset vector
  MemReadP                               ;
  inc si                                 ;
  MemReadPH                              ;
  mov si,ax                              ;get Reset address
  mov DWORD PTR [ePCL],esi               ;
  ret                                    ;done
ENDP
Reset65816 EQU eRESET
