// 175 opcodes...
#include <stdio.h>

// The beginning of the file
char *Beginning = {
  "BITS 32\n\n"
  "EXTERN SnesRom_      ; Pointer to the SNES's ROM 00-3f:8000-ffff, 40-7d:0000-ffff\n"
  "EXTERN SnesRam_      ; Pointer to the SNES's RAM 00-3f:0000-1fff, 7e-7f:0000-ffff\n"
  "EXTERN SnesReg_      ; Pointer to the SNES's register memory 00-3f:2100-4fff\n"
  "EXTERN IP_           ; the instruction pointer\n"
  "EXTERN A_            ; The A register (general purpose)\n"
  "EXTERN X_            ; The X register (general purpose)\n"
  "EXTERN Y_            ; The Y register (general purpose)\n"
  "EXTERN DB_           ; Data bank register\n"
  "EXTERN P_            ; The processor status byte (actually a word, I use smsw\n"
  "                     ; and lmsw so I don't have to manualy keep track of the\n"
  "                     ; math status bits)\n"
  "EXTERN Pe_           ; The emulation flag\n"
  "EXTERN D_            ; Direct page register\n"
  "EXTERN Pi_           ; The interrupt flag\n"
  "EXTERN Pn_           ; Negative flag\n"
  "EXTERN S_            ; The stack pointer (bank is always 0)\n"
  "EXTERN Pm_           ; Used to determine the size of A\n"
  "EXTERN Px_           ; Used to determine the size of X,Y\n\n"
  "EXTERN RegFxnB_      ; Register Function Called for byte register writes\n"
  "EXTERN RegFxnW_      ; same 'cept for words\n\n"
  "EXTERN RegValB_      ; hold the value of Byte registers\n"
  "EXTERN RegValW_      ; lalala\n"
  "EXTERN RegNum_       ; The register being written to\n\n"
  "SECTION .text\n\n"
};

// To be put at the start and end of every fxn
char *StartFxn = {
  "  push eax\n"
  "  push ebx\n"
  "  push edi\n"
  "  push esi\n"
};

char *EndFxn = {
  "  pop esi\n"
  "  pop edi\n"
  "  pop ebx\n"
  "  pop eax\n"
};

// Addressing Modes
char *AddrImmed = {
  "  mov esi, [IP_]\n"
  "  inc esi\n"
  "  mov edi, [esi]\n"
};

char *AddrAccu = {
  "  mov ax, [A_]\n"
};

char *AddrDir = {
  "  mov esi, [IP_]\n"
  "  inc esi\n"
  "  xor edi, edi\n"
  "  mov ax, [D_]\n"
  "  add al, [esi]\n"
  "  mov di, ax\n"
  "  add edi, [DB_]\n"
};

char *AddrDirX = {
  "  mov esi, [IP_]\n"
  "  inc esi\n"
  "  xor edi, edi\n"
  "  mov ax, [D_]\n"
  "  add al, [esi]\n"
  "  add ax, [X_]\n"
  "  mov di, ax\n"
};

char *AddrDirY = {
  "  mov esi, [IP_]\n"
  "  inc esi\n"
  "  xor edi, edi\n"
  "  mov ax, [D_]\n"
  "  add al, [esi]\n"
  "  add ax, [Y_]\n"
  "  mov di, ax\n"
};

char *AddrDirInd = {
  "  mov esi, [IP_]\n"
  "  inc esi\n"
  "  xor edi, edi\n"
  "  mov ax, [D_]\n"
  "  add al, [esi]\n"
  "  mov di, ax\n"
  "  add edi, [DB_]\n"
};

char *AddrDirIndX = {
  "  mov esi, [IP_]\n"
  "  inc esi\n"
  "  xor edi, edi\n"
  "  mov ax, [D_]\n"
  "  add al, [esi]\n"
  "  add ax, [Y_]\n"
  "  mov di, ax\n"
  "  add edi, [DB_]\n"
};

char *AddrDirIndY = {
  "  mov esi, [IP_]\n"
  "  inc esi\n"
  "  xor edi, edi\n"
  "  mov ax, [D_]\n"
  "  add al, [esi]\n"
  "  mov di, ax\n"
  "  add edi, [DB_]\n"
  "  xor ax, ax\n"
  "  mov ax, [Y_]\n"
  "  add edi, eax\n"
};

char *AddrDirIndL = {
  "  mov esi, [IP_]\n"
  "  inc esi\n"
  "  xor edi, edi\n"
  "  mov ax, [D_]\n"
  "  add al, [esi]\n"
  "  mov di, ax\n"
};

char *AddrAbs = {
  "  mov esi, [IP_]\n"
  "  inc esi\n"
  "  mov ah, [esi]\n"
  "  inc esi\n"
  "  xor edi, edi\n"
  "  mov al, [esi]\n"
  "  mov di, ax\n"
  "  add edi, [DB_]\n"
};

char *AddrAbsX = {
  "  mov esi, [IP_]\n"
  "  inc esi\n"
  "  mov ah, [esi]\n"
  "  inc esi\n"
  "  xor edi, edi\n"
  "  mov al, [esi]\n"
  "  mov di, ax\n"
  "  add di, [X_]\n"
  "  add edi, [DB_]\n"
};

char *AddrAbsY = {
  "  mov esi, [IP_]\n"
  "  inc esi\n"
  "  mov ah, [esi]\n"
  "  inc esi\n"
  "  xor edi, edi\n"
  "  mov al, [esi]\n"
  "  mov di, ax\n"
  "  add di, [Y_]\n"
  "  add edi, [DB_]\n"
};

char *AddrAbsL = {
  "  mov esi, [IP_]\n"
  "  inc esi\n"
  "  mov ah, [esi]\n"
  "  inc esi\n"
  "  xor edi, edi\n"
  "  mov al, [esi]\n"
  "  inc esi\n"
  "  mov di, ax\n"
  "  xor eax, eax\n"
  "  mov al, [esi]\n"
  "  shl eax, 16\n"
  "  add edi, eax\n"
};

char *AddrStackRel = {
  "  mov esi, [IP_]\n"
  "  inc esi\n"
  "  xor edi, edi\n"
  "  mov ax, [S_]\n"
  "  add al, [esi]\n"
  "  mov di, ax\n"
};

char *AddrStackRelIndY = {
  "  mov esi, [IP_]\n"
  "  inc esi\n"
  "  xor edi, edi\n"
  "  mov ax, [S_]\n"
  "  add al, [esi]\n"
  "  mov di, ax\n"
  "  add edi, [DB_]\n"
  "  xor eax, eax\n"
  "  mov ax, [Y_]\n"
  "  add edi, eax\n"
};

// Flag macros
char *GetFlags = {
  "%%macro GetFlags 0\n"
  "  and ax, 08000h\n"
  "  shr ax, 15\n"
  "  mov [Pn_], al\n"
  "  pushf\n"
  "  pop ax\n"
  "  mov [P_], ax\n"
  "%%endmacro\n"
};

char *SetFlags = {
  "%%macro SetFlags 0\n"
  "  mov ax, [P_]\n"
  "  push ax\n"
  "  popf\n"
  "%%endmacro\n"
};

// Memory macros
char *SnesMap = {
  "SnesMap_:\n"
  "  mov eax, edi\n"
  "  xor ebx, ebx\n"
  "  mov bx, ax\n"
  "  and eax, 07F0000h\n"
  "  cmp eax, 03FFFFFh\n"
  "  jg  .highmem\n"
  "  cmp bx, 08000h\n"
  "  jl  .notrom\n"
  "  shr eax, 1\n"
  "  and ebx, 07FFFh\n"
  "  or  eax, ebx\n"
  "  mov edi, eax\n"
  "  add edi, [SnesRom_]\n"
  "  jmp .done\n"
  ".notrom\n"
  "  cmp bx, 01FFFh\n"
  "  jg  .reg\n"
  "  mov edi, ebx\n"
  "  add edi, [SnesRam_]\n"
  "  jmp .done\n"
  ".reg\n"
  "  sub bx, 02100h\n"
  "  mov edi, ebx\n"
  "  add edi, [SnesReg_]\n"
  "  jmp .done\n"
  ".highmem\n"
  "  cmp eax, 07E0000h\n"
  "  jl  .done\n"
  "  mov edi, [SnesRam_]\n"
  "  add edi, ebx\n"
  ".done\n"
  "  ret\n"
};

char *GetWord = {
  "  call SnesMap_\n"
  "  mov ax, [edi]\n"
  "  xchg al, ah\n"
};

char *GetByte = {
  "  call SnesMap_\n"
  "  mov al, [edi]\n"
  "  cbw\n"
};

// Normal Opcode Instructions
char *adc = {
  "  mov bx, [A_]\n"
  "  SetFlags\n"
  "  adc bx, ax\n"
  "  GetFlags\n"
  "  mov [A_], bx\n"
};

char *and = {
  "  mov bx, [A_]\n"
  "  and bx, ax\n"
  "  GetFlags\n"
  "  mov [A_], bx\n"
};

char *bit = {
  "  mov bx, [A_]\n"
  "  and bx, ax\n"
  "  GetFlags\n"
}; 

char *cmp = {
  "  mov bx, [A_]\n"
  "  cmp bx, ax\n"
  "  GetFlags\n"
}; 

char *cpx = {
  "  mov bx, [X_]\n"
  "  cmp bx, ax\n"
  "  GetFlags\n"
  "  mov [A_], bx\n"
}; 

char *cpy = {
  "  mov bx, [Y_]\n"
  "  cmp bx, ax\n"
  "  GetFlags\n"
}; 

char *eor = {
  "  mov bx, [A_]\n"
  "  xor bx, ax\n"
  "  GetFlags\n"
  "  mov [A_], bx\n"
}; 

char *lda = {
  "  mov [A_], bx\n"
  "  GetFlags\n"
}; 

char *ldx = {
  "  mov [X_], bx\n"
  "  GetFlags\n"
}; 

char *ldy = {
  "  mov [Y_], bx\n"
  "  GetFlags\n"
};

char *ora = {
  "  mov bx, [A_]\n"
  "  or  bx, ax\n"
  "  GetFlags\n"
  "  mov [A_], bx\n"
}; 

char *sbc = {
  "  mov bx, [A_]\n"
  "  SetFlags\n"
  "  sbb bx, ax\n"
  "  GetFlags\n"
  "  mov [A_], bx\n"
}; 

// Accumulator opcodes
char *asl_a_b = {
  "  shl BYTE [A_+1], 1\n"
};

char *asl_a_w = {
  "  shl WORD [A_], 1\n"
};

char *asl_b = {
  "  xor ebx, ebx\n"
  "  mov bx, di\n"
  "  xor di, di\n"
  "  cmp edi, 03F0000h\n"
  "  jg  .trumem_b\n"
  "  mov edi, [SnesRam_]\n"
  "  add edi, ebx\n"
  "  mov al, [edi]\n"
  "  shl al, 1\n"
  "  mov [edi], al\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
  ".trumem_b\n"
  "  sub edi, 07E0000h\n"
  "  add edi, ebx\n"
  "  add edi, [SnesRam_]\n"
  "  mov al, [edi]\n"
  "  shl al, 1\n"
  "  mov [edi], al\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
}; 

char *asl_w = {        
  "  xor ebx, ebx\n"
  "  mov bx, di\n"
  "  xor di, di\n"
  "  cmp edi, 03F0000h\n"
  "  jg  .trumem_w\n"
  "  mov edi, [SnesRam_]\n"
  "  add edi, ebx\n"
  "  mov ax, [edi]\n"
  "  xchg ah, al\n"
  "  shl ax, 1\n"
  "  xchg ah, al\n"
  "  mov [edi], ax\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
  ".trumem_w\n"
  "  sub edi, 07E0000h\n"
  "  add edi, ebx\n"
  "  add edi, [SnesRam_]\n"
  "  mov ax, [edi]\n"
  "  xchg ah, al\n"
  "  shl ax, 1\n"
  "  xchg ah, al\n"
  "  mov [edi], ax\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
}; 

char *dec_a_b = {
  "  dec BYTE [A_+1]\n"
};

char *dec_a_w = {
  "  dec WORD [A_]\n"
};

char *dec_b = {
  "  xor ebx, ebx\n"
  "  mov bx, di\n"
  "  xor di, di\n"
  "  cmp edi, 03F0000h\n"
  "  jg  .trumem_b\n"
  "  mov edi, [SnesRam_]\n"
  "  add edi, ebx\n"
  "  mov al, [edi]\n"
  "  dec al\n"
  "  mov [edi], al\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
  ".trumem_b\n"
  "  sub edi, 07E0000h\n"
  "  add edi, ebx\n"
  "  add edi, [SnesRam_]\n"
  "  mov al, [edi]\n"
  "  dec al\n"
  "  mov [edi], al\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
}; 

char *dec_w = {        
  "  xor ebx, ebx\n"
  "  mov bx, di\n"
  "  xor di, di\n"
  "  cmp edi, 03F0000h\n"
  "  jg  .trumem_w\n"
  "  mov edi, [SnesRam_]\n"
  "  add edi, ebx\n"
  "  mov ax, [edi]\n"
  "  xchg ah, al\n"
  "  dec ax\n"
  "  xchg ah, al\n"
  "  mov [edi], ax\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
  ".trumem_w\n"
  "  sub edi, 07E0000h\n"
  "  add edi, ebx\n"
  "  add edi, [SnesRam_]\n"
  "  mov ax, [edi]\n"
  "  xchg ah, al\n"
  "  dec ax\n"
  "  xchg ah, al\n"
  "  mov [edi], ax\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
}; 

char *inc_a_b = {
  "  inc BYTE [A_+1]\n"
};

char *inc_a_w = {
  "  inc WORD [A_]\n"
};

char *inc_b = {
  "  xor ebx, ebx\n"
  "  mov bx, di\n"
  "  xor di, di\n"
  "  cmp edi, 03F0000h\n"
  "  jg  .trumem_b\n"
  "  mov edi, [SnesRam_]\n"
  "  add edi, ebx\n"
  "  mov al, [edi]\n"
  "  inc al\n"
  "  mov [edi], al\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
  ".trumem_b\n"
  "  sub edi, 07E0000h\n"
  "  add edi, ebx\n"
  "  add edi, [SnesRam_]\n"
  "  mov al, [edi]\n"
  "  inc al\n"
  "  mov [edi], al\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
}; 

char *inc_w = {        
  "  xor ebx, ebx\n"
  "  mov bx, di\n"
  "  xor di, di\n"
  "  cmp edi, 03F0000h\n"
  "  jg  .trumem_w\n"
  "  mov edi, [SnesRam_]\n"
  "  add edi, ebx\n"
  "  mov ax, [edi]\n"
  "  xchg ah, al\n"
  "  inc ax\n"
  "  xchg ah, al\n"
  "  mov [edi], ax\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
  ".trumem_w\n"
  "  sub edi, 07E0000h\n"
  "  add edi, ebx\n"
  "  add edi, [SnesRam_]\n"
  "  mov ax, [edi]\n"
  "  xchg ah, al\n"
  "  inc ax\n"
  "  xchg ah, al\n"
  "  mov [edi], ax\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
}; 

char *lsr_a_b = {
  "  shr BYTE [A_+1], 1\n"
};

char *lsr_a_w = {
  "  shr WORD [A_], 1\n"
};

char *lsr_b = {
  "  xor ebx, ebx\n"
  "  mov bx, di\n"
  "  xor di, di\n"
  "  cmp edi, 03F0000h\n"
  "  jg  .trumem_b\n"
  "  mov edi, [SnesRam_]\n"
  "  add edi, ebx\n"
  "  mov al, [edi]\n"
  "  shr al, 1\n"
  "  mov [edi], al\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
  ".trumem_b\n"
  "  sub edi, 07E0000h\n"
  "  add edi, ebx\n"
  "  add edi, [SnesRam_]\n"
  "  mov al, [edi]\n"
  "  shr al, 1\n"
  "  mov [edi], al\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
}; 

char *lsr_w = {        
  "  xor ebx, ebx\n"
  "  mov bx, di\n"
  "  xor di, di\n"
  "  cmp edi, 03F0000h\n"
  "  jg  .trumem_w\n"
  "  mov edi, [SnesRam_]\n"
  "  add edi, ebx\n"
  "  mov ax, [edi]\n"
  "  xchg ah, al\n"
  "  shr ax, 1\n"
  "  xchg ah, al\n"
  "  mov [edi], ax\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
  ".trumem_w\n"
  "  sub edi, 07E0000h\n"
  "  add edi, ebx\n"
  "  add edi, [SnesRam_]\n"
  "  mov ax, [edi]\n"
  "  xchg ah, al\n"
  "  shr ax, 1\n"
  "  xchg ah, al\n"
  "  mov [edi], ax\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
}; 

char *rol_a_b = {
  "  rol BYTE [A_+1], 1\n"
};

char *rol_a_w = {
  "  rol WORD [A_], 1\n"
};

char *rol_b = {
  "  xor ebx, ebx\n"
  "  mov bx, di\n"
  "  xor di, di\n"
  "  cmp edi, 03F0000h\n"
  "  jg  .trumem_b\n"
  "  mov edi, [SnesRam_]\n"
  "  add edi, ebx\n"
  "  mov al, [edi]\n"
  "  rol al, 1\n"
  "  mov [edi], al\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
  ".trumem_b\n"
  "  sub edi, 07E0000h\n"
  "  add edi, ebx\n"
  "  add edi, [SnesRam_]\n"
  "  mov al, [edi]\n"
  "  rol al, 1\n"
  "  mov [edi], al\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
}; 

char *rol_w = {        
  "  xor ebx, ebx\n"
  "  mov bx, di\n"
  "  xor di, di\n"
  "  cmp edi, 03F0000h\n"
  "  jg  .trumem_w\n"
  "  mov edi, [SnesRam_]\n"
  "  add edi, ebx\n"
  "  mov ax, [edi]\n"
  "  xchg ah, al\n"
  "  rol ax, 1\n"
  "  xchg ah, al\n"
  "  mov [edi], ax\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
  ".trumem_w\n"
  "  sub edi, 07E0000h\n"
  "  add edi, ebx\n"
  "  add edi, [SnesRam_]\n"
  "  mov ax, [edi]\n"
  "  xchg ah, al\n"
  "  rol ax, 1\n"
  "  xchg ah, al\n"
  "  mov [edi], ax\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
}; 

char *ror_a_b = {
  "  ror BYTE [A_+1], 1\n"
};

char *ror_a_w = {
  "  ror WORD [A_], 1\n"
};

char *ror_b = {
  "  xor ebx, ebx\n"
  "  mov bx, di\n"
  "  xor di, di\n"
  "  cmp edi, 03F0000h\n"
  "  jg  .trumem_b\n"
  "  mov edi, [SnesRam_]\n"
  "  add edi, ebx\n"
  "  mov al, [edi]\n"
  "  ror al, 1\n"
  "  mov [edi], al\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
  ".trumem_b\n"
  "  sub edi, 07E0000h\n"
  "  add edi, ebx\n"
  "  add edi, [SnesRam_]\n"
  "  mov al, [edi]\n"
  "  ror al, 1\n"
  "  mov [edi], al\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
}; 

char *ror_w = {        
  "  xor ebx, ebx\n"
  "  mov bx, di\n"
  "  xor di, di\n"
  "  cmp edi, 03F0000h\n"
  "  jg  .trumem_w\n"
  "  mov edi, [SnesRam_]\n"
  "  add edi, ebx\n"
  "  mov ax, [edi]\n"
  "  xchg ah, al\n"
  "  ror ax, 1\n"
  "  xchg ah, al\n"
  "  mov [edi], ax\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
  ".trumem_w\n"
  "  sub edi, 07E0000h\n"
  "  add edi, ebx\n"
  "  add edi, [SnesRam_]\n"
  "  mov ax, [edi]\n"
  "  xchg ah, al\n"
  "  ror ax, 1\n"
  "  xchg ah, al\n"
  "  mov [edi], ax\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
}; 

// Storing Opcodes (different depending on size)
char *sta_b = {
  "  xor ebx, ebx\n"
  "  mov bx, di\n"
  "  xor di, di\n"
  "  cmp edi, 03F0000h\n"
  "  jg  .trumem_b\n"
  "  cmp bx, 01FFFh\n"
  "  jle .reg_b\n"
  "  mov edi, [SnesRam_]\n"
  "  add edi, ebx\n"
  "  mov ax, [A_]\n"
  "  mov [edi], al\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
  ".reg_b\n"
  "  mov edi, [SnesReg_]\n"
  "  sub bx, 02100h\n"
  "  add edi, ebx\n"
  "  mov ax, [A_]\n"
  "  mov [edi], al\n"
  "  GetFlags\n"
  "  mov [RegValB_], al\n"
  "  mov [RegNum_], bx\n"
  "  call RegFxnB_\n"
  "  jmp .opdone\n"
  ".trumem_b\n"
  "  sub edi, 07E0000h\n"
  "  add edi, ebx\n"
  "  add edi, [SnesRam_]\n"
  "  mov ax, [A_]\n"
  "  mov [edi], al\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
}; 

char *sta_w = {        
  "  xor ebx, ebx\n"
  "  mov bx, di\n"
  "  xor di, di\n"
  "  cmp edi, 03F0000h\n"
  "  jg  .trumem_w\n"
  "  cmp bx, 01FFFh\n"
  "  jle .reg_w\n"
  "  mov edi, [SnesRam_]\n"
  "  add edi, ebx\n"
  "  mov ax, [A_]\n"
  "  xchg al, ah\n"
  "  mov [edi], ax\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
  ".reg_w\n"
  "  mov edi, [SnesReg_]\n"
  "  sub bx, 02100h\n"
  "  add edi, ebx\n"
  "  mov ax, [A_]\n"
  "  xchg ah, al\n"
  "  mov [edi], ax\n"
  "  GetFlags\n"
  "  mov [RegValW_], ax\n"
  "  mov [RegNum_], bx\n"
  "  call RegFxnW_\n"
  "  jmp .opdone\n"
  ".trumem_w\n"
  "  sub edi, 07E0000h\n"
  "  add edi, ebx\n"
  "  add edi, [SnesRam_]\n"
  "  mov ax, [A_]\n"
  "  xchg al, ah\n"
  "  mov [edi], ax\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
}; 

char *stx_b = {
  "  xor ebx, ebx\n"
  "  mov bx, di\n"
  "  xor di, di\n"
  "  cmp edi, 03F0000h\n"
  "  jg  .trumem_b\n"
  "  cmp bx, 01FFFh\n"
  "  jle .reg_b\n"
  "  mov edi, [SnesRam_]\n"
  "  add edi, ebx\n"
  "  mov ax, [X_]\n"
  "  mov [edi], al\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
  ".reg_b\n"
  "  mov edi, [SnesReg_]\n"
  "  sub bx, 02100h\n"
  "  add edi, ebx\n"
  "  mov ax, [X_]\n"
  "  mov [edi], al\n"
  "  GetFlags\n"
  "  mov [RegValB_], al\n"
  "  mov [RegNum_], bx\n"
  "  call RegFxnB_\n"
  "  jmp .opdone\n"
  ".trumem_b\n"
  "  sub edi, 07E0000h\n"
  "  add edi, ebx\n"
  "  add edi, [SnesRam_]\n"
  "  mov ax, [X_]\n"
  "  mov [edi], al\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
}; 

char *stx_w = {        
  "  xor ebx, ebx\n"
  "  mov bx, di\n"
  "  xor di, di\n"
  "  cmp edi, 03F0000h\n"
  "  jg  .trumem_w\n"
  "  cmp bx, 01FFFh\n"
  "  jle .reg_w\n"
  "  mov edi, [SnesRam_]\n"
  "  add edi, ebx\n"
  "  mov ax, [X_]\n"
  "  xchg al, ah\n"
  "  mov [edi], ax\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
  ".reg_w\n"
  "  mov edi, [SnesReg_]\n"
  "  sub bx, 02100h\n"
  "  add edi, ebx\n"
  "  mov ax, [X_]\n"
  "  xchg ah, al\n"
  "  mov [edi], ax\n"
  "  GetFlags\n"
  "  mov [RegValW_], ax\n"
  "  mov [RegNum_], bx\n"
  "  call RegFxnW_\n"
  "  jmp .opdone\n"
  ".trumem_w\n"
  "  sub edi, 07E0000h\n"
  "  add edi, ebx\n"
  "  add edi, [SnesRam_]\n"
  "  mov ax, [X_]\n"
  "  xchg al, ah\n"
  "  mov [edi], ax\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
}; 

char *sty_b = {
  "  xor ebx, ebx\n"
  "  mov bx, di\n"
  "  xor di, di\n"
  "  cmp edi, 03F0000h\n"
  "  jg  .trumem_b\n"
  "  cmp bx, 01FFFh\n"
  "  jle .reg_b\n"
  "  mov edi, [SnesRam_]\n"
  "  add edi, ebx\n"
  "  mov ax, [Y_]\n"
  "  mov [edi], al\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
  ".reg_b\n"
  "  mov edi, [SnesReg_]\n"
  "  sub bx, 02100h\n"
  "  add edi, ebx\n"
  "  mov ax, [Y_]\n"
  "  mov [edi], al\n"
  "  GetFlags\n"
  "  mov [RegValB_], al\n"
  "  mov [RegNum_], bx\n"
  "  call RegFxnB_\n"
  "  jmp .opdone\n"
  ".trumem_b\n"
  "  sub edi, 07E0000h\n"
  "  add edi, ebx\n"
  "  add edi, [SnesRam_]\n"
  "  mov ax, [Y_]\n"
  "  mov [edi], al\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
}; 

char *sty_w = {        
  "  xor ebx, ebx\n"
  "  mov bx, di\n"
  "  xor di, di\n"
  "  cmp edi, 03F0000h\n"
  "  jg  .trumem_w\n"
  "  cmp bx, 01FFFh\n"
  "  jle .reg_w\n"
  "  mov edi, [SnesRam_]\n"
  "  add edi, ebx\n"
  "  mov ax, [Y_]\n"
  "  xchg al, ah\n"
  "  mov [edi], ax\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
  ".reg_w\n"
  "  mov edi, [SnesReg_]\n"
  "  sub bx, 02100h\n"
  "  add edi, ebx\n"
  "  mov ax, [Y_]\n"
  "  xchg ah, al\n"
  "  mov [edi], ax\n"
  "  GetFlags\n"
  "  mov [RegValW_], ax\n"
  "  mov [RegNum_], bx\n"
  "  call RegFxnW_\n"
  "  jmp .opdone\n"
  ".trumem_w\n"
  "  sub edi, 07E0000h\n"
  "  add edi, ebx\n"
  "  add edi, [SnesRam_]\n"
  "  mov ax, [Y_]\n"
  "  xchg al, ah\n"
  "  mov [edi], ax\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
}; 

char *stz_b = {
  "  xor ebx, ebx\n"
  "  mov bx, di\n"
  "  xor di, di\n"
  "  cmp edi, 03F0000h\n"
  "  jg  .trumem_b\n"
  "  cmp bx, 01FFFh\n"
  "  jle .reg_b\n"
  "  mov edi, [SnesRam_]\n"
  "  add edi, ebx\n"
  "  xor ax, ax\n"
  "  mov [edi], al\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
  ".reg_b\n"
  "  mov edi, [SnesReg_]\n"
  "  sub bx, 02100h\n"
  "  add edi, ebx\n"
  "  xor ax, ax\n"
  "  mov [edi], al\n"
  "  GetFlags\n"
  "  mov [RegValB_], al\n"
  "  mov [RegNum_], bx\n"
  "  call RegFxnB_\n"
  "  jmp .opdone\n"
  ".trumem_b\n"
  "  sub edi, 07E0000h\n"
  "  add edi, ebx\n"
  "  add edi, [SnesRam_]\n"
  "  xor ax, ax\n"
  "  mov [edi], al\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
}; 

char *stz_w = {        
  "  xor ebx, ebx\n"
  "  mov bx, di\n"
  "  xor di, di\n"
  "  cmp edi, 03F0000h\n"
  "  jg  .trumem_w\n"
  "  cmp bx, 01FFFh\n"
  "  jle .reg_w\n"
  "  mov edi, [SnesRam_]\n"
  "  add edi, ebx\n"
  "  xor ax, ax\n"
  "  mov [edi], ax\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
  ".reg_w\n"
  "  mov edi, [SnesReg_]\n"
  "  sub bx, 02100h\n"
  "  add edi, ebx\n"
  "  xor ax, ax\n"
  "  mov [edi], ax\n"
  "  GetFlags\n"
  "  mov [RegValW_], ax\n"
  "  mov [RegNum_], bx\n"
  "  call RegFxnW_\n"
  "  jmp .opdone\n"
  ".trumem_w\n"
  "  sub edi, 07E0000h\n"
  "  add edi, ebx\n"
  "  add edi, [SnesRam_]\n"
  "  xor ax, ax\n"
  "  mov [edi], ax\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
}; 

// Yet more special needs opcodes...
char *trb_b = {
  "  xor ebx, ebx\n"
  "  mov bx, di\n"
  "  xor di, di\n"
  "  cmp edi, 03F0000h\n"
  "  jg  .trumem_b\n"
  "  mov edi, [SnesRam_]\n"
  "  add edi, ebx\n"
  "  mov al, [edi]\n"
  "  mov bx, [A_]\n"
  "  not bl\n"
  "  and al, bl\n"
  "  GetFlags\n"
  "  mov [edi], al\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
  ".trumem_b\n"
  "  sub edi, 07E0000h\n"
  "  add edi, ebx\n"
  "  add edi, [SnesRam_]\n"
  "  mov al, [edi]\n"
  "  mov bx, [A_]\n"
  "  not bl\n"
  "  and al, bl\n"
  "  GetFlags\n"
  "  mov [edi], al\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
}; 

char *trb_w = {        
  "  xor ebx, ebx\n"
  "  mov bx, di\n"
  "  xor di, di\n"
  "  cmp edi, 03F0000h\n"
  "  jg  .trumem_w\n"
  "  mov edi, [SnesRam_]\n"
  "  add edi, ebx\n"
  "  mov ax, [edi]\n"
  "  xchg ah, al\n"
  "  mov bx, [A_]\n"
  "  not bx\n"
  "  and ax, bx\n"
  "  GetFlags\n"
  "  xchg ah, al\n"
  "  mov [edi], ax\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
  ".trumem_w\n"
  "  sub edi, 07E0000h\n"
  "  add edi, ebx\n"
  "  add edi, [SnesRam_]\n"
  "  mov ax, [edi]\n"
  "  xchg ah, al\n"
  "  mov bx, [A_]\n"
  "  not bx\n"
  "  and ax, bx\n"
  "  GetFlags\n"
  "  xchg ah, al\n"
  "  mov [edi], ax\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
}; 

char *tsb_b = {
  "  xor ebx, ebx\n"
  "  mov bx, di\n"
  "  xor di, di\n"
  "  cmp edi, 03F0000h\n"
  "  jg  .trumem_b\n"
  "  mov edi, [SnesRam_]\n"
  "  add edi, ebx\n"
  "  mov al, [edi]\n"
  "  mov bx, [A_]\n"
  "  or  al, bl\n"
  "  GetFlags\n"
  "  mov [edi], al\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
  ".trumem_b\n"
  "  sub edi, 07E0000h\n"
  "  add edi, ebx\n"
  "  add edi, [SnesRam_]\n"
  "  mov al, [edi]\n"
  "  mov bx, [A_]\n"
  "  or  al, bl\n"
  "  GetFlags\n"
  "  mov [edi], al\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
}; 

char *tsb_w = {        
  "  xor ebx, ebx\n"
  "  mov bx, di\n"
  "  xor di, di\n"
  "  cmp edi, 03F0000h\n"
  "  jg  .trumem_w\n"
  "  mov edi, [SnesRam_]\n"
  "  add edi, ebx\n"
  "  mov ax, [edi]\n"
  "  xchg ah, al\n"
  "  mov bx, [A_]\n"
  "  or  ax, bx\n"
  "  GetFlags\n"
  "  xchg ah, al\n"
  "  mov [edi], ax\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
  ".trumem_w\n"
  "  sub edi, 07E0000h\n"
  "  add edi, ebx\n"
  "  add edi, [SnesRam_]\n"
  "  mov ax, [edi]\n"
  "  xchg ah, al\n"
  "  mov bx, [A_]\n"
  "  or  ax, bx\n"
  "  GetFlags\n"
  "  xchg ah, al\n"
  "  mov [edi], ax\n"
  "  GetFlags\n"
  "  jmp .opdone\n"
}; 

// size testing
char *CheckSize = {
  "  mov ax, [Pm_]\n"
  "  cmp ax, 1\n"
  "  je  near .bytesize\n"
  "  mov ax, [Px_]\n"
  "  cmp ax, 1\n"
  "  je  near .bytesize\n"
  "  mov ax, [Pe_]\n"
  "  cmp ax, 1\n"
  "  je  near .bytesize\n"
};

FILE *fp;

/////// TIME FOR HUGE FXN'S =) ///////////////////////////////////////////////

void DoADCs()
{
  fprintf(fp, "\n; ADC: Immediate Addressing Mode\n");
  fprintf(fp, "_adc_i_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrImmed);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, adc);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrImmed);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, adc);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ADC: Direct Addressing Mode\n");
  fprintf(fp, "_adc_d_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, adc);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, adc);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ADC: Direct,X Addressing Mode\n");
  fprintf(fp, "_adc_dx_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, adc);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, adc);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ADC: Direct Indirect Addressing Mode\n");
  fprintf(fp, "_adc_pd_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirInd);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, adc);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirInd);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, adc);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ADC: Direct Indirect,X Addressing Mode\n");
  fprintf(fp, "_adc_pdx_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirIndX);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, adc);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirIndX);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, adc);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ADC: Direct Indirect,Y Addressing Mode\n");
  fprintf(fp, "_adc_pd_y_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirIndY);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, adc);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirIndY);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, adc);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ADC: Direct Indirect Long Addressing Mode\n");
  fprintf(fp, "_adc_bd_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirIndL);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, adc);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirIndL);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, adc);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ADC: Absolute Addressing Mode\n");
  fprintf(fp, "_adc_a_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, adc);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, adc);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ADC: Absolute,X Addressing Mode\n");
  fprintf(fp, "_adc_ax_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, adc);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, adc);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ADC: Absolute,Y Addressing Mode\n");
  fprintf(fp, "_adc_y_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbsY);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, adc);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbsY);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, adc);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ADC: Absolute Long Addressing Mode\n");
  fprintf(fp, "_adc_al_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbsL);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, adc);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbsL);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, adc);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ADC: Stack Relative Addressing Mode\n");
  fprintf(fp, "_adc_ds_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrStackRel);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, adc);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrStackRel);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, adc);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ADC: Stack Relative Indirect,Y Addressing Mode\n");
  fprintf(fp, "_adc_pds_y_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrStackRelIndY);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, adc);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrStackRelIndY);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, adc);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");
}

void DoANDs()
{
  fprintf(fp, "\n; and: Immediate Addressing Mode\n");
  fprintf(fp, "_and_i_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrImmed);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, and);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrImmed);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, and);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; and: Direct Addressing Mode\n");
  fprintf(fp, "_and_d_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, and);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, and);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; and: Direct,X Addressing Mode\n");
  fprintf(fp, "_and_dx_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, and);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, and);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; and: Direct Indirect Addressing Mode\n");
  fprintf(fp, "_and_pd_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirInd);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, and);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirInd);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, and);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; and: Direct Indirect,X Addressing Mode\n");
  fprintf(fp, "_and_pdx_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirIndX);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, and);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirIndX);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, and);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; and: Direct Indirect,Y Addressing Mode\n");
  fprintf(fp, "_and_pd_y_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirIndY);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, and);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirIndY);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, and);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; and: Direct Indirect Long Addressing Mode\n");
  fprintf(fp, "_and_bd_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirIndL);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, and);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirIndL);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, and);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; and: Absolute Addressing Mode\n");
  fprintf(fp, "_and_a_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, and);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, and);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; and: Absolute,X Addressing Mode\n");
  fprintf(fp, "_and_ax_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, and);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, and);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; and: Absolute,Y Addressing Mode\n");
  fprintf(fp, "_and_y_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbsY);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, and);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbsY);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, and);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; and: Absolute Long Addressing Mode\n");
  fprintf(fp, "_and_al_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbsL);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, and);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbsL);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, and);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; and: Stack Relative Addressing Mode\n");
  fprintf(fp, "_and_ds_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrStackRel);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, and);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrStackRel);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, and);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; and: Stack Relative Indirect,Y Addressing Mode\n");
  fprintf(fp, "_and_pds_y_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrStackRelIndY);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, and);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrStackRelIndY);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, and);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");
}

void DoBITs()
{
  fprintf(fp, "\n; bit: Immediate Addressing Mode\n");
  fprintf(fp, "_bit_i_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrImmed);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, bit);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrImmed);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, bit);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; bit: Direct Addressing Mode\n");
  fprintf(fp, "_bit_d_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, bit);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, bit);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; bit: Direct,X Addressing Mode\n");
  fprintf(fp, "_bit_dx_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, bit);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, bit);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; bit: Absolute Addressing Mode\n");
  fprintf(fp, "_bit_a_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, bit);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, bit);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; bit: Absolute,X Addressing Mode\n");
  fprintf(fp, "_bit_ax_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, bit);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, bit);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");
}

void DoCMPs()
{
  fprintf(fp, "\n; cmp: Immediate Addressing Mode\n");
  fprintf(fp, "_cmp_i_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opercmp to determine the memory addr\n");
  fprintf(fp, AddrImmed);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; cmp finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cmp);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opercmp to determine the memory addr\n");
  fprintf(fp, AddrImmed);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; cmp finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cmp);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; cmp: Direct Addressing Mode\n");
  fprintf(fp, "_cmp_d_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opercmp to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; cmp finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cmp);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opercmp to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; cmp finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cmp);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; cmp: Direct,X Addressing Mode\n");
  fprintf(fp, "_cmp_dx_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opercmp to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; cmp finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cmp);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opercmp to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; cmp finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cmp);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; cmp: Direct Indirect Addressing Mode\n");
  fprintf(fp, "_cmp_pd_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opercmp to determine the memory addr\n");
  fprintf(fp, AddrDirInd);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; cmp finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cmp);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opercmp to determine the memory addr\n");
  fprintf(fp, AddrDirInd);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; cmp finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cmp);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; cmp: Direct Indirect,X Addressing Mode\n");
  fprintf(fp, "_cmp_pdx_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opercmp to determine the memory addr\n");
  fprintf(fp, AddrDirIndX);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; cmp finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cmp);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opercmp to determine the memory addr\n");
  fprintf(fp, AddrDirIndX);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; cmp finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cmp);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; cmp: Direct Indirect,Y Addressing Mode\n");
  fprintf(fp, "_cmp_pd_y_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opercmp to determine the memory addr\n");
  fprintf(fp, AddrDirIndY);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; cmp finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cmp);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opercmp to determine the memory addr\n");
  fprintf(fp, AddrDirIndY);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; cmp finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cmp);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; cmp: Direct Indirect Long Addressing Mode\n");
  fprintf(fp, "_cmp_bd_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opercmp to determine the memory addr\n");
  fprintf(fp, AddrDirIndL);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; cmp finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cmp);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opercmp to determine the memory addr\n");
  fprintf(fp, AddrDirIndL);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; cmp finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cmp);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; cmp: Absolute Addressing Mode\n");
  fprintf(fp, "_cmp_a_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opercmp to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; cmp finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cmp);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opercmp to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; cmp finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cmp);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; cmp: Absolute,X Addressing Mode\n");
  fprintf(fp, "_cmp_ax_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opercmp to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; cmp finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cmp);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opercmp to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; cmp finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cmp);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; cmp: Absolute,Y Addressing Mode\n");
  fprintf(fp, "_cmp_y_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opercmp to determine the memory addr\n");
  fprintf(fp, AddrAbsY);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; cmp finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cmp);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opercmp to determine the memory addr\n");
  fprintf(fp, AddrAbsY);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; cmp finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cmp);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; cmp: Absolute Long Addressing Mode\n");
  fprintf(fp, "_cmp_al_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opercmp to determine the memory addr\n");
  fprintf(fp, AddrAbsL);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; cmp finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cmp);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opercmp to determine the memory addr\n");
  fprintf(fp, AddrAbsL);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; cmp finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cmp);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; cmp: Stack Relative Addressing Mode\n");
  fprintf(fp, "_cmp_ds_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opercmp to determine the memory addr\n");
  fprintf(fp, AddrStackRel);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; cmp finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cmp);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opercmp to determine the memory addr\n");
  fprintf(fp, AddrStackRel);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; cmp finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cmp);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; cmp: Stack Relative Indirect,Y Addressing Mode\n");
  fprintf(fp, "_cmp_pds_y_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opercmp to determine the memory addr\n");
  fprintf(fp, AddrStackRelIndY);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; cmp finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cmp);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opercmp to determine the memory addr\n");
  fprintf(fp, AddrStackRelIndY);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; cmp finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cmp);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");
}

void DoCPXs()
{
  fprintf(fp, "\n; cpx: Immediate Addressing Mode\n");
  fprintf(fp, "_cpx_i_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opercpx to determine the memory addr\n");
  fprintf(fp, AddrImmed);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; cpx finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cpx);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opercpx to determine the memory addr\n");
  fprintf(fp, AddrImmed);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; cpx finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cpx);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; cpx: Direct Addressing Mode\n");
  fprintf(fp, "_cpx_d_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opercpx to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; cpx finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cpx);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opercpx to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; cpx finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cpx);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; cpx: Absolute Addressing Mode\n");
  fprintf(fp, "_cpx_a_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opercpx to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; cpx finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cpx);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opercpx to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; cpx finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cpx);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");
}

void DoCPYs()
{
  fprintf(fp, "\n; cpy: Immediate Addressing Mode\n");
  fprintf(fp, "_cpy_i_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opercpy to determine the memory addr\n");
  fprintf(fp, AddrImmed);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; cpy finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cpy);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opercpy to determine the memory addr\n");
  fprintf(fp, AddrImmed);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; cpy finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cpy);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; cpy: Direct Addressing Mode\n");
  fprintf(fp, "_cpy_d_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opercpy to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; cpy finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cpy);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opercpy to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; cpy finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cpy);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; cpy: Absolute Addressing Mode\n");
  fprintf(fp, "_cpy_a_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opercpy to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; cpy finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cpy);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opercpy to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; cpy finally we get to the actual fxn of the opcode\n");
  fprintf(fp, cpy);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");
}

void DoEORs()
{
  fprintf(fp, "\n; eor: Immediate Addressing Mode\n");
  fprintf(fp, "_eor_i_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opereor to determine the memory addr\n");
  fprintf(fp, AddrImmed);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; eor finally we get to the actual fxn of the opcode\n");
  fprintf(fp, eor);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opereor to determine the memory addr\n");
  fprintf(fp, AddrImmed);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; eor finally we get to the actual fxn of the opcode\n");
  fprintf(fp, eor);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; eor: Direct Addressing Mode\n");
  fprintf(fp, "_eor_d_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opereor to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; eor finally we get to the actual fxn of the opcode\n");
  fprintf(fp, eor);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opereor to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; eor finally we get to the actual fxn of the opcode\n");
  fprintf(fp, eor);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; eor: Direct,X Addressing Mode\n");
  fprintf(fp, "_eor_dx_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opereor to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; eor finally we get to the actual fxn of the opcode\n");
  fprintf(fp, eor);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opereor to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; eor finally we get to the actual fxn of the opcode\n");
  fprintf(fp, eor);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; eor: Direct Indirect Addressing Mode\n");
  fprintf(fp, "_eor_pd_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opereor to determine the memory addr\n");
  fprintf(fp, AddrDirInd);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; eor finally we get to the actual fxn of the opcode\n");
  fprintf(fp, eor);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opereor to determine the memory addr\n");
  fprintf(fp, AddrDirInd);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; eor finally we get to the actual fxn of the opcode\n");
  fprintf(fp, eor);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; eor: Direct Indirect,X Addressing Mode\n");
  fprintf(fp, "_eor_pdx_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opereor to determine the memory addr\n");
  fprintf(fp, AddrDirIndX);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; eor finally we get to the actual fxn of the opcode\n");
  fprintf(fp, eor);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opereor to determine the memory addr\n");
  fprintf(fp, AddrDirIndX);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; eor finally we get to the actual fxn of the opcode\n");
  fprintf(fp, eor);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; eor: Direct Indirect,Y Addressing Mode\n");
  fprintf(fp, "_eor_pd_y_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opereor to determine the memory addr\n");
  fprintf(fp, AddrDirIndY);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; eor finally we get to the actual fxn of the opcode\n");
  fprintf(fp, eor);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opereor to determine the memory addr\n");
  fprintf(fp, AddrDirIndY);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; eor finally we get to the actual fxn of the opcode\n");
  fprintf(fp, eor);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; eor: Direct Indirect Long Addressing Mode\n");
  fprintf(fp, "_eor_bd_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opereor to determine the memory addr\n");
  fprintf(fp, AddrDirIndL);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; eor finally we get to the actual fxn of the opcode\n");
  fprintf(fp, eor);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opereor to determine the memory addr\n");
  fprintf(fp, AddrDirIndL);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; eor finally we get to the actual fxn of the opcode\n");
  fprintf(fp, eor);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; eor: Absolute Addressing Mode\n");
  fprintf(fp, "_eor_a_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opereor to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; eor finally we get to the actual fxn of the opcode\n");
  fprintf(fp, eor);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opereor to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; eor finally we get to the actual fxn of the opcode\n");
  fprintf(fp, eor);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; eor: Absolute,X Addressing Mode\n");
  fprintf(fp, "_eor_ax_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opereor to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; eor finally we get to the actual fxn of the opcode\n");
  fprintf(fp, eor);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opereor to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; eor finally we get to the actual fxn of the opcode\n");
  fprintf(fp, eor);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; eor: Absolute,Y Addressing Mode\n");
  fprintf(fp, "_eor_y_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opereor to determine the memory addr\n");
  fprintf(fp, AddrAbsY);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; eor finally we get to the actual fxn of the opcode\n");
  fprintf(fp, eor);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opereor to determine the memory addr\n");
  fprintf(fp, AddrAbsY);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; eor finally we get to the actual fxn of the opcode\n");
  fprintf(fp, eor);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; eor: Absolute Long Addressing Mode\n");
  fprintf(fp, "_eor_al_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opereor to determine the memory addr\n");
  fprintf(fp, AddrAbsL);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; eor finally we get to the actual fxn of the opcode\n");
  fprintf(fp, eor);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opereor to determine the memory addr\n");
  fprintf(fp, AddrAbsL);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; eor finally we get to the actual fxn of the opcode\n");
  fprintf(fp, eor);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; eor: Stack Relative Addressing Mode\n");
  fprintf(fp, "_eor_ds_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opereor to determine the memory addr\n");
  fprintf(fp, AddrStackRel);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; eor finally we get to the actual fxn of the opcode\n");
  fprintf(fp, eor);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opereor to determine the memory addr\n");
  fprintf(fp, AddrStackRel);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; eor finally we get to the actual fxn of the opcode\n");
  fprintf(fp, eor);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; eor: Stack Relative Indirect,Y Addressing Mode\n");
  fprintf(fp, "_eor_pds_y_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the opereor to determine the memory addr\n");
  fprintf(fp, AddrStackRelIndY);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; eor finally we get to the actual fxn of the opcode\n");
  fprintf(fp, eor);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the opereor to determine the memory addr\n");
  fprintf(fp, AddrStackRelIndY);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; eor finally we get to the actual fxn of the opcode\n");
  fprintf(fp, eor);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");
}

void DoLDAs()
{
  fprintf(fp, "\n; lda: Immediate Addressing Mode\n");
  fprintf(fp, "_lda_i_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operlda to determine the memory addr\n");
  fprintf(fp, AddrImmed);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; lda finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lda);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operlda to determine the memory addr\n");
  fprintf(fp, AddrImmed);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; lda finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lda);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; lda: Direct Addressing Mode\n");
  fprintf(fp, "_lda_d_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operlda to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; lda finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lda);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operlda to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; lda finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lda);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; lda: Direct,X Addressing Mode\n");
  fprintf(fp, "_lda_dx_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operlda to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; lda finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lda);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operlda to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; lda finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lda);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; lda: Direct Indirect Addressing Mode\n");
  fprintf(fp, "_lda_pd_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operlda to determine the memory addr\n");
  fprintf(fp, AddrDirInd);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; lda finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lda);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operlda to determine the memory addr\n");
  fprintf(fp, AddrDirInd);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; lda finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lda);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; lda: Direct Indirect,X Addressing Mode\n");
  fprintf(fp, "_lda_pdx_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operlda to determine the memory addr\n");
  fprintf(fp, AddrDirIndX);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; lda finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lda);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operlda to determine the memory addr\n");
  fprintf(fp, AddrDirIndX);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; lda finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lda);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; lda: Direct Indirect,Y Addressing Mode\n");
  fprintf(fp, "_lda_pd_y_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operlda to determine the memory addr\n");
  fprintf(fp, AddrDirIndY);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; lda finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lda);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operlda to determine the memory addr\n");
  fprintf(fp, AddrDirIndY);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; lda finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lda);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; lda: Direct Indirect Long Addressing Mode\n");
  fprintf(fp, "_lda_bd_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operlda to determine the memory addr\n");
  fprintf(fp, AddrDirIndL);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; lda finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lda);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operlda to determine the memory addr\n");
  fprintf(fp, AddrDirIndL);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; lda finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lda);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; lda: Absolute Addressing Mode\n");
  fprintf(fp, "_lda_a_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operlda to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; lda finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lda);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operlda to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; lda finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lda);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; lda: Absolute,X Addressing Mode\n");
  fprintf(fp, "_lda_ax_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operlda to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; lda finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lda);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operlda to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; lda finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lda);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; lda: Absolute,Y Addressing Mode\n");
  fprintf(fp, "_lda_y_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operlda to determine the memory addr\n");
  fprintf(fp, AddrAbsY);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; lda finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lda);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operlda to determine the memory addr\n");
  fprintf(fp, AddrAbsY);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; lda finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lda);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; lda: Absolute Long Addressing Mode\n");
  fprintf(fp, "_lda_al_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operlda to determine the memory addr\n");
  fprintf(fp, AddrAbsL);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; lda finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lda);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operlda to determine the memory addr\n");
  fprintf(fp, AddrAbsL);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; lda finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lda);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; lda: Stack Relative Addressing Mode\n");
  fprintf(fp, "_lda_ds_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operlda to determine the memory addr\n");
  fprintf(fp, AddrStackRel);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; lda finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lda);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operlda to determine the memory addr\n");
  fprintf(fp, AddrStackRel);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; lda finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lda);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; lda: Stack Relative Indirect,Y Addressing Mode\n");
  fprintf(fp, "_lda_pds_y_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operlda to determine the memory addr\n");
  fprintf(fp, AddrStackRelIndY);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; lda finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lda);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operlda to determine the memory addr\n");
  fprintf(fp, AddrStackRelIndY);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; lda finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lda);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");
}

void DoLDXs()
{
  fprintf(fp, "\n; ldx: Immediate Addressing Mode\n");
  fprintf(fp, "_ldx_i_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operldx to determine the memory addr\n");
  fprintf(fp, AddrImmed);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; ldx finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ldx);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operldx to determine the memory addr\n");
  fprintf(fp, AddrImmed);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; ldx finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ldx);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ldx: Direct Addressing Mode\n");
  fprintf(fp, "_ldx_d_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operldx to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; ldx finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ldx);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operldx to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; ldx finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ldx);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ldx: Direct,Y Addressing Mode\n");
  fprintf(fp, "_ldx_dy_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operldx to determine the memory addr\n");
  fprintf(fp, AddrDirY);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; ldx finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ldx);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operldx to determine the memory addr\n");
  fprintf(fp, AddrDirY);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; ldx finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ldx);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ldx: Absolute Addressing Mode\n");
  fprintf(fp, "_ldx_a_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operldx to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; ldx finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ldx);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operldx to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; ldx finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ldx);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ldx: Absolute,X Addressing Mode\n");
  fprintf(fp, "_ldx_ax_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operldx to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; ldx finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ldx);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operldx to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; ldx finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ldx);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");
}

void DoLDYs()
{
  fprintf(fp, "\n; ldy: Immediate Addressing Mode\n");
  fprintf(fp, "_ldy_i_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operldy to determine the memory addr\n");
  fprintf(fp, AddrImmed);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; ldy finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ldy);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operldy to determine the memory addr\n");
  fprintf(fp, AddrImmed);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; ldy finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ldy);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ldy: Direct Addressing Mode\n");
  fprintf(fp, "_ldy_d_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operldy to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; ldy finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ldy);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operldy to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; ldy finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ldy);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ldy: Direct,X Addressing Mode\n");
  fprintf(fp, "_ldy_dx_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operldy to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; ldy finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ldy);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operldy to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; ldy finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ldy);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ldy: Absolute Addressing Mode\n");
  fprintf(fp, "_ldy_a_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operldy to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; ldy finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ldy);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operldy to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; ldy finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ldy);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ldy: Absolute,X Addressing Mode\n");
  fprintf(fp, "_ldy_ax_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operldy to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; ldy finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ldy);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operldy to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; ldy finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ldy);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");
}

void DoORAs()
{
  fprintf(fp, "\n; ora: Immediate Addressing Mode\n");
  fprintf(fp, "_ora_i_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operora to determine the memory addr\n");
  fprintf(fp, AddrImmed);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; ora finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ora);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operora to determine the memory addr\n");
  fprintf(fp, AddrImmed);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; ora finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ora);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ora: Direct Addressing Mode\n");
  fprintf(fp, "_ora_d_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operora to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; ora finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ora);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operora to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; ora finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ora);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ora: Direct,X Addressing Mode\n");
  fprintf(fp, "_ora_dx_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operora to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; ora finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ora);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operora to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; ora finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ora);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ora: Direct Indirect Addressing Mode\n");
  fprintf(fp, "_ora_pd_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operora to determine the memory addr\n");
  fprintf(fp, AddrDirInd);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; ora finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ora);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operora to determine the memory addr\n");
  fprintf(fp, AddrDirInd);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; ora finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ora);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ora: Direct Indirect,X Addressing Mode\n");
  fprintf(fp, "_ora_pdx_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operora to determine the memory addr\n");
  fprintf(fp, AddrDirIndX);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; ora finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ora);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operora to determine the memory addr\n");
  fprintf(fp, AddrDirIndX);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; ora finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ora);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ora: Direct Indirect,Y Addressing Mode\n");
  fprintf(fp, "_ora_pd_y_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operora to determine the memory addr\n");
  fprintf(fp, AddrDirIndY);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; ora finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ora);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operora to determine the memory addr\n");
  fprintf(fp, AddrDirIndY);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; ora finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ora);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ora: Direct Indirect Long Addressing Mode\n");
  fprintf(fp, "_ora_bd_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operora to determine the memory addr\n");
  fprintf(fp, AddrDirIndL);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; ora finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ora);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operora to determine the memory addr\n");
  fprintf(fp, AddrDirIndL);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; ora finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ora);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ora: Absolute Addressing Mode\n");
  fprintf(fp, "_ora_a_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operora to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; ora finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ora);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operora to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; ora finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ora);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ora: Absolute,X Addressing Mode\n");
  fprintf(fp, "_ora_ax_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operora to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; ora finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ora);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operora to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; ora finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ora);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ora: Absolute,Y Addressing Mode\n");
  fprintf(fp, "_ora_y_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operora to determine the memory addr\n");
  fprintf(fp, AddrAbsY);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; ora finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ora);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operora to determine the memory addr\n");
  fprintf(fp, AddrAbsY);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; ora finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ora);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ora: Absolute Long Addressing Mode\n");
  fprintf(fp, "_ora_al_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operora to determine the memory addr\n");
  fprintf(fp, AddrAbsL);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; ora finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ora);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operora to determine the memory addr\n");
  fprintf(fp, AddrAbsL);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; ora finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ora);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ora: Stack Relative Addressing Mode\n");
  fprintf(fp, "_ora_ds_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operora to determine the memory addr\n");
  fprintf(fp, AddrStackRel);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; ora finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ora);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operora to determine the memory addr\n");
  fprintf(fp, AddrStackRel);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; ora finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ora);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ora: Stack Relative Indirect,Y Addressing Mode\n");
  fprintf(fp, "_ora_pds_y_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operora to determine the memory addr\n");
  fprintf(fp, AddrStackRelIndY);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; ora finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ora);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operora to determine the memory addr\n");
  fprintf(fp, AddrStackRelIndY);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; ora finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ora);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");
}

void DoSBCs()
{
  fprintf(fp, "\n; sbc: Immediate Addressing Mode\n");
  fprintf(fp, "_sbc_i_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrImmed);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sbc);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrImmed);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sbc);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; sbc: Direct Addressing Mode\n");
  fprintf(fp, "_sbc_d_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sbc);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sbc);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; sbc: Direct,X Addressing Mode\n");
  fprintf(fp, "_sbc_dx_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sbc);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sbc);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; sbc: Direct Indirect Addressing Mode\n");
  fprintf(fp, "_sbc_pd_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirInd);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sbc);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirInd);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sbc);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; sbc: Direct Indirect,X Addressing Mode\n");
  fprintf(fp, "_sbc_pdx_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirIndX);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sbc);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirIndX);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sbc);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; sbc: Direct Indirect,Y Addressing Mode\n");
  fprintf(fp, "_sbc_pd_y_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirIndY);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sbc);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirIndY);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sbc);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; sbc: Direct Indirect Long Addressing Mode\n");
  fprintf(fp, "_sbc_bd_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirIndL);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sbc);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirIndL);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sbc);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; sbc: Absolute Addressing Mode\n");
  fprintf(fp, "_sbc_a_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sbc);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sbc);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; sbc: Absolute,X Addressing Mode\n");
  fprintf(fp, "_sbc_ax_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sbc);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sbc);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; sbc: Absolute,Y Addressing Mode\n");
  fprintf(fp, "_sbc_y_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbsY);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sbc);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbsY);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sbc);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; sbc: Absolute Long Addressing Mode\n");
  fprintf(fp, "_sbc_al_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbsL);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sbc);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbsL);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sbc);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; sbc: Stack Relative Addressing Mode\n");
  fprintf(fp, "_sbc_ds_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrStackRel);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sbc);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrStackRel);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sbc);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; sbc: Stack Relative Indirect,Y Addressing Mode\n");
  fprintf(fp, "_sbc_pds_y_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrStackRelIndY);
  fprintf(fp, GetWord);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sbc);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrStackRelIndY);
  fprintf(fp, GetByte);
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sbc);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");
}

void DoASLs()
{
  fprintf(fp, "\n; asl: Accumulator Addressing Mode\n");
  fprintf(fp, "_asl_A_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, asl_a_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n\n");
  fprintf(fp, asl_a_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; asl: Direct Addressing Mode\n");
  fprintf(fp, "_asl_d_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, asl_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, asl_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; asl: Direct,X Addressing Mode\n");
  fprintf(fp, "_asl_dx_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, asl_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, asl_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; asl: Absolute Addressing Mode\n");
  fprintf(fp, "_asl_a_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, asl_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, asl_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; asl: Absolute,X Addressing Mode\n");
  fprintf(fp, "_asl_ax_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, asl_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, asl_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");
}

void DoDECs()
{
  fprintf(fp, "\n; dec: Accumulator Addressing Mode\n");
  fprintf(fp, "_dec_A_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, dec_a_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n\n");
  fprintf(fp, dec_a_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; dec: Direct Addressing Mode\n");
  fprintf(fp, "_dec_d_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, dec_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, dec_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; dec: Direct,X Addressing Mode\n");
  fprintf(fp, "_dec_dx_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, dec_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, dec_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; dec: Absolute Addressing Mode\n");
  fprintf(fp, "_dec_a_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, dec_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, dec_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; dec: Absolute,X Addressing Mode\n");
  fprintf(fp, "_dec_ax_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, dec_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, dec_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");
}

void DoINCs()
{
  fprintf(fp, "\n; inc: Accumulator Addressing Mode\n");
  fprintf(fp, "_inc_A_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, inc_a_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n\n");
  fprintf(fp, inc_a_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; inc: Direct Addressing Mode\n");
  fprintf(fp, "_inc_d_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, inc_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, inc_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; inc: Direct,X Addressing Mode\n");
  fprintf(fp, "_inc_dx_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, inc_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, inc_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; inc: Absolute Addressing Mode\n");
  fprintf(fp, "_inc_a_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, inc_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, inc_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; inc: Absolute,X Addressing Mode\n");
  fprintf(fp, "_inc_ax_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, inc_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, inc_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");
}

void DoLSRs()
{
  fprintf(fp, "\n; lsr: Accumulator Addressing Mode\n");
  fprintf(fp, "_lsr_A_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lsr_a_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n\n");
  fprintf(fp, lsr_a_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; lsr: Direct Addressing Mode\n");
  fprintf(fp, "_lsr_d_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lsr_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lsr_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; lsr: Direct,X Addressing Mode\n");
  fprintf(fp, "_lsr_dx_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lsr_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lsr_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; lsr: Absolute Addressing Mode\n");
  fprintf(fp, "_lsr_a_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lsr_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lsr_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; lsr: Absolute,X Addressing Mode\n");
  fprintf(fp, "_lsr_ax_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lsr_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, lsr_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");
}

void DoROLs()
{
  fprintf(fp, "\n; rol: Accumulator Addressing Mode\n");
  fprintf(fp, "_rol_A_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, rol_a_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n\n");
  fprintf(fp, rol_a_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; rol: Direct Addressing Mode\n");
  fprintf(fp, "_rol_d_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, rol_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, rol_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; rol: Direct,X Addressing Mode\n");
  fprintf(fp, "_rol_dx_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, rol_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, rol_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; rol: Absolute Addressing Mode\n");
  fprintf(fp, "_rol_a_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, rol_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, rol_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; rol: Absolute,X Addressing Mode\n");
  fprintf(fp, "_rol_ax_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, rol_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, rol_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");
}

void DoRORs()
{
  fprintf(fp, "\n; ror: Accumulator Addressing Mode\n");
  fprintf(fp, "_ror_A_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ror_a_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n\n");
  fprintf(fp, ror_a_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ror: Direct Addressing Mode\n");
  fprintf(fp, "_ror_d_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ror_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrDir);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ror_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ror: Direct,X Addressing Mode\n");
  fprintf(fp, "_ror_dx_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ror_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ror_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ror: Absolute Addressing Mode\n");
  fprintf(fp, "_ror_a_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ror_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ror_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; ror: Absolute,X Addressing Mode\n");
  fprintf(fp, "_ror_ax_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ror_w);
  fprintf(fp, "  jmp .opdone\n");
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operbit to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  fprintf(fp, "\n  ; bit finally we get to the actual fxn of the opcode\n");
  fprintf(fp, ror_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");
}

void DoSTAs()
{
  fprintf(fp, "\n; : Direct Addressing Mode\n");
  fprintf(fp, "_sta_d_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDir);

  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sta_w);
  
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDir);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sta_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; : Direct,X Addressing Mode\n");
  fprintf(fp, "_sta_dx_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sta_w);
  
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sta_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; : Direct Indirect Addressing Mode\n");
  fprintf(fp, "_sta_pd_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirInd);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sta_w);
  
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirInd);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sta_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; : Direct Indirect,X Addressing Mode\n");
  fprintf(fp, "_sta_pdx_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirIndX);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sta_w);
  
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirIndX);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sta_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; : Direct Indirect,Y Addressing Mode\n");
  fprintf(fp, "_sta_pd_y_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirIndY);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sta_w);
  
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirIndY);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sta_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; : Direct Indirect Long Addressing Mode\n");
  fprintf(fp, "_sta_bd_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirIndL);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sta_w);
  
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirIndL);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sta_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; : Absolute Addressing Mode\n");
  fprintf(fp, "_sta_a_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sta_w);
  
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sta_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; : Absolute,X Addressing Mode\n");
  fprintf(fp, "_sta_ax_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sta_w);
  
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sta_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; : Absolute,Y Addressing Mode\n");
  fprintf(fp, "_sta_y_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbsY);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sta_w);
  
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbsY);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sta_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; : Absolute Long Addressing Mode\n");
  fprintf(fp, "_sta_al_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbsL);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sta_w);
  
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbsL);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sta_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; : Stack Relative Addressing Mode\n");
  fprintf(fp, "_sta_ds_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrStackRel);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sta_w);
  
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrStackRel);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sta_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; : Stack Relative Indirect,Y Addressing Mode\n");
  fprintf(fp, "_sta_pds_y_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrStackRelIndY);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sta_w);
  
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrStackRelIndY);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sta_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");
}

void DoSTXs()
{
  fprintf(fp, "\n; : Direct Addressing Mode\n");
  fprintf(fp, "_stx_d_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDir);

  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, stx_w);
  
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDir);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, stx_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; : Direct,Y Addressing Mode\n");
  fprintf(fp, "_stx_dy_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirY);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, stx_w);
  
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirY);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, stx_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; : Absolute Addressing Mode\n");
  fprintf(fp, "_stx_a_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, stx_w);
  
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, stx_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");
}

void DoSTYs()
{
  fprintf(fp, "\n; : Direct Addressing Mode\n");
  fprintf(fp, "_sty_d_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDir);

  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sty_w);
  
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDir);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sty_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; : Direct,X Addressing Mode\n");
  fprintf(fp, "_sty_dx_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sty_w);
  
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sty_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; : Absolute Addressing Mode\n");
  fprintf(fp, "_sty_a_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sty_w);
  
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, sty_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");
}

void DoSTZs()
{
  fprintf(fp, "\n; : Direct Addressing Mode\n");
  fprintf(fp, "_stz_d_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDir);

  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, stz_w);
  
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDir);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, stz_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; : Direct,X Addressing Mode\n");
  fprintf(fp, "_stz_dx_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, stz_w);
  
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDirX);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, stz_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; : Absolute Addressing Mode\n");
  fprintf(fp, "_stz_a_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, stz_w);
  
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, stz_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; : Absolute,X Addressing Mode\n");
  fprintf(fp, "_stz_ax_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, stz_w);
  
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbsX);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, stz_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");
}

void DoTRBs()
{
  fprintf(fp, "\n; : Direct Addressing Mode\n");
  fprintf(fp, "_trb_d_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDir);

  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, trb_w);
  
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDir);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, trb_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; : Absolute Addressing Mode\n");
  fprintf(fp, "_trb_a_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, trb_w);
  
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, trb_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");
}

void DoTSBs()
{
  fprintf(fp, "\n; : Direct Addressing Mode\n");
  fprintf(fp, "_tsb_d_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDir);

  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, tsb_w);
  
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrDir);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, tsb_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");

  fprintf(fp, "\n; : Absolute Addressing Mode\n");
  fprintf(fp, "_tsb_a_:\n");
  fprintf(fp, "  ; save all used registers\n");
  fprintf(fp, StartFxn);
  fprintf(fp, "\n  ; Is the A register a byte or word in size?\n");
  fprintf(fp, CheckSize);
  fprintf(fp, "\n  ; it's a word, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, tsb_w);
  
  fprintf(fp, ".bytesize\n");
  fprintf(fp, "\n  ; it's a byte, get the operand to determine the memory addr\n");
  fprintf(fp, AddrAbs);
  
  fprintf(fp, "\n  ; and finally we get to the actual fxn of the opcode\n");
  fprintf(fp, tsb_b);
  fprintf(fp, ".opdone\n");
  fprintf(fp, "\n  ; Restore all used registers\n");
  fprintf(fp, EndFxn);
  fprintf(fp, "  ret\n");
}

void main(void)
{
  fp = fopen("opcodes.asm", "wt");

  fprintf(fp, Beginning);
  fprintf(fp, SnesMap);
  fprintf(fp, "\n");
  fprintf(fp, SetFlags);
  fprintf(fp, "\n");
  fprintf(fp, GetFlags);
  fprintf(fp, "\n");

  DoADCs();
  DoANDs();
  DoASLs();
  DoBITs();
  DoCMPs();
  DoCPXs();
  DoCPYs();
  DoDECs();
  DoEORs();
  DoINCs();
  DoLDAs();
  DoLDXs();
  DoLDYs();
  DoLSRs();
  DoORAs();
  DoROLs();
  DoRORs();
  DoSBCs();
  DoSTAs();
  DoSTXs();
  DoSTYs();
  DoSTZs();
  DoTRBs();
  DoTSBs();

  fclose(fp);
}

