BITS 32

EXTERN SnesRom_      ; Pointer to the SNES's ROM 00-3f:8000-ffff, 40-7d:0000-ffff
EXTERN SnesRam_      ; Pointer to the SNES's RAM 00-3f:0000-1fff, 7e-7f:0000-ffff
EXTERN SnesReg_      ; Pointer to the SNES's register memory 00-3f:2100-4fff
EXTERN IP_           ; the instruction pointer
EXTERN A_            ; The A register (general purpose)
EXTERN X_            ; The X register (general purpose)
EXTERN Y_            ; The Y register (general purpose)
EXTERN DB_           ; Data bank register
EXTERN P_            ; The processor status byte (actually a word, I use smsw
                     ; and lmsw so I don't have to manualy keep track of the
                     ; math status bits)
EXTERN Pe_           ; The emulation flag
EXTERN D_            ; Direct page register
EXTERN Pi_           ; The interrupt flag
EXTERN S_            ; The stack pointer (bank is always 0)
EXTERN Pm_           ; Used to determine the size of A
EXTERN Px_           ; Used to determine the size of X,Y

EXTERN RegFxnB_      ; Register Function Called for byte register writes
EXTERN RegFxnW_      ; same 'cept for words

EXTERN RegValB_      ; hold the value of Byte registers
EXTERN RegValW_      ; lalala
EXTERN RegNum_       ; The register being written to

SECTION .text

SnesMap_:
  mov eax, edi
  xor ebx, ebx
  mov bx, ax
  and eax, 07F0000h
  cmp eax, 03FFFFFh
  jg  .highmem
  cmp bx, 08000h
  jl  .notrom
  shr eax, 1
  and ebx, 07FFFh
  or  eax, ebx
  mov edi, eax
  add edi, [SnesRom_]
  jmp .done
.notrom
  cmp bx, 01FFFh
  jg  .reg
  mov edi, ebx
  add edi, [SnesRam_]
  jmp .done
.reg
  sub bx, 02100h
  mov edi, ebx
  add edi, [SnesReg_]
  jmp .done
.highmem
  cmp eax, 07E0000h
  jl  .done
  mov edi, [SnesRam_]
  add edi, ebx
.done
  ret

%macro SetFlags 0
  mov ax, [P_]
  push ax
  popf
%endmacro

%macro GetFlags 0
  pushf
  pop ax
  mov [P_], ax
%endmacro


; ADC: Immediate Addressing Mode
_adc_i_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov edi, [esi]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  adc bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov edi, [esi]
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  adc bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ADC: Direct Addressing Mode
_adc_d_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  adc bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  adc bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ADC: Direct,X Addressing Mode
_adc_dx_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  adc bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  adc bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ADC: Direct Indirect Addressing Mode
_adc_pd_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  adc bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  adc bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ADC: Direct Indirect,X Addressing Mode
_adc_pdx_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [Y_]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  adc bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [Y_]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  adc bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ADC: Direct Indirect,Y Addressing Mode
_adc_pd_y_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor ax, ax
  mov ax, [Y_]
  add edi, eax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  adc bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor ax, ax
  mov ax, [Y_]
  add edi, eax
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  adc bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ADC: Direct Indirect Long Addressing Mode
_adc_bd_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  adc bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  adc bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ADC: Absolute Addressing Mode
_adc_a_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  adc bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  adc bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ADC: Absolute,X Addressing Mode
_adc_ax_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  adc bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  adc bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ADC: Absolute,Y Addressing Mode
_adc_y_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [Y_]
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  adc bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [Y_]
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  adc bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ADC: Absolute Long Addressing Mode
_adc_al_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  inc esi
  mov di, ax
  xor eax, eax
  mov al, [esi]
  shl eax, 16
  add edi, eax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  adc bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  inc esi
  mov di, ax
  xor eax, eax
  mov al, [esi]
  shl eax, 16
  add edi, eax
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  adc bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ADC: Stack Relative Addressing Mode
_adc_ds_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  adc bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  adc bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ADC: Stack Relative Indirect,Y Addressing Mode
_adc_pds_y_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor eax, eax
  mov ax, [Y_]
  add edi, eax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  adc bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor eax, eax
  mov ax, [Y_]
  add edi, eax
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  adc bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; and: Immediate Addressing Mode
_and_i_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov edi, [esi]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov edi, [esi]
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; and: Direct Addressing Mode
_and_d_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; and: Direct,X Addressing Mode
_and_dx_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; and: Direct Indirect Addressing Mode
_and_pd_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; and: Direct Indirect,X Addressing Mode
_and_pdx_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [Y_]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [Y_]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; and: Direct Indirect,Y Addressing Mode
_and_pd_y_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor ax, ax
  mov ax, [Y_]
  add edi, eax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor ax, ax
  mov ax, [Y_]
  add edi, eax
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; and: Direct Indirect Long Addressing Mode
_and_bd_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; and: Absolute Addressing Mode
_and_a_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; and: Absolute,X Addressing Mode
_and_ax_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; and: Absolute,Y Addressing Mode
_and_y_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [Y_]
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [Y_]
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; and: Absolute Long Addressing Mode
_and_al_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  inc esi
  mov di, ax
  xor eax, eax
  mov al, [esi]
  shl eax, 16
  add edi, eax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  inc esi
  mov di, ax
  xor eax, eax
  mov al, [esi]
  shl eax, 16
  add edi, eax
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; and: Stack Relative Addressing Mode
_and_ds_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; and: Stack Relative Indirect,Y Addressing Mode
_and_pds_y_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor eax, eax
  mov ax, [Y_]
  add edi, eax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor eax, eax
  mov ax, [Y_]
  add edi, eax
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; asl: Accumulator Addressing Mode
_asl_A_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; bit finally we get to the actual fxn of the opcode
  shl WORD [A_], 1
  jmp .opdone
.bytesize

  shl BYTE [A_+1], 1
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; asl: Direct Addressing Mode
_asl_d_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [edi]
  xchg ah, al
  shl ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [edi]
  xchg ah, al
  shl ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov al, [edi]
  shl al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov al, [edi]
  shl al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; asl: Direct,X Addressing Mode
_asl_dx_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [edi]
  xchg ah, al
  shl ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [edi]
  xchg ah, al
  shl ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov al, [edi]
  shl al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov al, [edi]
  shl al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; asl: Absolute Addressing Mode
_asl_a_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [edi]
  xchg ah, al
  shl ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [edi]
  xchg ah, al
  shl ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov al, [edi]
  shl al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov al, [edi]
  shl al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; asl: Absolute,X Addressing Mode
_asl_ax_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [edi]
  xchg ah, al
  shl ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [edi]
  xchg ah, al
  shl ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov al, [edi]
  shl al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov al, [edi]
  shl al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; bit: Immediate Addressing Mode
_bit_i_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov edi, [esi]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; bit finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov edi, [esi]
  call SnesMap_
  mov al, [edi]
  cbw

  ; bit finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; bit: Direct Addressing Mode
_bit_d_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; bit finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; bit finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; bit: Direct,X Addressing Mode
_bit_dx_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; bit finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax
  call SnesMap_
  mov al, [edi]
  cbw

  ; bit finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; bit: Absolute Addressing Mode
_bit_a_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; bit finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; bit finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; bit: Absolute,X Addressing Mode
_bit_ax_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; bit finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; bit finally we get to the actual fxn of the opcode
  mov bx, [A_]
  and bx, ax
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; cmp: Immediate Addressing Mode
_cmp_i_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opercmp to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov edi, [esi]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; cmp finally we get to the actual fxn of the opcode
  mov bx, [A_]
  cmp bx, ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the opercmp to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov edi, [esi]
  call SnesMap_
  mov al, [edi]
  cbw

  ; cmp finally we get to the actual fxn of the opcode
  mov bx, [A_]
  cmp bx, ax
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; cmp: Direct Addressing Mode
_cmp_d_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opercmp to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; cmp finally we get to the actual fxn of the opcode
  mov bx, [A_]
  cmp bx, ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the opercmp to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; cmp finally we get to the actual fxn of the opcode
  mov bx, [A_]
  cmp bx, ax
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; cmp: Direct,X Addressing Mode
_cmp_dx_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opercmp to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; cmp finally we get to the actual fxn of the opcode
  mov bx, [A_]
  cmp bx, ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the opercmp to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax
  call SnesMap_
  mov al, [edi]
  cbw

  ; cmp finally we get to the actual fxn of the opcode
  mov bx, [A_]
  cmp bx, ax
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; cmp: Direct Indirect Addressing Mode
_cmp_pd_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opercmp to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; cmp finally we get to the actual fxn of the opcode
  mov bx, [A_]
  cmp bx, ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the opercmp to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; cmp finally we get to the actual fxn of the opcode
  mov bx, [A_]
  cmp bx, ax
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; cmp: Direct Indirect,X Addressing Mode
_cmp_pdx_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opercmp to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [Y_]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; cmp finally we get to the actual fxn of the opcode
  mov bx, [A_]
  cmp bx, ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the opercmp to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [Y_]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; cmp finally we get to the actual fxn of the opcode
  mov bx, [A_]
  cmp bx, ax
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; cmp: Direct Indirect,Y Addressing Mode
_cmp_pd_y_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opercmp to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor ax, ax
  mov ax, [Y_]
  add edi, eax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; cmp finally we get to the actual fxn of the opcode
  mov bx, [A_]
  cmp bx, ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the opercmp to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor ax, ax
  mov ax, [Y_]
  add edi, eax
  call SnesMap_
  mov al, [edi]
  cbw

  ; cmp finally we get to the actual fxn of the opcode
  mov bx, [A_]
  cmp bx, ax
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; cmp: Direct Indirect Long Addressing Mode
_cmp_bd_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opercmp to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; cmp finally we get to the actual fxn of the opcode
  mov bx, [A_]
  cmp bx, ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the opercmp to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  call SnesMap_
  mov al, [edi]
  cbw

  ; cmp finally we get to the actual fxn of the opcode
  mov bx, [A_]
  cmp bx, ax
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; cmp: Absolute Addressing Mode
_cmp_a_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opercmp to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; cmp finally we get to the actual fxn of the opcode
  mov bx, [A_]
  cmp bx, ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the opercmp to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; cmp finally we get to the actual fxn of the opcode
  mov bx, [A_]
  cmp bx, ax
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; cmp: Absolute,X Addressing Mode
_cmp_ax_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opercmp to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; cmp finally we get to the actual fxn of the opcode
  mov bx, [A_]
  cmp bx, ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the opercmp to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; cmp finally we get to the actual fxn of the opcode
  mov bx, [A_]
  cmp bx, ax
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; cmp: Absolute,Y Addressing Mode
_cmp_y_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opercmp to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [Y_]
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; cmp finally we get to the actual fxn of the opcode
  mov bx, [A_]
  cmp bx, ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the opercmp to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [Y_]
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; cmp finally we get to the actual fxn of the opcode
  mov bx, [A_]
  cmp bx, ax
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; cmp: Absolute Long Addressing Mode
_cmp_al_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opercmp to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  inc esi
  mov di, ax
  xor eax, eax
  mov al, [esi]
  shl eax, 16
  add edi, eax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; cmp finally we get to the actual fxn of the opcode
  mov bx, [A_]
  cmp bx, ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the opercmp to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  inc esi
  mov di, ax
  xor eax, eax
  mov al, [esi]
  shl eax, 16
  add edi, eax
  call SnesMap_
  mov al, [edi]
  cbw

  ; cmp finally we get to the actual fxn of the opcode
  mov bx, [A_]
  cmp bx, ax
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; cmp: Stack Relative Addressing Mode
_cmp_ds_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opercmp to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; cmp finally we get to the actual fxn of the opcode
  mov bx, [A_]
  cmp bx, ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the opercmp to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  call SnesMap_
  mov al, [edi]
  cbw

  ; cmp finally we get to the actual fxn of the opcode
  mov bx, [A_]
  cmp bx, ax
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; cmp: Stack Relative Indirect,Y Addressing Mode
_cmp_pds_y_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opercmp to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor eax, eax
  mov ax, [Y_]
  add edi, eax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; cmp finally we get to the actual fxn of the opcode
  mov bx, [A_]
  cmp bx, ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the opercmp to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor eax, eax
  mov ax, [Y_]
  add edi, eax
  call SnesMap_
  mov al, [edi]
  cbw

  ; cmp finally we get to the actual fxn of the opcode
  mov bx, [A_]
  cmp bx, ax
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; cpx: Immediate Addressing Mode
_cpx_i_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opercpx to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov edi, [esi]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; cpx finally we get to the actual fxn of the opcode
  mov bx, [X_]
  cmp bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the opercpx to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov edi, [esi]
  call SnesMap_
  mov al, [edi]
  cbw

  ; cpx finally we get to the actual fxn of the opcode
  mov bx, [X_]
  cmp bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; cpx: Direct Addressing Mode
_cpx_d_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opercpx to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; cpx finally we get to the actual fxn of the opcode
  mov bx, [X_]
  cmp bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the opercpx to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; cpx finally we get to the actual fxn of the opcode
  mov bx, [X_]
  cmp bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; cpx: Absolute Addressing Mode
_cpx_a_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opercpx to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; cpx finally we get to the actual fxn of the opcode
  mov bx, [X_]
  cmp bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the opercpx to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; cpx finally we get to the actual fxn of the opcode
  mov bx, [X_]
  cmp bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; cpy: Immediate Addressing Mode
_cpy_i_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opercpy to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov edi, [esi]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; cpy finally we get to the actual fxn of the opcode
  mov bx, [Y_]
  cmp bx, ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the opercpy to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov edi, [esi]
  call SnesMap_
  mov al, [edi]
  cbw

  ; cpy finally we get to the actual fxn of the opcode
  mov bx, [Y_]
  cmp bx, ax
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; cpy: Direct Addressing Mode
_cpy_d_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opercpy to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; cpy finally we get to the actual fxn of the opcode
  mov bx, [Y_]
  cmp bx, ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the opercpy to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; cpy finally we get to the actual fxn of the opcode
  mov bx, [Y_]
  cmp bx, ax
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; cpy: Absolute Addressing Mode
_cpy_a_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opercpy to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; cpy finally we get to the actual fxn of the opcode
  mov bx, [Y_]
  cmp bx, ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the opercpy to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; cpy finally we get to the actual fxn of the opcode
  mov bx, [Y_]
  cmp bx, ax
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; dec: Accumulator Addressing Mode
_dec_A_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; bit finally we get to the actual fxn of the opcode
  dec WORD [A_]
  jmp .opdone
.bytesize

  dec BYTE [A_+1]
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; dec: Direct Addressing Mode
_dec_d_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [edi]
  xchg ah, al
  dec ax
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [edi]
  xchg ah, al
  dec ax
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov al, [edi]
  dec al
  mov [edi], al
  GetFlags
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov al, [edi]
  dec al
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; dec: Direct,X Addressing Mode
_dec_dx_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [edi]
  xchg ah, al
  dec ax
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [edi]
  xchg ah, al
  dec ax
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov al, [edi]
  dec al
  mov [edi], al
  GetFlags
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov al, [edi]
  dec al
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; dec: Absolute Addressing Mode
_dec_a_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [edi]
  xchg ah, al
  dec ax
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [edi]
  xchg ah, al
  dec ax
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov al, [edi]
  dec al
  mov [edi], al
  GetFlags
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov al, [edi]
  dec al
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; dec: Absolute,X Addressing Mode
_dec_ax_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [edi]
  xchg ah, al
  dec ax
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [edi]
  xchg ah, al
  dec ax
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov al, [edi]
  dec al
  mov [edi], al
  GetFlags
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov al, [edi]
  dec al
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; eor: Immediate Addressing Mode
_eor_i_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opereor to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov edi, [esi]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; eor finally we get to the actual fxn of the opcode
  mov bx, [A_]
  xor bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the opereor to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov edi, [esi]
  call SnesMap_
  mov al, [edi]
  cbw

  ; eor finally we get to the actual fxn of the opcode
  mov bx, [A_]
  xor bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; eor: Direct Addressing Mode
_eor_d_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opereor to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; eor finally we get to the actual fxn of the opcode
  mov bx, [A_]
  xor bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the opereor to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; eor finally we get to the actual fxn of the opcode
  mov bx, [A_]
  xor bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; eor: Direct,X Addressing Mode
_eor_dx_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opereor to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; eor finally we get to the actual fxn of the opcode
  mov bx, [A_]
  xor bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the opereor to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax
  call SnesMap_
  mov al, [edi]
  cbw

  ; eor finally we get to the actual fxn of the opcode
  mov bx, [A_]
  xor bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; eor: Direct Indirect Addressing Mode
_eor_pd_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opereor to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; eor finally we get to the actual fxn of the opcode
  mov bx, [A_]
  xor bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the opereor to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; eor finally we get to the actual fxn of the opcode
  mov bx, [A_]
  xor bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; eor: Direct Indirect,X Addressing Mode
_eor_pdx_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opereor to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [Y_]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; eor finally we get to the actual fxn of the opcode
  mov bx, [A_]
  xor bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the opereor to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [Y_]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; eor finally we get to the actual fxn of the opcode
  mov bx, [A_]
  xor bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; eor: Direct Indirect,Y Addressing Mode
_eor_pd_y_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opereor to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor ax, ax
  mov ax, [Y_]
  add edi, eax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; eor finally we get to the actual fxn of the opcode
  mov bx, [A_]
  xor bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the opereor to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor ax, ax
  mov ax, [Y_]
  add edi, eax
  call SnesMap_
  mov al, [edi]
  cbw

  ; eor finally we get to the actual fxn of the opcode
  mov bx, [A_]
  xor bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; eor: Direct Indirect Long Addressing Mode
_eor_bd_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opereor to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; eor finally we get to the actual fxn of the opcode
  mov bx, [A_]
  xor bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the opereor to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  call SnesMap_
  mov al, [edi]
  cbw

  ; eor finally we get to the actual fxn of the opcode
  mov bx, [A_]
  xor bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; eor: Absolute Addressing Mode
_eor_a_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opereor to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; eor finally we get to the actual fxn of the opcode
  mov bx, [A_]
  xor bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the opereor to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; eor finally we get to the actual fxn of the opcode
  mov bx, [A_]
  xor bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; eor: Absolute,X Addressing Mode
_eor_ax_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opereor to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; eor finally we get to the actual fxn of the opcode
  mov bx, [A_]
  xor bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the opereor to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; eor finally we get to the actual fxn of the opcode
  mov bx, [A_]
  xor bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; eor: Absolute,Y Addressing Mode
_eor_y_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opereor to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [Y_]
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; eor finally we get to the actual fxn of the opcode
  mov bx, [A_]
  xor bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the opereor to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [Y_]
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; eor finally we get to the actual fxn of the opcode
  mov bx, [A_]
  xor bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; eor: Absolute Long Addressing Mode
_eor_al_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opereor to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  inc esi
  mov di, ax
  xor eax, eax
  mov al, [esi]
  shl eax, 16
  add edi, eax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; eor finally we get to the actual fxn of the opcode
  mov bx, [A_]
  xor bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the opereor to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  inc esi
  mov di, ax
  xor eax, eax
  mov al, [esi]
  shl eax, 16
  add edi, eax
  call SnesMap_
  mov al, [edi]
  cbw

  ; eor finally we get to the actual fxn of the opcode
  mov bx, [A_]
  xor bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; eor: Stack Relative Addressing Mode
_eor_ds_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opereor to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; eor finally we get to the actual fxn of the opcode
  mov bx, [A_]
  xor bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the opereor to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  call SnesMap_
  mov al, [edi]
  cbw

  ; eor finally we get to the actual fxn of the opcode
  mov bx, [A_]
  xor bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; eor: Stack Relative Indirect,Y Addressing Mode
_eor_pds_y_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the opereor to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor eax, eax
  mov ax, [Y_]
  add edi, eax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; eor finally we get to the actual fxn of the opcode
  mov bx, [A_]
  xor bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the opereor to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor eax, eax
  mov ax, [Y_]
  add edi, eax
  call SnesMap_
  mov al, [edi]
  cbw

  ; eor finally we get to the actual fxn of the opcode
  mov bx, [A_]
  xor bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; inc: Accumulator Addressing Mode
_inc_A_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; bit finally we get to the actual fxn of the opcode
  inc WORD [A_]
  jmp .opdone
.bytesize

  inc BYTE [A_+1]
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; inc: Direct Addressing Mode
_inc_d_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [edi]
  xchg ah, al
  inc ax
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [edi]
  xchg ah, al
  inc ax
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov al, [edi]
  inc al
  mov [edi], al
  GetFlags
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov al, [edi]
  inc al
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; inc: Direct,X Addressing Mode
_inc_dx_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [edi]
  xchg ah, al
  inc ax
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [edi]
  xchg ah, al
  inc ax
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov al, [edi]
  inc al
  mov [edi], al
  GetFlags
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov al, [edi]
  inc al
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; inc: Absolute Addressing Mode
_inc_a_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [edi]
  xchg ah, al
  inc ax
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [edi]
  xchg ah, al
  inc ax
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov al, [edi]
  inc al
  mov [edi], al
  GetFlags
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov al, [edi]
  inc al
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; inc: Absolute,X Addressing Mode
_inc_ax_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [edi]
  xchg ah, al
  inc ax
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [edi]
  xchg ah, al
  inc ax
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov al, [edi]
  inc al
  mov [edi], al
  GetFlags
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov al, [edi]
  inc al
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; lda: Immediate Addressing Mode
_lda_i_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operlda to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov edi, [esi]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; lda finally we get to the actual fxn of the opcode
  mov [A_], bx
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operlda to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov edi, [esi]
  call SnesMap_
  mov al, [edi]
  cbw

  ; lda finally we get to the actual fxn of the opcode
  mov [A_], bx
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; lda: Direct Addressing Mode
_lda_d_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operlda to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; lda finally we get to the actual fxn of the opcode
  mov [A_], bx
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operlda to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; lda finally we get to the actual fxn of the opcode
  mov [A_], bx
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; lda: Direct,X Addressing Mode
_lda_dx_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operlda to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; lda finally we get to the actual fxn of the opcode
  mov [A_], bx
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operlda to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax
  call SnesMap_
  mov al, [edi]
  cbw

  ; lda finally we get to the actual fxn of the opcode
  mov [A_], bx
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; lda: Direct Indirect Addressing Mode
_lda_pd_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operlda to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; lda finally we get to the actual fxn of the opcode
  mov [A_], bx
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operlda to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; lda finally we get to the actual fxn of the opcode
  mov [A_], bx
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; lda: Direct Indirect,X Addressing Mode
_lda_pdx_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operlda to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [Y_]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; lda finally we get to the actual fxn of the opcode
  mov [A_], bx
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operlda to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [Y_]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; lda finally we get to the actual fxn of the opcode
  mov [A_], bx
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; lda: Direct Indirect,Y Addressing Mode
_lda_pd_y_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operlda to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor ax, ax
  mov ax, [Y_]
  add edi, eax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; lda finally we get to the actual fxn of the opcode
  mov [A_], bx
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operlda to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor ax, ax
  mov ax, [Y_]
  add edi, eax
  call SnesMap_
  mov al, [edi]
  cbw

  ; lda finally we get to the actual fxn of the opcode
  mov [A_], bx
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; lda: Direct Indirect Long Addressing Mode
_lda_bd_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operlda to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; lda finally we get to the actual fxn of the opcode
  mov [A_], bx
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operlda to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  call SnesMap_
  mov al, [edi]
  cbw

  ; lda finally we get to the actual fxn of the opcode
  mov [A_], bx
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; lda: Absolute Addressing Mode
_lda_a_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operlda to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; lda finally we get to the actual fxn of the opcode
  mov [A_], bx
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operlda to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; lda finally we get to the actual fxn of the opcode
  mov [A_], bx
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; lda: Absolute,X Addressing Mode
_lda_ax_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operlda to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; lda finally we get to the actual fxn of the opcode
  mov [A_], bx
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operlda to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; lda finally we get to the actual fxn of the opcode
  mov [A_], bx
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; lda: Absolute,Y Addressing Mode
_lda_y_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operlda to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [Y_]
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; lda finally we get to the actual fxn of the opcode
  mov [A_], bx
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operlda to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [Y_]
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; lda finally we get to the actual fxn of the opcode
  mov [A_], bx
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; lda: Absolute Long Addressing Mode
_lda_al_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operlda to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  inc esi
  mov di, ax
  xor eax, eax
  mov al, [esi]
  shl eax, 16
  add edi, eax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; lda finally we get to the actual fxn of the opcode
  mov [A_], bx
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operlda to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  inc esi
  mov di, ax
  xor eax, eax
  mov al, [esi]
  shl eax, 16
  add edi, eax
  call SnesMap_
  mov al, [edi]
  cbw

  ; lda finally we get to the actual fxn of the opcode
  mov [A_], bx
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; lda: Stack Relative Addressing Mode
_lda_ds_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operlda to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; lda finally we get to the actual fxn of the opcode
  mov [A_], bx
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operlda to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  call SnesMap_
  mov al, [edi]
  cbw

  ; lda finally we get to the actual fxn of the opcode
  mov [A_], bx
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; lda: Stack Relative Indirect,Y Addressing Mode
_lda_pds_y_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operlda to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor eax, eax
  mov ax, [Y_]
  add edi, eax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; lda finally we get to the actual fxn of the opcode
  mov [A_], bx
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operlda to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor eax, eax
  mov ax, [Y_]
  add edi, eax
  call SnesMap_
  mov al, [edi]
  cbw

  ; lda finally we get to the actual fxn of the opcode
  mov [A_], bx
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ldx: Immediate Addressing Mode
_ldx_i_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operldx to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov edi, [esi]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; ldx finally we get to the actual fxn of the opcode
  mov [X_], bx
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operldx to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov edi, [esi]
  call SnesMap_
  mov al, [edi]
  cbw

  ; ldx finally we get to the actual fxn of the opcode
  mov [X_], bx
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ldx: Direct Addressing Mode
_ldx_d_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operldx to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; ldx finally we get to the actual fxn of the opcode
  mov [X_], bx
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operldx to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; ldx finally we get to the actual fxn of the opcode
  mov [X_], bx
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ldx: Direct,Y Addressing Mode
_ldx_dy_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operldx to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [Y_]
  mov di, ax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; ldx finally we get to the actual fxn of the opcode
  mov [X_], bx
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operldx to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [Y_]
  mov di, ax
  call SnesMap_
  mov al, [edi]
  cbw

  ; ldx finally we get to the actual fxn of the opcode
  mov [X_], bx
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ldx: Absolute Addressing Mode
_ldx_a_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operldx to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; ldx finally we get to the actual fxn of the opcode
  mov [X_], bx
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operldx to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; ldx finally we get to the actual fxn of the opcode
  mov [X_], bx
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ldx: Absolute,X Addressing Mode
_ldx_ax_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operldx to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; ldx finally we get to the actual fxn of the opcode
  mov [X_], bx
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operldx to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; ldx finally we get to the actual fxn of the opcode
  mov [X_], bx
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ldy: Immediate Addressing Mode
_ldy_i_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operldy to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov edi, [esi]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; ldy finally we get to the actual fxn of the opcode
  mov [Y_], bx
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operldy to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov edi, [esi]
  call SnesMap_
  mov al, [edi]
  cbw

  ; ldy finally we get to the actual fxn of the opcode
  mov [Y_], bx
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ldy: Direct Addressing Mode
_ldy_d_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operldy to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; ldy finally we get to the actual fxn of the opcode
  mov [Y_], bx
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operldy to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; ldy finally we get to the actual fxn of the opcode
  mov [Y_], bx
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ldy: Direct,X Addressing Mode
_ldy_dx_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operldy to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; ldy finally we get to the actual fxn of the opcode
  mov [Y_], bx
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operldy to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax
  call SnesMap_
  mov al, [edi]
  cbw

  ; ldy finally we get to the actual fxn of the opcode
  mov [Y_], bx
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ldy: Absolute Addressing Mode
_ldy_a_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operldy to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; ldy finally we get to the actual fxn of the opcode
  mov [Y_], bx
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operldy to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; ldy finally we get to the actual fxn of the opcode
  mov [Y_], bx
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ldy: Absolute,X Addressing Mode
_ldy_ax_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operldy to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; ldy finally we get to the actual fxn of the opcode
  mov [Y_], bx
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operldy to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; ldy finally we get to the actual fxn of the opcode
  mov [Y_], bx
  GetFlags
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; lsr: Accumulator Addressing Mode
_lsr_A_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; bit finally we get to the actual fxn of the opcode
  shr WORD [A_], 1
  jmp .opdone
.bytesize

  shr BYTE [A_+1], 1
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; lsr: Direct Addressing Mode
_lsr_d_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [edi]
  xchg ah, al
  shr ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [edi]
  xchg ah, al
  shr ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov al, [edi]
  shr al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov al, [edi]
  shr al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; lsr: Direct,X Addressing Mode
_lsr_dx_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [edi]
  xchg ah, al
  shr ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [edi]
  xchg ah, al
  shr ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov al, [edi]
  shr al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov al, [edi]
  shr al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; lsr: Absolute Addressing Mode
_lsr_a_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [edi]
  xchg ah, al
  shr ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [edi]
  xchg ah, al
  shr ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov al, [edi]
  shr al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov al, [edi]
  shr al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; lsr: Absolute,X Addressing Mode
_lsr_ax_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [edi]
  xchg ah, al
  shr ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [edi]
  xchg ah, al
  shr ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov al, [edi]
  shr al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov al, [edi]
  shr al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ora: Immediate Addressing Mode
_ora_i_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operora to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov edi, [esi]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; ora finally we get to the actual fxn of the opcode
  mov bx, [A_]
  or  bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operora to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov edi, [esi]
  call SnesMap_
  mov al, [edi]
  cbw

  ; ora finally we get to the actual fxn of the opcode
  mov bx, [A_]
  or  bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ora: Direct Addressing Mode
_ora_d_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operora to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; ora finally we get to the actual fxn of the opcode
  mov bx, [A_]
  or  bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operora to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; ora finally we get to the actual fxn of the opcode
  mov bx, [A_]
  or  bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ora: Direct,X Addressing Mode
_ora_dx_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operora to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; ora finally we get to the actual fxn of the opcode
  mov bx, [A_]
  or  bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operora to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax
  call SnesMap_
  mov al, [edi]
  cbw

  ; ora finally we get to the actual fxn of the opcode
  mov bx, [A_]
  or  bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ora: Direct Indirect Addressing Mode
_ora_pd_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operora to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; ora finally we get to the actual fxn of the opcode
  mov bx, [A_]
  or  bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operora to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; ora finally we get to the actual fxn of the opcode
  mov bx, [A_]
  or  bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ora: Direct Indirect,X Addressing Mode
_ora_pdx_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operora to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [Y_]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; ora finally we get to the actual fxn of the opcode
  mov bx, [A_]
  or  bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operora to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [Y_]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; ora finally we get to the actual fxn of the opcode
  mov bx, [A_]
  or  bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ora: Direct Indirect,Y Addressing Mode
_ora_pd_y_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operora to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor ax, ax
  mov ax, [Y_]
  add edi, eax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; ora finally we get to the actual fxn of the opcode
  mov bx, [A_]
  or  bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operora to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor ax, ax
  mov ax, [Y_]
  add edi, eax
  call SnesMap_
  mov al, [edi]
  cbw

  ; ora finally we get to the actual fxn of the opcode
  mov bx, [A_]
  or  bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ora: Direct Indirect Long Addressing Mode
_ora_bd_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operora to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; ora finally we get to the actual fxn of the opcode
  mov bx, [A_]
  or  bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operora to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  call SnesMap_
  mov al, [edi]
  cbw

  ; ora finally we get to the actual fxn of the opcode
  mov bx, [A_]
  or  bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ora: Absolute Addressing Mode
_ora_a_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operora to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; ora finally we get to the actual fxn of the opcode
  mov bx, [A_]
  or  bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operora to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; ora finally we get to the actual fxn of the opcode
  mov bx, [A_]
  or  bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ora: Absolute,X Addressing Mode
_ora_ax_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operora to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; ora finally we get to the actual fxn of the opcode
  mov bx, [A_]
  or  bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operora to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; ora finally we get to the actual fxn of the opcode
  mov bx, [A_]
  or  bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ora: Absolute,Y Addressing Mode
_ora_y_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operora to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [Y_]
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; ora finally we get to the actual fxn of the opcode
  mov bx, [A_]
  or  bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operora to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [Y_]
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; ora finally we get to the actual fxn of the opcode
  mov bx, [A_]
  or  bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ora: Absolute Long Addressing Mode
_ora_al_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operora to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  inc esi
  mov di, ax
  xor eax, eax
  mov al, [esi]
  shl eax, 16
  add edi, eax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; ora finally we get to the actual fxn of the opcode
  mov bx, [A_]
  or  bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operora to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  inc esi
  mov di, ax
  xor eax, eax
  mov al, [esi]
  shl eax, 16
  add edi, eax
  call SnesMap_
  mov al, [edi]
  cbw

  ; ora finally we get to the actual fxn of the opcode
  mov bx, [A_]
  or  bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ora: Stack Relative Addressing Mode
_ora_ds_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operora to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; ora finally we get to the actual fxn of the opcode
  mov bx, [A_]
  or  bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operora to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  call SnesMap_
  mov al, [edi]
  cbw

  ; ora finally we get to the actual fxn of the opcode
  mov bx, [A_]
  or  bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ora: Stack Relative Indirect,Y Addressing Mode
_ora_pds_y_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operora to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor eax, eax
  mov ax, [Y_]
  add edi, eax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; ora finally we get to the actual fxn of the opcode
  mov bx, [A_]
  or  bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operora to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor eax, eax
  mov ax, [Y_]
  add edi, eax
  call SnesMap_
  mov al, [edi]
  cbw

  ; ora finally we get to the actual fxn of the opcode
  mov bx, [A_]
  or  bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; rol: Accumulator Addressing Mode
_rol_A_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; bit finally we get to the actual fxn of the opcode
  rol WORD [A_], 1
  jmp .opdone
.bytesize

  rol BYTE [A_+1], 1
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; rol: Direct Addressing Mode
_rol_d_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [edi]
  xchg ah, al
  rol ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [edi]
  xchg ah, al
  rol ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov al, [edi]
  rol al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov al, [edi]
  rol al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; rol: Direct,X Addressing Mode
_rol_dx_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [edi]
  xchg ah, al
  rol ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [edi]
  xchg ah, al
  rol ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov al, [edi]
  rol al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov al, [edi]
  rol al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; rol: Absolute Addressing Mode
_rol_a_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [edi]
  xchg ah, al
  rol ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [edi]
  xchg ah, al
  rol ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov al, [edi]
  rol al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov al, [edi]
  rol al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; rol: Absolute,X Addressing Mode
_rol_ax_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [edi]
  xchg ah, al
  rol ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [edi]
  xchg ah, al
  rol ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov al, [edi]
  rol al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov al, [edi]
  rol al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ror: Accumulator Addressing Mode
_ror_A_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; bit finally we get to the actual fxn of the opcode
  ror WORD [A_], 1
  jmp .opdone
.bytesize

  ror BYTE [A_+1], 1
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ror: Direct Addressing Mode
_ror_d_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [edi]
  xchg ah, al
  ror ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [edi]
  xchg ah, al
  ror ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov al, [edi]
  ror al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov al, [edi]
  ror al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ror: Direct,X Addressing Mode
_ror_dx_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [edi]
  xchg ah, al
  ror ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [edi]
  xchg ah, al
  ror ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov al, [edi]
  ror al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov al, [edi]
  ror al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ror: Absolute Addressing Mode
_ror_a_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [edi]
  xchg ah, al
  ror ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [edi]
  xchg ah, al
  ror ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov al, [edi]
  ror al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov al, [edi]
  ror al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; ror: Absolute,X Addressing Mode
_ror_ax_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [edi]
  xchg ah, al
  ror ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [edi]
  xchg ah, al
  ror ax, 1
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
  jmp .opdone
.bytesize

  ; it's a byte, get the operbit to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]

  ; bit finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov al, [edi]
  ror al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov al, [edi]
  ror al, 1
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; sbc: Immediate Addressing Mode
_sbc_i_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov edi, [esi]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  sbb bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov edi, [esi]
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  sbb bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; sbc: Direct Addressing Mode
_sbc_d_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  sbb bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  sbb bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; sbc: Direct,X Addressing Mode
_sbc_dx_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  sbb bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  sbb bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; sbc: Direct Indirect Addressing Mode
_sbc_pd_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  sbb bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  sbb bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; sbc: Direct Indirect,X Addressing Mode
_sbc_pdx_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [Y_]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  sbb bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [Y_]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  sbb bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; sbc: Direct Indirect,Y Addressing Mode
_sbc_pd_y_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor ax, ax
  mov ax, [Y_]
  add edi, eax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  sbb bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor ax, ax
  mov ax, [Y_]
  add edi, eax
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  sbb bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; sbc: Direct Indirect Long Addressing Mode
_sbc_bd_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  sbb bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  sbb bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; sbc: Absolute Addressing Mode
_sbc_a_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  sbb bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  sbb bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; sbc: Absolute,X Addressing Mode
_sbc_ax_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  sbb bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  sbb bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; sbc: Absolute,Y Addressing Mode
_sbc_y_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [Y_]
  add edi, [DB_]
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  sbb bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [Y_]
  add edi, [DB_]
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  sbb bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; sbc: Absolute Long Addressing Mode
_sbc_al_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  inc esi
  mov di, ax
  xor eax, eax
  mov al, [esi]
  shl eax, 16
  add edi, eax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  sbb bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  inc esi
  mov di, ax
  xor eax, eax
  mov al, [esi]
  shl eax, 16
  add edi, eax
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  sbb bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; sbc: Stack Relative Addressing Mode
_sbc_ds_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  sbb bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  sbb bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; sbc: Stack Relative Indirect,Y Addressing Mode
_sbc_pds_y_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor eax, eax
  mov ax, [Y_]
  add edi, eax
  call SnesMap_
  mov ax, [edi]
  xchg al, ah

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  sbb bx, ax
  GetFlags
  mov [A_], bx
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor eax, eax
  mov ax, [Y_]
  add edi, eax
  call SnesMap_
  mov al, [edi]
  cbw

  ; and finally we get to the actual fxn of the opcode
  mov bx, [A_]
  SetFlags
  sbb bx, ax
  GetFlags
  mov [A_], bx
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; : Direct Addressing Mode
_sta_d_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  cmp bx, 01FFFh
  jle .reg_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [A_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.reg_w
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [A_]
  xchg ah, al
  mov [edi], ax
  GetFlags
  mov [RegValW_], ax
  mov [RegNum_], bx
  call RegFxnW_
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [A_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  cmp bx, 01FFFh
  jle .reg_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [A_]
  mov [edi], al
  GetFlags
  jmp .opdone
.reg_b
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [A_]
  mov [edi], al
  GetFlags
  mov [RegValB_], al
  mov [RegNum_], bx
  call RegFxnB_
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [A_]
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; : Direct,X Addressing Mode
_sta_dx_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  cmp bx, 01FFFh
  jle .reg_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [A_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.reg_w
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [A_]
  xchg ah, al
  mov [edi], ax
  GetFlags
  mov [RegValW_], ax
  mov [RegNum_], bx
  call RegFxnW_
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [A_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  cmp bx, 01FFFh
  jle .reg_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [A_]
  mov [edi], al
  GetFlags
  jmp .opdone
.reg_b
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [A_]
  mov [edi], al
  GetFlags
  mov [RegValB_], al
  mov [RegNum_], bx
  call RegFxnB_
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [A_]
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; : Direct Indirect Addressing Mode
_sta_pd_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  cmp bx, 01FFFh
  jle .reg_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [A_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.reg_w
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [A_]
  xchg ah, al
  mov [edi], ax
  GetFlags
  mov [RegValW_], ax
  mov [RegNum_], bx
  call RegFxnW_
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [A_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  cmp bx, 01FFFh
  jle .reg_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [A_]
  mov [edi], al
  GetFlags
  jmp .opdone
.reg_b
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [A_]
  mov [edi], al
  GetFlags
  mov [RegValB_], al
  mov [RegNum_], bx
  call RegFxnB_
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [A_]
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; : Direct Indirect,X Addressing Mode
_sta_pdx_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [Y_]
  mov di, ax
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  cmp bx, 01FFFh
  jle .reg_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [A_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.reg_w
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [A_]
  xchg ah, al
  mov [edi], ax
  GetFlags
  mov [RegValW_], ax
  mov [RegNum_], bx
  call RegFxnW_
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [A_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [Y_]
  mov di, ax
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  cmp bx, 01FFFh
  jle .reg_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [A_]
  mov [edi], al
  GetFlags
  jmp .opdone
.reg_b
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [A_]
  mov [edi], al
  GetFlags
  mov [RegValB_], al
  mov [RegNum_], bx
  call RegFxnB_
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [A_]
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; : Direct Indirect,Y Addressing Mode
_sta_pd_y_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor ax, ax
  mov ax, [Y_]
  add edi, eax

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  cmp bx, 01FFFh
  jle .reg_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [A_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.reg_w
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [A_]
  xchg ah, al
  mov [edi], ax
  GetFlags
  mov [RegValW_], ax
  mov [RegNum_], bx
  call RegFxnW_
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [A_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor ax, ax
  mov ax, [Y_]
  add edi, eax

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  cmp bx, 01FFFh
  jle .reg_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [A_]
  mov [edi], al
  GetFlags
  jmp .opdone
.reg_b
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [A_]
  mov [edi], al
  GetFlags
  mov [RegValB_], al
  mov [RegNum_], bx
  call RegFxnB_
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [A_]
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; : Direct Indirect Long Addressing Mode
_sta_bd_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  cmp bx, 01FFFh
  jle .reg_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [A_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.reg_w
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [A_]
  xchg ah, al
  mov [edi], ax
  GetFlags
  mov [RegValW_], ax
  mov [RegNum_], bx
  call RegFxnW_
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [A_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  cmp bx, 01FFFh
  jle .reg_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [A_]
  mov [edi], al
  GetFlags
  jmp .opdone
.reg_b
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [A_]
  mov [edi], al
  GetFlags
  mov [RegValB_], al
  mov [RegNum_], bx
  call RegFxnB_
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [A_]
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; : Absolute Addressing Mode
_sta_a_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  cmp bx, 01FFFh
  jle .reg_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [A_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.reg_w
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [A_]
  xchg ah, al
  mov [edi], ax
  GetFlags
  mov [RegValW_], ax
  mov [RegNum_], bx
  call RegFxnW_
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [A_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  cmp bx, 01FFFh
  jle .reg_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [A_]
  mov [edi], al
  GetFlags
  jmp .opdone
.reg_b
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [A_]
  mov [edi], al
  GetFlags
  mov [RegValB_], al
  mov [RegNum_], bx
  call RegFxnB_
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [A_]
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; : Absolute,X Addressing Mode
_sta_ax_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  cmp bx, 01FFFh
  jle .reg_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [A_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.reg_w
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [A_]
  xchg ah, al
  mov [edi], ax
  GetFlags
  mov [RegValW_], ax
  mov [RegNum_], bx
  call RegFxnW_
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [A_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  cmp bx, 01FFFh
  jle .reg_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [A_]
  mov [edi], al
  GetFlags
  jmp .opdone
.reg_b
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [A_]
  mov [edi], al
  GetFlags
  mov [RegValB_], al
  mov [RegNum_], bx
  call RegFxnB_
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [A_]
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; : Absolute,Y Addressing Mode
_sta_y_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [Y_]
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  cmp bx, 01FFFh
  jle .reg_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [A_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.reg_w
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [A_]
  xchg ah, al
  mov [edi], ax
  GetFlags
  mov [RegValW_], ax
  mov [RegNum_], bx
  call RegFxnW_
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [A_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [Y_]
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  cmp bx, 01FFFh
  jle .reg_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [A_]
  mov [edi], al
  GetFlags
  jmp .opdone
.reg_b
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [A_]
  mov [edi], al
  GetFlags
  mov [RegValB_], al
  mov [RegNum_], bx
  call RegFxnB_
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [A_]
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; : Absolute Long Addressing Mode
_sta_al_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  inc esi
  mov di, ax
  xor eax, eax
  mov al, [esi]
  shl eax, 16
  add edi, eax

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  cmp bx, 01FFFh
  jle .reg_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [A_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.reg_w
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [A_]
  xchg ah, al
  mov [edi], ax
  GetFlags
  mov [RegValW_], ax
  mov [RegNum_], bx
  call RegFxnW_
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [A_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  inc esi
  mov di, ax
  xor eax, eax
  mov al, [esi]
  shl eax, 16
  add edi, eax

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  cmp bx, 01FFFh
  jle .reg_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [A_]
  mov [edi], al
  GetFlags
  jmp .opdone
.reg_b
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [A_]
  mov [edi], al
  GetFlags
  mov [RegValB_], al
  mov [RegNum_], bx
  call RegFxnB_
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [A_]
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; : Stack Relative Addressing Mode
_sta_ds_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  cmp bx, 01FFFh
  jle .reg_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [A_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.reg_w
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [A_]
  xchg ah, al
  mov [edi], ax
  GetFlags
  mov [RegValW_], ax
  mov [RegNum_], bx
  call RegFxnW_
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [A_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  cmp bx, 01FFFh
  jle .reg_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [A_]
  mov [edi], al
  GetFlags
  jmp .opdone
.reg_b
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [A_]
  mov [edi], al
  GetFlags
  mov [RegValB_], al
  mov [RegNum_], bx
  call RegFxnB_
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [A_]
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; : Stack Relative Indirect,Y Addressing Mode
_sta_pds_y_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor eax, eax
  mov ax, [Y_]
  add edi, eax

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  cmp bx, 01FFFh
  jle .reg_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [A_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.reg_w
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [A_]
  xchg ah, al
  mov [edi], ax
  GetFlags
  mov [RegValW_], ax
  mov [RegNum_], bx
  call RegFxnW_
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [A_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [S_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]
  xor eax, eax
  mov ax, [Y_]
  add edi, eax

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  cmp bx, 01FFFh
  jle .reg_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [A_]
  mov [edi], al
  GetFlags
  jmp .opdone
.reg_b
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [A_]
  mov [edi], al
  GetFlags
  mov [RegValB_], al
  mov [RegNum_], bx
  call RegFxnB_
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [A_]
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; : Direct Addressing Mode
_stx_d_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  cmp bx, 01FFFh
  jle .reg_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [X_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.reg_w
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [X_]
  xchg ah, al
  mov [edi], ax
  GetFlags
  mov [RegValW_], ax
  mov [RegNum_], bx
  call RegFxnW_
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [X_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  cmp bx, 01FFFh
  jle .reg_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [X_]
  mov [edi], al
  GetFlags
  jmp .opdone
.reg_b
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [X_]
  mov [edi], al
  GetFlags
  mov [RegValB_], al
  mov [RegNum_], bx
  call RegFxnB_
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [X_]
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; : Direct,Y Addressing Mode
_stx_dy_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [Y_]
  mov di, ax

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  cmp bx, 01FFFh
  jle .reg_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [X_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.reg_w
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [X_]
  xchg ah, al
  mov [edi], ax
  GetFlags
  mov [RegValW_], ax
  mov [RegNum_], bx
  call RegFxnW_
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [X_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [Y_]
  mov di, ax

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  cmp bx, 01FFFh
  jle .reg_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [X_]
  mov [edi], al
  GetFlags
  jmp .opdone
.reg_b
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [X_]
  mov [edi], al
  GetFlags
  mov [RegValB_], al
  mov [RegNum_], bx
  call RegFxnB_
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [X_]
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; : Absolute Addressing Mode
_stx_a_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  cmp bx, 01FFFh
  jle .reg_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [X_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.reg_w
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [X_]
  xchg ah, al
  mov [edi], ax
  GetFlags
  mov [RegValW_], ax
  mov [RegNum_], bx
  call RegFxnW_
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [X_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  cmp bx, 01FFFh
  jle .reg_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [X_]
  mov [edi], al
  GetFlags
  jmp .opdone
.reg_b
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [X_]
  mov [edi], al
  GetFlags
  mov [RegValB_], al
  mov [RegNum_], bx
  call RegFxnB_
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [X_]
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; : Direct Addressing Mode
_sty_d_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  cmp bx, 01FFFh
  jle .reg_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [Y_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.reg_w
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [Y_]
  xchg ah, al
  mov [edi], ax
  GetFlags
  mov [RegValW_], ax
  mov [RegNum_], bx
  call RegFxnW_
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [Y_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  cmp bx, 01FFFh
  jle .reg_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [Y_]
  mov [edi], al
  GetFlags
  jmp .opdone
.reg_b
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [Y_]
  mov [edi], al
  GetFlags
  mov [RegValB_], al
  mov [RegNum_], bx
  call RegFxnB_
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [Y_]
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; : Direct,X Addressing Mode
_sty_dx_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  cmp bx, 01FFFh
  jle .reg_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [Y_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.reg_w
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [Y_]
  xchg ah, al
  mov [edi], ax
  GetFlags
  mov [RegValW_], ax
  mov [RegNum_], bx
  call RegFxnW_
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [Y_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  cmp bx, 01FFFh
  jle .reg_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [Y_]
  mov [edi], al
  GetFlags
  jmp .opdone
.reg_b
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [Y_]
  mov [edi], al
  GetFlags
  mov [RegValB_], al
  mov [RegNum_], bx
  call RegFxnB_
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [Y_]
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; : Absolute Addressing Mode
_sty_a_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  cmp bx, 01FFFh
  jle .reg_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [Y_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.reg_w
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [Y_]
  xchg ah, al
  mov [edi], ax
  GetFlags
  mov [RegValW_], ax
  mov [RegNum_], bx
  call RegFxnW_
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [Y_]
  xchg al, ah
  mov [edi], ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  cmp bx, 01FFFh
  jle .reg_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [Y_]
  mov [edi], al
  GetFlags
  jmp .opdone
.reg_b
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  mov ax, [Y_]
  mov [edi], al
  GetFlags
  mov [RegValB_], al
  mov [RegNum_], bx
  call RegFxnB_
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [Y_]
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; : Direct Addressing Mode
_stz_d_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  cmp bx, 01FFFh
  jle .reg_w
  mov edi, [SnesRam_]
  add edi, ebx
  xor ax, ax
  mov [edi], ax
  GetFlags
  jmp .opdone
.reg_w
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  xor ax, ax
  mov [edi], ax
  GetFlags
  mov [RegValW_], ax
  mov [RegNum_], bx
  call RegFxnW_
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  xor ax, ax
  mov [edi], ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  cmp bx, 01FFFh
  jle .reg_b
  mov edi, [SnesRam_]
  add edi, ebx
  xor ax, ax
  mov [edi], al
  GetFlags
  jmp .opdone
.reg_b
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  xor ax, ax
  mov [edi], al
  GetFlags
  mov [RegValB_], al
  mov [RegNum_], bx
  call RegFxnB_
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  xor ax, ax
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; : Direct,X Addressing Mode
_stz_dx_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  cmp bx, 01FFFh
  jle .reg_w
  mov edi, [SnesRam_]
  add edi, ebx
  xor ax, ax
  mov [edi], ax
  GetFlags
  jmp .opdone
.reg_w
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  xor ax, ax
  mov [edi], ax
  GetFlags
  mov [RegValW_], ax
  mov [RegNum_], bx
  call RegFxnW_
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  xor ax, ax
  mov [edi], ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  add ax, [X_]
  mov di, ax

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  cmp bx, 01FFFh
  jle .reg_b
  mov edi, [SnesRam_]
  add edi, ebx
  xor ax, ax
  mov [edi], al
  GetFlags
  jmp .opdone
.reg_b
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  xor ax, ax
  mov [edi], al
  GetFlags
  mov [RegValB_], al
  mov [RegNum_], bx
  call RegFxnB_
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  xor ax, ax
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; : Absolute Addressing Mode
_stz_a_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  cmp bx, 01FFFh
  jle .reg_w
  mov edi, [SnesRam_]
  add edi, ebx
  xor ax, ax
  mov [edi], ax
  GetFlags
  jmp .opdone
.reg_w
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  xor ax, ax
  mov [edi], ax
  GetFlags
  mov [RegValW_], ax
  mov [RegNum_], bx
  call RegFxnW_
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  xor ax, ax
  mov [edi], ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  cmp bx, 01FFFh
  jle .reg_b
  mov edi, [SnesRam_]
  add edi, ebx
  xor ax, ax
  mov [edi], al
  GetFlags
  jmp .opdone
.reg_b
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  xor ax, ax
  mov [edi], al
  GetFlags
  mov [RegValB_], al
  mov [RegNum_], bx
  call RegFxnB_
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  xor ax, ax
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; : Absolute,X Addressing Mode
_stz_ax_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  cmp bx, 01FFFh
  jle .reg_w
  mov edi, [SnesRam_]
  add edi, ebx
  xor ax, ax
  mov [edi], ax
  GetFlags
  jmp .opdone
.reg_w
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  xor ax, ax
  mov [edi], ax
  GetFlags
  mov [RegValW_], ax
  mov [RegNum_], bx
  call RegFxnW_
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  xor ax, ax
  mov [edi], ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add di, [X_]
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  cmp bx, 01FFFh
  jle .reg_b
  mov edi, [SnesRam_]
  add edi, ebx
  xor ax, ax
  mov [edi], al
  GetFlags
  jmp .opdone
.reg_b
  mov edi, [SnesReg_]
  sub bx, 02100h
  add edi, ebx
  xor ax, ax
  mov [edi], al
  GetFlags
  mov [RegValB_], al
  mov [RegNum_], bx
  call RegFxnB_
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  xor ax, ax
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; : Direct Addressing Mode
_trb_d_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [edi]
  xchg ah, al
  mov bx, [A_]
  not bx
  and ax, bx
  GetFlags
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [edi]
  xchg ah, al
  mov bx, [A_]
  not bx
  and ax, bx
  GetFlags
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov al, [edi]
  mov bx, [A_]
  not bl
  and al, bl
  GetFlags
  mov [edi], al
  GetFlags
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov al, [edi]
  mov bx, [A_]
  not bl
  and al, bl
  GetFlags
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; : Absolute Addressing Mode
_trb_a_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [edi]
  xchg ah, al
  mov bx, [A_]
  not bx
  and ax, bx
  GetFlags
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [edi]
  xchg ah, al
  mov bx, [A_]
  not bx
  and ax, bx
  GetFlags
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov al, [edi]
  mov bx, [A_]
  not bl
  and al, bl
  GetFlags
  mov [edi], al
  GetFlags
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov al, [edi]
  mov bx, [A_]
  not bl
  and al, bl
  GetFlags
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; : Direct Addressing Mode
_tsb_d_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [edi]
  xchg ah, al
  mov bx, [A_]
  or  ax, bx
  GetFlags
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [edi]
  xchg ah, al
  mov bx, [A_]
  or  ax, bx
  GetFlags
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  xor edi, edi
  mov ax, [D_]
  add al, [esi]
  mov di, ax
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov al, [edi]
  mov bx, [A_]
  or  al, bl
  GetFlags
  mov [edi], al
  GetFlags
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov al, [edi]
  mov bx, [A_]
  or  al, bl
  GetFlags
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret

; : Absolute Addressing Mode
_tsb_a_:
  ; save all used registers
  push eax
  push ebx
  push edi
  push esi

  ; Is the A register a byte or word in size?
  mov ax, [Pm_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Px_]
  cmp ax, 1
  je  near .bytesize
  mov ax, [Pe_]
  cmp ax, 1
  je  near .bytesize

  ; it's a word, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_w
  mov edi, [SnesRam_]
  add edi, ebx
  mov ax, [edi]
  xchg ah, al
  mov bx, [A_]
  or  ax, bx
  GetFlags
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.trumem_w
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov ax, [edi]
  xchg ah, al
  mov bx, [A_]
  or  ax, bx
  GetFlags
  xchg ah, al
  mov [edi], ax
  GetFlags
  jmp .opdone
.bytesize

  ; it's a byte, get the operand to determine the memory addr
  mov esi, [IP_]
  inc esi
  mov ah, [esi]
  inc esi
  xor edi, edi
  mov al, [esi]
  mov di, ax
  add edi, [DB_]

  ; and finally we get to the actual fxn of the opcode
  xor ebx, ebx
  mov bx, di
  xor di, di
  cmp edi, 03F0000h
  jg  .trumem_b
  mov edi, [SnesRam_]
  add edi, ebx
  mov al, [edi]
  mov bx, [A_]
  or  al, bl
  GetFlags
  mov [edi], al
  GetFlags
  jmp .opdone
.trumem_b
  sub edi, 07E0000h
  add edi, ebx
  add edi, [SnesRam_]
  mov al, [edi]
  mov bx, [A_]
  or  al, bl
  GetFlags
  mov [edi], al
  GetFlags
  jmp .opdone
.opdone

  ; Restore all used registers
  pop esi
  pop edi
  pop ebx
  pop eax
  ret
