/*
Copyright (C) 2000 Chris Teague

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
*/

// emuschool.cpp : Defines the class behaviors for the application.
//

#include "emuschool.h"
#include "time.h"

#define HBLANK 115			// number of clock cycles to render one line
#define SCANLINES 240		// number of lines
#define VPERIOD 2875		// number of cycles a VBLANK lasts after it starts
#define benchmark 1

unsigned long total_6502_clocks;

void Run();
bool file_open(char *filename);
bool read_rom_header(char *filename, ROM_header &temp_header);
void read_ROM(char *filename, long int prg_size, generic_ram* rom, long int chr_size, generic_ram* vrom, BYTE trainer);


int main(int argc, char* argv[])
{
#ifdef benchmark
  struct timeval start_time, end_time;
  int total_usecs;

  total_6502_clocks = 0;  // count the number of simulated clock cylces
  gettimeofday(&start_time, NULL);
#endif

  if (argc > 1)  // then a filename was specified on command line
	{ 
	  if (file_open(argv[1])==false)   // open the file
		{
		  printf("error in input file\n");
		  exit(1);
		}
	  if (my_screen.init(&visual_ggi)==0x00)		// initialize the screen object
		{
		  printf("graphics init failed");
		  exit(1);
		}
	  printf("screen\n");
	  my_input.init(&visual_ggi);  // note that the screen must be init()'ed before the keyboard with ggi
	  Run();
	  my_screen.release_all();  // release the graphics objects
	  my_input.release_all();   // release the keyboard/joystick objects
#ifdef benchmark
	  gettimeofday(&end_time, NULL);
	  total_usecs = (end_time.tv_sec - start_time.tv_sec) * 1000000 + (end_time.tv_usec - start_time.tv_usec);
	  printf("Total time was %d uSec.\n", total_usecs);
	  printf("Total 6502 clock cycles simulated was %ld \n", total_6502_clocks);
	  double cps = (double)total_6502_clocks / ((double)total_usecs/(1000*1000));
	  printf("Simulated cycles per second (CPS) was %f \n", cps); 
	  printf("Efficiency was %f %% that of a real NES\n", cps/(1.7897725 * 1000 * 1000)*100 );
#endif
	  exit(0);
	}
  else  // no input arguements
	{
	  printf("error, no arguments\n");
	  printf("correct usage is:  emu <romname>\n\n");
	  printf("Keys are hardcoded for now:\n");
	  printf("Direction: arrows\n");
	  printf("A: period\n");
	  printf("B: slash\n");
	  printf("Start: [\n");
	  printf("Select: ]\n");
	  printf("Exit the emulator: ESC\n\n");
	  exit(1); // exit with error
	}
}


bool file_open(char *filename) 
{
  if (read_rom_header(filename, my_header) == false)		// read in first 16 bytes of ROM file
	{
	  return false;  // rom header not valid
	}
  my_prg_rom.set_size(my_header.prg_size);			// create the program rom		
  my_chr_vrom.set_size(my_header.chr_size);	// create chr VROM device
  my_sram.set_size(my_header.battery_backup*0x2000);
  my_ram.set_size(0x800);			// 2kb for internal ram
  my_vram.set_size(0x800);			// 2kb vram for ppu
  read_ROM(filename, my_header.prg_size, &my_prg_rom, my_header.chr_size, &my_chr_vrom, my_header.trainer);				
  switch(my_header.mapper)
  {
 	case 0:
 	  my_mapper = new mapper0(&my_prg_rom, &my_chr_vrom, &my_sram, my_header.prg_size);	// allocate memory for mapper				
 	  break;
 	case 2:
	  my_mapper = new mapper2(&my_prg_rom, &my_chr_vrom, &my_sram, my_header.prg_size);	// allocate memory for mapper				
 	  break;
	  
	default:
	  printf("Unsupported Mapper");
	  exit(1);
	  break;
	}
   my_ppu.set_values(&my_ram, &my_vram, my_header.mirroring, my_mapper, &my_screen); // create ppu 
   my_bus.set_values(&my_ram, my_mapper, &my_ppu, &my_gamepad1, &my_gamepad2);	// create memory bus
   my_cpu.set_bus(&my_bus);			// create cpu
   clock_cycles = 0;			// reset number of clocks

  scan_line = 0;				// start at top of screen
  my_cpu.reset();
  emulator_running = 0x01;	// 1 = running, start the emulation
  my_gamepad1.set_input_ptr(&my_input, 0);	// give ptr to input object to the gamepad object		
  my_gamepad2.set_input_ptr(&my_input, 1);	// give ptr to input object to the gamepad object
  return true;
}

bool read_rom_header(char *filename, ROM_header &temp_header)
{
// Opens the file specified in filename, and reads it's 16 byte iNES header
// Parameters: pointer to a ROM_header structure
// returns: header information in temp_header
  //{
 	FILE *fp;
 	BYTE header[16];	
	
 	if (!(fp = fopen(filename, "rb")))
	  {
 		printf("Unable to open ROM file\n");
		return false;
	  }
 	else
 	{
 		for (int i = 0; i <= 15; i++)	
 		{
 			header [i] = fgetc(fp);		// read in the header (first 16 bytes)	
 		}
 		fclose(fp);				// close the ROM file		

 		// make sure the first 4 bytes are 'NES'+0x1a
 		if ( (header[0] != 'N') || (header[1] != 'E') || (header[2] != 'S') || (header[3] != 0x1a) )
		  {
			printf("Not a valid NES ROM file\n");
			return false;  // the read failed
		  }
		
 		else
 		{
 			temp_header.prg_size = header[4]*0x4000;		// 16 KB pages for prg data
 			if (temp_header.chr_size == 0)	// if there is not VROM
 				temp_header.chr_size = 0x2000;	// create 8 KB of ram to use (not sure if this is right)
 			else		// there is VROM
 				temp_header.chr_size = header[5]*0x2000;		//  8 KB pages for chr data
 			temp_header.mapper = header[6];
 			temp_header.mapper >>= 4;					// shift upper bits into lower 4 bits
 			temp_header.mapper &= 0x0F;					// clear out upper 4 bits
 			header[7] &= 0xF0;				// make sure bottom is cleared
 			temp_header.mapper |= header[7];			// combine the two bytes into one;

			// set the battery back up info
			if(header[6] & 0x02)
			  temp_header.battery_backup=0x01;
			else
			  temp_header.battery_backup=0x00;
			
			// check if trainer exists
			if(header[6] & 0x04)
			  temp_header.trainer=0x01;
			else
			  temp_header.trainer=0x00;
			
			// check for 4 screen mirroring
			if(header[6] & 0x08)
			  {
				temp_header.four_screen_mirror=0x01;
				printf("Unsupported four screen miroring detected\n");
			  }
			else 
			  temp_header.four_screen_mirror=0x00;
			
			if (header[6] & 0x01)
			  temp_header.mirroring= 0x01;	// Vertical mirroring
			else
			  temp_header.mirroring= 0x00;	// Horizontal mirroring			

			return true;
		}		
	}	
}

void read_ROM(char * filename, long int prg_size, generic_ram* rom, long int chr_size, generic_ram* vrom, BYTE trainer)
// reads in the program data from the ROM file
{
	FILE *fp;			// file pointer	
	int i;

	if (!(fp = fopen(filename, "rb")))
		printf("Unable to open ROM file\n");
	else
	{
		fseek(fp,16,SEEK_SET);			// seek past the header
		if (trainer)
			fseek(fp, 512, SEEK_CUR);		// if a trainer exists, ignore it.
		
		for (i = 0; i< prg_size; i++)	// read in program data
			rom->write_byte(i, fgetc(fp));

		for (i=0; i<chr_size; i++)		// read in video data					
			vrom->write_byte(i, fgetc(fp));			
		
		fclose(fp);			// close the ROM file
	}

}

void Run() 
{
  while(emulator_running)
	{
	  // game code here
	  my_input.read_keyboard_state();
	  if (my_input.key_pressed(KEY_ESC))	// exit if escape pressed
		{
		  emulator_running = 0x00;  // exit the emulator
		}
	  else 
		{
		  if (my_input.key_pressed(KEY_F1))	// save screenshot if F1 pressed
			{
			  my_screen.screen_save();
			}
		  
		  emulator_running = my_cpu.execute_instruction(clock_cycles);
		  if (clock_cycles >= HBLANK)
			{
			  clock_cycles -= HBLANK;
#ifdef benchmark
			  total_6502_clocks += HBLANK;  // keep track of 6502 clock cycles completed
#endif
			  scan_line++;					// go to next scan line
			  if (scan_line == SCANLINES)		// we did the last one already
				{
				  my_ppu.start_vblank();	// a vblank should be performed				  
				  if ( my_ppu.read_byte(0x2000) & 0x80)
					{
					  my_cpu.vblank_interrupt(clock_cycles);	// execute an interrupt
					}
				}
			  if (scan_line == SCANLINES + VPERIOD/HBLANK)	
				{			
				  scan_line = 0;			// go back to top of screen.
				  my_ppu.end_vblank();	// end of vblank 
				}
			}	
		}
	}
  delete my_mapper;  // deallocate memory
}


