/*
Copyright (C) 2000 Chris Teague

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
*/

#include "ppu.h"

PPU::PPU()

{
	for (int i = 0; i<8; i++)
		ppu_regs[i]=0;			// reset all ppu registers
}

PPU::PPU(generic_ram* ram, BYTE mirroring, mapper* map, graphics* vid_screen)
// Constructor 
{	
	vram = ram;
	screen = vid_screen;
	ppu_mirroring = mirroring;	
	ppu_mapper = map;
	back_scroll_horizontal = 0;
	back_scroll_vertical = 0;
	for (int i = 0; i<8; i++)
		ppu_regs[i]=0;			// reset all ppu registers
}

PPU::~PPU()
// Destructor
{	
}

void PPU::set_values(generic_ram* ppu_ram, generic_ram* ppu_vram, BYTE mirroring, mapper* map, graphics* vid_screen)
// initialize values 
{	
	ram = ppu_ram;		// the NES internal ram
	vram = ppu_vram;	
	screen = vid_screen;
	ppu_mapper = map;
	ppu_mirroring = mirroring;	
	back_scroll_horizontal = 0;
	back_scroll_vertical = 0;
	ptn_table_updated = 0x01;		// indicate that the pattern table cache needs to be filled
	update_background = 0x01;		// indicates that the background needs to be redrawn	
	vram_data_buff = 0xff;			// for reads from 0x2007
	for (int i = 0; i<8; i++)
		ppu_regs[i]=0;			// reset all ppu registers
	
}

void PPU::start_vblank()
{
  //static int test=0;			// just to keep track of how many vblanks have occurred
  //TRACE("vblank #%d\n",test++);
	
  ppu_regs[2] |= 0x80;	// set bit #7 to indicate we are in vblank	
  
  display_pattern_table();	// fill back buffer
  //display_palette();
  draw_frame();				// draw to the back buffer
  
  screen->draw_frame();		// draw to the monitor		
}

void PPU::end_vblank()
{
	ppu_regs[2] &= 0x3f;	// clear bits 6 and 7 becuase vblank is over
}
	

BYTE PPU::read_byte(WORD address)
// Read from ppu registers
{
	BYTE temp_byte;
	WORD temp_word;
	
	switch(address)
	{
	case 0x2000:
		return ppu_regs[0];
		break;

	case 0x2001:
		return ppu_regs[1];
		break;

	case 0x2002:
		temp_byte = ppu_regs[2];
		ppu_regs[2] &= 0x7f;	// clear bit 7 after a read from this reg
		ppu_regs[5] = 0;		// reset 0x2005 after a read from this reg
		return temp_byte;
		break;

	case 0x2003:		// sprite ram address register
		return ppu_regs[3];
		break;

	case 0x2004:		// sprite ram IO register		
		ppu_regs[3] += 1;
		return sprite_ram[ (unsigned)(ppu_regs[3]-1) ];
		break;

	case 0x2005:
		return ppu_regs[5];
		break;

	case 0x2006:
		return ppu_regs[6];
		break;

	case 0x2007:
		temp_word = vram_addr_reg;		// save the address before we increment it

		if (ppu_regs[0] & 0x04)		// then bit 2 is set
			vram_addr_reg += 32;
		else
			vram_addr_reg++;		
		vram_addr_reg &= 0x3fff;	// make sure the address wraps at 0x4000
		if( (temp_word >= 0x3f00) && (temp_word < 0x4000) )
		{
			if (vram_addr_reg & 0x0010)	// then read from sprite palette
			{
				return sprite_palette[temp_word & 0x000f];				
			}
			else	// read from image palette
			{
				return image_palette[temp_word & 0x000f];				
			}			
		}
		else
		{
			temp_byte = vram_data_buff;		// read the value from the buffer (this will be returned)
			vram_data_buff = vram_read_byte(temp_word);		// read from VRAM
			return temp_byte;		// return the value that was in the buffer			
		}				
		break;

	default:
		printf("Invalid ppu memory read\n");
		return 0;
		break;
	}
}

void PPU::write_byte(WORD address, BYTE value)
// Write to ppu registers
{
  WORD temp_word;
  int i;	
  
  switch(address)
	{
	case 0x2000:
	  if ( (value & 0x13) ^ (ppu_regs[0] & 0x13) )	// then they are changeing pattern tables or name tables
		update_background = 0x01;
	  ppu_regs[0] = value;
	  break;
	  
	case 0x2001:
	  if ( (value & 0x08) ^ (ppu_regs[1] & 0x08) )	// then they have turned on/off background
		update_background = 0x01;
	  ppu_regs[1] = value;
	  break;
	  
	case 0x2002:
	  ppu_regs[2] = value;		
	  break;
	  
	case 0x2003:		// sprite_ram address register
	  ppu_regs[3] = value;
	  break;
	  
	case 0x2004:
	  //ppu_regs[4] = value;
	  sprite_ram[(unsigned)ppu_regs[3]] = value;		// write to sprite ram
	  ppu_regs[3] += 1;		// increment address register
	  break;
	  
	case 0x2005:
	  if (ppu_regs[5] == (BYTE)0x00)	//write to horizontal
		{					
		  if (((back_scroll_horizontal == 0)&&(value !=0))||((back_scroll_horizontal == 255)&&(value != 255)))
			{
			  update_background = 0x01;
			}
		  back_scroll_horizontal = value;
		  ppu_regs[5] = 0x01;		// toggle so next write is vertical
		}
	  else
		{			
		  if (value < 240)	// ignore if greater than 239
			{
			  if (((back_scroll_vertical == 0)&&(value !=0)) || ((back_scroll_vertical == 239)&&(value != 239)))
				{
				  update_background = 0x01;  // background needs to be redrawn
				}
			  back_scroll_vertical = value;  // update the scroll register
			}
		  ppu_regs[5] = 0x00;		// toggle so next write is horizontal
		}
	  break;
	  
	case 0x2006:		// VRAM address register
	  if (ppu_regs[6] == (BYTE)0x00)		// then we are writing the upper byte.
		{
			vram_addr_reg_buff = value & 0x3f;	// only take the lower 6 bits
			vram_addr_reg_buff <<= 8;		// put into upper byte
			ppu_regs[6] = 0x01;		// toggle to say next write should be lower half
		}
	  else
		{
		  vram_addr_reg = (vram_addr_reg_buff | value);	// put into lower byte
		  vram_addr_reg_buff = 0x0000;		// clear buffer
		  ppu_regs[6] = 0x00;		// toggle so next write is in upper half
		}
	  
	  break;
	  
	case 0x2007:	// VRAM IO register
	  ppu_regs[7] = value;		// not sure if this is necessary
	  if((vram_addr_reg >=0x2000)&&(vram_addr_reg<0x3000))
		{
		  vram_write_byte(vram_addr_reg, value);	// write to name/attribute tables
		  update_background = 0x01;		// tell ppu that background needs to be redrawn
		}
	  else if((vram_addr_reg >=0x3f00)&& (vram_addr_reg < 0x4000))	// then it's a palette access
		{			
			if (vram_addr_reg & 0x10)	// then it is a sprite palette write
			  {
				if (vram_addr_reg & 0x0003)	// then one of last two bits are non-zero
				  {
					sprite_palette[vram_addr_reg & 0x0f] = value;
				  }
				else // last two bits are zero, so mirror 3f10, 3f14, 3f18, 3f1c
				  {					
					sprite_palette[0x00] = value;					
					sprite_palette[0x04] = value;
					sprite_palette[0x08] = value;
					sprite_palette[0x0c] = value;
				  }				
			  }
			else		// it is an image palette write
			  {
				if (vram_addr_reg & 0x0003)	// then one of last two bits are non-zero
				  {
					image_palette[vram_addr_reg & 0x0f] = value;
				  }
				else // last two bits are zero, so mirror 3f00, 3f04, 3f18, 3f1c
				  {					
					image_palette[0x00] = value;					
					image_palette[0x04] = value;
					image_palette[0x08] = value;
					image_palette[0x0c] = value;
				  }								
			  }
			update_background = 0x01;		// tell ppu that background needs to be redrawn
			
		}		
		else if (vram_addr_reg < 0x2000)//  then it is a write to the pattern table
		  {
			ppu_mapper->write_byte(vram_addr_reg, value);
			ptn_table_updated = 0x01;
		  }
	  else
		{
		  printf("Invalid VRAM write address %X\n", vram_addr_reg);
		}
		if (ppu_regs[0] & 0x04)		// then bit 2 is set
		  vram_addr_reg += 32;
		else
		  vram_addr_reg++;
		vram_addr_reg &= 0x3fff;		// make sure it wraps around 0x4000
		break;
		
	case 0x4014:	// sprite ram DMA register
	  temp_word = (WORD)value << 8;		// calcualte address of 0x100 * value		
	  for(i=0; i<256; i++)		// transfer 256 bytes from NES internal ram to Sprite ram
		sprite_ram[i] = ram->read_byte(temp_word +i);		
	  break;
	  
	default:		
	  printf("Invalid ppu memory write\n");			
		break;
	}
  
  
}

void PPU::vram_write_byte(WORD address, BYTE value)
// handles writes to name and attribute table memory, including mirroring 
{
	if (ppu_mirroring)		// then it is vertical mirroring
	{
		vram->write_byte((address & 0x07ff), value);		// read from vram device
	}
	else	// horizontal mirroring
	{	
		if (address >= 0x2800)	// it's to tables 2 or 3
		{
			vram->write_byte( (address & 0x07ff) | 0x0400, value );
		}
		else // it's to tables 0 or 1
		{
			//TRACE("writing %X to address %X or %X\n", value, address, address & 0x0bff);
			vram->write_byte(address & 0x0bff, value);		// read from vram device
		}
	}
}



BYTE PPU::vram_read_byte(WORD address)
// Returns the byte located at the given address in the VRAM
{
	
	if (ppu_mirroring)		// then it is vertical mirroring
	{
		return vram->read_byte(address & 0x07ff);		// read from vram device
	}
	else	// horizontal mirroring
	{	
		if (address >= 0x2800)
		{
			return vram->read_byte( (address & 0x07ff) | 0x0400);	// read from vram device
		}
		else
		{
			return vram->read_byte(address & 0x0bff);	// read from vram device	
		}
		
	}
	
}

void PPU::display_palette()
// draws the palette on the screen
{
	#ifdef screen_on	
		screen->begin_set_pixel();		// lock the surface 
	#endif

	for(int i=0; i<16; i++)		// for each color in the palette
	{
		// draw a 10/10 block in the color
		for(int x=0; x<10; x++)
		{
			for(int y=0; y<10; y++)
			{
				#ifdef screen_on
				screen->set_pixel( i*10+x, y, image_palette[i]);	// image palette
				screen->set_pixel( i*10+x, y+20, sprite_palette[i]);	// sprite palette
				#endif
			}
		}
		
	}
	#ifdef screen_on	
		screen->end_set_pixel();		// unlock the surface 
	#endif
}

void PPU::draw_background()
// draws the background tiles
  // only updates the quadrants that are being displayed, based on the scroll values
  // I defined the quadrants as follows
  //  -------------------
  //  |        |        |
  //  |   2    |   3    |
  //  -------------------
  //  |        |        |
  //  {   0    |   1    |
  //  -------------------
  // where quadrant 0 always holds the name table pointed to by ppu_regs[0]

  // there are only 9 combinations of quadrants that could require redrawing
  // quad  value in tables_to_update
  // 0 -   %0101
  // 1 -   %0110
  // 2 -   %1001
  // 3 -   %1010 
  // 01 -  %0100
  // 02 -  %0001
  // 23 -  %1000
  // 13 -  %0010
  // 0123 -%0000

  //update 0 - 0x0x
  //update 1 - 0xx0
  //update 2 - x00x
  //update 3 - x0x0
{
  BYTE tables_to_update;
  WORD name_table_base_addr;	// the start of the name table
  BYTE start_col;			// used to say whether left 8 pixels are drawn in background
  BYTE tile_row, tile_col;	
  WORD tile_row_x32;		// just used so the calculation isn't done repeatedly
  WORD quad3_address;
  WORD quad2_address; // added after the switch
  WORD quad1_address;
  BYTE quad3_num;
  BYTE quad0_num;
  BYTE quad1_num;
  BYTE quad2_num;

  tables_to_update = 0x00;
  if( back_scroll_horizontal == 0)
	{
	  tables_to_update |= 0x01;
	}
  else if (back_scroll_horizontal == 255)
	{
	  tables_to_update |= 0x02;
	}
  if( back_scroll_vertical == 0)
	{
	  tables_to_update |= 0x04;
	}
  else if(back_scroll_vertical == 239)
	{
	  tables_to_update |= 0x08;
	}

  const BYTE NUM_ROWS = 30;			// these are just temporary, can be removed or made constants
  const BYTE NUM_COLS = 32;
  
  name_table_base_addr = ( (WORD)ppu_regs[0] << 10 );	// get name table from PPU ctrl reg
  name_table_base_addr &= 0x0fff;		// clear out top of address
  name_table_base_addr |= 0x2000;		// add the 0x2000 in
  
  quad1_address = (name_table_base_addr ^ 0x0400);	// base address of name table to be drawn in quadrant 1
  quad1_num = (BYTE)((quad1_address & 0x0c00) >> 10);		// name table number to be drawn in quadrant 1
  quad2_address = (name_table_base_addr ^ 0x0800);
  quad2_num = (BYTE)((quad2_address & 0x0c00) >> 10);
  quad3_address = (name_table_base_addr ^ 0x0c00);
  quad3_num = (BYTE)((quad3_address & 0x0c00) >> 10);
  quad0_num = (BYTE)((name_table_base_addr & 0x0c00) >> 10);
  
  start_col = !(ppu_regs[1] & 0x02);		// start_col = 1 means don't display left 8 
  
  // now draw the background
  if (ppu_regs[1] & 0x08)		// if background is enabled, draw it
	{
	  for (tile_row = 0; tile_row<NUM_ROWS; tile_row++)	
		{
		  tile_row_x32 = tile_row << 5;			
		  for (tile_col = start_col; tile_col<NUM_COLS; tile_col++)
			{
			  // only update the tables that need to be, based on the value calculated in tables_to_update
			  if ((tables_to_update & 0x0a)==0)  // then quad 0 needs to be updated
				{			
				  draw_tile(0, quad0_num, tile_row, tile_col, vram_read_byte( name_table_base_addr + tile_row_x32 + tile_col));
				}
			  if ((tables_to_update & 0x09)==0)
				{
				  draw_tile(1, quad1_num, tile_row, tile_col, vram_read_byte( quad1_address + tile_row_x32 + tile_col));
				}
			  if ((tables_to_update & 0x06)==0)
				{
				  draw_tile(2, quad2_num, tile_row, tile_col, vram_read_byte( quad2_address + tile_row_x32 + tile_col));
				}
			  if ((tables_to_update & 0x05)==0)
				{
				  draw_tile(3, quad3_num, tile_row, tile_col, vram_read_byte( quad3_address + tile_row_x32 + tile_col));
				}


			}
		}
	}
  else	// clear the background
	{
#ifdef screen_on	
	  screen->clear_NESBackground();
#endif
	}
  update_background =0x00;		// indicate that the background has been updated
  
}

void PPU::draw_frame()
// This function draws the current screen to the backbuffer
{	
	WORD ptn_table_addr;		// pattern table address	

	
	// if pattern table has been updated, need to reload the ptn_table_cache, then redraw screen
	if (ptn_table_updated != 0)	// only do this when pattern table has been updated
	{		
		fill_pattern_table_cache();		// no need to do this every time
		draw_background();		
	}
	else if (update_background != 0)// if name or attribute tables changed, or if a scroll has started,  we need to redraw screen, using cached pattern table		
	{
		draw_background();
		
	}
	// if nothing has been updated, we don't need to update the background buffer	
	
	#ifdef screen_on		
		screen->blit_NES_background(back_scroll_horizontal, back_scroll_vertical);		// draw the background
	#endif	
		
	// now it's time to draw the sprites on top of the background
	if (ppu_regs[1] & 0x10)		// if sprites are enabled, draw them
	{
		if (ppu_regs[0] & 0x20)	// if 8x16 sprites are enabled
		{
			for (int i=0; i<256; i+=4)
			{
				//TRACE("Drawing 8x16 sprite %d which is tile %X\n",i/4, sprite_ram[i+1]);
				draw_8x16_sprite(sprite_ram[i], sprite_ram[i+3], sprite_ram[i+1], sprite_ram[i+2]);
			}			
		}
		else	// draw 8x8 sprites
		{
		  ptn_table_addr = 0x0000;
		  if (ppu_regs[0] & 0x08)
			ptn_table_addr = 0x1000;
		  //		  ptn_table_addr = ((((WORD)ppu_regs[0]) & 0x0008) << 9);	// get pattern table from PPU ctrl reg			
		  for (int i=0; i<256; i+=4)
			{
			  if ((ppu_regs[0] & 0x08)&&(ptn_table_addr!=0x1000))
				printf("what the fuck???\n");
				//TRACE("Drawing sprite %d which is tile %X\n",i/4, sprite_ram[i+1]);				
				draw_sprite(sprite_ram[i], sprite_ram[i+3], sprite_ram[i+1], sprite_ram[i+2], ptn_table_addr);
			}
		}
	}	
	#ifdef screen_on
		screen->end_set_pixel();	
	#endif
	
	
}

void PPU::draw_8x16_sprite(BYTE Y, BYTE X, BYTE tile_index, BYTE attributes)
// draws an 8x16 sprite on the screen at location X,Y
{
  if (attributes & 0x80) // then flip it vertically (fold along horizontal line)
	{
	  if (tile_index & 0x01)  // then it is odd
		{
		  
		  draw_sprite(Y+8, X, tile_index, attributes, 0x1000);
		  draw_sprite(Y, X, tile_index-1, attributes, 0x1000);
		}
	  else  // it's even
		{
		  draw_sprite(Y+8, X, tile_index+1, attributes, 0x0000);	// draw the bottom half
		  draw_sprite(Y, X, tile_index, attributes, 0x0000);		// draw the top half of the sprite
		}
	}
  else // draw it normally
	{
	  if (tile_index & 0x01)  // then it is odd
		{
		  draw_sprite(Y, X, tile_index-1, attributes, 0x1000);
		  draw_sprite(Y+8, X, tile_index, attributes, 0x1000);
		}
	  else  // it's even
		{
		  draw_sprite(Y, X, tile_index, attributes, 0x0000);		// draw the top half of the sprite
		  draw_sprite(Y+8, X, tile_index+1, attributes, 0x0000);	// draw the bottom half
		}
	}
}

void PPU::draw_sprite(BYTE Y, BYTE X, BYTE tile_index, BYTE attributes, WORD ptn_table_addr)
// should be just like draw_tile() except it takes an absolute screen location
{	
  BYTE bit_field;
  BYTE temp_byte, temp_byte2;
  WORD temp_addr;
  BYTE color_index, attrib_color;
  int x_offset, y_offset;	
  
  attrib_color = ((attributes << 2) & 0x0C);		
  
  for(int c=0; c<8; c++)	// for each byte pair in this tile
	{
	  bit_field = 0x80;		// start with MSB set
	  temp_addr = ptn_table_addr + (WORD)(tile_index<<4)+c;
	  temp_byte = ppu_mapper->read_byte(temp_addr);
	  temp_byte2 = ppu_mapper->read_byte(temp_addr + 8);
	  
	  for(int d=0; d<8; d++)	// for each bit in this byte
		{						
		  color_index = (((temp_byte2 & bit_field) && 0x01) << 1) | (((temp_byte & bit_field) && 0x01));
		  color_index |= attrib_color;		// factor in the attribute table color index			
		  bit_field >>= 1;
		  if (color_index & 0x03)  // then the pixel is not transparent, draw it
			{
			  x_offset = d;
			  y_offset = c;
			  if (attributes & 0x40)	// if bit 6 set then flip horizontally (draw a vertical line down middle)
				x_offset = 7-d;
			  if (attributes & 0x80)	// if bit 7 set then flip vertically
				y_offset = 7-c;
			  
#ifdef screen_on
			  if (Y <= 232 )		// don't draw the sprite if it's off the screen (in vblank area of scanlines 240 through 262)
				{
				  if ( ((X + x_offset) > 8) || (ppu_regs[1] & 0x04) )	// check if leftmost column displayed
					{						
					  if (attributes & 0x20)	// then draw behind background
						screen->set_pixel_behind_background( X + x_offset, Y + y_offset, sprite_palette[color_index], sprite_palette[0]);
					  else	// draw it over the background
						screen->set_pixel( X + x_offset, Y + y_offset, sprite_palette[color_index]);					
					}
				}
#endif
			}
		}
	}
}

void PPU::draw_tile(BYTE quadrant_num, BYTE table_num, BYTE tile_row, BYTE tile_column, BYTE tile_index)
// table_num is the name table number, needed so we know which attribute table to look at
// quadrant_num indicates which of the four quadrants to draw it in 23
//																	01
{
  WORD ptn_table_addr;	
  BYTE color_index, attrib_color;
  WORD column, row;
  
  ptn_table_addr = (((WORD)ppu_regs[0] & 0x0010) << 8);	// get pattern table from PPU ctrl reg	
  //name_table_addr = 0x2000 + (0x400 * table_num);
  attrib_color = get_attrib_color(table_num, tile_row, tile_column);		
  column = (tile_column<<3)+ ((quadrant_num & 0x01)<<8);	
  if (quadrant_num & 0x02)
	row = (tile_row<<3);
  else
	row = (tile_row<<3) + 240;	
  
  for(int c=0; c<8; c++)	// for each row in this tile
	{				
	  for(int d=0; d<8; d++)	// for each col in this tile
		{						
		  if (ptn_table_addr == 0x0000)
			color_index = my_table_low[tile_index].get_color(d,c);
		  else
			color_index = my_table_high[tile_index].get_color(d,c);			
		  color_index |= attrib_color;		// factor in the attribute table color index
#ifdef screen_on			
		  screen->background_set_pixel( column+d, row+c, image_palette[color_index]);			
#endif
		}
	}
}

BYTE PPU::get_attrib_color(BYTE table_num, BYTE row, BYTE col)
// given the row 0-29 and the column 0-31 of the tile, returns the color info from attribute table
{
	BYTE attrib_table_row, attrib_table_col;
	WORD attrib_table_addr;
	BYTE color;
	BYTE bit_mask = 0x03;	// %00000011
	BYTE num_shifts=0;

	attrib_table_addr = 0x23c0 + (0x400 * table_num);	

	attrib_table_col = col >> 2;	// column divided by 4
	attrib_table_row = row >> 2;
	
	// read in the correct byte from the attribute table
	color = vram_read_byte(attrib_table_addr + (attrib_table_row<<3) + attrib_table_col);
	num_shifts = (row & 0x02);
	num_shifts |= ((col & 0x02) >> 1);
	num_shifts <<= 1;		// mult by 2
	bit_mask <<= num_shifts;
	color >>= num_shifts;
	color <<= 2;
	color &= 0x0c;	
	
	return color;
}

void PPU::fill_pattern_table_cache()
// updates the pattern_table_cache with the correct values from pattern table memory
{
  BYTE temp_byte, temp_byte2;
  WORD temp_addr;
  
  BYTE bit_field;
  BYTE color_index;
  
  bit_field = 0x80;		// start with MSB set
  
  int a, c, d;		// used as loop counters
  
  // pattern table at 0x0000
  for(a=0; a<256; a++)		// for each tile in this table
	{
	  for(c=0; c<8; c++)		// for each byte pair in this tile
		{
		  bit_field = 0x80;		// start with MSB set
		  temp_addr =  (a<<4) + c;	// index*16 + c
		  temp_byte = ppu_mapper->read_byte( temp_addr);
		  temp_byte2 = ppu_mapper->read_byte( temp_addr + 8);
		  for(d=0; d<8; d++)	// for each bit in this byte
			{
			  color_index = (((temp_byte2 & bit_field) && 0x01) << 1) | (((temp_byte & bit_field) && 0x01));				
			  bit_field >>= 1;
			  my_table_low[a].set_color(d, c, color_index);
			}
		}
	}	
  
  // pattern table at 0x1000
  for(a=0; a<256; a++)		// for each tile in this table
	{
	  for(c=0; c<8; c++)		// for each byte pair in this tile
		{
		  bit_field = 0x80;		// start with MSB set
		  temp_addr =  (a<<4) + c + 0x1000;	// index*16 + c
		  temp_byte = ppu_mapper->read_byte( temp_addr);
		  temp_byte2 = ppu_mapper->read_byte( temp_addr + 8);
		  for(d=0; d<8; d++)	// for each bit in this byte
			{			
			  color_index = (((temp_byte2 & bit_field) && 0x01) << 1) | (((temp_byte & bit_field) && 0x01));
			  bit_field >>= 1;
			  my_table_high[a].set_color(d, c, color_index);
			}
		}
	}
  ptn_table_updated = 0;		// indicate that the pattern table is current
}


void PPU::display_pattern_table()
// draws pattern table on the screen in a 256 wide by 128 pixel tall frame
{	

  int a,c,d;
  BYTE temp_byte, temp_byte2;
  WORD temp_addr;  
  BYTE bit_field;
  BYTE color_index;

  if (ptn_table_updated == 1)
	{
	  // pattern table at 0x0000
	  bit_field = 0x80;
	  for(a=0; a<256; a++)		// for each tile in this table
		{
		  for(c=0; c<8; c++)		// for each byte pair in this tile
			{
			  bit_field = 0x80;		// start with MSB set
			  temp_addr =  (a<<4) + c;	// index*16 + c
			  temp_byte = ppu_mapper->read_byte( temp_addr);
			  temp_byte2 = ppu_mapper->read_byte( temp_addr + 8);
			  for(d=0; d<8; d++)	// for each bit in this byte
				{
				  color_index = (((temp_byte2 & bit_field) && 0x01) << 1) | (((temp_byte & bit_field) && 0x01));				
				  bit_field >>= 1;
				  screen->pattern_table_set_pixel( ((a%16)*8)+d, ((a/16)*8)+c, color_index); 
				}
			}
		}
	  
	  // pattern table at 0x1000
	  for(a=0; a<256; a++)		// for each tile in this table
		{
		  for(c=0; c<8; c++)		// for each byte pair in this tile
			{
			  bit_field = 0x80;		// start with MSB set
			  temp_addr =  (a<<4) + c + 0x1000;	// index*16 + c
			  temp_byte = ppu_mapper->read_byte( temp_addr);
			  temp_byte2 = ppu_mapper->read_byte( temp_addr + 8);
			  for(d=0; d<8; d++)	// for each bit in this byte
				{			
				  color_index = (((temp_byte2 & bit_field) && 0x01) << 1) | (((temp_byte & bit_field) && 0x01));
				  bit_field >>= 1;
				  screen->pattern_table_set_pixel( ((a%16)*8)+d+128, ((a/16)*8)+c, color_index);
				}
			}
		}
	}
}
