/*
Copyright (C) 2000 Chris Teague

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
*/

#include "mem_bus.h"

mem_bus::mem_bus()
{
}

mem_bus::mem_bus(generic_ram* ram, mapper* map, PPU* ppu, gamepad* pad)
// Constructor 
// The memory bus must have a pointer to all the peripherals it will interface to
{
	my_cpu_ram = ram;		// assign the pointer to internal RAM
	my_mapper = map;
	my_ppu = ppu;
	my_gamepad1 = pad;
}

mem_bus::~mem_bus()
// Destructor
{
	my_cpu_ram = NULL;		// unassign pointer to internal RAM
}

void mem_bus::set_values(generic_ram* ram, mapper* map, PPU* ppu, gamepad* pad1, gamepad* pad2)
{
	my_cpu_ram = ram;		// assign the pointer to internal RAM
	my_mapper = map;
	my_ppu = ppu;
	my_gamepad1 = pad1;
	my_gamepad2 = pad2;
}



BYTE mem_bus::read_byte(WORD address)
// Read a byte from the memory bus
// This function must determine which device to read from, then perform the read
{	
	if (address < 0x2000)
	{		
		return my_cpu_ram->read_byte(address & 0x07FF);		// read from internal RAM
	}

	else if ( (address < 0x4000) )		// read from ppu registers
	{
		return my_ppu->read_byte(address & 0x2007);
	}

	else if ( (address <= 0xFFFF) && (address >= 0x6000) )		// read from mapper
		return my_mapper->read_byte(address);	

	else if (address == 0x4014)		// DMA sprite access
		return my_ppu->read_byte(address);

	else if (address <= 0x4015)			// then it is a sound register write
		return 0;		// do nothing yet, sound not implemented

	else if (address == 0x4016)			// then it is a gamepad1 register read
		return my_gamepad1->read_byte(address);
	else if (address == 0x4017)			// then it is a gamepad2 register read
	  {
		BYTE tempo = my_gamepad2->read_byte(address);
		//printf("attempt to read gamepad 2 returned %x\n", tempo);
		return tempo;
	  }

	else return 0;		// should not happen
}

void mem_bus::write_byte(WORD address, BYTE value)
// Write byte to bus
// This function must determine which device to write to, then perform the write
{	
	if (address < 0x2000)						// write to internal RAM
	{		
		my_cpu_ram->write_byte(address & 0x07ff, value);		
	}

	else if (address < 0x4000)		// write to ppu registers
		my_ppu->write_byte((address & 0x2007), value);

	else if ( (address <= 0xFFFF) && (address >= 0x6000) )		// write to mapper
		my_mapper->write_byte(address, value);	

	else if (address == 0x4014)			// sprite DMA access
		my_ppu->write_byte(address, value);

	else if (address <= 0x4015)			// then it is a sound register write
		address +=0;		// do nothing yet, sound not implemented

	else if (address == 0x4016)			// then it is a gamepad1 register write
	  {
		//printf("wrote %x to 4016\n", value);
		if ( my_gamepad1->write_byte(address, value) == 0x01) // then a reset took place
		  {
			my_gamepad2->reset();  // need to reset gamepad 2 as well
		  }

	  }
	else if (address == 0x4017)			// then it is a gamepad2 register write
	  {
		//printf("wrote %x to 4017\n", value);
		my_gamepad2->write_byte(address, value);
	  }

	else 
	{	
		printf("error with memory write\n");		
	}


}

WORD mem_bus::read_address(WORD address)
{
	WORD tempWord;
	BYTE tempByte;	
	
	tempByte = read_byte(address);	// read in lower half of address	
	tempWord = read_byte(address+1);	// read in upper half of address	
	tempWord <<= 8;					// shift lower half to top half;	
	tempWord |= (WORD)tempByte & 0x00FF;			// put them together to get address
	return tempWord;
}
