/*
Copyright (C) 2000 Chris Teague

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
*/

#include "stdafx.h"
#include "graphics.h"
#include "generic_ram.h"
#include "data_types.h"


#define SCREEN_WIDTH 320	// The screen resolution to use
#define SCREEN_HEIGHT 240	//


#define SAFE_RELEASE(ptr)  do { if ((ptr) != NULL) { (ptr)->Release(); (ptr) = NULL; } } while (0)


BOOL SaveBMP (LPDIRECTDRAWSURFACE4 SaveSurface, LPDIRECTDRAWSURFACE4 FrontBuffer,const char * szFilename);
BOOL SaveBMPToFile (LPDIRECTDRAWSURFACE4 SaveSurface,LPDIRECTDRAWSURFACE4 FrontBuffer, HANDLE file_out);

graphics::graphics()					// constructor
{
	g_pDD         = NULL;	// DirectDraw object
	g_pDDSPrimary = NULL;   // DirectDraw primary surface
	g_pDDSBack    = NULL;   // DirectDraw back surface	
	g_pNESBackground=NULL;	// NES pattern table
	g_bActive     = FALSE;  // App is running/active
	g_bReady      = FALSE;  // App is ready for updates
	g_bWindowed   = FALSE;   // App is in windowed mode
}

graphics::~graphics()
// Destructor 
{
	ReleaseAllObjects();

	// release the DD4 interface to avoid resource leaking
	SAFE_RELEASE(g_pDD);
}

//-----------------------------------------------------------------------------
// Name: ReleaseAllObjects()
// Desc: Release all DDraw objects we use
//-----------------------------------------------------------------------------
HRESULT graphics::ReleaseAllObjects()
{
    if (g_pDD != NULL)
    {
        g_pDD->SetCooperativeLevel(AfxGetMainWnd()->GetSafeHwnd(), DDSCL_NORMAL);
    	SAFE_RELEASE(g_pNESBackground);
    	SAFE_RELEASE(g_pDDSBack);
    	SAFE_RELEASE(g_pDDSPrimary);
    }
    return DD_OK;
}

//-----------------------------------------------------------------------------
// Name: InitFail()
// Desc: This function is called if an initialization function fails
//-----------------------------------------------------------------------------
HRESULT graphics::InitFail(HRESULT hRet, LPCTSTR szError, ...)
{
    char            szBuff[128];
    va_list         vl;

    va_start(vl, szError);
    vsprintf(szBuff, szError, vl);
    ReleaseAllObjects();
    MessageBox(AfxGetMainWnd()->GetSafeHwnd(), szBuff, "EmuSchool NES Emulator", MB_OK);
    DestroyWindow(AfxGetMainWnd()->GetSafeHwnd());
    va_end(vl);
    return hRet;
}

void graphics::clear_NESBackground()
// draws a black background, used when background is disabled
{
	DDBLTFX     ddbltfx;
    // Use the blter to do a color fill to clear the background
    ZeroMemory(&ddbltfx, sizeof(ddbltfx));
    ddbltfx.dwSize = sizeof(ddbltfx);
    ddbltfx.dwFillColor = 13;	// 13th color in the palette is black
    
	// This line clears the back buffer
	g_pNESBackground->Blt(NULL, NULL, NULL, DDBLT_COLORFILL | DDBLT_WAIT, &ddbltfx);	

}

BOOL graphics::blit_NES_background(BYTE horizontal_scroll, BYTE vertical_scroll)
// copies the pre-rendered background onto the backbuffer
{
    DDBLTFX     ddbltfx;	
	DDCOLORKEY color_key;
   
    // Use the blter to do a color fill to clear the back buffer
    ZeroMemory(&ddbltfx, sizeof(ddbltfx));
    ddbltfx.dwSize = sizeof(ddbltfx);
    ddbltfx.dwFillColor = 13;	// 13th color in the palette is black
    
	// This line clears the back buffer
	g_pDDSBack->Blt(NULL, NULL, NULL, DDBLT_COLORFILL | DDBLT_WAIT, &ddbltfx);	
	
	color_key.dwColorSpaceHighValue = 0;
	color_key.dwColorSpaceLowValue = 0;
	g_pNESBackground->SetColorKey(DDCKEY_SRCBLT, &color_key);

	RECT dest_rect, source_rect;

	dest_rect.left = 0;		// blit to the top left corner of the backbuffer
	dest_rect.right = 255;
	dest_rect.top = 0;
	dest_rect.bottom = 239;	
	
	source_rect.left = 0;		// blit from the lower left quadrant by default
	source_rect.right = 255;
	source_rect.top = 0;
	source_rect.bottom=239;
	
	source_rect.left += horizontal_scroll;
	source_rect.right += horizontal_scroll;
	source_rect.top += vertical_scroll;
	source_rect.bottom += vertical_scroll;
	
	
	g_pDDSBack->Blt(&dest_rect, g_pNESBackground, &source_rect, (DDBLT_WAIT | DDBLT_KEYSRC), NULL);		

    return TRUE;
}

//-----------------------------------------------------------------------------
// Name: InitSurfaces()
// Desc: Create all the needed DDraw surfaces and set the coop level
//-----------------------------------------------------------------------------
HRESULT graphics::InitSurfaces()
{
    HRESULT		        hRet;
    DDSURFACEDESC2      ddsd;
    DDSCAPS2            ddscaps;
    LPDIRECTDRAWCLIPPER pClipper;

    if (g_bWindowed)
    {
        // Get normal windowed mode
        hRet = g_pDD->SetCooperativeLevel(AfxGetMainWnd()->GetSafeHwnd(), DDSCL_NORMAL);
        if (hRet != DD_OK)
            return InitFail(hRet, "SetCooperativeLevel FAILED");

    	// Get the dimensions of the viewport and screen bounds
    	GetClientRect(AfxGetMainWnd()->GetSafeHwnd(), &g_rcViewport);
    	GetClientRect(AfxGetMainWnd()->GetSafeHwnd(), &g_rcScreen);
    	ClientToScreen(AfxGetMainWnd()->GetSafeHwnd(), (POINT*)&g_rcScreen.left);
    	ClientToScreen(AfxGetMainWnd()->GetSafeHwnd(), (POINT*)&g_rcScreen.right);

        // Create the primary surface
        ZeroMemory(&ddsd,sizeof(ddsd));
        ddsd.dwSize = sizeof(ddsd);
        ddsd.dwFlags = DDSD_CAPS;
        ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;
        hRet = g_pDD->CreateSurface(&ddsd, &g_pDDSPrimary, NULL);
        if (hRet != DD_OK)
            return InitFail(hRet, "CreateSurface FAILED");

        // Create a clipper object since this is for a Windowed render
        hRet = g_pDD->CreateClipper(0, &pClipper, NULL);
        if (hRet != DD_OK)
            return InitFail(hRet, "CreateClipper FAILED");

        // Associate the clipper with the window
        pClipper->SetHWnd(0, AfxGetMainWnd()->GetSafeHwnd());
        g_pDDSPrimary->SetClipper(pClipper);
    	SAFE_RELEASE(pClipper);

        // Get the backbuffer. For fullscreen mode, the backbuffer was created
        // along with the primary, but windowed mode still needs to create one.
        ddsd.dwFlags        = DDSD_WIDTH | DDSD_HEIGHT | DDSD_CAPS;
        ddsd.dwWidth        = SCREEN_WIDTH;
        ddsd.dwHeight       = SCREEN_HEIGHT;
        ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
        hRet = g_pDD->CreateSurface(&ddsd, &g_pDDSBack, NULL);
        if (hRet != DD_OK)
            return InitFail(hRet, "CreateSurface2 FAILED");
    }
    else
    {
        // Get exclusive mode
        hRet = g_pDD->SetCooperativeLevel(AfxGetMainWnd()->GetSafeHwnd(), DDSCL_EXCLUSIVE | DDSCL_ALLOWREBOOT |
                                                DDSCL_FULLSCREEN);
        if (hRet != DD_OK)
            return InitFail(hRet, "SetCooperativeLevel FAILED");

        // Set the video mode to SCREEN_WIDTHxSCREEN_HEIGHTx8
        hRet = g_pDD->SetDisplayMode( SCREEN_WIDTH, SCREEN_HEIGHT, 8, 0, 0);
        if (hRet != DD_OK)
            return InitFail(hRet, "SetDisplayMode FAILED");

    	// Get the dimensions of the viewport and screen bounds
    	// Store the rectangle which contains the renderer
    	SetRect(&g_rcViewport, 0, 0, SCREEN_WIDTH, SCREEN_HEIGHT );
    	memcpy(&g_rcScreen, &g_rcViewport, sizeof(RECT) );

        // Create the primary surface with 1 back buffer
        ZeroMemory(&ddsd,sizeof(ddsd));
        ddsd.dwSize = sizeof(ddsd);		
        ddsd.dwFlags = DDSD_CAPS |
                       DDSD_BACKBUFFERCOUNT;
        ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE |
                			  DDSCAPS_FLIP |
                			  DDSCAPS_COMPLEX;
        ddsd.dwBackBufferCount = 1;
        hRet = g_pDD->CreateSurface( &ddsd, &g_pDDSPrimary, NULL);
        if (hRet != DD_OK)
            return InitFail(hRet, "CreateSurface FAILED");
        ddscaps.dwCaps = DDSCAPS_BACKBUFFER;
        hRet = g_pDDSPrimary->GetAttachedSurface(&ddscaps, &g_pDDSBack);
        if (hRet != DD_OK)
            return InitFail(hRet, "GetAttachedSurface FAILED");
    }
	// now create the surface to hold the pattern table
	ZeroMemory(&ddsd, sizeof(ddsd));		
	ddsd.dwSize = sizeof(ddsd);	
	ddsd.dwFlags = DDSD_CAPS | DDSD_WIDTH | DDSD_HEIGHT;
	ddsd.dwWidth = 512;		// two screens wide
	ddsd.dwHeight = 480;	// two screens tall
	if (g_pDD->CreateSurface(&ddsd, &g_pNESBackground ,NULL) != DD_OK)
		return InitFail(hRet, "CreateSurface FAILED");
    return DD_OK;
}

//-----------------------------------------------------------------------------
// Name: ChangeCoopLevel()
// Desc: Called when the user wants to toggle between Full-Screen & Windowed
//-----------------------------------------------------------------------------
HRESULT graphics::ChangeCoopLevel()
{
    HRESULT hRet;

	g_bReady = FALSE;
	if (g_bWindowed)
        GetWindowRect(AfxGetMainWnd()->GetSafeHwnd(), &g_rcWindow);
    g_bWindowed = !g_bWindowed;    
    
    // Release all objects that need to be re-created for the new device
    if (FAILED(hRet = ReleaseAllObjects()))
        return InitFail(hRet, "ReleaseAllObjects FAILED");

    // In case we're coming from a fullscreen mode, restore the window size
    if (g_bWindowed)
    {
        SetWindowPos(AfxGetMainWnd()->GetSafeHwnd(), 
			         HWND_NOTOPMOST, g_rcWindow.left, g_rcWindow.top,
                     (g_rcWindow.right - g_rcWindow.left), 
                     (g_rcWindow.bottom - g_rcWindow.top), SWP_SHOWWINDOW );
    }

    // Re-create the surfaces
    hRet = InitSurfaces();
	g_bReady = TRUE;
    return hRet;
}

HRESULT graphics::init()
{
	
    HRESULT         hRet;
    LPDIRECTDRAW    pDD = NULL;            // DD1 interface, used to get DD4 interface

	// Save the window size/pos for switching modes
    GetWindowRect(AfxGetMainWnd()->GetSafeHwnd(), &g_rcWindow);

    // Create the main DirectDraw object
    hRet = DirectDrawCreate( NULL, &pDD, NULL);
    if (hRet != DD_OK)
        return InitFail(hRet, "DirectDrawCreate FAILED");

    // Fetch DirectDraw4 interface
    hRet = pDD->QueryInterface(IID_IDirectDraw4, (LPVOID *)&g_pDD);
    if (hRet != DD_OK)
        return InitFail(hRet, "QueryInterface FAILED");

    // Initialize all the surfaces we need
    hRet = InitSurfaces();
	init_palette();			// set up the palette to use
    if (FAILED(hRet))
    	return FALSE;

    g_bReady = TRUE;

	// release the DD1 interface to avoid resource leaking
	SAFE_RELEASE(pDD);

	return DD_OK;
}


void graphics::draw_frame()
{
	HRESULT hRet;

	if (g_bWindowed)
    {		
        //hRet = g_pDDSPrimary->Blt(&g_rcScreen, g_pDDSBack, &g_rcViewport, DDBLT_WAIT, NULL);
		hRet = g_pDDSPrimary->Blt(NULL, g_pDDSBack, NULL, DDBLT_WAIT, NULL);
    }
    else
    {
        // Else we are in fullscreen mode, so perform a flip.
        hRet = g_pDDSPrimary->Flip( NULL, DDFLIP_WAIT );
    }
    
    if (hRet == DDERR_SURFACELOST )
    {
        hRet = g_pDDSPrimary->Restore();    
    }
    

}

void graphics::init_palette()
// creates the palette to use
{
	PALETTEENTRY palette[256];		

	ZeroMemory(palette, 256*sizeof(PALETTEENTRY));
	
	// This palette stolen from NESTICLE.PAL
	NES_pal nes_palette[64] =
	{
		{0x7f,0x7f,0x7f}, {0x00,0x00,0xFF}, {0x00,0x00,0xBF}, {0x47,0x2b,0xbf},
		{0x97,0x00,0x87}, {0xab,0x00,0x23}, {0xab,0x13,0x00}, {0x8b,0x17,0x00},
		{0x53,0x30,0x00}, {0x00,0x78,0x00}, {0x00,0x6b,0x00}, {0x00,0x5b,0x00},
		{0x00,0x43,0x58}, {0x00,0x00,0x00}, {0x00,0x00,0x00}, {0x00,0x00,0x00},

		{0xbf,0xbf,0xbf}, {0x00,0x78,0xF8}, {0x00,0x58,0xF8}, {0x6B,0x47,0xff},
		{0xdb,0x00,0xcd}, {0xe7,0x00,0x5b}, {0xF8,0x38,0x00}, {0xe7,0x5f,0x13},
		{0xaf,0x7f,0x00}, {0x00,0xB8,0x00}, {0x00,0xab,0x00}, {0x00,0xab,0x47},
		{0x00,0x8b,0x8b}, {0x00,0x00,0x00}, {0x00,0x00,0x00}, {0x00,0x00,0x00},

		{0xF8,0xF8,0xF8}, {0x3f,0xbf,0xFF}, {0x6b,0x88,0xFF}, {0x98,0x78,0xF8},
		{0xF8,0x78,0xf8}, {0xF8,0x58,0x98}, {0xF8,0x78,0x58}, {0xFF,0xa3,0x47},
		{0xF8,0xb8,0x00}, {0xb8,0xf8,0x18}, {0x5b,0xDb,0x57}, {0x58,0xf8,0x98},
		{0x00,0xeb,0xdb}, {0x78,0x78,0x78}, {0x00,0x00,0x00}, {0x00,0x00,0x00},

		{0xFF,0xFF,0xFF}, {0xa7,0xe7,0xFF}, {0xB8,0xB8,0xF8}, {0xd8,0xB8,0xF8},
		{0xF8,0xB8,0xf8}, {0xFb,0xa7,0xC3}, {0xF0,0xd0,0xB0}, {0xFF,0xe3,0xAb},
		{0xFb,0xDb,0x7b}, {0xd8,0xf8,0x78}, {0xb8,0xf8,0xB8}, {0xb8,0xF8,0xd8},
		{0x00,0xff,0xFF}, {0xf8,0xD8,0xf8}, {0x00,0x00,0x00}, {0x00,0x00,0x00}
	};

	for(int i=0; i<64; i++)
	{
		palette[i].peRed = nes_palette[i].red;
		palette[i].peGreen = nes_palette[i].green;
		palette[i].peBlue = nes_palette[i].blue;
		palette[i].peFlags = PC_NOCOLLAPSE;
	}	

	if ((g_pDD->CreatePalette(DDPCAPS_8BIT | DDPCAPS_INITIALIZE | DDPCAPS_ALLOW256, 
                              palette, 
                              &lpddpal, 
                              NULL)) != DD_OK)
    {
		TRACE("Error creating palette\n");
    }
	else
	{
		g_pDDSPrimary->SetPalette(lpddpal);
        // release the pallete so no resource leaks
        SAFE_RELEASE(lpddpal);
	}
}

void graphics::begin_background_set_pixel()
// for the directX implementation this locks the background surface.  Needs to be changed
// for other graphics platforms
{
	ZeroMemory(&ddsd_background_set_pixel, sizeof(ddsd_background_set_pixel));
	ddsd_background_set_pixel.dwSize = sizeof(ddsd_background_set_pixel);
	g_pNESBackground->Lock(NULL, &ddsd_background_set_pixel, DDLOCK_SURFACEMEMORYPTR | DDLOCK_WAIT, NULL);	
	background_vid_buf = (UCHAR *)ddsd_background_set_pixel.lpSurface;	
}

void graphics::end_background_set_pixel()
// for the directX implementation this unlocks the background surface.
{
	g_pNESBackground->Unlock(NULL);
}

void graphics::background_set_pixel(int x, int y, BYTE color_index)
// this should not be using the pattern table surface, but it is for now
// The x and y are in screen coordinates, and the color_index is the color value (should be an index into the NES palette eventually)
{	
	background_vid_buf[x+y*ddsd_background_set_pixel.lPitch] = color_index;	// set the value	
}

void graphics::begin_set_pixel()
{
	ZeroMemory(&ddsd_set_pixel, sizeof(ddsd_set_pixel));
	ddsd_set_pixel.dwSize = sizeof(ddsd_set_pixel);
	g_pDDSBack->Lock(NULL, &ddsd_set_pixel, DDLOCK_SURFACEMEMORYPTR | DDLOCK_WAIT, NULL);	
	vid_buf = (UCHAR *)ddsd_set_pixel.lpSurface;	
}

void graphics::end_set_pixel()
{
	g_pDDSBack->Unlock(NULL);
}

void graphics::set_pixel(int x, int y, BYTE color_index)
// this should not be using the pattern table surface, but it is for now
// The x and y are in screen coordinates, and the color_index is the color value (should be an index into the NES palette eventually)
{	

	vid_buf[x+y*ddsd_set_pixel.lPitch] = color_index;	// set the value
}

void graphics::set_pixel_behind_background(int x, int y, BYTE color_index, BYTE background_color_index)
// just like set_pixel, but only draws the pixel if there is not something on the background already
{
	if (vid_buf[x+y*ddsd_set_pixel.lPitch] == background_color_index)
		vid_buf[x+y*ddsd_set_pixel.lPitch] = color_index;	// set the value
	
}

void graphics::screen_save()
// This is the screen save function.  For now, it only has the option to save to bmp format
{
	SaveBMP(g_pDDSPrimary, g_pDDSPrimary, "screenshot.bmp");
}

BOOL SaveBMP (LPDIRECTDRAWSURFACE4 SaveSurface, LPDIRECTDRAWSURFACE4 FrontBuffer,const char * szFilename)

{
	HANDLE file_out;

	// Create the outputfile...
	file_out = CreateFile(szFilename, GENERIC_WRITE, FILE_SHARE_WRITE, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
	if(file_out == INVALID_HANDLE_VALUE)
	{
		AfxMessageBox("Surface - Could not open output file for Bitmap: Screenshot.bmp!");
		return FALSE;
	}
	SaveBMPToFile(SaveSurface,FrontBuffer, file_out);
	CloseHandle(file_out);

 return TRUE;
}

BOOL SaveBMPToFile (LPDIRECTDRAWSURFACE4 SaveSurface,LPDIRECTDRAWSURFACE4 FrontBuffer, HANDLE file_out)
{
	HRESULT           rval;
	DWORD             numwrite;
	BITMAPFILEHEADER  fh;
	BITMAPINFOHEADER  bi;
	DWORD             outpixel;
	int               outbyte, loop, loop2, BufferIndex;
	BYTE              *WriteBuffer; 
	DDSURFACEDESC2     ddsd;
	int              Width, Height, Pitch;

	//First we need a ddsdription of the surface
	ZeroMemory(&ddsd, sizeof(ddsd));
	ddsd.dwSize = sizeof(ddsd); 
	rval = SaveSurface -> GetSurfaceDesc (&ddsd);
	if (rval != DD_OK)
	{
		AfxMessageBox("Couldn't get surface ddsd for bitmap save");
		return FALSE; 
	}

	//Setup output buffer stuff, since Windows has paging and we're in flat mode, I just made
	//it as big as the bitmap 
	BufferIndex = 0;
	Width = ddsd.dwWidth;
	Height = ddsd.dwHeight;
	Pitch = ddsd.lPitch;
	WriteBuffer = new BYTE [Width* Height * 3];  //width*height*24-bit

	//Write the file header
	((char *)&(fh . bfType))[0] = 'B';
	((char *)&(fh . bfType))[1] = 'M';
	fh . bfSize = (long)(sizeof (BITMAPINFOHEADER)+sizeof (BITMAPFILEHEADER)+Width*Height*3); //Size in BYTES
	fh . bfReserved1 = 0;
	fh . bfReserved2 = 0;
	fh . bfOffBits = sizeof (BITMAPINFOHEADER)+sizeof (BITMAPFILEHEADER);
	bi . biSize = sizeof (BITMAPINFOHEADER);
	bi . biWidth =Width;
	bi . biHeight =Height;
	bi . biPlanes = 1;
	bi . biBitCount = 24;
	bi . biCompression = BI_RGB;
	bi . biSizeImage = 0;
	bi . biXPelsPerMeter = 10000;
	bi . biYPelsPerMeter = 10000;
	bi . biClrUsed = 0;
	bi . biClrImportant = 0;

	WriteFile (file_out, (char *) &fh,sizeof (BITMAPFILEHEADER),&numwrite,NULL);
	WriteFile (file_out, (char *) &bi,sizeof (BITMAPINFOHEADER),&numwrite,NULL);
	if (ddsd.ddpfPixelFormat.dwRGBBitCount==32)    //16 bit surfaces
	{
		//lock the surface and start filling the output
        //buffer
		ZeroMemory(&ddsd, sizeof(ddsd));
		ddsd.dwSize = sizeof(ddsd);

		rval = SaveSurface -> Lock(NULL,&ddsd, DDLOCK_WAIT,NULL);
		if (rval != DD_OK)
		{
			AfxMessageBox("Couldn't lock source");
			delete [] WriteBuffer;
			return FALSE;
		}
		BYTE *Bitmap_in = (BYTE*)ddsd.lpSurface;
		for (loop =Height-1;loop>=0;loop--)    //Loop bottom up
			for (loop2=0;loop2<Width;loop2++)
			{
				outpixel = *((DWORD *)(Bitmap_in+loop2*4 + loop * Pitch)); //Load a word
                //Load up the Blue component and output it

				outbyte = (((outpixel)&0x000000ff));//blue
				WriteBuffer [BufferIndex++] = outbyte;

				//Load up the green component and output it 

				outbyte = (((outpixel>>8)&0x000000ff)); 
					WriteBuffer [BufferIndex++] = outbyte;

				//Load up the red component and output it 

				outbyte = (((outpixel>>16)&0x000000ff));
				WriteBuffer [BufferIndex++] = outbyte;
			}
			//At this point the buffer should be full, so just write it out
			WriteFile (file_out, WriteBuffer,BufferIndex,&numwrite,NULL);

			//Now unlock the surface and we're done
			SaveSurface -> Unlock(NULL);
	}
	if (ddsd.ddpfPixelFormat.dwRGBBitCount==24)    //24 bit surfaces
	{
		//So easy just lock the surface and output
		//lock the surface and start filling the output
		//buffer

                                ZeroMemory(&ddsd, sizeof(ddsd));
                                ddsd.dwSize = sizeof(ddsd);
                                rval = SaveSurface -> Lock(NULL,&ddsd, DDLOCK_WAIT,NULL);
                                if (rval != DD_OK)
                                {
                                        AfxMessageBox("Couldn't lock source");
                                        delete [] WriteBuffer;
                                        return FALSE;
                                }

                                BYTE *Bitmap_in = (BYTE*)ddsd.lpSurface;

                                 for (loop =Height-1;loop>=0;loop--)    //Loop bottom up
                                 for (loop2=0;loop2<Width;loop2++)
                                  {

                                   //Load up the Blue component and output it
                                   
                                   WriteBuffer [BufferIndex++] = *(Bitmap_in+loop2*3+2 + loop * Pitch); //Bug fix 6-5

                                   //Load up the green component and output it 

                                   WriteBuffer [BufferIndex++] = *(Bitmap_in+loop2*3+ 1 + loop * Pitch); //Bug fix 6-5

                                   //Load up the red component and output it 

                                   WriteBuffer [BufferIndex++] = *(Bitmap_in+loop2*3 + loop * Pitch);
                                  }
                                    
                                 //At this point the buffer should be full, so just write it out

                                        WriteFile (file_out, WriteBuffer,BufferIndex,&numwrite,NULL);

                                //Now unlock the surface and we're done

                                        SaveSurface -> Unlock(NULL);


 }
 else if (ddsd.ddpfPixelFormat.dwRGBBitCount==16)       //16 bit surfaces
 {

                          //lock the surface and start filling the output
                          //buffer

                                ZeroMemory(&ddsd, sizeof(ddsd));
                                ddsd.dwSize = sizeof(ddsd);
                                rval = SaveSurface -> Lock(NULL,&ddsd, DDLOCK_WAIT,NULL);
                                if (rval != DD_OK)
                                {
                                        AfxMessageBox("Couldn't lock source");
                                        delete [] WriteBuffer;
                                        return FALSE;
                                }

                                BYTE *Bitmap_in = (BYTE*)ddsd.lpSurface;


                          /*

                          According to DirectX docs, dwRGBBitCount is 2,4,8,16,24,32, BUT what about 15-bit surfaces
                          (5,5,5) I don't really know if its needed but here we check the green bitmask to see
                          if 5 or 6 bits are used for green.
                          
                          If the green bitmask equals 565 mode, do 16-bit mode, otherwise do 15-bit mode
                          NOTE: We are reversing the component order (ie. BGR instead of RGB)
                                and we are outputting it bottom up because BMP files are backwards and upside down.

                          */

                                if (ddsd .ddpfPixelFormat . dwGBitMask ==  0x07E0)
                                  {
                                         for (loop =Height-1;loop>=0;loop--)    //Loop bottom up
                                         for (loop2=0;loop2<Width;loop2++)
                                          {

                                           outpixel = *((WORD *)(Bitmap_in+loop2*2 + loop * Pitch)); //Load a word


                                           //Load up the Blue component and output it
                                           
                                           outbyte = (8*((outpixel)&0x001f));//blue
                                           WriteBuffer [BufferIndex++] = outbyte;

                                           //Load up the green component and output it 

                                           outbyte = (4*((outpixel>>5)&0x003f)); 
                                                WriteBuffer [BufferIndex++] = outbyte;

                                           //Load up the red component and output it 

                                           outbyte = (8*((outpixel>>11)&0x001f));
                                           WriteBuffer [BufferIndex++] = outbyte;
                                          }
                                    
                                   }
                                 else //Assume 555 mode. 15-bit mode
                                   {
                                         for (loop =Height-1;loop>=0;loop--)    //Loop bottom up
                                         for (loop2=0;loop2<Width;loop2++)
                                          {

                                           outpixel = *((WORD *)(Bitmap_in+loop2*2 + loop * Pitch)); //Load a word

                                           //Load up the Blue component and output it
                                           
                                           outbyte = (8*((outpixel)&0x001f));//blue
                                           WriteBuffer [BufferIndex++] = outbyte;

                                           //Load up the green component and output it 

                                           outbyte = (8*((outpixel>>5)&0x001f)); 
                                                WriteBuffer [BufferIndex++] = outbyte;

                                           //Load up the red component and output it 

                                           outbyte = (8*((outpixel>>10)&0x001f));  //BUG FIX here
                                           WriteBuffer [BufferIndex++] = outbyte;
                                          }
                                 }

                                 //At this point the buffer should be full, so just write it out

                                        WriteFile (file_out, WriteBuffer,BufferIndex,&numwrite,NULL);

                                //Now unlock the surface and we're done

                                        SaveSurface -> Unlock(NULL);

 }
 else  if (ddsd.ddpfPixelFormat.dwRGBBitCount==8) //8 bit surfaces
 {
        //Get the system palette so we can index each pixel to its corresponding color, this
        //is what the frontbuffer parameter is needed for

                        if (FrontBuffer == NULL)
                        {
                                AfxMessageBox("No Front Buffer for 8-bit BMP save");
                                delete [] WriteBuffer;
                                 return FALSE;

                        }

                        LPDIRECTDRAWPALETTE Pal;
                        char bytepal [256*4];

                        rval = FrontBuffer -> GetPalette(&Pal);
                        if (rval != DD_OK)
                        {
								AfxMessageBox("Surface - Couldn't get palette for 8-bit Bitmap Save");
                                delete [] WriteBuffer;
                                 return FALSE;
                        }

                        Pal -> GetEntries (0,0,256,(tagPALETTEENTRY *)&(bytepal[0]));
                        SAFE_RELEASE(Pal);

          //lock the surface and start filling the output
          //buffer

                        ZeroMemory(&ddsd, sizeof(ddsd));
                        ddsd.dwSize = sizeof(ddsd);
                        rval = SaveSurface -> Lock(NULL,&ddsd, DDLOCK_WAIT,NULL);
                        if (rval != DD_OK)
                        {
                                AfxMessageBox("Couldn't lock source");
                                delete [] WriteBuffer;
                                return FALSE;
                        }

                        BYTE *Bitmap_in = (BYTE*)ddsd.lpSurface;

         //Ok, now that we've got the palette and the 24-bit entries, we just look up the color and output it
         //NOTE: At the same time we are reversing the component order (ie. BGR instead of RGB)
         //      and we are outputting it bottom up. 

                         for (loop =Height-1;loop>=0;loop--)    //Loop bottom up
                         for (loop2=0;loop2<Width;loop2++)
                          {
                           outpixel = *(Bitmap_in+loop2 + loop * Pitch); //Load a byte from the surface

                           //Load up the Blue component and output it

                           outbyte = bytepal[outpixel*4+2];//blue
                           WriteBuffer [BufferIndex++] = outbyte;

                           //Load up the Green component and output it

                           outbyte = bytepal[outpixel*4+1];//green
                                WriteBuffer [BufferIndex++] = outbyte;

                           //Load up the Red component and output it

                           outbyte = bytepal[outpixel*4];//red
                                WriteBuffer [BufferIndex++] = outbyte;
                          }

                                 //At this point the buffer should be full, so just write it out

                                        WriteFile (file_out, WriteBuffer,BufferIndex,&numwrite,NULL);

                                //Now unlock the surface and we're done

                                        SaveSurface -> Unlock(NULL);
  }

  delete [] WriteBuffer;

 return TRUE;
}
