/*
Copyright (C) 2000 Chris Teague

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
*/
#include "stdafx.h"
#include "cpu.h"
#include <stdio.h>		// only needed for file operations on log file, debugging

const WORD NMI_VECTOR = 0xFFFA;
const WORD RESET_VECTOR = 0xFFFC;
const WORD IRQ_BRK_VECTOR = 0xFFFE;

//bit numbers for the flags in the Status REgister
#define SR_CARRY_FLAG     0  // carry 
#define SR_ZERO_FLAG      1  // zero
#define SR_INTR_FLAG      2  // interrupts disabled
#define SR_DECIMAL_FLAG   3  // decimal mode
#define SR_BREAK_FLAG     4  // break
#define SR_RSV_FLAG       5  // reserved, always 1
#define SR_OVERFLOW_FLAG  6  // overflow occurred
#define SR_NEG_FLAG       7  // negative

//sets the bit specified by pos in the status register
#define SR_SET_BIT(pos)   (SR) |= (1 << (pos))

//clears the bit specified by pos in the status register
#define SR_CLEAR_BIT(pos)   (SR) &= ~(1 << (pos))

//------------------------------------------------------------
// The following macro is used to set flags 
// in the status register (SR) of the 6502 CPU.
//------------------------------------------------------------
// sets the N and Z flags in SR based on the byte value passed
#define SET_NZ(value) \
	SR = (SR & 0x7d) | ( value ? ((value) & 0x80) : 0x02 )

// sets the N flag in SR based on the byte value passed
#define SR_SET_N(value) \
    SR = (SR & 0x7f) | ((value) & 0x80)

// sets the Z flag in SR based on the byte value passed
#define SR_SET_Z(value) \
    SR = (SR & 0xfd) | (!(value) ? 0x02 : 0x00)



//------------------------------------------------------------
// The following macros are used to compute the the common
// operand addressing modes for the 6502 CPU.
//------------------------------------------------------------

#define ABSOLUTE_OPERAND_ADDR() \
   (my_mem_bus->read_address(PC-2))

// the byte following the opcode is the operand, thus its address
// is relative the the PC  
#define IMMEDIATE_OPERAND_ADDR() \
    (PC-1)

// the word following the opcode is the address of the word 
// which is the address of the operand
#define INDIRECT_OPERAND_ADDR() \
    (my_mem_bus->read_address(my_mem_bus->read_address(PC-2)))

// the word following the opcode is added to register X
// to give the address of the operand
#define ABSOLUTEX_OPERAND_ADDR() \
    ((my_mem_bus->read_address(PC-2)+RegX) & 0xffff)

// the word following the opcode is added to register Y
// to give the address of the operand
#define ABSOLUTEY_OPERAND_ADDR() \
    ((my_mem_bus->read_address(PC-2)+RegY) & 0xffff)

// the byte following the opcode is the address on page 0
// retrieve the byte at that address
#define ZERO_PAGE_OPERAND_ADDR() \
    (my_mem_bus->read_byte(PC-1) & 0x00ff)

// the byte following the opcode is added to register X 
// to give the address on page 0 of the operand
#define ZERO_PAGEX_OPERAND_ADDR() \
    ((my_mem_bus->read_byte(PC-1)+RegX) & 0x00ff)

// the byte following the opcode is added to register Y 
// to give the address on page 0 of the operand
#define ZERO_PAGEY_OPERAND_ADDR() \
    ((my_mem_bus->read_byte(PC-1)+RegY) & 0x00ff)

// the byte following the opcode is added to register Y to give 
// the address on page 0 of the operand
#define INDIRECTY_OPERAND_ADDR() \
    (my_mem_bus->read_address(my_mem_bus->read_byte(PC-1) & 0x00ff) + RegY)

// the byte following the opcode is added to register X to give 
// the address on page 0 of the operand
#define INDIRECTX_OPERAND_ADDR() \
	(my_mem_bus->read_address((my_mem_bus->read_byte(PC-1) +RegX) & 0x00ff))

// the byte following the opcode is added to the PC to give the 
// address of the operand (the byte value is added via 2's complement,
// this means if the number is negative, we sign extend it)
#define RELATIVE_OPERAND_ADDR() \
    (PC + ((my_mem_bus->read_byte(PC-1) & 0x80) ? \
           (my_mem_bus->read_byte(PC-1) | 0xff00) : \
           (my_mem_bus->read_byte(PC-1) & 0x00ff)))

CPU::CPU()
// empty constructor
{
}

CPU::CPU(mem_bus* bus)
// Constructor 
{
	my_mem_bus = bus;		// assign pointer to memory bus	
}

CPU::~CPU()
// Destructor
{	
	my_mem_bus = NULL;		// unassign pointer to memory bus
#ifdef logfile_on
	fclose(fp);				// close the debugging log file
#endif
}

void CPU::set_bus(mem_bus* bus)
{
	my_mem_bus = bus;		// assign pointer to memory bus	
}

void CPU::reset()
// reset the state of the 6502
{
	SP = (WORD)0x01FF;				// reset stack pointer to top of stack
	SR = (BYTE)0x20;				// status register get's cleared except bit 5
	RegAcc = 0;				// clear registers to start with
	RegX = 0;
	RegY = 0;

	// now retrieve the correct starting value of PC from the RESET_VECTOR	
	PC = my_mem_bus->read_address(RESET_VECTOR);

#ifdef logfile_on
	// here for debugging
	start_instruction = 0;
	current_instruction = 0;
	end_instruction = 50000;
	if (!(fp = fopen("log.doc", "w")))
		AfxMessageBox("Unable to open LOG file\n");		
#endif
	
}

BYTE CPU::execute_instruction(long int &cycles)
// Execute the next instruction in the 6502 program
{
	BYTE temp_byte, temp_byte2;
	WORD temp_word;	
	
	static WORD pc1,pc2,pc3;
	pc3 = pc2;
	pc2 = pc1;
	pc1 = PC;
	if (PC == 0xd250)		// here as debug breakpoint
	{			
		int breaker =1;		// this variable does nothing
	}		
	IR = my_mem_bus->read_byte(PC);		// fetch the opcode;

#ifdef logfile_on
	if ((current_instruction >= start_instruction) && (current_instruction <= end_instruction))
	{
		fprintf(fp, "PC=%X | IR=%X | SR=%X -- ", PC, IR, SR);
		if (current_instruction % 3 == 0)
			fprintf(fp, "\n");
	}	
	current_instruction++;	
#endif

	switch(IR)
	{		
		case (BYTE)0x00:                  // BRK implied
			cycles +=7;				          // keep track of number of clock cycles
			TRACE("IRQ interrupt detected\n");
			push(PC);			// push the Program counter on stack
			push(SR);			// and the status register			
			PC = my_mem_bus->read_address(IRQ_BRK_VECTOR);
			SR_SET_BIT(SR_INTR_FLAG);
            SR_SET_BIT(SR_BREAK_FLAG);
			break;

		case (BYTE)0x01:		// ORA indirect x - Logical Or of mem and acc
			cycles +=6;				// keep track of number of clock cycles
			PC += 2;					// 2 byte opcode, so increment PC by 2			
			temp_word = INDIRECTX_OPERAND_ADDR();
			temp_byte = my_mem_bus->read_byte(temp_word);
			RegAcc |= temp_byte;		// perform the ORA operation
			SET_NZ(RegAcc);			// set flags in status register
			break;

		case (BYTE)0x05:		// ORA Z - Logical Or of mem and Acc Zero Page
			cycles += 3;
			PC += 2;
			temp_word = ZERO_PAGE_OPERAND_ADDR();
			temp_byte = my_mem_bus->read_byte(temp_word);	// get the data			
			RegAcc |= temp_byte;
			SET_NZ(RegAcc);			// set appropriate flags in status register
			break;

		case (BYTE)0x06:		// ASL Z - Arithmetic Shift Left zero page
			cycles += 5;
			PC +=2;		// two byte opcode			
			temp_word = ZERO_PAGE_OPERAND_ADDR();
			temp_byte = my_mem_bus->read_byte(temp_word);
			if (temp_byte & 0x80)		// if the bit to be shifted is a one
				SR_SET_BIT(SR_CARRY_FLAG);			// set the carry bit
			else
				SR_CLEAR_BIT(SR_CARRY_FLAG);		// clear the carry bit
			temp_byte <<= 1;	// shift left one
			SET_NZ(temp_byte);		// set N and Z correctly
			my_mem_bus->write_byte(temp_word, temp_byte);		// wite it back
			break;

		case (BYTE)0x08:		// PHP - push processor status (Status Reg) on stack
			cycles += 3;
			PC++;
			push(SR);
			// no flags affected
			break;

		case (BYTE)0x09:		// ORA #n - Logical Or of Mem and Acc
			cycles += 2;	// two cycles to execute
			PC += 2;		// two byets in the instruction
			temp_word = IMMEDIATE_OPERAND_ADDR();
			temp_byte = my_mem_bus->read_byte(temp_word);	// get the byte to OR A with
			RegAcc |= temp_byte;			// logical OR them together
			SET_NZ(RegAcc);				// set appropriate flags in status register
			break;

		case (BYTE)0x0A:		// ASL accumulator
			cycles += 2;
			PC ++;		// only one byte opcode
			if (RegAcc & 0x80)		// if the bit to be shifted is a one
				SR_SET_BIT(SR_CARRY_FLAG);			// set the carry bit
			else
				SR_CLEAR_BIT(SR_CARRY_FLAG);		// clear the carry bit
			RegAcc <<= 1;	// shift left one
			SET_NZ(RegAcc);		// set N and Z correctly
			break;

		case (BYTE)0x0D:		// ORA Q - Logical Or of mem and Acc absolute
			cycles += 4;
			PC += 3;			
			temp_word = ABSOLUTE_OPERAND_ADDR();		// get address
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data				
			RegAcc |= temp_byte;
			SET_NZ(RegAcc);			// set appropriate flags in status register
			break;

		case (BYTE)0x0E:		// ASL Q - Arithmetic Shift Left absolute
			cycles += 6;
			PC +=3;		// two byte opcode			
			temp_word = ABSOLUTE_OPERAND_ADDR();
			temp_byte = my_mem_bus->read_byte(temp_word);
			if (temp_byte & 0x80)		// if the bit to be shifted is a one
				SR_SET_BIT(SR_CARRY_FLAG);			// set the carry bit
			else
				SR_CLEAR_BIT(SR_CARRY_FLAG);		// clear the carry bit
			temp_byte <<= 1;	// shift left one
			SET_NZ(temp_byte);		// set N and Z correctly
			my_mem_bus->write_byte(temp_word, temp_byte);		// wite it back
			break;

		case (BYTE)0x10:		// BPL - Branch if Plus 
			PC += 2;					// only one byte opcode
			if (bit_set(SR_NEG_FLAG) )	// N=1
			{
				cycles += 2;		// only 2 cycles if no branch taken			
			}
			else			// N=0
			{
				cycles += 4;		// 4 cycles if branch taken
				PC = RELATIVE_OPERAND_ADDR();
			}		
			break;

		case (BYTE)0x11:		// ORA indirect Y - Logical Or of mem and acc
			cycles +=5;				// keep track of number of clock cycles
			PC += 2;					// 2 byte opcode, so increment PC by 2			
			temp_word = INDIRECTY_OPERAND_ADDR();
			temp_byte = my_mem_bus->read_byte(temp_word);
			RegAcc |= temp_byte;		// perform the ORA operation
			SET_NZ(RegAcc);			// set flags in status register
			break;

		case (BYTE)0x15:		// ORA Z Page, X - Logical Or of mem and Acc Zero Page x
			cycles += 4;
			PC += 2;
			temp_word = ZERO_PAGEX_OPERAND_ADDR();		// get the address			
			temp_byte = my_mem_bus->read_byte(temp_word);	// get the data
			RegAcc |= temp_byte;
			SET_NZ(RegAcc);			// set appropriate flags in status register
			break;

		case (BYTE)0x16:		// ASL Z,X - arithmetic shift left memory zero page X
			cycles += 6;
			PC += 2;
			temp_word = ZERO_PAGEX_OPERAND_ADDR();	// get the address			
			temp_byte = my_mem_bus->read_byte(temp_word);			
			if (temp_byte & 0x80)		// if the bit to be shifted is a one
				SR_SET_BIT(SR_CARRY_FLAG);			// set the carry bit
			else
				SR_CLEAR_BIT(SR_CARRY_FLAG);		// clear the carry bit
			temp_byte <<= 1;	// shift left one
			SET_NZ(temp_byte);		// set N and Z correctly
			my_mem_bus->write_byte(temp_word, temp_byte);		// wite it back
			break;

		case (BYTE)0x18:		// CLC - clear carry flag
			PC++;			// increment the PC by one
			cycles += 2;	// takes 2 cycles
			SR_CLEAR_BIT(SR_CARRY_FLAG);		// clear the C bit in SR		
			break;

		case (BYTE)0x19:		// ORA Q,Y- OR Acc absolute Y
			cycles += 4;
			PC +=3;
			temp_word = ABSOLUTEY_OPERAND_ADDR();		// get address			
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data				
			RegAcc |= temp_byte;
			SET_NZ(RegAcc);			// set appropriate flags in status register
			break;			

		case (BYTE)0x1D:		// ORA Q,X- OR Acc absolute X
			cycles += 4;
			PC +=3;
			temp_word = ABSOLUTEX_OPERAND_ADDR();		// get address			
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data				
			RegAcc |= temp_byte;
			SET_NZ(RegAcc);			// set appropriate flags in status register
			break;			

		case (BYTE)0x1E:		// ASL Q,X - Arithmetic Shift Left absolute,X
			cycles += 7;
			PC +=3;		// two byte opcode			
			temp_word = ABSOLUTEX_OPERAND_ADDR();
			temp_byte = my_mem_bus->read_byte(temp_word);
			if (temp_byte & 0x80)		// if the bit to be shifted is a one
				SR_SET_BIT(SR_CARRY_FLAG);			// set the carry bit
			else
				SR_CLEAR_BIT(SR_CARRY_FLAG);		// clear the carry bit
			temp_byte <<= 1;	// shift left one
			SET_NZ(temp_byte);		// set N and Z correctly
			my_mem_bus->write_byte(temp_word, temp_byte);		// wite it back
			break;

		case (BYTE)0x20:		// JSR - jump to subroutine
			cycles += 6;
            PC += 3;    
            temp_word = PC - 1;
            push(temp_word);
            temp_word = ABSOLUTE_OPERAND_ADDR();          // read in address
            PC = temp_word;                 // and make the JMP
            // no flags affected
            break;

		case (BYTE)0x21:		// AND (Z,X) - Logical AND of Mem and Acc
			cycles += 6;	// 5 cycles to execute
			PC += 2;		// 
			temp_word = INDIRECTX_OPERAND_ADDR();		// get address			
			temp_byte = my_mem_bus->read_byte(temp_word);	// get the byte to AND A with
			RegAcc &= temp_byte;			// logical AND them together
			SET_NZ(RegAcc);				// set appropriate flags in status register
			break;		

		case (BYTE)0x24:		// BIT Z - bit test with acc
			cycles += 3;
			PC += 2;			
			temp_word = ZERO_PAGE_OPERAND_ADDR();
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data to test			
			if (temp_byte & 0x80)
				SR_SET_BIT(SR_NEG_FLAG);			// set N flag
			else
				SR_CLEAR_BIT(SR_NEG_FLAG);		// clear N flag
			if (temp_byte & 0x40)
				SR_SET_BIT(SR_OVERFLOW_FLAG);			// set V flag
			else
				SR_CLEAR_BIT(SR_OVERFLOW_FLAG);			// clear V flag
			if (temp_byte & RegAcc)		// then they are nonzero
				SR_CLEAR_BIT(SR_ZERO_FLAG);		// clear Z flag
			else				// they are both zero
				SR_SET_BIT(SR_ZERO_FLAG);		// set Z flag
			break;

		case (BYTE)0x25:		// AND Z - Logical AND of mem and Acc Zero Page
			cycles += 3;
			PC += 2;			
			temp_word = ZERO_PAGE_OPERAND_ADDR();
			temp_byte = my_mem_bus->read_byte(temp_word);	// get the data			
			RegAcc &= temp_byte;
			SET_NZ(RegAcc);			// set appropriate flags in status register
			break;

		case (BYTE)0x26:		// ROL Z - rotate left Zero Page
			cycles += 5;
			PC +=2;
			temp_word = ZERO_PAGE_OPERAND_ADDR();
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data			
			if (temp_byte & 0x80)	// save MSB
				temp_byte2 = 0x01;
			else
				temp_byte2 = 0x00;			
			temp_byte <<= 1;		// shift left 1
			temp_byte |= (SR & 0x01);		// put C flag into LSB
			my_mem_bus->write_byte(temp_word, temp_byte);
			if (temp_byte2)		// put MSB into C flag
				SR_SET_BIT(SR_CARRY_FLAG);
			else 
				SR_CLEAR_BIT(SR_CARRY_FLAG);
			SET_NZ(temp_byte);
			break;

		case (BYTE)0x28:		// PLP - pull processor status (SR) off stack
			cycles +=4;
			PC++;
			SR = pop_byte();	// all the flags are set according to popped byte			
			break;

		case (BYTE)0x29:		// AND #n - Logical AND of Mem and Acc
			cycles += 2;	// two cycles to execute
			PC += 2;		// two byets in the instruction
			temp_word = IMMEDIATE_OPERAND_ADDR();
			temp_byte = my_mem_bus->read_byte(temp_word);	// get the byte to AND A with
			RegAcc &= temp_byte;			// logical AND them together
			SET_NZ(RegAcc);				// set appropriate flags in status register
			break;

		case (BYTE)0x2A:		// ROL - rotate left accumulator
			cycles += 2;
			PC++;
			if (RegAcc & 0x80)	// save MSB in temp_byte
				temp_byte = 0x01;
			else
				temp_byte = 0x00;			
			RegAcc <<= 1;		// shift left 1
			RegAcc |= (SR & 0x01);		// put C flag into LSB
			if (temp_byte)			// put MSB into C flag
				SR_SET_BIT(SR_CARRY_FLAG);
			else 
				SR_CLEAR_BIT(SR_CARRY_FLAG);
			SET_NZ(RegAcc);
			break;

		case (BYTE)0x2C:		// BIT Q - bit test with acc
			cycles += 4;
			PC += 3;
			temp_word = ABSOLUTE_OPERAND_ADDR();		// get address
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data			
			SR_SET_N(temp_byte);
            if (temp_byte & 0x40)
                SR_SET_BIT(SR_OVERFLOW_FLAG);              // set V flag
            else
                SR_CLEAR_BIT(SR_OVERFLOW_FLAG);              // clear V flag
            // check for the bit being set by AND'ing with accumulator
            SR_SET_Z(temp_byte & RegAcc);
            break;
		
		case (BYTE)0x2D:		// AND Q - Logical AND of mem and Acc absolute
			cycles += 4;
			PC += 3;			
			temp_word = ABSOLUTE_OPERAND_ADDR();
			temp_byte = my_mem_bus->read_byte(temp_word);	// get the data			
			RegAcc &= temp_byte;
			SET_NZ(RegAcc);			// set appropriate flags in status register
			break;

		case (BYTE)0x2E:		// ROL - rotate left absolute
			cycles += 6;
			PC +=3;
			temp_word = ABSOLUTE_OPERAND_ADDR();		// get address
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data
			if (temp_byte & 0x80)	// save MSB
				temp_byte2 = 0x01;
			else
				temp_byte2 = 0x00;			
			temp_byte <<= 1;		// shift left 1
			temp_byte |= (SR & 0x01);		// put Cflag into LSB
			my_mem_bus->write_byte(temp_word, temp_byte);
			if (temp_byte2)			// put MSB inc C flag
				SR_SET_BIT(SR_CARRY_FLAG);
			else 
				SR_CLEAR_BIT(SR_CARRY_FLAG);
			SET_NZ(temp_byte);
			break;

		case (BYTE)0x30:		// BMI - branch if minus
			PC += 2;					// only one byte opcode
			if (bit_set(SR_NEG_FLAG))	// N=1
			{
				cycles += 4;		// 4 cycles if branch taken
				PC = RELATIVE_OPERAND_ADDR();
                
			}
			else			// N=0
			{
				cycles += 2;		// only 2 cycles if no branch taken			
			}
			// no flags affected
			break;

		case (BYTE)0x31:		// AND (Z),Y - Logical AND of Mem and Acc
			cycles += 5;	// 5 cycles to execute
			PC += 2;		// 
			temp_word = INDIRECTY_OPERAND_ADDR();		// get address			
			temp_byte = my_mem_bus->read_byte(temp_word);	// get the byte to AND A with
			RegAcc &= temp_byte;			// logical AND them together
			SET_NZ(RegAcc);				// set appropriate flags in status register
			break;		

		case (BYTE)0x35:		// AND Z,X - load Y zero page X
			cycles += 4;
			PC += 2;
			temp_word = ZERO_PAGEX_OPERAND_ADDR();	// get address			
			temp_byte = my_mem_bus->read_byte(temp_word);	// read data			
			RegAcc &= temp_byte;			// logical AND them together
			SET_NZ(RegAcc);				// set appropriate flags in stat			
			break;

		case (BYTE)0x36:		// ROL Z,X - rotate left Zero Page X
			cycles += 6;
			PC +=2;
			temp_word = ZERO_PAGEX_OPERAND_ADDR();		// get address			
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data			
			if (temp_byte & 0x80)	// save MSB
				temp_byte2 = 0x01;
			else
				temp_byte2 = 0x00;			
			temp_byte <<= 1;		// shift left 1
			temp_byte |= (SR & 0x01);		// put C flag into LSB
			my_mem_bus->write_byte(temp_word, temp_byte);
			if (temp_byte2)			// put MSB into C flag
				SR_SET_BIT(SR_CARRY_FLAG);
			else 
				SR_CLEAR_BIT(SR_CARRY_FLAG);
			SET_NZ(temp_byte);
			break;

		case (BYTE)0x38:		// SEC - set carry flag
			cycles += 2;	// this instruction takes 2 cycles
			PC++;			// increment PC by one		
			SR_SET_BIT(SR_CARRY_FLAG);			// set the C bit in SR				
			break;

		case (BYTE)0x39:		// AND Q,Y - Logical AND of mem and Acc absolute, Y
			cycles += 4;
			PC += 3;			
			temp_word = ABSOLUTEY_OPERAND_ADDR();		// get address			
			temp_byte = my_mem_bus->read_byte(temp_word);	// get the data			
			RegAcc &= temp_byte;
			SET_NZ(RegAcc);			// set appropriate flags in status register
			break;

		case (BYTE)0x3D:		// AND Q,X - Logical AND of Mem and Acc
			cycles += 4;	// 4 cycles to execute
			PC += 3;		// 
			temp_word = ABSOLUTEX_OPERAND_ADDR();		// get address			
			temp_byte = my_mem_bus->read_byte(temp_word);	// get the byte to AND A with
			RegAcc &= temp_byte;			// logical AND them together
			SET_NZ(RegAcc);				// set appropriate flags in status register
			break;		

		case (BYTE)0x3E:		// ROL Q,X- rotate left absolute X
			cycles += 7;
			PC +=3;
			temp_word = ABSOLUTEX_OPERAND_ADDR();		// get address			
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data
			if (temp_byte & 0x80)	// save MSB
				temp_byte2 = 0x01;
			else
				temp_byte2 = 0x00;			
			temp_byte <<= 1;		// shift left 1
			temp_byte |= (SR & 0x01);		// put C flag into LSB
			my_mem_bus->write_byte(temp_word, temp_byte);
			if (temp_byte2)			// pub MSB into C flag
				SR_SET_BIT(SR_CARRY_FLAG);
			else 
				SR_CLEAR_BIT(SR_CARRY_FLAG);
			SET_NZ(temp_byte);
			break;

		case (BYTE)0x40:		// RTI - return from interrupt
			cycles +=6;
			SP++;			// move stack pointer
			SR = my_mem_bus->read_byte(SP);		// pop the status reg
			SP += 2;		// move stack pointer
			PC = my_mem_bus->read_address(SP-1);	// pop PC
			break;

		case (BYTE)0x41:		// EOR (Z,X) - exclusive OR memory indirect X
			cycles += 6;
			PC += 2;
			temp_word = INDIRECTX_OPERAND_ADDR();		// get address
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data
			RegAcc ^= temp_byte;					// EOR		
			SET_NZ(RegAcc);
			break;

		case (BYTE)0x45:		// EOR Z - exclusive OR memory zero page
			cycles += 3;
			PC += 2;			
			temp_word = ZERO_PAGE_OPERAND_ADDR();
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data			
			RegAcc ^= temp_byte;					// EOR		
			SET_NZ(RegAcc);
			break;

		case (BYTE)0x46:		// LSR Z - logical shift right zero page
			cycles += 5;
			PC += 2;		// 2 byte opcode
			temp_word = ZERO_PAGE_OPERAND_ADDR();
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data				
			if (temp_byte & 0x01)		// if the bit to be shifted is a one
				SR_SET_BIT(SR_CARRY_FLAG);			// set the carry bit
			else
				SR_CLEAR_BIT(SR_CARRY_FLAG);		// clear the carry bit
			temp_byte >>= 1;	// shift right one
			my_mem_bus->write_byte(temp_word, temp_byte);		// write it back
			SET_NZ(temp_byte);		// set N and Z correctly
			break;

		case (BYTE)0x48:		// PHA - push Acc on stack
			cycles += 3;
			PC++;
			push(RegAcc);
			// no flags affected
			break;

		case (BYTE)0x49:		// EOR #n - exclusive OR memory immediate
			cycles += 2;
			PC += 2;
			temp_word = IMMEDIATE_OPERAND_ADDR();
			temp_byte = my_mem_bus->read_byte(temp_word);	// read in data			
			RegAcc ^= temp_byte;					// EOR		
			SET_NZ(RegAcc);
			break;

		case (BYTE)0x4A:		// LSR accumulator
			cycles += 2;
			PC ++;		// only one byte opcode
			if (RegAcc & 0x01)		// if the bit to be shifted is a one
				SR_SET_BIT(SR_CARRY_FLAG);			// set the carry bit
			else
				SR_CLEAR_BIT(SR_CARRY_FLAG);		// clear the carry bit
			RegAcc >>= 1;	// shift right one
			SET_NZ(RegAcc);		// set N and Z correctly
			break;

		case (BYTE)0x4C:		// JMP absolute
			cycles += 3;
			PC +=3;
			PC = ABSOLUTE_OPERAND_ADDR();	// read in address			
			// no flags affected
			break;

		case (BYTE)0x4D:		// EOR Q - exclusive OR memory absolute
			cycles += 4;
			PC += 3;
			temp_word = ABSOLUTE_OPERAND_ADDR();          // read in address
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data
			RegAcc ^= temp_byte;					// EOR		
			SET_NZ(RegAcc);
			break;

		case (BYTE)0x4E:		// LSR Q - logical shift right absolute
			cycles += 6;
			PC += 3;		// three byte opcode
			temp_word = ABSOLUTE_OPERAND_ADDR();	// get address from opcode
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data
			if (temp_byte & 0x01)		// if the bit to be shifted is a one
				SR_SET_BIT(SR_CARRY_FLAG);			// set the carry bit
			else
				SR_CLEAR_BIT(SR_CARRY_FLAG);		// clear the carry bit
			temp_byte >>= 1;	// shift right one
			my_mem_bus->write_byte(temp_word, temp_byte);		// write it back
			SET_NZ(temp_byte);		// set N and Z correctly
			break;

		case (BYTE)0x50:		// BVC - branch if overflow clear
			PC += 2;					// two byte opcode
			if (bit_set(SR_OVERFLOW_FLAG) )	// V=1
			{
				cycles += 2;		// only 2 cycles if no branch taken			
			}
			else			// V=0
			{
				cycles += 4;		// 4 cycles if branch taken
				PC = RELATIVE_OPERAND_ADDR();
			}
			// no flags affected
			break;

		case (BYTE)0x51:		// EOR (Z),Y - exclusive OR memory indirect Y
			cycles += 5;
			PC += 2;
			temp_word = INDIRECTY_OPERAND_ADDR();		// get address
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data
			RegAcc ^= temp_byte;					// EOR		
			SET_NZ(RegAcc);
			break;

		case (BYTE)0x55:		// EOR Z Page, X - exclusive OR memory zero page X
			cycles += 4;
			PC += 2;
			temp_word = ZERO_PAGEX_OPERAND_ADDR();	// get address			
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data
			RegAcc ^= temp_byte;					// EOR		
			SET_NZ(RegAcc);
			break;

		case (BYTE)0x56:		// LSR Z Page, X - logical shift right zero page X
			cycles += 6;
			PC += 2;		// 2 byte opcode
			temp_word = ZERO_PAGEX_OPERAND_ADDR();			
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data			
			if (temp_byte & 0x01)		// if the bit to be shifted is a one
				SR_SET_BIT(SR_CARRY_FLAG);			// set the carry bit
			else
				SR_CLEAR_BIT(SR_CARRY_FLAG);		// clear the carry bit
			temp_byte >>= 1;	// shift right one
			my_mem_bus->write_byte(temp_word, temp_byte);		// write it back
			SET_NZ(temp_byte);		// set N and Z correctly
			break;

		case (BYTE)0x58:		// CLC - clear IRQ disable flag
			PC++;			// increment the PC by one
			cycles += 2;	// takes 2 cycles
			SR_CLEAR_BIT(SR_INTR_FLAG);		// clear the I bit in SR
			break;

		case (BYTE)0x59:		// EOR Q, Y - exclusive OR memory absolute Y
			cycles += 4;
			PC += 3;
			temp_word = ABSOLUTEY_OPERAND_ADDR();	// get address
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data
			RegAcc ^= temp_byte;					// EOR		
			SET_NZ(RegAcc);
			break;

		case (BYTE)0x5D:		// EOR Q, X - exclusive OR memory absolute X
			cycles += 4;
			PC += 3;
			temp_word = ABSOLUTEX_OPERAND_ADDR();	// get address
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data
			RegAcc ^= temp_byte;					// EOR		
			SET_NZ(RegAcc);
			break;

		case (BYTE)0x5E:		// LSR Q,X - logical shift right absolute X
			cycles += 7;
			PC += 3;		// 2 byte opcode
			temp_word = ABSOLUTEX_OPERAND_ADDR();			
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data			
			if (temp_byte & 0x01)		// if the bit to be shifted is a one
				SR_SET_BIT(SR_CARRY_FLAG);			// set the carry bit
			else
				SR_CLEAR_BIT(SR_CARRY_FLAG);		// clear the carry bit
			temp_byte >>= 1;	// shift right one
			my_mem_bus->write_byte(temp_word, temp_byte);		// write it back
			SET_NZ(temp_byte);		// set N and Z correctly
			break;

		case (BYTE)0x60:		// RTS return to subroutine
			cycles += 6;
			PC = my_mem_bus->read_address(SP+1)+1;	// get address off stack, add one when destack
			SP += 2;				// move stack pointer 
			// no flags affected
			break;

		case (BYTE)0x61:                  // ADC (Z,X) - add acc w/ carry
			cycles +=6;
			PC +=2;
			temp_word = INDIRECTX_OPERAND_ADDR();	// get address			
			temp_byte = my_mem_bus->read_byte(temp_word);	// read data			
			temp_byte2 = RegAcc;		// save original acc
			temp_word = temp_byte + RegAcc + 1*bit_set(SR_CARRY_FLAG);	// do the add
			RegAcc = temp_word & 0x00FF;
			SET_NZ(RegAcc);		// set N and Z flags
			if (temp_word & 0x100)	// Set Carry flag
				SR_SET_BIT(SR_CARRY_FLAG);
			else
				SR_CLEAR_BIT(SR_CARRY_FLAG);
			// now set overflow flag
			// if A and B have the *SAME* sign then overflow *IS* possible
			//    and if A and A' have different signs then overflow occured
			if (   !((temp_byte2 ^ temp_byte) & 0x80) 
				&& ((temp_byte2 ^ RegAcc) & 0x80))
				SR_SET_BIT(SR_OVERFLOW_FLAG);
			else
				SR_CLEAR_BIT(SR_OVERFLOW_FLAG); 
			break;

		case (BYTE)0x65:		// ADC Z - add acc with carry Zero Page
			cycles +=3;
			PC +=2;
			temp_word = ZERO_PAGE_OPERAND_ADDR();
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data	
			temp_byte2 = RegAcc;		// save original acc			
			temp_word = temp_byte + RegAcc + 1*bit_set(SR_CARRY_FLAG);	// do the add
			RegAcc = temp_word & 0x00FF;
			SET_NZ(RegAcc);		// set N and Z flags
			if (temp_word & 0x100)	// Set Carry flag
				SR_SET_BIT(SR_CARRY_FLAG);
			else
				SR_CLEAR_BIT(SR_CARRY_FLAG);
			// if A and B have the *SAME* sign then overflow *IS* possible
			//    and if A and A' have different signs then overflow occured
			if (   !((temp_byte2 ^ temp_byte) & 0x80) 
				&& ((temp_byte2 ^ RegAcc) & 0x80))
				SR_SET_BIT(SR_OVERFLOW_FLAG);
			else
				SR_CLEAR_BIT(SR_OVERFLOW_FLAG); 
			break;

		case (BYTE)0x66:		// ROR Z- rotate right zero page			
			cycles += 5;
			PC +=2;
			temp_word = ZERO_PAGE_OPERAND_ADDR();
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data				
			if (temp_byte & 0x01)	// save LSB in temp_byte
				temp_byte2 = 0x80;
			else
				temp_byte2 = 0x00;						
			temp_byte >>= 1;		// shift right 1
			temp_byte |= ((SR & 0x01) <<7);		// put C flag into MSB
			my_mem_bus->write_byte(temp_word, temp_byte);
			if (temp_byte2)			// put LSB into C flag
				SR_SET_BIT(SR_CARRY_FLAG);
			else 
				SR_CLEAR_BIT(SR_CARRY_FLAG);
			SET_NZ(temp_byte);
			break;			

		case (BYTE)0x68:		// PLA - pull Acc off stack
			cycles +=4;
			PC++;
			RegAcc = pop_byte();
			SET_NZ(RegAcc);		// set SR flags
			break;

		case (BYTE)0x69:		// ADC #n - add acc with carry
			cycles +=2;
			PC +=2;
			temp_word = IMMEDIATE_OPERAND_ADDR();
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data
			temp_byte2 = RegAcc;		// save original acc			
			temp_word = temp_byte + RegAcc + 1*bit_set(SR_CARRY_FLAG);	// do the add
			RegAcc = temp_word & 0x00FF;
			SET_NZ(RegAcc);		// set N and Z flags
			if (temp_word & 0x100)	// Set Carry flag
				SR_SET_BIT(SR_CARRY_FLAG);
			else
				SR_CLEAR_BIT(SR_CARRY_FLAG);
			// now set overflow flag
			// if A and B have the *SAME* sign then overflow *IS* possible
			//    and if A and A' have different signs then overflow occured
			if (   !((temp_byte2 ^ temp_byte) & 0x80) 
				&& ((temp_byte2 ^ RegAcc) & 0x80))
				SR_SET_BIT(SR_OVERFLOW_FLAG);
			else
				SR_CLEAR_BIT(SR_OVERFLOW_FLAG); 
			break;

		case (BYTE)0x6A:		// ROR - rotate right accumulator
			cycles += 2;
			PC++;
			if (RegAcc & 0x01)	// save LSB in temp_byte
				temp_byte = 0x80;
			else
				temp_byte = 0x00;			
			RegAcc >>= 1;		// shift right 1
			RegAcc |= ((SR & 0x01) <<7);		// put C flag into MSB
			if (temp_byte)			// put LSB into C flag
				SR_SET_BIT(SR_CARRY_FLAG);
			else 
				SR_CLEAR_BIT(SR_CARRY_FLAG);
			SET_NZ(RegAcc);
			break;

		case (BYTE)0x6C:		// JMP indirect
			cycles += 5;		// 5 cylces to execute
			PC += 3;                        // increment PC
			PC = INDIRECT_OPERAND_ADDR();
			// no flags affected
			break;

		case (BYTE)0x6D:		// ADC Q - add acc with carry absolute
			cycles +=4;
			PC +=3;
			temp_word = ABSOLUTE_OPERAND_ADDR();		// get address
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data
			temp_byte2 = RegAcc;		// save original acc			
			temp_word = temp_byte + RegAcc + 1*bit_set(SR_CARRY_FLAG);	// do the add
			RegAcc = temp_word & 0x00FF;
			SET_NZ(RegAcc);		// set N and Z flags
			if (temp_word & 0x100)	// Set Carry flag
				SR_SET_BIT(SR_CARRY_FLAG);
			else
				SR_CLEAR_BIT(SR_CARRY_FLAG);
			// now set overflow flag
			// if A and B have the *SAME* sign then overflow *IS* possible
			//    and if A and A' have different signs then overflow occured
			if (   !((temp_byte2 ^ temp_byte) & 0x80) 
				&& ((temp_byte2 ^ RegAcc) & 0x80))
				SR_SET_BIT(SR_OVERFLOW_FLAG);
			else
				SR_CLEAR_BIT(SR_OVERFLOW_FLAG); 
			break;

		case (BYTE)0x6E:		// ROR Q- rotate right absolute			
			cycles += 6;
			PC+=3;
			temp_word = ABSOLUTE_OPERAND_ADDR();		// get address
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data
			if (temp_byte & 0x01)	// save LSB in temp_byte
				temp_byte2 = 0x80;
			else
				temp_byte2 = 0x00;						
			temp_byte >>= 1;		// shift right 1
			temp_byte |= ((SR & 0x01) <<7);		// put C flag into MSB
			my_mem_bus->write_byte(temp_word, temp_byte);
			if (temp_byte2)			// pub LSB into C flag
				SR_SET_BIT(SR_CARRY_FLAG);
			else 
				SR_CLEAR_BIT(SR_CARRY_FLAG);
			SET_NZ(temp_byte);
			break;

		case (BYTE)0x70:                    // BVS - branch if overflow set
            PC += 2;                        // two byte opcode
            if (bit_set(SR_OVERFLOW_FLAG))  // V=1
            {
                cycles += 4;                // 4 cycles if branch taken
                PC = RELATIVE_OPERAND_ADDR();
            }
            else                            // V=0
            {
                cycles += 2;                // only 2 cycles if no branch taken         
            }
            // no flags affected
            break;

		case (BYTE)0x71:                  // ADC (Z),Y - add acc w/ carry
			cycles +=5;
			PC +=2;
			temp_word = INDIRECTY_OPERAND_ADDR();	// get address			
			temp_byte = my_mem_bus->read_byte(temp_word);	// read data			
			temp_byte2 = RegAcc;		// save original acc
			temp_word = temp_byte + RegAcc + 1*bit_set(SR_CARRY_FLAG);	// do the add
			RegAcc = temp_word & 0x00FF;
			SET_NZ(RegAcc);		// set N and Z flags
			if (temp_word & 0x100)	// Set Carry flag
				SR_SET_BIT(SR_CARRY_FLAG);
			else
				SR_CLEAR_BIT(SR_CARRY_FLAG);
			// now set overflow flag
			// if A and B have the *SAME* sign then overflow *IS* possible
			//    and if A and A' have different signs then overflow occured
			if (   !((temp_byte2 ^ temp_byte) & 0x80) 
				&& ((temp_byte2 ^ RegAcc) & 0x80))
				SR_SET_BIT(SR_OVERFLOW_FLAG);
			else
				SR_CLEAR_BIT(SR_OVERFLOW_FLAG); 
			break;

		case (BYTE)0x75:                  // ADC Z PAGE, X - add acc w/ carry
			cycles +=4;
			PC +=2;
			temp_word = ZERO_PAGEX_OPERAND_ADDR();	// get address			
			temp_byte = my_mem_bus->read_byte(temp_word);	// read data			
			temp_byte2 = RegAcc;		// save original acc
			temp_word = temp_byte + RegAcc + 1*bit_set(SR_CARRY_FLAG);	// do the add
			RegAcc = temp_word & 0x00FF;
			SET_NZ(RegAcc);		// set N and Z flags
			if (temp_word & 0x100)	// Set Carry flag
				SR_SET_BIT(SR_CARRY_FLAG);
			else
				SR_CLEAR_BIT(SR_CARRY_FLAG);
			// now set overflow flag
			// if A and B have the *SAME* sign then overflow *IS* possible
			//    and if A and A' have different signs then overflow occured
			if (   !((temp_byte2 ^ temp_byte) & 0x80) 
				&& ((temp_byte2 ^ RegAcc) & 0x80))
				SR_SET_BIT(SR_OVERFLOW_FLAG);
			else
				SR_CLEAR_BIT(SR_OVERFLOW_FLAG); 
			break;
			
		case (BYTE)0x76:		// ROR Z,X - rotate right Zero Page X
			cycles += 6;
			PC +=2;
			temp_word = ZERO_PAGEX_OPERAND_ADDR();		// get address			
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data			
			if (temp_byte & 0x01)	// save LSB
				temp_byte2 = 0x80;
			else
				temp_byte2 = 0x00;			
			temp_byte >>= 1;		// shift right 1
			temp_byte |= ((SR & 0x01) <<7);		// put C flag into MSB
			my_mem_bus->write_byte(temp_word, temp_byte);
			if (temp_byte2)
				SR_SET_BIT(SR_CARRY_FLAG);
			else 
				SR_CLEAR_BIT(SR_CARRY_FLAG);
			SET_NZ(temp_byte);
			break;

		case (BYTE)0x78:		// SEI - disable interrupts
			PC++;				// increment PC by one
			cycles += 2;		// this instruction takes 2 cycles
			SR_SET_BIT(SR_INTR_FLAG);		// set the I bit in SR			
			break;

		case (BYTE)0x79:		// ADC Q,Y - add acc with carry absolute Y
			cycles +=4;
			PC +=3;
			temp_word = ABSOLUTEY_OPERAND_ADDR();		// get address			
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data
			temp_byte2 = RegAcc;		// save original acc			
			temp_word = temp_byte + RegAcc + 1*bit_set(SR_CARRY_FLAG);	// do the add
			RegAcc = temp_word & 0x00FF;
			SET_NZ(RegAcc);		// set N and Z flags
			if (temp_word & 0x100)	// Set Carry flag
				SR_SET_BIT(SR_CARRY_FLAG);
			else
				SR_CLEAR_BIT(SR_CARRY_FLAG);
			// now set overflow flag
			// if A and B have the *SAME* sign then overflow *IS* possible
			//    and if A and A' have different signs then overflow occured
			if (   !((temp_byte2 ^ temp_byte) & 0x80) 
				&& ((temp_byte2 ^ RegAcc) & 0x80))
				SR_SET_BIT(SR_OVERFLOW_FLAG);
			else
				SR_CLEAR_BIT(SR_OVERFLOW_FLAG); 
			break;

		case (BYTE)0x7D:		// ADC Q,X - add acc with carry absolute X
			cycles +=4;
			PC +=3;
			temp_word = ABSOLUTEX_OPERAND_ADDR();		// get address			
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data
			temp_byte2 = RegAcc;		// save original acc			
			temp_word = temp_byte + RegAcc + 1*bit_set(SR_CARRY_FLAG);	// do the add
			RegAcc = temp_word & 0x00FF;
			SET_NZ(RegAcc);		// set N and Z flags
			if (temp_word & 0x100)	// Set Carry flag
				SR_SET_BIT(SR_CARRY_FLAG);
			else
				SR_CLEAR_BIT(SR_CARRY_FLAG);
			// now set overflow flag
			// if A and B have the *SAME* sign then overflow *IS* possible
			//    and if A and A' have different signs then overflow occured
			if (   !((temp_byte2 ^ temp_byte) & 0x80) 
				&& ((temp_byte2 ^ RegAcc) & 0x80))
				SR_SET_BIT(SR_OVERFLOW_FLAG);
			else
				SR_CLEAR_BIT(SR_OVERFLOW_FLAG); 
			break;

		case (BYTE)0x7E:		// ROR Q,X- rotate right absolute X
			cycles += 7;
			PC +=3;
			temp_word = ABSOLUTEX_OPERAND_ADDR();		// get address			
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data
			if (temp_byte & 0x01)	// save LSB in temp_byte
				temp_byte2 = 0x80;
			else
				temp_byte2 = 0x00;						
			temp_byte >>= 1;		// shift right 1			
			temp_byte |= ((SR & 0x01) <<7);		// put C flag into MSB
			my_mem_bus->write_byte(temp_word, temp_byte);
			if (temp_byte2)
				SR_SET_BIT(SR_CARRY_FLAG);
			else 
				SR_CLEAR_BIT(SR_CARRY_FLAG);
			SET_NZ(temp_byte);
			break;

		case (BYTE)0x81:		// STA (Z, X) - store acc indirect ,X
			cycles += 6;
			PC += 2;
			temp_word = INDIRECTX_OPERAND_ADDR();		// get address
			my_mem_bus->write_byte(temp_word, RegAcc);
			// no flags affected
			break;

		case (BYTE)0x84:		// STY - store Y in memory Zero Page
			cycles += 3;
			PC += 2;
			temp_word = ZERO_PAGE_OPERAND_ADDR();		// get address		
			my_mem_bus->write_byte(temp_word, RegY);
			// no flags altered
			break;

		case (BYTE)0x85:		// STA Z - Store A in Memory zero page
			cycles += 3;			// this instruction takes 4 NES clock cycles
			PC += 2;								// 3 byte instruction
			temp_word = ZERO_PAGE_OPERAND_ADDR();	// read in address		
			my_mem_bus->write_byte(temp_word, RegAcc);	// write to memory		
			// no flags affected
			break;

		case (BYTE)0x86:		// STX Z - Store X in Memory zero page
			cycles += 3;			// this instruction takes 3 NES clock cycles
			PC += 2;								// 3 byte instruction
			temp_word = ZERO_PAGE_OPERAND_ADDR();	// read in address		
			my_mem_bus->write_byte(temp_word, RegX);	// write to memory		
			// no flags affected
			break;

		case (BYTE)0x88:		// DEY - decrement Y
			cycles += 2;
			PC++;
			RegY--;			// decrement the register
			SET_NZ(RegY);	// set the flags
			break;

		case (BYTE)0x8A:		// TXA - transfer X to A
			cycles += 2;
			PC++;
			RegAcc = RegX;
			SET_NZ(RegAcc);
			break;

		case (BYTE)0x8C:		// STY Q - Store Y in Memory absolute
			cycles += 4;			// this instruction takes 4 NES clock cycles
			PC += 3;								// 3 byte instruction
			temp_word = ABSOLUTE_OPERAND_ADDR();	// read in address		
			my_mem_bus->write_byte(temp_word, RegY);	// write to memory		
			// no flags affected
			break;

		case (BYTE)0x8D:		// STA absolute- store accumulator
			cycles += 4;		// four cycles to execute
			PC += 3;			// three byte opcode
			temp_word = ABSOLUTE_OPERAND_ADDR();	// read in address			
			my_mem_bus->write_byte(temp_word, RegAcc);	// write to memory
			// no flags affected
			break;

		case (BYTE)0x8E:		// STX - Store X in Memory absolute
			cycles += 4;			// this instruction takes 4 NES clock cycles
			PC += 3;								// 3 byte instruction
			temp_word = ABSOLUTE_OPERAND_ADDR();	// read in address		
			my_mem_bus->write_byte(temp_word, RegX);	// write to memory		
			// no flags affected
			break;

		case (BYTE)0x90:		// BCC - branch if carry clear
			PC += 2;					// only one byte opcode
			if (bit_set(SR_CARRY_FLAG) )	// C=1
			{
				cycles += 2;		// only 2 cycles if no branch taken			
			}
			else			// C=0
			{
				cycles += 4;		// 4 cycles if branch taken
				PC = RELATIVE_OPERAND_ADDR();
			}
			// no flags affected
			break;

		case (BYTE)0x91:		// STA (Z),Y - store acc indirect ,Y
			cycles += 5;
			PC += 2;
			temp_word = INDIRECTY_OPERAND_ADDR();			
			my_mem_bus->write_byte(temp_word, RegAcc);
			// no flags affected
			break;

		case (BYTE)0x94:		// STY Z,X - store Y zero page X
			cycles += 4;
			PC += 2;
			temp_word = ZERO_PAGEX_OPERAND_ADDR();	// get address			
			my_mem_bus->write_byte(temp_word ,RegY);
			// no flags affected
			break;

		case (BYTE)0x95:		// STA Z,X - store Acc zero page X
			cycles += 4;
			PC += 2;
			temp_word = ZERO_PAGEX_OPERAND_ADDR();	// get address			
			my_mem_bus->write_byte(temp_word ,RegAcc);
			// no flags affected
			break;		

		case (BYTE)0x96:		// STX Z,Y - store Acc zero page Y
			cycles += 4;
			PC += 2;
			temp_word = ZERO_PAGEY_OPERAND_ADDR();	// get address			
			my_mem_bus->write_byte(temp_word ,RegX);
			// no flags affected
			break;

		case (BYTE)0x98:		// TYA - transfer Y to Acc
			cycles += 2;
			PC++;
			RegAcc = RegY;		// make the transfer
			SET_NZ(RegAcc);		// set the flags
			break;

		case (BYTE)0x99:		// STA Q,Y - store Acc absolute Y
			cycles += 4;
			PC += 3;
			temp_word = ABSOLUTEY_OPERAND_ADDR();	// read in address			
			my_mem_bus->write_byte(temp_word ,RegAcc);
			// no flags affected
			break;


		case (BYTE)0x9A:		// TXS - transfer X to S
			cycles += 2;
			PC++;
			SP = RegX;
			SP &= 0x01ff;
			SP |= 0x0100;
			// no flags affected
			break;

		case (BYTE)0x9D:		// STA Q,X - store Acc absolute X
			cycles += 4;
			PC += 3;
			temp_word = ABSOLUTEX_OPERAND_ADDR();	// read in address			
			my_mem_bus->write_byte(temp_word ,RegAcc);
			// no flags affected
			break;

		case (BYTE)0xA0:		// LDY - load Y immediate
			cycles += 2;
			PC +=2;
			temp_word = IMMEDIATE_OPERAND_ADDR();
			RegY = my_mem_bus->read_byte(temp_word);
			SET_NZ(RegY);
			break;

		case (BYTE)0xA1:		// LDA (Z,X) - load Acc indirect X
			cycles += 6;
			PC += 2;			
			temp_word = INDIRECTX_OPERAND_ADDR();
			RegAcc = my_mem_bus->read_byte(temp_word);
			SET_NZ(RegAcc);
			break;

		case (BYTE)0xA2:		// LDX #n - Load X immediate
			PC += 2;		// increment PC by two because this instruction is 2 bytes long;
			cycles += 4;
			temp_word = IMMEDIATE_OPERAND_ADDR();
			RegX = my_mem_bus->read_byte(temp_word);		// fetch the immediate value
			SET_NZ(RegX);		// set the status register flags			
			break;

		case (BYTE)0xA4:		// LDY Z - load register Y zero page
			cycles += 3;
			PC += 2;
			temp_word = ZERO_PAGE_OPERAND_ADDR();		// get the address		
			RegY = my_mem_bus->read_byte(temp_word);
			SET_NZ(RegY);
			break;

		case (BYTE)0xA5:		// LDA - load accumulator zero page
			cycles += 3;
			PC += 2;
			temp_word = ZERO_PAGE_OPERAND_ADDR();		// get the address		
			RegAcc = my_mem_bus->read_byte(temp_word);
			SET_NZ(RegAcc);
			break;

		case (BYTE)0xA6:		// LDX - load register X zero page
			cycles += 3;
			PC += 2;			
			temp_word = ZERO_PAGE_OPERAND_ADDR();		// get the address		
			RegX = my_mem_bus->read_byte(temp_word);
			SET_NZ(RegX);
			break;

		case (BYTE)0xA8:		// TAY - tranfer A to Y
			cycles += 2;
			PC++;
			RegY = RegAcc;
			SET_NZ(RegY);
			break;

		case (BYTE)0xA9:		// LDA immediate
			cycles += 2;
			PC += 2;
			temp_word = IMMEDIATE_OPERAND_ADDR();
			RegAcc = my_mem_bus->read_byte(temp_word);	// read in value
			SET_NZ(RegAcc);				// set appropriate flags in status register
			break;

		case (BYTE)0xAA:		// TAX - transfer A to X
			cycles += 2;
			PC++;
			RegX = RegAcc;	// make the tranfer
			SET_NZ(RegX);
			break;
			
		case (BYTE)0xAC:		// LDY Q - load register Y absolute
			cycles += 4;
			PC += 3;
			temp_word = ABSOLUTE_OPERAND_ADDR();	// read in address
			RegY = my_mem_bus->read_byte(temp_word);
			SET_NZ(RegY);
			break;

		case (BYTE)0xAE:		// Load X absolute
			cycles += 4;			// this instruction takes 4 NES clock cycles
			PC += 3;				// this instruction is 3 bytes long
			temp_word = ABSOLUTE_OPERAND_ADDR();	// read in address			
			RegX = my_mem_bus->read_byte(temp_word);		
			SET_NZ(RegX);			// set the status register flags
			break;

		case (BYTE)0xAD:		// Load Accumulator absolute
			cycles += 4;			// this instruction takes 4 NES clock cycles
			PC += 3;				// this instruction is 3 bytes long
			temp_word = ABSOLUTE_OPERAND_ADDR();	// read in address			
			RegAcc = my_mem_bus->read_byte(temp_word);		
			SET_NZ(RegAcc);			// set the status register flags			
			break;

		case (BYTE)0xB0:		// BCS - branch if carry set
			PC += 2;					// two byte opcode
			if (bit_set(SR_CARRY_FLAG) )	// C=1
			{
				cycles += 4;		// 4 cycles if branch taken
				PC = RELATIVE_OPERAND_ADDR();
			}
			else			// Z=0
			{
				cycles += 2;		// only 2 cycles if no branch taken						
			}
			// no flags affected
			break;

		case (BYTE)0xB1:		// LDA (Z),Y - load acc indirect Y
			cycles += 5;
			PC += 2;
			temp_word = INDIRECTY_OPERAND_ADDR();			
			RegAcc = my_mem_bus->read_byte(temp_word);
			SET_NZ(RegAcc);
			break;

		case (BYTE)0xB4:		// LDY Z,X - load Y zero page X
			cycles += 4;
			PC += 2;
			temp_word = ZERO_PAGEX_OPERAND_ADDR();	// get address			
			temp_byte = my_mem_bus->read_byte(temp_word);	// read data
			RegY = temp_byte;
			SET_NZ(RegY);
			break;

		case (BYTE)0xB5:		// LDA Z,X - load Acc zero page X
			cycles += 4;
			PC += 2;
			temp_word = ZERO_PAGEX_OPERAND_ADDR();	// get address			
			temp_byte = my_mem_bus->read_byte(temp_word);	// read data
			RegAcc = temp_byte;
			SET_NZ(RegAcc);
			break;

		case (BYTE)0xB6:		// LDX Z,Y- load X zero page Y
			cycles += 4;
			PC +=2;
			temp_word = ZERO_PAGEY_OPERAND_ADDR();	// get address			
			RegX = my_mem_bus->read_byte(temp_word);	// get data			
			SET_NZ(RegX);
			break;

		case (BYTE)0xB8:		// CLV - clear overflow flag
			PC++;			// increment the PC by one
			cycles += 2;	// takes 2 cycles
			SR_CLEAR_BIT(SR_OVERFLOW_FLAG);		// clear the D bit in SR		
			break;

		case (BYTE)0xB9:		// LDA Q,Y- load A absolute Y
			cycles += 4;
			PC += 3;
			temp_word = ABSOLUTEY_OPERAND_ADDR();			
			RegAcc = my_mem_bus->read_byte(temp_word);
			SET_NZ(RegAcc);
			break;

		case (BYTE)0xBA:		// TSX - transfer S to X
			cycles += 2;
			PC++;
			RegX = (BYTE)(SP&0x00ff);
			SET_NZ(RegX);
			break;

		case (BYTE)0xBC:		// LDY Q,X- load Y absolute X
			cycles += 4;
			PC +=3;
			temp_word = ABSOLUTEX_OPERAND_ADDR();		// get address			
			RegY = my_mem_bus->read_byte(temp_word);	// get data			
			SET_NZ(RegY);
			break;

		case (BYTE)0xBD:		// LDA Q,X- load Acc absolute X
			cycles += 4;
			PC += 3;
			temp_word = ABSOLUTEX_OPERAND_ADDR();			
			RegAcc = my_mem_bus->read_byte(temp_word);
			SET_NZ(RegAcc);
			break;

		case (BYTE)0xBE:		// LDX Q,Y- load X absolute Y
			cycles += 4;
			PC +=3;
			temp_word = ABSOLUTEY_OPERAND_ADDR();		// get address			
			RegX = my_mem_bus->read_byte(temp_word);	// get data			
			SET_NZ(RegX);
			break;

		case (BYTE)0xC0:		// CPY #n - compare mem and Y immediate
			cycles += 2;
			PC += 2;
			temp_word = IMMEDIATE_OPERAND_ADDR();
			temp_byte = my_mem_bus->read_byte(temp_word);	// read in data
			SetCMPFlags(temp_byte, RegY);		// set the flags accordingly
			break;

		case (BYTE)0xC1:		// CMP (Z,X) - compare mem and Acc indirect X
			cycles += 6;
			PC += 2;
			temp_word = INDIRECTX_OPERAND_ADDR();		// get address
			temp_byte = my_mem_bus->read_byte(temp_word);	// read data			
			SetCMPFlags(temp_byte, RegAcc);		// set the flags accordingly
			break;		

		case (BYTE)0xC4:		// CPY Z - compare mem and Y zero page
			cycles += 3;
			PC += 2;
			temp_word = ZERO_PAGE_OPERAND_ADDR();
			temp_byte = my_mem_bus->read_byte(temp_word);	// read in data
			SetCMPFlags(temp_byte, RegY);		// set the flags accordingly
			break;

		case (BYTE)0xC5:		// CMP Z - compare mem and Acc zero page
			cycles += 3;
			PC += 2;
			temp_word = ZERO_PAGE_OPERAND_ADDR();		// read in address
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data
			SetCMPFlags(temp_byte, RegAcc);		// set the flags accordingly
			break;

		case (BYTE)0xC6:		// DEC - decrement memory by one Zero Page
			cycles += 5;
			PC += 2;
			temp_word = ZERO_PAGE_OPERAND_ADDR();		// read in address
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data
			temp_byte--;		// decrement it
			my_mem_bus->write_byte(temp_word, temp_byte);	// put it back
			SET_NZ(temp_byte);		// set flags
			break;

		case (BYTE)0xC8:		// INY - increment Y
			PC++;			// increment the PC by one
			cycles += 2;	// takes 2 cycles
			RegY++;		// increment
			SET_NZ(RegY);
			break;

		case (BYTE)0xC9:		// CMP #n - compare mem and Acc Absolute
			cycles += 2;
			PC += 2;
			temp_word = IMMEDIATE_OPERAND_ADDR();
			temp_byte = my_mem_bus->read_byte(temp_word);	// read in data
			SetCMPFlags(temp_byte, RegAcc);		// set the flags accordingly
			break;

		case (BYTE)0xCA:		// DEX - decrement X
			cycles += 2;
			PC ++;			// one byte;
			RegX--;			// decrement x
			SET_NZ(RegX);	// set the flags
			break;

		case (BYTE)0xCC:		// CPY Q - compare mem and Y absolute
			cycles += 4;
			PC += 3;
			temp_word = ABSOLUTE_OPERAND_ADDR();
			temp_byte = my_mem_bus->read_byte(temp_word);	// read in data
			SetCMPFlags(temp_byte, RegY);		// set the flags accordingly
			break;

		case (BYTE)0xCD:		// CMP Q - compare mem and Acc Absolute
			cycles += 4;
			PC += 3;
			temp_word = ABSOLUTE_OPERAND_ADDR();	// get address from opcode
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data
			SetCMPFlags(temp_byte, RegAcc);		// set the flags accordingly
			break;

		case (BYTE)0xCE:		// DEC Q - Decrement memory absolute
			cycles += 6;
			PC += 3;
			temp_word = ABSOLUTE_OPERAND_ADDR();	// get address from opcode
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data
			temp_byte--;					// Decrement
			my_mem_bus->write_byte(temp_word, temp_byte);	// store it back
			SET_NZ(temp_byte);
			break;

		case (BYTE)0xD0:		// BNE - branch if not equal		
			PC += 2;					// only one byte opcode
			if (bit_set(SR_ZERO_FLAG))	// Z=1
			{
				cycles += 2;		// only 2 cycles if no branch taken			
			}
			else			// Z=0
			{
				cycles += 4;		// 4 cycles if branch taken
				PC = RELATIVE_OPERAND_ADDR();
			}
			// no flags affected
			break;

		case (BYTE)0xD1:		// CMP (Z), Y - compare mem and Acc indirect Y
			cycles += 5;
			PC += 2;
			temp_word = INDIRECTY_OPERAND_ADDR();	// get address			
			temp_byte = my_mem_bus->read_byte(temp_word);	// read data			
			SetCMPFlags(temp_byte, RegAcc);		// set the flags accordingly
			break;

		case (BYTE)0xD5:		// CMP Z PAGE, X - compare mem and Acc Absolute
			cycles += 4;
			PC += 2;
			temp_word = ZERO_PAGEX_OPERAND_ADDR();	// get address			
			temp_byte = my_mem_bus->read_byte(temp_word);	// read data			
			SetCMPFlags(temp_byte, RegAcc);		// set the flags accordingly
			break;

		case (BYTE)0xD6:		// DEC Z,X - decrement memory zero page X
			cycles += 6;
			PC += 2;
			temp_word = ZERO_PAGEX_OPERAND_ADDR();			
			temp_byte = my_mem_bus->read_byte(temp_word);			
			temp_byte--;			// do the decrement
			my_mem_bus->write_byte(temp_word, temp_byte);	// write it back
			SET_NZ(temp_byte);
			break;

		case (BYTE)0xD8:		// CLD - clear decimal mode
			PC++;			// increment the PC by one
			cycles += 2;	// takes 2 cycles
			SR_CLEAR_BIT(SR_DECIMAL_FLAG);		// clear the D bit in SR		
			break;

		case (BYTE)0xD9:		// CMP Q,Y - compare mem and Acc Absolute Y
			cycles += 4;
			PC += 3;
			temp_word = ABSOLUTEY_OPERAND_ADDR();	// get address			
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data
			SetCMPFlags(temp_byte, RegAcc);		// set the flags accordingly
			break;

		case (BYTE)0xDD:		// CMP Q,X - compare mem and Acc Absolute X
			cycles += 4;
			PC += 3;
			temp_word = ABSOLUTEX_OPERAND_ADDR();	// get address			
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data
			SetCMPFlags(temp_byte, RegAcc);		// set the flags accordingly
			break;

		case (BYTE)0xDE:		// DEC Q,X - decrement memory absolute X
			cycles += 7;
			PC += 3;
			temp_word = ABSOLUTEX_OPERAND_ADDR();	// get address			
			temp_byte = my_mem_bus->read_byte(temp_word);			
			temp_byte--;			// do the decrement
			my_mem_bus->write_byte(temp_word, temp_byte);	// write it back
			SET_NZ(temp_byte);
			break;

		case (BYTE)0xE0:		// CPX #n - compare mem and X immediate
			cycles += 2;
			PC += 2;
			temp_word = IMMEDIATE_OPERAND_ADDR();
			temp_byte = my_mem_bus->read_byte(temp_word);	// read in data
			SetCMPFlags(temp_byte, RegX);		// set the flags accordingly
			break;

		case (BYTE)0xE1:		// SBC (Z,X) - Subtract acc with carry indirect X
			cycles +=6;
			PC +=2;
			temp_word = INDIRECTX_OPERAND_ADDR();		// get address
			temp_byte = my_mem_bus->read_byte(temp_word);			
			temp_byte2 = RegAcc;		//save the original Acc
			temp_word = RegAcc - temp_byte  - (1-1*bit_set(SR_CARRY_FLAG));	// do the sub
			RegAcc = temp_word & 0x00FF;
			SET_NZ(RegAcc);		// set N and Z flags
			if (temp_word & 0x100)	// Set Carry flag
				SR_CLEAR_BIT(SR_CARRY_FLAG);
			else
				SR_SET_BIT(SR_CARRY_FLAG);				
			// now set overflow flag:
			// if A and B are different signs then overflow possible
			// and if A and A' are different signs then overflow occured
			if (   ((temp_byte2 ^ temp_byte) & 0x80)
				&& ((temp_byte2 ^ RegAcc) & 0x80))
				SR_SET_BIT(SR_OVERFLOW_FLAG);
			else
				SR_CLEAR_BIT(SR_OVERFLOW_FLAG);
			break;

		case (BYTE)0xE4:		// CPX Z - compare mem and X zero page
			cycles += 3;
			PC += 2;
			temp_word = ZERO_PAGE_OPERAND_ADDR();
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data			
			SetCMPFlags(temp_byte, RegX);		// set the flags accordingly
			break;

		case (BYTE)0xE5:		// SBC z - Subtract acc with carry zero page			
			cycles +=3;
			PC +=2;			
			temp_word = ZERO_PAGE_OPERAND_ADDR();
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data				
			temp_byte2 = RegAcc;		//save the original Acc
			temp_word = RegAcc - temp_byte  - (1-1*bit_set(SR_CARRY_FLAG));	// do the add
			RegAcc = temp_word & 0x00FF;
			SET_NZ(RegAcc);		// set N and Z flags
			if (temp_word & 0x100)	// Set Carry flag
				SR_CLEAR_BIT(SR_CARRY_FLAG);
			else
				SR_SET_BIT(SR_CARRY_FLAG);				
			// now set overflow flag:
			// if A and B are different signs then overflow possible
			// and if A and A' are different signs then overflow occured
			if (   ((temp_byte2 ^ temp_byte) & 0x80)
				&& ((temp_byte2 ^ RegAcc) & 0x80))
				SR_SET_BIT(SR_OVERFLOW_FLAG);
			else
				SR_CLEAR_BIT(SR_OVERFLOW_FLAG);
			break;

		case (BYTE)0xE6:		// INC Z - increment memory by one Zero Page
			cycles += 5;
			PC += 2;
			temp_word = ZERO_PAGE_OPERAND_ADDR();		// read in address
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data
			temp_byte++;		// increment it
			my_mem_bus->write_byte(temp_word, temp_byte);	// put it back
			SET_NZ(temp_byte);		// set flags
			break;

		case (BYTE)0xE8:		// INX - increment X
			PC++;			// increment the PC by one
			cycles += 2;	// takes 2 cycles
			RegX++;		// increment
			SET_NZ(RegX);
			break;

		case (BYTE)0xE9:		// SBC #n - Subtract acc with carry
			cycles +=2;
			PC +=2;
			temp_word = IMMEDIATE_OPERAND_ADDR();
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data
			temp_byte2 = RegAcc;		//save the original Acc
			temp_word = RegAcc - temp_byte  - (1-1*bit_set(SR_CARRY_FLAG));	// do the add
			RegAcc = temp_word & 0x00FF;
			SET_NZ(RegAcc);		// set N and Z flags
			if (temp_word & 0x100)	// Set Carry flag
				SR_CLEAR_BIT(SR_CARRY_FLAG);
			else
				SR_SET_BIT(SR_CARRY_FLAG);				
			// now set overflow flag:
			// if A and B are different signs then overflow possible
			// and if A and A' are different signs then overflow occured
			if (   ((temp_byte2 ^ temp_byte) & 0x80)
				&& ((temp_byte2 ^ RegAcc) & 0x80))
				SR_SET_BIT(SR_OVERFLOW_FLAG);
			else
				SR_CLEAR_BIT(SR_OVERFLOW_FLAG);
			break;

		case (BYTE)0xEA:		// NOP - no operation
			cycles += 2;
			PC++;
			// not hard to implement :)
			break;

		case (BYTE)0xEC:		// CPX Q - compare mem and X absolute
			cycles += 4;
			PC += 3;
			temp_word = ABSOLUTE_OPERAND_ADDR();
			temp_byte = my_mem_bus->read_byte(temp_word);	// read in data
			SetCMPFlags(temp_byte, RegX);		// set the flags accordingly
			break;

		case (BYTE)0xED:		// SBC Q - Subtract acc with carry Abolute
			cycles +=4;
			PC +=3;
			temp_word = ABSOLUTE_OPERAND_ADDR();
			temp_byte = my_mem_bus->read_byte(temp_word);	// get data
			temp_byte2 = RegAcc;		//save the original Acc
			temp_word = RegAcc - temp_byte  - (1-1*bit_set(SR_CARRY_FLAG));	// do the add
			RegAcc = temp_word & 0x00FF;
			SET_NZ(RegAcc);		// set N and Z flags
			if (temp_word & 0x100)	// Set Carry flag
				SR_CLEAR_BIT(SR_CARRY_FLAG);
			else
				SR_SET_BIT(SR_CARRY_FLAG);				
			// now set overflow flag:
			// if A and B are different signs then overflow possible
			// and if A and A' are different signs then overflow occured
			if (   ((temp_byte2 ^ temp_byte) & 0x80)
				&& ((temp_byte2 ^ RegAcc) & 0x80))
				SR_SET_BIT(SR_OVERFLOW_FLAG);
			else
				SR_CLEAR_BIT(SR_OVERFLOW_FLAG);
			break;

		case (BYTE)0xEE:		// INC Q - increment memory by one Zero Page
			cycles += 6;
			PC += 3;
			temp_word = ABSOLUTE_OPERAND_ADDR();	// get address from opcode			;
			temp_byte = my_mem_bus->read_byte(temp_word);			
			temp_byte++;		// increment it
			my_mem_bus->write_byte(temp_word, temp_byte);	// put it back
			SET_NZ(temp_byte);		// set flags
			break;

		case (BYTE)0xF0:		// BEQ - branch if equal		
			PC += 2;					// only one byte opcode
			if (bit_set(SR_ZERO_FLAG) )	// Z=1
			{
				cycles += 4;		// 4 cycles if branch taken
				PC = RELATIVE_OPERAND_ADDR();
			}
			else			// Z=0
			{
				cycles += 2;		// only 2 cycles if no branch taken						
			}
			// no flags affected
			break;

		case (BYTE)0xF1:		// SBC (Z),Y - Subtract acc with carry indirect Y
			cycles +=5;
			PC +=2;
			temp_word = INDIRECTY_OPERAND_ADDR();		// get address
			temp_byte = my_mem_bus->read_byte(temp_word);			
			temp_byte2 = RegAcc;		//save the original Acc
			temp_word = RegAcc - temp_byte  - (1-1*bit_set(SR_CARRY_FLAG));	// do the sub
			RegAcc = temp_word & 0x00FF;
			SET_NZ(RegAcc);		// set N and Z flags
			if (temp_word & 0x100)	// Set Carry flag
				SR_CLEAR_BIT(SR_CARRY_FLAG);
			else
				SR_SET_BIT(SR_CARRY_FLAG);				
			// now set overflow flag:
			// if A and B are different signs then overflow possible
			// and if A and A' are different signs then overflow occured
			if (   ((temp_byte2 ^ temp_byte) & 0x80)
				&& ((temp_byte2 ^ RegAcc) & 0x80))
				SR_SET_BIT(SR_OVERFLOW_FLAG);
			else
				SR_CLEAR_BIT(SR_OVERFLOW_FLAG);
			break;

		case (BYTE)0xF5:		// SBC Z,X - Subtract acc with carry zero page, X
			cycles +=4;
			PC +=2;
			temp_word = ZERO_PAGEX_OPERAND_ADDR();	// get address
			temp_byte = my_mem_bus->read_byte(temp_word);			
			temp_byte2 = RegAcc;		//save the original Acc
			temp_word = RegAcc - temp_byte  - (1-1*bit_set(SR_CARRY_FLAG));	// do the sub
			RegAcc = temp_word & 0x00FF;
			SET_NZ(RegAcc);		// set N and Z flags
			if (temp_word & 0x100)	// Set Carry flag
				SR_CLEAR_BIT(SR_CARRY_FLAG);
			else
				SR_SET_BIT(SR_CARRY_FLAG);				
			// now set overflow flag:
			// if A and B are different signs then overflow possible
			// and if A and A' are different signs then overflow occured
			if (   ((temp_byte2 ^ temp_byte) & 0x80)
				&& ((temp_byte2 ^ RegAcc) & 0x80))
				SR_SET_BIT(SR_OVERFLOW_FLAG);
			else
				SR_CLEAR_BIT(SR_OVERFLOW_FLAG);
			break;

		case (BYTE)0xF6:		// INC Z,X - increment memory zero page X
			cycles += 6;
			PC += 2;
			temp_word = ZERO_PAGEX_OPERAND_ADDR();			
			temp_byte = my_mem_bus->read_byte(temp_word);			
			temp_byte++;			// do the increment
			my_mem_bus->write_byte(temp_word, temp_byte);	// write it back
			SET_NZ(temp_byte);
			break;

		case (BYTE)0xF8:                  // SED - set decimal flag
			cycles += 2;			          // keep track of number of clock cycles
			PC += 1;					          // 2 byte opcode, so increment PC by 2
			SR_SET_BIT(SR_DECIMAL_FLAG);	 // set the decimal flag
			break;

		case (BYTE)0xF9:		// SBC Q,Y - Subtract acc with carry			
			cycles +=4;
			PC +=3;
			temp_word = ABSOLUTEY_OPERAND_ADDR();	// get address			
			temp_byte = my_mem_bus->read_byte(temp_word);			
			temp_byte2 = RegAcc;		//save the original Acc
			temp_word = RegAcc - temp_byte  - (1-1*bit_set(SR_CARRY_FLAG));	// do the sub
			RegAcc = temp_word & 0x00FF;
			SET_NZ(RegAcc);		// set N and Z flags
			if (temp_word & 0x100)	// Set Carry flag
				SR_CLEAR_BIT(SR_CARRY_FLAG);
			else
				SR_SET_BIT(SR_CARRY_FLAG);				
			// now set overflow flag:
			// if A and B are different signs then overflow possible
			// and if A and A' are different signs then overflow occured
			if (   ((temp_byte2 ^ temp_byte) & 0x80)
				&& ((temp_byte2 ^ RegAcc) & 0x80))
				SR_SET_BIT(SR_OVERFLOW_FLAG);
			else
				SR_CLEAR_BIT(SR_OVERFLOW_FLAG);
			break;

		case (BYTE)0xFD:		// SBC Q,X - Subtract acc with carry absolute, X
			cycles +=4;
			PC +=3;
			temp_word = ABSOLUTEX_OPERAND_ADDR();	// get address
			temp_byte = my_mem_bus->read_byte(temp_word);			
			temp_byte2 = RegAcc;		//save the original Acc
			temp_word = RegAcc - temp_byte  - (1-1*bit_set(SR_CARRY_FLAG));	// do the sub
			RegAcc = temp_word & 0x00FF;
			SET_NZ(RegAcc);		// set N and Z flags
			if (temp_word & 0x100)	// Set Carry flag
				SR_CLEAR_BIT(SR_CARRY_FLAG);
			else
				SR_SET_BIT(SR_CARRY_FLAG);				
			// now set overflow flag:
			// if A and B are different signs then overflow possible
			// and if A and A' are different signs then overflow occured
			if (   ((temp_byte2 ^ temp_byte) & 0x80)
				&& ((temp_byte2 ^ RegAcc) & 0x80))
				SR_SET_BIT(SR_OVERFLOW_FLAG);
			else
				SR_CLEAR_BIT(SR_OVERFLOW_FLAG);
			break;

		case (BYTE)0xFE:		// INC Q,X - increment memory absolute X
			cycles += 7;
			PC += 3;
			temp_word = ABSOLUTEX_OPERAND_ADDR();	// get address
			temp_byte = my_mem_bus->read_byte(temp_word);			
			temp_byte++;			// do the increment
			my_mem_bus->write_byte(temp_word, temp_byte);	// write it back
			SET_NZ(temp_byte);
			break;

		default:
			TRACE("Unsupported opcode encountered %#2x at %x PC\n", (int)IR, PC);
			return 0;		// stop emulation
			break;

	}
	return 1;		// tell main program to continue;
}


BYTE CPU::bit_set(short bit_number)
// returns 1 if the specified bit in the status register is a 1, returns 0 if specified bit is a 0
{
	return ( SR & (0x01 << bit_number) );
}

void CPU::vblank_interrupt(long int &cycles)
// execute a vblank interrupt
{
	cycles += 7;		// 7 cycle interrupt latency
	push(PC);			// push the Program counter on stack
	push(SR);			// and the status register
	PC = my_mem_bus->read_address(NMI_VECTOR);		// the NMI vector
}

void CPU::push(BYTE value)
// pushes byte onto the stack
{	
	my_mem_bus->write_byte(SP, value);	// put it on the stack
	SP --;				// update the stack pointer
	
}

void CPU::push(WORD value)
// pushes word onto the stack
{	
	BYTE tempByte = 0;

	tempByte = (value >> 8);		// get the high byte
	push(tempByte);
	tempByte = value & 0x00FF;		// get the low byte of the word
	push(tempByte);	
}

BYTE CPU::pop_byte()
// Pops a byte off of the stack, and returns it
{
	SP++;			// increment Stack pointer
	return my_mem_bus->read_byte(SP);
}

WORD CPU::pop_word()
// Pops a word off of the stack, and returns it.
{
	SP += 2;			// increment Stack pointer
	return my_mem_bus->read_address(SP-1);
}

void CPU::SetCMPFlags(BYTE value1, BYTE value2)
// sets the status register flags accordingly
// value2 = regAcc, should do value2 - value1

{
	BYTE temp_byte;

	temp_byte = value2 - value1;	// do the comparison
	SET_NZ(temp_byte);		// set N and Z
	if (value2 >= value1)	// set C
		SR_SET_BIT(SR_CARRY_FLAG);
	else
		SR_CLEAR_BIT(SR_CARRY_FLAG);
}

