/*
Copyright (C) 1999 Chris Teague

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
*/

#include "stdafx.h"
#include "debugger.h"
#include "conio.h"
#include <windows.h> 
#include "stdlib.h"
#include "string.h"

void gotoxy(int x, int y);
void redraw_static();


debugger::debugger(CPU* temp_cpu, mem_bus* bus)
// Constructor 
{
	cpu = temp_cpu;		// assign the pointer to cpu
	debug_mem_bus = bus;
	cpu_mem_addr = 0;
	ppu_mem_addr = 0;
	prev_instr_line = NULL;
	pc_target = NULL;	
	pc_base = NULL;
	//cpu_mem_range_changed = 1;		// redraw the first time through
	redraw_static();			// redraw the part of the screen that does not change

}

debugger::~debugger()
// Destructor
{
	cpu = NULL;		// unassign pointer to cpu
}

BYTE read_kb_key()
{
	return getch();		// get a character from keyboard
}

BYTE debugger::update()
// This function retrieves all data needed by the debugger and redraws the debugger screen
{	
	BYTE key = NULL;
	char buffer[50];		// to hold the command
	int i=0, q=0;
	int pc_ptr;				
	BYTE invalid_command;		// flag to indicate whether the typed command is valid or not
	
	cpu->get_debug_info(acc, x, y, pc, sr, sp);		// get values from the cpu for the debugger to display	
	if (pc_target != NULL)		// if break point set
	{
		if (pc_target == pc)	// if we are at the break point
			pc_target = NULL;	// stop the cpu
		else
			return 1;			// tell cpu to continue
	}
	
	while(1)		// do loop indefinately (this loop is exited using return() statements)
	{
		cpu->get_debug_info(acc, x, y, pc, sr, sp);		// get values from the cpu for the debugger to display
		debug_mem_bus->get_debug_info(cpu_mem_addr, cpu_mem_data, pc, program_data);			// get values from the cpu address space
		
		
		gotoxy(2,1);
		printf("PC $%4.4X    SP $%4.4X    ACC $%2.2X    SR $%2.2X %1.1X%1.1X%1.1X%1.1X%1.1X%1.1X%1.1X%1.1X",pc, sp, acc, sr, (sr&0x80)>>7, (sr&0x40)>>6, (sr&0x20)>>5, (sr&0x10)>>4, (sr&0x08)>>3, (sr&0x04)>>2, (sr&0x02)>>1, sr&0x01);
		gotoxy(3,2);
		printf("X $%2.2X       Y $%2.2X                        NV BDIZC", x, y);		

		//if (cpu_mem_range_changed)		// only redraw the memory if needed
		//{
			for (q = 0; q <5; q++)	// this loop prints the CPU memory values
			{		
				if (cpu_mem_addr != prev_cpu_mem_addr)		// need to redraw the addresses
				{
					for (i=0; i<5; i++)
					{
						gotoxy(0,4+i);
						printf("| $%4.4X",cpu_mem_addr+(i<<4));
					}
					prev_cpu_mem_addr = cpu_mem_addr;						
				}
				for (i = 0; i <16; i++)
					if ( prev_cpu_mem_data[(q<<4)+i] != cpu_mem_data[(q<<4)+i] )	// if data has changed
					{						
						gotoxy(3*i+7,4+q);
						printf(" %2.2X",cpu_mem_data[(q<<4)+i]);				
						prev_cpu_mem_data[(q<<4)+i] = cpu_mem_data[(q<<4)+i];
					}
			}
			cpu_mem_range_changed = 0;		
		//}		
		
		if ( (pc_base==NULL) || (pc > instructions[11]) || (pc < instructions[1]) )		// check if instructions need to be redrawn
		{
			pc_base = pc;		// set the pc_base
			pc_ptr = 0;			
			for(i=0; i<13; i++)		// print the instructions
			{
				instructions[i] = pc+pc_ptr;		// assign starting address for this instruction
				gotoxy(1,10+i);
				printf(" $%4.4X ", pc + pc_ptr);		// print the address of the instruction
				print_mnemonic(pc_ptr, pc);	// print the mnemonic
				//printf("                            |                     |");
			}		
			gotoxy(1,10);
			printf(">");		// draw new indicator
			prev_instr_line = 10;		// update previous indicator
		}
		else		// we don't need to redraw entire instruction section
		{
			for(i=0; i<13; i++)		// update the current instruction indicator
			{
				if (pc == instructions[i])
				{
					gotoxy(1,prev_instr_line);
					printf(" ");		// erase previous indicator
					gotoxy(1,10+i);
					printf(">");		// draw new indicator
					prev_instr_line = 10+i;		// update previous indicator
				}
			}

		}
		
		gotoxy(0,24);		// always leave cursor at bottom of screen
		key = read_kb_key();
		switch(key)
		{
		case 0x63:	// 'c' for command
			while(kbhit())
			{
			}
			key = NULL;
			gotoxy(0,24);
			printf("command> ");			
			i = 0;
			while( (key = getche()) != '\r')		// this loop gets the command and puts it in buffer
				buffer[i++] = key;			
			buffer[i]=NULL;
			gotoxy(0,24);
			printf("                       ");		// clear the command line from the screen
			invalid_command =0;					// assume the command is valid until we find otherwise
			key = NULL;
			if ( (buffer[0]=='g') && (buffer[1]==' ')&&(strlen(buffer)==6) )		// the 'go to' command
			{
				for (i=2; i<=5; i++)
				{
					if (buffer[i] & 0x40)		// then it is not a digit, may be A-F
					{
						buffer[i] &= 0xdf;
						if ( (buffer[i] > 0x40) && (buffer[i] < 0x47) )	// then it is a A-F
							buffer[i] -= 7;
						else
							invalid_command =1;			// then it is not an A-F and is invalid
					}
					
				}
				
				if (invalid_command == 0)
				{
					pc_target = buffer[5]-0x30;
					pc_target += (buffer[4]-0x30)<<4;
					pc_target += (buffer[3]-0x30)<<8;
					pc_target += (buffer[2]-0x30)<<12;
					return 1;		// tell cpu to go
				}
			}
			else if ( (buffer[0]=='m') && (buffer[1]=='e') && (buffer[2]=='m') && (buffer[3]==' ') && (buffer[4]=='c') && (buffer[5]=='p') && (buffer[6]=='u') && (buffer[7]==' ') && (strlen(buffer)==12) )		// the 'mem cpu' command
			{
				for (i=8; i<=11; i++)
				{
					if (buffer[i] & 0x40)		// then it is not a digit, may be A-F
					{
						buffer[i] &= 0xdf;
						if ( (buffer[i] > 0x40) && (buffer[i] < 0x47) )	// then it is a A-F
							buffer[i] -= 7;
						else
							invalid_command =1;			// then it is not an A-F and is invalid
					}
					
				}
				
				if (invalid_command == 0)
				{
					cpu_mem_addr = buffer[11]-0x30;
					cpu_mem_addr += (buffer[10]-0x30)<<4;
					cpu_mem_addr += (buffer[9]-0x30)<<8;
					cpu_mem_addr += (buffer[8]-0x30)<<12;					
				}	
				
			}
			else if ( (buffer[0]=='r') && (buffer[1]=='u') && (buffer[2]=='n')&&(strlen(buffer)==3) )		// the 'run' command
			{
				pc_target = 0xFFFC;		// the reset vector, cpu won't stop until a reset performed
				return 1;				// tell cpu to go
			}
			break;
		
		case 0x73:	// 's' for step
			key = NULL;
			return 1;
			break;

		case 0x74:	// 't' for trace
			key = NULL;
			return 1;
			break;

		case 'q':	// quit
			return 0;		// tell main program to quit
			break;

		default:
			break;
		}
	}	// while (1)			
		
}



void gotoxy(int x, int y) 
{ 
	HANDLE hConsoleOutput; 
	COORD dwCursorPosition; 

	dwCursorPosition.X = x; 
	dwCursorPosition.Y = y; 
	hConsoleOutput = GetStdHandle(STD_OUTPUT_HANDLE); 
	SetConsoleCursorPosition(hConsoleOutput,dwCursorPosition); 
}

void debugger::print_mnemonic(int &pc_ptr, WORD pc )
// print the opcode and mnemonic in a 21 character string at the current screen location
{
	BYTE IR;		// holds the instruction
	IR = program_data[pc_ptr];			// get the opcode
	BYTE temp_byte;
	WORD temp_word;
	switch(IR)
	{
		case (BYTE)0x01:		// ORA indirect x - Logical Or of mem and acc			
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("ORA ($%2.2X,X)", program_data[pc_ptr+1]);
			pc_ptr += 2;		// two bytes in the instruction
			break;

		case (BYTE)0x05:		// ORA Z - Logical Or of mem and Acc Zero Page
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("ORA ($%2.2X)  ", program_data[pc_ptr+1]);
			pc_ptr += 2;		// two bytes in the instruction
			break;

		case (BYTE)0x06:		// ASL Z - Arithmetic Shift Left zero page
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("ASL ($%2.2X)  ", program_data[pc_ptr+1]);
			pc_ptr += 2;		// two bytes in the instruction
			break;

		case (BYTE)0x09:		// ORA #n - Logical Or of Mem and Acc
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("ORA #$%2.2X   ", program_data[pc_ptr+1]);
			pc_ptr += 2;		// two bytes in the instruction
			break;

		case (BYTE)0x0A:		// ASL accumulator
			printf("%2.2X        ", program_data[pc_ptr]);			
			printf("ASL        ");
			pc_ptr ++;		// only one byte opcode
			break;

		case (BYTE)0x10:		// BPL - Branch if Plus 
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			pc_ptr += 2;
			temp_byte = program_data[pc_ptr-1];
			if (temp_byte > 0x7f)	// then it is a negative number
			{
				temp_byte = (0xff - temp_byte)+1;
				temp_word = (pc+pc_ptr) - temp_byte;
			}
			else 
				temp_word = pc+pc_ptr+temp_byte;
			printf("BPL $%4.4X  ", temp_word);// only one byte opcode
			break;

		case (BYTE)0x18:		// CLC - clear carry flag
			printf("%2.2X        ", program_data[pc_ptr]);						
			printf("CLC        ");
			pc_ptr++;			// increment the PC by one			
			break;

		case (BYTE)0x20:		// JSR - jump to subroutine
			printf("%2.2X %2.2X %2.2X  ", program_data[pc_ptr], program_data[pc_ptr+1], program_data[pc_ptr+2]);
			printf("JSR $%2.2X%2.2X  ", program_data[pc_ptr+2], program_data[pc_ptr+1]);
			pc_ptr += 3;	
			break;

		case (BYTE)0x24:		// BIT Z - bit test with acc
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("BIT ($%2.2X)  ", program_data[pc_ptr+1]);
			pc_ptr += 2;		// two byets in the instruction
			break;

		case (BYTE)0x25:		// AND Z - Logical AND of mem and Acc Zero Page
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("AND ($%2.2X)  ", program_data[pc_ptr+1]);
			pc_ptr += 2;		// two byets in the instruction
			break;

		case (BYTE)0x26:		// ROL Z - rotate left Zero Page
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("ROL ($%2.2X)  ", program_data[pc_ptr+1]);
			pc_ptr += 2;		// two byets in the instruction
			break;

		case (BYTE)0x29:		// AND #n - Logical AND of immeiate and Acc
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("AND #$%2.2X   ", program_data[pc_ptr+1]);
			pc_ptr += 2;		// two byets in the instruction
			break;

		case (BYTE)0x2A:		// ROL - rotate left accumulator
			printf("%2.2X        ", program_data[pc_ptr]);						
			printf("ROL A      ");
			pc_ptr++;			// move stack pointer
			break;

		case (BYTE)0x2E:		// ROL - rotate left absolute
			printf("%2.2X %2.2X %2.2X  ", program_data[pc_ptr], program_data[pc_ptr+1], program_data[pc_ptr+2]);
			printf("ROL ($%2.2X%2.2X)", program_data[pc_ptr+2], program_data[pc_ptr+1]);
			pc_ptr += 3;	
			break;

		case (BYTE)0x36:		// ROL Z,X - rotate left Zero Page X
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("ROL $%2.2X,X  ", program_data[pc_ptr+1]);
			pc_ptr += 2;		// two byets in the instruction
			break;

		case (BYTE)0x38:		// SEC - set carry flag
			printf("%2.2X        ", program_data[pc_ptr]);						
			printf("SEC        ");
			pc_ptr++;			// increment the PC by one			
			break;

		case (BYTE)0x3E:		// ROL Q,X- rotate left absolute X
			printf("%2.2X %2.2X %2.2X  ", program_data[pc_ptr], program_data[pc_ptr+1], program_data[pc_ptr+2]);
			printf("ROL $%2.2X%2.2X,X", program_data[pc_ptr+2], program_data[pc_ptr+1]);			
			pc_ptr +=3;
			break;

		case (BYTE)0x40:		// RTI - return from interrupt
			printf("%2.2X        ", program_data[pc_ptr]);						
			printf("RTI        ");
			pc_ptr++;			// move stack pointer
			break;

		case (BYTE)0x46:		// LSR Z - logical shift right zero page
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("LSR ($%2.2X)  ", program_data[pc_ptr+1]);
			pc_ptr += 2;		// two byets in the instruction
			break;

		case (BYTE)0x48:		// PHA - push Acc on stack
			printf("%2.2X        ", program_data[pc_ptr]);						
			printf("PHA        ");
			pc_ptr++;
			break;

		case (BYTE)0x4A:		// LSR accumulator
			printf("%2.2X        ", program_data[pc_ptr]);			
			printf("LSR        ");
			pc_ptr++;
			break;

		case (BYTE)0x4E:		// LSR Q - logical shift right absolute
			printf("%2.2X %2.2X %2.2X  ", program_data[pc_ptr], program_data[pc_ptr+1], program_data[pc_ptr+2]);
			printf("LSR $%2.2X%2.2X  ", program_data[pc_ptr+2], program_data[pc_ptr+1]);			
			pc_ptr +=3;
			break;

		case (BYTE)0x4C:		// JMP absolute
			printf("%2.2X %2.2X %2.2X  ", program_data[pc_ptr], program_data[pc_ptr+1], program_data[pc_ptr+2]);
			printf("JMP $%2.2X%2.2X  ", program_data[pc_ptr+2], program_data[pc_ptr+1]);			
			pc_ptr +=3;
			break;

		case (BYTE)0x4D:		// EOR Q - exclusive OR memory absolute
			printf("%2.2X %2.2X %2.2X  ", program_data[pc_ptr], program_data[pc_ptr+1], program_data[pc_ptr+2]);
			printf("EOR ($%2.2X%2.2X)", program_data[pc_ptr+2], program_data[pc_ptr+1]);			
			pc_ptr +=3;
			break;

		case (BYTE)0x50:		// BVC - branch if overflow clear
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);			
			pc_ptr += 2;					// only one byte opcode
			temp_byte = program_data[pc_ptr-1];
			if (temp_byte > 0x7f)	// then it is a negative number
			{
				temp_byte = (0xff - temp_byte)+1;
				temp_word = (pc+pc_ptr) - temp_byte;
			}
			else 
				temp_word = pc+pc_ptr+temp_byte;
			printf("BVC $%4.4X  ", temp_word);
			break;

		case (BYTE)0x58:		// CLI - clear IRQ disable flag
			printf("%2.2X        ", program_data[pc_ptr]);
			printf("CLI        ");
			pc_ptr++;			// increment the PC by one
			break;

		case (BYTE)0x60:		// RTS return to subroutine
			printf("%2.2X        ", program_data[pc_ptr]);			
			printf("RTS        ");
			pc_ptr++;
			break;

		case (BYTE)0x65:		// ADC Z - add acc with carry Zero Page
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("ADC ($%2.2X)  ", program_data[pc_ptr+1]);
			pc_ptr +=2;
			break;

		case (BYTE)0x66:		// ROR Z- rotate right zero page			
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("ROR $%2.2X    ", program_data[pc_ptr+1]);
			pc_ptr +=2;
			break;

		case (BYTE)0x68:		// PLA - pull Acc off stack
			printf("%2.2X        ", program_data[pc_ptr]);			
			printf("PLA        ");
			pc_ptr++;
			break;

		case (BYTE)0x69:		// ADC #n - add acc with carry
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("ADC #$%2.2X   ", program_data[pc_ptr+1]);
			pc_ptr +=2;
			break;

		case (BYTE)0x6A:		// ROR - rotate right accumulator
			printf("%2.2X        ", program_data[pc_ptr]);			
			printf("ROR A      ");
			pc_ptr++;
			break;

		case (BYTE)0x6C:		// JMP indirect
			printf("%2.2X %2.2X %2.2X  ", program_data[pc_ptr], program_data[pc_ptr+1], program_data[pc_ptr+2]);
			printf("JMP ($%2.2X%2.2X)", program_data[pc_ptr+2], program_data[pc_ptr+1]);
			pc_ptr += 3;			// increment PC
			break;

		case (BYTE)0x6E:		// ROR Q- rotate right absolute
			printf("%2.2X %2.2X %2.2X  ", program_data[pc_ptr], program_data[pc_ptr+1], program_data[pc_ptr+2]);
			printf("ROR $%2.2X%2.2X  ", program_data[pc_ptr+2], program_data[pc_ptr+1]);
			pc_ptr += 3;			// increment PC
			break;

		case (BYTE)0x76:		// ROR Z,X - rotate right Zero Page X
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("ROR ($%2.2X),X", program_data[pc_ptr+1]);
			pc_ptr += 2;
			break;

		case (BYTE)0x78:		// SEI - disable interrupts
			printf("%2.2X        ", program_data[pc_ptr]);			
			printf("SEI        ");
			pc_ptr++;
			break;

		case (BYTE)0x7E:		// ROR Q,X- rotate right absolute X
			printf("%2.2X %2.2X %2.2X  ", program_data[pc_ptr], program_data[pc_ptr+1], program_data[pc_ptr+2]);
			printf("ROR $%2.2X%2.2X,X", program_data[pc_ptr+2], program_data[pc_ptr+1]);
			pc_ptr += 3;			// increment PC
			break;


		case (BYTE)0x84:		// STY - store Y in memory Zero Page
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("STY $%4.4X  ", program_data[pc_ptr+1]);
			pc_ptr += 2;
			break;

		case (BYTE)0x85:		// STA Z - Store A in Memory zero page
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("STA $%2.2X    ", program_data[pc_ptr+1]);
			pc_ptr += 2;								// 3 byte instruction
			break;

		case 0x86:		// STX Z - Store X in Memory zero page
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("STX $%2.2X    ", program_data[pc_ptr+1]);
			pc_ptr += 2;								// 3 byte instruction
			break;

		case (BYTE)0x88:		// DEY - decrement Y
			printf("%2.2X        ", program_data[pc_ptr]);			
			printf("DEY        ");
			pc_ptr++;
			break;

		case (BYTE)0x8A:		// TXA - transfer X to A
			printf("%2.2X        ", program_data[pc_ptr]);			
			printf("TXA        ");
			pc_ptr++;
			break;

		case (BYTE)0x8C:		// STY Q - Store Y in Memory absolute
			printf("%2.2X %2.2X %2.2X  ", program_data[pc_ptr], program_data[pc_ptr+1], program_data[pc_ptr+2]);
			printf("STY $%2.2X%2.2X  ", program_data[pc_ptr+2], program_data[pc_ptr+1]);
			pc_ptr += 3;			// three byte opcode
			break;

		case (BYTE)0x8D:		// STA absolute- store accumulator
			printf("%2.2X %2.2X %2.2X  ", program_data[pc_ptr], program_data[pc_ptr+1], program_data[pc_ptr+2]);
			printf("STA $%2.2X%2.2X  ", program_data[pc_ptr+2], program_data[pc_ptr+1]);
			pc_ptr += 3;			// three byte opcode
			break;

		case (BYTE)0x8E:		// STX - Store X in Memory absolute			
			printf("%2.2X %2.2X %2.2X  ", program_data[pc_ptr], program_data[pc_ptr+1], program_data[pc_ptr+2]);
			printf("STX $%2.2X%2.2X  ", program_data[pc_ptr+2], program_data[pc_ptr+1]);
			pc_ptr += 3;								// 3 byte instruction
			break;

		case (BYTE)0x90:		// BCC - branch if carry clear
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			pc_ptr += 2;
			temp_byte = program_data[pc_ptr-1];
			if (temp_byte > 0x7f)	// then it is a negative number
			{
				temp_byte = (0xff - temp_byte)+1;
				temp_word = (pc+pc_ptr) - temp_byte;
			}
			else 
				temp_word = pc+pc_ptr+temp_byte;
			printf("BCC $%4.4X  ", temp_word);// only one byte opcode			
			break;

		case (BYTE)0x91:		// STA (Z),Y - store acc indirect ,Y
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("STA ($%2.2X),Y", program_data[pc_ptr+1]);
			pc_ptr += 2;
			break;

		case (BYTE)0x94:		// STY Z,Y - store Y zero page X
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("STY ($%2.2X),Y", program_data[pc_ptr+1]);
			pc_ptr += 2;
			break;

		case (BYTE)0x95:		// STA Z,X - store Acc zero page X
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("STA ($%2.2X),X", program_data[pc_ptr+1]);
			pc_ptr += 2;
			break;

		case (BYTE)0x96:		// STX Z,Y - store Acc zero page Y
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("STX ($%2.2X),X", program_data[pc_ptr+1]);
			pc_ptr += 2;
			break;

		case (BYTE)0x98:		// TYA - transfer Y to Acc
			printf("%2.2X        ", program_data[pc_ptr]);
			printf("TYA        ");			
			pc_ptr++;
			break;

		case (BYTE)0x9A:		// TXS - transfer X to S
			printf("%2.2X        ", program_data[pc_ptr]);
			printf("TXS        ");			
			pc_ptr++;
			break;		

		case (BYTE)0xA0:		// LDY - load Y immediate
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("LDY #$%2.2X   ",program_data[pc_ptr+1]);
			pc_ptr +=2;
			break;

		case (BYTE)0xA1:		// LDA (Z,X) - load Acc indirect X
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("LDa ($%2.2X,X)",program_data[pc_ptr+1]);
			pc_ptr += 2;		// increment PC by two because this instruction is 2 bytes long;
			break;

		case (BYTE)0xA2:		// LDX #n - Load X immediate
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("LDX #$%2.2X   ",program_data[pc_ptr+1]);
			pc_ptr += 2;		// increment PC by two because this instruction is 2 bytes long;
			break;

		case (BYTE)0xA4:		// LDY Z - load register Y zero page
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("LDY ($%2.2X)  ", program_data[pc_ptr+1]);
			pc_ptr += 2;
			break;

		case (BYTE)0xA5:		// LDA - load accumulator zero page
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("LDA ($%2.2X)  ", program_data[pc_ptr+1]);
			pc_ptr += 2;
			break;

		case (BYTE)0xA6:		// LDX - load register X zero page
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("LDX ($%4.4X)", program_data[pc_ptr+1]);
			pc_ptr += 2;			
			break;

		case 0xA8:		// TAY - tranfer A to Y
			printf("%2.2X        ", program_data[pc_ptr]);
			printf("TAY        ");
			pc_ptr++;
			break;

		case (BYTE)0xA9:		// LDA immediate
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("LDA #$%2.2X   ",program_data[pc_ptr+1]);
			pc_ptr += 2;
			break;

		case (BYTE)0xAA:		// TAX - transfer A to X
			printf("%2.2X        ", program_data[pc_ptr]);
			printf("TAX        ");
			pc_ptr++;
			break;

		case (BYTE)0xAE:		// Load X absolute
			printf("%2.2X %2.2X %2.2X  ", program_data[pc_ptr], program_data[pc_ptr+1], program_data[pc_ptr+2]);
			printf("LDX $%2.2X%2.2X  ", program_data[pc_ptr+2],program_data[pc_ptr+1]);
			pc_ptr += 3;
			break;

		case (BYTE)0xAD:		// Load Accumulator absolute
			printf("%2.2X %2.2X %2.2X  ", program_data[pc_ptr], program_data[pc_ptr+1], program_data[pc_ptr+2]);
			printf("LDA $%2.2X%2.2X  ", program_data[pc_ptr+2],program_data[pc_ptr+1]);
			pc_ptr += 3;
			break;

		case (BYTE)0xB0:		// BCS - branch if carry set
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			pc_ptr += 2;
			temp_byte = program_data[pc_ptr-1];
			if (temp_byte > 0x7f)	// then it is a negative number
			{
				temp_byte = (0xff - temp_byte)+1;
				temp_word = (pc+pc_ptr) - temp_byte;
			}
			else 
				temp_word = pc+pc_ptr+temp_byte;
			printf("BCS $%4.4X  ", temp_word);// only one byte opcode			
			break;

		case (BYTE)0xB1:		// LDA (Z),Y - load acc indirect Y
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("LDA $(%2.2X),Y",program_data[pc_ptr+1]);
			pc_ptr += 2;
			break;		

		case (BYTE)0xB4:		// LDY Z,X - load Y zero page X
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("LDY $(%2.2X),X",program_data[pc_ptr+1]);
			pc_ptr += 2;
			break;	

		case (BYTE)0xB5:		// LDA (Z),X - load Acc zero page X
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("LDA $(%2.2X),X",program_data[pc_ptr+1]);
			pc_ptr += 2;
			break;

		case (BYTE)0xB8:		// CLV - clear overflow flag
			printf("%2.2X        ", program_data[pc_ptr]);
			printf("CLV        ");
			pc_ptr++;			// increment the PC by one
			break;

		case (BYTE)0xB9:		// LDA Q,Y- load A absolute Y
			printf("%2.2X %2.2X %2.2X  ", program_data[pc_ptr], program_data[pc_ptr+1], program_data[pc_ptr+2]);
			printf("LDA $%2.2X%2.2X,Y", program_data[pc_ptr+2],program_data[pc_ptr+1]);
			pc_ptr += 3;
			break;

		case (BYTE)0xBA:		// TSX - transfer S to X
			printf("%2.2X        ", program_data[pc_ptr]);
			printf("TSX        ");
			pc_ptr++;			// increment the PC by one
			break;

		case (BYTE)0xBD:		// LDA Q,X- load Acc absolute X
			printf("%2.2X %2.2X %2.2X  ", program_data[pc_ptr], program_data[pc_ptr+1], program_data[pc_ptr+2]);
			printf("LDA $%2.2X%2.2X,X", program_data[pc_ptr+2],program_data[pc_ptr+1]);
			pc_ptr += 3;
			break;

		case (BYTE)0xC0:		// CPY #n - compare mem and Y immediate
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("CPY #$%2.2X  ", program_data[pc_ptr+1]);
			pc_ptr += 2;
			break;

		case (BYTE)0xC6:		// DEC - decrement memory by one Zero Page
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("DEC ($%2.2X) ", program_data[pc_ptr+1]);
			pc_ptr += 2;
			break;

		case (BYTE)0xC8:		// INY - increment Y
			printf("%2.2X        ", program_data[pc_ptr]);
			printf("INY        ");
			pc_ptr++;			// increment the PC by one
			break;

		case (BYTE)0xC9:		// CMP #n - compare mem and Acc Absolute
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("CMP #$%2.2X  ", program_data[pc_ptr+1]);
			pc_ptr += 2;
			break;

		case (BYTE)0xCA:		// DEX - decrement X	
			printf("%2.2X        ", program_data[pc_ptr]);
			printf("DEX        ");
			pc_ptr ++;			// one byte;
			break;

		case (BYTE)0xD0:		// BNE - branch if not equal		
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			pc_ptr += 2;
			temp_byte = program_data[pc_ptr-1];
			if (temp_byte > 0x7f)	// then it is a negative number
			{
				temp_byte = (0xff - temp_byte)+1;
				temp_word = (pc+pc_ptr) - temp_byte;
			}
			else 
				temp_word = pc+pc_ptr+temp_byte;
			printf("BNE $%4.4X  ", temp_word);// only one byte opcode			
			break;

		case (BYTE)0xD6:		// DEC Z,X - decrement memory zero page X
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("DEC ($%2.2X),X", program_data[pc_ptr+1]);
			pc_ptr += 2;
			break;

		case (BYTE)0xD8:		// CLD - clear decimal mode
			printf("%2.2X        ", program_data[pc_ptr]);
			printf("CLD        ");
			pc_ptr++;
			break;

		case (BYTE)0xE0:		// CPX #n - compare mem and X immediate
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("CPX #$%2.2X   ", program_data[pc_ptr+1]);
			pc_ptr += 2;
			break;

		case (BYTE)0xE6:		// INC Z - increment memory by one Zero Page
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("INC ($%2.2X)  ", program_data[pc_ptr+1]);
			pc_ptr += 2;
			break;

		case (BYTE)0xE8:		// INX - increment X
			printf("%2.2X        ", program_data[pc_ptr]);
			printf("INX        ");
			pc_ptr++;
			break;

		case (BYTE)0xE9:		// SBC #n - Subtract acc with carry
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);
			printf("SBC #$%2.2X   ", program_data[pc_ptr+1]);
			pc_ptr += 2;
			break;

		case (BYTE)0xEA:		// NOP - no operation
			printf("%2.2X        ", program_data[pc_ptr]);
			printf("NOP        ");
			pc_ptr++;
			break;

		case (BYTE)0xF0:		// BEQ - branch if equal		
			printf("%2.2X %2.2X     ", program_data[pc_ptr], program_data[pc_ptr+1]);			
			pc_ptr += 2;					// only one byte opcode
			temp_byte = program_data[pc_ptr-1];
			if (temp_byte > 0x7f)	// then it is a negative number
			{
				temp_byte = (0xff - temp_byte)+1;
				temp_word = (pc+pc_ptr) - temp_byte;
			}
			else 
				temp_word = pc+pc_ptr+temp_byte;
			printf("BEQ $%4.4X  ", temp_word);
			break;


		default:
			printf("%2.2X        ", program_data[pc_ptr]);
			printf("UNKNOWN    ");
			pc_ptr++;
			break;
		
	}

}


	
void redraw_static()
// redraws the parts of the debugger screen that don't change
// this should be done as little as possible since printf() is really slow
{
	gotoxy(0,0);
	printf("|------------------------------------------------------------------------------|");
	gotoxy(0,1);
	printf("|                                                                              |");
	gotoxy(0,2);
	printf("|                                                                              |");
	gotoxy(0,3);
	printf("|-----------------------------CPU----------------------------------PPU---------|");				
	gotoxy(0,4);
	printf("|                                                        |                     |");
	gotoxy(0,5);
	printf("|                                                        |                     |");
	gotoxy(0,6);
	printf("|                                                        |                     |");
	gotoxy(0,7);
	printf("|                                                        |                     |");
	gotoxy(0,8);
	printf("|                                                        |                     |");
	gotoxy(0,9);
	printf("|--------------------------------------------------------|                     |");
	gotoxy(0,10);
	printf("|                                                        |                     |");
	gotoxy(0,11);
	printf("|                                                        |                     |");
	gotoxy(0,12);
	printf("|                                                        |                     |");
	gotoxy(0,13);
	printf("|                                                        |                     |");
	gotoxy(0,14);
	printf("|                                                        |                     |");
	gotoxy(0,15);
	printf("|                                                        |                     |");
	gotoxy(0,16);
	printf("|                                                        |                     |");
	gotoxy(0,17);
	printf("|                                                        |                     |");
	gotoxy(0,18);
	printf("|                                                        |                     |");
	gotoxy(0,19);
	printf("|                                                        |                     |");
	gotoxy(0,20);
	printf("|                                                        |                     |");
	gotoxy(0,21);
	printf("|                                                        |                     |");
	gotoxy(0,22);
	printf("|                                                        |                     |");	
	gotoxy(0,23);
	printf("--------------------------------------------------------------------------------");
}