/*
Copyright (C) 2000 Chris Teague

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
*/

// emuschool.cpp : Defines the class behaviors for the application.
//

#include "stdafx.h"
#include "emuschool.h"
#include "data_types.h"

#include "MainFrm.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CEmuschoolApp

BEGIN_MESSAGE_MAP(CEmuschoolApp, CWinApp)
	//{{AFX_MSG_MAP(CEmuschoolApp)
	ON_COMMAND(ID_APP_ABOUT, OnAppAbout)
	ON_COMMAND(ID_FILE_OPEN, OnFileOpen)
	ON_COMMAND(ID_ALT_ENTER, OnAltEnter)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()
/////////////////////////////////////////////////////////////////////////////
#define HBLANK 115			// number of clock cycles to render one line
#define SCANLINES 240		// number of lines
#define VPERIOD 2875		// number of cycles a VBLANK lasts after it starts

#define _RGB16BIT(r,g,b) ((b%32) + ((g%32) << 5) + ((r%32) << 10))

void read_rom_header(CString filename, ROM_header &temp_header);
void read_ROM(CString filename, long int prg_size, generic_ram* rom, long int chr_size, generic_ram* vrom, BYTE trainer);

// CEmuschoolApp construction

CEmuschoolApp::CEmuschoolApp()
{
	// TODO: add construction code here,
	// Place all significant initialization in InitInstance
	emulator_running = 0x00;		// 0= false or not running
}

/////////////////////////////////////////////////////////////////////////////
// The one and only CEmuschoolApp object

CEmuschoolApp theApp;

// global objects

CMainFrame*		g_pMainWnd;			// main application window

/////////////////////////////////////////////////////////////////////////////
// CEmuschoolApp initialization

BOOL CEmuschoolApp::InitInstance()
{
	AfxEnableControlContainer();

	// Standard initialization
	// If you are not using these features and wish to reduce the size
	//  of your final executable, you should remove from the following
	//  the specific initialization routines you do not need.

#ifdef _AFXDLL
	Enable3dControls();			// Call this when using MFC in a shared DLL
#else
	Enable3dControlsStatic();	// Call this when linking to MFC statically
#endif


	// Change the registry key under which our settings are stored.
	// TODO: You should modify this string to be something appropriate
	// such as the name of your company or organization.
	SetRegistryKey(_T("Local AppWizard-Generated Applications"));


	// To create the main window, this code creates a new frame window
	// object and then sets it as the application's main window object.

	CMainFrame* pFrame = new CMainFrame;
	m_pMainWnd = pFrame;
	g_pMainWnd = pFrame;

	// create and load the frame with its resources

	pFrame->LoadFrame(IDR_MAINFRAME,
		WS_OVERLAPPEDWINDOW | FWS_ADDTOTITLE, NULL,
		NULL);




	// The one and only window has been initialized, so show and update it.
	pFrame->ShowWindow(SW_SHOW);
	pFrame->UpdateWindow();		

	return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
// CEmuschoolApp message handlers





/////////////////////////////////////////////////////////////////////////////
// CAboutDlg dialog used for App About

class CAboutDlg : public CDialog
{
public:
	CAboutDlg();

// Dialog Data
	//{{AFX_DATA(CAboutDlg)
	enum { IDD = IDD_ABOUTBOX };
	//}}AFX_DATA

	// ClassWizard generated virtual function overrides
	//{{AFX_VIRTUAL(CAboutDlg)
	protected:
	virtual void DoDataExchange(CDataExchange* pDX);    // DDX/DDV support
	//}}AFX_VIRTUAL

// Implementation
protected:
	//{{AFX_MSG(CAboutDlg)
		// No message handlers
	//}}AFX_MSG
	DECLARE_MESSAGE_MAP()
};

CAboutDlg::CAboutDlg() : CDialog(CAboutDlg::IDD)
{
	//{{AFX_DATA_INIT(CAboutDlg)
	//}}AFX_DATA_INIT
}

void CAboutDlg::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CAboutDlg)
	//}}AFX_DATA_MAP
}

BEGIN_MESSAGE_MAP(CAboutDlg, CDialog)
	//{{AFX_MSG_MAP(CAboutDlg)
		// No message handlers
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

// App command to run the dialog
void CEmuschoolApp::OnAppAbout()
{
	CAboutDlg aboutDlg;
	aboutDlg.DoModal();
}

/////////////////////////////////////////////////////////////////////////////
// CEmuschoolApp message handlers


void CEmuschoolApp::OnFileOpen() 
// This is called when the user selects File -> Open
{
	

	char szFilter[]="NES Roms(*.nes)|*.nes||";
	CFileDialog FileDlg(TRUE, NULL, NULL, OFN_HIDEREADONLY, szFilter);

	if (FileDlg.DoModal() == IDOK)
	{		
		read_rom_header(FileDlg.GetPathName(), my_header);		// read in first 16 bytes of ROM file
		my_prg_rom.set_size(my_header.prg_size);			// create the program rom		
		my_chr_vrom.set_size(my_header.chr_size);	// create chr VROM device
		my_sram.set_size(my_header.battery_backup*0x2000);
		my_ram.set_size(0x800);			// 2kb for internal ram
		my_vram.set_size(0x800);			// 2kb vram for ppu
		read_ROM(FileDlg.GetPathName(), my_header.prg_size, &my_prg_rom, my_header.chr_size, &my_chr_vrom, my_header.trainer);				
		switch(my_header.mapper)
		{
			case 0:
				my_mapper = new mapper0(&my_prg_rom, &my_chr_vrom, &my_sram, my_header.prg_size);	// allocate memory for mapper
				
				break;

			default:
				AfxMessageBox("Unsupported Mapper");
				exit(1);
				break;
		}
		my_ppu.set_values(&my_ram, &my_vram, my_header.mirroring, my_mapper, &my_screen);					// create ppu 
		my_bus.set_values(&my_ram, my_mapper, &my_ppu, &my_gamepad1, &my_gamepad2);	// create memory bus
		my_cpu.set_bus(&my_bus);			// create cpu
		clock_cycles = 0;			// reset number of clocks
		#ifdef screen_on	
			if (my_screen.init() != DD_OK)		// initialize the Direct Draw object
				AfxMessageBox("screen not initialized");
		#endif
		scan_line = 0;				// start at top of screen
		my_cpu.reset();
		emulator_running = 0x01;	// 1 = running, start the emulation
		my_input.init();			// initialize the input object
		my_gamepad1.set_input_ptr(&my_input, 0);	// give ptr to input object to the gamepad object		
		my_gamepad2.set_input_ptr(&my_input, 1);	// give ptr to input object to the gamepad object
	}	
}


void read_rom_header(CString filename,ROM_header &temp_header)
// Opens the file specified in filename, and reads it's 16 byte iNES header
// Parameters: pointer to a ROM_header structure
// returns: header information in temp_header
{
	FILE *fp;
	BYTE header[16];	
	
	if (!(fp = fopen(filename, "rb")))
		AfxMessageBox("Unable to open ROM file\n");
	else
	{
		for (int i = 0; i <= 15; i++)	
		{
			header [i] = fgetc(fp);		// read in the header (first 16 bytes)	
		}
		fclose(fp);				// close the ROM file		

		// make sure the first 4 bytes are 'NES'+0x1a
		if ( (header[0] != 'N') || (header[1] != 'E') || (header[2] != 'S') || (header[3] != 0x1a) )
			AfxMessageBox("Not a valid NES ROM file\n");
		else
		{
			temp_header.prg_size = header[4]*0x4000;		// 16 KB pages for prg data
			if (temp_header.chr_size == 0)	// if there is not VROM
				temp_header.chr_size = 0x2000;	// create 8 KB of ram to use (not sure if this is right)
			else		// there is VROM
				temp_header.chr_size = header[5]*0x2000;		//  8 KB pages for chr data

			temp_header.mapper = header[6];
			temp_header.mapper >>= 4;					// shift upper bits into lower 4 bits
			temp_header.mapper &= 0x0F;					// clear out upper 4 bits
			header[7] &= 0xF0;				// make sure bottom is cleared
			temp_header.mapper |= header[7];			// combine the two bytes into one;

		// set the battery back up info
		if(header[6] & 0x02)
			temp_header.battery_backup=0x01;
		else
			temp_header.battery_backup=0x00;

		// check if trainer exists
		if(header[6] & 0x04)
			temp_header.trainer=0x01;
		else
			temp_header.trainer=0x00;

		// check for 4 screen mirroring
		if(header[6] & 0x08)
		{
			temp_header.four_screen_mirror=0x01;
			TRACE("Unsupported four screen miroring detected\n");
		}
		else 
			temp_header.four_screen_mirror=0x00;

		if (header[6] & 0x01)
			temp_header.mirroring= 0x01;	// Vertical mirroring
		else
			temp_header.mirroring= 0x00;	// Horizontal mirroring			
		}		
	}	
}

void read_ROM(CString filename, long int prg_size, generic_ram* rom, long int chr_size, generic_ram* vrom, BYTE trainer)
// reads in the program data from the ROM file
{
	FILE *fp;			// file pointer	

	if (!(fp = fopen(filename, "rb")))
		AfxMessageBox("Unable to open ROM file\n");
	else
	{
		fseek(fp,16,SEEK_SET);			// seek past the header
		if (trainer)
			fseek(fp, 512, SEEK_CUR);		// if a trainer exists, ignore it.
		
		for (int i = 0; i< prg_size; i++)	// read in program data
			rom->write_byte(i, fgetc(fp));

		for (i =0; i < chr_size; i++)		// read in video data					
			vrom->write_byte(i, fgetc(fp));			
		

		fclose(fp);			// close the ROM file	

	}

}

int CEmuschoolApp::Run() 
{
	// TODO: Add your specialized code here and/or call the base class	
	while(TRUE)
	{
		MSG msg;
		if(PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE))
		{
			if(!GetMessage(&msg, NULL, 0,0))
			{
				break;
			}
			CMainFrame* pFrame = (CMainFrame*)m_pMainWnd;
			if(!TranslateAccelerator(m_pMainWnd->GetSafeHwnd(),pFrame->GetAccelTable(), &msg))
			{				
				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}
		}				

		// game code here
		if(emulator_running)
		{
			int i=0;

			my_input.read_keyboard_state();
			if (my_input.key_pressed(ESC))	// exit if escape pressed
			{
				my_input.release_all();		// cleans up the DirectInput object
				// there are still memory leaks, this is not a good way to exit
				AfxPostQuitMessage(0);		
			}
			else if (my_input.key_pressed(F1))	// save screenshot if F1 pressed
			{
				my_screen.screen_save();
			}
			
			emulator_running = my_cpu.execute_instruction(clock_cycles);
			if (clock_cycles >= HBLANK)
			{
				clock_cycles -= HBLANK;
				//if (scan_line < 240)
				//	my_ppu.render_scanline(scan_line);	// draw this scan line to back buffer
				scan_line++;					// go to next scan line
				if (scan_line == SCANLINES)		// we did the last one already
				{
					my_ppu.start_vblank();	// a vblank should be performed			
					if ( my_ppu.read_byte(0x2000) & 0x80)
						my_cpu.vblank_interrupt(clock_cycles);	// execute an interrupt
				}
				if (scan_line == SCANLINES + VPERIOD/HBLANK)	
				{			
					scan_line = 0;			// go back to top of screen.
					my_ppu.end_vblank();	// end of vblank 
				}

			}
		}
	}
	return ExitInstance();		
}




int CEmuschoolApp::ExitInstance() 
{
	// TODO: Add your specialized code here and/or call the base class	

	delete my_mapper;			// deallocate memory

	return CWinApp::ExitInstance();
}

void CEmuschoolApp::OnAltEnter() 
// Switch screen modes
{
	#ifdef screen_on
	my_screen.ChangeCoopLevel();	
	#endif
}

