// Millipede Emulator 0.2
// ----------------------
// by Ivan Mackintosh email: ivan@rcp.co.uk
//
//
// Still to be completed:
// ----------------------
// Colour Palettes
// Sound
// A way of controlling the game speed automatically
//
// Thanks to:
// ----------
// Allard van der Bas - providing the repository
// Neil Bradley for supplying 6502 code



#include <stdio.h>
#include <stdlib.h>
#include <varargs.h>
#include <conio.h>
#include <ctype.h>
#include <pc.h>
#include <allegro.h>  
#include "6502.h"
#include "dipswtch.h"

/* prototypes */
void DisplayMessage(void);
int LoadRom(BYTE * pAddress, char * Name, size_t Size);
void DisplayCharSet(void);
void DrawSingleChar(int CharNo, int x, int y, int palette);
void DrawSprite(int Offset, int x, int y);
int GetBit(int);
void setupvideo(void);
void WarpChars(void);
void RedrawScreen(void);
void LoadStatus(void);
void ReadKeyboard(void);
void SaveStatus(void);
void DipSwitches(void);


#define SAVEFILE	"milliped.sav"
#define CODE_ROM1	"136013.104"
#define CODE_ROM2	"136013.103"
#define CODE_ROM3	"136013.102"
#define CODE_ROM4	"136013.101"
#define GRPX_ROM1	"136013.106"
#define GRPX_ROM2	"136013.107"

#define CODE_START	(0x4000)
#define	HISCORE_BYTES	(0x2F)
#define RAM_HISCORES	(0x64)
#define	ROM_HISCORES	(0x788E)
#define DIPSW1		(0x408)
#define DIPSW2		(0x808)
#define DIPSW3    (0x2000)
#define SPRITE_OFFSET	(0x13C0)
#define SPRITE_XPOS	(0x13D0)
#define SPRITE_YPOS	(0x13E0)
#define SPRITE_COL	(0x13F0)
#define SCREEN_MEM	(0x1000)


int addrmask = 0x7fff;
BYTE * gameImage = NULL;
BYTE CharSet[4096];
BYTE WarpCharSet[16384];
BITMAP *piccie;
BYTE pal[768];
WORD ExitAddress;
extern volatile char key[128];


int DisplayChars = 0;
int nojoy = 0;
int CPURunning = 0;
int InstructionsPerUserInput = 4000;
int CyclesPerInterrupt = 10000;


int main(int argc, char * argv[])
{
   int i;
   int EditDS = 0;
   
   DisplayMessage();
   
   // parse command line
   for (i = 1; i < argc; i ++)
   {
      if ((argv[i][0] == '-') || (argv[i][0] == '/'))
      {
	 if (strcmp(&argv[i][1], "chars") == 0)
	    DisplayChars = 1;

	 else if(strcmp(&argv[i][1], "nojoy") == 0)
	    nojoy = 1;
	 
	 else if (strcmp(&argv[i][1], "switch") == 0)
	    EditDS = 1;

    else if (strcmp(&argv[i][1], "cycles") == 0)
       CyclesPerInterrupt = atoi(argv[i + 1]);

    else if (strcmp(&argv[i][1], "input") == 0)
       InstructionsPerUserInput = atoi(argv[i + 1]);
 
      }
   }

   /* create memory map */
   if ((gameImage = (BYTE *)malloc(0x8000)) == NULL)
   { printf("Memory Allocation failed\n"); exit(1); }

   // clear memory
   for (i = 0; i < 0x4000; i ++)
      gameImage[i] = 0;
   
   /* load roms */
   printf("Loading Roms...\n");
   if (!LoadRom(&gameImage[CODE_START], CODE_ROM1, 0x1000)) 
      { free (gameImage); exit(1); }
   if (!LoadRom(&gameImage[CODE_START + 0x1000], CODE_ROM2, 0x1000))
      { free (gameImage); exit(1); }
   if (!LoadRom(&gameImage[CODE_START + 0x2000], CODE_ROM3, 0x1000))
      { free (gameImage); exit(1); }
   if (!LoadRom(&gameImage[CODE_START + 0x3000], CODE_ROM4, 0x1000))
      { free (gameImage); exit(1); }
   if (!LoadRom(&CharSet[0], GRPX_ROM1, 0x800))
      { free (gameImage); exit(1); }
   if (!LoadRom(&CharSet[0x800], GRPX_ROM2, 0x800))
      { free (gameImage); exit(1); }
   
   WarpChars();

   init6502(CyclesPerInterrupt);
   reset6502();


   // detect joystick
   if (!nojoy) 
   {
      int det = initialise_joystick();
      
      if (det == 0)
	 printf("Joystick detected\n");
      else
	 nojoy = 1;
   }


   install_keyboard();
   
   printf("\nPress a key to start\n");
   readkey();
   
   setupvideo();

   // if -chars switch is set then display chars and exit
   if (DisplayChars)
   {
      DisplayCharSet();
      readkey();
      set_gfx_mode(GFX_TEXT,80,25,0,0);
      destroy_bitmap(piccie);
      allegro_exit();
      free(gameImage);
      exit(1);
   }


   LoadStatus();


   // if -switch param display edit dip switch menu
   if (EditDS) DipSwitches();
   
   CPURunning = 1;
   do
   {
      ExitAddress = exec6502(InstructionsPerUserInput);
      ReadKeyboard();
   } 
   while(CPURunning);


   // save highscores + dipswitches
   SaveStatus();
         
   free (gameImage);
   destroy_bitmap(piccie);
   set_gfx_mode(GFX_TEXT,80,25,0,0);
   allegro_exit();
   printf("Exit Address = %x\n", ExitAddress);
   
   exit(0);
}




int LoadRom(BYTE * pAddress, char * Name, size_t Size)
{
   FILE * fp;

   if ((fp = fopen(Name, "rb")) == NULL)
   {
      printf("Failed to open %s\n", Name);
      return 0;
   }

   if (fread(pAddress, 1, Size, fp) != Size)
   {
      printf("Failed to read %s\n", Name);
      free(gameImage);
      fclose(fp);
      return 0;
   }

   fclose(fp);
   return 1;
}






void DrawSingleChar(int CharNo, int x, int y, int palette)
{
   int cx, cy;
   int Start = CharNo * 64;
   
   for (cy = 0; cy < 8; cy ++)
   {
      for (cx = 0; cx < 8; cx ++)
      {
	 piccie->line[(y * 8) + cy][(x * 8) + cx] = 
	    (BYTE)(WarpCharSet[Start] | (palette << 2));
	 Start++;
      }
   }
}


void DrawSprite(int Offset, int x, int y)
{
   int cx, cy;
   int Start;
   int SpOffset = Offset & 0x3E;
   int Flip = 1;
   Start = SpOffset * 64;
   
   // use second set of sprite data?
   if ((Offset & 0x01) == 0x01)
      Start += 8192; 

   // flip sprite horizontally?
   if ((Offset & 0x80) == 0x80)
      Flip = 0;
   
   for (cy = 7; cy >=0; cy--)
   {
      for (cx = 0; cx < 8; cx++)
      {
	 BYTE ypos = 255-(y+cy);
	 
	 if (!Flip)
	 {
	    BYTE xpos = 238 - (x - (8 - cx));

	    if ((WarpCharSet[Start] != 0) && ((BYTE)(xpos+8) < 0xf0))
	       piccie->line[ypos][(BYTE)(xpos+8)] = WarpCharSet[Start];

	    if ((WarpCharSet[Start+64] != 0) && (xpos < 0xf0))
	       piccie->line[ypos][xpos] = WarpCharSet[Start+64];
	 }
	 else
	 {
	    BYTE xpos = 238 - (x - cx);
	    
	    if ((WarpCharSet[Start] != 0) && (xpos < 0xf0))
	       piccie->line[ypos][xpos] = WarpCharSet[Start];

	    if ((WarpCharSet[Start+64] != 0) && ((BYTE)(xpos+8) < 0xf0))
	       piccie->line[ypos][(BYTE)(xpos+8)] = WarpCharSet[Start+64];
	    
	 }
	 Start++;
      }
   }
}



void DisplayCharSet(void)
{
   int x, y;
   for (x = 0; x < 32; x ++)
   {
      for (y = 0; y < 8; y ++)
      {
	 DrawSingleChar((y * 32) + x, x, y, 0);	 
      }
   }

   blit(piccie, screen, 0,0, 0,0, 256,256);
}



void setcolor(char red, char green, char blue, unsigned short color)
{
   color = 3*color;
   pal[color++]=red;
   pal[color++]=green;
   pal[color]=blue;
}

void setpallete ()
{
   int count;

   outportb(0x3c8,0x00);
   for(count=0;count<768;count++)
      outportb(0x3c9,pal[count]);
}

void setupvideo ()
{
   WORD teller;
   WORD count1,count2;

   set_gfx_mode(GFX_MODEX,256,256,0,0);
   for (teller=0;teller<256;teller++)
      setcolor(0,0,0,teller);

   setcolor(0,0,0,0);		// Black
   setcolor(0,63,0,1);		// Green
   setcolor(63,0,0,2);		// Red
   setcolor(63,63,63,3);	// White

   setcolor(0,0,0,4);		// Black
   setcolor(32,32,32,5);	// Blue
   setcolor(0,63,0,6);  	// Green
   setcolor(63,63,63,7);	// White

   setcolor(0,0,0,8);		// Black
   setcolor(0,63,63,9);		// Light Blue
   setcolor(0,63,0,10);		// Green
   setcolor(63,63,63,11);	// White

   piccie = create_bitmap(256,256);
   for (count1=0;count1<256;count1++)
      for (count2=0;count2<256;count2++)
         piccie->line[count1][count2] = 0;

   setpallete();
}




int GetBit(int startbit)
{
   int nbyte = startbit/8;
   int nbit = startbit%8;
   int mask = 0x80 >> nbit;
   
   return ((CharSet[nbyte] & mask) == mask);
}

void WarpChars(void)
{
   int x,cx,cy, bit;
   BYTE result = 0;
   int Count = 0;

   for (cy = 0; cy < 8; cy ++)
   {
      for (cx = 0; cx < 32; cx ++)
      {
         for (bit = 7; bit >= 0; bit --)
         {
            for (x = 0; x < 8; x ++)
            {
               result = 0;
               if (GetBit(bit + (x * 8) + (cx * 64) + (cy * 2048))) result = 1;
               if (GetBit(bit + (x * 8) + (cx * 64) + (cy * 2048) + 16384)) result ^= 2;
         
	       WarpCharSet[Count] = result;
	       Count++;
            }
         }
      }
   }
}


void RedrawScreen(void)
{
   int x, y;
   int count = 0;

   for (x = 0; x < 30; x++)
   {
      for (y = 31; y >= 0; y--)
      {
	 BYTE Char = gameImage[SCREEN_MEM + count] & 0x3F;
	 BYTE Add = 64;
	 
	 if (gameImage[SCREEN_MEM + count] & 0x40)
	    Add = 192;
	 
	 DrawSingleChar(Char + Add, x, y, 0);
	 count ++;
      }
   }

   // draw sprites
   for (x = 0; x < 16; x ++)
   {
      int xpos = gameImage[SPRITE_XPOS + x];
      int ypos = gameImage[SPRITE_YPOS + x];
      int offset = gameImage[SPRITE_OFFSET + x];

      if (ypos < 0xf8)
	 DrawSprite(offset, xpos, ypos);
   }

   blit(piccie, screen, 0,0, 0,0, 256,256);
}



void ReadKeyboard(void)
{
   if (key[KEY_ESC])
      CPURunning = 0;

   RedrawScreen();

   if (!nojoy)
      poll_joystick();
   
}



void M_WRMEM(WORD A,BYTE V)
{
   // protect dip switches 
   if ((A == 0x408) || (A == 0x808))
      return;
   
   if (A == 0x2600)
      clockticks6502 = 0;
   
   // don't allow code to overwrite itself
   if (A >= CODE_START)
      return;
   
   gameImage[A] = V;
}


BYTE M_RDMEM(WORD A)
{
   if (A == 0x2000)
   {
      // 0x2000 doubles as dipswitch
      BYTE Val = gameImage[A] & 0x0F; // read and mask lower nibble
      Val ^= 0xF0; // set upper nibble to all 1s
      
      if (key[KEY_1])
         Val &= ~0x20;  // 1 player start
      
      if (key[KEY_CONTROL])
         Val &= ~0x10;  // player fire (keyboard)
      
      if (!nojoy && joy_b1)
         Val &= ~0x10;  // player fire (joystick)
      
      return Val;
   }

   // random generators
   if ((A == 0x40a) || (A == 0x80a))
      return (BYTE)rand();
   
   if (A == 0x2001)
   {
      BYTE Val = 0xFF;
      
      if (key[KEY_2])
	 Val &= ~0x20;  // 1 player start
      
      if (key[KEY_CONTROL])
	 Val &= ~0x10;  // player fire (keyboard)

      if (!nojoy && joy_b1)
	 Val &= ~0x10;  // player fire (joystick)
      
      return Val;
   }
   
   if ((A == 0x2010) || (A == 0x2011))
   {
      BYTE Val = 0xFF;
      
      // player1 move (keyboard)
      if (key[KEY_UP])
	 Val &= ~0x08;
      if (key[KEY_DOWN])
	 Val &= ~0x04;
      if (key[KEY_LEFT])
	 Val &= ~0x02;
      if (key[KEY_RIGHT])
	 Val &= ~0x01;
      
      if (!nojoy)
      {
	 // player1 move (joystick)
	 if (joy_up)
	    Val &= ~0x08;
	 if (joy_down)
	    Val &= ~0x04;
	 if (joy_left)
	    Val &= ~0x02;
	 if (joy_right)
	    Val &= ~0x01;
      }
        
      if ((A == 0x2010) && (key[KEY_3]))
	 Val &= ~0x20;
      	
      return Val;
   }

      
   return(gameImage[(A & addrmask)]);
}



// If the status file centiped.sav exists then load the
// hiscore table and dip switch settings from it
void LoadStatus(void)
{
   FILE * fp;
   int i;
   
   // open save file if exists
   fp = fopen(SAVEFILE, "rb");
   if (fp == NULL) return;
   
   // load in high score over ROM high scores
   for (i = 0; i < HISCORE_BYTES; i ++)
      gameImage[ROM_HISCORES + i] = (BYTE)fgetc(fp);
   
   // load in dipswitch settings
   gameImage[DIPSW1] = (BYTE)fgetc(fp);
   gameImage[DIPSW2] = (BYTE)fgetc(fp);
   gameImage[DIPSW3] = (BYTE)fgetc(fp);
   
   fclose(fp);
}

// Saves the RAM based high scores and the dipswitch
// settings - called on emulator exit
void SaveStatus(void)
{
   FILE * fp;
   int i;
   
   // open/create save file
   fp = fopen(SAVEFILE, "wb");
   if (fp == NULL) return;
   
   // save the high scores from RAM
   for (i = 0; i < HISCORE_BYTES; i ++)
      fputc(gameImage[RAM_HISCORES + i], fp);
   
   // save dipswitch settings
   fputc(gameImage[DIPSW1], fp);
   fputc(gameImage[DIPSW2], fp);
   fputc(gameImage[DIPSW3], fp);
   
   fclose(fp);
}




void DisplayMessage(void)
{
   printf("\nPLEASE DO NOT DISTRIBUTE THE SOURCE FILES OR THE EXECUTABLE WITH"
      " ROM IMAGES.\nDOING SO WILL HARM FURTHER EMULATOR DEVELOPMENT AND WILL"
      " CONSIDERABLY ANNOY\nTHE RIGHTFUL COPYRIGHT HOLDERS OF THOSE ROM"
      " IMAGES AND CAN RESULT IN LEGAL\nACTION UNDERTAKEN BY EARLIER"
      " MENTIONED COPYRIGHT HOLDERS.\n\n\n");

   printf("Command line switches:\n");
   printf("-chars    Display character set\n");
   printf("-nojoy    Disable joystick support\n");
   printf("-switch   Display dip switch menu\n");
   printf("-cycles nnnnn   Specifies duration between IRQs - default 10000\n");
   printf("                Making this number small will speed up emulation\n");
   printf("-input nnnnn    Specifies duration between user inputs checks - default 4000\n");
   printf("                Making this number larger will speed up emulation\n\n");
   
   printf("Keys:\n");
   printf("Pressing 1 or 2 starts the game, 3 inserts a coin. Move around using the\n"
      "joystick or cursor keys and <ctrl> to fire. Pressing <esc> quits"
      " the emulator.\n\n");
}




// Clears the bitmap that represents the screen
void ClearBitmap(void)
{
   int x, y;
   for (x = 0; x < 256; x ++)
      for (y = 0; y < 256; y ++)
         piccie->line[y][x] = 0;
}



// Draws a given alphanumeric string using the centipede chars
// at a specified place on the screen.
void DrawString(int x, int y, char * Text, int TextLen, int Palette)
{
   int i;

   for (i = 0; i < TextLen; i ++)
   {
      if (Text[i] == ' ')
         DrawSingleChar(64, x + i, y, Palette);
      else if (isalpha(Text[i]))
         DrawSingleChar(Text[i], x + i , y, Palette);
      else if (isdigit(Text[i]))
         DrawSingleChar((Text[i] - '0') + 96, x + i, y, Palette);
   }
}





// DipSwitches
// -----------
// Decodes and displays the current dip switch setting and allows
// these setting to be modified. This function is invoked on start
// up if the -switch parameter is specified.
void DipSwitches(void)
{
   int i, t, ExitDS = 0;
   // debounce 
   int DBUp = 0, DBDown = 0, DBFire = 0;

   int Setting = 0;
   int CurValues[NUM_SWITCHES];
   
   // don't do anything until the user lets go of <return>
   // otherwise the menu will exit straight away
   while (key[KEY_SPACE])
      ;
   
   ClearBitmap();
   DrawString(6, 2, "DIP SWITCH SETTINGS", 19, 1);

   // dipswitch settings
   for (i = 0; i < NUM_SWITCHES; i ++)
   {
      DrawString(5, 7 + (i * 2), Dipswitch[i].Heading, HEADING_SIZE, 
      (Setting == i)?2:0);

      for (t = 0; t < Dipswitch[i].NumOptions; t ++)
      {
         if ((gameImage[Dipswitch[i].Switch] & ~Dipswitch[i].Mask) ==
	    Dipswitch[i].Value[t])
	    break;
      }

      // if not found set default 
      if (t == Dipswitch[i].NumOptions)
         t = Dipswitch[i].Default; 
      DrawString(6 + HEADING_SIZE, 7 + (i * 2), Dipswitch[i].Options[t],
		 OPTION_SIZE, (Setting == i)?2:0);

      CurValues[i] = t;
      gameImage[Dipswitch[i].Switch] &= Dipswitch[i].Mask;
      gameImage[Dipswitch[i].Switch] |= Dipswitch[i].Value[t];
   }
   
   DrawString(1, 25, "USE UP DOWN AND FIRE TO CYCLE", 30, 1);
   DrawString(2, 26, "THROUGH DIP SWITCH SETTINGS", 27, 1);
   DrawString(0, 29, "PRESS SPACE TO START MILLIPEDE", 31, 1);
   
   blit(piccie, screen, 0,0, 0,0, 256,256);

   while(!ExitDS)
   {
      if (key[KEY_UP] && (Setting > 0) && !DBUp)
      {
         // remove old highlight
	 
	 DrawString(5, 7 + (Setting * 2), Dipswitch[Setting].Heading, 
	    HEADING_SIZE, 0);
	 DrawString(6 + HEADING_SIZE, 7 + (Setting * 2),
	    Dipswitch[Setting].Options[CurValues[Setting]], OPTION_SIZE, 0);

	 Setting --;

	 // add new highlight
	 DrawString(5, 7 + (Setting * 2), Dipswitch[Setting].Heading, 
	    HEADING_SIZE, 2);
	 DrawString(6 + HEADING_SIZE, 7 + (Setting * 2),
	    Dipswitch[Setting].Options[CurValues[Setting]], OPTION_SIZE, 2);

         DBUp = 1;
         blit(piccie, screen, 0,0, 0,0, 256,256);
      }
      if (key[KEY_DOWN] && (Setting < NUM_SWITCHES - 1) && !DBDown)
      {
         // remove old highlight
	 DrawString(5, 7 + (Setting * 2), Dipswitch[Setting].Heading, 
	    HEADING_SIZE, 0);
	 DrawString(6 + HEADING_SIZE, 7 + (Setting * 2),
	    Dipswitch[Setting].Options[CurValues[Setting]], OPTION_SIZE, 0);

	 Setting ++;

	 // add new highlight
	 DrawString(5, 7 + (Setting * 2), Dipswitch[Setting].Heading, 
	    HEADING_SIZE, 2);
	 DrawString(6 + HEADING_SIZE, 7 + (Setting * 2),
	    Dipswitch[Setting].Options[CurValues[Setting]], OPTION_SIZE, 2);

	 DBDown = 1;
         blit(piccie, screen, 0,0, 0,0, 256,256);
      }
      if (key[KEY_CONTROL] && !DBFire)
      {
         CurValues[Setting]++;
         if (CurValues[Setting] == Dipswitch[Setting].NumOptions)
            CurValues[Setting] = 0;

         gameImage[Dipswitch[Setting].Switch] &= Dipswitch[Setting].Mask;
         gameImage[Dipswitch[Setting].Switch] |= Dipswitch[Setting].Value[CurValues[Setting]];
         
         DBFire = 1;
         DrawString(6 + HEADING_SIZE, 7 + (Setting * 2),
                    Dipswitch[Setting].Options[CurValues[Setting]], OPTION_SIZE, 2);
         blit(piccie, screen, 0,0, 0,0, 256,256);
      }
      if (key[KEY_SPACE])
         ExitDS = 1;

      // debounce reset
      if (!key[KEY_UP]) DBUp = 0;
      if (!key[KEY_DOWN]) DBDown = 0;
      if (!key[KEY_CONTROL]) DBFire = 0;
   }
}

