;---------------------------------------------------------------------------------------------
;
; CLEAR 39999 BEFORE LAUNCHING THIS ROUTINE.
;
; This routine will program into ROM BANK 2 AND 3 a modified ZX-Spectrum internal ROM which jumps
; into the BOOTROM bank on NMI. The BOOTROM bank we jump into is RAM BANK 31.
; The BOOTROM resides into ROM BANK 0 and is started at power-on. It will copy itself to
; RAM BANK 31.
;
; This ROM should be selected by user in place of the internal basic rom, when serial
; communication over NMI is required during normal operation.
;
; Four RAM banks (64K) are destroyed (beginning from the one in the RAMBANK EQU).
; The zx-spectrum upper 16K are destroyed as well (needs CLEAR 39999)
;
; NOTE: IT SHOULD BE USED ON A 16/48K MACHINE because the internal rom is used as a starting point.
;
;
;
; V2.01 11/01/2008
; Bytes at address 6E, 6F are no longer modified in the patched ROM. In previous version, these
; bytes was changed to NOPs in order to balance other changes and obtain a NMI handler which
; does not hang if ran without a zxmmc+ installed (in such a situation, the fastpaging does not
; take place).
;
; However, a program could call the 006F JP(HL) instruction to obtain a CALL(HL). With this new
; version, the trick works again.
;
;
;
; V2.00 12/06/2007
; MODIFIED NMI patch, which SAVES FASTPAGE content on the stack, while mantaining
; a size small enough to avoid any need to place code somewhere else than in the $66 vector.
;
;


OUT_PORT equ	$1F		; SD chip select and FLASH WR enable register

FASTPAGE equ	$7F		; RAM/ROM fast paging register
;
; D7 set = RAM WR enable (standalone)
; D6 set = PAGE IN active
; D5 '0' = ram is paged; '1' = rom is paged
; D4:D0 = bank number


BORDER		equ	$FE

SCRATCH		equ	$C000	; 16K buffer used as scratchpad in zx-spectrum standard memory
RAMBANK		equ	27	; first of 4 16K banks in the zxmmc+ memory used as 64K backup for block erase

FLASHWR_CODE	equ	$A0	; writing this code to the OUT_PORT register will enable ROM CS on Z80 WR cycles
OUT_IDLE	equ	$F7	; acceptable idle state for OUT_PORT register


        org 42000

fastp_save	equ	$
		org $+1
flags		equ	$
		org $+1
ram_bank	equ	$
		org $+1
rom_bank	equ	$
		org $+1


	org 40000

	di
	in a,(FASTPAGE)
	ld (fastp_save),a	; FASTPAGE content is saved

	ld a,6
	out (BORDER),a
	call copy_2ram		; reads 64K bytes from ROM banks 0-3 to RAM

	ld a,5
	out (BORDER),a
	ld a,2			; creates RAM BANK 2
	call sinclair
	ld a,3			; creates RAM BANK 3
	call sinclair

	ld a,4
	out (BORDER),a
	call erase

	ld a,3
	out (BORDER),a
	call program

	ld a,2
	out (BORDER),a
	call compare

	ld a,(fastp_save)
	out (FASTPAGE),a

	ei
	ret

;
;---------------------------------------------------------------------------------------------
; This subroutine will copy a given ROM block from ROM to RAM (64Kbytes).
; Returns: nothing
;---------------------------------------------------------------------------------------------
copy_2ram
	ld a,RAMBANK
	set 7,a			; WR enabled

	ld b,4			; number of 16K blocks
	ex af,af'
	ld a,$60		; ROM BANK
	ex af,af'

next_btoram
	ex af,af'
	out (FASTPAGE),a	; rom is selected
	inc a
	push bc

	ld hl,0			; copy to sinclair ram
	ld de,SCRATCH
	ld bc,16384
	ldir

	ex af,af'
	out (FASTPAGE),a	; RAM is selected in WR mode
	inc a

	ld hl,SCRATCH		; copy from sinclair ram to zxmmc+ ram
	ld de,0
	ld bc,16384
	ldir

	pop bc
	djnz next_btoram

	ret


;
;---------------------------------------------------------------------------------------------
; This subroutine will compare a given ROM block with RAM content (64Kbytes).
;
; Returns:
; 0 = Identical; 1 = Mismatch which can be programmed; 2 = mismatch which needs BLOCK ERASE.
;---------------------------------------------------------------------------------------------
compare
	xor a
	ld (flags),a		; return code = OK

	ld a,RAMBANK
	set 6,a			; RD enabled

	ex af,af'
	ld a,$60
	ex af,af'		; A' = rom bank

	ld b,4			; number of 16K blocks

next_bcompare
	ex af,af'
	out (FASTPAGE),a	; rom is selected
	inc a
	push bc

	ld hl,0
	ld de,SCRATCH
	ld bc,16384
	ldir

	ex af,af'
	out (FASTPAGE),a	; RAM is selected in RD mode
	inc a

	push af			; COMPARE LOOP
	ld hl,0			; PAGED-IN RAM pointer
	ld de,SCRATCH		; scratchram (ROM CONTENT) pointer
	ld bc,16384
l_compare
	ld a,(de)		; this is ROM content byte, which was copied to sinclair RAM
	cp (hl)			; compare with RAM buffer (paged-in)
	jr z,byte_ok
	and (hl)		; are there '0' bits which should be turned into '1'?
	cp (hl)
	jr z,programmable	; no, this byte can be programmed without erasing the 64K block
	ld a,2
	ld (flags),a		; 2 = ERASE is needed
	pop af			; can't be worse: exit immediately
	pop bc
	jr exit_compare
programmable
	ld a,1
	ld (flags),a		; 1 = ROM content mismatch, but still programmable
byte_ok
	inc hl
	inc de
	dec bc
	ld a,b
	or c
	jr nz,l_compare
	pop af

	pop bc
	djnz next_bcompare

exit_compare
	ld a,(flags)		; return code
	ld c,a
	ld b,0
	ret

;
;---------------------------------------------------------------------------------------------
; This subroutine will program a given ROM block (64Kbytes).
;---------------------------------------------------------------------------------------------
program
	exx
	push bc
	push de
	push hl
	ld hl,$555		; quick access to program codes
	ld bc,$2AA
	ld e,$AA
	exx

	ld a,$60		; ROM page in
	ld (rom_bank),a

	ld a,RAMBANK		; first bank of ram to be programmed
	set 6,a			; RAM RD enabled
	ld (ram_bank),a

	ld a,FLASHWR_CODE	; this code will enable the ROM CS on Z80 write cycles
	out (OUT_PORT),a

next_bank
	ld a,(ram_bank)
	out (FASTPAGE),a
	inc a
	ld (ram_bank),a

	ld hl,0			; reads the current 16K ram buffer into scratchpad ram
	ld de,SCRATCH
	ld bc,16384
	ldir

	ld a,(rom_bank)
	out (FASTPAGE),a	; select ROM for read too (needed to check programmed byte)
	inc a
	ld (rom_bank),a

	ld hl,SCRATCH		; buffer to be programmed
	ld de,0			; A0:A13 address bit to ROM (16K) - remainder from PAGED-IN ROM BANK
	ld bc,16384

program_16k
	ld a,(de)		; current ROM content
	cp (hl)			; desired content
	jr z,no_program		; byte OK already

	exx
	ld a,$55
	ld (hl),e		; $555 <- $AA
	ld (bc),a		; $2AA <- $55
	ld (hl),$A0		; $555 <- $A0
	exx

	ld a,(hl)		; byte to program
	ld (de),a		; writes to ROM

	ex hl,de
	push bc			; wait for write completion
	ld bc,0			; timeout counter
l_waitprogram
	cp (hl)
	jr z,prog_do
	dec c
	jr nz,l_waitprogram
	dec b
	jr nz,l_waitprogram
	pop bc

	ld (hl),$F0		; RESET command to flash rom (clears write error and switch to read mode)

	ld bc,0			; datasheet says the chip needs at least 10us before any read attempt.
l_resetwait			; we wait quite more.
	dec bc
	ld a,b
	or c
	jr nz,l_resetwait

	ld bc,1			; program error
	jr program_exit

prog_do
	pop bc
	ex hl,de
no_program
	inc hl
	inc de
	dec bc
	ld a,b
	or c
	jr nz,program_16k

	ld a,(rom_bank)
	and 3			; "00" on LSB bits means 4 banks programmed
	jr nz,next_bank

	ld bc,0			; OK code
program_exit
	ld a,OUT_IDLE
	out (OUT_PORT),a	; disables ROM WR cycles

	exx
	pop hl
	pop de
	pop bc
	exx
	ret


;
;---------------------------------------------------------------------------------------------
; This subroutine will erase a given ROM block (64Kbytes).
;---------------------------------------------------------------------------------------------
erase
	ld a,$60		; ROM paged-in for read
	out (FASTPAGE),a

	ld a,FLASHWR_CODE	; this code will enable the ROM CS on Z80 write cycles
	out (OUT_PORT),a

	ld hl,$555
	ld bc,$2AA
	ld e,$AA
	ld a,$55

	ld (hl),e		; $555 <- $AA
	ld (bc),a		; $2AA <- $55
	ld (hl),$80		; $555 <- $80
	ld (hl),e		; $555 <- $AA
	ld (bc),a		; $2AA <- $55

	ld hl,0
	ld (hl),$30		; block erase command (address don't really care: only 3 MSB 64K-BLOCK bits)


	ld e,8			; about 6 seconds timeout
	ld bc,0
l_waiterase
	ld a,(hl)
	cp $ff
	jr z,erased_ok
	dec c
	jr nz,l_waiterase
	dec b
	jr nz,l_waiterase
	dec e
	jr nz,l_waiterase

	ld (hl),$F0		; RESET command to flash rom (clears write error and switch to read mode)

	ld bc,1			; timeout error
	jr erase_exit

erased_ok
	ld bc,0			; OK code

erase_exit
	ld a,OUT_IDLE
	out (OUT_PORT),a	; disables ROM WR cycles

	ret


;
;---------------------------------------------------------------------------------------------
; This subroutine will copy the sinclair ROM to the specified (0-3) programming ram buffer,
; changing the power-on logo and NMI handler.
;
; NMI Handler:
; Switch to the last (31) rom bank, where BOOTROM firmware resides. The BOOTROM content
; will begin execution from the point we switch between banks, at address $6E.
; HL must be saved by BOOTROM NMI handler.
;---------------------------------------------------------------------------------------------
sinclair
	add a,RAMBANK+128		; RAM bank offset + D7 set (WR enable)
	out (FASTPAGE),a

	ld hl,0
	ld de,0
	ld bc,16384
	ldir				; sinclair rom to ram copy

	ld hl,string			; new power-on logo
	ld de,$1552			; overwrite the 'Ltd' leading text
	ld bc,3
	ldir

	ld hl,nmi_block			; new NMI handler patch
	ld de,$66			; destination address
	ld bc,end_nmi-nmi_block
	ldir				; NMI handler is now patched

	ret

nmi_block
	push af				; 11T patch code for NMI handler
	in a,(FASTPAGE)			; 11T
	push af				; 11T fastpage content is saved as well
	ld a,$DF			; 7T RAM BANK 31 (bootrom) in RD/WR mode
	out (FASTPAGE),a		; 11T switch to BOOTROM BANK

					; 51T states to switch to BOOTROM BANK (29 in previous V1.00 version: +22).

;
; following code removed in version 2.01 for compatibility reason. This will remain unchanged in the patched rom.
;
;
;	nop				; this instruction is never executed: --> BOOTROM NMI handler (at $6E!), which
;	nop				; holds a JR (only first byte!) (operand = $D3 (the OUT that follows in the bootrom firmware))
;
;	pop af				; In the bootrom, here we have an OUT (FASTPAGE),A. It is the BOOTROM return point: $6F.
;					; 'A' should hold PATCHED ROM BANK NUMBER
;					; THIS POP AF IS NEVER EXECUTED! (but balances the second PUSH AF in case the rom was ran
;					; without the zxmmc+ hardware :-)
;
;
;					; after the OUT (FASTPAGE) is fetched from $6F address in the BOOTROM bank, here we come:
;	pop af				; this is same as original content, at address $71. THIS IS EXECUTED HERE.
;	retn				; NOTE: this allows a snapshot restore even with non-patched sinclair ROM.
end_nmi



string
        defb 'N','m','I'+128


        end 

