;---------------------------------------------------------------------------------------------
;
; ZXMMC+ nonvolatile RAM test.
; This program copies the whole 512K ram space to a (spare? 256MB offset!) region of the first
; found sd card, then compares it (the compare function should be used after powerdown/powerup).
; NOTE: NEW VERSION that will shift SD CARD address at +512K (an alternate save place) when
; location 23728 is > 0.
;
; The compare buffer is located at address 49152 ($C000).
; A CLEAR 29999 is needed.
;
; This is the assembly part used in the basic program "diagno", which can be snapped into FLASHROM
; using the NMI -> 'R' snapshot function provided by ZXMMC+ Bootrom Firmware then provided
; by default in new cards. The flashrom snapshot function uses the three last ROM banks
; 29,30,31 (each 16KB in size), for a complete 48K snap.
;
;
; 4 Functions are provided:
;
; 30.000:	Compare RAM against SD card. Returns number of mismatches (max 65535)
; 30.003:	Copy RAM to SD card
; 30.006:	Restore RAM from SD card
; 30.009:	TEST - COMPLEMENTS THE ENTIRE 512K RAM CONTENT
;
;
; The COMPARE function will exit after the first 4095 mismatches.
; For each mismatch, a 5 bytes record is saved starting from address 33000:
;
; 1 byte  RAM PAGE
; 2 bytes RAM address 0-3FFF
; 1 byte  RAM content
; 1 byte  CARD content
;
;
;   NOTE: before using this program (USR 30000 = COMPARE), make a CLEAR 29999.
;
;---------------------------------------------------------------------------------------------
;


; CPLD REGISTERS
;
SPI_PORT	equ $3F
OUT_PORT	equ $1F		; port for CS control (D1:D0)
FASTPAGE	equ $7F


MMC_0		equ $F6		; D0 LOW = SLOT0 active; D3 low = NMI disabled
MMC_1		equ $F5		; D1 LOW = SLOT1 active; D3 low = NMI disabled


BORDER		equ $FE


; SD/MMC RELATED DEFINES:

MMC_SNAP16	equ	1	; il cluster contiene uno snapshot da 16K

MMC_GO_IDLE_STATE	equ	$40
MMC_SEND_OP_COND	equ	$41
MMC_READ_CID		equ	$4A
MMC_SET_BLOCK_SIZE	equ	$50
MMC_READ_SINGLE_BLOCK	equ	$51
MMC_READ_MULTIPLE_BLOCK	equ	$52
MCC_TERMINATE_MULTI_READ equ	$4C
MMC_WRITE_SINGLE_BLOCK	equ	$58
MMC_WRITE_MULTIPLE_BLOCK equ	$59
MMC_STOP_TRAN		equ	$FD


BLOCKSIZE	equ	512	; SD/MMC block size (bytes)

FATSIZE		equ	8	; dimensioni FAT in termini di blocchi da 64KBytes (in pratica e` l'offset
				; da caricare nella word MSB dell'indirizzo della SD per accedere al primo
				; cluster di dati, subito oltre la FAT). 8 = 512KBytes (8192 entries da 64 bytes)


	org 32000		; variables
card_select	equ	$
	org $+1
found_cards	equ	$
	org $+1
fast_save	equ	$
	org $+1
c_result	equ	$
	org $+2
temp_ram	equ	$
	org $+512


MISMATCHES	equ	33000	; buffer for (max to) 4095 mismatches:
				; bank number, address 0-3FFF, nonvola value, card value



;********************************************************************************************************
;********************************************************************************************************

	org 30000

	jp compare		; COMPARE RAM data against CARD
	jp saveram		; WRITE RAM data to CARD
	jp restore		; RESTORE RAM data from CARD
	jp complement		; COMPLEMENTS the entire 512K RAM CONTENT


search_sd
	ld a,MMC_0		; SLOT 0 ENABLE (also disables NMI on RS-232 byte reception (D3 low))
	ld (card_select),a
	call sd_check		; controlla eventuale presenza di SD/MMM
	cp 0
	ret z			; card rilevata; rimane selezionata nel registro CARD_SELECT

	ld a,MMC_1		; SLOT 1 ENABLE
	ld (card_select),a
	call sd_check		; controlla eventuale presenza di SD/MMM
	ret			; rientra in ogni caso, segnalando l'eventuale presenza (Z)



saveram
	di

	in a,(FASTPAGE)		; FASTPAGE register is saved
	ld (fast_save),a

	call search_sd		; initializes and selects the first found card
	jr nz,no_card

	ld hl,$1000		; unit value = 64KBytes offset; 0x1000 = 256MBytes offset
	ld a,(23728)
	cp 0
	jr z,base_a		; if 23728 != 0, use an alternate buffer
	ld hl,$1008		; +512K
base_a
	ld de,0			; LSB part of the address in the card

	ld c,$40		; RAM_BANK = 0
l_128ksd
	ld a,c			; actual bank
	out (FASTPAGE),a
	ld b,16384/BLOCKSIZE
	ld ix,0			; read address
	call mmc_write_data	; saves this 16K bank to the SD CARD and increments the HL:DE pointer
	cp 0
	jr nz,l_128_error
	inc c
	ld a,c
	and 31			; 32 banks
	out (BORDER),a		; shows progress
	jr nz,l_128ksd
	ld a,7
	out (BORDER),a		; leaves a white border
	ld bc,0
	jr exit_ok
l_128_error
no_card
	ld bc,1
exit_ok
	ld a,(fast_save)	; FASTPAGE register is restored
	out (FASTPAGE),a
	ei
	ret



compare
	di
	ld hl,1			; error counter: 1 for 'card not found'
	ld (c_result),hl

	call search_sd		; initializes and selects the first found card
	jr z,card_found
	ld bc,(c_result)
	ei
	ret

card_found
	ld hl,0			; initializes the error counter
	ld (c_result),hl

	exx
	push bc
	push de
	push hl
	exx
	push iy			; mismatch pointer

	ld iy,MISMATCHES

	in a,(FASTPAGE)		; FASTPAGE register is saved
	ld (fast_save),a

	ld hl,$1000		; unit value = 64KBytes offset; 0x1000 = 256MBytes offset
	ld a,(23728)
	cp 0
	jr z,base_b		; if 23728 != 0, use an alternate buffer
	ld hl,$1008		; +512K
base_b
	ld de,0			; LSB part of the address in the card

	ld c,$40		; RAM_BANK = 0
l_128krd
	ld a,c			; actual bank
	out (FASTPAGE),a
	ld b,16384/BLOCKSIZE
	ld ix,49152		; read address
	call mmc_read_multidata	; reads 16K into spectrum ram
	cp 0
	jr nz,read_error

	exx
	ld hl,49152		; compare loop. 49152 ram holds CARD data
	ld de,0			; compare with 0 (nonvolatile ram)
	ld bc,16384		; n. of bytes to compare
l_compare
	ld a,(de)
	cp (hl)
	jr z,match		; equal
	push hl
	ld hl,(c_result)
	inc hl			; increments error counter
	ld a,h
	cp $10
	jr z,more_than4k	; max allowable is 4096 mismatches
	ld (c_result),hl
	in a,(FASTPAGE)
	and $1F
	ld (iy),a		; page number is saved
	ld (iy+1),d		; address 0-3FFF is saved
	ld (iy+2),e
	ld a,(de)
	ld (iy+3),a		; data from RAM

	ld hl,5
	ex hl,de
	add iy,de		; IX is incremented
	ex hl,de

	pop hl

	ld a,(hl)
	ld (iy-1),a		; data in the card

match
	inc hl
	inc de
	dec bc
	ld a,b
	or c
	jr nz,l_compare
	exx

	ld a,$40		; NOTE: MMC_READ_MULTIDATA DOES NOT INCREMENT THE HL:DE POINTER!
	add d			; adding $40 to D means 16KBytes displacement
	jr nc,no_flowhl
	inc hl			; increase MSB pointer
no_flowhl
	ld d,a

	inc c			; next bank
	ld a,c
	and 31			; 32 banks
	out (BORDER),a		; shows progress
	jr nz,l_128krd
	ld a,7
	out (BORDER),a		; leaves a white border
	jr exit_rdok

read_error
	ld hl,1
	ld (c_result),hl
	jr exit_rdok

more_than4k
	pop hl
exit_rdok
	ld a,(fast_save)	; FASTPAGE content is restored
	out (FASTPAGE),a

	pop ix
	exx
	pop hl
	pop de
	pop bc
	exx
no_card_cp
	ld bc,(c_result)
	ei
	ret



restore
	di

	in a,(FASTPAGE)		; FASTPAGE register is saved
	ld (fast_save),a

	call search_sd		; initializes and selects the first found card
	jr nz,no_card_c

	ld hl,$1000		; unit value = 64KBytes offset; 0x1000 = 256MBytes offset
	ld a,(23728)
	cp 0
	jr z,base_c		; if 23728 != 0, use an alternate buffer
	ld hl,$1008		; +512K
base_c
	ld de,0			; LSB part of the address in the card

	ld c,$80		; RAM_BANK = 0, WR enabled
l_128kre
	ld a,c			; actual bank
	out (FASTPAGE),a
	ld b,16384/BLOCKSIZE
	ld ix,0			; read address
	call mmc_read_multidata	; reads 16K into zxmmc+ RAM
	cp 0
	jr nz,read_errore

	ld a,$40		; NOTE: MMC_READ_MULTIDATA DOES NOT INCREMENT THE HL:DE POINTER!
	add d			; adding $40 to D means 16KBytes displacement
	jr nc,no_flowhlr
	inc hl			; increase MSB pointer
no_flowhlr
	ld d,a

	inc c			; next bank
	ld a,c
	and 31			; 32 banks
	out (BORDER),a		; shows progress
	jr nz,l_128kre
	ld a,7
	out (BORDER),a		; leaves a white border
	ld bc,0
	jr exit_ok_c

read_errore
no_card_c
	ld bc,1
exit_ok_c
	ld a,(fast_save)	; FASTPAGE register is restored
	out (FASTPAGE),a
	ei
	ret



complement
	di

	in a,(FASTPAGE)		; FASTPAGE register is saved
	ld (fast_save),a

	ld e,$C0		; RAM_BANK = 0, RD/WR

bank_comple
	ld a,e
	out (FASTPAGE),a
	ld hl,0
	ld bc,16384
l_comple
	ld a,(hl)
	xor $FF
	ld (hl),a
	inc hl
	dec bc
	ld a,b
	or c
	jr nz,l_comple

	inc e			; next bank
	ld a,e
	and 31			; 32 banks
	out (BORDER),a		; shows progress
	jr nz,bank_comple

	ld a,7
	out (BORDER),a		; leaves a white border

	ld a,(fast_save)	; FASTPAGE register is restored
	out (FASTPAGE),a
	ei
	ret



;
;---------------------------------------------------------------------------------------------
; Subroutine che verifica la presenza di una SD/MMC inserita nello slot.
; Rientra con A = 0 se OK; 1 in caso di errore.
;---------------------------------------------------------------------------------------------
sd_check
	push bc
	ld hl,temp_ram		; tenta di leggere il CID di una eventuale SD/MMC salvandolo a TEMP_RAM
	call mmc_getcid_poweron
	pop bc
	ret


;*************************************************************************************************
;*************************************************************************************************
; MMC READ/WRITE ROUTINES that can operate @21.25MHz processor clock.
;
; RD/WR are on 512 bytes boundary ONLY.
; MULTIPLE_BLOCK READ is supported, while by now MULTIPLE_BLOCK WR is not
;
;*************************************************************************************************
;*************************************************************************************************

;-------------------------------------------------------------------------------------------
; This subroutine should be called at power-on or when a MMC has been inserted.
; It tries to get the MMC CID (writing at the provided HL pointer) and, in case of failure,
; it calls the INIT procedure then tries again.
;
; Returns A = 0 if OK, or 1 = error (no MMC found or MMC error)
;
; iF != 0 AND != $ff, the mmc_get_cid returned error code is displayed on screen.
;-------------------------------------------------------------------------------------------
mmc_getcid_poweron
	call mmc_get_cid		; try to read the MMC CID INFO --> (HL)
	cp 0				; if it fails, then che SD/MMC needs to be initialized
	ret z				; to SPI mode (MMC_INIT) communications (once after power-on)

	cp 255				; card is probably not in SPI mode
	jr z,need_init

	bit 0,a				; IDLE/still initializing bit
	jr nz,need_init

	ld a,1				; unknown response: exit
	ret

need_init
	call mmc_init
	cp 0
	ret nz				; INIT error: MMC not detected.

	ld de,BLOCKSIZE
	call mmc_send_blocksize
	jr mmc_getcid_poweron


;
;-----------------------------------------------------------------------------------------
; READ MULTIPLE BLOCK OF DATA TEST subroutine
;
; This routine only works for blocksize = 512 (two INI sequence).
;
; HL, DE= MSB, LSB of 32bit address in MMC memory
; B	= number of 512 bytes blocks to be read
; IX	= ram buffer address
;
; RETURN code in A:
; 0 = OK
; 1 = read_block command error
; 2 = no wait_data token from MMC
;
; DESTROYS AF, B
;-----------------------------------------------------------------------------------------
mmc_read_multidata
	ld a,MMC_READ_MULTIPLE_BLOCK	; Command code for multiple block read
	call mmc_send_command
	cp 0
	ret nz				; ERRORE
	push hl				; HL should be saved

	push ix
	pop hl				; INI usa HL come puntatore
jrhere
	call mmc_waitdata_token
	cp $FE
	jr z,read_mmc_multiblock	; OK
	ld a,2				; no data token from MMC
	pop hl
	ret

read_mmc_multiblock
	push bc
	ld bc,SPI_PORT			; B = 0 = 256 bytes for the first INI; C = I/O port

;	inir
;	nop
;	inir

ini_loop1
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	jr nz,ini_loop1

ini_loop2
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	jr nz,ini_loop2

	nop
	nop

	in a,(SPI_PORT)			; CRC
	nop
	nop
	in a,(SPI_PORT)			; CRC

	pop bc
	djnz jrhere

	ld a,MCC_TERMINATE_MULTI_READ
	call mmc_write_command

	in a,(SPI_PORT)			; CRC?
	nop
	nop
	in a,(SPI_PORT)

	call mmc_wait_response		; waits for the MMC to reply "0"
	ld b,a				; error code

	call cs_high			; set cs high
	call clock32			; 32 more clock cycles
	call clock32			; 32 more clock cycles
	call clock32			; 32 more clock cycles
	call clock32			; 32 more clock cycles

	pop hl
	ld a,b
	ret


;
;-----------------------------------------------------------------------------------------
; READ A SINGLE BLOCK OF DATA subroutine
;
; This routine only works for blocksize = 512 (two INIR sequence).
;
; HL, DE= MSB, LSB of 32bit address in MMC memory
; IX	= ram buffer address
;
; RETURN code in A:
; 0 = OK
; 1 = read_block command error
; 2 = no wait_data token from MMC
;
; DESTROYS AF, HL
;-----------------------------------------------------------------------------------------
mmc_read_data

	ld a,MMC_READ_SINGLE_BLOCK	; Command code for block read
	call mmc_send_command
	cp 0
	ret nz				; ERRORE

	call mmc_waitdata_token
	cp $FE
	jr z,read_mmc_block		; OK
	ld a,2				; no data token from MMC
	ret

read_mmc_block
	push ix
	pop hl				; HL = INIR write pointer

	push bc
	ld bc,SPI_PORT			; B = 0 = 256 bytes for the first INIR; C = I/O port

;	inir
;	nop
;	inir

ini_loop3
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	jr nz,ini_loop3

ini_loop4
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	jr nz,ini_loop4

	nop
	nop

	in a,(SPI_PORT)			; CRC
	nop
	nop
	in a,(SPI_PORT)			; CRC

	call cs_high			; set cs high
	call clock32			; 32 more clock cycles

	pop bc

	xor a				; no error
	ret


clock32
	push bc
	ld b,4
l_4bytes
	in a,(SPI_PORT)			; some more clock cycles
	djnz l_4bytes
	pop bc
	ret


;-----------------------------------------------------------------------------------------
; READ FIRST 2 BYTES OF SD/MMC CARD and copies them to DE.
;
; RETURN code in A:
; 0 = OK
; 1 = read_block command error
; 2 = no wait_data token from MMC
;
; DESTROYS AF, DE
;-----------------------------------------------------------------------------------------
mmc_get_firstword
	push hl
	ld a,MMC_READ_SINGLE_BLOCK	; Command code for block read
	ld hl,0
	ld de,0
	call mmc_send_command
	pop hl
	cp 0
	ret nz				; ERRORE
	call mmc_waitdata_token
	cp $FE
	jr z,read_mmc_first		; OK
	ld a,2				; no data token from MMC
	ret

read_mmc_first
	in a,(SPI_PORT)
	ld e,a
	nop
	in a,(SPI_PORT)
	ld d,a

	push bc
	ld bc,BLOCKSIZE-2		; reads the remaining 510 bytes and throws them
read_firstword
	in a,(SPI_PORT)
	dec bc
	ld a,b
	or c
	jr nz,read_firstword
	pop bc

	in a,(SPI_PORT)			; CRC
	nop
	nop
	in a,(SPI_PORT)			; CRC

	call cs_high			; set cs high
	call clock32			; 32 more clock cycles

	xor a
	ret

;
;-----------------------------------------------------------------------------------------
; WRITE BLOCK OF DATA subroutine. By now, we don't use the MULTIPLE_BLOCK transfer.
;
; This routine only works for blocksize = 512 (two OUTI sequence).
;
; HL, DE= MSB, LSB of 32bit address in MMC memory
; B	= number of 512 bytes blocks to write
; IX	= ram buffer address
;
; RETURN code in A:
; 0 = OK
; 1 = read_block command error
; 2 = write error (no "5" response from MMC)
;
; Destroys AF, B, DE, HL, IX.
;-----------------------------------------------------------------------------------------
mmc_write_data
	ld a,MMC_WRITE_SINGLE_BLOCK	; Command code for block read
	call mmc_send_command
	cp 0
	ret nz				; ERRORE

	ld a,$FE
	out (SPI_PORT),a		; first byte to be sent = DATA TOKEN

write_mmc_block
	push bc
	ld bc,$3F			; C = PORT, B = 0 for 256 bytes on first OTIR
	push hl
	push ix
	pop hl

;	otir
;	nop				; without this NOP, the 2nd half of the 512 bytes block would miss some write cycles
;	otir				; on the CPLD if Z80 runs at 21MHz

out_loop1
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	jr nz,out_loop1

out_loop2
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	jr nz,out_loop2

	push hl
	pop ix
	pop hl

	ld a,$95			; CRC... (!)
	out (SPI_PORT),a
	nop
	nop
	out (SPI_PORT),a

	call mmc_wait_response

	pop bc

	and $1F				; masks useful response bits
	cp 5
	jr nz,write_error

	ld a,d				; aggiorna il puntatore al blocco da leggere
	add BLOCKSIZE/256		; 2 se blocksize = 512
	jr nc,no_overw
	inc hl
no_overw
	ld d,a

wait_busy
	call mmc_wait_response		; MMC will report "00" until busy
	cp 0
	jr z,wait_busy

	call cs_high
	call clock32			; 32 more clock cycles

	djnz jrmmc_write_data		; next block
	xor a
	ret
jrmmc_write_data
	jp mmc_write_data

write_error
	ld a,2				; write error code
	ret


;
;-----------------------------------------------------------------------------------------
; MMC SPI MODE initialization. RETURNS ERROR CODE IN A register:
;
; 0 = OK
; 1 = Card RESET ERROR
; 2 = Card INIT ERROR
;
; Destroys AF, B.
;-----------------------------------------------------------------------------------------
mmc_init
	call	cs_high			; set cs high
	ld	b,10			; sends 80 clocks
	ld a,$FF
l_init
	out (SPI_PORT),a
	djnz l_init
	nop
	
	call	cs_low			; set cs low
	
	ld a,MMC_GO_IDLE_STATE
	call mmc_write_command

	call mmc_wait_response
	cp $01				; MMC should respond 01 to this command
	jr nz,mmc_reset_failed		; fail to reset

	ld bc,120			; retry counter*256 (corrisponde a circa 5 secondi @3.5MHz)
mmc_reset_ok
	call cs_high			; set cs high
	ld a,$FF
	out (SPI_PORT),a		; 8 extra clock cycles
	nop
	nop
	
	call	cs_low			; set cs low
	
	ld a,MMC_SEND_OP_COND		; Sends OP_COND command
	call mmc_write_command

	call mmc_wait_response		; MMC_WAIT_RESPONSE tries to receive a response reading an SPI

	bit 0,a				; D0 SET = initialization still in progress...
	jr z,mmc_init_ok

	djnz mmc_reset_ok		; if no response, tries to send the entire block 254 more times
	dec c
	jr nz,mmc_reset_ok

	ld a,2				; error code for INIT ERROR
	jr mmc_errorx

mmc_init_ok
	call cs_high			; set cs high
	in a,(SPI_PORT)			; some extra clock cycles
	call pause1
	xor a
	ret
	
mmc_reset_failed			; MMC Reset error
	ld	a,1
mmc_errorx
	call	cs_high
	ret


;debug_display				; TEST: displays the code in A at current cursor position
;	push af				; ('A' based): A = 0x00, B = 0x01, etc.
;	push bc
;	push de
;	push hl
;	add 'A'
;	call disp_ch
;	ld bc,0
;lll
;	djnz lll
;	dec c
;	jr nz,lll
;	pop hl
;	pop de
;	pop bc
;	pop af
;	ret


;
;-----------------------------------------------------------------------------------------
; READ MMC CID subroutine. Data is stored at address (HL).
;
; Returns error code in A register:
; 0 = no error
; 1 = Read CID command error
; 2 = no wait_data token from MMC
;
; Destroys AF, HL.
;-----------------------------------------------------------------------------------------
mmc_get_cid
	push bc
	push de
	push hl
	ld hl,0
	ld de,0
	ld a,MMC_READ_CID
	call mmc_send_command
	pop hl
	cp 0
	jr z,cmd_cidok			; MMC_SEND_COMMAND reports 0 (ok) or MMC error code
	push af				; error code on STACK
	jr cid_exit
cmd_cidok
	call mmc_waitdata_token
	cp $FE
	jr z,waitda_cid_ok
	ld a,2
	push af
	jr cid_exit

waitda_cid_ok
	ld b,18				; 16 bytes + CRC?
	ld c,SPI_PORT

	inir

;mmc_cidl
;	ini
;	jr nz,mmc_cidl

	xor a
	push af
cid_exit
	call cs_high			; set cs high
	in a,(SPI_PORT)
	pop af				; error code
	pop de
	pop bc
	ret


;
;-----------------------------------------------------------------------------------------
; SEND COMMAND TO MMC subroutine
;
; A = COMMAND CODE;
; H, L, D, E = 32 bit parameter (MSB ... LSB);
;
; Sends a $FF fake checksum
;
; RETURNS: 0 = OK; != 0 = MMC error code
;
; On OK, the CHIP SELECT will be LOW on exit
; On error, the CHIP SELECT will be deasserted on exit
;
; Destroys AF.
;-----------------------------------------------------------------------------------------
mmc_send_command
	push bc
	ld c,a
	call cs_high			; cs high
	call clock32
	nop
	call cs_low			; cs low
	ld a,c				; command code is the first byte to be sent
	out (SPI_PORT),a
	ld a,h
	nop
	out (SPI_PORT),a
	ld a,l
	nop
	out (SPI_PORT),a
	ld a,d
	nop
	out (SPI_PORT),a
	ld a,e
	nop
	out (SPI_PORT),a
	ld a,$ff
	nop
	out (SPI_PORT),a

	call mmc_wait_response		; waits for the MMC to reply != $FF
	cp 0
	jr nz,mmc_commande
	pop bc
	ret				; 0 = no error

mmc_commande
	push af				; saves the error code
	call cs_high			; set cs high
	in a,(SPI_PORT)
	pop af
	pop bc
	ret				; returns the error code got from MMC


;
;-----------------------------------------------------------------------------------------
; WAIT FOR DATA TOKEN ($FE) subroutine (calls MMC_WAIT_RESPONSE up to 256 times)
; Returns with A = code read from MMC (ok if $FE). Destroys AF.
;-----------------------------------------------------------------------------------------
mmc_waitdata_token
	push bc
	ld b,10				; retry counter
mmc_waitl
	call mmc_wait_response
	cp $FE				; waits for the MMC to reply $FE (DATA TOKEN)
	jr z,exit_wda
	cp $FF				; but if not $FF, exits immediately (error code from MMC)
	jr nz,exit_wda
	djnz mmc_waitl
exit_wda
	pop bc
	ret


;
;-----------------------------------------------------------------------------------------
; SENDS BLOCK_SIZE COMMAND TO MMC.
;
; Parameters: DE = nbytes. Destroys AF.
;-----------------------------------------------------------------------------------------
mmc_send_blocksize
	call	cs_low			; set cs low
	ld a,MMC_SET_BLOCK_SIZE
	out (SPI_PORT),a
	xor a
	nop
	out (SPI_PORT),a
	nop
	nop
	out (SPI_PORT),a
	ld a,d
	nop
	out (SPI_PORT),a
	ld a,e
	nop
	out (SPI_PORT),a
	ld a,$ff			; fake checksum
	nop
	out (SPI_PORT),a
	nop
	nop

	in a,(SPI_PORT)
	nop
	nop
	in a,(SPI_PORT)
	nop
	nop
	call	cs_high
	
	ret


;
;-----------------------------------------------------------------------------------------
; Sends a command with parameters = 00 and checksum = $95. Destroys AF.
;-----------------------------------------------------------------------------------------
mmc_write_command
	push bc
	out (SPI_PORT),a	; sends the command
	ld b,4
	xor a
l_sendc0
	out (SPI_PORT),a	; then sends four "00" bytes (parameters = NULL)
	djnz l_sendc0
	ld a,$95		; $95 is only needed when the CARD INIT is being performed,
	nop
	out (SPI_PORT),a	; then this byte is ignored.
	pop bc
	ret


;
;-----------------------------------------------------------------------------------------
; Waits for the MMC to respond. Returns with A = response code; $FF = NO RESPONSE
;
; Responses from CARD are in R1 format for all command except SEND_STATUS.
; When SET, a bit indicates:
;
; D0 = Idle state / init not completed yet
; D1 = Erase Reset
; D2 = Illegal Command
; D3 = Com CRC Error
; D4 = Erase Sequence Error
; D5 = Address Error
; D6 = Parameter Error
; D7 = ALWAYS LOW
;
; Destroys AF.
;-----------------------------------------------------------------------------------------
mmc_wait_response
	push bc
	ld bc,50		; retry counter
l_response
	in a,(SPI_PORT)		; reads a byte from MMC
	cp $FF			; $FF = no card data line activity
	jr nz,resp_ok
	djnz l_response
	dec c
	jr nz,l_response
resp_ok
	pop bc
	ret


;
;------------------------------------------------------------------------------------
; CHIP_SELECT HIGH subroutine. Destroys no registers. Entire port is tied to '1'.
;------------------------------------------------------------------------------------
cs_high
	push af
	ld a,255
	out (OUT_PORT),a
	pop af
	ret

	
;
;------------------------------------------------------------------------------------
; CHIP_SELECT LOW subroutine. Destroys no registers. The card to be selected should
; specified in CARD_SELECT (D1 = SLOT1, D0 = SLOT0, active LOW)
;------------------------------------------------------------------------------------
cs_low
	push af
	ld a,(card_select)
	out (OUT_PORT),a
	pop af
	ret


pause1
	push hl
	ld hl,$8000		; OK for 3.5MHz ONLY (7MHz requires two calls or ld hl,0 and so on)
loop3	dec hl
	ld a,h
	or l
	jr nz,loop3
	pop hl
	ret


; VARIABILI IN CODA

snap_torestore	equ	$
	org $+2
entry_addr	equ	$
	org $+2


        end 

