;---------------------------------------------------------------------------------------------
;
; ZXMMC+ test software; allows reading/writing blocks to/from the screen and restoring
; 16K snapshots in ZX-Badaloc's BOOTROM firmware format.
;
; Unica modifica rispetto a zxmmc: la copia dei 512K avviene ad inizio card.
;
;
; MMC routines are from ZX-Badaloc BOOTROM Version 4.96 01/03/2007, that could run at 21MHz.
; NEW VERSION with OUTI/INI instruction unrolling, suggested by Paolo Ferraris, for maximum
; performance (218Kbytes/sec @3.5MHz)
;
; 4 Functions are provided:
;
; 40.000:	CARD INIT. Returns number of found cards (0, 1 or 2)
; 40.003:	13 blocks (512 bytes each) at offset +512KB are written to the card (from screen content)
; 40.006:	13 blocks (512 bytes each) from offset +512KB are read from the card to the screen
; 40.009:	Restores a 16K snapshot, that should have been previously saved to the card with
;		ZX-Badaloc. The snapshot number should be poked at address 23728 (LSB) and 23729 (MSB).
; 40.012:	Copy one megabyte at offset +512K from SLOT0 to SLOT1, in 8K blocks.
; 40.015:	Compare one megabyte at offset +512K between SLOT0 and SLOT1.
;
; For all functions except the first one and last two, card to be accessed should be specified at location
; 41400 as follows: $FE for SLOT0, $FD for SLOT1.
;
; It is not possible to load 48K snapshots, since the upper portion of memory holds this routine.
;
;---------------------------------------------------------------------------------------------
;


; CPLD REGISTERS
;
SPI_PORT	equ $3F
OUT_PORT	equ $1F		; port for CS control (D1:D0)

MMC_0		equ $F6		; D0 LOW = SLOT0 active; D3 low = NMI disabled
MMC_1		equ $F5		; D1 LOW = SLOT1 active; D3 low = NMI disabled


BORDER	equ $FE			; PORT setup border sui bit D2 - D0 (gli altri riguardano SPKR/TAPE)
BORDCR	equ 23624		; variabile SPECTRUM contenente il colore del border *8


; SD/MMC RELATED DEFINES:

MMC_SNAP16	equ	1	; il cluster contiene uno snapshot da 16K

MMC_GO_IDLE_STATE	equ	$40
MMC_SEND_OP_COND	equ	$41
MMC_READ_CID		equ	$4A
MMC_SET_BLOCK_SIZE	equ	$50
MMC_READ_SINGLE_BLOCK	equ	$51
MMC_READ_MULTIPLE_BLOCK	equ	$52
MCC_TERMINATE_MULTI_READ equ	$4C
MMC_WRITE_SINGLE_BLOCK	equ	$58
MMC_WRITE_MULTIPLE_BLOCK equ	$59
MMC_STOP_TRAN		equ	$FD


BLOCKSIZE	equ	512	; SD/MMC block size (bytes)

FATSIZE		equ	8	; dimensioni FAT in termini di blocchi da 64KBytes (in pratica e` l'offset
				; da caricare nella word MSB dell'indirizzo della SD per accedere al primo
				; cluster di dati, subito oltre la FAT). 8 = 512KBytes (8192 entries da 64 bytes)


	org 41400		; variables
card_select	equ	$
	org $+1
found_cards	equ	$
	org $+1
temp_ram	equ	$
	org $+512



;********************************************************************************************************
;********************************************************************************************************

	org 40000

	jp init			; CARD INIT
	jp writecard		; WRITE TO CARD
	jp readcard		; READ FROM CARD
	jp read_snap		; RESTORE A SNAPSHOT (desired snapshot number should be poked at address 23728)
	jp copy0to1		; COPY 1 MEGABYTE at offset 0 from SLOT0 to SLOT1
	jp compare		; COMPARE 1 MEGABYTE at offset 0 between SLOT0 and SLOT1


init
	di
	in a,(RXREG)		; clears UART RX register

	ld c,0			; found cards

	ld a,MMC_0		; SLOT 0 ENABLE (also disables NMI on RS-232 byte reception (D3 low))
	ld (card_select),a
	call sd_check		; controlla eventuale presenza di SD/MMM
	cp 0
	jr nz,no_detect0	; salta se la card non e` stata rilevata
	inc c

no_detect0
	ld a,MMC_1		; SLOT 1 ENABLE
	ld (card_select),a
	call sd_check		; controlla eventuale presenza di SD/MMM
	cp 0
	jr nz,no_detect1	; salta se la card non e` stata rilevata
	inc c

no_detect1
	ld a,c
	ld (found_cards),a
	ld b,0

	ei
	ret



readcard
	di
	ld hl,FATSIZE		; MSB = FATSIZE = reads from a 512Kbytes offset
	ld de,0
	ld ix,16384
	ld b,13
	call mmc_read_multidata
	ld b,0
	ld c,a
	ei
	ret


writecard
	di
	ld hl,FATSIZE		; MSB = FATSIZE = writes to a 512Kbytes offset
	ld de,0
	ld ix,16384
	ld b,13
	call mmc_write_data
	ld b,0
	ld c,a
	ei
	ret


copy0to1
	di
;	ld hl,FATSIZE		; +512K offset
	ld hl,0
	ld de,0
	ld b,128		; number of 8K blocks to copy (128 = 1MB)
next_block
	ld a,MMC_0		; source card
	ld (card_select),a
	push bc
	ld ix,42000		; buffer address
	ld b,16			; sectors to read (512 bytes) = 8KB
	push hl
	push de
	call mmc_read_multidata

	ld hl,42000		; checkpoint: copies the first 6912 bytes to screen area
	ld de,16384
	ld bc,6912
	ldir

	pop de
	pop hl
	cp 0
	jr nz,cpread_error

	ld a,MMC_1		; destination card
	ld (card_select),a
	push hl
	push de
	ld ix,42000		; buffer address
	ld b,16			; sectors to write (512 bytes) = 8KB
	call mmc_write_data
	pop de
	pop hl
	pop bc
	cp 0
	jr nz,cpwrite_error

	ld a,d
	add 32			; INC LSB pointer (+8K offset)
	ld d,a
	jr nc,next_8copy
	inc hl			; INC MSB pointer
next_8copy
	djnz next_block
	ld bc,0			; no errors
	ei
	ret
cpread_error
	pop bc
	ld bc,1
	ei
	ret
cpwrite_error
	ld bc,2
	ei
	ret



compare
	di
;	ld hl,FATSIZE		; +512K offset
	ld hl,0
	ld de,0
	ld b,128		; number of 8K blocks to compare (128 = 1MB)
next_b_compare
	ld a,MMC_0		; SLOT 0
	ld (card_select),a
	push bc
	ld ix,42000		; buffer address
	ld b,16			; sectors to read (512 bytes) = 8KB
	push hl
	push de
	call mmc_read_multidata

	ld hl,42000		; checkpoint: copies the first 6912 bytes to screen area
	ld de,16384
	ld bc,6912
	ldir

	pop de
	pop hl
	cp 0
	jr nz,co0read_error

	ld a,MMC_1		; SLOT 1
	ld (card_select),a
	push hl
	push de
	ld ix,52000		; buffer address
	ld b,16			; sectors to read (512 bytes) = 8KB
	call mmc_read_multidata

	ld hl,52000		; checkpoint: copies the first 6912 bytes to screen area
	ld de,16384
	ld bc,6912
	ldir

	cp 0
	jr nz,co1read_error

	ld bc,8192		; bytes to compare
	ld hl,42000		; address of buffer 0
	ld de,52000		; address of buffer 1
l_compare
	ld a,(de)
	cp (hl)
	jr nz,compare_error
	inc hl
	inc de
	dec bc
	ld a,c
	or b
	jr nz,l_compare

	pop de
	pop hl
	pop bc

	ld a,d
	add 32			; INC LSB pointer (+8K)
	ld d,a
	jr nc,next_8compare
	inc hl			; INC MSB pointer
next_8compare
	djnz next_b_compare
	ld bc,0			; no errors
	ei
	ret
co0read_error
	pop bc
	ld bc,1			; error reading SLOT 0
	ei
	ret
co1read_error
	pop de
	pop hl
	pop bc
	ld bc,2			; error reading SLOT 1
	ei
	ret
compare_error
	pop de
	pop hl
	pop bc
	ld bc,3			; data error
	ei
	ret




read_snap
	di
	ld hl,(23728)		; snapshot number (16 bit from 23728, 23729)


;
;*****************************************************************************************************************
; RIPRISTINO SNAPSHOT 16K
;*****************************************************************************************************************
; READ EXAMPLE

	ld sp,temp_ram+2048		; stack in upper ram

	ld (snap_torestore),hl		; salva il numero dello snapshot da ripristinare
	ex hl,de

	call calc_hldec			; calcola l'indirizzo del blocco --> HL, DE e l'offset 64 bytes in C

	ld ix,temp_ram			; scratchpad area
	call mmc_read_data		; performs the single block read
	cp 0
        jr z,restore_ok1

restore_error
	ld a,2
	out (BORDER),a			; segnala l'errore
gasp
        jr gasp

restore_ok1
	ld b,c				; BC = offset 64 bytes entry da leggere
	ld c,0
	srl b
	rr c
	srl b
	rr c

	ld hl,temp_ram			; scratchpad area
	add hl,bc			; aggiunge l'offset relativo all'entry da leggere (ce ne sono 8 in 512 bytes)
	ld (entry_addr),hl

	ld ix,(entry_addr)
	ld a,(ix+2)			; n. blocchi da ripristinare: 1 = 16K, 2 = 48K, 8 = 128K
	cp MMC_SNAP16			; 1
	jr z,snap_valid_nb
	ld a,3
	out (BORDER),a
	jr gasp


snap_valid_nb

	ld hl,(snap_torestore)
	ex hl,de
	ld hl,FATSIZE			; n. blocchi da 64KB che costituiscono la FAT (e` l'offset per la word MSB)
	sla e
	rl d				; moltiplica per 2 (ogni entry corrisponde ad un cluster da 128KB)
	add hl,de			; HL = word MSB scrittura in SD/MMC
	ld de,0				; word LSB = 0 (insieme a D0 = 0 della word MSB = 128K)

	ld b,16384/BLOCKSIZE

	ld ix,$4000			; ram buffer (restore point)
	call mmc_read_multidata		; 16K/48K restore. we can't rely on stack for temporary data
	cp 0				; error?
        jr nz,restore_error

	ld a,(BORDCR)			; colore attuale del border *8 (variabile del basic sinclair)
	rra
	rra
	rra
	and 7				; rimette a posto il colore originale
	out (BORDER),a			; gli altri bit possono essere ZERO (SPKR, TAPE)

	ld ix,(entry_addr)

	ld a,(ix+44)
	ld i,a				; ripristina lo stato del registro I

	ld l,(ix+42)			; RIPRISTINA LO STATO DELLO STACK POINTER
	ld h,(ix+43)
	ld sp,hl

	ld h,(ix+46)			; Registro F
	di				; not really necessary, but who knows...
	im 1
	ld a,h				; for exiting the handler and restore the BASIC ROM.
	and 3				; if the host's int handler need the basic rom, it would crash
	cp 0
	jr z,imzerommc
	cp 1
	jr z,completedeimmc
	im 2
	jr completedeimmc
imzerommc
	im 0

completedeimmc
	bit 2,h				; interrupt enable real status
	jr z,no_eneimmc
	ei				; this will execute our fake handler (EI, RETI)
no_eneimmc				; now we have 20 (or 10) ms before next interrupt will occur

	exx
	pop ix
	pop hl
	pop de
	ex af,af'
	pop af
	ex af,af'
	pop iy
	pop bc
	exx
	pop de
	pop bc
	pop hl
	pop af
	retn
end_snaprestore


; Subroutine che calcola il blocco da 512 bytes che contiene l'entry (file) nella FAT.
; Richiede: DE = n. del file (snapshot)
; Restituisce: HL, DE = indirizzo 32 bit settore nells SD/MMC
; C = offset 0-7 che identifica l'header desiderato (ciascuno e` 64 bytes) all'interno
; di questo blocco da 512 bytes

calc_hldec
	ld a,e
	and 7
	ld c,a				; salva i 3 bit LSB in C
	ld a,e
	and $F8				; e li elimina da DE
	ld e,a

	ld hl,0
	ld b,6				; moltiplica *64 (6 shift) = dimensioni di ciascuna entry
mul6
	sla e
	rl d
	rl l				; shift a 24 bit: dovendo indirizzare 512KB, coinvolge anche i 3 bit LSB di L
	djnz mul6
	ret


;********************************************************************************************
; FINE RIPRISTINO SNAPSHOT DA SD/MMC 16K/48K/128K


;
;---------------------------------------------------------------------------------------------
; Subroutine che verifica la presenza di una SD/MMC inserita nello slot.
; Rientra con A = 0 se OK; 1 in caso di errore.
;---------------------------------------------------------------------------------------------
sd_check
	push bc
	ld hl,temp_ram		; tenta di leggere il CID di una eventuale SD/MMC salvandolo a TEMP_RAM
	call mmc_getcid_poweron
	pop bc
	ret


;*************************************************************************************************
;*************************************************************************************************
; MMC READ/WRITE ROUTINES that can operate @21.25MHz processor clock.
;
; RD/WR are on 512 bytes boundary ONLY.
; MULTIPLE_BLOCK READ is supported, while by now MULTIPLE_BLOCK WR is not
;
;*************************************************************************************************
;*************************************************************************************************

;-------------------------------------------------------------------------------------------
; This subroutine should be called at power-on or when a MMC has been inserted.
; It tries to get the MMC CID (writing at the provided HL pointer) and, in case of failure,
; it calls the INIT procedure then tries again.
;
; Returns A = 0 if OK, or 1 = error (no MMC found or MMC error)
;
; iF != 0 AND != $ff, the mmc_get_cid returned error code is displayed on screen.
;-------------------------------------------------------------------------------------------
mmc_getcid_poweron
	call mmc_get_cid		; try to read the MMC CID INFO --> (HL)
	cp 0				; if it fails, then che SD/MMC needs to be initialized
	ret z				; to SPI mode (MMC_INIT) communications (once after power-on)

	cp 255				; card is probably not in SPI mode
	jr z,need_init

	bit 0,a				; IDLE/still initializing bit
	jr nz,need_init

	ld a,1				; unknown response: exit
	ret

need_init
	call mmc_init
	cp 0
	ret nz				; INIT error: MMC not detected.

	ld de,BLOCKSIZE
	call mmc_send_blocksize
	jr mmc_getcid_poweron


;
;-----------------------------------------------------------------------------------------
; READ MULTIPLE BLOCK OF DATA TEST subroutine
;
; This routine only works for blocksize = 512 (two INI sequence).
;
; HL, DE= MSB, LSB of 32bit address in MMC memory
; B	= number of 512 bytes blocks to be read
; IX	= ram buffer address
;
; RETURN code in A:
; 0 = OK
; 1 = read_block command error
; 2 = no wait_data token from MMC
;
; DESTROYS AF, B
;-----------------------------------------------------------------------------------------
mmc_read_multidata
	ld a,MMC_READ_MULTIPLE_BLOCK	; Command code for multiple block read
	call mmc_send_command
	cp 0
	ret nz				; ERRORE
	push hl				; HL should be saved

	push ix
	pop hl				; INI usa HL come puntatore
jrhere
	call mmc_waitdata_token
	cp $FE
	jr z,read_mmc_multiblock	; OK
	ld a,2				; no data token from MMC
	pop hl
	ret

read_mmc_multiblock
	push bc
	ld bc,SPI_PORT			; B = 0 = 256 bytes for the first INI; C = I/O port

;	inir
;	nop
;	inir

ini_loop1
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	jr nz,ini_loop1

ini_loop2
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	jr nz,ini_loop2

	nop
	nop

	in a,(SPI_PORT)			; CRC
	nop
	nop
	in a,(SPI_PORT)			; CRC

	pop bc
	djnz jrhere

	ld a,MCC_TERMINATE_MULTI_READ
	call mmc_write_command

	in a,(SPI_PORT)			; CRC?
	nop
	nop
	in a,(SPI_PORT)

	call mmc_wait_response		; waits for the MMC to reply "0"
	ld b,a				; error code

	call cs_high			; set cs high
	call clock32			; 32 more clock cycles
	call clock32			; 32 more clock cycles
	call clock32			; 32 more clock cycles
	call clock32			; 32 more clock cycles

	pop hl
	ld a,b
	ret


;
;-----------------------------------------------------------------------------------------
; READ A SINGLE BLOCK OF DATA subroutine
;
; This routine only works for blocksize = 512 (two INIR sequence).
;
; HL, DE= MSB, LSB of 32bit address in MMC memory
; IX	= ram buffer address
;
; RETURN code in A:
; 0 = OK
; 1 = read_block command error
; 2 = no wait_data token from MMC
;
; DESTROYS AF, HL
;-----------------------------------------------------------------------------------------
mmc_read_data

	ld a,MMC_READ_SINGLE_BLOCK	; Command code for block read
	call mmc_send_command
	cp 0
	ret nz				; ERRORE

	call mmc_waitdata_token
	cp $FE
	jr z,read_mmc_block		; OK
	ld a,2				; no data token from MMC
	ret

read_mmc_block
	push ix
	pop hl				; HL = INIR write pointer

	push bc
	ld bc,SPI_PORT			; B = 0 = 256 bytes for the first INIR; C = I/O port

;	inir
;	nop
;	inir

ini_loop3
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	jr nz,ini_loop3

ini_loop4
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	ini
	jr nz,ini_loop4

	nop
	nop

	in a,(SPI_PORT)			; CRC
	nop
	nop
	in a,(SPI_PORT)			; CRC

	call cs_high			; set cs high
	call clock32			; 32 more clock cycles

	pop bc

	xor a				; no error
	ret


clock32
	push bc
	ld b,4
l_4bytes
	in a,(SPI_PORT)			; some more clock cycles
	djnz l_4bytes
	pop bc
	ret


;-----------------------------------------------------------------------------------------
; READ FIRST 2 BYTES OF SD/MMC CARD and copies them to DE.
;
; RETURN code in A:
; 0 = OK
; 1 = read_block command error
; 2 = no wait_data token from MMC
;
; DESTROYS AF, DE
;-----------------------------------------------------------------------------------------
mmc_get_firstword
	push hl
	ld a,MMC_READ_SINGLE_BLOCK	; Command code for block read
	ld hl,0
	ld de,0
	call mmc_send_command
	pop hl
	cp 0
	ret nz				; ERRORE
	call mmc_waitdata_token
	cp $FE
	jr z,read_mmc_first		; OK
	ld a,2				; no data token from MMC
	ret

read_mmc_first
	in a,(SPI_PORT)
	ld e,a
	nop
	in a,(SPI_PORT)
	ld d,a

	push bc
	ld bc,BLOCKSIZE-2		; reads the remaining 510 bytes and throws them
read_firstword
	in a,(SPI_PORT)
	dec bc
	ld a,b
	or c
	jr nz,read_firstword
	pop bc

	in a,(SPI_PORT)			; CRC
	nop
	nop
	in a,(SPI_PORT)			; CRC

	call cs_high			; set cs high
	call clock32			; 32 more clock cycles

	xor a
	ret

;
;-----------------------------------------------------------------------------------------
; WRITE BLOCK OF DATA subroutine. By now, we don't use the MULTIPLE_BLOCK transfer.
;
; This routine only works for blocksize = 512 (two OUTI sequence).
;
; HL, DE= MSB, LSB of 32bit address in MMC memory
; B	= number of 512 bytes blocks to write
; IX	= ram buffer address
;
; RETURN code in A:
; 0 = OK
; 1 = read_block command error
; 2 = write error (no "5" response from MMC)
;
; Destroys AF, B, DE, HL, IX.
;-----------------------------------------------------------------------------------------
mmc_write_data
	ld a,MMC_WRITE_SINGLE_BLOCK	; Command code for block read
	call mmc_send_command
	cp 0
	ret nz				; ERRORE

	ld a,$FE
	out (SPI_PORT),a		; first byte to be sent = DATA TOKEN

write_mmc_block
	push bc
	ld bc,$3F			; C = PORT, B = 0 for 256 bytes on first OTIR
	push hl
	push ix
	pop hl

;	otir
;	nop				; without this NOP, the 2nd half of the 512 bytes block would miss some write cycles
;	otir				; on the CPLD if Z80 runs at 21MHz

out_loop1
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	jr nz,out_loop1

out_loop2
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	outi
	jr nz,out_loop2

	push hl
	pop ix
	pop hl

	ld a,$95			; CRC... (!)
	out (SPI_PORT),a
	nop
	nop
	out (SPI_PORT),a

	call mmc_wait_response

	pop bc

	and $1F				; masks useful response bits
	cp 5
	jr nz,write_error

	ld a,d				; aggiorna il puntatore al blocco da leggere
	add BLOCKSIZE/256		; 2 se blocksize = 512
	jr nc,no_overw
	inc hl
no_overw
	ld d,a

wait_busy
	call mmc_wait_response		; MMC will report "00" until busy
	cp 0
	jr z,wait_busy

	call cs_high
	call clock32			; 32 more clock cycles

	djnz jrmmc_write_data		; next block
	xor a
	ret
jrmmc_write_data
	jp mmc_write_data

write_error
	ld a,2				; write error code
	ret


;
;-----------------------------------------------------------------------------------------
; MMC SPI MODE initialization. RETURNS ERROR CODE IN A register:
;
; 0 = OK
; 1 = Card RESET ERROR
; 2 = Card INIT ERROR
;
; Destroys AF, B.
;-----------------------------------------------------------------------------------------
mmc_init
	call	cs_high			; set cs high
	ld	b,10			; sends 80 clocks
	ld a,$FF
l_init
	out (SPI_PORT),a
	djnz l_init
	nop
	
	call	cs_low			; set cs low
	
	ld a,MMC_GO_IDLE_STATE
	call mmc_write_command

	call mmc_wait_response
	cp $01				; MMC should respond 01 to this command
	jr nz,mmc_reset_failed		; fail to reset

	ld bc,120			; retry counter*256 (corrisponde a circa 5 secondi @3.5MHz)
mmc_reset_ok
	call cs_high			; set cs high
	ld a,$FF
	out (SPI_PORT),a		; 8 extra clock cycles
	nop
	nop
	
	call	cs_low			; set cs low
	
	ld a,MMC_SEND_OP_COND		; Sends OP_COND command
	call mmc_write_command

	call mmc_wait_response		; MMC_WAIT_RESPONSE tries to receive a response reading an SPI

	bit 0,a				; D0 SET = initialization still in progress...
	jr z,mmc_init_ok

	djnz mmc_reset_ok		; if no response, tries to send the entire block 254 more times
	dec c
	jr nz,mmc_reset_ok

	ld a,2				; error code for INIT ERROR
	jr mmc_errorx

mmc_init_ok
	call cs_high			; set cs high
	in a,(SPI_PORT)			; some extra clock cycles
	call pause1
	xor a
	ret
	
mmc_reset_failed			; MMC Reset error
	ld	a,1
mmc_errorx
	call	cs_high
	ret


;debug_display				; TEST: displays the code in A at current cursor position
;	push af				; ('A' based): A = 0x00, B = 0x01, etc.
;	push bc
;	push de
;	push hl
;	add 'A'
;	call disp_ch
;	ld bc,0
;lll
;	djnz lll
;	dec c
;	jr nz,lll
;	pop hl
;	pop de
;	pop bc
;	pop af
;	ret


;
;-----------------------------------------------------------------------------------------
; READ MMC CID subroutine. Data is stored at address (HL).
;
; Returns error code in A register:
; 0 = no error
; 1 = Read CID command error
; 2 = no wait_data token from MMC
;
; Destroys AF, HL.
;-----------------------------------------------------------------------------------------
mmc_get_cid
	push bc
	push de
	push hl
	ld hl,0
	ld de,0
	ld a,MMC_READ_CID
	call mmc_send_command
	pop hl
	cp 0
	jr z,cmd_cidok			; MMC_SEND_COMMAND reports 0 (ok) or MMC error code
	push af				; error code on STACK
	jr cid_exit
cmd_cidok
	call mmc_waitdata_token
	cp $FE
	jr z,waitda_cid_ok
	ld a,2
	push af
	jr cid_exit

waitda_cid_ok
	ld b,18				; 16 bytes + CRC?
	ld c,SPI_PORT

	inir

;mmc_cidl
;	ini
;	jr nz,mmc_cidl

	xor a
	push af
cid_exit
	call cs_high			; set cs high
	in a,(SPI_PORT)
	pop af				; error code
	pop de
	pop bc
	ret


;
;-----------------------------------------------------------------------------------------
; SEND COMMAND TO MMC subroutine
;
; A = COMMAND CODE;
; H, L, D, E = 32 bit parameter (MSB ... LSB);
;
; Sends a $FF fake checksum
;
; RETURNS: 0 = OK; != 0 = MMC error code
;
; On OK, the CHIP SELECT will be LOW on exit
; On error, the CHIP SELECT will be deasserted on exit
;
; Destroys AF.
;-----------------------------------------------------------------------------------------
mmc_send_command
	push bc
	ld c,a
	call cs_high			; cs high
	call clock32
	nop
	call cs_low			; cs low
	ld a,c				; command code is the first byte to be sent
	out (SPI_PORT),a
	ld a,h
	nop
	out (SPI_PORT),a
	ld a,l
	nop
	out (SPI_PORT),a
	ld a,d
	nop
	out (SPI_PORT),a
	ld a,e
	nop
	out (SPI_PORT),a
	ld a,$ff
	nop
	out (SPI_PORT),a

	call mmc_wait_response		; waits for the MMC to reply != $FF
	cp 0
	jr nz,mmc_commande
	pop bc
	ret				; 0 = no error

mmc_commande
	push af				; saves the error code
	call cs_high			; set cs high
	in a,(SPI_PORT)
	pop af
	pop bc
	ret				; returns the error code got from MMC


;
;-----------------------------------------------------------------------------------------
; WAIT FOR DATA TOKEN ($FE) subroutine (calls MMC_WAIT_RESPONSE up to 256 times)
; Returns with A = code read from MMC (ok if $FE). Destroys AF.
;-----------------------------------------------------------------------------------------
mmc_waitdata_token
	push bc
	ld b,10				; retry counter
mmc_waitl
	call mmc_wait_response
	cp $FE				; waits for the MMC to reply $FE (DATA TOKEN)
	jr z,exit_wda
	cp $FF				; but if not $FF, exits immediately (error code from MMC)
	jr nz,exit_wda
	djnz mmc_waitl
exit_wda
	pop bc
	ret


;
;-----------------------------------------------------------------------------------------
; SENDS BLOCK_SIZE COMMAND TO MMC.
;
; Parameters: DE = nbytes. Destroys AF.
;-----------------------------------------------------------------------------------------
mmc_send_blocksize
	call	cs_low			; set cs low
	ld a,MMC_SET_BLOCK_SIZE
	out (SPI_PORT),a
	xor a
	nop
	out (SPI_PORT),a
	nop
	nop
	out (SPI_PORT),a
	ld a,d
	nop
	out (SPI_PORT),a
	ld a,e
	nop
	out (SPI_PORT),a
	ld a,$ff			; fake checksum
	nop
	out (SPI_PORT),a
	nop
	nop

	in a,(SPI_PORT)
	nop
	nop
	in a,(SPI_PORT)
	nop
	nop
	call	cs_high
	
	ret


;
;-----------------------------------------------------------------------------------------
; Sends a command with parameters = 00 and checksum = $95. Destroys AF.
;-----------------------------------------------------------------------------------------
mmc_write_command
	push bc
	out (SPI_PORT),a	; sends the command
	ld b,4
	xor a
l_sendc0
	out (SPI_PORT),a	; then sends four "00" bytes (parameters = NULL)
	djnz l_sendc0
	ld a,$95		; $95 is only needed when the CARD INIT is being performed,
	nop
	out (SPI_PORT),a	; then this byte is ignored.
	pop bc
	ret


;
;-----------------------------------------------------------------------------------------
; Waits for the MMC to respond. Returns with A = response code; $FF = NO RESPONSE
;
; Responses from CARD are in R1 format for all command except SEND_STATUS.
; When SET, a bit indicates:
;
; D0 = Idle state / init not completed yet
; D1 = Erase Reset
; D2 = Illegal Command
; D3 = Com CRC Error
; D4 = Erase Sequence Error
; D5 = Address Error
; D6 = Parameter Error
; D7 = ALWAYS LOW
;
; Destroys AF.
;-----------------------------------------------------------------------------------------
mmc_wait_response
	push bc
	ld bc,50		; retry counter
l_response
	in a,(SPI_PORT)		; reads a byte from MMC
	cp $FF			; $FF = no card data line activity
	jr nz,resp_ok
	djnz l_response
	dec c
	jr nz,l_response
resp_ok
	pop bc
	ret


;
;------------------------------------------------------------------------------------
; CHIP_SELECT HIGH subroutine. Destroys no registers. Entire port is tied to '1'.
;------------------------------------------------------------------------------------
cs_high
	push af
	ld a,255
	out (OUT_PORT),a
	pop af
	ret

	
;
;------------------------------------------------------------------------------------
; CHIP_SELECT LOW subroutine. Destroys no registers. The card to be selected should
; specified in CARD_SELECT (D1 = SLOT1, D0 = SLOT0, active LOW)
;------------------------------------------------------------------------------------
cs_low
	push af
	ld a,(card_select)
	out (OUT_PORT),a
	pop af
	ret


pause1
	push hl
	ld hl,$8000		; OK for 3.5MHz ONLY (7MHz requires two calls or ld hl,0 and so on)
loop3	dec hl
	ld a,h
	or l
	jr nz,loop3
	pop hl
	ret


; VARIABILI IN CODA

snap_torestore	equ	$
	org $+2
entry_addr	equ	$
	org $+2


        end 

