;---------------------------------------------------------------------------------------------
; BOOTROM test program for ZXMMC+
;
; Can be uploaded into interface's RAM for testing purposes on bank 31
; Can be programmed into interface's ROM BANK 0 to act as a BOOT MANAGER. In this case, the
; whole bank self-copies to RAM BANK 31 in order to get RD/WR ram for variable space
;
; Version with TIMEOUT in RS-232 RX routine (start bit detect). Can generate a NMI on
; start bit detect; the handler is fast enough to receive the incoming byte in software,
; at 19K2 baud speed. This allows full ZX-Com compatibility.
;---------------------------------------------------------------------------------------------
;


;
; V2.20e 15/06/2008
;
; Added a function to backup and restore the 512K Flashrom into a 4-blocks sd-card snapshot,
; as already available for the ram chip. This allows quick flash programming when the zxmmc+
; is powered for the first time. The only block to be loaded by tape is the bootrom.


;
; V2.12e 24/03/2008
;
; The $7FFD register's lock bit is now set by default for any 16K and 48K snapshot.
; Small optimization in the 512K backup/restore routines


;
; V2.11e 11/02/2008
;
; Fastest video 'print character' routines by Eugenio Ciceri
; Squeezed out to the limit: now it takes 200 t-states less than v2.10e to print each character.
; Optimized to work with 5, 6 and 7 bits wide fonts.


;
; V2.10e 11/02/2008
;
; Faster video 'print character' routines by Eugenio Ciceri
; 'U' key now displayed in the power-on menu (ram backup)


;
; V2.10 03/02/2008
;
; FASTPAGE register is now saved and restored by rs-232 snapshots. This change required a
; new zx-com (and Badaloc firmware) release.
;
; Added a function for complete 512K ram backup/restore, using four 128K sd-card snapshot slots


;
; V2.00 02/02/2008
;
; Card initialization asks for an offset for sd-card utilization; snapshot header is now
; searched at multiple offsets. The CARD_OFFSET byte is displayed next to card's CID.
;
; All rd/wr routines refer to the card taking care of this offset. This way, che card
; may hold different partitions as long as they provide free space in the upper side.
;


;
; V1.41 19/01/2008
;
; Fixed a bug in the flashrom routine, which caused a fixed destination bank = 0
;


;
; V1.40 30/12/2007
;
; RS-232 16/48/128K snapshot functionality added! To restore a 128K snap, the machine should
; be running bootrom in order to ensure that the SP points to private, zxmmc+ onboard ram.
;
; The 'R' (receive block) and 'T' (send block) now work in the proper, requested RAM BANK
; for 128K machines. The 'T' (send block) command uses a separate, scratch STACK in order
; to work even while snapping a 128K program with SP on the upper (banked) area.
;
; The $7FFD set register now works. 7FFD and 1FFD is guessed on NMI handler entry.
; When a block of data is read or written, the selected ram bank is switched to the one
; requested in the header, then restored with known value.
; improvements.
;
; Every time the ram bank is switched, the I register content is saved then the register is
; set to ZERO before actual switch takes place. This avoids strange SNOW effects (and crashes).
; This was also the reason for strange screen behaviour during 128K SD-CARD snapshot writing,
; sometimes leading to system crash. Solved by resetting (and restoring) the I register.
;
; The ROM Flashing routine can now receive and program larger blocks (up to 64K).
;
; Bugfix in the guess_128 subroutine: using the stack during bank switch could have lead to
; failure in detecting an unlocked 7FFD register
;


;
; V1.33 28/12/2007
;
; Added a "bsod" (black screen of death) subroutine which displays a dump of register's content.
;
; Added a "retry" capability to the mmc_read_multidata. The routine may also call the BSOD to
; display register's content after an unsuccessful read attempt.
;


;
; V1.32 29/06/2007
;
; Fixed a bug in the 'ROM SNAPSHOT' which did not save the active ROM/RAM bank info.
;


;
; V1.31 12/06/2007
;
; Minor bugfixes;
; More friendly zx-com NMI handling when the spectrum is reset and zx-com think it's still logged-in
; Nested NMI no longer destroy the saved (active) FASTPAGE content's internal variable, allowing
; the user to perform the snapshot anyway (a nested NMI (bounce?) call will be signaled by RED border,
; but everything will be fine anyway).
;
; 48K ram is cleared before jumping into the 128K ROM0: surprisingly, this allows some 128K operation
; on rubber key 48K machines (128K menu, 128K editor and simple basic programs will work!)
;


;
; V1.30 11/06/2007
;
; The NMI handler is now capable of saving previous FASTPAGE register's content. Should be used
; in conjunction with the new version (2.00) of the patched 48K ROM, generated by MKSINCLA.ASM.
; In this new version, the FASTPAGE register is read and saved onto STACK by 48K rom before
; overwriting it with the BOOTROM paging value. This can be luckily done WITHIN the 14-bytes
; space available in the original 48K rom: $66 = start of NMI handler, $74 = start of next routine.
; The value is then popped from stack and saved in a local variable by bootrom firmware.
;
; To achieve this in just 14 bytes, a tricky jump is performed which uses the first byte of next
; instruction ($D3 - OUT) as a relative jump operand. Program execution of NMI handler continues
; at (spare) address $43 then jumps into the real routine.
;
; This allows returning to previously-active ROM BANK on NMI-handler exit.
;
; The saved FASTPAGE content is also written into SD-CARD snapshots (rs-232 have not been implemented yet)
; in a DEDICATED BYTE (offset [60]), which is loaded by "00" in ZX-Badaloc's snapshots. This is done because
; in zxmmc+ the fastpage register is the only ROM-mapping information, while in ZX-Badaloc this is handled
; by real ROM address data held in dedicated registers and FASTPAGE should not be loaded with values other
; than "00" unless the register was really in use. Snapshot created here will have "00" in the zx-badaloc
; fastpage byte.


;
; V1.21 09/06/2007
;
; 'GUESS128' subroutine added, which tries to determine $1FFD and $7FFD register's content
; by manipulating ram banks and trying to find determinate patterns, for snapshot purposes.
; The 'SPECIAL ADDRESS MODE' of $1FFD register is not recognized by now; only bit D2 is
; set accordingly with high bit ROM selection. (The low bit is on D4 of $7FFD).
; These registers are saved for compatibility with zx-badaloc as they are meaningless
; when restoring on zxmmc+: since snapshot save/restore can only be performed while NMI-patched
; rom are used, the FASTPAGE register is the only thing that needs to be properly restored.
;
; Fixed a bug that prevented restoring a snapshot on > 48K machines. The SCREEN 1 was selected
; and the register locked (due to a register READ, which is not possible on original hardware)
;
; Fixed a but into rs-232 'I' command, which used to send the wrong bank.
;


;
; V1.20 07/06/2007
;
; Added support to zx-com's 'I' command. This let download a RAM or FLASH bank from zxmmc+
; NOTE: due to time wasted processing the command decode chain, all functions should receive
; a single byte command code; then, zx-com should WAIT FOR ACKNOWLEDGE ('p' byte response)
; before sending any parameter.
;


;
; V1.10 05/06/2007
;
; Smaller changes to accomodate the NEW 48K/IF1 ROMS into banks 2 and 3 (the "USR-14500 and
; the SEA CHANGE ROM" was dropped).
;
; It should be noted that the "USR 14500" rom may be programmed into any other bank, then
; switched-in by a simple OUT 127,#bank BASIC command issued from the NMI-modified or even the
; internal basic ROM. The system won't crash as these rom are almost identical.
;


;
; V1.00 02/06/2007
;
; Full ZX-Badaloc's style SD-CARD snapshot management (currently only for the 16/48K Spectrum).
;
; Thanks to the on-board zxmmc+ ram, the boot menu with 8 sd-card snapshot entries is now
; displayed as on zx-badaloc and is capable of snapping/restoring 48K programs, as no sinclair
; memory is ever touched during the entire process.
;
; Any snapshot can be backed-up/restored/verified via ZX-Com RS-232 program (as on ZX-Badaloc).
;
; Added the capability to perform a single snapshot into FLASHROM (using last banks 29,30,31).
; This can be very useful to speed-up ResiDOS reinstallation. After first loading the installer
; (from tape), the user can snap the spectrum state with FLASH SNAP. If needed, ResiDOS installer
; can be reloaded in a eyeblink directly from the bootrom's main menu.
;
; ZX-Badaloc has an advanced NMI switching capability which automatically pages-in the bootrom
; firmware (15K) and dedicated scratchpad ram (1K from one of two spare 16K banks, unused on
; the 512K chip thanks to the additional dual-port 32K chip for video management) on $66 opcode
; fetch. Here on zxmmc+ there are no such hardware resources, but a good degree of compatibility
; can still be achieved by copying the BOOTROM bank (ROM 0) into last RAM bank ($1F) at power-on
; (last bank can no longer be used by ResiDOS). Then, the entire bootrom firmware runs into RAM,
; using the last KB as a private variable space. No sinclair ram is ever needed for any purpose
; (even the stack is moved into private ram during critical operation such as snapshot saving
; or restoring).
; The bootrom page-in issue is a little bit tricky, as the cpld is unable to detect the $66 fetch.
; For this thing to work, ROM 0 should hold this bootrom firmware while ROM 1 should contain a
; patched copy of the original Sinclair 16/48K (rubber or +) ROM. The patch consists of just 5
; bytes at address $66 (the NMI handler) and will PAGE-IN the RAM BANK $1F (this firmware).
; Here, the code is placed in such a way that execution can continue on BANK crossing, straight
; to the real NMI handler provided here. Jumping back to the sinclair rom works in the same way:
; the program should load the needed bank to be paged-in (usually ROM 1, for continued NMI
; service) into A register then jump to fixed location $6F. Here, a two-bytes OUT instructions
; will page-in the required bank. Both the NMI patched rom and ORIGINAL Sinclair ROM are
; suitable for this, as two bytes forward (at address $70) the processor will find a POP HL,
; POP AF, RETN. Of course, the caller should leave just these registers on stack. The snapshot
; restore routine does the same, and since the stack was just restored (as part of the
; entire ram restore process), the RETN will jump right at the place the processor was executing
; code at the time the snapshot was taken.
;
; 16/48K snapshot saved by ZX-Badaloc can be successfully loaded by a rubber keyed 48K spectrum,
; and vice-versa. As on ZX-Badaloc, ResiDOS is detected in nonvolatile RAM and a proper boot
; option ('R' key) is displayed.


;
; V0.30 23/05/2007
;
; FLASHROM routines added (from ZX-Badaloc's bootrom source code)
;
; From version V0.30, this firmware will correctly handle the 'F' (flash) RS-232 command.
; ZX-Com Win32 program can now program a ROM file into any of the 32 16K FLASHROM banks
; with a few mouse clicks.
;
; ZX-Com's separate command for bootrom firmware upgrade is working as well.


;
; V0.28 21/05/2007
;
; Diagnostic mode added. When the system is powered on with Kempston D4 input active (this can
; be easily done by a jumper, as D4 is next to the ground pin), a slow color cycle is shown
; on border, without ANY need of sinclair ram. In other words, this should work even with a
; total ram failure. If kempston D4 is found active for the whole loop, then the ROM 0 (this
; program) is copied to RAM 0 (this will destroy ResiDOS, but copying to a different bank is
; difficult without using sinclair ram as a buffer). Then, the program will run as always, but
; from this version on, it never relies on any sinclair ram location.
;
; Provided that Z80 can execute the program, this will allow to 'login' via RS-232 and zx-com
; and be able to read and write any memory region. Being able to dump memory content is more
; than a clue when trying to find out which ram chip failed.


;
; V0.27 10/05/2007
;
; SD-MMC card minimal support added. Now the firmware properly responds to ZX-Com 'Z' and 'X'
; commands as ZX-Badaloc does, for RAW DATA read/write to a PC file. The first found card is
; used. Cards are initialized if necessary when one of the above commands is received for the
; first time since power-on. The Card ID is displayed on the screen.


DEBUGMODE       equ     1       ; 1 = some functions will call the BSOD subroutine
FASTER          equ     1       ; 1 = some functions will run faster but could take much more memory
TOOFAST         equ     0       ; 1 = some functions will run too fast to be really accurate


; CPLD REGISTERS
;
KEMPSTON        equ     $1F
OUT_REG         equ     $1F
SPI_PORT        equ     $3F
FASTPAGE        equ     $7F

IF1_F7          equ     $F7
IF1_EF          equ     $EF


CS_0            equ     2       ; SLOT 0 select
CS_1            equ     1       ; SLOT 1 select
CS_DISABLE      equ     3
NMI_ENABLE      equ     8       ; D3 SET = NMI enabled (same register as CHIP SELECTS)

NMI_PATCHED_ROM equ     $62     ; FASTPAGE register's vaule for sinclair NMI patched ROM (48K machines!) $62 = BANK 2
NMI_PATCHED_128 equ     $67     ; FASTPAGE register's vaule for sinclair NMI patched ROM 1 of the 128 two rom set

BOOT_BANK       equ     $DF     ; RAM BANK for bootrom execution/variable space. DF = last (31) bank, RD/WR


CHAR_5     equ 5                ; larghezza dei caratteri nel font 51 cpl (che e` cmq 8x8)
MASK_5     equ $07              ; base bitmask (8 bit) for the 5 bits wide character
MASK_5_16  equ $07FF            ; base bitmask (16 bit) for the 5 bits wide character

CHAR_6     equ 6                ; larghezza dei caratteri nel font 42 cpl (che e` cmq 8x8)
MASK_6     equ $03              ; base bitmask (8 bit) for the 6 bits wide character
MASK_6_16  equ $03FF            ; base bitmask (16 bit) for the 6 bits wide character

CHAR_7     equ 7                ; larghezza dei caratteri nel font 36 cpl (che e` cmq 8x8)
MASK_7     equ $01              ; base bitmask (8 bit) for the 7 bits wide character
MASK_7_16  equ $01FF            ; base bitmask (16 bit) for the 7 bits wide character

CHAR_X     equ CHAR_6           ; larghezza dei caratteri scelta
MASK_X     equ MASK_6           ; base bitmask (8 bit) scelta
MASK_X_16  equ MASK_6_16        ; base bitmask (16 bit) scelta

BORDER  equ $FE                 ; PORT setup border sui bit D2 - D0 (gli altri riguardano SPKR/TAPE)
BORDCR  equ 23624               ; variabile SPECTRUM contenente il colore del border *8
ATTRIB1 equ 22528               ; first byte of attributes



; FLASHROM related defines

RAMBANK         equ     27      ; first of 4 16K banks in the zxmmc+ memory used as 64K backup for block erase.
                                ; leaves the LAST 16K bank untouched, as it may contain the BOOTROM

FLASHWR_CODE    equ     $A0     ; writing this code to the OUT_REG register will enable ROM CS on Z80 WR cycles
OUT_IDLE        equ     $F7     ; acceptable idle state for OUT_REG register



; SD/MMC RELATED DEFINES:

BLOCKSIZE               equ     512

USE_INI                 equ     1       ; 1 = code unrolling for maximum SD-CARD SPI port throughput


MMC_GO_IDLE_STATE       equ     $40
MMC_SEND_OP_COND        equ     $41
MMC_READ_CID            equ     $4A
MMC_SET_BLOCK_SIZE      equ     $50
MMC_READ_SINGLE_BLOCK   equ     $51
MMC_READ_MULTIPLE_BLOCK equ     $52
MCC_TERMINATE_MULTI_READ equ    $4C
MMC_WRITE_SINGLE_BLOCK  equ     $58
MMC_WRITE_MULTIPLE_BLOCK equ    $59
MMC_STOP_TRAN           equ     $FD




; SNAPSHOT RELATED DEFINES
;---------------------------------------------------------------------------------------------
;
FATSIZE         equ     8       ; dimensioni FAT in termini di blocchi da 64KBytes (in pratica e` l'offset
                                ; da caricare nella word MSB dell'indirizzo della SD per accedere al primo
                                ; cluster di dati, subito oltre la FAT). 8 = 512KBytes (8192 entries da 64 bytes)

MMCSIZE         equ     8192    ; dimensioni della MMC in termini di blocchi da 128K: 8192 = 1GB
                                ; Prima di prendere/ripristinare uno snapshot, il programmma verifica che l'ubicazione
                                ; dei dati non sia oltre il limite fisico, tenendo anche conto dello spazio occupato
                                ; dalla FAT. Con una FATSIZE = 8 (4 blocchi da 128K) e una MMC da 1GB (8192 blocchi)
                                ; sara` possibile gestire 8192-4 = 8188 snapshots.
                                ; NOTA: non tiene conto dell'eventuale uso di CARD_OFFSET, che potrebbe ridurre
                                ; sensibilmente il numero di snapshot. Si tratta cmq di un valore di riferimento


OFFSET_GRAN     equ     16      ; sd_check ora cerca l'header snapshot a piu' offset, spaziati di OFFSET_GRAN.
                                ; con 16 avremo salti di 256MB (per cui controlla a 0, +256, +512, etc

MAXOFFSET       equ     112     ; limite ricerca (compreso): con 48 cerca fino a +768MB, con 96 fino a +1,5GB


; DESCRIZIONE IPOTESI DI FILESYSTEM PER SALVATAGGIO DI SNAPSHOTS:
;
; Clusters = 128KBytes: possono contenere uno snapshot da 16/48/128K. Ciascun file ha lunghezza
; fissa pari ad 1 cluster.
;
; Primi 512KBytes contengono la FAT, con 64 bytes per entry, max 8192 files, secondo il seguente formato:
;
; [0-1]         Numero della prima ENTRY LIBERA (valido solo nel PRIMO FILE, negli altri e` inutilizzato)
; [2]           Codice del contenuto: 0 = cluster libero; 1 = SNAP16K, 2 = SNAP48K, 8 = SNAP128K
; [3-34]        32 bytes Filename
; [35-41]       7 bytes Snapshot date and time from RTC chip
; [42-43]       STACK POINTER
; [44]          I Register
; [45]          $7FFD CPLD register
; [46]          F register (INT enable, IM MODE, SPECTRUM_TYPE, DOUBLE_CLOCK BIT)
; [47]          $24DF CPLD register
; [48-57]       10 bytes joystick programming
; [58]          $1FFD CPLD register
; [59]          $54DF CPLD register (ZX-Badaloc FASTPAGE register)
; [60]          ZXMMC+ $7F register (ZXMMC+ FASTPAGE register, which is the only useful info about active ROM bank)
; [61-63]       spare
;
;
; Codici utilizzati correntemente per il byte [2]:
;
MMC_FREECLUSTER equ     0
MMC_SNAP16      equ     1       ; il cluster contiene uno snapshot da 16K
MMC_SNAP48      equ     2       ; snapshot 48K
MMC_SNAP128     equ     8       ; snapshot 128K
MMC_RAWDATA     equ     3       ; 128K raw data, readable and writable on 512-bytes basis (1 byte offset, 1 byte length)
MMC_RAMBACKUP   equ     4       ; first of 4 blocks for 512KB RAM backup (subsequent 3 blocks will be RAWDATA type)
MMC_FLASHBACKUP equ     5       ; first of 4 blocks for 512KB FLASH backup (subsequent 3 blocks will be RAWDATA type)


PLUS3REG equ    $1FFD           ; registro standard nello spectrum 128K +3
S128KREG equ    $7FFD           ; registro standard gestione banchi dello spectrum 128K

;********************************************************************************************************
;********************************************************************************************************


SHADOW_VARS     equ     $3D00   ; SHADOW_ROM space used as scratchpad rd/wr RAM.
                                ; This is also the amount of bytes copied from EPROM to SHADOW_ROM
                                ; when making a copy of BOOTROM firmware prior to switch
                                ; to SHADOW_ROM execution for any reason, in order to preserve
                                ; variables content

SHADOW_RAM      equ     $3E00   ; 512 bytes scratchpad for snapshot SD/MMC operations
;SHADOW_RAM     equ     $4000   ; USE VIDEO RAM (for test purposes)

SHADOW_STACK    equ     $3E00   ; stack in shadow_ram. SP will start writing at $3DFF
TEMP_STACK      equ     $3F00   ; SP location during critical operation ('T', sendblock with bank switching)


EXECUTE_VARS    equ     30000   ; Non- zxmmc+ram variable space (ROM flashing). STACK IS ALSO LOADED WITH THIS VALUE!
EXECUTE_RAM     equ     30100   ; Sinclair RAM from which we execute subroutines who need to swap BANKS
SCRATCH         equ     32768   ; Sinclair RAM 16K block receive buffer (for rom flashing)

NOULA_RAM       equ     32768   ; ram space where ULA can't bother (rs_232_tx execution for 'i' command)



; The following variables are allocated in the ZXMMC+ RAM space and are used by SD/MMC snapshot
; subroutines in order to leave all the spectrum memory untouched. THIS 256 BYTES PAGE IS SHARED
; WITH SHADOW_STACK.
;

        org SHADOW_VARS

residos_sign    equ     $       ; two bytes ResiDOS signature (should be 'R', 'D')
        org $+2
residos_version equ     $       ; two bytes ResiDOS version (BCD)
        org $+2
im_mode         equ     $       ; used to detect the IM MODE currently selected in the processor
        org $+1
nmi_count       equ     $       ; used to detect nested NMI calls
        org $+1
sh_flags        equ     $       ; various flags in SHADOW_RAM
        org $+1
whichbank       equ     $       ; tracks the active bank at NMI
        org $+1
boot_from       equ     $
        org $+1
cs_outreg       equ     $       ; found card chip select. both disabled = no cards found (yet)
        org $+1
x_pos           equ     $       ; coordinata X attuale posizione di scrittura (IN PIXEL, 0 - 255)
        org $+1
y_pos           equ     $       ; coordinata Y attuale posizione di scrittura (in caratteri, 0 - 23)
        org $+1
wr_video        equ     $       ; puntatore ram video (2 bytes)
        org $+2
attribute       equ     $       ; attributi usati dalle routines di scrittura a video
        org $+1
frames          equ     $       ; loop counter per ritardi vari (4 bytes)
        org $+4
wr_attribute    equ     $       ; puntatore all'area attributi dello schermo (routines di scrittura) (2 bytes)
        org $+2
tmp_str         equ     $
        org $+50
entry_addr      equ     $
        org $+2
real_sp         equ     $
        org $+2
snap_torestore  equ     $
        org $+2
snap_1ffd       equ     $
        org $+1
snap_7ffd       equ     $
        org $+1
snap_i          equ     $
        org $+1
guess_i         equ     $
        org $+1
firstfree       equ     $
        org $+2
card_offset     equ     $       ; bits 31-24 (4o MSB byte) of 32 bit absolute base card address for snapshots.
        org $+1                 ; 0 = begin of card; 16 = +256MB offset; 32 = +512MB offset; 48 = +768MB offset
start_item      equ     $
        org $+1
sd_entry        equ     $
        org $+2
flags2          equ     $
        org $+1
repeat_timer    equ     $
        org $+1
rewrite_timer   equ     $
        org $+1
saved_attrib    equ     $
        org $+1
startbit_state  equ     $
        org $+1
new_whichbank   equ     $
        org $+1
tmp_reg         equ     $
        org $+10



; SH_FLAGS DEFINES

IF1_enabled     equ     0       ; '1' = IF1 dip switch is enabled
sd_present      equ     1       ; '1' = sd_check found a card in currently selected slot
sd_initsnap     equ     2       ; '1' = the found card is initialized for snapshots
residos_present equ     3       ; '1' = ResiDOS signature was found into nonvolatile RAM


; FLAGS2 DEFINES

update_mmc      equ     0
key_pressed     equ     1
restore_attrib  equ     2
save_attrib     equ     3       ; used for snapshot selection
force_points    equ     4       ; used by build_timestring
sp_pressed      equ     5
suppress_leadz  equ     6       ; conv_4cifre will suppress leading zeroes


        org     EXECUTE_VARS


fastp_save      equ     $       ; used by flashrom routines to save FASTPAGE register's content
        org $+1
flashcompare    equ     $       ; flash compare result code
        org $+1
rom_64block     equ     $       ; flashrom block to be programmed
        org $+1
flash_towr      equ     $       ; nbytes to be programmed
        org $+2
ram_bank        equ     $
        org $+1
rom_bank        equ     $
        org $+1
shouldcopy      equ     $
        org $+1
moreblocks      equ     $
        org $+1
first_flashed   equ     $
        org $+1
backupmedia	equ	$	; make_512Kbackup: 0 = ram; 1 = rom
	org $+1



        org 0

        di
        ld sp,SHADOW_STACK      ; NOTE: This is in the ZXMMC+ RAM. STACK SHOULD NOT BE USED before
                                ; code is copied from ROM to RAM and this is accessible in RD/WR.
        jp reset_vector

        org $8
        ret

        org $10
        ret

        org $18
        ret

        org $20
        ret

        org $28
        ret

        org $30
        ret


        org $38                 ; MASKABLE INTERRUPT HANDLER: non deve e non puo`
        ei                      ; fare niente, potremmo appena aver abilitato gli INT
        reti                    ; dopo un ripristino SNAPSHOT



;
;------------------------------------------------------------------------------------------------
; NMI VECTOR
;
; The patched ROM requires 8 bytes to switch to our (this) bank, so the first executed
; instruction is JR ($D3). However, if BOOTROM was active when NMI occurred, then
; equivalent code is executed in order to save AF and FASTPAGE register content.
;
; To exit the NMI handler switching back to patched ROM, a JP $6F should be performed with
; 'A' holding the patchROM bank number (previously saved FASTPAGE register).
; This will 'CHAIN' to the patched ROM code so that execution will continue there, after
; the OUT (FASTPAGE),A is executed.
;
; NOTE: the patched ROM switches here by activating the RAM BANK 31 in RD/WR MODE.
;------------------------------------------------------------------------------------------------

        org $43                 ; this is pointed by the DEFB $18 followed by $D3 in the NMI vector handler
                                ; 61/63 T-states after NMI

        push bc                 ; 11T save extra registers
        ld bc,IF1_F7            ; 10T
        in a,(c)                ; 12T = 94/96 (was 83/85) T-states since NMI: START-BIT READ NOW! RS-232 RX pin on D7
        ld (startbit_state),a   ; 13T
        pop bc                  ; 10T (bit time = 182,3 T-states)

        jp nmi_handler          ; 10T

                                ; total time to JP to nmi_handler = 127/129 T-states, with start bit already READ.
                                ; time-offset from start_bit read = 33/35T-states



        org $66

        push af                 ; 11T 8 NOT executed bytes: we are still in PATCHED ROM BANK
        in a,(FASTPAGE)         ; 11T but if not, perform same action
        push af                 ; 11T fastpage content (our bank!) is saved as the patched 48K rom does
        nop                     ; 4T
        nop                     ; the patched rom takes 8 bytes to jump here (these 4 NOPs balance the
        nop                     ; LD A,# and OUT instructions)
        nop

                                ; total time HERE = 49T-states; from PATCHED-ROM = 51T-states

        defb $18                ; 12T? address $6E: the patched rom has just selected our bank. Follwing byte is
                                ; the OUT instruction ($D3) which will lead the jump to address $43.

                                ; NOTE: 61 (was 37) T-states to jump into handler. If patched ROM was active, they are 63 (was 39 in V1.00).
                                ; this means +24 T-states for saving FASTPAGE REGISTER (22) and JR (takes 12 instead of JP = 10) (!!)


; $6F                           ; NMI handler RETURN POINT: $6F. 'A' holds the ROM BANK we should jump into.
        out (FASTPAGE),a        ; Execution continues in the selected rom bank, which will POP AF then RETN.


                                ; This could not be even a single byte ahead, since in the 48K rom we have just 3 bytes
                                ; at address $71 before next subroutine starts, and we should perform POP AF, RETN (3 bytes).


;
;---------------------------------------------------------------------------------------------
; Get IM MODE Subroutine.
;
; Call with D = previous I state, that will be restored on exit.
;
; Loads IM_MODE with 1 or 2 depending on the actual IM_MODE of the processor. IM0 is detected
; as IM1. REQUIRES THE SHADOW_ROM TO BE ENABLED BEFORE CALLING.
;
; When IM_MODE 2 is detected, the border is turned MAGENTA.
;
; Destroys AF
;---------------------------------------------------------------------------------------------
get_immode

        ld a,1
        ld i,a                  ; restart vector for IM2 = $01xx
        ld (im_mode),a          ; default to IM1
        ei
        halt                    ; wait at least 1 INT
        di                      ; NOTE: INT will need to be re-enabled if IFF2 was SET
        ld a,d
        ld i,a                  ; restores I content
        ret



        ; The IM_2 vector table (points to $202 regardless of the value fetched from databus)

        org $100
        dw      $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202
        dw      $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202
        dw      $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202
        dw      $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202
        dw      $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202
        dw      $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202
        dw      $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202
        dw      $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202, $202
        dw      $202

        push af                 ; This instruction is located at address $202
        ld a,2                  ; If called, this routine loads "2" in IM_MODE and turns
        ld (im_mode),a          ; magenta the border. This will only happen if IM_2
        inc a                   ; was selected in the processor by the host program
        out (BORDER),a
        pop af
        reti                    ; exits without re-enabling interrupts


;
;------------------------------------------------------------------------------------------------
; NMI HANDLER ROUTINE
;
; 1 bit a 19200 = 52,0833us = 182.3 T-states @3.5MHz
; We enter this routine with an overhead of 127 (NMI within bootrom) or 129 (within patched rom) T.
;
; This routine takes 147 more T-states before calling rs_232_giabit for serial receiving,
; which is then entered after 274/276 T-states after NMI occurred. (1,5 bit = 273 T-states).
; time-offset from START-BIT READ = 180/182 T-states
;------------------------------------------------------------------------------------------------
; stack badaloc: AF, HL, BC, DE, BC', IY.
nmi_handler
        pop af                  ; 10T previous FASTPAGE register is now saved in a local variable
        ld (new_whichbank),a    ; 13T this is not the real WHICHBANK (a nested NMI call would overwrite
                                ; the correct data with BOOTROM BANK CODE!)

        push hl                 ; 11T ; stack content is now HL, AF, RETN

        ld hl,nmi_count         ; 10T increments NMI-nested calls counter
        inc (hl)                ; 11T

        push bc                 ; 11T
        push de                 ; 11T save extra registers. Now on stack = DE BC HL AF RETN

        ld bc,OUT_REG           ; 10T disables any further NMI from RS-232 LINE
        ld e,CS_DISABLE         ; 7T
        out (c),e               ; 12T

        ld a,(startbit_state)   ; 13T
        rla                     ; 4T D7 into carry
        jr nc,no_232rx          ; 7T if not met (no jump)

        call rs_232_giabit      ; 17T START BIT detected: we should receive a byte from rs-232 port
                                ; we call RS_232_GIABIT 274/276T-states after NMI. Bit time = 182,3 T-states

        ld a,(new_whichbank)    ; was not a 'button' NMI: we assume it can't be a nested call, so update
        ld (whichbank),a        ; whichbank immediately

        ld a,e                  ; received character
        cp $55
        jp z,rs_enter_ok        ; jumps if correct header byte was received
        ld a,'!'
        call rs_232_tx          ; returns error code

no_232rx
        ld a,(nmi_count)        ; we can now check for nested NMI calls
        out (BORDER),a          ; border will turn BLUE if a single NMI is being processed (as it should be)
        cp 1
        jp nz,exit_handler      ; jump if a nested NMI call was detected
        ld a,(new_whichbank)    ; otherwise, save the FASTPAGE content to the proper (real) variable
        ld (whichbank),a

        cp BOOT_BANK            ; FASTPAGE code for bootrom bank
        jr nz,w_keyclock        ; bootrom was NOT active: snapshot handling is 'legal'

        ld a,(16384)            ; a BUTTON-NMI while in bootrom is not legal: we will exit now
        inc a                   ; do something visible
        ld (16384),a

        jp exit_handler         ; exit immediately.





; SINGLE-KEY COMMANDS ON NMI HANDLING

w_keyclock
        call key_scan
        jr nz,w_keyclock        ; piu` di 2 tasti: ignora
        ld a,e
        cp $FF
        jr z,w_keyclock         ; no keys pressed

        call key_number         ; A = numero del tasto premuto (1 - 40)

        ld e,MMC_SNAP16         ; codice snapshot da 16KBytes
        cp 11                   ; [Q] tasto SNAPSHOT 16K
        jr z,takesnap

        ld e,MMC_SNAP48         ; codice snapshot da 48KBytes
        cp 12                   ; [W] tasto SNAPSHOT 48K
        jr z,takesnap

        ld e,MMC_SNAP128        ; codice snapshot da 128KBytes
        cp 13                   ; [E] tasto SNAPSHOT 128K
        jr z,takesnap           ; AL MOMENTO, INATTIVO.

        cp 14                   ; [R]
        jp z,rom_snapshot       ; takes a snapshot into last 3 ROM banks

        jr w_keyclock           ; attesa di un tasto valido (eventualmente mettere TIMEOUT)

filename
        db      "New SnapShot  ...               "

rtc_data
        db      0,0,13,6,2,6,7  ; dummy date: we don't have an rtc chip, as on ZX-Badaloc


;
; SAVING A SNAPSHOT subroutine
;
;

takesnap
        ld hl,sh_flags          ; if the SD CARD is not initialized for our proprietary snapshot scheme,
        bit sd_initsnap,(hl)    ; turn the border WHITE then exit immediately
        jr nz,card_ok
        ld a,7
        out (BORDER),a
        jp exit_handler

card_ok
        exx                     ; la strana sequenza di salvataggio ricalca quella usata nella routine UART,
        push bc                 ; al fine di mantenere una certa compatibilita` negli snapshot. Questi due
        push iy                 ; registri vengono usati attivamente nelle routines a seguire.

                                ; nello snapshot UART, tutti i registri che seguono non vengono salvati sullo
                                ; stack ma bensi` trasmessi e scritti nel file .snap, per minimizzare l'uso dello
                                ; stack stesso. Nel nostro caso, invece, li salviamo tutti sullo stack.
                                ; la sequenza e` stata rispettata, con la sola differenza che prima di AF'
                                ; viene trasmesso lo SP.

        ex af,af'               ; ALTRI REGISTRI, salvati per averli sullo stack e poterli poi ripristinare
        push af                 ; in fase di recupero dello snapshot stesso.
        ex af,af'
        push de
        push hl                 ; salvataggio secondo la sequenza con cui i registri vengono salvati nel file .snap
        exx                     ; (solo per avere una qualche forma di equivalenza qualora si decida di scrivere
                                ; un programma di conversione: sotto ogni altro punto di vista, e` inutile).

        push ix

        ld hl,0
        add hl,sp               ; SP in HL
        ld (real_sp),hl         ; saves to a shadow_rom variable
        ld sp,SHADOW_STACK

        ld a,i
        ld (snap_i),a
        xor a
        ld i,a

        call guess_128          ; tries to guess the $7FFD and $1FFD register's content

        ld a,(snap_7ffd)
        bit 5,a                 ; the LOCK BIT
        jr z,no_locked

;        res 5,a                 ; NOTE: By default, the LOCK BIT IS NOT SET INTO SNAPSHOT DATA.
;        ld (snap_7ffd),a        ; if needed, the user should manually SET IT.

        ld a,e                  ; requested snapshot type
        cp MMC_SNAP128          ; a 128K mode snapshot was selected?
        jr nz,no_locked         ; no, jump forward
        ld e,MMC_SNAP48         ; yes, BUT REGISTER IS LOCKED! turns it into a 48K snapshot
	jr no_forcelock

no_locked
	ld a,e
	cp MMC_SNAP128
	jr z,no_forcelock
	ld a,(snap_7ffd)
        set 5,a                 ; LOCK BIT IS SET INTO 16/48 SNAPSHOTS.
        ld (snap_7ffd),a

no_forcelock
        ld a,(snap_1ffd)
        bit 0,a                 ; D0 SET = modalita` di indirizzamento speciale: tutti i banchi dovranno essere salvati
        jr z,no_specialmode
        ld e,MMC_SNAP128        ; se e` attiva, forza uno snapshot da 128K
        xor a
        ld bc,PLUS3REG
        out (c),a               ; disattiva lo special mode: dovra` essere ripristinato prima di uscire
        ld a,4
        out (BORDER),a          ; segnala l'intervento mediante border verde

no_specialmode
        ld a,e                  ; codice dello snapshot (1 = 16K, 2 = 48K, 8 = 128K)
        cp MMC_SNAP16
        jr nz,no16ksnap         ; non e` 16K
        bit 7,h                 ; lo stack punta ad un'area oltre i primi 16K?
        jr z,proceed_snap       ; se e` basso, prosegue
        inc e                   ; altrimenti FORZA UNO SNAPSHOT 48K: se no, lo stack rimarrebbe fuori! (no clear 32599?)
        ld a,e
        out (BORDER),a          ; segnala l'intervento mediante BORDER ROSSO
        jr proceed_snap
no16ksnap
proceed_snap

        ld c,e                  ; saves the snap type
        call mmc_get_firstword  ; legge i primi 2 bytes della MMC --> DE, contenenti il numero della prima ENTRY LIBERA

        push de

; Calcola l'indirizzo di scrittura in SD/MMC per questo snapshot; DE = numero dell'entry


        ld l,FATSIZE            ; n. blocchi da 64KB che costituiscono la FAT (e` l'offset per la word MSB)
        ld a,(card_offset)
        ld h,a                  ; offset area snapshot nella card
        sla e
        rl d                    ; moltiplica per 2 (ogni entry corrisponde ad un cluster da 128KB)
        add hl,de               ; HL = word MSB scrittura in SD/MMC
        ld de,0                 ; word LSB = 0 (insieme a D0 = 0 della word MSB = 128K)

        ld a,c                  ; dimensioni dello snapshot (1 = 16K, 2 = 48K, 8 = 128K)
        ld iyl,c                ; backup size of snapshot

        ld b,16384/BLOCKSIZE    ; dimensioni per uno snapshot da 16K
        cp MMC_SNAP48           ; MMC_SNAP48 = 2
        jr nz,mmc_snapnow       ; 2 = 48K
        ld b,49152/BLOCKSIZE    ; e` da 48K


; Esegue lo snapshot

mmc_snapnow
        cp MMC_SNAP128
        jr z,snap128sd          ; salta se e` uno snapshot da 128K

        ld ix,$4000             ; ram buffer address
        call mmc_write_data     ; takes the snapshot (16/48K)
        pop de                  ; recovers the cluster number from stack
        and a
        jr z,read_entry_mmc     ; jumps if no error
        jp mmcwrite_error


snap128sd                       ; $7FFD register is not readable on real hardware!
        ld a,(snap_7ffd)
        and $F8                 ; RAM_BANK = 0
        ld c,a                  ; C = RAM_BANK counter, that keeps all other bits
l_128ksd
        ld a,c                  ; actual bank
        exx
        ld bc,S128KREG          ; BC' for CPLD addressing (mmc_read_data overwrites it)
        out (c),a               ; new ram_bank
        exx
        ld b,16384/BLOCKSIZE
        ld ix,$C000
        call mmc_write_data     ; saves this 16K bank
        jr nz,l_128k_error
        inc c
        ld a,c
        and 7
        out (BORDER),a          ; shows progress
        jr nz,l_128ksd

        ld a,(snap_7ffd)        ; original ram bank
        ld bc,S128KREG
        out (c),a
        pop de
        jr read_entry_mmc
l_128k_error
        pop de
        ld a,(snap_7ffd)        ; original ram bank
        ld bc,S128KREG
        out (c),a
        jp mmcwrite_error       ; termina



; Calcola l'indirizzo in cui si trova il blocco da 64 bytes che descrive questa entry (nella FAT) e ne legge
; il blocco (512 bytes)

read_entry_mmc
        push de                 ; salva di nuovo il numero dello snapshot: serve per riscrivere l'entry
        call calc_hldec         ; calcola l'indirizzo del blocco --> HL, DE e l'offset 64 bytes in C

        ld ix,SHADOW_RAM        ; scratchpad area
        call mmc_read_data      ; performs the single block read
        pop de                  ; clears the stack before possible jump
        and a
        jp nz,mmcwrite_error

        push de                 ; saves back to the stack (block number that contains the entry)


; Ora calcola l'offset di QUESTA entry all'interno dei 512 bytes del blocco letto (3 bit LSB = 8 entry da 64 bytes)

        ld d,c                  ; D = 3 bit LSB numero dell'entry
        ld e,0                  ; E = 0, quindi = entry * 256
        srl d                   ; divide /4: adesso DE e` un offset di 64 bytes per ogni valore unitario
        rr e                    ; che avevano i 3 bit LSB, ovvero l'offset della entry da 64 bytes
        srl d                   ; all'interno del blocco da 512 letto precedentemente
        rr e

        ld hl,SHADOW_RAM+2      ; NOTA: SALTA I PRIMI 2 BYTES, utili solo nella prima entry
        add hl,de               ; HL = puntatore al byte con offset [2] dell'entry da modificare

        ld e,iyl
        ld (hl),e               ; E = size of snapshot (1 = 16K, 2 = 48K, 8 = 128K) at offset +2
        inc hl
        ex hl,de
        ld hl,filename          ; copia 32 bytes di filename ad offset +3 (sostituire con il vero nome?)
        ld bc,32
        ldir

        ld hl,rtc_data          ; copia 7 bytes di data ad offset +35
        ld bc,7
        ldir

        ex hl,de

        ld ix,(real_sp)
        ld de,ix
        ld (hl),e               ; saves SP (2 bytes) at offset +42
        inc hl
        ld (hl),d
        inc hl

;       ld a,i
        ld a,(snap_i)
        ld i,a
        ld a,i
        push af                 ; LD A,I transfers IFF2 state (INT enable) in P/V FLAG (D2 in flag register)
        ld d,a                  ; saves I to D too
        ld (hl),a               ; saves I register at offset +44
        inc hl

        ld a,(snap_7ffd)        ; previously saved
        ld (hl),a               ; saves $7FFD register at offset +45
        inc hl

        call get_immode         ; loads IM_MODE with the actual IM MODE selected in the processor

        pop bc                  ; C = flag register
        ld a,c
        and 4                   ; keeps D2 (IFF STATE)
        ld c,a
        ld a,(im_mode)          ; IM MODE previously detected
        add c
        ld e,a                  ; E = result to be saved as F register

        ld a,iyl                ; IYL still holds the snapshot size
        cp MMC_SNAP128          ; 8 = 128K snapshot
        jr z,s128mode
        set 3,e                 ; selects 48K mode for spectrum_type bit (badaloc compatibility)
s128mode
;       ld a,(snap_34df)
;       bit 1,a                 ; D1 = INT doubler bit
;       jr z,no_doublei
;       set 4,e                 ; che deve essere salvato su D4 del registro 'F'
;no_doublei
;       and $60                 ; bit 5 and 6 should be copied to F register
;       or e                    ; (D5 = context_switch disable; D6 = all non-standard registers LOCK)
;       ld e,a
;       exx
;       ld a,l                  ; get the $24DF register content without our clock shift
;       exx
;       res 6,a                 ; disables the SHADOW_ROM WR BIT, although it's ignored at restore time
        ld (hl),e               ; saves IM ENABLE, IM MODE and SPECTRUM_TYPE at offset +46
        inc hl
        xor a                   ; FAKE
        ld (hl),a               ; saves $24DF register at offset +47
        inc hl

        ld iyl,e                ; IYL now holds the F register (for INT ENABLE STATUS)

        xor a
        ld b,10                 ; 10 bytes joystick programming are set to ZERO
l_clearbytes
        ld (hl),a
        inc hl
        djnz l_clearbytes

        ld a,(snap_1ffd)        ; original $1FFD register content
        and $1f
        ld (hl),a               ; saves it at offset +58
        inc hl
        ld bc,PLUS3REG
        out (c),a               ; restores the register content (previously set to zero in order to access all ram banks at $C000)

        ld bc,S128KREG
        ld a,(snap_7ffd)        ; previously saved (restores LOCK status)
        out (c),a

;       ld bc,FASTPAGE          ; $54DF register (zx-badaloc fastpage)
;       in a,(c)
        xor a                   ; FAKE: should be 0 for badaloc compatibility and is ignored when restored on zxmmc+
        ld (hl),a               ; saves it at offset +59
        inc hl

        ld a,(whichbank)
        ld (hl),a               ; +60: this is OUR fastpage register, ignored by ZX-Badaloc (who puts it = "00")
        inc hl

        xor a
        ld b,3                  ; 3 spare bytes are set to ZERO
l_sparebytes
        ld (hl),a
        inc hl
        djnz l_sparebytes


; Il blocco dati e` stato modificato: e` ora di riscriverlo nella SD/MMC

        pop de                  ; recupera l'indirizzo usato per la lettura
        call calc_hldec         ; calcola l'indirizzo del blocco --> HL, DE e l'offset 64 bytes in C
        ld b,1                  ; number of blocks to write
        ld ix,SHADOW_RAM        ; scratchpad area
        call mmc_write_data
        and a
        jr nz,mmcwrite_error


; A questo punto occorre aggiornare il numero della prima entry libera, che e` memorizzato nei due bytes
; iniziali della prima entry (ovvero i primi due bytes della SD/MMC). Notare che questo campo e` inutilizzato
; in tutte le altre entry.

; Lo incrementiamo +1

        ld de,0
        ld a,(card_offset)      ; MSB byte (bit 31-24) = snapshot area offset (1 = +16MB)
        ld h,a
        ld l,0
        ld ix,SHADOW_RAM        ; scratchpad area
        call mmc_read_data      ; performs the single block read
        and a
        jr nz,mmcwrite_error
        ld hl,(SHADOW_RAM)
        inc hl                  ; incremento a 16 bit
        call check_hlspace      ; se andiamo oltre il limite della SD/MMC, non incrementa
        jr nc,dontincr
        ld (SHADOW_RAM),hl
        ld de,0
        ld a,(card_offset)      ; MSB byte (bit 31-24) = snapshot area offset (1 = +16MB)
        ld h,a
        ld l,0
        ld b,1                  ; 1 block
        ld ix,SHADOW_RAM        ; scratchpad area
        call mmc_write_data     ; performs the block write
        and a
        jr nz,mmcwrite_error

dontincr
        ld a,$D5                ; test: wr ok
        ld ($5800),a
        jr end_write


mmcwrite_error
        ld a,18                 ; RED on RED
        ld ($5801),a            ; error

end_write
        ld hl,(real_sp)
        ld sp,hl

;       exx
;       ld bc,CLKREG
;       out (c),l               ; restores Z80 CLOCK
;       exx

;       ld bc,SPECTR_MODE
;       ld a,(snap_34df)        ; restores D6 content (non-standard register's lock)
;       out (c),a

        ld a,iyl
end_romsnap_entrypoint
        bit 2,a
        jr z,no_intenable
        and 3                   ; checks the IM-MODE
        cp 2
        jr nz,no_im2            ; IM-MODE is not IM2: safe to enable interrupts

        im 1                    ; IM2: if the host's interrupt handler deals with BASIC ROM,
        ei                      ; the program will crash as we are still running in BOOTROM.
        halt                    ; Shoots a couple of interrupts with IM1 mode selected, then
        halt                    ; switches back to IM2 with enough time for exiting the rom
        im 2                    ; before next interrupt will occur
        jr no_intenable
no_im2
        ei
no_intenable
        pop ix
        exx
        pop hl
        pop de
        ex af,af'
        pop af
        ex af,af'
        pop iy

        pop bc
        exx


; exiting handler

exit_handler
        pop de
        pop bc                  ; now only HL, AF, RETN are left on stack

        ld hl,sh_flags
        ld a,CS_DISABLE         ; both CS inactive
        bit IF1_enabled,(hl)    ; is IF1 capability enabled?
        jr z,no_rs232nmi
        or NMI_ENABLE           ; re-enables NMI from RS-232 as we will activate the ROM1 bank (nmi patched rom)
no_rs232nmi
        out (OUT_REG),a

        pop hl                  ; now only AF is left on the stack

        ld a,(nmi_count)        ; we now decrement the nested NMI calls counter and check if it reaches ZERO
        dec a
        ld (nmi_count),a
        and a   ; ------------------- could be useless
        jr nz,nested_exit       ; if not, we exit immediately by RETN, jumping into the 'main' NMI handler thread

        ld a,(whichbank)        ; FASTPAGE content at the time NMI occurred
        cp BOOT_BANK            ; was BOOTROM active when NMI occurred?
        jp nz,$6F               ; no, exit the handler and jumps back into one PATCHED ROM (held in 'A' register)

nested_exit
        pop af                  ; exit gracefully if NMI took place while BOOTROM was already active,
        retn                    ; or return to MAIN NMI CALL if a nested NMI was detected


;
;------------------------------------------------------------------------------------------------
; This subroutine tries to guess registers $7FFD and $1FFD content.
;
; NOTE: current version does not detect a SPECIAL ADDRESS MODE into $1FFD register:
; only D2 (high ROM address bit) is detected for this register.
;
; If more than one RAM BANK matches, then the LOCK BIT is assumed to be SET
; Turns cyan the border if unable to identify the selected ROM (!)
;
; Returns register's setting into snap_1ffd and snap_7ffd shadow_ram variables.
;
; Bit D3 of 7FFD register is not changed, as there is no way to determine it.
; This bit is only changed by the 'G' command (set the 7FFD register's content)
;
; Destroys: nothing
;------------------------------------------------------------------------------------------------
guess_128
        push af
        push bc
        push de
        push hl
        ex af,af'
        push af
        ex af,af'
        exx
        push bc
        push de
        push hl
        exx

        ld a,i
        ld (guess_i),a                  ; si accerta che I non sia > 0
        xor a
        ld i,a

        ld hl,49152                     ; start address of banked ram
        ld de,SHADOW_RAM                ; private buffer for ram content saving
        ld bc,16
        ldir                            ; marks current bank AND prepare relocated ROMFIND routine

        ld hl,guess_128                 ; first 16 bytes of this routine are the signature to seek for
        ld de,49152                     ; start address of banked ram
        ld bc,16
        ldir                            ; now the active ram bank is 'marked'


        ld hl,33000                     ; now finds some place to run the rom detect routine
        ld de,SHADOW_RAM+16
        ld bc,end_romfind-rom_find
        ldir

        ld hl,rom_find
        ld de,33000
        ld bc,end_romfind-rom_find
        ldir

        jp 33000

rom_find                                ; NOTE: this routine should not exceed (512-16) bytes
        in a,(FASTPAGE)
        ex af,af'
        xor a                           ; disable zxmmc+ PAGE-IN
        out (FASTPAGE),a                ; NOTE: STACK (in shadow ram) IS NO LONGER ACCESSIBLE

        ld de,$0010                     ; D is for $1FFD; E is for $7FFD. This value has D4 of E SET (ROM_1)
        ld hl,(0)                       ; first two bytes of currently active Sinclair ROM

        ld bc,PLUS3REG                  ; $1FFD write. WARNING: on 128K machines, this will corrupt $7FFD register!
        out (c),d
        ld bc,S128KREG
        out (c),e                       ; ROM_1 is now selected

        ld bc,(0)                       ; check for ROM 1
        and a
        sbc hl,bc
        jr z,found_rom
        add hl,bc

        res 4,e                         ; check for ROM 0
        ld bc,S128KREG
        out (c),e
        ld bc,(0)
        and a
        sbc hl,bc
        jr z,found_rom
        add hl,bc

        set 2,d                         ; check for ROM 2 (D = $1FFD register)
        ld bc,PLUS3REG
        out (c),d
        ld bc,S128KREG                  ; restores $7FFD content
        out (c),e
        ld bc,(0)
        and a
        sbc hl,bc
        jr z,found_rom
        add hl,bc

        set 4,e                         ; check for ROM 3
        ld bc,S128KREG
        out (c),e
        ld bc,(0)
        and a
        sbc hl,bc
        jr z,found_rom

        ld de,0                         ; NO ROM IDENTIFIED! (?)
        ld a,5                          ; cyan border
        out (BORDER),a

found_rom                               ; DE now holds bit A15 and A14 for ROM selection
        ex af,af'
        out (FASTPAGE),a                ; bootrom firmware and STACK are now accessible
        jp end_romfind
end_romfind

        ld a,d
        ld (snap_1ffd),a                ; $1FFD register has been determined (BIT D2 ONLY!!)

        ld d,0                          ; D is now a bank-is-equal counter
        ld bc,S128KREG                  ; now tries to determine the ACTIVE RAM BANK
l_seekram
        ld a,e                          ; E = $7FFD register
        out (c),a

        exx
        ld hl,guess_128
        ld de,49152
        ld bc,16
l_cmp_ram
        ld a,(de)
        cpi                             ; compare A against (HL), inc hl, dec bc
        jr nz,no_this                   ; mismatch: next bank should be checked
        inc de
        jp pe,l_cmp_ram                 ; jump if BC > 0
        exx
        ld a,d
        and a
        jr nz,no_firsttime
        ld a,(snap_7ffd)                ; D3 of 7FFD register (active video bank) is preserved
        and 8
        or e
        ld (snap_7ffd),a                ; BANK FOUND! (updated only on FIRST MATCH)
no_firsttime
        inc d
        exx                             ; balance

no_this
        exx
        inc e                           ; try another bank
        ld a,e
        and 7
        jr nz,l_seekram                 ; jump if another bank is to be checked

        ld a,(snap_7ffd)                ; selects first match
        out (c),a

        ld a,d                          ; saves # of matches

        ld hl,SHADOW_RAM                ; private buffer for ram content saving
        ld de,49152                     ; start address of banked ram
        ld bc,16
        ldir                            ; restore current bank content

        ld hl,SHADOW_RAM+16
        ld de,33000                     ; restore 'exec ram' content
        ld bc,end_romfind-rom_find
        ldir

        cp 2                            ; A = number of matching banks
        jr c,no_moret1                  ; no more than 1
        ld a,(snap_7ffd)
        set 5,a                         ; otherwise, set the REGISTER LOCK bit
        ld (snap_7ffd),a
no_moret1

        ld a,(guess_i)
        ld i,a

        exx
        pop hl
        pop de
        pop bc
        exx
        ex af,af'
        pop af
        ex af,af'
        pop hl
        pop de
        pop bc
        pop af
        ret


;
;------------------------------------------------------------------------------------------------
; ROM SNAPSHOT: the current 48K machine state is saved into 3 fixed ROM banks (29,30,31).
;
; Video ram $4000 - $47FF is used for program execution, while a few bytes from $4800 are
; SP and vital variable save space. Backup starts from $4800.
;------------------------------------------------------------------------------------------------
rom_snapshot
        exx
        push bc
        push iy
        ex af,af'
        push af
        ex af,af'
        push de
        push hl
        exx
        push ix

        ld hl,0
        add hl,sp               ; SP in HL
        ld ($4800),hl           ; saves stack content to VIDEO RAM +2KB
        ld sp,$47FF             ; stack is now moved to VIDEO RAM (will write from $47FE and back)

        ld a,i
        push af                 ; LD A,I transfers IFF2 state (INT enable) in P/V FLAG (D2 in flag register)
        ld d,a                  ; saves I to D too
        ld hl,$4802
        ld (hl),a               ; saves I register
        inc hl

        call get_immode         ; loads IM_MODE with the actual IM MODE selected in the processor

        pop bc                  ; C = flag register
        ld a,c
        and 4                   ; keeps D2 (IFF STATE)
        ld c,a
        ld a,(im_mode)          ; IM MODE previously detected
        add c
        set 3,a                 ; selects 48K mode for spectrum_type bit
        ld (hl),a               ; saves IM ENABLE, IM MODE and SPECTRUM_TYPE at address $4803
        inc hl

        call guess_128
        ld a,(snap_7ffd)
        res 5,a                 ; avoid BANK LOCK
        ld (hl),a               ; $7FFD content to $4804
        inc hl
        ld a,(snap_1ffd)
        ld (hl),a               ; $1FFD content to $4805

        ld a,(whichbank)
        inc hl
        ld (hl),a               ; saves ACTIVE BANK to $4806

        ld de,$4000
        ld hl,romsnap_relocate
        ld bc,end_romsnaprelocate
        ldir
        jp $4000

romsnap_relocate
;
; **** ERASING PHASE ***
;
;

        ld a,6
        out (BORDER),a          ; yellow border during ERASE

        ld a,$7C                ; ROM paged-in for read, last 4 banks
        out (FASTPAGE),a

        ld a,FLASHWR_CODE       ; this code will enable the ROM CS on Z80 write cycles
        out (OUT_REG),a

        ld hl,$555
        ld bc,$2AA
        ld e,$AA
        ld a,$55

        ld (hl),e               ; $555 <- $AA
        ld (bc),a               ; $2AA <- $55
        ld (hl),$80             ; $555 <- $80
        ld (hl),e               ; $555 <- $AA
        ld (bc),a               ; $2AA <- $55

        ld hl,0
        ld (hl),$30             ; block erase command (address don't really care: only 3 MSB 64K-BLOCK bits)


        ld e,8                  ; about 6 seconds timeout
        ld bc,0
l_waiter
        ld a,(hl)
        cp $ff
        jr z,erased_k
        dec c
        jr nz,l_waiter
        dec b
        jr nz,l_waiter
        dec e
        jr nz,l_waiter

        ld (hl),$F0             ; RESET command to flash rom (clears write error and switch to read mode)
        jp romsnap_exit
erased_k

;
; **** PROGRAMMING PHASE ****
;
;
;

        ld a,5
        out (BORDER),a

        exx
        ld hl,$555              ; quick access to program codes
        ld bc,$2AA
        ld e,$AA
        exx

        ld a,$7D                ; by now we don't take care of bank n. 28 ($7C) (erased but not reprogrammed).
                                ; to add BANK 28 saving, the program may copy it to the corresponding RAM BANK
                                ; (no sinclair ram can be used as a buffer) by activating ROM RD and RAM WR,
                                ; before performing the BLOCK ERASE, then here it should be added one more
                                ; step which saves back to rom USING SINCLAIR RAM AS A BUFFER, BUT ONLY AFTER
                                ; HAVING SAVED ALL 48K MEMORY.

        ld ($47FF),a            ; bank counter

        ld hl,$4800             ; buffer to be programmed (entire 48K Sinclair RAM)
        ld de,$800              ; A0:A13 address bit to ROM (16K) - remainder from PAGED-IN ROM BANK
        ld bc,16384-$800

next_ba
        ld a,($47FF)
        out (FASTPAGE),a        ; select ROM for read too (needed to check programmed byte)
        inc a
        ld ($47FF),a

program_16kk
        ld a,(de)               ; current ROM content
        cp (hl)                 ; desired content
        jr z,no_prog            ; byte OK already

        exx
        ld a,$55
        ld (hl),e               ; $555 <- $AA
        ld (bc),a               ; $2AA <- $55
        ld (hl),$A0             ; $555 <- $A0
        exx

        ld a,(hl)               ; byte to program
        ld (de),a               ; writes to ROM

        ex hl,de
        push bc                 ; wait for write completion (stack is located at address $4010)
        ld bc,0                 ; timeout counter
l_waitprog
        cp (hl)                 ; NOTE: comparison would be fine even if ram content changed,
        jr z,prog_doo           ; as the compared byte is the PREVIOUS ONE in 'A' register
        dec c
        jr nz,l_waitprog
        dec b
        jr nz,l_waitprog
        pop bc

        ld (hl),$F0             ; RESET command to flash rom (clears write error and switch to read mode)

        ld bc,0                 ; datasheet says the chip needs at least 10us before any read attempt.
l_resetwa                       ; we wait quite more.
        dec bc
        ld a,b
        or c
        jr nz,l_resetwa

        jp romsnap_exit         ; error

prog_doo
        pop bc
        ex hl,de
no_prog
        inc hl
        inc de
        dec bc
        ld a,b
        or c
        jr nz,program_16kk

        ld a,($47FF)
        and 3                   ; "00" on LSB bits means 4 banks programmed
        jr z,white_end

        ld de,0                 ; new 16K block (HL should advance so it's fine already)
        ld bc,16384
        jr next_ba

white_end
        ld a,7
        out (BORDER),a


romsnap_exit
        ld a,OUT_IDLE
        out (OUT_REG),a         ; disables ROM WR cycles
        ld a,BOOT_BANK          ; RAM BANK n. 31 selected in RD/WR mode
        out (FASTPAGE),a
        jp end_romsnaprelocate  ; bootrom bank is paged-in again: jump in it to complete execution

end_romsnaprelocate
        ld a,($4802)
        ld i,a
        ld hl,($4800)
        ld sp,hl
        ld a,($4803)            ; FLAGS register
        jp end_romsnap_entrypoint


;
;------------------------------------------------------------------------------------------------
; ROM SNAPSHOT: the current 48K machine state is RESTORED from 3 fixed ROM banks (29,30,31).
;------------------------------------------------------------------------------------------------
romsnap_restore
        ld hl,romsnaprestore_relocate
        ld de,$4000
        ld bc,end_romsnapre
        ldir
        jp $4000

romsnaprestore_relocate
        ld a,$7D                ; first of three ROM banks to restore
        out (FASTPAGE),a

        ld hl,$800              ; source buffer (first 2K are unused)
        ld de,$4800             ; destination address (we gracefully avoid to overwrite this program)
        ld bc,16384-$800
        ldir

        inc a                   ; next bank
        out (FASTPAGE),a
        ld hl,0
        ld bc,16384
        ldir

        inc a                   ; last bank
        out (FASTPAGE),a
        ld hl,0
        ld bc,16384
        ldir

        ld a,BOOT_BANK          ; RAM BANK n. 31 selected in RD/WR mode
        out (FASTPAGE),a
        jp end_romsnapre        ; bootrom bank is paged-in again: jump in it to complete execution

end_romsnapre
        ld a,7
        out (BORDER),a

        ld a,($4802)            ; I register saved into video ram
        ld i,a
        ld hl,($4800)           ; SP saved into video ram
        ld sp,hl

        ld a,($4804)            ; the $7FFD register content was saved here
        ld (snap_7ffd),a        ; may be not needed
        ld bc,S128KREG
        out (c),a

        ld a,($4805)            ; the $1FFD register content was saved here
        and $1f
        ld (snap_1ffd),a        ; may be not needed
        ld bc,PLUS3REG
        out (c),a

        ld a,($4806)            ; active bank when the snapshot was stored
        ld (whichbank),a

        ld a,($4803)            ; FLAGS register
        ld h,a                  ; should be in H
        jp romsnaprestore_entrypoint

;
;------------------------------------------------------------------------------------------------
; RS-232 entering sequence (4 bytes)
;------------------------------------------------------------------------------------------------
rs_enter_ok
        ld a,'*'
        call rs_232_tx          ; OK code

        ld h,4                  ; sequenza di altri 4 bytes identificativo entrata nella routine
        ld l,$65                ; codice del secondo byte
lcheckl
                                ; ENTER/EXIT rs_232_tx overhead = 50T-states (without the CALL)
        call rs_232_rx          ; 17T riceve il byte.
        jr nc,exit_node         ; 7T if not met
        ld a,e                  ; 4T
        cp l                    ; 4T
        jr nz,exit_node         ; 7T if not met
        sla l                   ; 8T terzo codice = $CA; poi $94, $28
        dec h                   ; 4T
        jr nz,lcheckl           ; 12T attende il prossimo byte

                                ; total LOOP time including rs_232_rx overhead = 113T-states. Bit time = 182.
                                ; this means that we re-enter the rs_232_rx shortly after the new start bit
                                ; is placed on the data line.

        exx
        push bc
        exx                     ; (BC' viene usato nella gestione dei comandi R e T)
        push iy                 ; stack: IY, BC', DE, BC, HL, AF.

;       ld a,$F9
;       ld (ATTRIB1),a          ; cubetto lampeggiante bianco/blu (test)

        ld a,(BORDCR)           ; colore attuale del border *8 (variabile del basic sinclair)
        rra
        rra
        rra
        dec a                   ; lo decrementa di 1
        and 7
        out (BORDER),a          ; gli altri bit possono essere ZERO (SPKR, TAPE)

        ld a,'*'                ; codice OK di conferma ricezione sequenza di entrata
        call rs_232_tx
        jr waitcmd

exit_node
        ld a,'!'
        call rs_232_tx
        jp exit_handler


;
;------------------------------------------------------------------------------------------------
; RS-232 Command receiving loop. Should exit via JP EXIT_HANDLER.
;
waitcmd
        call rs_232_rx
        jr nc,waitcmd

        ld a,e

;
;------------------------------------------------------------------------------------------------
; ESC command handler (NMI-handler exit).
;
; If we just took an rs-232 snapshot, then we know that the IM-MODE detection routine has
; overwritten (reset) the IFF2 interrupt enable flag, so maskable interrupt won't be enabled
; on NMI EXIT. This would lock the host program when the NMI handler will be quit, by
; pressing the LogOUT button on the zx-com PC software.
;
; To solve the problem, PC software checks if a take-snapshot with interrupt enabled took
; place. If this is the case, when the user presses the LogOUT button, a special ESC code is
; sent:
;
; ESC+1 (28) = re-enable interrupt;
; ESC+2 (29) = set IM2 MODE and re-enable interrupt.
;
; Of course, the command 29 is sent only if IM2 mode was detected by snapshot routines.
;
; There should be no need to deal with IM MODE. Hovewer, since it's not possible to just set
; the IFF2 flag (that would enable maskable interrupt on RETN execution), our only chance is
; to execute a real EI (that sets both IFF1 & IFF2).
; Unfortunately, if IM MODE was 2 AND the host's program interrupt handler relies on the
; zx-spectrum BASIC ROM, then it will crash because at least one interrupt will be shooted
; while we are still in the BOOTROM firmware.
;
; To solve this problem, if the host sends ESC+2, the IM-MODE is initially set to IM1 and EI
; is executed, followed by a couple of HALT instructions (that will halt the processor until
; a maskable interrupt is executed). Then, we have about 20ms (10 if interrupt works @100Hz)
; to set IM2 then exit the bootrom firmware. The host's maskable interrupt handler will then
; find the correct rom.
;---------------------------------------------------------------------------------------------

        cp 27
        jr z,esc        ; standard exit mode
        cp 28           ; ESC+1 = exit EI execution
        jr z,esc_ei1
        cp 29
        jr nz,no_esc

        ld a,'*'        ; we should set IM2 and enable interrupts
        call rs_232_tx  ; sends OK code now, to save time later
        im 1            ; IM1
        ei              ; enable interrupt
        halt            ; sync with INT hardware signal
        halt
        im 2            ; IM2: dangerous if the handler deals with BASIC ROM
        jr exit_sub     ; exit immediately, before a new INT will be generated by ULA

esc_ei1
        ei
esc
        ld a,'*'
        call rs_232_tx
exit_sub
        pop iy
        exx
        pop bc
        exx
        ld a,(BORDCR)           ; colore attuale del border *8 (variabile del basic sinclair)
        rra
        rra
        rra
        and 7
        out (BORDER),a          ; gli altri bit possono essere ZERO (SPKR, TAPE)
        jp exit_handler


no_esc
        cp 'S'                  ; ask registers
        jr nz,nos

l_sendreg
        call guess_128
;       ld a,16                 ; $7FFD: ROM A14 SET = ROM 1 (for compatibility with possible 128K machines: rom 1 = 48K basic)
        ld a,(snap_7ffd)
        call rs_232_tx
        xor a                   ; 24DF (fake)
        call rs_232_tx
        xor a                   ; 34DF (fake)
        call rs_232_tx
;       ld a,4                  ; 1FFD: ROM A15 set = ROM 2 (for compatibility with zx-badaloc 48K mode snapshot)
;       xor a
        ld a,(snap_1ffd)
        call rs_232_tx
        in a,(FASTPAGE)         ; 54DF (FASTPAGE; here it's $7F)
        call rs_232_tx
ok_waitcmd
        ld a,'*'
        call rs_232_tx
        jr waitcmd


;
;---------------------------------------------------------------------------------------------
; Gestione comando "G" (ricezione dall'host dati da scrivere nel registro 7FFD).
;
; GRAZIE AL CONTEXT SWITCH, possiamo scrivere anche nel bit D4 (LSB ROM_SELECT)
;
; Restituisce un codice di errore '$' se il registro risulta bloccato in scrittura.
;
; Distrugge AF, BC, E
;---------------------------------------------------------------------------------------------
nos
        cp 'G'
        jr nz,nog
        call rs_232_rx
        ld bc,S128KREG          ; Registro $7FFD
        out (c),e               ; scrive il dato nel registro
        ld a,e
        ld (snap_7ffd),a        ; lo salva nella nostra variabile

        jr ok_waitcmd


;
;---------------------------------------------------------------------------------------------
; Gestione comando "H" (impostazione registro $1FFD)
;
; Distrugge AF, BC, E
;---------------------------------------------------------------------------------------------
nog
        cp 'H'
        jr nz,noh
        call rs_232_rx
        ld bc,PLUS3REG
        out (c),e               ; scrive il dato nel registro
        ld a,e
        ld (snap_1ffd),a        ; e lo salva
        ld bc,S128KREG          ; Registro $7FFD
        ld a,(snap_7ffd)
        out (c),a               ; rinfresca il 7FFD: su macchine 128K la scrittura su 1FFD
        jp ok_waitcmd           ; viene presa per 7FFD, sporcando il registro


;
;---------------------------------------------------------------------------------------------
; Gestione comando 'Q' (trasmissione contenuto di tutti i registri che non abbiamo salvato
; sullo stack) ad uso SNAPSHOT.
;
; Ordine di trasmissione dei registri:
;
; SP    registro n. 1   2 bytes
; AF    sullo stack, non trasmesso
; BC    "
; DE    "
; HL    "
; AF'   registro n. 2   2 bytes
; BC'   sullo stack, non trasmesso
; DE'   registro n. 3   2
; HL'   registro n. 4   2
; IX    registro n. 5   2
; IY    sullo stack, non trasmesso
;
; I     registro n. 7 (1 byte)
; $7FFD CPLD (1 byte)
;
; F     altro byte per salvataggio stato IFF (1 byte)
; $24DF CPLD (1 byte)
;
; $1FFD CPLD (1 byte)
; $54DF CPLD (1 byte)
;
; $7F   CPLD (1 byte)
; dummy byte (devono essere pari)
;
; TOTALE = 18 bytes
;
; NUOVA VERSIONE che maschera il registro F a ZERO tranne D2 (stato IFF flag) e carica nei
; bit D1-D0 lo stato IM MODE, realmente rilevato.
; Tale rilevamento provoca l'azzeramento di eventuale IFF2, per cui non ci sara` riabilitazione
; degli INT all'uscita dall'handler NMI. A questo problema ovvia zx-com, che nel momento in cui
; l'utente decide di uscire dalla sessione (col bottone "LogOUT) provvede eventualmente a
; riabilitare gli interrupt.
;
; REGISTRO F:
; bit d1-d0 = IM MODE. 00 = IM 0, 01 = IM 1, 10 = IM 2.
; D2: 0 = DI, 1 = EI.
; D3: Spectrum mode 128/48K. Restituiamo SEMPRE ZERO, ci pensa il C++ a settarlo.
; D4: Double int status (proviene da D1 SPECTR_MODE)
; D5: Context_switch disable (proviene da D5 SPECTR_MODE)
; D6: All non-standard registers LOCKED (proviene da D6 SPECTRUM_MODE)
;
; Corrispondenza con i relativi bit nel registro SPECTR_MODE:
;
; F Register    D7      D6      D5      D4      D3      D2      D1      D0
; SPECTR_MODE   --      D6      D5      D1    not D0    --      --      --
;---------------------------------------------------------------------------------------------
noh
        cp 'Q'
        jp nz,noq

        ld iy,0                 ; checksum
        add iy,sp
        ld a,iyl
        call rs_232_tx          ; trasmette SP
        ld a,iyh
        call rs_232_tx

        ex af,af'
        push af
        ex af,af'
        pop bc
        add iy,bc
        ld a,c
        call rs_232_tx          ; trasmette AF'
        ld a,b
        call rs_232_tx

        exx

        add iy,de
        ld a,e
        call rs_232_tx          ; trasmette DE'
        ld a,d
        call rs_232_tx

        push hl                 ; salva per checksum
        ld a,l
        call rs_232_tx          ; trasmette HL'
        ld a,h
        call rs_232_tx

        exx

        pop bc
        add iy,bc               ; somma HL' al checksum

        push ix
        pop bc
        add iy,bc
        ld a,ixl                ; trasmette IX
        call rs_232_tx
        ld a,ixh
        call rs_232_tx

        ld a,i
        push af                 ; LD A,I ha trasferito lo stato di IFF2 nel flag P/V (D2 del flag register)
        ld e,a                  ; LSB somma checksum
        call rs_232_tx          ; trasmette I

        call guess_128          ; tries to guess the $7FFD and $1FFD register's content

        ld a,(snap_7ffd)
        ld d,a                  ; MSB checksum
        add iy,de               ; somma gli ultimi 2 bytes trasmessi
        call rs_232_tx

        ld d,e
        call get_immode         ; loads IM_MODE with 1 for IM0 and IM1, and 2 for IM2. WARNING: this will overwrite
                                ; the IFF2 flag, so INT won't be re-enabled on exit from NMI handler.

        pop bc                  ; C = flag register
        ld a,c                  ; trasmettiamo il registro dei flag
        and 4                   ; preserva il solo D2 (IFF STATE)
        ld c,a
        ld a,(im_mode)
        add c
        ld e,a

;       ld bc,SPECTR_MODE
;       in a,(c)
;       bit 1,a                 ; INT doubler bit
;       jr z,no_indou
;       set 4,e
;no_indou
;       and $60                 ; bit 5 and 6 should be just copied
;       or e
        ld e,0
        ld e,a
        call rs_232_tx          ; trasmette il registro F

;       ld bc,CLKREG
;       in a,(c)
        xor a
        ld d,a
        call rs_232_tx          ; trasmette il registro $24DF

        add iy,de               ; aggiorna il checksum

        ld a,(snap_1ffd)
        ld e,a                  ; LSB checksum
        call rs_232_tx          ; trasmette il registro $1FFD

        xor a
        ld d,a
        call rs_232_tx          ; trasmette il registro $54DF
        add iy,de               ; aggiorna il checksum

        ld a,(whichbank)        ; fastpage zxmmc+
        ld e,a
        call rs_232_tx

        xor a                   ; dummy byte (devono essere pari)
        ld d,a
        call rs_232_tx
        add iy,de               ; aggiorna il checksum

        ld a,iyl
        call rs_232_tx          ; trasmette LSB checksum
        ld a,iyh
        call rs_232_tx          ; trasmette MSB checksum

        jp ok_waitcmd


;
;---------------------------------------------------------------------------------------------
; Gestione comando 'W' (ricezione contenuto di tutti i registri che non abbiamo salvato
; sullo stack) + $7FFD, $1FFD e $54DF CPLD ad uso SNAPSHOT.
;
; Non ripristina F (non FLAG, ma quello usato per conoscere lo stato degli interrupt
; mascherabili al momento dello snap, gestito dal comando 'P').
;
; Chiamare PRIMA DI RIPRISTINARE LA MEMORIA.
;
; L'utilita` di settare lo SP ed il RAM_BANK prima di ricevere i blocchi di memoria consiste
; nel fatto che le routines di ricezione sporcano almeno 2 livelli di stack, ed e` quindi
; opportuno che cio` avvenga dove lo stack e` effettivamente presente (cioe` dove si trovava
; al momento dello SNAP).
;
; NOTA: nella presente versione, questo comando sposta lo stack nella ram della zxmmc+
; perche` diversamente non sarebbe possibile ricevere dati in rambank diversi: lo stack
; gia` in posizione e` incompatibile con la chiama a subroutine per RS_232_RX (infatti
; nello zx-badaloc viene gestito direttamente l'hardware senza chiamate a subtoutine.
; Lo stack del programma viene salvato in REAL_SP, da dove verra` ripristinato dal comando 'P'
; (che esegue le operazioni finali e salta nel programma snappato).
;
; E` cmq necessaria una operazione speciale per D5 del registro $7FFD (LOCK BIT):
; Non si deve bloccare adesso il registro, perche` il successivo comando 'P' avra` bisogno
; di scrivere in $1FFD. Per questo motivo, la presente routine salva il valore integrale
; nella variabile shadow "snap_7ffd" ed imposta il registro con D5 forzato basso.
; Il comando 'P', dopo che avra` opportunamente impostato $1FFD, provvedera` a traferire
; sul $7FFD il valore salvato nella variabile.
;
; E` importante relegare l'impostazione di $1FFD (e anche $54DF, fast_paging) al comando 'P'
; per poter effettuare il ripristino dei banchi di ram senza indirizzamenti speciali in corso.
; Anche in caso di indirizzamento speciale mediante 1FFD, l'intera ram dovra` essere scritta
; mediante il consueto ram_paging su $7FFD (su $C000 - $FFFF) per poi, alla fine, attivare
; la modalita` speciale su $1FFD.
;
;
; Ordine di trasmissione dei registri:
;
; SP    registro n. 1   2 bytes
; AF    sullo stack, non trasmesso
; BC    "
; DE    "
; HL    "
; AF'   registro n. 2   2 bytes
; BC'   sullo stack, non trasmesso
; DE'   registro n. 3   2
; HL'   registro n. 4   2
; IX    registro n. 5   2
; IY    sullo stack, non trasmesso
;
; I     registro n. 7 (1 solo byte)
; Registro CPLD $7FFD (1 byte)
;
; Totale 12 bytes
;---------------------------------------------------------------------------------------------
noq
        cp 'W'
        jp nz,now

        call rs_232_rx  ; riceve SP usando HL come area temporanea
        ld l,e
        call rs_232_rx
        ld h,e
        ld (real_sp),hl
;       ld sp,hl        ; in questo momento dovrebbe essersi ricostituita la situazione allo SNAP
        push hl
        pop iy          ; inizializza checksum

;       ld hl,SHADOW_STACK
;       ld sp,hl

        call rs_232_rx
        ld l,e          ; riceve AF' usando HL come area temporanea
        call rs_232_rx
        ld h,e
        push hl
        push hl
        ex af,af'
        pop af
        ex af,af'
        pop de
        add iy,de       ; somma al checksum

        call rs_232_rx  ; riceve DE'
        ld l,e
        call rs_232_rx
        ld h,e
        push hl
        exx
        pop de
        add iy,de
        exx

        call rs_232_rx  ; riceve HL'
        ld l,e
        call rs_232_rx
        ld h,e
        push hl
        push hl
        exx
        pop hl
        exx
        pop de
        add iy,de       ; somma al checksum

        call rs_232_rx  ; riceve IX
        ld ixl,e
        call rs_232_rx
        ld ixh,e
        push ix
        pop de
        add iy,de

        call rs_232_rx  ; riceve I
        ld a,e
        ld (snap_i),a   ; per qualche motivo, settare subito il registro I crea problemi in upload multibanco,
;       ld i,a          ; ovvero quelli su macchine 128K. Su switch in banco 1, va in crash con neve sul video.
        ld l,e          ; LSB CHECKSUM

        call rs_232_rx  ; riceve il registro $7FFD
        ld d,e          ; salva per il checksum
        ld e,l
        add iy,de       ; aggiorna il checksum
        ld a,d
        ld (snap_7ffd),a; salva il valore INTEGRALE del registro
        res 5,a         ; dopodiche` disattiva il LOCK BIT
        ld bc,S128KREG
        out(c),a        ; imposta il registro evitando il LOCK: verra` inserito dal comando 'P'

        call rs_232_rx
        ld l,e          ; parte LSB checksum trasmesso dal PC
        call rs_232_rx
        ld a,e
        cp iyh          ; confronta la parte MSB
        jr nz,rty_err
        ld a,l
        cp iyl
        jr nz,rty_err

        jp ok_waitcmd   ; termina con successo

rty_err
        ld a,'#'
        call rs_232_tx  ; codice di errore
        jp waitcmd      ; NOTA: I REGISTRI SONO COMPROMESSI!


;
;---------------------------------------------------------------------------------------------
; Gestione comando 'P': ricezione di F (abilitazione interrupt), $24DF, $1FFD, $54DF, $7F CPLD.
;
; Termina con un SALTO NEL PROGRAMMA SNAPPATO.
;
; I dati relativi ai registri presenti solo sullo zx-badaloc vengono scartati.
;
; In uscita, attiva la rom NMI PATCHED 16/48K. Per poter ricevere anche snapshot 128K e` opportuno
; identificare il tipo di snap ed eventualmente attivare la ROM 1 della coppia 16+16K del 128K.
;
;
; REGISTRI RICEVUTI DA QUESTO COMANDO:
;
; F     byte per salvataggio stato INTERRUPT e spectrum_mode = 1 byte
; $24DF CPLD = 1 byte (scartato)
; $1FFD CPLD = 1 byte (scartato)
; $54DF CPLD = 1 byte (scartato)
; $7F   CPLD = 1 byte (FASTPAGE)
; 1 dummy byte
;
;
; CONTENUTO DEL BYTE F:
;
; D1 - D0 = IM MODE (0, 1, 2)
;
; D2 = interrupt enable bit
;
;
; Bit validi solo sullo zx-badaloc:
;
; D3 = stato da caricare INVERTITO nello SPECTRUM_TYPE bit (D0 $34DF) (0 = 128K, 1 = 48K)
; (infatti quel bit deve essere 0 per selezionare il 48K)
;
; D4 = stato da caricare nel DOUBLE_INT bit (D1 $34DF) (0 = normale, 1 = double INT speed)
;
; D5 = '1' = context_switch disabled (no bootrom swap-in will occur on $66 opcode fetch)
;
; D6 = non-standard registers LOCK bit
;---------------------------------------------------------------------------------------------
now
        cp 'P'
        jr nz,no_p

        call rs_232_rx  ; riceve registro F per stabilire come settare IFF
        ld h,e          ; SALVA IN H, che non deve essere utilizzato dalle prossime funzioni

        call rs_232_rx  ; riceve il registro $24DF (e lo scarta)

        call rs_232_rx
        ld bc,PLUS3REG
        out (c),e       ; registro $1FFD

        ld a,(snap_7ffd); e` ora di impostare (se necessario) il LOCK BIT. Il valore integrale
        ld bc,S128KREG  ; del registro $7FFD e` stato salvato in questa variabile dal comando 'W',
        out (c),a       ; che lo ha impostato tutto tranne D5 (LOCK).
                        ; E` ANCHE IMPORTANTE riscrivere il $7FFD dopo il $1FFD, in quanto su
                        ; macchine 128K il 7ffd viene sovrascritto dal WR su 1ffd!!

        call rs_232_rx  ; registro $54DF: scartato

        call rs_232_rx  ; registro $7F
        ld a,e
        ld (whichbank),a

        call rs_232_rx  ; dummy byte

        ld a,'*'
        call rs_232_tx

        ld a,(BORDCR)   ; colore attuale del border *8 (variabile del basic sinclair)
        rra
        rra
        rra
        and 7           ; rimette a posto il colore originale
        out (BORDER),a  ; gli altri bit possono essere ZERO (SPKR, TAPE)

        ld iy,(real_sp) ; ripristina lo stack
        ld sp,iy

        pop iy
        exx
        pop bc          ; recupera BC'
        exx
        pop de          ; recupera DE
        pop bc

        ld a,CS_DISABLE ; ENABLE NMI ON RS-232 STARTBIT
        or NMI_ENABLE
        out (OUT_REG),a

        di              ; not really necessary, but who knows...
        im 1            ; default IM1
        bit 2,h         ; interrupt enable real status
        jr z,no_enei
        ei              ; this will execute our fake handler (EI, RETI)
        halt            ; sync with the hardware maskable interrupt generator
        halt
no_enei                 ; now we have 20 (or 10) ms before next interrupt will occur
        ld a,h          ; recupera lo stato di F dopo il LD A,I all'atto della creazione dello snapshot
        and 3           ; bit d1-d0 = IM MODE. 00 = IM 0, 01 = IM 1, 10 = IM 2.
        and a   ; ------------------- could be useless
        jr z,imzero
        cp 1
        jr z,completedei
        ld a,(snap_i)
        ld i,a
        im 2
        jr completedei
imzero
        im 0
completedei
        xor a
        ld (nmi_count),a

        pop hl

        ld a,(whichbank)
        and a                           ; the FASTPAGE info may be missing (example: zx-badaloc)
        jr nz,bank_ok
        ld a,NMI_PATCHED_ROM            ; default: NMI patched 16/48K ROM

                                        ; NOTA INTERESSANTE: se si tratta di un banco di tipo "ROM 48K", non c'e'
                                        ; bisogno che sia stato patchato per funzionare: all'indirizzo in cui
                                        ; vi saltiamo ($71) c'e' quello che serve per default: POP AF, RETN!

                                        ; (diversa e` la storia pre PRENDERE lo snapshot, caso in cui senza la patch
bank_ok                                 ; un NMI provoca un reset della macchina).
        jp $6F


;
;---------------------------------------------------------------------------------------------
; Gestione comando "R" (ricezione di un blocco)
;
; Distrugge AF, BC, BC', DE, HL
;
; Il banco richiesto in RX_DBLOCK viene caricato nel registro solo dopo il rientro dalla
; RX_DBLOCK stessa: lo stack potrebbe sparire. Per lo stesso motivo non usiamo la subroutine
; RS_RX ma una sua copia, in locale.
;---------------------------------------------------------------------------------------------
no_p
        cp 'R'                  ; codice "R" = riceviamo un blocco
        jr nz, nor

        call rx_dblock          ; HL = puntatore; BC = len; RAM_BANK voluto --> A
        ld e,a                  ; ram bank
        ld a,d                  ; flag di errore
        and a
        jr z,rxd_ok             ; salta se tutto OK
        jp rx_error

rxd_ok
        push bc
        ld a,e
        exx
        ld bc,S128KREG
        out (c),a               ; seleziona il RAMBANK
        pop bc                  ; sposta il contatore di bytes su BC'
        exx
        ld a,'*'                ; codice OK
        call rs_232_tx
        ld iy,0                 ; contatore di checksum

l_ric
        call rs_232_rx          ; 17T
        ld (hl),e               ; 7T salva il byte
        add iy,de               ; 15T somma al checksum (D e` zero se la ricezione e` andata a buon fine)
        inc hl                  ; 6T
        exx
        dec bc
        ld a,b
        or c
        exx
        jr nz,l_ric

        ld bc,S128KREG          ; saltiamo in questo rambank: probabilmente e` quello preesistente
        ld a,(snap_7ffd)
        out (c),a

        call rs_232_rx
        ld l,0
        ld a,e
        cp iyl                  ; confronta LSB
        jr z,rtr_ok1
        ld l,1                  ; flag di errore
rtr_ok1
        call rs_232_rx
        ld a,e
        cp iyh                  ; confronta la parte MSB
        jr nz,errex             ; errore

        ld a,l                  ; recupera il flag di errore
        and a
        jp z,ok_waitcmd         ; rientra con trasmissione di codice OK
errex
        ld a,'#'                ; codice di errore
        call rs_232_tx
        jp waitcmd


;
;---------------------------------------------------------------------------------------------
; Gestione comando "T" (trasmissione di un blocco)
;
; Distrugge AF, BC, DE, HL
;
; Il banco richiesto in RX_DBLOCK viene caricato nel registro solo dopo il rientro dalla
; RX_DBLOCK stessa: lo stack potrebbe sparire. Per lo stesso motivo non usiamo la subroutine
; RS_TX ma una sua copia, in locale.
;---------------------------------------------------------------------------------------------
nor
        cp 'T'                  ; codice "T" = trasmettiamo un blocco
        jp nz,not

        call rx_dblock          ; HL = puntatore; BC = len; RAM_BANK voluto --> A
        ld e,a                  ; ram bank
        ld a,d                  ; flag di errore
        and a
        jr z,rxd_ok2            ; salta se tutto OK
        jp rx_error
rxd_ok2
        ld (tmp_reg),hl
        ld hl,0
        add hl,sp
        ld (tmp_reg+2),hl
        ld hl,TEMP_STACK
        ld sp,hl
        ld hl,(tmp_reg)

        ld a,i
        ld (tmp_reg+4),a
        xor a
        ld i,a

        push bc
        ld bc,S128KREG          ; seleziona il banco desiderato
        out (c),e
        pop bc

        ld a,'*'                ; codice OK
        call rs_232_tx
        ld d,0                  ; DE usato per il calcolo del checksum
        ld iy,0                 ; contatore di checksum

l_tra
        ld a,(hl)               ; trasmette il byte
        ld e,a
        add iy,de               ; aggiorna il checksum
        call rs_232_tx
        inc hl
        dec bc
        ld a,b
        or c
        jr nz,l_tra

        ld a,iyl                ; trasmette LSB checksum
        call rs_232_tx
        ld a,iyh                ; trasmette MSB checksum
        call rs_232_tx

        ld a,(snap_7ffd)
        ld bc,S128KREG          ; torna sul banco precedente (o supposto tale)
        out (c),a

        ld hl,(tmp_reg+2)
        ld sp,hl

        ld a,(tmp_reg+4)
        ld i,a

        jp ok_waitcmd           ; trasmette OK e rientra in loop attesa comandi



;
;---------------------------------------------------------------------------------------------
; Subroutine che riceve i dati per il trasferimento. Per primo il RAM_BANK, poi address e len.
;
; RAM_BANK e` di competenza dello ZX-Badaloc; tuttavia, la routine di ricezione di un blocco
; da programmare in FLASHROM lo utilizza come numero di banco di destinazione.
; Rientra con:
;
; HL = indirizzo del blocco
; BC = lunghezza del blocco
;  A = rtx_rambank (usato come n. blocco da FLASH_PROG)
;
; Distrugge AF, BC, DE, HL
;---------------------------------------------------------------------------------------------
rx_dblock
        ld a,'p'
        call rs_232_tx  ; clear to receive parameters


        call rs_232_rx  ; riceve il primo byte di dati: e` il RAM_BANK
        ld a,e
        push af         ; rtx_rambank
        ld a,d
        and a
        jr nz,rx_der

        call rs_232_rx  ; riceve un byte
        ld a,d
        and a
        jr nz,rx_der
        ld l,e
        call rs_232_rx
        ld a,d
        and a
        jr nz,rx_der
        ld h,e          ; HL = puntatore al blocco
        call rs_232_rx
        ld a,d
        and a
        jr nz,rx_der
        ld c,e
        push bc
        call rs_232_rx
        pop bc
        ld a,d
        and a
        jr nz,rx_der
        ld b,e          ; BC = lunghezza blocco
rx_der
        pop af
        ret



;
;---------------------------------------------------------------------------------------------
; Gestione comando J, salto ad un indirizzo ricevuto come parametro. Il contesto va` perso
;---------------------------------------------------------------------------------------------
not
        cp 'J'
        jr nz,noj

        call rs_232_rx
        jp nc,rx_error
        ld l,e                  ; riceve 2 bytes: indirizzo del salto
        call rs_232_rx
        jp nc,rx_error
        ld h,e
        push hl                 ; indirizzo del salto sullo stack
        ld a,'*'
        call rs_232_tx          ; trasmette OK

        ld hl,nmi_count
        ld (hl),0

        ld a,CS_DISABLE
        or NMI_ENABLE           ; re-enables NMI from RS-232
        out (OUT_REG),a

        retn                    ; vedere di sostituire questa cosa con una modifica all'indirizzo
                                ; di rientro salvato sullo stack (se ha senso)



;
;---------------------------------------------------------------------------------------------
; Subroutine che TRASMETTE all'host un blocco da 512 bytes letti dalla SD/MMC.
; L'host deve trasmettere l'indirizzo del blocco da leggere. La SD/MMC e` indirizzata da un
; puntatore a 32bit, ma dato che leggiamo allineati ai 512 bytes l'host ne trasmette solo 24:
; si tratta del NUMERO DEL BLOCCO, che viene quindi SCALATO verso SX (moltiplicato *2) e poi
; caricato nei 3 bytes MSB dell'indirizzo (il byte LSB e` azzerato).
;
; Utilizza la SHADOW_RAM come buffer temporaneo
;---------------------------------------------------------------------------------------------
noj
        cp 'Z'
        jr nz,noz

        ld a,(cs_outreg)
        and 3
        cp 3
        call z,lookfor_sd       ; tries to locate a suitable SD-CARD
        ld a,(cs_outreg)
        and 3
        cp 3
        jp z,errex_mmc          ; no cards

        ld a,'*'
        call rs_232_tx          ; now we are ready to go (ZX-Com has been modified to wait for this)

        call rs_232_rx          ; byte MSB indirizzo richiesto
        ld h,e

        call rs_232_rx
        ld l,e

        call rs_232_rx
        ld d,e

        sla d                   ; moltiplica *2
        rl l
        rl h

        ld e,0                  ; byte LSB = 0. Adesso abbiamo l'indirizzo completo a 32bit
        ld ix,SHADOW_RAM
        call mmc_read_data
        and a
        jr z,r_sectok
        ld a,'#'                ; error code
        call rs_232_tx
        jp waitcmd
r_sectok
        ld a,'*'
        call rs_232_tx

        ld d,0                  ; DE usato per il calcolo del checksum
        ld iy,0                 ; contatore di checksum
        ld hl,SHADOW_RAM
        ld bc,BLOCKSIZE
l_tra_mmc
        ld a,(hl)
        ld e,a
        call rs_232_tx
        add iy,de               ; aggiorna il checksum
        inc hl
        dec bc
        ld a,b
        or c
        jr nz,l_tra_mmc

        ld a,iyl                ; trasmette LSB checksum
        call rs_232_tx
        ld a,iyh                ; trasmette MSB checksum
        call rs_232_tx

        jp waitcmd


;
;---------------------------------------------------------------------------------------------
; Subroutine che RICEVE dall'host un blocco da 512 bytes, da scrivere nella SD/MMC.
; L'host trasmette i 512 bytes di dati da scrivere, seguiti dal numero del blocco in SD/MMC
; piu` il checksum.
;
; Utilizza la SHADOW_RAM come buffer temporaneo
;---------------------------------------------------------------------------------------------
noz
        cp 'X'
        jp nz,nox

        ld a,(cs_outreg)
        and 3
        cp 3
        call z,lookfor_sd       ; tries to locate a suitable SD-CARD
        ld a,(cs_outreg)
        and 3
        cp 3
        jp z,errex_mmc          ; no cards

        ld a,'*'                ; codice OK
        call rs_232_tx


        ld iy,0                 ; contatore di checksum
        exx
        ld bc,BLOCKSIZE         ; contatore di bytes = BC'
        exx
        ld hl,SHADOW_RAM
l_ric_mmc
        call rs_232_rx
        ld (hl),e               ; salva il byte
        add iy,de               ; somma al checksum. se non ci sono stati errori, D = 0
        inc hl
        exx
        dec bc
        ld a,b
        or c
        exx
        jr nz,l_ric_mmc

        call rs_232_rx          ; byte MSB indirizzo richiesto
        ld h,e
        add iy,de               ; aggiorna il checksum

        call rs_232_rx
        ld l,e
        add iy,de               ; aggiorna il checksum

        call rs_232_rx
        add iy,de               ; aggiorna il checksum

        ld d,e                  ; questo byte deve andare in D (adesso che abbiamo finito con i checksum)
        push de                 ; DE viene distrutto da rs_232_rx

        call rs_232_rx
        ld a,e
        cp iyl                  ; confronta LSB
        jr z,rtr_ok1_mmc
        exx
        ld c,1                  ; flag di errore. BC' e` stato azzerato a fine loop di ricezione.
        exx
rtr_ok1_mmc
        call rs_232_rx          ; riceve MSB CHECKSUM
        ld a,e
        pop de                  ; recupera D
        cp iyh                  ; confronta la parte MSB
        jr nz,errex_mmc         ; errore

        exx
        ld a,c                  ; recupera il flag di errore da C'
        exx
        and a
        jr nz,errex_mmc

        sla d                   ; moltiplica *2
        rl l
        rl h
        ld e,0                  ; byte LSB = 0. Adesso abbiamo l'indirizzo completo a 32bit

        ld ix,SHADOW_RAM
        ld b,1                  ; blocchi da scrivere
        call mmc_write_data
        and a
        jr nz,errex_mmc
        jp ok_waitcmd

errex_mmc
        ld a,'#'                ; codice di errore
        call rs_232_tx
        jp waitcmd



;
;---------------------------------------------------------------------------------------------
; 16Kbytes FLASH ROM RECEIVE/PROGRAM routine
;---------------------------------------------------------------------------------------------
nox
        cp 'F'
        jp nz,nof

        ld a,1
        ld (shouldcopy),a               ; la ROM deve essere backuppata

another_bank
        call rx_dblock                  ; BC = len
        ld (rom_64block),a              ; banco da programmare (variabile in ram zxmmc+)
        ld a,d
        and a
        jr z,okf_rom                    ; salta se OK
        ld a,'#'                        ; codice di errore
        call rs_232_tx
        jp 0                            ; errore
;
; Parametro 'HL': normalmente ADDRESS del blocco da trasferire, viene usato per indicare che seguiranno altri blocchi
;                 e non si deve quindi programmare subito la flash
;
okf_rom
        ld (flash_towr),bc              ; idem
        ld a,l
        ld (moreblocks),a               ; 1 = seguono ulteriori blocchi, non programmare immediatamente

        ld sp,EXECUTE_VARS              ; stack fuori dall'area di esecuzione in RAM SINCLAIR (si estende verso il basso)

        ld a,(shouldcopy)               ; copia solo in occasione del primo (o unico) blocco da 16K
        and a
        jr z,dontcopy

        call cls
        xor a
        ld (x_pos),a
        ld (y_pos),a
        out (BORDER),a
        ld hl,backup                    ; ROM BACKUP message
        call disp_text

        ld hl,copy_2ram                 ; subroutine to be called: ROM to RAM 64KBYTES BACKUP
        ld bc,end_copy_2ram-copy_2ram   ; length
        call sinclair_execute           ; executes the subroutine into sinclair ram (as bank switch is needed!)
        xor a
        ld (shouldcopy),a

        ld a,(rom_64block)              ; salva il numero del primo (o unico) banco da programmare:
        ld (first_flashed),a            ; serve per la visualizzazione prima della conferma

dontcopy
        ld hl,flashreceive              ; DATA RECEIVING
        xor a
        ld (x_pos),a
        ld a,(y_pos)
        inc a
        ld (y_pos),a
        call disp_text

        ld a,'*'                        ; codice OK
        call rs_232_tx

        exx
        ld bc,(flash_towr)              ; contatore di bytes su BC'
        exx
        ld iy,0                         ; contatore di checksum

        ld hl,SCRATCH                   ; buffer di ricezione

l_ricflash
        call rs_232_rx                  ; 17T
        ld (hl),e                       ; 7T salva il byte
        add iy,de                       ; 15T somma al checksum (D e` zero se la ricezione e` andata a buon fine)
        inc hl                          ; 6T
        exx
        dec bc
        ld a,b
        or c
        exx
        jr nz,l_ricflash

        call rs_232_rx
        ld l,0
        ld a,e
        cp iyl                          ; confronta LSB
        jr z,rtrflash_ok1
        ld l,1                          ; flag di errore
rtrflash_ok1
        call rs_232_rx
        ld a,e
        cp iyh                          ; confronta la parte MSB
        jr nz,errexflash                ; errore

        ld a,l                          ; recupera il flag di errore
        exx
        and a
        jr z,ok_flash                   ; nessun errore: procede
errexflash
        ld a,'#'                        ; codice di errore
        call rs_232_tx
        jp 0                            ; situazione irrecuperabile: riavvia.

ok_flash
        ld hl,anyrom                    ; SCRATCH --> proper zxmmc+ RAM BANK, for (flash_towr) bytes
        ld bc,end_anyrom-anyrom
        call sinclair_execute

        ld a,'*'                        ; OK code
        call rs_232_tx

        ld a,(moreblocks)               ; se ci sono altri blocchi da ricevere, salta
        and a
        jp nz,wait_another

        call flash_screen1              ; displays user screen for eprom flashing

w_keyflash
        call key_scan
        jr nz,w_keyflash                ; piu` di 2 tasti: ignora
        ld a,e
        cp 8                            ; N key
        jp z,0                          ; reboot: ormai la ram e` compromessa
no_no
        cp 2                            ; Y key
        jr nz,w_keyflash

        ld a,5
        ld (y_pos),a

	call do_64K			; 64K compare/erase/program/verify

        ld hl,flash_programok
        ld (x_pos),a
        ld a,20
        ld (y_pos),a
        call disp_text
        ld hl,flash_final
        ld (x_pos),a
        ld a,21
        ld (y_pos),a
        call disp_text

        ld a,1                          ; un eventuale nuovo invio richiede il backup della ROM in RAM
        ld (shouldcopy),a

wait_another
        call rs_232_rx
        jr nc,wait_another              ; no byte received
        ld a,d
        and a
        jr nz,wait_another              ; framing error

        ld a,e                          ; received command
        cp 'F'
        jr nz,wait_another
        jp another_bank


flash_screen1
        call cls                ; queste scritte verranno opportunamente rivelate dal chiamante, che
        xor a                   ; essendo rilocato in ram non potra` piu` accedere alla DISP_TEXT.
        ld (y_pos),a
        ld (x_pos),a
        ld hl,flashmsg1
        call disp_text
        ld a,(first_flashed)    ; n. blocco ricevuto per primo
        ld c,a
        push bc
        call disp_hex
        pop bc
        ld a,(rom_64block)      ; n. blocco ricevuto per ultimo
        cp c
        jr z,single_blockflash  ; salta se il primo ed ultimo blocco coincidono
        ld hl,flashmsg1bis
        push af
        call disp_text
        pop af
        ld c,a
        call disp_hex

single_blockflash
        ld a,2
        ld (y_pos),a
        xor a
        ld (x_pos),a
        ld hl,flashmsg2
        call disp_text
        ret


flashmsg1
        db      7,"About to FLASH ROM_BANK $",0
flashmsg1bis
        db      7," - $",0
flashmsg2
        db      7,"Proceed? (Y/N) ",0

backup
        db      7,"Backing-up 64K ROM Content...",0
flashreceive
        db      7,"Receiving Data...",0

programming
        db      7,"Programming...",0
comparing
        db      7,"Comparing...",0
erasing
        db      7,"Erasing...",0
flash_programok
        db      7,"ROM Successfully Flashed. Please RESET",0
flash_final
        db      7,"or send another ROM now (no other cmd).",0
flash_program_error
        db      7,"Programming Error! Please Reset ",0



;
;---------------------------------------------------------------------------------------------
; 64K Flash compare / erase / program / verify.
; rom_64block should contain the number of base block (0-31, 4 increments = 64K)
; RAMBANK (4 banks) should hold data to be programmed
;---------------------------------------------------------------------------------------------
do_64K
        ld hl,comparing                 ; COMPARING
        xor a
        out (BORDER),a
        ld (x_pos),a
        ld a,(y_pos)
        inc a
        ld (y_pos),a
        call disp_text
        ld hl,compare                   ; subroutine address
        ld bc,end_compare-compare       ; length
        call sinclair_execute           ; executes the subroutine into sinclair ram (as bank switch is needed!)
        and a
        ret z                           ; current content is equal!
        cp 1
        jr z,go_program                 ; directly programmable


        ld hl,erasing                   ; ERASING
        xor a
        out (BORDER),a
        ld (x_pos),a
        ld a,(y_pos)
        inc a
        ld (y_pos),a
        call disp_text
        ld hl,erase                     ; subroutine address
        ld bc,end_erase-erase
        call sinclair_execute


go_program
        ld hl,programming               ; PROGRAMMING
        xor a
        out (BORDER),a
        ld (x_pos),a
        ld a,(y_pos)
        inc a
        ld (y_pos),a
        call disp_text
        ld hl,program
        ld bc,end_program-program
        call sinclair_execute
        jr do_64K


;
;---------------------------------------------------------------------------------------------
; This subroutine sends (or receives) a 16K bytes block, accessing memory through FASTPAGE
; register. Up to 1 MegaByte (512KB ROM and 512KB RAM) can then be accessed.
;
; The main loop is relocated into video ram in order to access the requested bank, which
; will appear on ROM space (0 - $3FFF).
;
; FASTPAGE, CLKREG and CONTEXT SWITCH are altered but restored on exit.
; However, REGISTER ARE UNLOCKED and lock is NOT restored.
;---------------------------------------------------------------------------------------------
nof
        cp 'I'
        jp nz,noi

        ld a,'p'
        call rs_232_tx  ; clear to receive parameters

        call rs_232_rx  ; receives the byte to be written to the FASTPAGE register
        ld a,e
        exx
        ld l,a          ; L' = desired bank
        exx
        and $E0         ; at least one of these three bits should be HIGH
        and a   ; ------------------- could be useless
        jr nz,valid_bank
        ld a,'!'        ; error code
        call rs_232_tx
        jp waitcmd

valid_bank
        bit 7,a
        jr z,no_ramwrite
        ld a,'#'                ; receiving a ram block is currently not supported
        call rs_232_tx
        jp waitcmd

no_ramwrite
        ld a,'*'
        call rs_232_tx          ; OK

        ld hl,reloc_fastpaging
        ld de,NOULA_RAM
        ld bc,back_rom-reloc_fastpaging
        ldir

        ld hl,rs_232_tx         ; we need this subroutine
        ld de,NOULA_RAM+$100
        ld bc,end_rs232tx-rs_232_tx
        ldir
        jp NOULA_RAM            ; execute from Sinclair RAM

reloc_fastpaging
        ld hl,0
        add hl,sp
        push hl
        exx
        pop de                  ; SP in DE'
        ld sp,NOULA_RAM+$100
        ld bc,FASTPAGE
        in h,(c)                ; H' = current fastpage content
        out (c),l               ; L' = desired bank (previously saved)
        exx                     ; now address space 0 - $3FFF contains the desired memory space

        ld d,0                  ; DE is used for checksum computation
        ld iy,0                 ; IY is the checksum register
        ld bc,16384             ; bytes to be sent
        ld hl,0                 ; address pointer

l_sendpage
        ld a,(hl)
        ld e,a
        call NOULA_RAM+$100     ; the rs_232_tx subroutine
        add iy,de               ; checksum
        inc hl
        dec bc
        ld a,b
        or c
        jr nz,l_sendpage

        exx
        out (c),h               ; FASTPAGE content is restored
        ex hl,de
        ld sp,hl                ; SP is restored
        exx

        jp back_rom             ; now we can jump back to firmware romspace
back_rom
        ld a,iyl                ; LSB checksum
        call rs_232_tx
        ld a,iyh                ; MSB checksum
        call rs_232_tx

        jp waitcmd              ; end of job



noi
        ld a,'?'                ; unknown command, but within the RS-232 NMI LOOP
        call rs_232_tx
        jp waitcmd


rx_error
        ld a,'#'        ; trasmette '#' (fuori sincrono)
        call rs_232_tx
        jp waitcmd      ; rientra in loop


;
;------------------------------------------------------------------------------------------------
; RS-232 RX frontend
;
; It waits for a START_BIT (0,9 sec timeout). The timeout overhead can be up to 61T-states
; in worst case.
;
; Returns:
; E = received character if D = 0
; D = 1 if stop bit was not detected (framing error)
;
; Start bit detect:
; Min time (asserted exactly on IN) = 47T-states to reach l_delstart;
; Max time (asserted just after IN) = 84T-states to reach l_delstart
; Average = 65,5T
;
; Destroys AF, BC, DE
;------------------------------------------------------------------------------------------------
rs_232_rx
        ld bc,IF1_F7            ; 10T
        ld de,0                 ; 10T timeout counter. 65536 = 0,9 sec
l_waitstart
        in a,(c)                ; 12T
        rla                     ; 4T
        jr c,startfound         ; 7 if not met

        dec de                  ; 6T
        ld a,e                  ; 4T
        or d                    ; 4T
        jr nz,l_waitstart       ; 12T if met; 7 otherwise. 49T states/loop! = 14us
        xor a                   ; reset carry
        ld d,1                  ; error
        ret

startfound                      ; here we come after START BIT DETECTION
        ld b,14
l_delstart
        djnz l_delstart

                                ; 65+8+7+10+12 = 102 (il primo 8 e` il ciclo finale del DJNZ L_DELSTART)
                                ; 273 (1,5 bit) - 102 = 171; 171/13 = 13,15 so value to load = 14

                                ; for some reason, the correct value is 13 instead of the 14 computed.
                                ; perhaps due to overhead in receiving routine (calling routine)
                                ; NOTA: FORSE PERCHE` NON TENEVA CONTO DEL JR PER ENTRARE IN GIABIT (ora eliminato)


                                ; 13*13 + 8 = 177T-states
                                ; Total time (with average delay = 65) = 65+177+7+10+12 = 271T
                                ; 1+1/2 bit time = 273T

;------------------------------------------------------------------------------------------------
; rs_232_giabit START_BIT ALREADY FOUND ON NMI ENTRY POINT
;
; called when start bit was already detected and with a delay of 180/182T-states
; from it's middle. Since 1 bit = 182.3 T-states, we should start receiving IMMEDIATELY!

rs_232_giabit

        ld d,8                  ; 7T bit counter - this adds undesired delay
        ld bc,IF1_F7            ; 10T I/O pointer

rx_newbit
        in a,(c)                ; 12T
        rla                     ; 4T D7 (data in line) into carry
        ccf                     ; 4T
        rr e                    ; 8T receive register

        ld b,10                 ; 7T
        nop                     ; 4T

l_rxbit
        djnz l_rxbit            ; 9*13T + 8T = 125T

        dec d                   ; 4T
        jr nz,rx_newbit         ; 12T if met, 7 otherwise. Total = 180T-states /cycle

;       nop                     ; 4T
        in a,(c)                ; stop bit
        rla                     ; 4T
        jr nc,rx_doneok         ; 12T if met
        ld de,0x100             ; D = 1 = framing error
        scf                     ; something (or error) was received
        ret
rx_doneok

;       xor a
;       out (IF1_EF),a          ; TEST
;       ld a,$ff
;       out (IF1_EF),a

        ld d,0                  ; 7T D = 0 = no errors, E = data
        scf                     ; 4T something (or error) was received
        ret                     ; 10T


;
;------------------------------------------------------------------------------------------------
; RS-232 19200 baud TX routine. The DTR input line is ignored and TX takes place anyway.
; Bit time is 182.3T-states
;
; Destroys AF
;------------------------------------------------------------------------------------------------
rs_232_tx
        push bc
        push de
        xor $ff
        ld e,a                  ; byte to be sent
        ld d,9                  ; bit counter (1 start + 8 data bits)

        ld a,$FF
        out (IF1_EF),a          ; selects RS-232 and also ties the CTS HIGH
        ld bc,IF1_F7            ; I/O port full 16 bit address (avoids any ULA clock issue)

        scf                     ; start bit is '1' output (as it is negated by the MAX232)
next_txbit
        rla                     ; 4T carry --> D0
        out (c),a               ; 12T bit in output

        ld b,11                 ; 7T
l_bittx
        djnz l_bittx            ; 10*13T + 8T = 138T

        rr e                    ; 4T
        dec d                   ; 4T
        jr nz,next_txbit        ; 12T if met; 7T if not. 181T/bit

        ld a,0                  ; 7T
        out (c),a               ; 12T stop bit

        ld b,13                 ; 7T
l_bitstop
        djnz l_bitstop          ; 12*13T + 8T = 164T

        pop de
        pop bc
        ret
end_rs232tx


;
;---------------------------------------------------------------------------------------------
; This front-end will JUMP into the subroutine specified in HL, BC = length, after copying the code to sinclair RAM.
; The subroutine should take care of FASTPAGE register handling (if altered, should be restored before exit).
; The subroutine should terminate with a RET, which will jump back to the caller.
;
; ANY ROUTINE CALLED THIS WAY should place it's variable into Sinclair RAM!
;
sinclair_execute
        ld de,EXECUTE_RAM
        ldir
        jp EXECUTE_RAM


;
;---------------------------------------------------------------------------------------------
; This subroutine will copy a given ROM block from ROM to RAM (64Kbytes).
; Returns: nothing.
; Base destination RAM block is specified in RAMBANK equ.
;
; SHOULD BE COPIED TO SINCLAIR RAM BEFORE EXECUTE!
;---------------------------------------------------------------------------------------------
copy_2ram
        in a,(FASTPAGE)
        ld (fastp_save),a       ; FASTPAGE content is saved

        ld a,RAMBANK            ; base destination bank
        set 7,a                 ; WR enabled

        ex af,af'
        ld a,(rom_64block)      ; desired ROM bank (0-31)
        and $1C                 ; 2 LSB bit should be ignored
        or $60                  ; ROM paged-in
        ex af,af'               ; A' = rom bank

        ld b,4                  ; number of 16K blocks

next_btoram
        ex af,af'
        out (FASTPAGE),a        ; rom is selected
        inc a
        push bc

        ld hl,0                 ; copy to sinclair ram
        ld de,SCRATCH
        ld bc,16384
        ldir

        ex af,af'
        out (FASTPAGE),a        ; RAM is selected in WR mode
        inc a

        ld hl,SCRATCH           ; copy from sinclair ram to zxmmc+ ram
        ld de,0
        ld bc,16384
        ldir

        pop bc
        djnz next_btoram

        ld a,(fastp_save)
        out (FASTPAGE),a        ; FASTPAGE content is restored
        ret
end_copy_2ram


;
;---------------------------------------------------------------------------------------------
; This subroutine will compare a given ROM block with RAM content (64Kbytes).
;
; Returns in A:
; 0 = Identical; 1 = Mismatch which can be programmed; 2 = mismatch which needs BLOCK ERASE.
;
; SHOULD BE COPIED TO SINCLAIR RAM BEFORE EXECUTION!
;---------------------------------------------------------------------------------------------
compare
        in a,(FASTPAGE)
        ld (fastp_save),a               ; FASTPAGE content is saved
        xor a
        ld (flashcompare),a             ; return code = OK

        ld a,RAMBANK
        set 6,a                         ; RD enabled

        ex af,af'
        ld a,(rom_64block)              ; desired ROM bank (0-31)
        and $1C                         ; 2 LSB bit should be ignored
        or $60                          ; ROM paged-in
        ex af,af'                       ; A' = rom bank

        ld b,4                          ; number of 16K blocks

next_bcompare
        ex af,af'
        out (FASTPAGE),a                ; rom is selected
        inc a
        push bc

        ld hl,0
        ld de,SCRATCH
        ld bc,16384
        ldir

        ex af,af'
        out (FASTPAGE),a                ; RAM is selected in RD mode
        inc a

        push af                         ; COMPARE LOOP
        ld hl,0                         ; PAGED-IN RAM pointer
        ld de,SCRATCH                   ; scratchram (ROM CONTENT) pointer
        ld bc,16384
l_compare
        ld a,(de)                       ; this is ROM content byte, which was copied to sinclair RAM
        cp (hl)                         ; compare with RAM buffer (paged-in)
        jr z,byte_ok
        and (hl)                        ; are there '0' bits which should be turned into '1'?
        cp (hl)
        jr z,programmable               ; no, this byte can be programmed without erasing the 64K block
        ld a,2
        ld (flashcompare),a             ; 2 = ERASE is needed
        pop af                          ; can't be worse: exit immediately
        pop bc
        jr exit_compare
programmable
        ld a,1
        ld (flashcompare),a             ; 1 = ROM content mismatch, but still programmable
byte_ok
        inc hl
        inc de
        dec bc
        ld a,b
        or c
        jr nz,l_compare
        pop af

        pop bc
        djnz next_bcompare

exit_compare
        ld a,(fastp_save)
        out (FASTPAGE),a                ; FASTPAGE content is restored

        ld a,(flashcompare)             ; return code
        ret
end_compare


;
;---------------------------------------------------------------------------------------------
; This subroutine will program a given ROM block (64Kbytes).
;---------------------------------------------------------------------------------------------
program
        exx
        ld hl,$555              ; quick access to program codes
        ld bc,$2AA
        ld e,$AA
        exx

        in a,(FASTPAGE)
        ld (fastp_save),a       ; FASTPAGE content is saved

        ld a,(rom_64block)      ; desired ROM bank (0-31)
        and $1C                 ; 2 LSB bit should be ignored
        or $60                  ; ROM page in
        ld (rom_bank),a

        ld a,RAMBANK            ; first bank of ram to be programmed
        set 6,a                 ; RAM RD enabled
        ld (ram_bank),a

        ld a,FLASHWR_CODE       ; this code will enable the ROM CS on Z80 write cycles
        out (OUT_REG),a

next_bank
        ld a,(ram_bank)
        out (FASTPAGE),a
        inc a
        ld (ram_bank),a

        ld hl,0                 ; reads the current 16K ram buffer into scratchpad ram
        ld de,SCRATCH
        ld bc,16384
        ldir

        ld a,(rom_bank)
        out (FASTPAGE),a        ; select ROM for read too (needed to check programmed byte)
        inc a
        ld (rom_bank),a

        ld hl,SCRATCH           ; buffer to be programmed
        ld de,0                 ; A0:A13 address bit to ROM (16K) - remainder from PAGED-IN ROM BANK
        ld bc,16384

program_16k
        ld a,(de)               ; current ROM content
        cp (hl)                 ; desired content
        jr z,no_program         ; byte OK already

        exx
        ld a,$55
        ld (hl),e               ; $555 <- $AA
        ld (bc),a               ; $2AA <- $55
        ld (hl),$A0             ; $555 <- $A0
        exx

        ld a,(hl)               ; byte to program
        ld (de),a               ; writes to ROM

        ex hl,de
        push bc                 ; wait for write completion
        ld bc,0                 ; timeout counter
l_waitprogram
        cp (hl)
        jr z,prog_do
        dec c
        jr nz,l_waitprogram
        dec b
        jr nz,l_waitprogram
        pop bc

        ld (hl),$F0             ; RESET command to flash rom (clears write error and switch to read mode)

        ld bc,0                 ; datasheet says the chip needs at least 10us before any read attempt.
l_resetwait                     ; we wait quite more.
        dec bc
        ld a,b
        or c
        jr nz,l_resetwait

        ld bc,1                 ; program error
        jr program_exit

prog_do
        pop bc
        ex hl,de
no_program
        inc hl
        inc de
        dec bc
        ld a,b
        or c
        jr nz,program_16k

        ld a,(rom_bank)
        and 3                   ; "00" on LSB bits means 4 banks programmed
        jr nz,next_bank

        ld bc,0                 ; OK code
program_exit
        ld a,OUT_IDLE
        out (OUT_REG),a         ; disables ROM WR cycles

        ld a,(fastp_save)
        out (FASTPAGE),a        ; FASTPAGE content is restored

        ret
end_program


;
;---------------------------------------------------------------------------------------------
; This subroutine will erase a given ROM block (64Kbytes).
;---------------------------------------------------------------------------------------------
erase
        in a,(FASTPAGE)
        ld (fastp_save),a       ; FASTPAGE content is saved

        ld a,(rom_64block)      ; desired ROM bank (0-31)
        and $1C                 ; 2 LSB bit should be ignored
        or $60                  ; ROM paged-in for read
        out (FASTPAGE),a

        ld a,FLASHWR_CODE       ; this code will enable the ROM CS on Z80 write cycles
        out (OUT_REG),a

        ld hl,$555
        ld bc,$2AA
        ld e,$AA
        ld a,$55

        ld (hl),e               ; $555 <- $AA
        ld (bc),a               ; $2AA <- $55
        ld (hl),$80             ; $555 <- $80
        ld (hl),e               ; $555 <- $AA
        ld (bc),a               ; $2AA <- $55

        ld hl,0
        ld (hl),$30             ; block erase command (address don't really care: only 3 MSB 64K-BLOCK bits)


        ld e,8                  ; about 6 seconds timeout
        ld bc,0
l_waiterase
        ld a,(hl)
        cp $ff
        jr z,erased_ok
        dec c
        jr nz,l_waiterase
        dec b
        jr nz,l_waiterase
        dec e
        jr nz,l_waiterase

        ld (hl),$F0             ; RESET command to flash rom (clears write error and switch to read mode)

        ld c,1                  ; timeout error
        jr erase_exit

erased_ok
        ld c,0                  ; OK code

erase_exit
        ld a,OUT_IDLE
        out (OUT_REG),a         ; disables ROM WR cycles

        ld a,(fastp_save)
        out (FASTPAGE),a        ; FASTPAGE content is restored
        ret
end_erase


;
;---------------------------------------------------------------------------------------------
; This subroutine will copy a previously loaded ROM from SCRATCHPAD spectrum ram to the
; specified (0-3) ram buffer
;---------------------------------------------------------------------------------------------
anyrom
        in a,(FASTPAGE)
        ld (fastp_save),a               ; FASTPAGE content is saved

        ld a,(rom_64block)
        and 3
        add RAMBANK+128                 ; RAM bank offset + D7 set (WR enable)
        out (FASTPAGE),a

        ld hl,SCRATCH                   ; reads data from SCRATCHPAD area
        ld de,0
        ld bc,(flash_towr)
        ldir

        ld a,(fastp_save)
        out (FASTPAGE),a                ; FASTPAGE content is restored
        ret
end_anyrom


;
;---------------------------------------------------------------------------------------------
; This test subroutine will copy 16K from the given ROM BANK (0-31) to SCRATCHPAD spectrum ram
;---------------------------------------------------------------------------------------------
romcopy
        in a,(FASTPAGE)
        ld (fastp_save),a       ; FASTPAGE content is saved

        ld a,(rom_64block)      ; desired ROM BANK (0-31!)
        and $1F
        or $60                  ; ROM is paged-in
        out (FASTPAGE),a

        ld hl,0
        ld de,SCRATCH
        ld bc,16384
        ldir

        ld a,(fastp_save)
        out (FASTPAGE),a        ; FASTPAGE content is restored
        ret
end_romcopy

;
;------------------------------------------------------------------------------------------------
; MAIN PROGRAM ENTRY POINT
;
; If program is executing from ROM BANK 0, then the first 15KBytes are copied to RAM BANK 31
; then control is transferred there. If not, we already are in RAM BANK 31 (debug session?).
; NOTE: ResiDOS signature is correctly detected only when executing from ROM.
;
; The last KB is not affected as it's the variable area, except first 4 bytes (ResiDOS signature)
;
; STACK SHOULD NOT be used before executing code in RAM.
;
; IF KEMPSTON D4 IS FOUND ACTIVE, then the Diagnostic mode is entered (see top of file)
;------------------------------------------------------------------------------------------------
reset_vector
        in a,(FASTPAGE)
        cp $60                  ; are we running on ROM BANK 0? (power-on)
        jr z,copy_it            ; yes: should copy to ZXMMC+ RAM
        ld a,1
        ld (boot_from),a        ; probably a RAM debug session: signals it for menu string selection
        jp g_ahead


copy_it
        in a,(KEMPSTON)         ; TEST FOR DIAGNOSTIC MODE REQUESTED
        cp 16
        jr nz,no_diagnostic
        ld bc,0                 ; loop counter
l_diagno
        in a,(KEMPSTON)
        cp 16
        jr nz,no_diagnostic     ; kempston released: exit diagnostic mode
        dec c
        jr nz,l_diagno
        ld a,b
        rlca
        rlca
        rlca
        and 7
        out (BORDER),a          ; slow border cycling
        and 1
        or 2
        out (OUT_REG),a         ; SD-CARD SLOT 0 flashing led
        dec b
        jr nz,l_diagno

        ld a,$E0                ; now entering DIAGNOSTIC MODE: read FLASHROM, write RAM
        out (FASTPAGE),a

        ld hl,0                 ; full bank copy from FLASHROM to RAM, same bank (0)
        ld de,0
        ld bc,SHADOW_VARS       ; NOTE: 15K. Leaves alone the LAST KILOBYTE (variable space)
        ldir

        ld a,$C0                ; RAM BANK n. 0 selected in RD/WR mode
        out (FASTPAGE),a        ; FROM NOW ON, STACK CAN BE USED. VARIABLES CAN BE USED.


        ld bc,0                 ; displays a quick border cycling to tell that rom->ram copy
l_dispborder                    ; succeeded
        dec c
        jr nz,l_dispborder
        ld a,b
        and 7
        out (BORDER),a
        and 1
        or 2
        out (OUT_REG),a         ; SD-CARD SLOT 0 flashing led
        dec b
        jr nz,l_dispborder

        ld a,2                  ; code 'debug mode'
        ld (boot_from),a
        jr g_ahead



no_diagnostic
        ld hl,0                 ; full bank copy into Spectrum ram
        ld de,$4000             ; uses video ram for visual inspection of what happens
        ld bc,SHADOW_VARS       ; NOTE: 15K. Leaves alone the LAST KILOBYTE (variable space)
        ldir

        jp screen_memory+$4000  ; jumps into the copy before altering FASTPAGE register
screen_memory

        ld a,$40                ; RAM BANK 0
        out (FASTPAGE),a
        ld hl,$34               ; ResiDOS signature location (4 bytes)
        ld bc,4
        ldir                    ; the 4 bytes string is copied at the beginning of bootrom's variable space

        ld a,BOOT_BANK          ; RAM BANK n. 31 selected in RD/WR mode
        out (FASTPAGE),a        ; FROM NOW ON, STACK CAN BE USED

        ld hl,$4000             ; makes a copy of ROM BANK 0 into RAM BANK 31
        ld de,0
        ld bc,SHADOW_VARS+4     ; Leaves alone the LAST KILOBYTE except first 4 bytes of variable space
        ldir

        xor a
        ld (boot_from),a        ; 'code comes from ROM' flag
        jp g_ahead              ; bootrom start entry point: now we jump into RAM BANK 31



g_ahead
        call set_vars           ; VARIABLES INITIALIZATION

        ld a,CS_DISABLE         ; both CARD CS = '1' (disabled):
        ld (cs_outreg),a        ; no cards found (yet)


        ld (attribute),a
        call cls                ; clear screen with 'attribute' colors

        ld a, CS_DISABLE        ; ensures no CS active, NMI on 232 rx disabled
        out (OUT_REG),a
        ld a,$AA                ; signals to the ZX-Com program that we LOGGED OUT
        call rs_232_tx          ; if a RX/TX "echo" loop is present, this will not generate a NMI

        call residos_detect     ; detects residos into nonvolatile ram
        call lookfor_sd         ; looks for a suitable SD card (for proprietary snapshots)
        call if1_feature        ; checks if IF1 switch was enabled: if so, NMI are enabled too - ENABLES RS-232 MODE!
        call disp_menu          ; displays the menu

        ld a,(sh_flags)
        bit sd_present,a
        jr z,no_mmc             ; salta se non l'ha trovata

        bit sd_initsnap,a
        jr z,no_mmc

        ld hl,0
        ld (sd_entry),hl        ; numero di snapshot per visualizzazione

        call display_mmcdata

no_mmc
        xor a
        ld (start_item),a       ; oggetto con cubetto lampeggiante lanciato per default alla pressione di ENTER

        ld hl,flags2
        set save_attrib,(hl)    ; alla prima chiamata, SHOWSEL_ITEM salvera` l'attributo prima di sovrascriverlo

;
;------------------------------------------------------------------------------------------------
; MAIN LOOP
;------------------------------------------------------------------------------------------------
mainloop
        ld hl,frames
        inc (hl)                ; incrementa byte LSB prescaler
        jr nz,mainloop
        inc hl
        inc (hl)                ; incrementa byte MSB prescaler
        jr nz,mainloop
        ld a,$F0                ; che viene pero` reinizializzato a $F0. Quando questo conteggio va` in overflow,
        ld (hl),a               ; si ha un ciclo di lettura KEYBOARD
        inc hl
        inc (hl)                ; incrementa byte LSB contatore
        jr nz,nomsbinc
        inc hl
        inc (hl)                ; incrementa byte MSB contatore (10 secondi/incremento @3.5MHz)
nomsbinc
        ld a,(hl)               ; cambia colore ad ogni ciclo
        call showsel_item       ; carica attributo A alla voce indicata da START_ITEM

        call gest_rewrite       ; eventuale gestione riscrittura di FIRSTFREE in SD/MMC

;       call inc_rtc            ; gestione incremento DATE/TIME basato sul segnale a 1Hz dal chip RTC
                                ; se reload_rtc in sh_flags e` settato, il chip viene riletto.

;       ld l,1                  ; default rom 0
;       ld a,(frames+3)         ; dopo un certo periodo, esegue l'opzione selezionata
;       cp 100                  ; durata unitaria = 10 secondi circa
;       jp nc,start_the_item

        call key_scan           ; lettura tastiera
        jr nz,mainloop          ; piu` di 2 tasti: ignora
        ld a,e
        cp $ff                  ; nessun tasto premuto?
        jr nz,something

        call g_nokeys           ; gestione reset flags tasto premuto per eventuali autorepeat
        jr mainloop             ; salvare il prossimo snapshot

something
        call key_number         ; restituisce il numero del tasto premuto (1 - 40)
        cp 9                    ; tasti da 1 a 9 = selezione di uno snapshot
        jr nc,no_mmcread
        dec a
jp_snapshot                     ; entry point per pressione ENTER, A contiene n. snapshot 0-based
        ld hl,sh_flags
        bit sd_initsnap,(hl)    ; is the card initialized for snapshots?
        jr z,mainloop           ; no, ignore the command
        ld e,a
        ld d,0
        ld hl,(sd_entry)        ; base n. entry visualizzata nel menu
        add hl,de               ; HL = numero di snapshot da ripristinare
        jp read_snap

no_mmcread
        call key_decode         ; restituisce il codice ASCII del tasto premuto

        cp 't'
        jp z,romsnap_restore

        ld l,0                  ; L = 0 = default (NMI-patched ROM)
        cp 'z'                  ; tasto 'Z'
        jr z,r_loadthis
        inc l                   ; L = 1 = 48K + IF1 modified ROMS
        cp 'x'
        jr z,r_loadthis
        inc l                   ; L = 2 = 128K roms (32K)
        cp 'c'
        jr z,r_loadthis
        inc l                   ; L = 3 = ResiDOS
        cp 'r'
        jr z,r_loadthis
        inc l                   ; L = 4 = BBC roms (32K)
        cp 'v'
        jr z,r_loadthis
        inc l                   ; L = 5 = Internal ROM
        cp 1                    ; 1 = codice del CAPS SHIFT premuto da solo
        jr z,r_loadthis

check_enter
        cp 13                   ; ENTER attiva l'opzione selezionata dal cubetto lampeggiante
        jr nz,noenter
start_the_item
        ld a,(start_item)
        cp 5
        jr nc,norom             ; salta se non e` selezionata una ROM
        ld l,a

r_loadthis
        jp loadthis
norom
        sub 5
        jr jp_snapshot


noenter
        ld hl,sh_flags
        bit sd_present,(hl)
        jp z,no_cardpresent     ; se la SD/MMC non e` stata trovata, salta diverse opzioni
        bit sd_initsnap,(hl)
        jp z,no_oursdcard       ; se la SD/MMC non e` inizializzata per snapshot, salta diverse opzioni

        ld de,8
        cp '9'                  ; 9 = indietro di un blocco da 8 entries
        jr z,indblock
        ld de,128
        cp 'n'
        jr nz,nolessentry
indblock
        ld hl,(sd_entry)
        and a
        sbc hl,de
        jr nc,nroll2
        ld hl,0
nroll2
        ld (sd_entry),hl
        call display_entries
        jp mainloop


nolessentry
        ld de,8
        cp '0'                  ; 0 = avanti di un blocco da 8 entries
        jr z,avablock
        ld de,128
        cp 'm'
        jr nz,nomz
avablock
        ld hl,(sd_entry)
        add hl,de
        push hl
        ld de,MMCSIZE
        and a
        sbc hl,de
        pop hl
        jr c,noroll
        ld hl,MMCSIZE-8
noroll
        ld (sd_entry),hl
        call display_entries
        jp mainloop


nomz
        cp 'e'                  ; EDIT filename
        jp nz,noedit
        call wait_keyrelease    ; waits for the key to be released

        call get_hlentry        ; loads the entry from SD/MMC and sets HL to point to the correct item
        jp nz,mainloop          ; returns NONZERO if entry is invalid

        ld a,(start_item)
        add 5
        ld (y_pos),a
        inc hl
        push hl                 ; address of the ascii text in TEMP_RAM (sd/mmc scratch area)
        call make_blank
        ld hl,tmp_str
        ld (hl),$0F             ; attribute for edit space
        ld b,32                 ; available space
        inc hl
l_edit
        push hl
        push bc
        ld a,'_'
        ld (hl),a
        ld hl,tmp_str
        ld a,51
        ld (x_pos),a
        call disp_text
l_key
        call key_scan
        jr nz,l_key
        call key_decode         ; codice ASCII
        pop bc
        pop hl
        cp 13
        jr z,end_edit
        cp 8                    ; symbol shift = DELETE (cambiare con codice 8)
        jr nz,nodelete
        ld a,b
        cp 32
        jr z,l_edit             ; niente da cancellare
        ld (hl),' '             ; toglie il cursore
        dec hl
        inc b
        inc b
        jr next_key
nodelete
        cp ' '
        jr c,l_edit             ; non accetta nessun altro codice inferiore a SPAZIO
        ld (hl),a
        inc hl
next_key
        call wait_keyrelease    ; waits for the key to be released

        djnz l_edit
end_edit
        ld a,b
        and a
        jr z,notrailing
        ld (hl),' '             ; se la stringa non e` piena, elimina il codice del cursore
notrailing
        pop de                  ; address of string in SD/MMC scratchpad area
        ld hl,tmp_str+1
        ld bc,32
        ldir
        call put_entry
        call display_entries
        jp mainloop


noedit
        cp 'w'                  ; EDIT Parameters of snapshot
        jp nz,noparameters
        call wait_keyrelease    ; waits for the key to be released
        call get_hlentry        ; loads the entry from SD/MMC and sets HL to point to the correct item
        jp nz,mainloop          ; returns NONZERO if entry is invalid

        inc hl
        push hl
        ld a,7
        ld (attribute),a
        call cls

        call print_logo

        ld hl,istruz1
        xor a
        ld (x_pos),a
        ld a,19
        ld (y_pos),a
        call disp_text

        ld hl,istruz2
        xor a
        ld (x_pos),a
        ld a,21
        ld (y_pos),a
        call disp_text

        ld hl,istruz3
        ld a,45
        ld (x_pos),a
        ld a,23
        ld (y_pos),a
        call disp_text

        pop hl                  ; visualizza il nome dello snapshot
        push hl
        ld de,tmp_str+1
        ld bc,32
        ldir
        xor a
        ld (x_pos),a
        ld a,4
        ld (y_pos),a
        ld hl,tmp_str
        ld (hl),7
        call disp_text

        pop iy                  ; indirizzo blocco dati
        ld de,39                ; offset al byte LSB dello SP
        add iy,de
refresh_param
        call disp_param         ; visualizza i parametri di questo snapshot

l_paramkey
        call key_scan
        jr nz,l_paramkey
        call key_number         ; A = numero del tasto premuto
        and a
        jr z,l_paramkey
;       cp 1
;       jr nz,nofreqch
;       ld hl,flags2
;       set update_mmc,(hl)
;       ld a,(iy+5)
;       and 7
;       inc a                   ; next frequency step on bits D2:D0
;       cp 6
;       jr c,no_over
;       xor a
;no_over
;       ld c,a
;       ld a,(iy+5)
;       and $F8
;       or c
;       ld (iy+5),a
;       call wait_keyrelease
;       jr refresh_param


nofreqch
        cp 2
        jr nz,noimch
        ld hl,flags2
        set update_mmc,(hl)
        ld a,(iy+4)             ; F BYTE
        inc a                   ; IM MODE +1
        ld c,a
        and 3                   ; IM MODE 3 non esiste
        cp 3
        jr nz,no_i3
        res 0,c
        res 1,c
no_i3
        ld (iy+4),c
        call wait_keyrelease
        jr refresh_param

noimch
        cp 3
        jr nz,noint_toggle
        ld hl,flags2
        set update_mmc,(hl)
        ld a,(iy+4)
        xor 4                   ; bit abilitazione INT
        ld (iy+4),a
        call wait_keyrelease
        jp refresh_param

noint_toggle
        cp 32                   ; 'Z'
        jr nz,nospectrum_toggle
        ld hl,flags2
        set update_mmc,(hl)
        ld a,(iy+4)
        xor 8                   ; bit SPECTRUM MODE
        ld (iy+4),a
        call wait_keyrelease
        jp refresh_param

nospectrum_toggle
        cp 33                   ; 'X'
        jr nz,no_doubleinttoggle
        ld hl,flags2
        set update_mmc,(hl)
        ld a,(iy+4)
        xor 16                  ; bit DOUBLE_INT mode
        ld (iy+4),a
        call wait_keyrelease
        jp refresh_param

no_doubleinttoggle
        cp 30                   ; ENTER
        jp nz,l_paramkey        ; fine tasti decodificati

        ld hl,flags2
        bit update_mmc,(hl)
        jr z,no_upmmc1
        res update_mmc,(hl)
        call put_entry          ; salva in SD/MMC
no_upmmc1
        ld a,7
        ld (attribute),a
        call cls
        call disp_menu
        call display_mmcdata
        jp mainloop


noparameters
        cp 'u'                  ; backup entire zxmmc+ RAM content into four snapshot slots (512K)
        jr nz,norambackup
	xor a
	ld (backupmedia),a	; source: RAM
        ld hl,backup_confirm
	jr do_ramrombackup
norambackup
	cp 'i'			; backup entire zxmmc+ FLASHROM content
	jp nz,noramrombackup
	ld a,1
	ld (backupmedia),a	; source: ROM
	ld hl,backup_confirmrom

do_ramrombackup
	call yesno_confirm
	jp nc,mainloop

        xor a
        ld (x_pos),a
        ld a,2
        ld (y_pos),a
        ld hl,backup_inprogress
        call disp_text

        call make_512kbackup
	jr c,no_backup

        ld a,5
        ld (y_pos),a
        xor a
        ld (x_pos),a
        ld hl,back_done
        call disp_text
anykey
        call key_scan
        ld a,e
        cp $ff                          ; no keys
        jr z,anykey

no_backup
        call cls
        call disp_menu
        call display_mmcdata
        jp mainloop


backup_confirm
        db      7,"Confirm 512K Ram Backup into sd-card (Y/N)",0
backup_confirmrom
        db      7,"Confirm 512K FlashBackup into sdcard (Y/N)",0
backup_inprogress
        db      7,"Backup in progress. Please wait.",0
back_done
        db      7,"Backup Successful. Press any key.",0


noramrombackup                  ; fine comandi specifici per sd-card presente ed inizializzata
        jr common_keys

no_oursdcard                    ; COMMANDS AVAILABLE WHEN THE CARD IS NOT INITIALIZED FOR SNAPSHOTS
        cp 'q'                  ; Q = inizializza la card per l'uso con gli snapshot
        jr nz,common_keys
        call gest_cardinit      ; initializes the card
        jp mainloop


no_cardpresent
common_keys
;
; Following KEYS are subject to AUTOREPEAT function
;
        ld c,a                  ; salva il codice del tasto
        ld a,(repeat_timer)
        and a
        jr z,nomz_pro           ; autorepeat scaduto: procede al rilevamento della pressione dei tasti
        dec a
        ld (repeat_timer),a     ; altrimenti decrementa semplicemente la variabile
        ld a,c
        jp cha_wrnum            ; e salta la gestione di queste funzioni


nomz_pro
        ld a,1
        ld (repeat_timer),a     ; velocita` di autoripetizione
        ld hl,flags2            ; gestione ritardo/autorepeat per i comandi a seguire
        bit key_pressed,(hl)
        jr nz,no_moredelay
        ld a,12                 ; ritardo iniziale prima dello scatto dell'autorepeat
        ld (repeat_timer),a
        set key_pressed,(hl)
no_moredelay
        ld a,c                  ; recupera il codice del tasto


        cp 'p'                  ; P = avanti di 1 item
        jr nz,nomp
        ld hl,flags2
        set restore_attrib,(hl)
        call showsel_item       ; ripristina attributi normali sulla voce che sta per essere lasciata
        set save_attrib,(hl)    ; alla prossima chiamata, SHOWSEL_ITEM salvera` l'attributo prima di sovrascriverlo
        ld b,2
        ld a,(start_item)
        inc a
        cp 3
        jr nz,noresi
        ld hl,sh_flags
        bit residos_present,(hl)
        jr nz,no3_ss
        inc a
        ld b,1
noresi
        cp 4
        jr nz,no3_ss
        ld hl,sh_flags
        bit sd_initsnap,(hl)
        jr nz,sdfound1
        ld a,b
sdfound1
        inc a                   ; salta la parte vuota
no3_ss
        cp 13                   ; siamo oltre il menu`?
        jr nz,no_omenu
        ld a,5
        ld (start_item),a
        ld de,8
        jp avablock             ; prosegue nella routine di avanzamento di un blocco
no_omenu
        ld (start_item),a
        ld hl,flags2
        set force_points,(hl)
        call disptimedate       ; displays DATE and TIME for the (start_item-5) entry
        jr r_wait_key


nomp
        cp 'o'                  ; O = indietro di 1 item
        jr nz,noop
        ld hl,flags2
        set restore_attrib,(hl)
        call showsel_item       ; ripristina attributi normali sulla voce che sta per essere lasciata
        set save_attrib,(hl)    ; alla prossima chiamata, SHOWSEL_ITEM salvera` l'attributo prima di sovrascriverlo
        ld a,(start_item)
        and a
        jr z,r_wait_key
        dec a
        cp 4                    ; siamo oltre il menu`?
        jr nz,no_omenu2
        ld hl,(sd_entry)        ; ci sono altri blocchi di entry su cui retrocedere?
        ld a,l
        or h
        jr nz,some_moreitems    ; si, salta
        ld a,3                  ; altrimenti rientra nella zona della selezione di una ROM
        ld hl,sh_flags
        bit residos_present,(hl)
        jr nz,no_omenu2
        dec a
        jr no_omenu2
some_moreitems
        ld a,12
        ld (start_item),a
        ld de,8
        jp indblock             ; prosegue nella routine di indietreggiamento di un blocco
no_omenu2
        ld (start_item),a
        ld hl,flags2
        set force_points,(hl)
        call disptimedate       ; displays DATE and TIME for the (start_item-5) entry
        jr r_wait_key


noop

;
; Gestione pulsanti [s] e [ ] (symbol shift e spazio) per la gestione di incremento/decremento del numero di entry
; in cui il prossimo snapshot avra` luogo. E` basato su di un contatore di ritardo per l'aggiornamento della SD/MMC
; che produce anche un meccanismo di autorepeat.
;
; Utilizza un conteggio di autorepeat dedicato, che gestisce anche la riscrittura del dato aggiornato.
; Non aggiungere decodifica di altri tasti dopo di questi, perche` il conteggio di autorepeat interferirebbe.
;
cha_wrnum
        ld c,a
        ld a,(rewrite_timer)
        cp $30
        jr nc,r_wait_key        ; attesa autorepeat

        xor a                   ; impedisce uscita dal menu` per timeout
        ld (frames+3),a

        ld a,c

        cp 2                    ; Symbol Shift decrementa il numero di ENTRY per il prossimo snapshot
        jr nz,nosymb            ; 2 = codice del symbol shift premuto da solo

        ld hl,(firstfree)       ; ma solo se e` > 0
        ld a,l
        or h
        jr z,r_wait_key
        dec hl
loadfirstfree
        ld (firstfree),hl
        call disp_firstfree

        ld a,$40                ; ritardo autorepeat e timer riscrittura in SD/MMC
        ld hl,flags2
        bit sp_pressed,(hl)     ; era premuto anche prima?
        jr z,first_pressed      ; alla prima pressione carica $40, alle successive $30 (autorepeat veloce)
        ld a,$30
first_pressed
        ld (rewrite_timer),a
        set sp_pressed,(hl)
r_wait_key
        jp mainloop

nosymb
        cp ' '                  ; spazio incrementa il numero di ENTRY per il prossimo snapshot
        jr nz,r_wait_key
        ld hl,(firstfree)
        inc hl                  ; incrementa il puntatore
        call check_hlspace      ; verifica che non abbia raggiunto il limite
        jp nc,mainloop
        jr loadfirstfree


;
; Gestione nessun tasto premuto
;
g_nokeys
        ld hl,flags2

        res key_pressed,(hl)    ; gestione autorepeat generico
        xor a
        ld (repeat_timer),a

        bit sp_pressed,(hl)     ; gestione autorepeat cambio entry in cui salvare il prossimo snapshot,
        ret z                   ; che ne gestisce anche la riscrittura in SD/MMC

        res sp_pressed,(hl)
        ld a,$30
        ld (rewrite_timer),a    ; pronta risposta in caso venisse premuto di nuovo
        ret



loadthis
        ld a,CS_DISABLE         ; NMI on RS-232 start bit is disabled by default
        out (OUT_REG),a

        ld a,l                  ; code of requested startup media (0-based)
        cp 1                    ; 0 = nmi=patched rom; 1 = 48K/IF1, 2 = 128K, 3 = ResiDOS, 4 = BBC, 5 = Internal ROM
        jr z,load_bank          ; rom selection 1 (48K/IF1) have no NMI on start bit as rs-232 may be used by basic

        cp 2                    ; rom selection = 2 = 128K rom
        jr nz,no_128krom
        exx
        ld hl,16384             ; FUNNY: clearing the 48K memory allows 128K system to start on a rubber 48K machine
        ld bc,49152
l_resetram
        ld (hl),0
        inc hl
        dec bc
        ld a,b
        or c
        jr nz,l_resetram
        exx
        jr enable_nmi

no_128krom
        cp 3
        jr z,residos

        cp 5                    ; rom selection 4 = BBC, 5 = internal rom
        jr nz,enable_nmi
        xor a                   ; disable any rom paging from ZXMMC+ in order to activate the internal ROM
        ld bc,PLUS3REG
        out (c),a
        ld bc,S128KREG
        out (c),a
        jr proceed_load
enable_nmi
        ld a,CS_DISABLE
        or NMI_ENABLE           ; NMI is enabled on RS-232 start bit
        out (OUT_REG),a

load_bank
        ld a,l
        cp 4                    ; 4 = BBC: should be rolled back because ResiDos takes one position
        jr z,noinc
        inc a
noinc
        rlca                    ; each "rom" takes two banks
        add $60                 ; ROM is paged
        jr proceed_load
residos
        ld a,(sh_flags)
        bit residos_present,a
        jp z,mainloop

        ld a,$40                ; otherwise, loads ResiDOS with NMI disabled

proceed_load
        ld hl,bank_switch_code
        ld de,$4000
        ld bc,end_switch-bank_switch_code
        ldir
        jp $4000

bank_switch_code
        out (FASTPAGE),a        ; bank switching can only take place away from paged-in banks
        jp 0
end_switch



;---------------------------------------------------------------------------------------------
; This subroutine clears the screen and displays a (HL) string.
; Returns carry SET if key Y is pressed. If any other key is pressed, the main menu is
; displayed and the routine returns carry reset.
;---------------------------------------------------------------------------------------------
yesno_confirm
	push hl
        ld a,7
        ld (attribute),a
        call cls
        xor a
        ld (x_pos),a
        ld (y_pos),a
	pop hl
        call disp_text
        call wait_keyrelease            ; waits for the key to be released

w_keybackup
        call key_scan
        jr nz,w_keybackup               ; piu` di 2 tasti: ignora
        ld a,e
        cp $ff                          ; no keys
        jr z,w_keybackup
        cp 2                            ; Y key
        jr nz,answerno
	scf
	ret				; return carry SET (Y was pressed)
answerno
        call cls
        call disp_menu
        call display_mmcdata
        and a
	ret

;---------------------------------------------------------------------------------------------
; The BSOD (black screen of death) subroutine: screen is cleared and content of all registers
; is displayed.
;---------------------------------------------------------------------------------------------
regnum  equ     10
bsod
        ld (tmp_str),hl         ; temporary HL saveplace
        push af
        pop hl
        ld (tmp_str+4),hl       ; AF is saved here

        pop hl                  ; HL = return address
        push hl
        dec hl
        dec hl
        dec hl
        ld (tmp_str+2),hl       ; saves caller address here

        ld hl,(tmp_str)         ; HL is restored
        push af                 ; used registers are now saved on stack
        push bc
        push de
        push hl

; the stack will be now populated with values to be displayed

        ld hl,(tmp_str+2)
        push hl                 ; last displayed value is caller address (PC)

        ld hl,12                ; +2 (RET address) +8 (saved registers) +2 (displayed return address)
        add hl,sp               ; calling routine's STACK POINTER VALUE in HL
        push hl                 ; SP value will be displayed right before the PC

        ld a,i
        ld h,a
        ld a,r
        ld l,a
        push hl                 ; IR is displayed

        ld hl,(tmp_str+4)
        push hl
        pop af                  ; AF is restored
        ld hl,(tmp_str)         ; HL is restored

        push iy
        push ix
        exx
        ex af,af'
        push hl
        push de
        push bc
        push af
        ex af,af'
        exx
        push hl
        push de
        push bc
        push af                 ; registers to be displayed --> on stack in reverse order

        call cls
        xor a
        ld (x_pos),a
        ld (y_pos),a

        ld b,regnum+3           ; # of registers +2: displays IR, stack and return address too!
        ld hl,bsod_regs
l_dispreg
        pop de                  ; current register content to be displayed
        push bc
        push de
        call disp_text          ; HL is advanced to next string
        pop bc                  ; current register content to be displayed
        push hl
        push bc
        ld c,b                  ; display MSB nibble
        call disp_hex
        pop bc                  ; display LSB nibble
        call disp_hex
        pop hl
        pop bc
        inc hl
        xor a
        ld (x_pos),a
        ld a,(y_pos)
        inc a
        ld (y_pos),a
        djnz l_dispreg

        ld bc,0
l_bsod
        dec bc
        ld a,b
        or c
        jr nz,l_bsod

        xor a
        ld (x_pos),a
        ld a,20
        ld (y_pos),a
        ld hl,bsod_text
        call disp_text

lkey_bsod
        call key_scan
        jr nz,lkey_bsod         ; piu` di 2 tasti: ignora
        ld a,e
        cp $FF
        jr z,lkey_bsod          ; no keys pressed

        call wait_keyrelease

        pop hl
        pop de
        pop bc
        pop af
        ret

bsod_regs
        db      7,"AF  ",0,7,"BC  ",0,7,"DE  ",0,7,"HL  ",0
        db      7,"AF' ",0,7,"BC' ",0,7,"DE' ",0,7,"HL' ",0
        db      7,"IX  ",0,7,"IY  ",0
        db      7,"IR  ",0,7,"SP  ",0,7,"PC  ",0

bsod_text
        db      7,"The BSOD. Press any key to continue.",0


;---------------------------------------------------------------------------------------------
; Subroutine che visualizza i parametri di snapshot puntati da IY (che punta al byte LSB SP)
;
; Distrugge AF, BC, DE, HL, IX
;---------------------------------------------------------------------------------------------
disp_param
        ld a,7
        ld (y_pos),a

        ld hl,M21
        ld ixl,2
        call disp_clkset

        ld hl,M14
        ld ixl,5
        call disp_clkset

        ld hl,M72
        ld ixl,4
        call disp_clkset

        ld hl,M71
        ld ixl,1
        call disp_clkset

        ld hl,M35
        ld ixl,0
        call disp_clkset


        ld hl,im_modestr
        ld de,tmp_str+1
        ld bc,5
        ldir

        ld b,3
l_dis_p1
        ld a,(iy+4)                     ; byte F
        and 3                           ; D1 - D0 = IM MODE
        inc a

        ld ix,tmp_str
        ld c,7                          ; attributi per voce non attiva
        cp b                            ; e` quella attiva?
        jr nz,noattiva
        ld c,$0F
noattiva
        ld (ix),c
        ld a,b
        add '0'-1
        ld (ix+4),a
        ld a,12
        sub b
        ld (y_pos),a
        ld a,120
        ld (x_pos),a
        ld hl,tmp_str
        call disp_text
        djnz l_dis_p1



        ld hl,int_enabledt
        bit 2,(iy+4)
        jr nz,int_k
        ld hl,int_disabledt
int_k
        ld a,170
        ld (x_pos),a
        call disp_text

        ld hl,spec_typet
        ld a,14
        ld (y_pos),a
        ld a,70
        ld (x_pos),a
        call disp_text


        ld hl,t48k_t
        bit 3,(iy+4)
        jr nz,type_k
        ld hl,t128k_t
type_k
        call disp_text

        ld a,16                         ; displays the SP register content (from snapshot data)
        ld (y_pos),a
        ld a,70
        ld (x_pos),a
        ld hl,spval_t
        call disp_text
        ld c,(iy+1)                     ; MSB byte
        call disp_hex
        ld c,(iy)                       ; LSB byte
        call disp_hex

        ld a,142
        ld (x_pos),a
        ld hl,ival_t                    ; displays the I register content (from snapshot data)
        call disp_text
        ld c,(iy+2)
        call disp_hex

        ret


disp_clkset
        ld a,(iy+4)                     ; F register (IM MODE(D1,0), INT Enable(D2), spectrum type(D3), int doubler(D4))
        bit 4,a
        jr z,hl_isok
        ld de,16                        ; string length offset (points to the "double INT frequency" version)
        add hl,de
hl_isok
        ld de,tmp_str
        ld bc,16                        ; string length
        ldir
        ld hl,tmp_str
        ld a,(iy+5)                     ; CPLD $2FFD register
        and 7                           ; clock select bits
        cp ixl
        jr nz,noqq1
        ld (hl),$0F
noqq1
        xor a
        ld (x_pos),a
        call disp_text
        ld a,(y_pos)
        inc a
        ld (y_pos),a
        ret

im_modestr
        db      "IM  ",0
int_enabledt
        db      2,"INT Enabled ",0
int_disabledt
        db      7,"INT Disabled",0
spec_typet
        db      7,"Spectrum Type: ",0
t48k_t
        db      7,"48K ",0
t128k_t
        db      7,"128K",0

M35
        db      7,"3.5MHz -  50Hz",0    ; "000"
        db      7,"3.5MHz - 100Hz",0    ; "000"
M71
        db      7,"  7MHz -  50Hz",0    ; "001"
        db      7,"  7MHz - 100Hz",0    ; "001"
M72
        db      7,"  7MHz - 100Hz",0    ; "100"
        db      7,"  7MHz - 200Hz",0    ; "100"
M14
        db      7," 14MHz - 100Hz",0    ; "101"
        db      7," 14MHz - 200Hz",0    ; "101"
M21
        db      7," 21MHz -  50Hz",0    ; "010"
        db      7," 21MHz - 100Hz",0    ; "010"
spval_t
        db      7,"SP: $",0
ival_t
        db      7,"I: $",0


istruz1
        db      4,"[1] Z80 Clock; [2] IM MODE; [3] INT On/Off",0
istruz2
        db      4,"[Z] Toggle Spectrum Type; [X] INT Doubler",0
istruz3
        db      2,"Press [ENTER] to Terminate",0



;
;------------------------------------------------------------------------------------------------
; This subroutine saves the entire 512K zxmmc+ ram or rom content into 4 empty snapshot slots
;------------------------------------------------------------------------------------------------
make_512kbackup
        ld hl,BACKUPSTR         ; stringa nomefile primo slot utilizzato
        ld c,MMC_RAMBACKUP      ; tipo di dati: backup ram
	ld a,(backupmedia)	; 0 = RAM; 1 = FLASHROM
	cp 0
	jr z,backram
	ld hl,BACKUPSTRROM
        ld c,MMC_FLASHBACKUP    ; tipo di dati: backup flash
backram
        ld (tmp_reg),hl
        ld a,c
        ld (tmp_reg+2),a

        ld hl,(firstfree)       ; primo slot libero
        ld (tmp_reg+3),hl       ; serve piu` tardi per il calcolo della posizione del blocco dati

        call write_slotdata     ; lo occupa, assegnando il nomefile ed il tipo

        ld hl,RAWSTR            ; ora occupa i successivi 3 slot (512K in totale)
	ld de,tmp_str
	ld bc,32
	ldir
	ld hl,tmp_str		; usa tmp_str in modo da poter variare il numero nel nome del blocco
        ld (tmp_reg),hl
        ld a,MMC_RAWDATA        ; tipo di dati: RAW (non ripristinabile)
        ld (tmp_reg+2),a
        ld hl,(firstfree)       ; primo slot libero (non e` ancora aggiornato)
        ld b,3
loop_3slot
	ld a,'5'
	sub b
	ld (tmp_str+12),a	; scrive il numero di questo blocco nel nome del file
        inc hl
        push bc
        push hl
        call write_slotdata
        pop hl
        pop bc
        ret c                   ; rientra in caso di errore
        djnz loop_3slot
        inc hl
        ld (firstfree),hl       ; aggiorna il n. di slot libero (+4)
        call save_firstfree     ; lo salva nella card

        ld hl,ramcopy           ; parte che deve essere eseguita in ram spectrum
        ld bc,ramcopyend-ramcopy
        ld de,EXECUTE_RAM
        ldir

; Calcola l'indirizzo di scrittura in SD/MMC per questo snapshot

        ld hl,(tmp_reg+3)       ; numero del primo dei 4 slot occupati
        ex hl,de
        ld l,FATSIZE            ; n. blocchi da 64KB che costituiscono la FAT (e` l'offset per la word MSB)
        ld a,(card_offset)
        ld h,a                  ; offset area snapshot nella card
        sla e
        rl d                    ; moltiplica per 2 (ogni entry corrisponde ad un cluster da 128KB)
        add hl,de               ; HL = word MSB scrittura in SD/MMC
        ld de,0                 ; word LSB = 0 (insieme a D0 = 0 della word MSB = 128K)

        ld b,32                 ; blocchi da 16K da copiare

        in a,(FASTPAGE)
        ld (fastp_save),a       ; FASTPAGE content is saved (in spectrum ram)

	ld c,$60		; RAM backup: diventa 0-31 con D6 set dopo la sottrazione di B (ram selected)
	ld a,(backupmedia)
	cp 0			; 0 = backup ram
	jr z,l_rambackup
	ld c,$80		; ROM backup: diventa 0-31 con D6+D5 set
l_rambackup
        push bc
        push hl
        push de

        ld a,c
        sub b                   ; lo trasforma in numero 0-31 con solo D6 SET (ram paged in) o D6+D5 SET (rom paged in)

        call EXECUTE_RAM        ; esecuzione in ram sinclair

        pop de                  ; indirizzo di scrittura in SD-CARD
        pop hl

        ld b,16384/BLOCKSIZE    ; dimensioni per uno snapshot da 16K

        ld ix,SCRATCH           ; ram buffer address
        call mmc_write_data     ; takes the snapshot
        and a
        jr z,wr_okbackup
        ld a,2
        out (BORDER),a
        pop bc
	scf
        ret

wr_okbackup
        pop bc
        djnz l_rambackup        ; other banks? (HL/DE is incremented by mmc_write_data)
	and a
        ret

ramcopy
        out (FASTPAGE),a        ; seleziona il banco
        and 7
        out (BORDER),a
        ld hl,0
        ld de,SCRATCH
        ld bc,16384
        ldir                    ; lo copia in SCRATCH
        ld a,(fastp_save)
        out (FASTPAGE),a        ; riattiva la bootrom
        ret		        ; rientra
ramcopyend

BACKUPSTR
        db      "512K Ram Backup                 "
BACKUPSTRROM
        db      "512K Flash Backup               "
RAWSTR
        db      "Backup part x of 4              "

;
; Routine di scrittura di una entry nella FAT
;
write_slotdata
        ex hl,de
        push de
        call calc_hldec         ; calcola l'indirizzo del blocco --> HL, DE e l'offset 64 bytes in C

        ld ix,SHADOW_RAM        ; scratchpad area
        call mmc_read_data      ; performs the single block read
        and a
        jr z,b512_1
b512_error_de
        pop de
b512_error
        ld a,2
        out (BORDER),a
        scf                     ; carry set = error
        ret

b512_1
        ld d,c                  ; D = 3 bit LSB numero dell'entry
        ld e,0                  ; E = 0, quindi = entry * 256
        srl d                   ; divide /4: adesso DE e` un offset di 64 bytes per ogni valore unitario
        rr e                    ; che avevano i 3 bit LSB, ovvero l'offset della entry da 64 bytes
        srl d                   ; all'interno del blocco da 512 letto precedentemente
        rr e

        ld hl,SHADOW_RAM+2      ; NOTA: SALTA I PRIMI 2 BYTES, utili solo nella prima entry
        add hl,de               ; HL = puntatore al byte con offset [2] dell'entry da modificare

        ld a,(tmp_reg+2)
        ld (hl),a               ; data block type
        inc hl
        ex hl,de
        ld hl,(tmp_reg)         ; copia 32 bytes di filename ad offset +3
        ld bc,32
        ldir

        xor a
        ld b,29                 ; 29 spare bytes are set to ZERO
        ex hl,de
l_spare
        ld (hl),a
        inc hl
        djnz l_spare

        pop de                  ; recupera l'indirizzo usato per la lettura
        call calc_hldec         ; calcola l'indirizzo del blocco --> HL, DE e l'offset 64 bytes in C
        ld b,1                  ; number of blocks to write
        ld ix,SHADOW_RAM        ; scratchpad area
        call mmc_write_data
        and a
        jr nz,b512_error
        and a                   ; carry reset = no error
        ret


;
;------------------------------------------------------------------------------------------------
; POWER-ON menu
;------------------------------------------------------------------------------------------------
disp_menu
        xor a
        out (BORDER),a

        call print_logo

        ld hl,rom0
        xor a
        ld (x_pos),a
        ld a,5
        ld (y_pos),a
        call disp_text

        ld hl,rom1
        xor a
        ld (x_pos),a
        ld a,6
        ld (y_pos),a
        call disp_text

        ld hl,rom2
        xor a
        ld (x_pos),a
        ld a,7
        ld (y_pos),a
        call disp_text

        ld a,(sh_flags)
        bit residos_present,a
        jr z,no_respres
        xor a
        ld (x_pos),a
        ld a,8
        ld (y_pos),a
        ld hl,resi_ram
        call disp_text
        call resi_version

no_respres
        ld hl,sh_flags
        bit sd_initsnap,(hl)    ; is the present SD/MMC initialized for our proprietary snapshot scheme?
        jr z,no_sdprop
        call proprietary_menu
        ret
no_sdprop
        call extended_menu      ; displays alternate menu
        ret


proprietary_menu
        ld a,19                 ; posizione di visualizzazione
        ld (y_pos),a
        xor a
        ld (x_pos),a
        ld hl,yellowarrow
        call disp_text

        ld hl,please            ; [O,P] [9,0] [N,M] menu for snapshot navigation
        xor a
        ld (x_pos),a
        ld a,2
        ld (y_pos),a
        call disp_text

        ld hl,snap1             ; snapshot buttons
        xor a
        ld (x_pos),a
        ld a,3
        ld (y_pos),a
        call disp_text
        ld hl,snap2
        call disp_text
        ld hl,snap3
        call disp_text
        ld hl,snap4
        call disp_text
        ld hl,snap5
        call disp_text

        ld hl,rambackup
        xor a
        ld (x_pos),a
        ld a,4
        ld (y_pos),a
        call disp_text

        ld hl,keys1             ; other snapshot properties buttons
        xor a
        ld (x_pos),a
        ld a,20
        ld (y_pos),a
        call disp_text

        ld hl,keys2
        xor a
        ld (x_pos),a
        ld a,21
        ld (y_pos),a
        call disp_text
        ret

extended_menu
        ld hl,extend_1          ; card init button
        xor a
        ld (x_pos),a
        ld a,2
        ld (y_pos),a
        call disp_text
        ret


print_logo
        xor a
        ld (x_pos),a
        ld (y_pos),a
        ld hl,logo              ; default logo
        ld (hl),7               ; white on black by default
        ld a,(sh_flags)
        bit IF1_enabled,a       ; is IF1 enabled?
        jr z,no_if1
        ld (hl),32              ; black on green
no_if1
        call disp_text

        ld a,144
        ld (x_pos),a
        ld a,(boot_from)
        ld hl,logo_ok           ; standard logo
        and a
        jr z,hl_ok
        ld hl,logo_ram          ; 'RAM debug' logo
        cp 1
        jr z,hl_ok
        ld hl,logo_diagnostic   ; diagnostic logo
hl_ok
        ld a,(logo)
        ld (hl),a
        jp disp_text


logo                    db      7,"ZXMMC+ Bootrom V2.20e -",0
logo_ok                 db      7,"ROM 0 ---> RAM $1F",0
logo_ram                db      7,"RAM $1F    (Debug)",0
logo_diagnostic db      7,"RAM 0 (Diagnostic)",0

please          db      6,"[O,P] = Cursor;  [9,0] +- 8;  [N,M] +- 128",0
rom0            db      7,"[Z] = NMI-Patched 16/48K Rom",0
rom1            db      7,"[X] = 16/48K ROM with IF1 Shadow ROM (24K)",0
rom2            db      7,"[C] = 128K ROMs; [V] BBC; [CAPS] Internal",0
resi_ram        db      7,"[R] = ResiDOS (ZXmmc+ RAM) V",0
mmc             db      7,"SD/MMC CID = ",0
next_snapstr    db      7,"Snap --> ",0
keys1           db      7,"[W] Edit Parameters; [E] Edit Filename",0
keys2           db      7,"[s,' '] Change 'Next Snap' slot number",0
snap1           db      6,"Snap: NMI --> ",0
snap2           db      4,"[Q]16K ",0
snap3           db      5,"[W]48K ",0
snap4           db      2,"[E]128K ",0
snap5           db      3,"[R]ROM",0
rambackup       db      6,"[U]Ram Backup [I] Flash Backup",0

extend_1        db      6,"Press [Q] to initialize CARD for SnapShots",0


;
;------------------------------------------------------------------------------------------------
; This subroutine displays the version of residos found in the nonvolatile RAM chip
;------------------------------------------------------------------------------------------------
resi_version
        ld hl,tmp_str                   ; now displays the ResiDOS version
        ld (hl),7                       ; white color
        inc hl
        ld a,(residos_version)
        ld c,a
        srl a
        srl a
        srl a
        srl a
        and a
        jr z,suppress_zero
        add '0'
        ld (hl),a
        inc hl
suppress_zero
        ld a,c
        and $0F
        add '0'
        ld (hl),a
        inc hl
        ld (hl),'.'
        inc hl
        ld (hl),0                       ; asciz
        ld hl,tmp_str
        call disp_text
        ld a,(residos_version+1)        ; remaining byte can be processed by standard disp_hex call
        ld c,a
        call disp_hex
        ret


;
;---------------------------------------------------------------------------------------------
; Subroutine che imposta le variabili ed il colore del BORDER.
;---------------------------------------------------------------------------------------------
set_vars
        xor a
        ld (x_pos),a
        ld (y_pos),a
        ld (frames),a
        ld (frames+2),a
        ld (frames+3),a
        ld (rewrite_timer),a
        ld (repeat_timer),a
        ld (sh_flags),a
        ld (nmi_count),a        ; NMI counter (nested NMI calls detect)
        ld (flags2),a
        out (BORDER),a          ; border nero
        ld (BORDCR),a           ; variabile BASIC che conserva il colore del border
        ld (card_offset),a

        ld a,$C0                ; MSB contatore viene inizializzato in modo da sveltire la comparsa
        ld (frames+1),a         ; del cubetto lampeggiante (e scansione tastiera)

        ld a,7                  ; PAPER 0, INK 7
        ld (attribute),a

        ld hl,0
        ld (snap_torestore),hl

        ld bc,PLUS3REG          ; default 1FFD and 7FFD registers status: Ram bank 0, Rom bank 1 (48K mode):
        xor a                   ; useful when current rom bank is guessed by guess_128 in order to give
        out (c),a               ; a default value of ROM1 instead of ROM0 when a 16/48k rom is running
        ld (snap_1ffd),a
        ld bc,S128KREG          ; (for example a nmi-patched rom).
        ld a,$10                ; A14 set, Screen 0
        out (c),a
        ld (snap_7ffd),a

        ret

;
;------------------------------------------------------------------------------------------------
; Residos detection
;------------------------------------------------------------------------------------------------
residos_detect
        ld a,(residos_sign)             ; ResiDOS signature
        cp 'R'                          ; first signature byte should be 'R'
        ret nz
        ld a,(residos_sign+1)
        cp 'D'                          ; second signature byte should be 'D'
        ret nz
        ld hl,sh_flags
        set residos_present,(hl)
        ret


;
;------------------------------------------------------------------------------------------------
; This subroutine detects the state of the IF1 switch. If the feature is enabled, then the
; cpld is instructed to issue an NMI on RS-232 Start-bit (for zx-com communication)
; The NMI is later disabled if a nonpatched rom is booted.
;------------------------------------------------------------------------------------------------
if1_feature
        ld bc,IF1_EF            ; IF1 I/O register, full 16 bit
        ld hl,sh_flags
        ld e,0                  ; default NMI disabled on RS-232 start bit
        in d,(c)
        in a,(c)
        or d
        cp $ff
        jr z,no_if1enabled
        set IF1_enabled,(hl)
        ld e,NMI_ENABLE         ; enables NMI from RS-232

        ld d,$FF
        xor a
        out (IF1_F7),a          ; TX Line resting (this will produce a glitch over the network, which is currently active)
        out (c),d               ; selects RS-232 and also ties the CTS HIGH (as fast as possible)

no_if1enabled
        ld a,(cs_outreg)
        or e                    ; the CS register should have proper NMI bit set
        ld (cs_outreg),a

        ld a,CS_DISABLE
        or e
        out (OUT_REG),a         ; enable/disable NMI and keeps both CS inactive
        ret


;
;--------------------------------------------------------------------------------------------------
; MMC card initialization for proprietary SNAPSHOT scheme
;--------------------------------------------------------------------------------------------------
gest_cardinit
        ld a,7
        ld (attribute),a
        call cls
        xor a
        ld (x_pos),a
        ld (y_pos),a
        ld hl,init_confirm_msg
        call disp_text

        ld bc,1                 ; B = offset, C = key number
        ld hl,tmp_str
        ld a,7
        ld (hl),a               ; attributi
        inc hl
        inc hl
        ld a,' '
        ld (hl),a
        inc hl
        ld a,'+'
        ld (hl),a
l_dispoffsets
        ld iy,tmp_str+4         ; posizione di caricamento per CONV_4CIFRE
        ld a,c
        add '0'
        ld (iy-3),a
        ld l,b                  ; valore offset attuale (unitario = 16MB)
        ld h,0
        sla l                   ; moltiplica per 16 per esprimere in megabytes
        rl h
        sla l
        rl h
        sla l
        rl h
        sla l
        rl h
        ld a,(flags2)
        set suppress_leadz,a
        ld (flags2),a
        call conv_4cifre
        xor a
        ld (x_pos),a
        ld a,c
        inc a
        ld (y_pos),a
        ld hl,tmp_str
        call disp_text
        inc c
        ld a,b
        add OFFSET_GRAN         ; offset granularity (16 = +256MB)
        ld b,a
        cp MAXOFFSET+1
        jr nc,max_exceeded
        ld a,(card_offset)
        cp b
        jr nc,l_dispoffsets
max_exceeded
        push bc
        call wait_keyrelease    ; waits for the key to be released

        xor a
        ld (x_pos),a
        ld a,(y_pos)
        add a,2
        ld (y_pos),a
        ld hl,noinit
        call disp_text

w_keyinitsd
        call key_scan
        jr nz,w_keyinitsd       ; piu` di 2 tasti: ignora
        ld a,e
        cp $ff                  ; no keys
        jr z,w_keyinitsd
        call key_number         ; restituisce il numero del tasto premuto (1 - 40)
        pop bc
        cp c
        jr nc,dont_init         ; supera il massimo (tasto 0 = key n. 10)
        dec a
        and a   ; ------------------- could be useless
        jr z,l_cadone           ; 0 = non c'e' bisogno di calcolare l'offset
        ld b,a
        xor a
l_calco
        add OFFSET_GRAN         ; computes the corresponding value
        djnz l_calco
l_cadone
        ld (card_offset),a      ; and saves it to the offset variable
        call sd_init_for_snap
        jp 0

dont_init
        call cls
        call disp_menu
        call display_mmcdata
        jp mainloop

init_confirm_msg
        db      7,"Select an Initialization Offset (MBytes):",0
noinit
        db      7,"...or press '0' to Abort",0


;
;---------------------------------------------------------------------------------------------
; Subroutine che visualizza 8 entries e la posizione del prossimo snapshot
;---------------------------------------------------------------------------------------------
display_mmcdata
        ld a,(sh_flags)
        bit sd_present,a        ; rientra subito se la SD/MMC non e` stata trovata
        ret z
        bit sd_initsnap,a       ; rientra subito se la SD/MMC non e` stata trovata
        ret z

        call display_entries    ; visualizza 8 entries a partire da quella puntata da SD_ENTRY

        ld hl,next_snapstr
        ld a,178
        ld (x_pos),a
        ld a,23
        ld (y_pos),a
        call disp_text
;                               prosegue in DISP_FIRSTFREE
;
;---------------------------------------------------------------------------------------------
; Subroutine che visualizza il contenuto di FIRSTFREE in basso a destra
;---------------------------------------------------------------------------------------------
disp_firstfree
        ld hl,(firstfree)
        ld iy,tmp_str+1         ; posizione di caricamento per CONV_4CIFRE
        ld a,7
        ld (iy-1),a             ; attributi
        call conv_4cifre
        ld a,230
        ld (x_pos),a
        ld a,23
        ld (y_pos),a
        ld hl,tmp_str
        call disp_text
        ret


;---------------------------------------------------------------------------------------------
; Subroutine che gestisce la temporizzazione per la riscrittura in SD/MMC di FIRSTFREE
;---------------------------------------------------------------------------------------------
gest_rewrite
        ld a,(rewrite_timer)
        and a
        ret z                           ; nessuna temporizzazione in corso: rientra subito

        dec a
        ld (rewrite_timer),a
        ld ($5AFA),a                    ; cubetto multicolore che evidenzia il conteggio in corso
        ld ($5AFB),a                    ; cubetto multicolore che evidenzia il conteggio in corso
        and a
        ret nz                          ; se non e` arrivato a zero, rientra
        ld a,2
        ld ($5AFA),a                    ; cubetto multicolore che evidenzia il conteggio in corso
        ld ($5AFB),a                    ; cubetto multicolore che evidenzia il conteggio in corso

save_firstfree
        ld a,(card_offset)              ; MSB byte (bit 31-24) = snapshot area offset (1 = +16MB)
        ld h,a
        ld l,0
        ld de,0                         ; leggiamo il primo blocco della SD/MMC
        ld ix,SHADOW_RAM
        call mmc_read_data              ; performs the single block read
        and a
        ret nz

        ld hl,(firstfree)
        ld (SHADOW_RAM),hl

        ld a,(card_offset)              ; MSB byte (bit 31-24) = snapshot area offset (1 = +16MB)
        ld h,a
        ld l,0
        ld de,0                         ; riscriviamo il blocco nella SD/MMC
        ld b,1                          ; 1 block
        ld ix,SHADOW_RAM
        call mmc_write_data
        ret



;
;------------------------------------------------------------------------------------------------
; The DISPLAY_SD_ENTRIES subroutine
;------------------------------------------------------------------------------------------------
display_entries
        call get_entry
        ld b,8                  ; number of entries to display
        ld ix,SHADOW_RAM+3      ; offset inizio filename

        ld a,10                 ; posizione di visualizzazione
        ld (y_pos),a

l_dispe
        ld a,(ix-1)             ; tipo di snapshot: 1 = 16K, 2 = 48K, 8 = 128K
        ld c,4                  ; verde per snapshot 16K
        cp MMC_SNAP16           ; 1
        jr z,l_dispp
        ld c,5                  ; cyan per snapshot 48K
        cp MMC_SNAP48           ; 2
        jr z,l_dispp
        ld c,2                  ; rosso per snapshot 128K
        cp MMC_SNAP128          ; 8
        jr z,l_dispp
        ld c,7                  ; bianco per raw_data
        cp MMC_RAWDATA          ; 3
        jr z,l_dispp
        ld c,6                  ; giallo per 512K RAM backup
        cp MMC_RAMBACKUP
        jr z,l_dispp
        cp MMC_FLASHBACKUP
        jr z,l_dispp
        ld c,$41                ; invalid: blu
l_dispp
        push bc
        call disp_number        ; displays KEY and SD/MMC entry number

        ld a,(attribute)
        cp 1
        jr nz,displayit
        call make_blank         ; creates a blank string at TMP_STR+1
        jr godisplay

displayit
        ld a,7                  ; white
        ld (tmp_str),a
        push ix
        pop hl
        ld de,tmp_str+1
        ld b,32                 ; copia il nome del file
copyname
        ld a,(hl)
        and a                   ; sopprime eventuali 0 sostituendoli con spazi
        jr nz,noblanka
        ld a,' '
noblanka
        ld (de),a
        inc de
        inc hl
        djnz copyname
        ex hl,de

godisplay
        ld (hl),0

        ld a,(x_pos)
        add a,3
        ld (x_pos),a
        ld hl,tmp_str
        call disp_text          ; visualizza la stringa

next_ent
        ld de,64
        add ix,de
        ld a,(y_pos)
        inc a
        ld (y_pos),a

        pop bc
        djnz l_dispe

        ld hl,flags2
        set force_points,(hl)
        call disptimedate       ; displays DATE and TIME for the (start_item-5) entry
        ret


get_entry
        ld hl,(sd_entry)        ; gets the SD_ENTRY data in SHADOW_RAM
        ex hl,de
        call calc_hldec         ; HL, DE = 32bit address of block to be read
        ld ix,SHADOW_RAM
        call mmc_read_data      ; reads the single block
        ret

put_entry
        ld hl,(sd_entry)        ; puts the SD_ENTRY data --> SD/MMC card
        ex hl,de
        call calc_hldec         ; HL, DE = 32bit address of block to be read
        ld ix,SHADOW_RAM
        ld b,1
        call mmc_write_data     ; writes the block
        ret

make_blank
        ld hl,tmp_str+1         ; creates a BLANK STRING at tmp_str+1
        ld b,32
        ld a,' '
l_blankname
        ld (hl),a
        inc hl
        djnz l_blankname
        ld (hl),0               ; asciz
        ret

;
;------------------------------------------------------------------------------------------------
; Subroutine that displays entry key-select and entry number, with C attribute
;
; (SD_ENTRY) = base number (first entry number)
; 8-B = this entry offset number
; IX = Address of filename in entry space
; C = attribute code
;
; Destroys AF, BC, DE, HL, IY
;------------------------------------------------------------------------------------------------
disp_number
        ld hl,(sd_entry)        ; entries base number
        ld a,8
        sub b
        ld e,a
        ld d,0
        add hl,de

        ld iy,tmp_str+5
        ld (iy-4),'['
        ld (iy-2),']'
        ld (iy-1),' '
        add '1'
        ld (iy-3),a
        ld (iy-5),c

        call conv_4cifre        ; trasforma HL in ascii all'indirizzo IY

        ld hl,tmp_str
        xor a
        ld (x_pos),a
        call disp_text
        ret


;
;---------------------------------------------------------------------------------------------
; Subroutine that displays TIME and DATE saved in the currently highlighted snapshot item
;---------------------------------------------------------------------------------------------
disptimedate
        ld ix,SHADOW_RAM+35     ; indirizzo inizio dati RTC del primo item
        ld de,64                ; offset per blocco
        ld a,(start_item)
        sub 5
        ret c
        jr z,ix_disok
        ld b,a
l_calix
        add ix,de
        djnz l_calix
ix_disok
        call build_timestring

        ld a,19                 ; posizione di visualizzazione
        ld (y_pos),a
        ld a,51
        ld (x_pos),a
        ld hl,tmp_str
        call disp_text
        ret

yellowarrow                     ; questa e` visualizzata da 'disp_menu'
        db      6,"Took -->",0


;
;------------------------------------------------------------------------------------------------
; Subroutine that converts an integer (HL) to a string (max 4 characters), writing at (IY).
; Leading zero are suppressed if proper flag is set.
;
; Returns: String written at (IY), IY pointing to last valid character in string
; Destroys AF, DE, HL, IY
;------------------------------------------------------------------------------------------------
conv_4cifre
        push bc
        ld a,(flags2)
        ld c,a

        ld de,1000
        call conv_cifra
        ld de,100
        call conv_cifra
        ld de,10
        call conv_cifra
        ld a,l
        add '0'
        ld (iy),a
        ld (iy+1),0                     ; asciz

        res suppress_leadz,c
        ld a,c
        ld (flags2),a
        pop bc
        ret

conv_cifra
        ld a,'0'
l_1000
        and a
        sbc hl,de
        jr c,nomille
        inc a
        res suppress_leadz,c
        jr l_1000
nomille
        ld (iy),a
        bit suppress_leadz,c
        jr nz,sup_1000
        inc iy
sup_1000
        add hl,de
        ret


;
;--------------------------------------------------------------------------------------------------
; This subroutine builds a printable string with DATE/TIME informations at address TMP_STR,
; reading data from (IX) (7 bytes in BINARY format)
;
; The ':' symbol is displayed only when the RTC's 1Hz line is LOW (update occurs on falling edge)
; or when flags2,force_points is SET.
;
; Destroys AF, BC, DE, HL
;--------------------------------------------------------------------------------------------------
build_timestring
        ld de,tmp_str
        ld a,6                          ; yellow on black
        ld (de),a
        inc de
        ld c,(ix+3)                     ; day of week
        sla c                           ; *4
        sla c
        ld b,0
        ld hl,day_names
        add hl,bc
        ld bc,4                         ; length of day's name
        ldir
no_dayname
        ld b,':'                        ; the ':' symbol
        ld hl,flags2
        bit force_points,(hl)           ; this flag forces the ':' symbol
        jr nz,dupoint
;       in a,(STATUS)
;       bit 5,a                         ; otherwise, we test the 1Hz line
;       jr z,dupoint
        ld b,' '                        ; blank
dupoint
        res force_points,(hl)

        ex hl,de                        ; write pointer to HL

        ld a,(ix+4)                     ; day of month
        cp 32
        jr nc,rtc_invalid
        and a
        jr z,rtc_invalid
        call ldhl_binary
        ld (hl),'/'
        inc hl
        ld a,(ix+5)                     ; month
        cp 13
        jr nc,rtc_invalid
        and a
        jr z,rtc_invalid
        call ldhl_binary
        ld (hl),'/'
        inc hl
        ld (hl),'2'
        inc hl
        ld (hl),'0'                     ; someone should change this on year 2100 (please)
        inc hl
        ld a,(ix+6)                     ; year
        cp $9A
        jr nc,rtc_invalid
        call ldhl_binary
        ld (hl),' '
        inc hl
        ld (hl),' '
        inc hl
        ld a,(ix+2)                     ; hours
        cp 24
        jr nc,rtc_invalid
        call ldhl_binary
        ld (hl),b
        inc hl
        ld a,(ix+1)                     ; minutes
        cp 60
        jr nc,rtc_invalid
        call ldhl_binary
        ld (hl),b
        inc hl
        ld a,(ix)                       ; seconds
        cp 60
        jr nc,rtc_invalid
        call ldhl_binary
        ld (hl),0                       ; null terminated string
        ret

rtc_invalid
        ld hl,tmp_str+1
        ld b,24                         ; string length
        ld a,' '                        ; returns a blank string suitable for deleting any previous data
l_clearstr
        ld (hl),a
        inc hl
        djnz l_clearstr
        xor a
        ld (hl),a                       ; null terminated string
        ret


ldhl_binary
        ld c,'0'
l_bincalc
        sub 10
        jr c,over_bin
        inc c
        jr l_bincalc
over_bin
        ld (hl),c
        add 10+'0'
        inc hl
        ld (hl),a
        inc hl
        ret

day_names
        db      "    Mon Tue Wed Thu Fri Sat Sun "


;
;------------------------------------------------------------------------------------------------
; THE 'KEYBOARD SCANNING' SUBROUTINE, copiata pari-pari da quella della ROM 48K
;
; The E register is returned with a value in the range of +00 to +27, the value being different
; for each of the forty keys of the keyboard, or the value +FF, for no-key.
; The D register is returned with a value that indicates which single shift key is being pressed.
; If both shift keys are being pressed then the D and E registers are returned with the values
; for the CAPS SHIFT and SYMBOL SHIFT keys respectively.
; If no keys is being pressed then the DE register pair is returned holding +FFFF.
; The zero flag is returned reset if more than two keys are being pressed, or neither key of a
; pair of keys is a shift key.
;
; KEY MAP FOR 'E' REGISTER:
;
;      COL 1  2  3  4 5 6  7  8  9 10
;
;ROW 1    36 28 20 12 4 3 11 19 27 35   1234567890
;ROW 2    37 29 21 13 5 2 10 18 26 34   QWERTYUIOP
;ROW 3    38 30 22 14 6 1  9 17 25 33   ASDFGHJKL[ENTER]
;ROW 4    39 31 23 15 7 0  8 16 24 32   [caps]ZXCVBNM[symbol][SPACE]
;
; DISTRUGGE AF, BC, DE, HL
;------------------------------------------------------------------------------------------------
key_scan
        ld l,$2F
        ld de,$FFFF
        ld bc,$FEFE
key_line
        in a,(c)
        cpl
        and $1F
        jr z,key_done
        ld h,a
        ld a,l
key_3keys
        inc d
        ret nz
key_bits
        sub 8
        srl h
        jr nc,key_bits
        ld d,e
        ld e,a
        jr nz,key_3keys
key_done
        dec l
        rlc b
        jr c,key_line
        ld a,d
        inc a
        ret z
        cp $28
        ret z
        cp $19
        ret z
        ld a,e
        ld e,d
        ld d,a
        cp $18
        ret


;
;------------------------------------------------------------------------------------------------
; Subroutine che attende il rilascio di tutti i tasti. Distrugge AF.
;------------------------------------------------------------------------------------------------
wait_keyrelease
        push bc
        push de
        push hl
l_wrel
        call key_scan
        jr nz,l_wrel
        ld a,e
        cp $FF
        jr z,no_key
        cp 24                   ; symbol shift non viene considerato un tasto, rientriamo
        jr z,no_key
        cp 39                   ; stessa cosa per il caps shift
        jr nz,l_wrel
no_key
        pop hl
        pop de
        pop bc
        ret

;
;------------------------------------------------------------------------------------------------
; Key table subroutine. Returns a single, decoded character in A register based on
; keyscan code in DE register.
;
; NEVER WRITES to any ram variable, since we may have been called in NMI for snapshot purposes
;
; RETURNS:
; 0 = no valid keycode
; 1 = caps shift pressed (alone)
; 2 = symbol shift pressed (alone)
; 8 = BACKSPACE (caps_shift + '0')
; 13 = ENTER
; ASCII code (letters and numbers, capitals if caps shift pressed as well)
; Symbol ascii codes if symbol shift is pressed as well
;
; Destroys AF, D, HL
;------------------------------------------------------------------------------------------------
key_decode
        ld a,d
        cp 24                   ; D = 24 = symbol shift premuto insieme ad un altro tasto
        jr z,symbolshift        ; (se premuto da solo, il suo codice viene letto da tabella ascii)

        ld a,e
        cp 40
        jr c,validk             ; salta se E e` entro 0-39 (tasto premuto)
        ld a,d
        cp 39                   ; D = 39 = caps shift premuto
        jr z,capsshift
        xor a
        ret
capsshift
        ld a,1                  ; rientra con il codice del caps shift
        ret

validk
        push de                 ; salva lo stato del caps shift
        ld hl,asciitable
        ld d,0
        add hl,de
        ld a,(hl)               ; legge il codice ascii
        pop de                  ; notare che se il caps shift e` premuto da solo, il suo codice
        push af                 ; verra` letto dalla tabella
        ld a,d
        cp 39                   ; D = 39 = caps shift premuto
        jr nz,nocaps            ; non e` premuto: rientra con il codice letto da tabella
        pop af
        cp '0'
        jr nz,nobackspa
        ld a,8                  ; 0 + CAPS LOCK = BACKSPACE
        ret
nobackspa
        cp 'a'
        ret c                   ; rientra se non e` un carattere alfabetico
        sub 'a'-'A'             ; altrimenti lo trasforma in maiuscolo
        ret

nocaps
        pop af
        ret

symbolshift
        ld a,e                  ; c'e' un altro tasto premuto contemporaneamente?
        cp 40
        jr c,altrota            ; si, salta (siamo entro la tabella, quindi non e` $FF)
        ld a,2                  ; altrimenti rientra con codice ASCII $02 (lo usiamo per SYMBOL SHIFT)
        ret
altrota
        ld hl,symboltable       ; trova il codice ascii del SIMBOLO sul tasto premuto
        ld d,0
        add hl,de
        ld a,(hl)
        ret


asciitable
        defb    'b','h','y','6','5','t','g','v' ; tasti relativi ai codici da 0 a 7
        defb    'n','j','u','7','4','r','f','c' ; 8 - 15
        defb    'm','k','i','8','3','e','d','x' ; 16 - 23
        defb    2,'l','o','9','2','w','s','z'   ; 24 - 31 (0 = symbol shift)
        defb    ' ',13,'p','0','1','q','a',1    ; 32 - 39 (0 = caps shift), 13 = enter

symboltable
        defb    '*','^','y','&','%','>','g','/' ; tasti relativi ai codici da 0 a 7
        defb    ',','-','u',39,'$','<','f','?'  ; 8 - 15
        defb    '.','+','i','(','#','e','d','x' ; 16 - 23
        defb    0,'=',';',')','@','w','s',':'   ; 24 - 31 (0 = symbol shift)
        defb    ' ',13,'"','_','!','q','a',0    ; 32 - 39 (0 = caps shift, 13 = enter)


;
;------------------------------------------------------------------------------------------------
; Subroutine that returns the KEY NUMBER (1 - 40) of the pressed key, or 0 for no key.
;
; Destroys AF
;------------------------------------------------------------------------------------------------
key_number
        ld a,e
        cp 40                   ; la tabella contiene 40 elementi
        jr nc,over
        push de
        push hl
        ld hl,keyposition
        ld d,0
        add hl,de
        ld a,(hl)
        pop hl
        pop de
        ret
over
        xor a                   ; no-key
        ret

keyposition
        defb    36, 26, 16, 6, 5, 15, 25, 35
        defb    37, 27, 17, 7, 4, 14, 24, 34
        defb    38, 28, 18, 8, 3, 13, 23, 33
        defb    39, 29, 19, 9, 2, 12, 22, 32
        defb    40, 30, 20, 10, 1, 11, 21, 31


;
;------------------------------------------------------------------------------------------------
; Subroutine di visualizzazione della stringa puntata da HL
;
; INPUT:
; HL = puntatore alla stringa ASCIIZ
; X_POS = coordinata X di visualizzazione, espressa in PIXEL (0 - 255)
; Y_POS = coordinata Y di visualizzazione, espressa in CARATTERI (0 - 23)
;
; Chiama COMPUTE_ADD per calcolare il contenuto di WR_VIDEO in base a X_POS e Y_POS
; Chiama DISP_CH per visualizzare ciascun carattare
;
; OUTPUT:
; X_POS, Y_POS aggiornati alla coordinata del prossimo carattere da visualizzare
; HL punta al byte di fine stringa (NULL)
;
; DISTRUGGE AF, DE, HL.
;------------------------------------------------------------------------------------------------
disp_text
        push iy
        push bc
        call compute_add        ; calcola l'indirizzo di inizio scrittura in ram video in base a X_POS, Y_POS
        ld a,(hl)
        ld (attribute),a        ; primo byte = attributi
        inc hl
l_text
        ld a,(hl)
        and a
        jr z,f_disptext         ; fine lavoro
        call disp_ch            ; visualizza il carattere
        inc hl
        jr l_text
f_disptext
        pop bc
        pop iy
        ret                     ; rientra con HL che punta al NULL byte


;------------------------------------------------------------------------------------------------
; Subroutine che calcola WR_VIDEO in base al contenuto di X_POS e Y_POS
;
; INPUT:
; X_POS = coordinata X di visualizzazione, espressa in PIXEL (0 - 255)
; Y_POS = coordinata Y di visualizzazione, espressa in CARATTERI (0 - 23)
;
; OUTPUT:
; WR_VIDEO puntera` al primo byte in cui scrivere; il numero di bit gia` occupati in questo
; byte e` (era gia`) espresso nei 3 bit LSB di X_POS, che infatti non prendono parte alla
; costituzione di WR_VIDEO
;
; DISTRUGGE AF, DE
; PRESERVA  HL, BC (not really...)
;
; MODDED: Eugenio Ciceri 20080209
;------------------------------------------------------------------------------------------------

                if FASTER = 1
compute_add
;       push bc                 ; the original code was not preserving BC, so why should I? ;-)
        push hl

        ld bc,(x_pos)           ; b = y_pos, c = x_pos
        srl c
        srl c
        srl c
        ld a,b
        and $18
        ld h,a
        set 6,h
        rrca
        rrca
        rrca
        or $58
        ld d,a
        ld a,b
        and 7
        rrca
        rrca
        rrca
        add a,c
        ld l,a
        ld e,a
        ld a,(de)
        ld (wr_video),hl
        ld (wr_attribute),de

        pop hl
;       pop bc
        ret

                else

compute_add
        push hl
        ld hl,$4000             ; indirizzo base VIDEO RAM

        ld a,(y_pos)            ; l'offset per ciascun blocco di 8 righe e` $800
        and $F8
        ld d,a                  ; lo somma quindi direttamente ad HL, sul byte MSB
        ld e,0                  ; e dopo aver mascherato i 3 bit LSB
        add hl,de

        ld a,(y_pos)
        and 7                   ; i 3 bit LSB valgono offset +32 ciascuno
        rrca                    ; moltiplica *5
        rrca
        rrca
        ld e,a
        ld d,0
        add hl,de               ; somma

        ld a,(x_pos)            ; i 5 bit MSB costituiscono l'offset di COLONNA in questa riga
        srl a
        srl a                   ; shifta a destra 3 volte (offset 0 - 31)
        srl a
        ld e,a                  ; D e` gia` ZERO
        add hl,de               ; somma offset
        ld (wr_video),hl


        ld hl,$5800             ; attribute base address
        ld a,(y_pos)            ; il valore unitario di Y_POS corrisponde ad un offset di +32 bytes
        ld e,a                  ; negli attributi
        ld d,0
        ld b,5
l_muldd
        sla e                   ; moltiplica *5 a 16 bit
        rl d
        djnz l_muldd
        add hl,de               ; somma l'offset

        ld a,(x_pos)            ; i 5 bit MSB costituiscono l'offset di COLONNA in questa riga
        srl a
        srl a                   ; shifta a destra 3 volte (offset 0 - 31)
        srl a
        ld e,a
        ld d,0
        add hl,de               ; somma offset
        ld (wr_attribute),hl

        pop hl
        ret

                endif

;------------------------------------------------------------------------------------------------
; Subroutine di visualizzazione del carattere in 'A'
;
; INPUT:
; WR_VIDEO deve puntare al primo byte in ram video in cui scrivere;
; X_POS, che contiene la coordinata X espressa IN PIXEL (ovvero 0 - 255) viene utilizzato come
; indicatore di bit OCCUPATI nel byte puntato da WR_VIDEO, dato espresso dai suoi 3 bit LSB;
; (i 5 MSB sono gia` stati integrati in WR_VIDEO ad opera di COMPUTE_ADD)
;
; Y_POS (qui non utilizzata) contiene la coordinata Y espressa in CARATTERI (0-23).
; Viene usata da COMPUTE_ADD per calcolare WR_VIDEO.
;
; OUTPUT:
; Il carattere viene scritto nella ram video; X_POS incrementato del n. di bit costituenti la
; larghezza del carattere. Se necessario, ovvero se siamo andati oltre il byte correntemente
; puntato da WR_VIDEO, esso viene incrementato in modo da non richiedere altri calcoli in caso di
; chiamate consecutive (per visualizzare una stringa).
;
; DISTRUGGE AF, BC, DE, IY (nella versione lenta).
;
; MODDED: Eugenio Ciceri 20080210
;------------------------------------------------------------------------------------------------

                if FASTER = 1
disp_ch
        push hl

        ; ----------------------- compute the character data address
        sub ' '                 ; SPACE = first character in the font block
        ld l,a
        ld h,0
        add hl,hl
        add hl,hl
        add hl,hl               ; multiply by 8 (16 bit wide)
        ld de,font_block
        add hl,de               ; hl holds the character data address

        ld de,MASK_X_16         ; de holds the 16 bit mask

        ; ----------------------- find which routine is best suited
        ld a,(x_pos)
        and 7
        ld c,a                  ; c holds the number of bits to be shifted
        jp z,no_shift           ; do we need to shift?
        cp 8-CHAR_X             ; yes, but do we need two bytes?
        jp c,single_byte_shift


        ; ----------------------- routine to print a two bytes wide shifted char
two_byte_shift                  ; worst case: we need two bytes to print the char
        ld a,(x_pos)
        cp 255-CHAR_X           ; is there space enough to print the whole char?
        jr c,two_byte_mask_shift; yes, do it

        xor a                   ; no, will print at the start of the...
        ld (x_pos),a

        ld a,(y_pos)            ; ...next line
        inc a
        ld (y_pos),a

        call compute_add        ; new video address
        jp no_shift             ; of course there is no more need to shift

        ; ----------------------- build a 16 bit mask
two_byte_mask_shift             ; de holds the 16 bit mask
        ld b,c                  ; c holds the number of bits to be shifted
        scf                     ; set the the carry flag for the first rotation
next_bit_2
        rr d
        rr e
        djnz next_bit_2         ; the last bit of the mask will rotate to d MSB

        push de
        exx
        pop bc                  ; bc' holds the rotated 16 bit mask
        ld hl,(wr_video)        ; hl' holds the display file address
        exx

        ; ----------------------- row number 1
        ld e,b                  ; b holds '0' after the previous djnz loop
        ld d,(hl)               ; fetch the character data

        ld b,c                  ; c holds the number of bits to be shifted
shift_left_2_1
        srl d                   ; shift the character data
        rr e
        djnz shift_left_2_1     ; de holds the shifted character data

        push de
        exx
        pop de                  ; de' holds the shifted character data
        ld a,(hl)               ; fetch the first video address content
        and b                   ; mask it
        or d                    ; OR print the character
        ld (hl),a               ; print back the result to the screen
        inc hl                  ; next byte video address
        ld a,(hl)               ; fetch the second video address content
        and c                   ; mask it
        or e                    ; OR print the character
        ld (hl),a               ; print back the result to the screen
        dec hl                  ; back to the first byte video address

        inc h                   ; next line video address
        exx
        inc hl                  ; next character line

        ; ----------------------- row number 2
        ld e,b                  ; b holds '0' after the previous djnz loop
        ld d,(hl)               ; fetch the character data

        ld b,c                  ; c holds the number of bits to be shifted
shift_left_2_2
        srl d                   ; shift the character data
        rr e
        djnz shift_left_2_2     ; de holds the shifted character data

        push de
        exx
        pop de                  ; de' holds the shifted character data
        ld a,(hl)               ; fetch the first video address content
        and b                   ; mask it
        or d                    ; OR print the character
        ld (hl),a               ; print back the result to the screen
        inc hl                  ; next byte video address
        ld a,(hl)               ; fetch the second video address content
        and c                   ; mask it
        or e                    ; OR print the character
        ld (hl),a               ; print back the result to the screen
        dec hl                  ; back to the first byte video address

        inc h                   ; next line video address
        exx
        inc hl                  ; next character line

        ; ----------------------- row number 3
        ld e,b                  ; b holds '0' after the previous djnz loop
        ld d,(hl)               ; fetch the character data

        ld b,c                  ; c holds the number of bits to be shifted
shift_left_2_3
        srl d                   ; shift the character data
        rr e
        djnz shift_left_2_3     ; de holds the shifted character data

        push de
        exx
        pop de                  ; de' holds the shifted character data
        ld a,(hl)               ; fetch the first video address content
        and b                   ; mask it
        or d                    ; OR print the character
        ld (hl),a               ; print back the result to the screen
        inc hl                  ; next byte video address
        ld a,(hl)               ; fetch the second video address content
        and c                   ; mask it
        or e                    ; OR print the character
        ld (hl),a               ; print back the result to the screen
        dec hl                  ; back to the first byte video address

        inc h                   ; next line video address
        exx
        inc hl                  ; next character line

        ; ----------------------- row number 4
        ld e,b                  ; b holds '0' after the previous djnz loop
        ld d,(hl)               ; fetch the character data

        ld b,c                  ; c holds the number of bits to be shifted
shift_left_2_4
        srl d                   ; shift the character data
        rr e
        djnz shift_left_2_4     ; de holds the shifted character data

        push de
        exx
        pop de                  ; de' holds the shifted character data
        ld a,(hl)               ; fetch the first video address content
        and b                   ; mask it
        or d                    ; OR print the character
        ld (hl),a               ; print back the result to the screen
        inc hl                  ; next byte video address
        ld a,(hl)               ; fetch the second video address content
        and c                   ; mask it
        or e                    ; OR print the character
        ld (hl),a               ; print back the result to the screen
        dec hl                  ; back to the first byte video address

        inc h                   ; next line video address
        exx
        inc hl                  ; next character line

        ; ----------------------- row number 5
        ld e,b                  ; b holds '0' after the previous djnz loop
        ld d,(hl)               ; fetch the character data

        ld b,c                  ; c holds the number of bits to be shifted
shift_left_2_5
        srl d                   ; shift the character data
        rr e
        djnz shift_left_2_5     ; de holds the shifted character data

        push de
        exx
        pop de                  ; de' holds the shifted character data
        ld a,(hl)               ; fetch the first video address content
        and b                   ; mask it
        or d                    ; OR print the character
        ld (hl),a               ; print back the result to the screen
        inc hl                  ; next byte video address
        ld a,(hl)               ; fetch the second video address content
        and c                   ; mask it
        or e                    ; OR print the character
        ld (hl),a               ; print back the result to the screen
        dec hl                  ; back to the first byte video address

        inc h                   ; next line video address
        exx
        inc hl                  ; next character line

        ; ----------------------- row number 6
        ld e,b                  ; b holds '0' after the previous djnz loop
        ld d,(hl)               ; fetch the character data

        ld b,c                  ; c holds the number of bits to be shifted
shift_left_2_6
        srl d                   ; shift the character data
        rr e
        djnz shift_left_2_6     ; de holds the shifted character data

        push de
        exx
        pop de                  ; de' holds the shifted character data
        ld a,(hl)               ; fetch the first video address content
        and b                   ; mask it
        or d                    ; OR print the character
        ld (hl),a               ; print back the result to the screen
        inc hl                  ; next byte video address
        ld a,(hl)               ; fetch the second video address content
        and c                   ; mask it
        or e                    ; OR print the character
        ld (hl),a               ; print back the result to the screen
        dec hl                  ; back to the first byte video address

        inc h                   ; next line video address
        exx
        inc hl                  ; next character line

        ; ----------------------- row number 7
        ld e,b                  ; b holds '0' after the previous djnz loop
        ld d,(hl)               ; fetch the character data

        ld b,c                  ; c holds the number of bits to be shifted
shift_left_2_7
        srl d                   ; shift the character data
        rr e
        djnz shift_left_2_7     ; de holds the shifted character data

        push de
        exx
        pop de                  ; de' holds the shifted character data
        ld a,(hl)               ; fetch the first video address content
        and b                   ; mask it
        or d                    ; OR print the character
        ld (hl),a               ; print back the result to the screen
        inc hl                  ; next byte video address
        ld a,(hl)               ; fetch the second video address content
        and c                   ; mask it
        or e                    ; OR print the character
        ld (hl),a               ; print back the result to the screen
        dec hl                  ; back to the first byte video address

        inc h                   ; next line video address
        exx
        inc hl                  ; next character line

        ; ----------------------- row number 8
        ld e,b                  ; b holds '0' after the previous djnz loop
        ld d,(hl)               ; fetch the character data

        ld b,c                  ; c holds the number of bits to be shifted
shift_left_2_8
        srl d                   ; shift the character data
        rr e
        djnz shift_left_2_8     ; de holds the shifted character data

        push de
        exx
        pop de                  ; de' holds the shifted character data
        ld a,(hl)               ; fetch the first video address content
        and b                   ; mask it
        or d                    ; OR print the character
        ld (hl),a               ; print back the result to the screen
        inc hl                  ; next byte video address
        ld a,(hl)               ; fetch the second video address content
        and c                   ; mask it
        or e                    ; OR print the character
        ld (hl),a               ; print back the result to the screen
;       dec hl                  ; back to the first byte video address

;       inc h                   ; next line video address
;       exx
;       inc hl                  ; next character line

        ; ----------------------- print the attributes for both the two bytes involved
        ld a,(attribute)
        ld hl,(wr_attribute)
        ld (hl),a
        inc hl
        ld (hl),a

        ; ----------------------- compute the attribute address for the next char
        ld (wr_attribute),hl

        ; ----------------------- compute the video address for the next char
        ld a,(x_pos)
        add CHAR_X
        ld (x_pos),a
        ld hl,wr_video
        inc (hl)

        jp end_print


        ; ----------------------- routine to print a single byte shifted char
single_byte_shift

        ; ----------------------- build a 8 bit mask
single_byte_mask_shift          ; d holds the 8 bit mask
        ld b,c                  ; c holds the number of bits to be shifted
        scf                     ; set the the carry flag for the first rotation
next_bit_1
        rr d
        djnz next_bit_1         ; the last bit of the mask will rotate to d MSB

        push de
        exx
        pop bc                  ; b' holds the rotated 8 bit mask
        ld hl,(wr_video)        ; hl' holds the display file address
        exx

        ; ----------------------- row number 1
        ld d,(hl)               ; fetch the character data

        ld b,c                  ; c holds the number of bits to be shifted
shift_left_1_1
        srl d                   ; shift the character data
        djnz shift_left_1_1     ; d holds the shifted character data

        push de
        exx
        pop de                  ; d' holds the shifted character data
        ld a,(hl)               ; fetch the video address content
        and b                   ; mask it
        or d                    ; OR print the character
        ld (hl),a               ; print back the result to the screen

        inc h                   ; next line video address
        exx
        inc hl                  ; next character line

        ; ----------------------- row number 2
        ld d,(hl)               ; fetch the character data

        ld b,c                  ; c holds the number of bits to be shifted
shift_left_1_2
        srl d                   ; shift the character data
        djnz shift_left_1_2     ; d holds the shifted character data

        push de
        exx
        pop de                  ; d' holds the shifted character data
        ld a,(hl)               ; fetch the video address content
        and b                   ; mask it
        or d                    ; OR print the character
        ld (hl),a               ; print back the result to the screen

        inc h                   ; next line video address
        exx
        inc hl                  ; next character line

        ; ----------------------- row number 3
        ld d,(hl)               ; fetch the character data

        ld b,c                  ; c holds the number of bits to be shifted
shift_left_1_3
        srl d                   ; shift the character data
        djnz shift_left_1_3     ; d holds the shifted character data

        push de
        exx
        pop de                  ; d' holds the shifted character data
        ld a,(hl)               ; fetch the video address content
        and b                   ; mask it
        or d                    ; OR print the character
        ld (hl),a               ; print back the result to the screen

        inc h                   ; next line video address
        exx
        inc hl                  ; next character line

        ; ----------------------- row number 4
        ld d,(hl)               ; fetch the character data

        ld b,c                  ; c holds the number of bits to be shifted
shift_left_1_4
        srl d                   ; shift the character data
        djnz shift_left_1_4     ; d holds the shifted character data

        push de
        exx
        pop de                  ; d' holds the shifted character data
        ld a,(hl)               ; fetch the video address content
        and b                   ; mask it
        or d                    ; OR print the character
        ld (hl),a               ; print back the result to the screen

        inc h                   ; next line video address
        exx
        inc hl                  ; next character line

        ; ----------------------- row number 5
        ld d,(hl)               ; fetch the character data

        ld b,c                  ; c holds the number of bits to be shifted
shift_left_1_5
        srl d                   ; shift the character data
        djnz shift_left_1_5     ; d holds the shifted character data

        push de
        exx
        pop de                  ; d' holds the shifted character data
        ld a,(hl)               ; fetch the video address content
        and b                   ; mask it
        or d                    ; OR print the character
        ld (hl),a               ; print back the result to the screen

        inc h                   ; next line video address
        exx
        inc hl                  ; next character line

        ; ----------------------- row number 6
        ld d,(hl)               ; fetch the character data

        ld b,c                  ; c holds the number of bits to be shifted
shift_left_1_6
        srl d                   ; shift the character data
        djnz shift_left_1_6     ; d holds the shifted character data

        push de
        exx
        pop de                  ; d' holds the shifted character data
        ld a,(hl)               ; fetch the video address content
        and b                   ; mask it
        or d                    ; OR print the character
        ld (hl),a               ; print back the result to the screen

        inc h                   ; next line video address
        exx
        inc hl                  ; next character line

        ; ----------------------- row number 7
        ld d,(hl)               ; fetch the character data

        ld b,c                  ; c holds the number of bits to be shifted
shift_left_1_7
        srl d                   ; shift the character data
        djnz shift_left_1_7     ; d holds the shifted character data

        push de
        exx
        pop de                  ; d' holds the shifted character data
        ld a,(hl)               ; fetch the video address content
        and b                   ; mask it
        or d                    ; OR print the character
        ld (hl),a               ; print back the result to the screen

        inc h                   ; next line video address
        exx
        inc hl                  ; next character line

        ; ----------------------- row number 8
        ld d,(hl)               ; fetch the character data

        ld b,c                  ; c holds the number of bits to be shifted
shift_left_1_8
        srl d                   ; shift the character data
        djnz shift_left_1_8     ; d holds the shifted character data

        push de
        exx
        pop de                  ; d' holds the shifted character data
        ld a,(hl)               ; fetch the video address content
        and b                   ; mask it
        or d                    ; OR print the character
        ld (hl),a               ; print back the result to the screen

;       inc h                   ; next line video address
;       exx
;       inc hl                  ; next character line

        ; ----------------------- print the attributes for the byte involved
        ld a,(attribute)
        ld hl,(wr_attribute)
        ld (hl),a

        ; ----------------------- compute the video address for the next char
        ld a,(x_pos)
        add CHAR_X
        ld (x_pos),a

        jp end_print


                        if TOOFAST = 1
        ; ----------------------- routine to print a single byte unshifted char
        ; ----------------------- this version is too fast: it is not masking the printed data
        ; ----------------------- so an extra space of few bits is created to the right of the
        ; ----------------------- character. If this character is the last in the string
        ; ----------------------- then the extra space could be visible on the screen.
no_shift
        ld bc,(wr_video)        ; take the display file address

        ; ----------------------- row number 1
        ld a,(hl)               ; fetch the character data
        ld (bc),a               ; print it to the screen
        inc hl                  ; next character line
        inc b                   ; next video address

        ; ----------------------- row number 2
        ld a,(hl)               ; fetch the character data
        ld (bc),a               ; print it to the screen
        inc hl                  ; next character line
        inc b                   ; next video address

        ; ----------------------- row number 3
        ld a,(hl)               ; fetch the character data
        ld (bc),a               ; print it to the screen
        inc hl                  ; next character line
        inc b                   ; next video address

        ; ----------------------- row number 4
        ld a,(hl)               ; fetch the character data
        ld (bc),a               ; print it to the screen
        inc hl                  ; next character line
        inc b                   ; next video address

        ; ----------------------- row number 5
        ld a,(hl)               ; fetch the character data
        ld (bc),a               ; print it to the screen
        inc hl                  ; next character line
        inc b                   ; next video address

        ; ----------------------- row number 6
        ld a,(hl)               ; fetch the character data
        ld (bc),a               ; print it to the screen
        inc hl                  ; next character line
        inc b                   ; next video address

        ; ----------------------- row number 7
        ld a,(hl)               ; fetch the character data
        ld (bc),a               ; print it to the screen
        inc hl                  ; next character line
        inc b                   ; next video address

        ; ----------------------- row number 8
        ld a,(hl)               ; fetch the character data
        ld (bc),a               ; print it to the screen
;       inc hl                  ; next character line
;       inc b                   ; next video address

                        else

        ; ----------------------- routine to print a single byte unshifted char
no_shift                        ; d holds the 8 bit mask
        ld bc,(wr_video)        ; take the display file address

        ; ----------------------- row number 1
        ld a,(bc)               ; fetch the video address content
        and d                   ; mask it
        or (hl)                 ; OR print the character
        ld (bc),a               ; print back the result to the screen
        inc hl                  ; next character line
        inc b                   ; next video address

        ; ----------------------- row number 2
        ld a,(bc)               ; fetch the video address content
        and d                   ; mask it
        or (hl)                 ; OR print the character
        ld (bc),a               ; print back the result to the screen
        inc hl                  ; next character line
        inc b                   ; next video address

        ; ----------------------- row number 3
        ld a,(bc)               ; fetch the video address content
        and d                   ; mask it
        or (hl)                 ; OR print the character
        ld (bc),a               ; print back the result to the screen
        inc hl                  ; next character line
        inc b                   ; next video address

        ; ----------------------- row number 4
        ld a,(bc)               ; fetch the video address content
        and d                   ; mask it
        or (hl)                 ; OR print the character
        ld (bc),a               ; print back the result to the screen
        inc hl                  ; next character line
        inc b                   ; next video address

        ; ----------------------- row number 5
        ld a,(bc)               ; fetch the video address content
        and d                   ; mask it
        or (hl)                 ; OR print the character
        ld (bc),a               ; print back the result to the screen
        inc hl                  ; next character line
        inc b                   ; next video address

        ; ----------------------- row number 6
        ld a,(bc)               ; fetch the video address content
        and d                   ; mask it
        or (hl)                 ; OR print the character
        ld (bc),a               ; print back the result to the screen
        inc hl                  ; next character line
        inc b                   ; next video address

        ; ----------------------- row number 7
        ld a,(bc)               ; fetch the video address content
        and d                   ; mask it
        or (hl)                 ; OR print the character
        ld (bc),a               ; print back the result to the screen
        inc hl                  ; next character line
        inc b                   ; next video address

        ; ----------------------- row number 8
        ld a,(bc)               ; fetch the video address content
        and d                   ; mask it
        or (hl)                 ; OR print the character
        ld (bc),a               ; print back the result to the screen
;       inc hl                  ; next character line
;       inc b                   ; next video address

                        endif

        ; ----------------------- print the attributes for the byte involved
        ld a,(attribute)
        ld hl,(wr_attribute)
        ld (hl),a

        ; ----------------------- compute the video address for the next char
        ld a,(x_pos)
        add CHAR_X
        ld (x_pos),a


end_print
        pop hl
        ret

                else

disp_ch
        push hl
        push ix
        sub ' '                 ; SPAZIO = carattere 0 nel font
;       ld e,a                  ; carattere in DE
;       ld d,0
;       sla e                   ; moltiplica per 8 (dimensioni MAX di un carattere nel font)
;       rl d
;       sla e
;       rl d
;       sla e
;       rl d

        ld l,a
        ld h,0
        add hl,hl
        add hl,hl
        add hl,hl               ; moltiplica per 8 (dimensioni MAX di un carattere nel font)
        ex de,hl

        ld iy,font_block        ; indirizzo base FONT
        add iy,de

        ld de,MASK_X_16
;       ld d,$FF                ; carica in DE una maschera di AND 16 bit utile ad azzerare i pixel
;       ld b,CHAR_X             ; sui quali verra` caricato il bitmap del font. Per un font largo 6,
;ldefinemask                    ; DE conterra` $03FF (ovvero 6 bit MSB azzerati, gli altri tutto SET).
;       srl d
;       djnz ldefinemask
;       ld e,$FF

        ld a,(x_pos)            ; posizione X di scrittura (in pixel)
        and 7                   ; 3 bit LSB = bit da non utilizzare nel presente byte
;       and a
        jr z,mask_done
        ld b,a
        scf                     ; sposta la maschera AND in modo corrispondente al pattern da caricare
l_andmask
        rr d
        rr e
        djnz l_andmask
mask_done
        push de
        pop ix                  ; IX = maschera AND

        ld b,8                  ; n. linee da caricare
        ld hl,(wr_video)        ; indirizzo prima linea da caricare

next_row
        push af                 ; salva n. bit gia` occupati nel byte corrente
        ld d,(iy)               ; DE = bitmap della riga di pixel corrente per questo carattere
        ld e,0

l_sh1
        and a                   ; A contiene il numero di pixel gia` usati nel byte corrente
        jr z,shiftdone
        srl d
        rr e                    ; shift a 16 bit verso destra del pattern da caricare

        dec a                   ; alla fine la maschera a 16 bit conterra` una "finestra" di bit RESET
        jr l_sh1                ; nella posizione in cui il nuovo pattern andra` caricato

shiftdone
        ld a,(hl)               ; byte in ram video
        and ixh                 ; maschera AND, azzera i bit da modificare
        or d                    ; inserisce i bit dal bitmap font
        ld (hl),a

        ld a,ixl
        cp 255                  ; se LSB maschera != 255 abbiamo una parte eccedente da scrivere
        jr z,nosecondo          ; nel byte successivo, altrimenti salta.

        inc hl                  ; scrive parte eccedente nel byte alla destra di quello precedente
        ld a,(hl)
        and ixl
        or e
        ld (hl),a
        dec hl

nosecondo
        inc h                   ; la riga successiva si trova ad un offset +256
        inc iy                  ; nuova riga nel font
        pop af
        djnz next_row           ; procede per 8 righe

        ld a,(attribute)
        ld hl,(wr_attribute)
        ld (hl),a
        ld c,a
        ld a,ixl
        cp 255                  ; se LSB maschera != 255 dobbiamo settare anche l'attributo per il byte successivo
        jr z,nos_attr
        ld a,c
        inc hl
        ld (hl),a

nos_attr
        ld a,(x_pos)            ; caricamento completato:
        add CHAR_X              ; aggiunge larghezza del carattere appena visualizzato alla coordinata
        ld (x_pos),a            ; X del prossimo carattere, i cui 3 bit LSB indicano il n. di bit occupati
        cp 255-CHAR_X           ; non c'e' posto per un ulteriore carattere in questa riga?
        jr c,sispace            ; si, salta

        xor a                   ; altrimenti azzera coordinata X
        ld (x_pos),a
        ld a,(y_pos)
        inc a                   ; passa alla riga sottostante (inserire un controllo di SCROLL!)
        ld (y_pos),a
        call compute_add        ; occorre ricalcolare l'indirizzo
        jr norepp

sispace
        and 7                   ; se necessario aggiorna WR_VIDEO, in modo da evitare la necessita`
        cp CHAR_X               ; di richiamare COMPUTE_ADD
        jr nc,norepp
        ld hl,wr_video
        inc (hl)
        ld hl,wr_attribute
        inc (hl)
norepp
        pop ix
        pop hl
        ret

                endif

;------------------------------------------------------------------------------------------------
; Inizializza lo schermo con il colore indicato in ATTRIBUTE
;
; MODDED: Eugenio Ciceri 20080209
;------------------------------------------------------------------------------------------------

                if FASTER = 1

cls     ; this entry point uses the attribute variable to clear the attributes area
        ld a,(attribute)
clsa    ; this entry point expects the attribute code already loaded in the accumulator
        ld hl,$4000
        ld d,h
        ld e,1
        ld b,$18
        ld c,l
        ld (hl),l
        ldir
        ld(hl),a
        ld bc,$02ff
        ldir
        ret

                else

cls
        ld hl,$4000
        ld bc,6144
l_clear
        ld (hl),0               ; pulisce il video
        inc hl
        dec bc
        ld a,b
        or c
        jr nz,l_clear

        ld bc,768               ; spazio attributi
        ld a,(attribute)
        ld e,a
l_attr
        ld (hl),e
        inc hl
        dec bc
        ld a,b
        or c
        jr nz,l_attr
        ret

        endif


;
;------------------------------------------------------------------------------------------------
; Subroutine that displays the C content in HEX format, 2 chars.
;------------------------------------------------------------------------------------------------
disp_hex
        ld hl,tmp_str
        ld (hl),7               ; white color
        inc hl
        ld a,c
        srl a
        srl a
        srl a
        srl a
        ld b,2
nextcc
        ld e,'0'
        cp 10
        jr c,letter
        ld e,'A'-10
letter
        add e
        ld (hl),a
        inc hl
        ld a,c
        and $0F
        djnz nextcc

        ld (hl),0               ; asciz
        ld hl,tmp_str
        call disp_text
        ret



;*************************************************************************************************
;*************************************************************************************************
; MMC READ/WRITE ROUTINES from ZX-Badaloc Firmware Version 4.96b 17/03/2007
;
; RD/WR are on 512 bytes boundary ONLY.
; MULTIPLE_BLOCK READ is supported, while by now MULTIPLE_BLOCK WR is not
;
;*************************************************************************************************
;*************************************************************************************************



;
;---------------------------------------------------------------------------------------------
; This subroutine finds the first available SD-CARD and loads it's CS layout into CS_OUTREG,
; so it will be used by default.
;---------------------------------------------------------------------------------------------
lookfor_sd
        ld a,CS_0               ; D0 LOW = MMC0 selected
        ld (cs_outreg),a

        call sd_check           ; controlla eventuale presenza di SD/MMC
        ld a,(sh_flags)
        bit sd_initsnap,a       ; se e` presente ed inizializzata per i nostri snapshot, non cerca altro
        jr nz,card_okforus

        push af                 ; saves current SH_FLAGS status
        ld a,CS_1               ; D1 LOW = MMC1 selected
        ld (cs_outreg),a
        call sd_check           ; controlla eventuale presenza di SD/MMC
        pop bc
        ld a,(sh_flags)         ; se assente, riseleziona per default l'altra card. E` possibile che nello
        bit sd_present,a        ; slot 0 ce ne sia una, anche se non inizializzata per i nostri snapshot.
        jr nz,card_okforus
        ld a,CS_0               ; D0 LOW = MMC0 selected
        ld (cs_outreg),a
        ld a,b                  ; restores previous FLAGS state (card0 found or not)
        ld (sh_flags),a
        bit sd_present,a
        jr nz,card_okforus
        ld a,CS_DISABLE         ; no cards
        ld (cs_outreg),a
card_okforus
        ret


;
;---------------------------------------------------------------------------------------------
; Subroutine che verifica la presenza di una SD/MMC inserita nello slot.
; Se la trova ne visualizza il CID e verifica che i primi due bytes del secondo blocco
; contengano $AA55. Se e` cosi`, legge il numero del prossimo snapshot dai primi due bytes
; della card, altrimenti inizializza entrambi i campi. Setta il flag SD_PRESENT.
;
; Nuova versione che controlla a diversi offset nella card, spaziati di OFFSET_GRAN
; (valore unitario = 16MB) fino al superamento di MAXOFFSET. Serve per poter usare una card
; con gli snapshot e avere una prima parte libera per qualche filesystem.
;---------------------------------------------------------------------------------------------
sd_check
        ld a,(sh_flags)
        res sd_present,a
        res sd_initsnap,a
        ld (sh_flags),a

        ld hl,SHADOW_RAM        ; tenta di leggere il CID di una eventuale SD/MMC salvandolo a SHADOW_RAM
        call mmc_getcid_poweron
        and a
        ret nz                  ; errore

        ld hl,mmc               ; visualizza il CID della SD rilevata
        ld a,3
        ld (x_pos),a
        ld a,23
        ld (y_pos),a
        call disp_text
        ld hl,SHADOW_RAM+8
        xor a                   ; la stringa e` ASCIZ
        ld (hl),a
        ld hl,SHADOW_RAM        ; primo carattere stampabile (per 7 caratteri) = +1; questo viene
        ld a,7                  ; sovrascritto con il codice dell'attributo desiderato
        ld (hl),a
        call disp_text

        ld hl,sh_flags
        set sd_present,(hl)

        xor a
        ld (card_offset),a      ; starts from beginning of card (offset 0)
l_newoffset
        ld l,0
        ld h,a                  ; card_offset is loaded on 4o byte (address bits 31-24) of card absolute address
        ld de,$200              ; second 512 bytes block
        ld ix,SHADOW_RAM
        call mmc_read_data      ; performs the single block read
        and a
        jr z,rseek_ok
        ld a,(card_offset)      ; read error: marks the last successful read address
        dec a                   ; and returns
        ld (card_offset),a
        ret

rseek_ok
        ld hl,(SHADOW_RAM)      ; checks the first 2 bytes of the second block
        ld a,l                  ; that should contain $AA55 for cards that have been
        cp $55                  ; initialized for our proprietary snapshot filesystem
        jr nz,no_mmcok
        ld a,h
        cp $AA
        jr z,check_ok

no_mmcok                        ; now we seek for header on other card offsets
        ld a,(card_offset)
        add OFFSET_GRAN         ; offset granularity (16 = +256MB)
        ld (card_offset),a
        cp MAXOFFSET+1
        ret nc                  ; none of card offsets contains a valid header
        jr l_newoffset
        ret

sd_init_for_snap                ; SD INIT ENTRY POINT
        ld a,(card_offset)
        ld h,a
        ld l,0
        push hl
        ld de,0
        ld b,2
        ld ix,SHADOW_RAM
        call mmc_read_multidata ; reads the first 2 blocks
        ld hl,$AA55
        ld (SHADOW_RAM+512),hl  ; initializes the identifier
        ld hl,0
        ld (SHADOW_RAM),hl      ; initializes the "first snapshot entry" counter

        pop hl
        ld de,0
        ld b,2
        ld ix,SHADOW_RAM
        call mmc_write_data     ; writes the data back to the SD/MMC

check_ok
        ld a,(card_offset)
        ld h,a
        ld l,0
        ld de,0                 ; first 512 bytes block
        ld ix,SHADOW_RAM
        call mmc_read_data      ; performs the single block read
        ld hl,(SHADOW_RAM)
        ld (firstfree),hl

        ld hl,sh_flags          ; this flag indicates that the present SD/MMC has been found to be
        set sd_initsnap,(hl)    ; initialized for our proprietary snapshot scheme

        ld iy,tmp_str+2
        ld (iy-2),7             ; color
        ld (iy-1),'+'
        ld a,(card_offset)
        ld l,a                  ; valore offset attuale (unitario = 16MB)
        ld h,0
        sla l                   ; moltiplica per 16 per esprimere in megabytes
        rl h
        sla l
        rl h
        sla l
        rl h
        sla l
        rl h
        ld a,(flags2)
        set suppress_leadz,a
        ld (flags2),a
        call conv_4cifre
        ld (iy+1),'M'
        ld (iy+2),'B'
        ld (iy+3),0
        ld a,132
        ld (x_pos),a
        ld hl,tmp_str
        call disp_text

        ret


;
;---------------------------------------------------------------------------------------------
; Subroutine che carica l'attributo A alla posizione indicata da START_ITEM
;---------------------------------------------------------------------------------------------
showsel_item
        push hl
        ld c,a                  ; attributo

        ld a,(start_item)
        ld d,a
        ld e,0
        srl d
        rr e
        srl d
        rr e
        srl d
        rr e
        ld hl,$58A0             ; posizione prima voce di menu
        add hl,de

        ld a,(flags2)
        bit restore_attrib,a    ; dobbiamo ripristinare l'attributo precedentemente sovrascritto?
        jr z,norestore
        res restore_attrib,a
        ld (flags2),a
        ld a,(saved_attrib)
        ld c,a
        jr nosave
norestore
        bit save_attrib,a       ; dobbiamo salvare l'attributo prima di sovrascriverlo?
        jr z,nosave
        res save_attrib,a
        ld (flags2),a
        ld a,(hl)
        ld (saved_attrib),a
nosave
        ld (hl),c               ; cubetto multicolore scelta di default?
        inc hl
        ld (hl),c
        pop hl
        ret


;
;-------------------------------------------------------------------------------------------
; This subroutine should be called at power-on or when a MMC has been inserted.
; It tries to get the MMC CID (writing at the provided HL pointer) and, in case of failure,
; it calls the INIT procedure then tries again.
;
; Returns A = 0 if OK, or 1 = error (no MMC found or MMC error)
;
; iF != 0 AND != $ff, the mmc_get_cid returned error code is displayed on screen.
;-------------------------------------------------------------------------------------------
mmc_getcid_poweron
        call mmc_get_cid                ; try to read the MMC CID INFO --> (HL)
        and a                           ; if it fails, then che SD/MMC needs to be initialized
        ret z                           ; to SPI mode (MMC_INIT) communications (once after power-on)

        cp 255                          ; card is probably not in SPI mode
        jr z,need_init

        bit 0,a                         ; IDLE/still initializing bit
        jr nz,need_init

        ld c,a                          ; unknown response: displays the code and exits
        ld hl,mmcreturned
        xor a
        ld (x_pos),a
        ld a,21
        ld (y_pos),a
        call disp_text
        call disp_hex                   ; displays HEX code in C

        ld a,1                          ; error
        ret

need_init
        call mmc_init
        and a
        ret nz                          ; INIT error: MMC not detected.

        ld de,BLOCKSIZE
        call mmc_send_blocksize
        jr mmc_getcid_poweron

mmcreturned
        db      2,"MMC send_cmd returned  $",0


;
;-----------------------------------------------------------------------------------------
; READ MULTIPLE BLOCK OF DATA TEST subroutine
;
; This routine only works for blocksize = 512 (two INI sequence).
;
; HL, DE= MSB, LSB of 32bit address in MMC memory
; B     = number of 512 bytes blocks to be read
; IX    = ram buffer address
;
; RETURN code in A:
; 0 = OK
; 1 = read_block command error
; 2 = no wait_data token from MMC
;
; DESTROYS AF, B
;-----------------------------------------------------------------------------------------
mmc_read_multidata
        ld (tmp_reg),bc                 ; temporary parameter's saveplace for retry function
        ld (tmp_reg+2),de
        ld (tmp_reg+4),hl
        ld a,5                          ; # of retries
        ld (tmp_reg+6),a
        jr readtry_again

read_bsod
        if DEBUGMODE = 1
        call bsod                       ; black screen of death
        endif

        ld a,MCC_TERMINATE_MULTI_READ
        call mmc_write_command
        call cs_high                    ; set cs high
        call clock32                    ; 32 more clock cycles
        call clock32                    ; 32 more clock cycles
        call clock32                    ; 32 more clock cycles
        call clock32                    ; 32 more clock cycles
        ld bc,(tmp_reg)
        ld de,(tmp_reg+2)
        ld hl,(tmp_reg+4)
        ld a,(tmp_reg+6)
        dec a
        ld (tmp_reg+6),a
        and a
        jr nz,readtry_again
        ld a,2                          ; no data token from MMC
        ld hl,(tmp_reg+4)               ; HL should be restored. DE was not modified.
        ret

readtry_again
        ld a,MMC_READ_MULTIPLE_BLOCK    ; Command code for multiple block read
        call mmc_send_command
        and a
        ret nz                          ; ERRORE

        push ix
        pop hl                          ; INI usa HL come puntatore

jrhere
        call mmc_waitdata_token
        cp $FE
        jr nz,read_bsod                 ; error

read_mmc_multiblock
        push bc
        ld bc,SPI_PORT                  ; B = 0 = 256 bytes for the first INI; C = I/O port

        if USE_INI = 0
        inir
        nop
        inir
        endif

        if USE_INI = 1
ini_loop1
        ini
        ini
        ini
        ini
        ini
        ini
        ini
        ini
        ini
        ini
        ini
        ini
        ini
        ini
        ini
        ini
        jr nz,ini_loop1
ini_loop2
        ini
        ini
        ini
        ini
        ini
        ini
        ini
        ini
        ini
        ini
        ini
        ini
        ini
        ini
        ini
        ini
        jr nz,ini_loop2
        endif

        nop
        nop

        in a,(SPI_PORT)                 ; CRC
        nop
        nop
        in a,(SPI_PORT)                 ; CRC

        pop bc
        djnz jrhere

        ld a,MCC_TERMINATE_MULTI_READ
        call mmc_write_command

        in a,(SPI_PORT)                 ; CRC?
        nop
        nop
        in a,(SPI_PORT)

        call mmc_wait_response          ; waits for the MMC to reply "0"
        ld b,a                          ; error code

        call cs_high                    ; set cs high
        call clock32                    ; 32 more clock cycles
        call clock32                    ; 32 more clock cycles
        call clock32                    ; 32 more clock cycles
        call clock32                    ; 32 more clock cycles

        ld hl,(tmp_reg+4)               ; HL should be restored. DE was not modified.
        ld a,b
        ret


;
;-----------------------------------------------------------------------------------------
; READ A SINGLE BLOCK OF DATA subroutine
;
; This routine only works for blocksize = 512 (two INIR sequence).
;
; HL, DE= MSB, LSB of 32bit address in MMC memory
; IX    = ram buffer address
;
; RETURN code in A:
; 0 = OK
; 1 = read_block command error
; 2 = no wait_data token from MMC
;
; DESTROYS AF, HL
;-----------------------------------------------------------------------------------------
mmc_read_data

        ld a,MMC_READ_SINGLE_BLOCK      ; Command code for block read
        call mmc_send_command
        and a
        ret nz                          ; ERRORE

        call mmc_waitdata_token
        cp $FE
        jr z,read_mmc_block             ; OK
        ld a,2                          ; no data token from MMC
        ret

read_mmc_block
        push ix
        pop hl                          ; HL = INIR write pointer

        push bc
        ld bc,SPI_PORT                  ; B = 0 = 256 bytes for the first INIR; C = I/O port

        inir
        nop
        inir

        nop
        nop

        in a,(SPI_PORT)                 ; CRC
        nop
        nop
        in a,(SPI_PORT)                 ; CRC

        call cs_high                    ; set cs high
        call clock32                    ; 32 more clock cycles

        pop bc

        xor a                           ; no error
        ret


clock32
        push bc
        ld b,4
l_4bytes
        in a,(SPI_PORT)                 ; some more clock cycles
        djnz l_4bytes
        pop bc
        ret


;-----------------------------------------------------------------------------------------
; READ FIRST 2 BYTES OF SD/MMC CARD and copies them to DE.
;
; RETURN code in A:
; 0 = OK
; 1 = read_block command error
; 2 = no wait_data token from MMC
;
; DESTROYS AF, DE
;-----------------------------------------------------------------------------------------
mmc_get_firstword
        push hl
        ld a,(card_offset)              ; MSB byte (bit 31-24) = snapshot area offset (1 = +16MB)
        ld h,a
        ld l,0
        ld de,0
        ld a,MMC_READ_SINGLE_BLOCK      ; Command code for block read
        call mmc_send_command
        pop hl
        and a
        ret nz                          ; ERRORE
        call mmc_waitdata_token
        cp $FE
        jr z,read_mmc_first             ; OK
        ld a,2                          ; no data token from MMC
        ret

read_mmc_first
        in a,(SPI_PORT)
        ld e,a
        nop
        in a,(SPI_PORT)
        ld d,a

        push bc
        ld bc,BLOCKSIZE-2               ; reads the remaining 510 bytes and throws them
read_firstword
        in a,(SPI_PORT)
        dec bc
        ld a,b
        or c
        jr nz,read_firstword
        pop bc

        in a,(SPI_PORT)                 ; CRC
        nop
        nop
        in a,(SPI_PORT)                 ; CRC

        call cs_high                    ; set cs high
        call clock32                    ; 32 more clock cycles

        xor a
        ret

;
;-----------------------------------------------------------------------------------------
; WRITE BLOCK OF DATA subroutine. By now, we don't use the MULTIPLE_BLOCK transfer.
;
; This routine only works for blocksize = 512 (two OUTI sequence).
;
; HL, DE= MSB, LSB of 32bit address in MMC memory
; B     = number of 512 bytes blocks to write
; IX    = ram buffer address
;
; RETURN code in A:
; 0 = OK
; 1 = read_block command error
; 2 = write error (no "5" response from MMC)
;
; Destroys AF, B, DE, HL, IX.
;-----------------------------------------------------------------------------------------
mmc_write_data
        ld a,MMC_WRITE_SINGLE_BLOCK     ; Command code for block read
        call mmc_send_command
        and a
        ret nz                          ; ERRORE

        ld a,$FE
        out (SPI_PORT),a                ; first byte to be sent = DATA TOKEN

write_mmc_block
        push bc
        ld bc,SPI_PORT                  ; C = PORT, B = 0 for 256 bytes on first OTIR
        push hl
        push ix
        pop hl

        otir
        nop                             ; without this NOP, the 2nd half of the 512 bytes block would miss some write cycles
        otir                            ; on the CPLD if Z80 runs at 21MHz

        push hl
        pop ix
        pop hl

        ld a,$95                        ; CRC... (!)
        out (SPI_PORT),a
        nop
        nop
        out (SPI_PORT),a

        call mmc_wait_response

        pop bc

        and $1F                         ; masks useful response bits
        cp 5
        jr nz,write_error

        ld a,d                          ; aggiorna il puntatore al blocco da scrivere
        add BLOCKSIZE/256               ; 2 se blocksize = 512
        jr nc,no_overw
        inc hl
no_overw
        ld d,a

wait_busy
        call mmc_wait_response          ; MMC will report "00" until busy
        and a
        jr z,wait_busy

        call cs_high
        call clock32                    ; 32 more clock cycles

        djnz mmc_write_data             ; next block

        xor a
        ret

write_error
        ld a,2                          ; write error code
        ret


;
;-----------------------------------------------------------------------------------------
; MMC SPI MODE initialization. RETURNS ERROR CODE IN A register:
;
; 0 = OK
; 1 = Card RESET ERROR
; 2 = Card INIT ERROR
;
; Destroys AF, B.
;-----------------------------------------------------------------------------------------
mmc_init
        call    cs_high                 ; set cs high
        ld      b,10                    ; sends 80 clocks
        ld a,$FF
l_init
        out (SPI_PORT),a
        djnz l_init
        nop

        call    cs_low                  ; set cs low

        ld a,MMC_GO_IDLE_STATE
        call mmc_write_command

        call mmc_wait_response
        cp $01                          ; MMC should respond 01 to this command
        jr nz,mmc_reset_failed          ; fail to reset

        ld bc,120                       ; retry counter*256 (corrisponde a circa 5 secondi @3.5MHz)
mmc_reset_ok
        call cs_high                    ; set cs high
        ld a,$FF
        out (SPI_PORT),a                ; 8 extra clock cycles
        nop
        nop

        call    cs_low                  ; set cs low

        ld a,MMC_SEND_OP_COND           ; Sends OP_COND command
        call mmc_write_command

        call mmc_wait_response          ; MMC_WAIT_RESPONSE tries to receive a response reading an SPI

        bit 0,a                         ; D0 SET = initialization still in progress...
        jr z,mmc_init_ok

        djnz mmc_reset_ok               ; if no response, tries to send the entire block 254 more times
        dec c
        jr nz,mmc_reset_ok

        ld a,2                          ; error code for INIT ERROR
        jr mmc_errorx

mmc_init_ok
        call cs_high                    ; set cs high
        in a,(SPI_PORT)                 ; some extra clock cycles
        call pause1
        xor a
        ret

mmc_reset_failed                        ; MMC Reset error
        ld      a,1
mmc_errorx
        call    cs_high
        ret


;debug_display                          ; TEST: displays the code in A at current cursor position
;       push af                         ; ('A' based): A = 0x00, B = 0x01, etc.
;       push bc
;       push de
;       push hl
;       add 'A'
;       call disp_ch
;       ld bc,0
;lll
;       djnz lll
;       dec c
;       jr nz,lll
;       pop hl
;       pop de
;       pop bc
;       pop af
;       ret


;
;-----------------------------------------------------------------------------------------
; READ MMC CID subroutine. Data is stored at address (HL).
;
; Returns error code in A register:
; 0 = no error
; 1 = Read CID command error
; 2 = no wait_data token from MMC
;
; Destroys AF, HL.
;-----------------------------------------------------------------------------------------
mmc_get_cid
        push bc
        push de
        push hl
        ld hl,0
        ld de,0
        ld a,MMC_READ_CID
        call mmc_send_command
        pop hl
        and a
        jr z,cmd_cidok                  ; MMC_SEND_COMMAND reports 0 (ok) or MMC error code
        push af                         ; error code on STACK
        jr cid_exit
cmd_cidok
        call mmc_waitdata_token
        cp $FE
        jr z,waitda_cid_ok
        ld a,2
        push af
        jr cid_exit

waitda_cid_ok
        ld b,18                         ; 16 bytes + CRC?
        ld c,SPI_PORT

        inir

;mmc_cidl
;       ini
;       jr nz,mmc_cidl

        xor a
        push af
cid_exit
        call cs_high                    ; set cs high
        in a,(SPI_PORT)
        pop af                          ; error code
        pop de
        pop bc
        ret


;
;-----------------------------------------------------------------------------------------
; SEND COMMAND TO MMC subroutine
;
; A = COMMAND CODE;
; H, L, D, E = 32 bit parameter (MSB ... LSB);
;
; Sends a $FF fake checksum
;
; RETURNS: 0 = OK; != 0 = MMC error code
;
; On OK, the CHIP SELECT will be LOW on exit
; On error, the CHIP SELECT will be deasserted on exit
;
; Destroys AF.
;-----------------------------------------------------------------------------------------
mmc_send_command
        push bc
        ld c,a
        call cs_high                    ; cs high
        call clock32
        nop
        call cs_low                     ; cs low
        ld a,c                          ; command code is the first byte to be sent
        out (SPI_PORT),a
        ld a,h
        nop
        out (SPI_PORT),a
        ld a,l
        nop
        out (SPI_PORT),a
        ld a,d
        nop
        out (SPI_PORT),a
        ld a,e
        nop
        out (SPI_PORT),a
        ld a,$ff
        nop
        out (SPI_PORT),a

        call mmc_wait_response          ; waits for the MMC to reply != $FF
        and a
        jr nz,mmc_commande
        pop bc
        ret                             ; 0 = no error

mmc_commande
        push af                         ; saves the error code
        call cs_high                    ; set cs high
        in a,(SPI_PORT)
        pop af
        pop bc
        ret                             ; returns the error code got from MMC


;
;-----------------------------------------------------------------------------------------
; WAIT FOR DATA TOKEN ($FE) subroutine (calls MMC_WAIT_RESPONSE up to 256 times)
; Returns with A = code read from MMC (ok if $FE). Destroys AF.
;-----------------------------------------------------------------------------------------
mmc_waitdata_token
        push bc
        ld b,10                         ; retry counter
mmc_waitl
        call mmc_wait_response
        cp $FE                          ; waits for the MMC to reply $FE (DATA TOKEN)
        jr z,exit_wda
        cp $FF                          ; but if not $FF, exits immediately (error code from MMC)
        jr nz,exit_wda
        djnz mmc_waitl
exit_wda
        pop bc
        ret


;
;-----------------------------------------------------------------------------------------
; SENDS BLOCK_SIZE COMMAND TO MMC.
;
; Parameters: DE = nbytes. Destroys AF.
;-----------------------------------------------------------------------------------------
mmc_send_blocksize
        call    cs_low                  ; set cs low
        ld a,MMC_SET_BLOCK_SIZE
        out (SPI_PORT),a
        xor a
        nop
        out (SPI_PORT),a
        nop
        nop
        out (SPI_PORT),a
        ld a,d
        nop
        out (SPI_PORT),a
        ld a,e
        nop
        out (SPI_PORT),a
        ld a,$ff                        ; fake checksum
        nop
        out (SPI_PORT),a
        nop
        nop

        in a,(SPI_PORT)
        nop
        nop
        in a,(SPI_PORT)
        nop
        nop
        call    cs_high

        ret


;
;-----------------------------------------------------------------------------------------
; Sends a command with parameters = 00 and checksum = $95. Destroys AF.
;-----------------------------------------------------------------------------------------
mmc_write_command
        push bc
        out (SPI_PORT),a        ; sends the command
        ld b,4
        xor a
l_sendc0
        out (SPI_PORT),a        ; then sends four "00" bytes (parameters = NULL)
        djnz l_sendc0
        ld a,$95                ; $95 is only needed when the CARD INIT is being performed,
        nop
        out (SPI_PORT),a        ; then this byte is ignored.
        pop bc
        ret


;
;-----------------------------------------------------------------------------------------
; Waits for the MMC to respond. Returns with A = response code; $FF = NO RESPONSE
;
; Responses from CARD are in R1 format for all command except SEND_STATUS.
; When SET, a bit indicates:
;
; D0 = Idle state / init not completed yet
; D1 = Erase Reset
; D2 = Illegal Command
; D3 = Com CRC Error
; D4 = Erase Sequence Error
; D5 = Address Error
; D6 = Parameter Error
; D7 = ALWAYS LOW
;
; Destroys AF.
;-----------------------------------------------------------------------------------------
mmc_wait_response
        push bc
        ld bc,50                ; retry counter
l_response
        in a,(SPI_PORT)         ; reads a byte from MMC
        cp $FF                  ; $FF = no card data line activity
        jr nz,resp_ok
        djnz l_response
        dec c
        jr nz,l_response
resp_ok
        pop bc
        ret


;
;------------------------------------------------------------------------------------
; ALL CHIP_SELECT HIGH subroutine. Destroys no registers.
;
; If we are within the NMI handler, then keeps NMI on start bit DISABLED; otherwise,
; the setting in CS_OUTREG for this bit is kept.
;------------------------------------------------------------------------------------
cs_high
        push af
        ld a,(nmi_count)
        and a                   ; != 0 means that we are servicing an NMI
        ld a,(cs_outreg)
        jr z,maybenmi
        and CS_DISABLE          ; keeps NMI inactive
maybenmi
        or CS_DISABLE           ; ensures no CS active
        out (OUT_REG),a
        pop af
        ret


;
;------------------------------------------------------------------------------------
; CHIP_SELECT 0/1 (D0/D1) LOW subroutine (based on MMCID content). Destroys no registers.
;------------------------------------------------------------------------------------
cs_low
        push af
        ld a,(nmi_count)
        and a                   ; != 0 means that we are servicing an NMI
        ld a,(cs_outreg)
        jr z,maybenmi2
        and CS_DISABLE          ; keeps NMI inactive
maybenmi2
        out (OUT_REG),a
        pop af
        ret


pause1
        push hl
        ld hl,$8000             ; OK for 3.5MHz ONLY (7MHz requires two calls or ld hl,0 and so on)
loop3   dec hl
        ld a,h
        or l
        jr nz,loop3
        pop hl
        ret



;*****************************************************************************************************************
; RIPRISTINO SNAPSHOT DA SD/MMC 16K/48K/128K
;
; Vecchie considerazioni durante la stesura su zx-badaloc senza flashrom:
; Lo snapshot viene preso mediante pulsante NMI + tasto perdeterminato, in modo da non MODIFICARE NULLA
; in ram o sullo schermo (come per il cambio di clock). Il ripristino, invece, avviene dal menu` principale
; al power-on (come per la scelta di una ROM). SE STIAMO GIRANDO IN SHADOW_ROM per collaudare il software,
; non avremo il CONTEXT MODE in quanto saremo arrivati NON da un vero POWER-ON ma da un upload che lo avra`
; disattivato. Quindi, con buone probabilita` di riuscita, testiamo lo stato del bit SHADOW RD ENABLE e
; se lo troviamo attivo eseguiamo la rilocazione in RAM (video: non c'e' altro posto sicuro, dopo il ripristino
; dello snapshot) mentre in caso contrario cio` non avviene (collaudare se funziona una volta che sara`
; fatta l'EPROM). La rilocazione comprende la parte di codice che altera il ROM_BANK.
;
; Non esegue il ripristino se il numero di snapshot specificato e` oltre la capacita` della SD/MMC.
;*****************************************************************************************************************
; READ EXAMPLE
read_snap
        call check_hlspace              ; controlla se lo snapshot specificato in HL e` oltre la capacita` della SD/MMC
        jr c,snap_ok                    ; no, possiamo ripristinare

        ld a,2
        out (BORDER),a                  ; segnala l'errore
        jp mainloop                     ; errore: rientra in loop

snap_ok
        ex hl,de                        ; number of snapshot in DE

        ld hl,0
        add hl,sp                       ; SP in HL
        ld (real_sp),hl                 ; saves to a shadow_rom variable
        ld sp,SHADOW_STACK		; stack is then moved to zxmmc+ ram. this makes everything far simpler.

        ex hl,de
        ld (snap_torestore),hl          ; salva il numero dello snapshot da ripristinare
        ex hl,de

        call calc_hldec                 ; calcola l'indirizzo del blocco --> HL, DE e l'offset 64 bytes in C

        ld ix,SHADOW_RAM                ; scratchpad area
        call mmc_read_data              ; performs the single block read
        and a
        jr z,restore_ok1

restore_error
        add a,2
        out (BORDER),a                  ; segnala l'errore: colore del border indica il codice

restore_abort
        call cs_high
        ld hl,(real_sp)                 ; restores the SP
        ld sp,hl
        jp mainloop

restore_ok1
        ld b,c                          ; BC = offset 64 bytes entry da leggere
        ld c,0
        srl b
        rr c
        srl b
        rr c

        ld hl,SHADOW_RAM                ; scratchpad area
        add hl,bc                       ; aggiunge l'offset relativo all'entry da leggere (ce ne sono 8 in 512 bytes)
        ld (entry_addr),hl

        ld ix,(entry_addr)
        ld a,(ix+2)                     ; n. blocchi da ripristinare: 1 = 16K, 2 = 48K, 8 = 128K

        cp MMC_SNAP16                   ; 1
        jr z,snap_valid_nb
        cp MMC_SNAP48                   ; 2
        jr z,snap_valid
        cp MMC_SNAP128                  ; 8
        jr z,snap_valid_nb
        cp MMC_RAMBACKUP
        jr z,snap_valid_nb
        cp MMC_FLASHBACKUP
        jr z,snap_valid_nb
        jr restore_error

snap_valid
        ld a,(ix+45)                    ; stato del registro $7FFD (S128KREG)
        and 7
        ld bc,S128KREG                  ; impostiamo subito il RAM_BANK: in caso di snap 48K sara` bene scrivere
        out (c),a                       ; nel banco che alla fine diventera` quello attivo

        ld bc,PLUS3REG                  ; si accerta che non sia attiva una modalita` di indirizzamento speciale:
        xor a                           ; impedirebbe l'azione del RAM_BANK nella scelta dei banchi 0-7 su $C000-$FFFF,
        out (c),a                       ; che e` l'indirizzo di ripristino per i 128K
                                        ; notare che D2 e` A15 della ROM, ma noi stiamo girando in CONTEXT MODE

snap_valid_nb
        ld hl,(snap_torestore)
        ex hl,de
        ld l,FATSIZE                    ; n. blocchi da 64KB che costituiscono la FAT (e` l'offset per la word MSB)
        ld a,(card_offset)
        ld h,a
        sla e
        rl d                            ; moltiplica per 2 (ogni entry corrisponde ad un cluster da 128KB)
        add hl,de                       ; HL = word MSB lettura SD/MMC

        ld de,0                         ; word LSB = 0 (insieme a D0 = 0 della word MSB = 128K)

        ld a,(ix+2)                     ; n. blocchi da ripristinare: 1 = 16K, 2 = 48K, 8 = 128K
        cp MMC_RAMBACKUP
        jp nz,no_restorebackup


;
; Ripristino di un backup RAM 512K zxmmc+
;
	push hl
	push de
        ld hl,restore_confirm
	call yesno_confirm
	pop de
	pop hl
	jr nc,restore_abort

	xor a
	ld (backupmedia),a		; flag "restore ram"

	exx
        ld hl,ramread           	; parte da eseguire in ram spectrum
        ld bc,ramreadend-ramread
        ld de,EXECUTE_RAM
        ldir
	exx

        in a,(FASTPAGE)
        ld (fastp_save),a       	; FASTPAGE content is saved (in spectrum ram)

        ld a,32                 	; blocchi da 16K da leggere
l_ramrestore
        push af

        ld b,16384/BLOCKSIZE    	; dimensioni per uno snapshot da 16K
        ld ix,SCRATCH           	; ram buffer address
        call mmc_read_multidata 	; reads the card
        and a
        jp nz,restore_error
        ld a,$40
        add d                   	; adding $40 to D means 16KBytes displacement
        jr nc,no_flohl
        inc hl                  	; increase MSB pointer
no_flohl
        ld d,a

        pop bc                  	; B = numero blocco corrente
        push bc

        push hl
        push de

        ld a,$E0
        sub b                   	; lo trasforma in numero 0-31 con D6 (ram paged in) e D7 (ram WR) attivi

        call EXECUTE_RAM        	; esecuzione in ram sinclair

        pop de                  	; indirizzo di lettura in SD-CARD
        pop hl
        pop af
        dec a
        cp 1                    	; NON RIPRISTINA L'ULTIMO BANCO (bootrom!)
        jr nz,l_ramrestore

end_ramromrestore
        ld hl,(real_sp)         	; restores the SP
        ld sp,hl

        call cls
        xor a
        ld (x_pos),a
        ld (y_pos),a
        out (BORDER),a
        ld hl,restore_done1
        ld a,(backupmedia)
	push af
        cp 0
	jr z,ramdone
	ld hl,restore_rom_done
ramdone
        call disp_text
	pop af
        cp 0
	jr nz,ranykey
        ld a,2
        ld (y_pos),a
        xor a
        ld (x_pos),a
        ld hl,restore_done2
        call disp_text
ranykey
        call key_scan
        ld a,e
        cp $ff                          ; no keys
        jr z,ranykey
        call cls
        call disp_menu
        call display_mmcdata
        jp mainloop

ramread
        out (FASTPAGE),a        	; seleziona il banco
        and 7
        out (BORDER),a
        ld hl,SCRATCH
        ld de,0
        ld bc,16384
        ldir                    	; vi copia i dati
        ld a,(fastp_save)
        out (FASTPAGE),a        	; attiva la FLASHROM:
        ret	                	; rientra
ramreadend

restore_done1
        db      7,"512K Ram content has been restored",0
restore_done2
        db      7,"except last 16K bank (firmware)",0
restore_rom_done
        db      7,"Bootrom is reloaded from Flash on hardwareReset. Press any key.",0

restore_confirm
	db	7,"Confirm 512K Ram Restore? (Y/N)",0
romrestore_confirm
	db	7,"Confirm 512K Flash Restore? (Y/N)",0


no_restorebackup

        cp MMC_FLASHBACKUP
        jp nz,no_flashrestorebackup

;
; Ripristino di un backup FLASH 512K zxmmc+
;
	push hl
	push de
        ld hl,romrestore_confirm
	call yesno_confirm
	pop de
	pop hl
	jp nc,restore_abort

	ld a,1
	ld (backupmedia),a		; flag "restore rom"

        ld sp,EXECUTE_VARS      	; stack fuori dall'area di esecuzione in RAM SINCLAIR (si estende verso il basso).
					; lo stack non puo` restare in ram zxmmc+ perche` le routines di scrittura in FLASH
					; lo utilizzano e tale area non e` disponibile durante le operazioni.

        in a,(FASTPAGE)
        ld (fastp_save),a       	; FASTPAGE content is saved (in spectrum ram)

        ld a,8                  	; blocchi da 64K da leggere
l_romrestore
        push af

	exx
        ld hl,ramread           	; parte da eseguire in ram spectrum
        ld bc,ramreadend-ramread
        ld de,EXECUTE_RAM
        ldir
	exx

        ld a,4                  	; blocchi da 16K da leggere per completare un banco da 64K
l_romrestoreinner
        push af

        ld b,16384/BLOCKSIZE    	; dimensioni per uno snapshot da 16K
        ld ix,SCRATCH           	; ram buffer address
        call mmc_read_multidata 	; reads the card
        and a
        jp nz,restore_error
        ld a,$40
        add d                   	; adding $40 to D means 16KBytes displacement
        jr nc,romno_flohl
        inc hl                  	; increase MSB pointer
romno_flohl
        ld d,a

        pop bc                  	; B = numero blocco corrente contatore INNER, quindi 1-4
        push bc

        push hl
        push de

        ld a,$C4			; D7, D6 SET (ram WR, ram paged in) e base per sottrazione = 4
        sub b                   	; lo trasforma in numero 0-3
	add RAMBANK			; utilizza i 4 blocchi preposti all'uso

        call EXECUTE_RAM        	; esecuzione in ram sinclair

        pop de                  	; indirizzo di lettura in SD-CARD
        pop hl
        pop af
        dec a
        jr nz,l_romrestoreinner		; completa la lettura di 4 blocchi, pari a 64K

	pop af				; outer loop counter
	push af
	rlca
	rlca				; moltiplica per 4
	ld c,a
	ld a,32
	sub c				; A = numero del banco base destinazione per questi 64K
	ld (rom_64block),a

	push hl
	push de

	call cls
        xor a
        ld (y_pos),a

	call do_64K			; programma questi 64K nella flash

	pop de
	pop hl

	pop af
	dec a
	jr nz,l_romrestore

	jp end_ramromrestore


no_flashrestorebackup
        ld b,16384/BLOCKSIZE
        cp MMC_SNAP48
        jr nz,ripr_this
        ld b,49152/BLOCKSIZE

ripr_this
        cp MMC_SNAP128                  ; A = 8 = snapshot 128K
        jr z,restore_128sd

        ld ix,$4000                     ; ram buffer (restore point)
        call mmc_read_multidata         ; 16K/48K restore. we can't rely on stack for temporary data
        and a                           ; error?
        jp nz,restore_error
        jr proceed_restore

restore_128sd
        ld c,0
        ld ix,$C000                     ; restore buffer write address
r_128ksd
        ld a,c                          ; current bank
        exx
        ld bc,S128KREG                  ; BC' for CPLD addressing
        out (c),a                       ; new ram_bank
        exx

        ld b,16384/BLOCKSIZE
        call mmc_read_multidata         ; reads this 16K bank without affecting IX, HL and DE
        and a
        jp nz,restore_error
        ld a,$40
        add d                           ; adding $40 to D means 16KBytes displacement
        jr nc,no_flowhl
        inc hl                          ; increase MSB pointer
no_flowhl
        ld d,a

        inc c                           ; next ram bank
        ld a,c
        and 7
        out (BORDER),a                  ; shows progress
        jr nz,r_128ksd                  ; roll over = compleded


proceed_restore
        ld a,(BORDCR)                   ; colore attuale del border *8 (variabile del basic sinclair)
        rra
        rra
        rra
        and 7                           ; rimette a posto il colore originale
        out (BORDER),a                  ; gli altri bit possono essere ZERO (SPKR, TAPE)

        ld ix,(entry_addr)

        ld a,(ix+44)
        ld i,a                          ; ripristina lo stato del registro I

        ld l,(ix+42)                    ; RIPRISTINA LO STATO DELLO STACK POINTER
        ld h,(ix+43)
        ld sp,hl

        ; FASE RIPRISTINO REGISTRI CPLD, IM_MODE E INT ENABLE da E', D', H'

        ld a,(ix+58)                    ; stato del registro $1FFD
        and $1f
        ld bc,PLUS3REG
        out (c),a

        ld bc,S128KREG                  ; ripristina il registro $7FFD
        ld a,(ix+45)                    ; ora il $1FFD e $7FFD potrebbero essere lockati mediante D5 SET
        out (c),a

;       ld a,(ix+59)                    ; stato del registro $54DF (FASTPAGE!)
;       ld bc,FASTPAGE                  ; viene ignorato: sul badaloc, eventuale banco ROM non e` scelto qui
;       out (c),a


        ld a,(ix+60)                    ; contenuto del NOSTRO FASTPAGE al momento dello snapshot. Se zero
        and a                           ; significa che lo snap e` stato preso sul Badaloc o da una versione precedente.
        jr nz,some_fastpage             ; valore valido
        ld h,NMI_PATCHED_ROM            ; altrimenti H = banco di default per snapshot di tipo 16/48K
        ld a,(ix+2)                     ; tipo di snapshot
        cp MMC_SNAP16                   ; 1
        jr z,snap_16_48
        cp MMC_SNAP48                   ; 2
        jr z,snap_16_48
        ld h,NMI_PATCHED_128            ; ROM 1 (48K) della coppia di rom del 128K (e` la piu` probabile)
snap_16_48
        ld a,h
some_fastpage
        ld (whichbank),a                ; WHICHBANK verra` usato in fase di uscita dall'handler


        ld a,(ix+46)                    ; registro F: D5, D6 da copiare; D4 --> D1; D3 --> NOT D0
        ld h,a
;       and $60
;       bit 3,h
;       jr nz,bit3highf
;       set 0,a                         ; copia invertito
;bit3highf
;       bit 4,h
;       jr z,bit4lowf
;       set 1,a                         ; copia
;bit4lowf
;       ld bc,SPECTR_MODE
;       out (c),a                       ; ATTENZIONE: D6 SET blocca ogni altro accesso WR ai registri non standard!

romsnaprestore_entrypoint
        di                              ; not really necessary, but who knows...
        im 1                            ; default IM1
        bit 2,h                         ; interrupt enable real status
        jr z,no_eneimmc
        ei                              ; this will execute our fake handler (EI, RETI)
        halt                            ; sync with the hardware maskable interrupt generator
        halt
no_eneimmc                              ; now we have 20 (or 10) ms before next interrupt will occur
        ld a,h                          ; for exiting the handler and restore the BASIC ROM.
        and 3                           ; if the host's int handler need the basic rom, it would crash
        and a   ; ------------------- could be useless
        jr z,imzerommc
        cp 1
        jr z,completedeimmc
        im 2
        jr completedeimmc
imzerommc
        im 0
completedeimmc

        exx
        pop ix
        pop hl
        pop de
        ex af,af'
        pop af
        ex af,af'
        pop iy
        pop bc
        exx
        pop de
        pop bc
        pop hl
;       pop af

; Punto molto importante: bisogna decidere se attivare la FLASHROM BANK 1 + NMI su START BIT.
; Questo e` utile per poter successivamente entrare di nuovo.
;
; Al momento, l'NMI viene comunque abilitato; si puo` in seguito basarsi sul contenuto di WHICHBANK, ora
; reale, che indica qual e` il banco in cui saltare (il valore e` stato recuperato dai dati dello snapshot
; ed indica il contenuto di FASTPAGE al momento dello snap stesso).


        ld a,CS_DISABLE                 ; ENABLE FLASHROM BANK 1 AND NMI ON RS-232 STARTBIT
        or NMI_ENABLE
        out (OUT_REG),a

        ld a,(whichbank)                ; banco in cui saltare
                                        ; NOTA INTERESSANTE: se si tratta di un banco di tipo "ROM 48K", non c'e'
                                        ; bisogno che sia stato patchato per funzionare: all'indirizzo in cui
                                        ; vi saltiamo ($71) c'e' quello che serve per default: POP AF, RETN!

                                        ; (diversa e` la storia pre PRENDERE lo snapshot, caso in cui senza la patch
                                        ; un NMI provoca un reset della macchina).
        jp $6F

end_snaprestore


; Subroutine che calcola il blocco da 512 bytes che contiene l'entry (file) nella FAT.
; Richiede: DE = n. del file (snapshot)
; Restituisce: HL, DE = indirizzo 32 bit settore nells SD/MMC
; C = offset 0-7 che identifica l'header desiderato (ciascuno e` 64 bytes) all'interno
; di questo blocco da 512 bytes

calc_hldec
        ld a,e
        and 7
        ld c,a                          ; salva i 3 bit LSB in C
        ld a,e
        and $F8                         ; e li elimina da DE
        ld e,a

        ld a,(card_offset)
        ld h,a
        ld l,0
        ld b,6                          ; moltiplica *64 (6 shift) = dimensioni di ciascuna entry
mul6
        sla e
        rl d
        rl l                            ; shift a 24 bit: dovendo indirizzare 512KB, coinvolge anche i 3 bit LSB di L
        djnz mul6
        ret


;
;--------------------------------------------------------------------------------------------------
; This subroutines returns NO CARRY if HL points to a snapshot number which is above the maximum
; SD/MMC capacity
;
; Destroys AF, DE
;--------------------------------------------------------------------------------------------------
check_hlspace
        push hl
        ld de,FATSIZE           ; dimensioni della FAT in termini di blocchi da 64K
        srl e                   ; lo trasforma in numero di blocchi da 128K
        ld hl,MMCSIZE           ; dimensioni della MMC in termini di blocchi da 128K
        and a
        sbc hl,de               ; HL = numero complessivo di snapshot memorizzabili
        ex hl,de
        pop hl
        push hl
        and a
        sbc hl,de               ; siamo oltre il massimo?
        pop hl
        ret                     ; rientra con CARRY SET se non siamo oltre il massimo


;********************************************************************************************
; FINE RIPRISTINO SNAPSHOT DA SD/MMC 16K/48K/128K
;
; Gestione scelta di una ROM DI BOOT
;********************************************************************************************


;
;--------------------------------------------------------------------------------------------------
; This subroutine is called by the [E] (edit filename, [W] (edit parameters) and [Q] edit Joystick.
; It loads the SD/MMC BLOCK and sets HL to the desired item, at offset +2 (snapshot size).
;--------------------------------------------------------------------------------------------------
get_hlentry
        ld a,(start_item)
        cp 5
        jr nc,it_ok             ; aborts if no snapshot entry is selected
        ret                     ; returns with NONZERO
it_ok
        call get_entry          ; reads the sd/mmc block to SHADOW_RAM
        ld a,(start_item)
        sub 5                   ; 0-based entry pointer
        ld d,a
        ld e,0
        srl d                   ; each item takes 64 bytes
        rr e
        srl d
        rr e
        ld hl,SHADOW_RAM+2      ; base address (+2 = snapshot type)
        add hl,de
        ld a,(hl)
        cp MMC_SNAP16
        ret z                   ; only snapshots with code 1, 2 or 8 are valid: return ZERO
        cp MMC_SNAP48
        ret z                   ; 2 = ok = return ZERO
        cp MMC_RAMBACKUP        ; 4 = ok
        ret z
        cp MMC_FLASHBACKUP      ; 5 = ok
        ret z
        cp MMC_SNAP128
        ret                     ; return NONZERO if <> 8

font_block
        end

