; Questo loader serve per testare una ROM di avvio per la ZXMMC+
;
; Il file binario della rom da testare, che viene messo in coda al presente programma, verra` copiato
; sul banco RAM n. 31 (l'ultimo), dopodiche` lo si abilita in lettura e si esegue un JP 0.
;
;
; METTE A DISPOSIZIONE UNA SECONDA ENTRY POINT SU 30100 che PROGRAMMA LA FLASHROM (banco 0).
;
;


FASTPAGE	equ	$7F


        org 30000

	di		; eseguendo dal BASIC, e` fondamentale disabilitare gli INT

	ld a,$DF	; ultimo banco di ram abilitato in RD/WR su 0 - 3FFF
	out (FASTPAGE),a

	ld hl,PROGBLOCK
	ld de,0
	ld bc,16384	; massima dimensione possibile
	ldir

	jp 0		; salta ad inizio "ROM"



rom_64block	equ	$	; usato per mantenere standard le routines di gestione FLASHROM
		org $+1
flags		equ	$
		org $+1
ram_bank	equ	$
		org $+1
rom_bank	equ	$
		org $+1



; SECONDA ENTRY POINT per programmazione in FLASH ROM sul banco 0.
; Termina saltando nella ROM appena programmata.


SCRATCH	equ	49152	; usa gli ultimi 16K di ram sinclair
RAMBANK	equ	27	; usa i blocchi 27,28,29,30 (il 31 e` usato dalla BOOTROM)

OUT_PORT	equ	$1F

FLASHWR_CODE	equ	$A0	; writing this code to the OUT_PORT register will enable ROM CS on Z80 WR cycles
OUT_IDLE	equ	$F7	; acceptable idle state for OUT_PORT register


	org 30100

	di
	ld sp,30000		; in questo modo non c'e' bisogno del CLEAR 29999

	xor a
	ld (rom_64block),a	; BANCO 0
	ld a,6
	out ($FE),a		; giallo
	call copy_2ram		; copia 64K del banco selezionato in RAM (usa gli ultimi 4 banchi)

	ld a,5
	out($FE),a		; CYAN = copia 16K
	call upload_romcode	; copia i 16K del programma in coda (la nuova bootrom) nel primo banco RAM

	ld a,4			; VERDE = confronto preliminare
	out($FE),a
	call compare		; e` gia` ok?

	ld a,(flags)
	cp 0
	jr z,compare_ok		; si, salta
	cp 2
	jr nz,no_erase		; salta se non e` necessario un ciclo di ERASE
	ld a,3			; MAGENTA = erase
	out ($FE),a
	call erase

no_erase
	ld a,1			; BLU = programmazione
	out ($FE),a
	call program		; programmazione

compare_ok
	ld a,$60		; banco 0 ROM
	out (FASTPAGE),a
	jp 0

	

;
;---------------------------------------------------------------------------------------------
upload_romcode
	ld a,RAMBANK
	set 7,a			; WR enabled
	out (FASTPAGE),a

	ld hl,PROGBLOCK
	ld de,0
	ld bc,16384
	ldir
	ret


;
;---------------------------------------------------------------------------------------------
; This subroutine will copy a given ROM block from ROM to RAM (64Kbytes).
; Returns: nothing
;---------------------------------------------------------------------------------------------
copy_2ram
	ld a,RAMBANK
	set 7,a			; WR enabled

	ex af,af'
	ld a,(rom_64block)	; desired ROM block
	and 7
	sla a			; forms a 5 bit 16K page pointer
	sla a
	or $60			; ROM paged-in
	ex af,af'		; A' = rom bank

	ld b,4			; number of 16K blocks

next_btoram
	ex af,af'
	out (FASTPAGE),a	; rom is selected
	inc a
	push bc

	ld hl,0			; copy to sinclair ram
	ld de,SCRATCH
	ld bc,16384
	ldir

	ex af,af'
	out (FASTPAGE),a	; RAM is selected in WR mode
	inc a

	ld hl,SCRATCH		; copy from sinclair ram to zxmmc+ ram
	ld de,0
	ld bc,16384
	ldir

	pop bc
	djnz next_btoram

	ret


;
;---------------------------------------------------------------------------------------------
; This subroutine will compare a given ROM block with RAM content (64Kbytes).
;
; Returns:
; 0 = Identical; 1 = Mismatch which can be programmed; 2 = mismatch which needs BLOCK ERASE.
;---------------------------------------------------------------------------------------------
compare
	xor a
	ld (flags),a		; return code = OK

	ld a,RAMBANK
	set 6,a			; RD enabled

	ex af,af'
	ld a,(rom_64block)	; desired ROM block
	and 7
	sla a			; forms a 5 bit 16K page pointer
	sla a
	or $60			; ROM paged-in
	ex af,af'		; A' = rom bank

	ld b,4			; number of 16K blocks

next_bcompare
	ex af,af'
	out (FASTPAGE),a	; rom is selected
	inc a
	push bc

	ld hl,0
	ld de,SCRATCH
	ld bc,16384
	ldir

	ex af,af'
	out (FASTPAGE),a	; RAM is selected in RD mode
	inc a

	push af			; COMPARE LOOP
	ld hl,0			; PAGED-IN RAM pointer
	ld de,SCRATCH		; scratchram (ROM CONTENT) pointer
	ld bc,16384
l_compare
	ld a,(de)		; this is ROM content byte, which was copied to sinclair RAM
	cp (hl)			; compare with RAM buffer (paged-in)
	jr z,byte_ok
	and (hl)		; are there '0' bits which should be turned into '1'?
	cp (hl)
	jr z,programmable	; no, this byte can be programmed without erasing the 64K block
	ld a,2
	ld (flags),a		; 2 = ERASE is needed
	pop af			; can't be worse: exit immediately
	pop bc
	jr exit_compare
programmable
	ld a,1
	ld (flags),a		; 1 = ROM content mismatch, but still programmable
byte_ok
	inc hl
	inc de
	dec bc
	ld a,b
	or c
	jr nz,l_compare
	pop af

	pop bc
	djnz next_bcompare

exit_compare
	ret

;
;---------------------------------------------------------------------------------------------
; This subroutine will program a given ROM block (64Kbytes).
;---------------------------------------------------------------------------------------------
program
	exx
	ld hl,$555		; quick access to program codes
	ld bc,$2AA
	ld e,$AA
	exx

	ld a,(rom_64block)	; desired ROM block
	and 7
	sla a			; forms a 5 bit 16K page pointer
	sla a
	or $60			; ROM page in
	ld (rom_bank),a

	ld a,RAMBANK		; first bank of ram to be programmed
	set 6,a			; RAM RD enabled
	ld (ram_bank),a

	ld a,FLASHWR_CODE	; this code will enable the ROM CS on Z80 write cycles
	out (OUT_PORT),a

next_bank
	ld a,(ram_bank)
	out (FASTPAGE),a
	inc a
	ld (ram_bank),a

	ld hl,0			; reads the current 16K ram buffer into scratchpad ram
	ld de,SCRATCH
	ld bc,16384
	ldir

	ld a,(rom_bank)
	out (FASTPAGE),a	; select ROM for read too (needed to check programmed byte)
	inc a
	ld (rom_bank),a

	ld hl,SCRATCH		; buffer to be programmed
	ld de,0			; A0:A13 address bit to ROM (16K) - remainder from PAGED-IN ROM BANK
	ld bc,16384

program_16k
	ld a,(de)		; current ROM content
	cp (hl)			; desired content
	jr z,no_program		; byte OK already

	exx
	ld a,$55
	ld (hl),e		; $555 <- $AA
	ld (bc),a		; $2AA <- $55
	ld (hl),$A0		; $555 <- $A0
	exx

	ld a,(hl)		; byte to program
	ld (de),a		; writes to ROM

	ex hl,de
	push bc			; wait for write completion
	ld bc,0			; timeout counter
l_waitprogram
	cp (hl)
	jr z,prog_do
	dec c
	jr nz,l_waitprogram
	dec b
	jr nz,l_waitprogram
	pop bc

	ld (hl),$F0		; RESET command to flash rom (clears write error and switch to read mode)

	ld bc,0			; datasheet says the chip needs at least 10us before any read attempt.
l_resetwait			; we wait quite more.
	dec bc
	ld a,b
	or c
	jr nz,l_resetwait

	ld bc,1			; program error
	jr program_exit

prog_do
	pop bc
	ex hl,de
no_program
	inc hl
	inc de
	dec bc
	ld a,b
	or c
	jr nz,program_16k

	ld a,(rom_bank)
	and 3			; "00" on LSB bits means 4 banks programmed
	jr nz,next_bank

	ld bc,0			; OK code
program_exit
	ld a,OUT_IDLE
	out (OUT_PORT),a	; disables ROM WR cycles

	ret


;
;---------------------------------------------------------------------------------------------
; This subroutine will erase a given ROM block (64Kbytes).
;---------------------------------------------------------------------------------------------
erase
	ld a,(rom_64block)	; desired ROM block
	and 7
	sla a			; forms a 5 bit 16K page pointer
	sla a
	or $60			; ROM paged-in for read
	out (FASTPAGE),a

	ld a,FLASHWR_CODE	; this code will enable the ROM CS on Z80 write cycles
	out (OUT_PORT),a

	ld hl,$555
	ld bc,$2AA
	ld e,$AA
	ld a,$55

	ld (hl),e		; $555 <- $AA
	ld (bc),a		; $2AA <- $55
	ld (hl),$80		; $555 <- $80
	ld (hl),e		; $555 <- $AA
	ld (bc),a		; $2AA <- $55

	ld hl,0
	ld (hl),$30		; block erase command (address don't really care: only 3 MSB 64K-BLOCK bits)


	ld e,8			; about 6 seconds timeout
	ld bc,0
l_waiterase
	ld a,(hl)
	cp $ff
	jr z,erased_ok
	dec c
	jr nz,l_waiterase
	dec b
	jr nz,l_waiterase
	dec e
	jr nz,l_waiterase

	ld (hl),$F0		; RESET command to flash rom (clears write error and switch to read mode)

	ld bc,1			; timeout error
	jr erase_exit

erased_ok
	ld bc,0			; OK code

erase_exit
	ld a,OUT_IDLE
	out (OUT_PORT),a	; disables ROM WR cycles

	ret


PROGBLOCK

	end
