;---------------------------------------------------------------------------------------------
;
; based on Version 4.96b 17/03/2007 of ZX-Badaloc's BOOTROM.
;
; These routines allow read/compare/programming/erasing the ST/AMD M29F040B 4Mbit FLASHROM
; installed on the ZXMMC+ interface.
;
; The chip may be programmed on a byte basis, but can only be erased on 64KBytes blocks.
; For this reason, all routines will work on a 64Kbyte basis.
;
; When in ERASED state, a ROM byte will read as $FF.
; Erase is needed only when the new content should turn a programmed bit '0' into
; a nonprogrammed '1'. In the opposite case, the program routine can 'overwrite' previous
; ROM content without the need to first erase the entire block.
;
; MODE OF OPERATION:
;
; The program will use four contiguous 16K RAM banks from the zxmmc+ ram chip, beginning at
; bank n. RAMBANK (equ) as a 64Kbytes compare/program buffer.
; The program will also use 16K of the zx-spectrum memory at address SCRATCH (equ) as a
; temporary scratchpad copy/paste buffer, as we don't want to switch between RAM and ROM
; PAGING on the zxmmc+ interface for every byte to be read/programmed/compared.
;
;
; THE 16K BANK to be read/compared/programmed/erased should be POKEd at address 42000.
; Since the device can only be erased/programmed on 64K basis (four banks), some functions
; will ignore some of the bank number bits. For example, the COPY_2RAM will read ALL 4 BANKS
; to the 64K ram buffer: the two LSB bank number bits are ignored (this allows full backup
; of the 64K bank prior to erasing it). The ERASE command works in the same way. When erasing
; BANK 3, banks 0,1,2,3 will be erased. When erasing bank 4, banks 4,5,6,7 will be erased.
; The ANYROM functiom, which copies 16K from Sincialr 49152 address (upper 16K) to one of
; the four RAM banks, will ignore the upper 3 bits (only offset 0-3 is used).
;
;
;
; PROVIDED SUBROUTINES:
;
; 'COPY_2RAM', address 40000:
;
; This routine will copy 64KBytes from ROM to RAM, including the specified 16K bank.
; 4 RAM banks will be overwritten, beginning
; from the one specified in the EQU 'RAMBANK'. This routine is useful to read the entire 64K
; block before erasing. Location 42000 should contain a 0-31 16K bank number comprised in
; the 64K that will be read.
; Returns: nothing
;
;
;
; 'COMPARE', address 40003:
;
; This routine will compare 64K bytes of data between ROM and RAM. As for the COPY_2RAM
; function, the caller should specify the 16K bank number 0-31 at location 42000.
; Returns: 0 = Equal; 1 = mismatch that can be programmed; 2 = mismatch that needs BLOCK ERASE.
;
;
;
; 'PROGRAM', address 40006:
;
; This routine will actually PROGRAM 64K bytes of data from the RAM to FLASHROM. As always,
; the bank to be programmed (0-31) should be specified at location 42000. Four banks will be
; programmed, including the desired one.
; Returns: 0 = OK; 1 = program error.
; Attempting to program when the COMPARE function returns code 2 without performing the
; necessary block erase function, will result in the first not-programmable byte being a logic
; AND between previous and desired contents, and remaining bytes left unchanged.
;
;
;
; 'ERASE', address 40009:
;
; This function will erase a 64K bytes block which contains the specified (0-31) 16K block
; specified at location 42000.
; Returns: 0 = OK; 1 = erase error.
;
;
;
; 'TEST', address 40012:
;
; Provided just for testing purposes, this routine will read a 16K rom bank into the zx-spectrum
; scratchpad memory (at address EQU SCRATCH). Bank number (0-31) should be specified at
; location 42000.
;
;
;
; 'SINCLAIR', address 40015:
;
; This routine will copy the 16K system ROM content (sinclair basic) to one of the four RAM
; banks that can be compared/programmed into FLASHROM. Destination ram bank (0-3) should be
; specified at location 42000. Since MSB bits are ignored, the location may contain the 16K
; 0-31 final destination ROM BANK. For testing purposes, the rom content is patched so that the
; power-on logo will show 'ROM' in place of 'Ltd' and the NMI handler will increment the first
; byte of video memory.
;
; This rom will have the NMI handler patched as obtained by MKSINCLA.ASM, so can be used
; for ZX-Com RS-232 communication.
;
; It also insert a "GETROM' routine at address 14500 (spare in original rom).
; This routine will copy a selected ROM bank to a fixed location 32768 in Spectrum RAM.
; The desired bank number should be POKEd at address 23728 (0-31).

; HOWTO: It's likely that ROM BANK 0 holds the BOOTROM firmware and BANK 1 the NMI-patched ROM.
; To place a copy of this patched sinclair rom into BANK 2:
;
; - POKE 42000,2
; - call COPY_2RAM (USR 40000)		this will copy banks 0,1,2,3
; - call ERASE	   (USR 40009)		this will erase banks 0,1,2,3
; - call SINCLAIR  (USR 40015)		this will overwrite RAM bank n.2
; - call PROGRAM   (USR 40006)		this will program banks 0,1,2,3 with RAM data
; - call COMPARE   (USR 40003)		this will compare banks 0,1,2,3
;
;
;
; 'ANYROM', address 40018:
;
; As 'SINCLAIR', but reads data from the SCRATCH buffer (usually at 49152) so the user could
; load a ROM from tape then copy it to one ZXMMC+ RAM BANK for FLASH programming.
; HOWTO: To copy a ROM into ROM BANK 3:
;
; - POKE 42000,3
; - call COPY_2RAM (USR 40000)		this will copy banks 0,1,2,3
; - call ERASE	   (USR 40009)		this will erase banks 0,1,2,3
; - Load the ROM binary file from TAPE at address SCRATCH (49152)
; - call ANYROM  (USR 40018)		this will copy the 49152 16K buffer to RAM BANK 3
; - call PROGRAM   (USR 40006)		this will program banks 0,1,2,3 with RAM data
; - call COMPARE   (USR 40003)		this will compare banks 0,1,2,3
;
;
; BEFORE using the ERASE command, check that the system is NOT running from the ROM being erased :-)


OUT_PORT equ	$1F		; SD chip select and FLASH WR enable register

FASTPAGE equ	$7F		; RAM/ROM fast paging register
;
; D7 set = RAM WR enable (standalone)
; D6 set = PAGE IN active
; D5 '0' = ram is paged; '1' = rom is paged
; D4:D0 = bank number


BORDER	equ $FE			; PORT setup border sui bit D2 - D0 (gli altri riguardano SPKR/TAPE)


SCRATCH		equ	$C000	; 16K buffer used as scratchpad in zx-spectrum standard memory
RAMBANK		equ	27	; first of 4 16K banks in the zxmmc+ memory used as 64K backup for block erase.
				; leaves the LAST 16K bank untouched, as it may contain the BOOTROM

FLASHWR_CODE	equ	$A0	; writing this code to the OUT_PORT register will enable ROM CS on Z80 WR cycles
OUT_IDLE	equ	$F7	; acceptable idle state for OUT_PORT register


        org 42000

rom_64block	equ	$	; number of 64K bank to be erased/read/programmed (0-7 = 512K)
		org $+1
fastp_save	equ	$
		org $+1
flags		equ	$
		org $+1
ram_bank	equ	$
		org $+1
rom_bank	equ	$
		org $+1



	org 40000
	jp copy_2ram		; 40000 64K bytes ROM --> RAM
	jp compare		; 40003 return code: 0 = OK, 1 = programmable; 2 = need block erase
	jp program		; 40006 return code: 0 = OK, 1 = program timeout error
	jp erase		; 40009 return code: 0 = OK, 1 = erase timeout error
	jp test			; 40012 get specified 16K rom bank into SCRATCHPAD spectrum RAM
	jp sinclair		; 40015 copy 16K from sinclair rom to specified 0-3 RAM program buffer
	jp anyrom		; 40018 as 'sinclair', but reads data from the SCRATCHPAD spectrum RAM



;
;---------------------------------------------------------------------------------------------
; This subroutine will copy a given ROM block from ROM to RAM (64Kbytes).
; Returns: nothing
;---------------------------------------------------------------------------------------------
copy_2ram
	di
	in a,(FASTPAGE)
	ld (fastp_save),a	; FASTPAGE content is saved

	ld a,RAMBANK
	set 7,a			; WR enabled

	ex af,af'
	ld a,(rom_64block)	; desired ROM bank (0-31)
	and $1C			; 2 LSB bit should be ignored
	or $60			; ROM paged-in
	ex af,af'		; A' = rom bank

	ld b,4			; number of 16K blocks

next_btoram
	ex af,af'
	out (FASTPAGE),a	; rom is selected
	inc a
	push bc

	ld hl,0			; copy to sinclair ram
	ld de,SCRATCH
	ld bc,16384
	ldir

	ex af,af'
	out (FASTPAGE),a	; RAM is selected in WR mode
	inc a

	ld hl,SCRATCH		; copy from sinclair ram to zxmmc+ ram
	ld de,0
	ld bc,16384
	ldir

	pop bc
	djnz next_btoram

	ld a,(fastp_save)
	out (FASTPAGE),a	; FASTPAGE content is restored
	ei
	ret


;
;---------------------------------------------------------------------------------------------
; This subroutine will compare a given ROM block with RAM content (64Kbytes).
;
; Returns:
; 0 = Identical; 1 = Mismatch which can be programmed; 2 = mismatch which needs BLOCK ERASE.
;---------------------------------------------------------------------------------------------
compare
	di
	in a,(FASTPAGE)
	ld (fastp_save),a	; FASTPAGE content is saved
	xor a
	ld (flags),a		; return code = OK

	ld a,RAMBANK
	set 6,a			; RD enabled

	ex af,af'
	ld a,(rom_64block)	; desired ROM bank (0-31)
	and $1C			; 2 LSB bit should be ignored
	or $60			; ROM paged-in
	ex af,af'		; A' = rom bank

	ld b,4			; number of 16K blocks

next_bcompare
	ex af,af'
	out (FASTPAGE),a	; rom is selected
	inc a
	push bc

	ld hl,0
	ld de,SCRATCH
	ld bc,16384
	ldir

	ex af,af'
	out (FASTPAGE),a	; RAM is selected in RD mode
	inc a

	push af			; COMPARE LOOP
	ld hl,0			; PAGED-IN RAM pointer
	ld de,SCRATCH		; scratchram (ROM CONTENT) pointer
	ld bc,16384
l_compare
	ld a,(de)		; this is ROM content byte, which was copied to sinclair RAM
	cp (hl)			; compare with RAM buffer (paged-in)
	jr z,byte_ok
	and (hl)		; are there '0' bits which should be turned into '1'?
	cp (hl)
	jr z,programmable	; no, this byte can be programmed without erasing the 64K block
	ld a,2
	ld (flags),a		; 2 = ERASE is needed
	pop af			; can't be worse: exit immediately
	pop bc
	jr exit_compare
programmable
	ld a,1
	ld (flags),a		; 1 = ROM content mismatch, but still programmable
byte_ok
	inc hl
	inc de
	dec bc
	ld a,b
	or c
	jr nz,l_compare
	pop af

	pop bc
	djnz next_bcompare

exit_compare
	ld a,(fastp_save)
	out (FASTPAGE),a	; FASTPAGE content is restored

	ld a,(flags)		; return code
	ld c,a
	ld b,0
	ei
	ret

;
;---------------------------------------------------------------------------------------------
; This subroutine will program a given ROM block (64Kbytes).
;---------------------------------------------------------------------------------------------
program
	di
	exx
	push bc
	push de
	push hl
	ld hl,$555		; quick access to program codes
	ld bc,$2AA
	ld e,$AA
	exx

	in a,(FASTPAGE)
	ld (fastp_save),a	; FASTPAGE content is saved

	ld a,(rom_64block)	; desired ROM bank (0-31)
	and $1C			; 2 LSB bit should be ignored
	or $60			; ROM page in
	ld (rom_bank),a

	ld a,RAMBANK		; first bank of ram to be programmed
	set 6,a			; RAM RD enabled
	ld (ram_bank),a

	ld a,FLASHWR_CODE	; this code will enable the ROM CS on Z80 write cycles
	out (OUT_PORT),a

next_bank
	ld a,(ram_bank)
	out (FASTPAGE),a
	inc a
	ld (ram_bank),a

	ld hl,0			; reads the current 16K ram buffer into scratchpad ram
	ld de,SCRATCH
	ld bc,16384
	ldir

	ld a,(rom_bank)
	out (FASTPAGE),a	; select ROM for read too (needed to check programmed byte)
	inc a
	ld (rom_bank),a

	ld hl,SCRATCH		; buffer to be programmed
	ld de,0			; A0:A13 address bit to ROM (16K) - remainder from PAGED-IN ROM BANK
	ld bc,16384

program_16k
	ld a,(de)		; current ROM content
	cp (hl)			; desired content
	jr z,no_program		; byte OK already

	exx
	ld a,$55
	ld (hl),e		; $555 <- $AA
	ld (bc),a		; $2AA <- $55
	ld (hl),$A0		; $555 <- $A0
	exx

	ld a,(hl)		; byte to program
	ld (de),a		; writes to ROM

	ex hl,de
	push bc			; wait for write completion
	ld bc,0			; timeout counter
l_waitprogram
	cp (hl)
	jr z,prog_do
	dec c
	jr nz,l_waitprogram
	dec b
	jr nz,l_waitprogram
	pop bc

	ld (hl),$F0		; RESET command to flash rom (clears write error and switch to read mode)

	ld bc,0			; datasheet says the chip needs at least 10us before any read attempt.
l_resetwait			; we wait quite more.
	dec bc
	ld a,b
	or c
	jr nz,l_resetwait

	ld bc,1			; program error
	jr program_exit

prog_do
	pop bc
	ex hl,de
no_program
	inc hl
	inc de
	dec bc
	ld a,b
	or c
	jr nz,program_16k

	ld a,(rom_bank)
	and 3			; "00" on LSB bits means 4 banks programmed
	jr nz,next_bank

	ld bc,0			; OK code
program_exit
	ld a,OUT_IDLE
	out (OUT_PORT),a	; disables ROM WR cycles

	ld a,(fastp_save)
	out (FASTPAGE),a	; FASTPAGE content is restored

	exx
	pop hl
	pop de
	pop bc
	exx
	ei
	ret


;
;---------------------------------------------------------------------------------------------
; This subroutine will erase a given ROM block (64Kbytes).
;---------------------------------------------------------------------------------------------
erase
	di
	in a,(FASTPAGE)
	ld (fastp_save),a	; FASTPAGE content is saved

	ld a,(rom_64block)	; desired ROM bank (0-31)
	and $1C			; 2 LSB bit should be ignored
	or $60			; ROM paged-in for read
	out (FASTPAGE),a

	ld a,FLASHWR_CODE	; this code will enable the ROM CS on Z80 write cycles
	out (OUT_PORT),a

	ld hl,$555
	ld bc,$2AA
	ld e,$AA
	ld a,$55

	ld (hl),e		; $555 <- $AA
	ld (bc),a		; $2AA <- $55
	ld (hl),$80		; $555 <- $80
	ld (hl),e		; $555 <- $AA
	ld (bc),a		; $2AA <- $55

	ld hl,0
	ld (hl),$30		; block erase command (address don't really care: only 3 MSB 64K-BLOCK bits)


	ld e,8			; about 6 seconds timeout
	ld bc,0
l_waiterase
	ld a,(hl)
	cp $ff
	jr z,erased_ok
	dec c
	jr nz,l_waiterase
	dec b
	jr nz,l_waiterase
	dec e
	jr nz,l_waiterase

	ld (hl),$F0		; RESET command to flash rom (clears write error and switch to read mode)

	ld bc,1			; timeout error
	jr erase_exit

erased_ok
	ld bc,0			; OK code

erase_exit
	ld a,OUT_IDLE
	out (OUT_PORT),a	; disables ROM WR cycles

	ld a,(fastp_save)
	out (FASTPAGE),a	; FASTPAGE content is restored
	ei
	ret


;
;---------------------------------------------------------------------------------------------
; This test subroutine will copy 16K from the given ROM BANK (0-31) to SCRATCHPAD spectrum ram
;---------------------------------------------------------------------------------------------
test
	di
	in a,(FASTPAGE)
	ld (fastp_save),a	; FASTPAGE content is saved

	ld a,(rom_64block)	; desired ROM BANK (0-31!)
	and $1F
	or $60			; ROM is paged-in
	out (FASTPAGE),a

	ld hl,0
	ld de,SCRATCH
	ld bc,16384
	ldir

	ld a,(fastp_save)
	out (FASTPAGE),a	; FASTPAGE content is restored
	ei
	ret


;
;---------------------------------------------------------------------------------------------
; This subroutine will copy the sinclair ROM to the specified (0-3) programming ram buffer,
; changing the power-on logo and NMI handler.
;---------------------------------------------------------------------------------------------
sinclair
	di
	in a,(FASTPAGE)
	ld (fastp_save),a		; FASTPAGE content is saved

	ld a,(rom_64block)
	and 3
	add RAMBANK+128			; RAM bank offset + D7 set (WR enable)
	out (FASTPAGE),a

	ld hl,0
	ld de,0
	ld bc,16384
	ldir				; sinclair rom to ram copy

	ld hl,string			; new power-on logo
	ld de,$1552			; overwrite the 'Ltd' leading text
	ld bc,3
	ldir

	ld hl,nmi_block			; new NMI handler patch
	ld de,$66			; destination address
	ld bc,end_nmi-nmi_block
	ldir				; NMI handler is now patched


	ld hl,getrom_block		; new GETROM routine copied at address 14500
	ld de,14500			; destination address
	ld bc,end_getrom-getrom_block
	ldir

	ld a,(fastp_save)
	out (FASTPAGE),a		; FASTPAGE content is restored
	ei
	ret

nmi_block
	push af				; 11T patch code for NMI handler
	in a,(FASTPAGE)			; 11T
	push af				; 11T fastpage content is saved as well
	ld a,$DF			; 7T RAM BANK 31 (bootrom) in RD/WR mode
	out (FASTPAGE),a		; 11T switch to BOOTROM BANK

					; 51T states to switch to BOOTROM BANK (29 in previous V1.00 version: +22).


	nop				; this instruction is never executed: --> BOOTROM NMI handler (at $6E!), which
	nop				; holds a JR (only first byte!) (operand = $D3 (the OUT that follows in the bootrom firmware))

	pop af				; In the bootrom, here we have an OUT (FASTPAGE),A. It is the BOOTROM return point: $6F.
					; 'A' should hold PATCHED ROM BANK NUMBER
					; THIS POP AF IS NEVER EXECUTED! (but balances the second PUSH AF in case the rom was ran
					; without the zxmmc+ hardware :-)

					; after the OUT (FASTPAGE) is fetched from $6F address in the BOOTROM bank, here we come:
	pop af				; this is same as original content, at address $71. THIS IS EXECUTED HERE.
	retn				; NOTE: this allows a snapshot restore even with non-patched sinclair ROM.
end_nmi


;nmi_block
;	push af				; patch code for NMI handler
;	push hl
;	ld hl,16384
;	inc (hl)
;	nop
;	nop
;	nop
;	nop
;	pop hl				; this is same as original content, at address $70
;	pop af
;	retn
;end_nmi


getrom_block					; WE ASSUME THIS RUNS AT ADDRESS 14500! If not, change the
	di					; ld HL formula.
	ld hl,(reloc_start-getrom_block)+14500	; routine should be relocated somewhere in RAM
	ld de,16384				; as we can't change the FASTPAGE register
	ld bc,end_getrom-reloc_start		; without loosing control
	ldir
	jp 16384
reloc_start
	in a,(FASTPAGE)
	push af
	ld a,(23728)			; desired rom bank
	and 31
	or $60				; ROM PAGE-IN
	out (FASTPAGE),a
	ld hl,0
	ld de,30000			; rom is copied to address 30000
	ld bc,16384
	ldir
	pop af
	out (FASTPAGE),a		; fastpage is restored
	ei
	ret
end_getrom


string
        defb 'R','O','M'+128



;
;---------------------------------------------------------------------------------------------
; This subroutine will copy a previously loaded ROM from SCRATCHPAD spectrum ram to the
; specified (0-3) ram buffer
;---------------------------------------------------------------------------------------------
anyrom
	di
	in a,(FASTPAGE)
	ld (fastp_save),a		; FASTPAGE content is saved

	ld a,(rom_64block)
	and 3
	add RAMBANK+128			; RAM bank offset + D7 set (WR enable)
	out (FASTPAGE),a

	ld hl,SCRATCH			; reads data from SCRATCHPAD area
	ld de,0
	ld bc,16384
	ldir

	ld a,(fastp_save)
	out (FASTPAGE),a		; FASTPAGE content is restored
	ei
	ret
        end 

