/***************************************************************************

  Mini-Pac Copyright (C) 1999 by Michael Balfour.  All Rights Reserved.


  Pac-Man Copyright (C) 1980 by Namco, Ltd.  All Rights Reserved.


  Multi-Z80 32 Bit emulator 
  Copyright (C) 1996-1999 Neil Bradley, All rights reserved
  Author      : Neil Bradley (neil@synthcom.com)
  Distribution: ftp://ftp.synthcom.com/pub/emulators/cpu/makez80.zip (latest)

 ***************************************************************************/

// INCLUDES ///////////////////////////////////////////////////////////////////

#include "minipac.h"

static CODE_ROM namcoRomList[] = 
{
	"namcopac.6e",0x0000,0x1000,
	"namcopac.6f",0x1000,0x1000,
	"namcopac.6h",0x2000,0x1000,
	"namcopac.6j",0x3000,0x1000,
	NULL,-1,-1
};

#define TILE_GRAPHICS_ROM_NAME		"namcopac.5e"
#define SPRITE_GRAPHICS_ROM_NAME	"namcopac.5f"

#define COLOR_PALETTE_ROM_NAME		"82s123.7f"
#define COLOR_LOOKUP_TABLE_ROM_NAME	"82s126.4a"

#define SOUND_ROM_NAME				"82s126.1m"

/*-----------------------------------------------------------------------------
   Name: Interpreter()
   Desc: Construct our main object
  -----------------------------------------------------------------------------*/

Interpreter::Interpreter()
{
	emuMemory = NULL;
	dirtyVideoMemory = NULL;

	tilePtr = NULL;
	spritePtr = NULL;
	spriteHFlipPtr = NULL;
	spriteVFlipPtr = NULL;
	spriteHVFlipPtr = NULL;

	origPalette = NULL;
	origColorLookup = NULL;

	// Init Emulation Memory
	InitEmuMemory(namcoRomList);

	// Init Sound
	InitSound();

	// Init Graphics
	InitGraphics();

	// Init Colors
	InitColors();

	// Init Sprite List
	InitSprites();

	// Init inputs and DIP switches
	InitInputPorts();
}

/*-----------------------------------------------------------------------------
   Name: ~Interpreter()
   Desc: Destroy our main object
  -----------------------------------------------------------------------------*/

Interpreter::~Interpreter()
{
	if (emuMemory != NULL)			free (emuMemory);
	if (dirtyVideoMemory != NULL)	free (dirtyVideoMemory);

	if (tilePtr != NULL)			free (tilePtr);
	if (spritePtr != NULL)			free (spritePtr);
	if (spriteHFlipPtr != NULL)		free (spriteHFlipPtr);
	if (spriteVFlipPtr != NULL)		free (spriteVFlipPtr);
	if (spriteHVFlipPtr != NULL)	free (spriteHVFlipPtr);

	if (origPalette != NULL)		free (origPalette);
	if (origColorLookup != NULL)	free (origColorLookup);
}

/*-----------------------------------------------------------------------------
   Name: CreateEmuMemory()
   Desc: Load code ROMs and create a memory space for the emulation
  -----------------------------------------------------------------------------*/
bool Interpreter::InitEmuMemory(CODE_ROM *romList)
{
	FILE *f;
	CODE_ROM *curRom;

	curRom = romList;

	/* Create the emulated game area */
	emuMemory = (UCHAR *)calloc(0x10000,1);

	/* Load the game ROMs */
	while (curRom->name != NULL)
	{
		f = fopen(curRom->name,"rb");
		if (f == NULL)
		{
			throw "Unable to locate game ROM";
		}

		if (fread(&(emuMemory[curRom->location]),curRom->size,1,f) != 1)
		{
			throw "Unable to read game ROM";
		}
		fclose(f);
        curRom++;
	}

	dirtyVideoMemory = (UCHAR *)malloc(0x400);
	if (dirtyVideoMemory == NULL)
	{
		throw "Error allocating dirtyMemory block";
	}

	memset(dirtyVideoMemory,1,0x400);

	return true;
}

/*-----------------------------------------------------------------------------
   Name: InitInputPorts()
   Desc: Initialize input ports and DIP switches
  -----------------------------------------------------------------------------*/
void Interpreter::InitInputPorts(void)
{
/*  0x5000
		0x01 = up
		0x02 = left
		0x04 = right
		0x08 = down
		0x10 = rack test
		0x20 = coin 1
		0x40 = coin 2
		0x80 = coin 3
	0x5040
		0x01 = up (cocktail)
		0x02 = left (cocktail)
		0x04 = right (cocktail)
		0x08 = down (cocktail)
		0x10 = service mode
		0x20 = start 1
		0x40 = start 2
		0x80 = cabinet (cocktail, upright)
	0x5080
		0x00 = Free play
		0x01 = 1 coin/1 credit
		0x02 = 1 coin/2 credits
		0x03 = 2 coins/1 credit

		0x00 = 1 life
		0x04 = 2 lives
		0x08 = 3 lives
		0x0C = 5 lives

		0x00 = bonus life @ 10000
		0x10 = bonus life @ 15000
		0x20 = bonus life @ 20000
		0x30 = no bonus life

		0x00 = hard difficulty
		0x40 = normal difficulty

		0x00 = alternate ghost names
		0x80 = normal ghost names
	0x50C0 - unused
*/

	UCHAR data = 0x00;

	/* 1 coin/1 credit - 0x01 */
	data |= 0x01;

	/* 5 lives - 0x0c */
	data |= 0x0C;

	/* Bonus life @ 10000 - 0x00 */
	data |= 0x00;

	/* Difficulty normal - 0x40 */
	data |= 0x40;

	/* Ghost names normal - 0x80 */
	data |= 0x80;

	emuMemory[0x5000] = 0xFF;
	emuMemory[0x5040] = 0xFF;
	emuMemory[0x5080] = data;
	emuMemory[0x50C0] = 0xFF;
}

/*-----------------------------------------------------------------------------
   Name: InitSound()
   Desc: Load the sound ROM
  -----------------------------------------------------------------------------*/
void Interpreter::InitSound(void)
{
	FILE *f;
	int i;

	for (i = 0; i < NUM_SOUND_VOICES; i++ )
	{
		frequency[i] = 0;
		volume[i] = 0;
		wave[i] = 0;
	}

	f = fopen(SOUND_ROM_NAME,"rb");
	if (f == NULL)
	{
		throw "Error opening sound ROM";
	}

	if (fread(waveBuffer,0x100,1,f) != 1)
	{
		throw "Error reading sound ROM";
	}

	fclose(f);

	// Scale from a 0-F range to a 00-FF range.
	for (i = 0; i < 0x100; i++ )
	{
		waveBuffer[i] |= ((waveBuffer[i] & 0x0F) << 4);
	}
}

/*-----------------------------------------------------------------------------
   Name: InitGraphics()
   Desc: Initialize and decode graphics
  -----------------------------------------------------------------------------*/
void Interpreter::InitGraphics(void)
{
	FILE *f;
	UCHAR buffer[0x1000];

	/* Load the tiles */

	f = fopen(TILE_GRAPHICS_ROM_NAME,"rb");
	if (f == NULL)
	{
		throw "Error opening tile graphics ROM";
	}

	if (fread(buffer,0x1000,1,f) != 1)
	{
		throw "Error reading tile graphics ROM";
	}

	fclose(f);

	tilePtr = (UCHAR *)malloc(8*8*256);

	if (tilePtr == NULL)
	{
		throw "Error creating tile graphics memory";
	}

	DecodeTiles(buffer, tilePtr);

	/* Load the sprites */

	f = fopen(SPRITE_GRAPHICS_ROM_NAME,"rb");
	if (f == NULL)
	{
		throw "Error opening sprite graphics ROM";
	}

	if (fread(buffer,0x1000,1,f) != 1)
	{
		throw "Error reading sprite graphics ROM";
	}

	fclose(f);

	spritePtr        = (UCHAR *)malloc(16*16*64);
	spriteHFlipPtr   = (UCHAR *)malloc(16*16*64);
	spriteVFlipPtr   = (UCHAR *)malloc(16*16*64);
	spriteHVFlipPtr  = (UCHAR *)malloc(16*16*64);

	if ((spritePtr == NULL)      || (spriteHFlipPtr == NULL) || 
		(spriteVFlipPtr == NULL) || (spriteHVFlipPtr == NULL))
	{
		throw "Error creating sprite graphics memory";
	}

	DecodeSprites(buffer, spritePtr);
	FlipSpritesHoriz(spritePtr, spriteHFlipPtr);
	FlipSpritesVert(spritePtr, spriteVFlipPtr);
	FlipSpritesHorizVert(spritePtr, spriteHVFlipPtr);

}

/*-----------------------------------------------------------------------------
   Name: InitColors()
   Desc: Initialize and load colors
  -----------------------------------------------------------------------------*/
void Interpreter::InitColors(void)
{
	FILE *f;

    origPalette = (UCHAR *)malloc(0x20);
	origColorLookup = (UCHAR *)malloc(0x100);

	if ((origPalette == NULL) || (origColorLookup == NULL))
	{
		throw "Error creating color memory";
	}

	/* Load the color palette and lookup table */
	f = fopen(COLOR_PALETTE_ROM_NAME,"rb");
	if (f == NULL)
	{
		throw "error opening palette ROM";
	}
	if (fread(origPalette,0x20,1,f) != 1)
	{
		throw "error reading palette ROM";
	}
	fclose(f);

	f = fopen(COLOR_LOOKUP_TABLE_ROM_NAME,"rb");
	if (f == NULL)
	{
		throw "error opening color lookup ROM";
	}
	if (fread(origColorLookup,0x100,1,f) != 1)
	{
		throw "error reading color lookup ROM";
	}
	fclose(f);
}

/*-----------------------------------------------------------------------------
   Name: DecodeTiles()
   Desc: Decode graphics from original buffer
  -----------------------------------------------------------------------------*/
void Interpreter::DecodeTiles(UCHAR *srcBuffer, UCHAR *destBuffer)
{
	UCHAR *s;
	UCHAR *d;

	for (int tile = 0; tile < 256; tile++)
	{
		s = &(srcBuffer[tile * 16]);
		d = &(destBuffer[tile * (8 * 8)]);

		for (int y = 7; y >= 0; y--)
		{
			d[8*0 + y] = ((s[8] & 0x80) >> 6) | ((s[8] & 0x08) >> 3);
			d[8*1 + y] = ((s[8] & 0x40) >> 5) | ((s[8] & 0x04) >> 2);
			d[8*2 + y] = ((s[8] & 0x20) >> 4) | ((s[8] & 0x02) >> 1);
			d[8*3 + y] = ((s[8] & 0x10) >> 3) | ((s[8] & 0x01) >> 0);

			d[8*4 + y] = ((s[0] & 0x80) >> 6) | ((s[0] & 0x08) >> 3);
			d[8*5 + y] = ((s[0] & 0x40) >> 5) | ((s[0] & 0x04) >> 2);
			d[8*6 + y] = ((s[0] & 0x20) >> 4) | ((s[0] & 0x02) >> 1);
			d[8*7 + y] = ((s[0] & 0x10) >> 3) | ((s[0] & 0x01) >> 0);

			s++;
		}
	}
}

/*-----------------------------------------------------------------------------
   Name: DecodeSprites()
   Desc: Decode graphics from original buffer
  -----------------------------------------------------------------------------*/
void Interpreter::DecodeSprites(UCHAR *srcBuffer, UCHAR *destBuffer)
{
	UCHAR *s;
	UCHAR *d;

	for (int sprite = 0; sprite < 64; sprite++)
	{
		s = &(srcBuffer[sprite * 64]);
		d = &(destBuffer[sprite * (16 * 16)]);

		for (int y = 15; y >= 0; y--)
		{
			if (s == &(srcBuffer[(sprite * 64) + 0x08]))
				s = &(srcBuffer[(sprite * 64) + 0x20]);

			d[16*0  + y] = ((s[8] & 0x80) >> 6) | ((s[8] & 0x08) >> 3);
			d[16*1  + y] = ((s[8] & 0x40) >> 5) | ((s[8] & 0x04) >> 2);
			d[16*2  + y] = ((s[8] & 0x20) >> 4) | ((s[8] & 0x02) >> 1);
			d[16*3  + y] = ((s[8] & 0x10) >> 3) | ((s[8] & 0x01) >> 0);

			d[16*4  + y] = ((s[16] & 0x80) >> 6) | ((s[16] & 0x08) >> 3);
			d[16*5  + y] = ((s[16] & 0x40) >> 5) | ((s[16] & 0x04) >> 2);
			d[16*6  + y] = ((s[16] & 0x20) >> 4) | ((s[16] & 0x02) >> 1);
			d[16*7  + y] = ((s[16] & 0x10) >> 3) | ((s[16] & 0x01) >> 0);
	
			d[16*8  + y] = ((s[24] & 0x80) >> 6) | ((s[24] & 0x08) >> 3);
			d[16*9  + y] = ((s[24] & 0x40) >> 5) | ((s[24] & 0x04) >> 2);
			d[16*10 + y] = ((s[24] & 0x20) >> 4) | ((s[24] & 0x02) >> 1);
			d[16*11 + y] = ((s[24] & 0x10) >> 3) | ((s[24] & 0x01) >> 0);
	
			d[16*12 + y] = ((s[0] & 0x80) >> 6) | ((s[0] & 0x08) >> 3);
			d[16*13 + y] = ((s[0] & 0x40) >> 5) | ((s[0] & 0x04) >> 2);
			d[16*14 + y] = ((s[0] & 0x20) >> 4) | ((s[0] & 0x02) >> 1);
			d[16*15 + y] = ((s[0] & 0x10) >> 3) | ((s[0] & 0x01) >> 0);

			s++;
		}
	}
}

/*-----------------------------------------------------------------------------
   Name: FlipSpritesHoriz()
   Desc: Take decoded sprites and horizontally flip them
  -----------------------------------------------------------------------------*/
void Interpreter::FlipSpritesHoriz(UCHAR *srcBuffer, UCHAR *flipBuffer)
{
	UCHAR *s;
	UCHAR *f;

	for (int sprite = 0; sprite < 64; sprite++)
	{
		int spriteOffset;

		spriteOffset = sprite * (16 * 16);

		for (int y = 15; y >= 0; y--)
		{
			s = &(srcBuffer[spriteOffset + (16 * y)]);
			f = &(flipBuffer[spriteOffset + (16 * y)]);

			for (int x = 15; x >= 0; x--)
			{
				f[x] = s[15 - x];				
			}
		}
	}
}

/*-----------------------------------------------------------------------------
   Name: FlipSpritesVert()
   Desc: Take decoded sprites and vertically flip them
  -----------------------------------------------------------------------------*/
void Interpreter::FlipSpritesVert(UCHAR *srcBuffer, UCHAR *flipBuffer)
{
	UCHAR *s;
	UCHAR *f;

	for (int sprite = 0; sprite < 64; sprite++)
	{
		int spriteOffset;

		spriteOffset = sprite * (16 * 16);

		for (int y = 15; y >= 0; y--)
		{
			s = &(srcBuffer[spriteOffset + (16 * y)]);
			f = &(flipBuffer[spriteOffset + (16 * (15 - y))]);
			memcpy(f,s,16);
		}
	}
}

/*-----------------------------------------------------------------------------
   Name: FlipSpritesHorizVert()
   Desc: Take decoded sprites and horizontally/vertically flip them
  -----------------------------------------------------------------------------*/
void Interpreter::FlipSpritesHorizVert(UCHAR *srcBuffer, UCHAR *flipBuffer)
{
	for (int sprite = 0; sprite < 64; sprite++)
	{
		int spriteOffset;

		spriteOffset = sprite * (16 * 16);

		for (int i = 0; i < (16 * 16); i++)
		{
			flipBuffer[spriteOffset + i] = srcBuffer[spriteOffset + ((16 * 16) - 1) - i];
		}
	}
}

/*-----------------------------------------------------------------------------
   Name: InitSprites()
   Desc: Initialize the interpreted sprite list
  -----------------------------------------------------------------------------*/
void Interpreter::InitSprites(void)
{
	for (int sprite = 0; sprite < NUM_SPRITES; sprite++)
	{
		spriteList[sprite].x = 0;
		spriteList[sprite].y = 0;
		spriteList[sprite].color = 0;
		spriteList[sprite].graphicsPtr = NULL;
	}
}

/*-----------------------------------------------------------------------------
   Name: Interpret()
   Desc: Interpret emulated memory into usable format
  -----------------------------------------------------------------------------*/
void Interpreter::Interpret(void)
{
	InterpretSprites();

	InterpretSound();
}

/*-----------------------------------------------------------------------------
   Name: InterpretSprites()
   Desc: Read emulated memory and refresh the interpreted sprite list
  -----------------------------------------------------------------------------*/
void Interpreter::InterpretSprites(void)
{
	int pic, flip;
	SPRITE_STRUCT *curSprite;

	curSprite = spriteList;

	// Flip sprite order for correct drawing priority
	for (int sprite = NUM_SPRITES - 1; sprite >= 0; sprite--)
	{
		curSprite->x = 255 - emuMemory[0x5060 + sprite*2];
		curSprite->y = 255 - emuMemory[0x5061 + sprite*2];
		curSprite->color = emuMemory[0x4ff1 + sprite*2] & 0x1F;
		pic   = emuMemory[0x4ff0 + sprite*2] >> 2;
		flip  = emuMemory[0x4ff0 + sprite*2] & 0x03;

		switch (flip)
		{
			/* No flip */
			case 0:
				curSprite->graphicsPtr = &(spritePtr[pic * (16*16)]);
				break;
			/* Vert flip */
			case 1:
				curSprite->graphicsPtr = &(spriteVFlipPtr[pic * (16*16)]);
				break;
			/* Horiz flip */
			case 2:
				curSprite->graphicsPtr = &(spriteHFlipPtr[pic * (16*16)]);
				break;
			/* Horiz/Vert flip */
			case 3:
				curSprite->graphicsPtr = &(spriteHVFlipPtr[pic * (16*16)]);
				break;
		}

		curSprite++;
	}
}

/*-----------------------------------------------------------------------------
   Name: InterpretSound()
   Desc: Read emulated memory and refresh sound parameters
  -----------------------------------------------------------------------------*/
void Interpreter::InterpretSound(void)
{
	if (emuMemory[0x5001] == 0)
		soundEnable = false;
	else
		soundEnable = true;

	for (int i = 0; i < NUM_SOUND_VOICES; i++ )
	{
		// Only mem locations 5045, 504A, 504F, and 5050-505F are used.

		wave[i]   = emuMemory[0x5045 + i*5] & 0x07;
		volume[i] = emuMemory[0x5055 + i*5] & 0x0F;

		frequency[i]  = (emuMemory[0x5054 + i*5] & 0x0F) << 16;
		frequency[i] |= (emuMemory[0x5053 + i*5] & 0x0F) << 12;
		frequency[i] |= (emuMemory[0x5052 + i*5] & 0x0F) << 8;
		frequency[i] |= (emuMemory[0x5051 + i*5] & 0x0F) << 4;

		// Special case - first voice gets an extra nibble of accuracy
		if (i == 0)
			frequency[i] |= (emuMemory[0x5050] & 0x0F);
	}
}

