/***************************************************************************

  Mini-Pac Copyright (C) 1999 by Michael Balfour.  All Rights Reserved.


  Pac-Man Copyright (C) 1980 by Namco, Ltd.  All Rights Reserved.


  Multi-Z80 32 Bit emulator 
  Copyright (C) 1996-1999 Neil Bradley, All rights reserved
  Author      : Neil Bradley (neil@synthcom.com)
  Distribution: ftp://ftp.synthcom.com/pub/emulators/cpu/makez80.zip (latest)

 ***************************************************************************/

// INCLUDES ///////////////////////////////////////////////////////////////////

#include "minipac.h"

/*-----------------------------------------------------------------------------
   Name: MiniPac()
   Desc: Construct our main object
  -----------------------------------------------------------------------------*/

MiniPac::MiniPac(HWND hwnd)
{
	// initalize our Emulation Interpreter
	// (We do this first in case there's a problem loading files)
    pacInterpreter = new Interpreter;

    // initialize direct draw, might throw exception
	DDInit(hwnd);
	CreateSurfaces();

    // initialize the sub classes
    pac2D = new Pac2D(ddObject,pacInterpreter);
    pacEngine = new PacEngine(pacInterpreter);
	pacSound = new PacSound(pacInterpreter, hwnd);

    // Init the palette
    InitPalette();

	// Figure out timer stuff
	QueryPerformanceFrequency(&timerFreq);
	QueryPerformanceCounter(&lastTimerValue);
	timeTillSecond = timerFreq;
	QueryPerformanceCounter(&lastFrameValue);
	frameFreq.QuadPart = timerFreq.QuadPart / 60;
	realFrameFreq = frameFreq;
	timeTillFrame = frameFreq;

}

/*-----------------------------------------------------------------------------
   Name: ~MiniPac()
   Desc: Destroy our main object
  -----------------------------------------------------------------------------*/

MiniPac::~MiniPac()
{
    // remove our sub classes
    delete pac2D;
    delete pacEngine;
	delete pacSound;
    delete pacInterpreter;

	// shut down direct draw
	DDShutdown();

}


/*-----------------------------------------------------------------------------
   Name: DDInit()
   Desc: Create our main DirectDraw object
  -----------------------------------------------------------------------------*/

void MiniPac::DDInit(HWND hwnd)
{
	// now that the windows portion is complete, start up direct draw
	if (DirectDrawCreate(NULL,&ddObject,NULL)!=DD_OK)
	{
		// shutdown any other dd objects and kill window
		DDShutdown();
		throw "Direct Draw Create failed!";
	} // end if

	// now set the coop level to exclusive and set for full screen and mode x
	if (ddObject->SetCooperativeLevel(hwnd, DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN | DDSCL_ALLOWREBOOT)!=DD_OK)
	{
		// shutdown any other dd objects and kill window
		DDShutdown();
		throw "Set Cooperative Level failed!";
	} // end if

	// now set the display mode
	if (ddObject->SetDisplayMode(SCREEN_WIDTH,SCREEN_HEIGHT,SCREEN_BPP)!=DD_OK)
	{
		// shutdown any other dd objects and kill window
		DDShutdown();
		throw "Set Display Mode failed!";
	} // end if
}

/*-----------------------------------------------------------------------------
   Name: CreateSurfaces()
   Desc: Create our DirectDraw surfaces
  -----------------------------------------------------------------------------*/

void MiniPac::CreateSurfaces(void)
{
    DDSURFACEDESC        ddSurfaceDesc;						// a direct draw surface description struct

	// clear all the palette entries to RGB 0,0,0
	memset(color_palette,0,256*sizeof(PALETTEENTRY));

	// Create the primary surface
	ddSurfaceDesc.dwSize            = sizeof(ddSurfaceDesc);
	ddSurfaceDesc.dwFlags           = DDSD_CAPS;
	ddSurfaceDesc.ddsCaps.dwCaps    = DDSCAPS_PRIMARYSURFACE;

	if (ddObject->CreateSurface(&ddSurfaceDesc,&ddPrimarySurface,NULL)!=DD_OK)
	{
		// shutdown any other dd objects and kill window
		DDShutdown();
		throw "Create Primary Surface failed!";
	} // end if

	// create the palette and attach it to the primary surface

	// now create the palette object, note that it is a member of the dd object itself
	if (ddObject->CreatePalette((DDPCAPS_8BIT | DDPCAPS_INITIALIZE),color_palette,&ddPalette,NULL)!=DD_OK)
	{
		// shutdown any other dd objects and kill window
		DDShutdown();
		throw "Create Palette failed!";
	} // end if

	// now attach the palette to the primary surface
	ddPrimarySurface->SetPalette(ddPalette);

    ZeroMemory(&ddSurfaceDesc, sizeof(ddSurfaceDesc));
	ddSurfaceDesc.dwSize         = sizeof(ddSurfaceDesc);
    ddSurfaceDesc.dwFlags        = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH;
    ddSurfaceDesc.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
    ddSurfaceDesc.dwHeight       = SCREEN_HEIGHT;
    ddSurfaceDesc.dwWidth        = SCREEN_WIDTH;
    if (ddObject->CreateSurface(&ddSurfaceDesc, &ddOffscreenSurface, NULL) != DD_OK)
    {   
        DDShutdown();
        throw "Create Offscreen Surface failed!";
    }

    // return success if we got this far
	return;

}

/*-----------------------------------------------------------------------------
   Name: DDShutdown()
   Desc: Clean up the Direct Draw stuff
  -----------------------------------------------------------------------------*/
// this function tests for dd components that have been created and releases
// them back to the operating system
void MiniPac::DDShutdown(void)
{

	// test if the dd object exists
	if (ddObject != NULL)
	{
		// test if there is a primary surface
		if(ddPrimarySurface != NULL)
		{
			// release the memory and set pointer to NULL
			ddPrimarySurface->Release();
			ddPrimarySurface = NULL;
		} // end if

		// test if there is an offscreen surface
		if(ddOffscreenSurface != NULL)
		{
			// release the memory and set pointer to NULL
			ddOffscreenSurface->Release();
			ddOffscreenSurface = NULL;
		} // end if

		// now release the dd object itself
		ddObject->Release();
		ddObject = NULL;
	} // end if

}

/*-----------------------------------------------------------------------------
   Name: InitPalette()
   Desc: Initialize the palette to the colors we need
  -----------------------------------------------------------------------------*/

void MiniPac::InitPalette(void)
{
    int startOffset = 0;

    // Init shadow palette
	pac2D->InitPalette(color_palette);

    // Copy to real palette
    for (int i = 0; i < 256; i++ )
    {
    	// set the color palette entry
    	ddPalette->SetEntries(0,i,1,&color_palette[i]);
    }
}

/*-----------------------------------------------------------------------------
   Name: ExecuteFrame()
   Desc: Perform all functions necessary for one frame of video
  -----------------------------------------------------------------------------*/

int MiniPac::ExecuteFrame(void)
{
    static int onscreen_fps = 0;
    UCHAR               *video_buffer;             // pointer to video ram
    DDSURFACEDESC       ddSurfaceDesc;				// a direct draw surface description struct
    RECT                rcRect;
    HRESULT             hRet;

	// Run one frame worth of Z80.
	pacEngine->RunFrame();
	// Interpret results of game engine
	pacInterpreter->Interpret();

	// Handle sound
	pacSound->PlayFrame();

    // acquire pointer to video ram, note it is always linear
	memset(&ddSurfaceDesc,0,sizeof(ddSurfaceDesc));
	ddSurfaceDesc.dwSize = sizeof(ddSurfaceDesc);
	if (ddOffscreenSurface->Lock(NULL,&ddSurfaceDesc,DDLOCK_SURFACEMEMORYPTR,NULL) == DD_OK)
	{
		video_buffer = (UCHAR *)ddSurfaceDesc.lpSurface;

		// draw graphics
		pac2D->Draw(video_buffer);
		pac2D->DrawFPS(video_buffer, onscreen_fps);

		// release pointer to video ram
		ddOffscreenSurface->Unlock(ddSurfaceDesc.lpSurface);

	    // Blit offscreen to onscreen (pac emu rectangle + FPS)
	    rcRect.left = 0;
	    rcRect.top = 0;
	    rcRect.right = (256 - 32);
	    rcRect.bottom = (256 + 32) + 8;

		while (true)
		{
	        hRet = ddPrimarySurface->BltFast(0, 0, ddOffscreenSurface, &rcRect, false);
			if (hRet == DD_OK)
			{
	            break;
			}
			else if (hRet == DDERR_SURFACELOST)
			{
				ddPrimarySurface->Restore();
				ddOffscreenSurface->Restore();
			}
			else if (hRet != DDERR_WASSTILLDRAWING)
	            return 0;
	    }
	}

	// Pause for the rest of frame
    do
	{
		QueryPerformanceCounter(&frameValue);
		timeTillFrame.QuadPart = timeTillFrame.QuadPart - (frameValue.QuadPart - lastFrameValue.QuadPart);
		lastFrameValue = frameValue;
	} while (timeTillFrame.QuadPart > 0);
	lastFrameValue.QuadPart -= timeTillFrame.QuadPart;
	timeTillFrame = frameFreq;

	QueryPerformanceCounter(&timerValue);
	fps++;
	timeTillSecond.QuadPart = timeTillSecond.QuadPart - (timerValue.QuadPart - lastTimerValue.QuadPart);
	lastTimerValue = timerValue;
	if (timeTillSecond.QuadPart < 0)
	{
        onscreen_fps = fps;
		fps = 0;
		timeTillSecond = timerFreq;
	}

    return 1;
}

/*-----------------------------------------------------------------------------
   Name: Throttle()
   Desc: Turn on speed Throttling
  -----------------------------------------------------------------------------*/

void MiniPac::Throttle(void)
{
    frameFreq = realFrameFreq;
}

/*-----------------------------------------------------------------------------
   Name: Unthrottle()
   Desc: Turn off speed Throttling
  -----------------------------------------------------------------------------*/

void MiniPac::Unthrottle(void)
{
    frameFreq.QuadPart = 0;
}

/*-----------------------------------------------------------------------------
   Name: Reset()
   Desc: Reset the emulation
  -----------------------------------------------------------------------------*/

void MiniPac::Reset(void)
{
    pacEngine->Reset();
}


