/***************************************************************************

  Mini-Pac Copyright (C) 1999 by Michael Balfour.  All Rights Reserved.


  Pac-Man Copyright (C) 1980 by Namco, Ltd.  All Rights Reserved.


  Multi-Z80 32 Bit emulator 
  Copyright (C) 1996-1999 Neil Bradley, All rights reserved
  Author      : Neil Bradley (neil@synthcom.com)
  Distribution: ftp://ftp.synthcom.com/pub/emulators/cpu/makez80.zip (latest)

 ***************************************************************************/

// INCLUDES ///////////////////////////////////////////////////////////////////

#include "minipac.h"

#define SAMPLES_PER_SEC (3072000/32)

/*-----------------------------------------------------------------------------
   Name: PacSound()
   Desc: Construct our main object
  -----------------------------------------------------------------------------*/

PacSound::PacSound(Interpreter *com, HWND hwnd)
{
	int i;

	soundEnable = false;

	ddSound = NULL;

	for (i = 0; i < NUM_SOUND_VOICES; i++)
	{
		dsBuffer[i] = NULL;
		lastVolume[i] = 0;
		lastFrequency[i] = 0;
		lastWaveform[i] = 0;
	}

	for (i = 0; i < 8; i++)
		wavePtr[i] = NULL;

    c = com;

	if (DirectSoundCreate(NULL, &ddSound, NULL) != DS_OK)
	{
		// We won't discriminate against people without a sound card :)
		soundEnable = false;
		return;
	}

	if (ddSound->SetCooperativeLevel(hwnd, DSSCL_PRIORITY) != DS_OK)
	{
		throw "DS SetCooperativeLevel failed";
	}

	InitWaveBuffers();

	CreateBuffers();
}

/*-----------------------------------------------------------------------------
   Name: ~PacSound()
   Desc: Destroy our main object
  -----------------------------------------------------------------------------*/
PacSound::~PacSound()
{
	int w;

	for (w = 0; w < 8; w++ )
	{
		if (wavePtr[w] != NULL)
			free(wavePtr[w]);
	}

	if (ddSound != NULL)
	{
		for (w = 0; w < NUM_SOUND_VOICES; w++)
		{
			if (dsBuffer[w] != NULL)
			{
				dsBuffer[w]->Stop();
				dsBuffer[w]->Release();
			}
		}

		ddSound->Release();
	}

}

/*-----------------------------------------------------------------------------
   Name: InitWaveBuffers()
   Desc: Create 8 0x10000-byte wave buffers for our 8 0x20-byte waves.
  -----------------------------------------------------------------------------*/
void PacSound::InitWaveBuffers(void)
{
	for (int w = 0; w < 8; w++ )
	{
		wavePtr[w] = (char *)malloc(SAMPLES_PER_SEC);
		if (wavePtr[w] == NULL)
		{
			throw "Error allocating wave pointer memory.";
		}

		for (int i = 0; i < SAMPLES_PER_SEC; i+=0x20 )
		{
			memcpy(&(wavePtr[w][i]), &(c->waveBuffer[w*0x20]), 0x20);
		}
	}
}


/*-----------------------------------------------------------------------------
   Name: EnableSound()
   Desc: Enable sound
  -----------------------------------------------------------------------------*/
void PacSound::EnableSound(void)
{
	if ((ddSound != NULL) && (!soundEnable))
	{
		soundEnable = true;
		for (int w = 0; w < NUM_SOUND_VOICES; w++)
		{
			if (c->frequency[w] != 0)
				dsBuffer[w]->Play(0,0,DSBPLAY_LOOPING);
		}
	}
}

/*-----------------------------------------------------------------------------
   Name: DisableSound()
   Desc: Disable sound
  -----------------------------------------------------------------------------*/
void PacSound::DisableSound(void)
{
	if ((ddSound != NULL) && (soundEnable))
	{
		soundEnable = false;
		for (int w = 0; w < NUM_SOUND_VOICES; w++)
		{
			DWORD status;

			dsBuffer[w]->GetStatus(&status);

			if ((status & DSBSTATUS_PLAYING) != 0)
				dsBuffer[w]->Stop();
		}
	}
}

/*-----------------------------------------------------------------------------
   Name: SetWaveform()
   Desc: Change the waveform for one of the voices
  -----------------------------------------------------------------------------*/
void PacSound::SetWaveform(int voice, int wave)
{
	LPVOID ptr1, ptr2;
	DWORD  size1, size2;
	DWORD status;

	dsBuffer[voice]->GetStatus(&status);
	if ((status & DSBSTATUS_PLAYING) != 0)
	{
		dsBuffer[voice]->Stop();
	}
	if (dsBuffer[voice]->Lock(0, SAMPLES_PER_SEC, &ptr1, &size1, &ptr2, &size2, 0) == DS_OK)
	{
		CopyMemory(ptr1, wavePtr[wave], size1);
		if (size2 != 0)
			CopyMemory(ptr2, wavePtr[wave] + size1, size2);
	}
	dsBuffer[voice]->Unlock(ptr1, size1, ptr1, size2);
	if ((status & DSBSTATUS_PLAYING) != 0)
	{
		dsBuffer[voice]->Play(0,0,DSBPLAY_LOOPING);
	}
}

/*-----------------------------------------------------------------------------
   Name: CreateBuffers()
   Desc: Create our DirectSound sound buffer and an internal buffer
  -----------------------------------------------------------------------------*/

void PacSound::CreateBuffers(void)
{
	PCMWAVEFORMAT	pcm;
	DSBUFFERDESC	dsDesc;

	LPVOID ptr1, ptr2;
	DWORD  size1, size2;

	// Describe our Pac-Man sound buffer
	memset(&pcm, 0, sizeof(PCMWAVEFORMAT));
	pcm.wf.wFormatTag = WAVE_FORMAT_PCM;
	pcm.wf.nChannels = 1;
//	pcm.wf.nSamplesPerSec = 3072000/32;	// based on Pac's system clock
	pcm.wf.nSamplesPerSec = SAMPLES_PER_SEC;
    pcm.wBitsPerSample = 8; 
    pcm.wf.nBlockAlign = pcm.wBitsPerSample * pcm.wf.nChannels / 8;
	pcm.wf.nAvgBytesPerSec = pcm.wf.nSamplesPerSec * pcm.wf.nBlockAlign;

	// Set up DSBUFFERDESC structure. 
    memset(&dsDesc, 0, sizeof(DSBUFFERDESC));
    dsDesc.dwSize = sizeof(DSBUFFERDESC); 
    // Need default controls (pan, volume, frequency). 
    dsDesc.dwFlags = DSBCAPS_CTRLDEFAULT;
//    dsDesc.dwBufferBytes = 1 * pcm.wf.nAvgBytesPerSec;      // 1-second buffer
    dsDesc.dwBufferBytes = SAMPLES_PER_SEC;
    dsDesc.lpwfxFormat = (LPWAVEFORMATEX)&pcm;     // Create buffer. 

	for (int w = 0; w < NUM_SOUND_VOICES; w++)
	{
		if (ddSound->CreateSoundBuffer(&dsDesc, &(dsBuffer[w]), NULL) != DS_OK)
		{
			throw "Failed to create sound buffer!";
		}

		if (dsBuffer[w]->Lock(0, SAMPLES_PER_SEC, &ptr1, &size1, &ptr2, &size2, 0) == DS_OK)
		{
			CopyMemory(ptr1, wavePtr[0], size1);
			if (size2 != 0)
				CopyMemory(ptr2, wavePtr[0] + size1, size2);
		}
		dsBuffer[w]->Unlock(ptr1, size1, ptr1, size2);
	}
}

/*-----------------------------------------------------------------------------
   Name: PlayFrame()
   Desc: Add one frame's worth of sound to the buffer
  -----------------------------------------------------------------------------*/
void PacSound::PlayFrame(void)
{
	if (ddSound == NULL)
		return;

	if (c->soundEnable)
		EnableSound();
	else
		DisableSound();

	if (!soundEnable)
		return;

	for (int w = 0; w < NUM_SOUND_VOICES; w++ )
	{
		DWORD status;

		dsBuffer[w]->GetStatus(&status);

		if (c->frequency[w] == 0)
		{
			if ((status & DSBSTATUS_PLAYING) != 0) 
				dsBuffer[w]->Stop();
		}
		else
		{
			if ((status & DSBSTATUS_PLAYING) == 0)
				dsBuffer[w]->Play(0,0,DSBPLAY_LOOPING);

			if (c->frequency[w] != lastFrequency[w])
				dsBuffer[w]->SetFrequency(c->frequency[w]*3);

			if (c->volume[w] != lastVolume[w])
			{
				if (c->volume[w] == 0)
					dsBuffer[w]->SetVolume(DSBVOLUME_MIN);
				else
					dsBuffer[w]->SetVolume(-(c->volume[w]*16));
			}
		}

		if (c->wave[w] != lastWaveform[w])
			SetWaveform(w, c->wave[w]);

		lastFrequency[w] = c->frequency[w];
		lastVolume[w] = c->volume[w];
		lastWaveform[w] = c->wave[w];
	}
}
