/***************************************************************************

  Mini-Pac Copyright (C) 1999 by Michael Balfour.  All Rights Reserved.


  Pac-Man Copyright (C) 1980 by Namco, Ltd.  All Rights Reserved.


  Multi-Z80 32 Bit emulator 
  Copyright (C) 1996-1999 Neil Bradley, All rights reserved
  Author      : Neil Bradley (neil@synthcom.com)
  Distribution: ftp://ftp.synthcom.com/pub/emulators/cpu/makez80.zip (latest)

 ***************************************************************************/

#include "minipac.h"

#include "MZ80.h"

/////////////////////////////////////////////////////////////////////////////////////////////

UINT8 *emu_mem;
UINT8 *dirty_mem;

static CONTEXTMZ80 z80info;
static bool interrupt_enabled = true;
static UINT8 interrupt_vector = 0;

// PROTOTYPES ////////////////////////////////////////////////////////////////

void null_write(UINT32 addr, UINT8 data, struct MemoryWriteByte *wmem);
void interrupt_vector_write(UINT16 addr, UINT8 data, struct z80PortWrite *wport);
void interrupt_enable_write(UINT32 addr, UINT8 data, struct MemoryWriteByte *wmem);

UINT8 input_port_read(UINT32 addr, struct MemoryReadByte *rmem);

void video_color_write(UINT32 addr, UINT8 data, struct MemoryWriteByte *wmem);

void sound_register_write(UINT32 addr, UINT8 data, struct MemoryWriteByte *wmem);

/////////////////////////////////////////////////////////////////////////////////////////////

static struct z80PortRead read_ports[] =
{
	{(UINT16) -1, (UINT16) -1, NULL}
};

static struct z80PortWrite write_ports[] =
{
	{0x00, 0x00, interrupt_vector_write },
	{(UINT16) -1, (UINT16) -1, NULL}
};

static struct MemoryReadByte read_memory[] =
{
	{0x5000, 0x50ff, input_port_read },
	{(UINT32) -1, (UINT32) -1, NULL}
};

static struct MemoryWriteByte write_memory[] =
{
	{0x4000, 0x47ff, video_color_write  },
	{0x5000, 0x5000, interrupt_enable_write },
	{0x5040, 0x5040, null_write },				// not used for sound
//	{0x5040, 0x505f, sound_register_write},
	{0x5070, 0x50ff, null_write },
	{0xc000, 0xc7ff, video_color_write },
	{(UINT32) -1, (UINT32) -1, NULL}
};

/////////////////////////////////////////////////////////////////////////////////////////////

PacEngine::PacEngine(Interpreter *com)
{
    c = com;

	/* Create the emulated game area */
	emu_mem   = (UINT8 *)(c->emuMemory);
	dirty_mem = (UINT8 *)(c->dirtyVideoMemory);

	/* Prime the Z80 */
	z80info.z80Base     = emu_mem;
	z80info.z80MemRead  = read_memory;
	z80info.z80MemWrite = write_memory;
	z80info.z80IoRead   = read_ports;
	z80info.z80IoWrite  = write_ports;
	mz80SetContext(&z80info);
	mz80reset();
}

PacEngine::~PacEngine()
{
}

void PacEngine::RunFrame(void)
{
	mz80exec((18432000/6)/60);

	// Cause an interrupt on the Z80?
	if (interrupt_enabled)
	{
		mz80int(interrupt_vector);
		interrupt_enabled = false;
	}
}

void PacEngine::Reset(void)
{
    mz80reset();
}

/////////////////////////////////////////////////////////////////////////////////////////////

void null_write(UINT32 addr, UINT8 data, struct MemoryWriteByte *wmem)
{
}

void interrupt_vector_write(UINT16 addr, UINT8 data, struct z80PortWrite *wport)
{
	interrupt_vector = data;
}

void interrupt_enable_write(UINT32 addr, UINT8 data, struct MemoryWriteByte *wmem)
{
	interrupt_enabled = true;
}

/////////////////////////////////////////////////////////////////////////////////////////////

UINT8 input_port_read(UINT32 addr, struct MemoryReadByte *rmem)
{
	// Limit our reads to addresses $5000, $5040, $5080, $50C0
	return emu_mem[addr & 0x50C0];
}

/////////////////////////////////////////////////////////////////////////////////////////////

void video_color_write(UINT32 addr, UINT8 data, struct MemoryWriteByte *wmem)
{
	emu_mem[addr & 0x7FFF] = data;
	emu_mem[addr | 0x8000] = data;
	dirty_mem[addr & 0x3FF] = 1;
}

void sound_register_write(UINT32 addr, UINT8 data, struct MemoryWriteByte *wmem)
{
}

