/***************************************************************************
                          cfg.c  -  description
                             -------------------
    begin                : Sun Oct 28 2001
    copyright            : (C) 2001 by Pete Bernert
    email                : BlackDove@addcom.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version. See also the license.txt file for *
 *   additional informations.                                              *
 *                                                                         *
 ***************************************************************************/

//*************************************************************************//
// History of changes:
//
// 2007/10/27 - Pete
// - added SSSPSX frame limit mode and MxC stretching modes
//
// 2005/04/15 - Pete
// - changed user frame limit to floating point value
//
// 2004/02/08 - Pete
// - added Windows zn config file handling (no need to change it for Linux version)
//
// 2002/11/06 - Pete
// - added 2xSai, Super2xSaI, SuperEagle cfg stuff
//
// 2002/10/04 - Pete
// - added Win debug mode & full vram view key config
//
// 2002/09/27 - linuzappz
// - separated linux gui to conf.c
//
// 2002/06/09 - linuzappz
// - fixed linux about dialog
//
// 2002/02/23 - Pete
// - added capcom fighter special game fix
//
// 2002/01/06 - lu
// - Connected the signal "destroy" to gtk_main_quit() in the ConfDlg, it
//   should fix a possible weird behaviour
//
// 2002/01/06 - lu
// - now fpse for linux has a configurator, some cosmetic changes done.
//
// 2001/12/25 - linuzappz
// - added gtk_main_quit(); in linux config
//
// 2001/12/20 - syo
// - added "Transparent menu" switch
//
// 2001/12/18 - syo
// - added "wait VSYNC" switch
// - support refresh rate change
// - modified key configuration (added toggle wait VSYNC key)
//   (Pete: fixed key buffers and added "- default"
//    refresh rate (=0) for cards not supporting setting the
//    refresh rate)
//
// 2001/12/18 - Darko Matesic
// - added recording configuration
//
// 2001/12/15 - lu
// - now fpsewp has his save and load routines in fpsewp.c
//
// 2001/12/05 - syo
// - added  "use system memory" switch
// - The bug which fails in the change in full-screen mode from window mode is corrected.
// - added  "Stop screen saver" switch
//
// 2001/11/20 - linuzappz
// - added WriteConfig and rewrite ReadConfigFile
// - added SoftDlgProc and AboutDlgProc for Linux (under gtk+-1.2.5)
//
// 2001/11/11 - lu
// - added some ifdef for FPSE layer
//
// 2001/11/09 - Darko Matesic
// - added recording configuration
//
// 2001/10/28 - Pete
// - generic cleanup for the Peops release
//
//*************************************************************************//

#include "stdafx.h"

#define _IN_CFG

//-------------------------------------------------------------------------// windows headers

#ifdef _WINDOWS

#include "stdafx.h"
#include "record.h"
#include <vfw.h>
#include <stdio.h>

#include "externals.h"
#include "cfg.h"
#include "gpu.h"
#include "DisplayResolutions.h"

//-------------------------------------------------------------------------// linux headers
#else

#include <sys/stat.h>
#undef FALSE
#undef TRUE
#define MAKELONG(low,high)     ((unsigned long)(((unsigned short)(low)) | (((unsigned long)((unsigned short)(high))) << 16)))

#include "externals.h"
#include "cfg.h"
#include "gpu.h"

#endif

/////////////////////////////////////////////////////////////////////////////
// CONFIG FILE helpers.... used in (non-fpse) Linux and ZN Windows
/////////////////////////////////////////////////////////////////////////////

char* pConfigFile=NULL;

#ifndef _FPSE

#include <sys/stat.h>

// some helper macros:

#define GetValue(name, var) \
    p = strstr(pB, name); \
    if (p != NULL) { \
        p+=strlen(name); \
        while ((*p == ' ') || (*p == '=')) p++; \
        if (*p != '\n') var = atoi(p); \
    }

#define GetFloatValue(name, var) \
    p = strstr(pB, name); \
    if (p != NULL) { \
        p+=strlen(name); \
        while ((*p == ' ') || (*p == '=')) p++; \
        if (*p != '\n') var = (float)atof(p); \
    }

#define SetValue(name, var) \
    p = strstr(pB, name); \
    if (p != NULL) { \
        p+=strlen(name); \
        while ((*p == ' ') || (*p == '=')) p++; \
        if (*p != '\n') { \
            len = sprintf(t1, "%d", var); \
            strncpy(p, t1, len); \
            if (p[len] != ' ' && p[len] != '\n' && p[len] != 0) p[len] = ' '; \
        } \
    } \
    else { \
        size+=sprintf(pB+size, "%s = %d\n", name, var); \
    }

#define SetFloatValue(name, var) \
    p = strstr(pB, name); \
    if (p != NULL) { \
        p+=strlen(name); \
        while ((*p == ' ') || (*p == '=')) p++; \
        if (*p != '\n') { \
            len = sprintf(t1, "%.1f", (double)var); \
            strncpy(p, t1, len); \
            if (p[len] != ' ' && p[len] != '\n' && p[len] != 0) p[len] = ' '; \
        } \
    } \
    else { \
        size+=sprintf(pB+size, "%s = %.1f\n", name, (double)var); \
    }

/////////////////////////////////////////////////////////////////////////////

void ReadConfigFile()
{
    struct stat buf;
    FILE* in;
    char t[256];
    int len, size;
    char* pB, * p;

    if (pConfigFile)
        strcpy(t,pConfigFile);
    else
    {
        strcpy(t,"cfg/gpuPeopsSoftX.cfg");
        in = fopen(t,"rb");
        if (!in)
        {
            strcpy(t,"gpuPeopsSoftX.cfg");
            in = fopen(t,"rb");
            if (!in) sprintf(t,"%s/gpuPeopsSoftX.cfg",getenv("HOME"));
            else    fclose(in);
        }
        else     fclose(in);
    }

    if (stat(t, &buf) == -1) return;
    size = buf.st_size;

    in = fopen(t,"rb");
    if (!in) return;

    pB=(char*)malloc(size);
    memset(pB,0,size);

    len = fread(pB, 1, size, in);
    fclose(in);

    GetValue("ResX", iResX);
    if (iResX<20) iResX=20;
    iResX=(iResX/4)*4;

    GetValue("ResY", iResY);
    if (iResY<20) iResY=20;
    iResY=(iResY/4)*4;

    iWinSize=MAKELONG(iResX,iResY);

    GetValue("NoStretch", iUseNoStretchBlt);

    GetValue("Dithering", iUseDither);

    GetValue("FullScreen", iWindowMode);
    if (iWindowMode!=0) iWindowMode=0;
    else               iWindowMode=1;

    GetValue("ShowFPS", iShowFPS);
    if (iShowFPS<0) iShowFPS=0;
    if (iShowFPS>1) iShowFPS=1;

    GetValue("SSSPSXLimit", bSSSPSXLimit);
    if (iShowFPS<0) iShowFPS=0;
    if (iShowFPS>1) iShowFPS=1;

    GetValue("UseFrameLimit", UseFrameLimit);
    if (UseFrameLimit<0) UseFrameLimit=0;
    if (UseFrameLimit>1) UseFrameLimit=1;

    GetValue("UseFrameSkip", UseFrameSkip);
    if (UseFrameSkip<0) UseFrameSkip=0;
    if (UseFrameSkip>1) UseFrameSkip=1;

    GetValue("FPSDetection", iFrameLimit);
    if (iFrameLimit<1) iFrameLimit=1;
    if (iFrameLimit>2) iFrameLimit=2;

    GetFloatValue("FrameRate", fFrameRate);
    if (fFrameRate<10.0f)   fFrameRate=10.0f;
    if (fFrameRate>1000.0f) fFrameRate=1000.0f;

    GetValue("CfgFixes", dwCfgFixes);

    GetValue("UseFixes", iUseFixes);
    if (iUseFixes<0) iUseFixes=0;
    if (iUseFixes>1) iUseFixes=1;

    free(pB);

}

#endif

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

#ifdef _WINDOWS

/////////////////////////////////////////////////////////////////////////////
// globals

char szKeyDefaults[11] = { VK_DELETE, VK_INSERT, VK_HOME, 0x4f, 0x49, 0x50, VK_MULTIPLY, VK_SUBTRACT, VK_ADD, VK_F12, 0x00 }; // MOD2!!!
char szDevName[128];

////////////////////////////////////////////////////////////////////////
// prototypes

BOOL OnInitCfgDialog(HWND hW);
void OnCfgOK(HWND hW);
BOOL OnInitSoftDialog(HWND hW);
void OnSoftOK(HWND hW);
void OnCfgCancel(HWND hW);
void OnCfgDef1(HWND hW);
void OnBugFixes(HWND hW);

void OnRecording(HWND hW);

void SelectDev(HWND hW);
BOOL bTestModes(void);
void OnKeyConfig(HWND hW);
void GetSettings(HWND hW);
void DoDevEnum(HWND hW);

////////////////////////////////////////////////////////////////////////
// funcs

BOOL CALLBACK SoftDlgProc(HWND hW, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
    switch (uMsg)
    {
        case WM_INITDIALOG:
            return OnInitSoftDialog(hW);

        case WM_COMMAND:
        {
            switch (LOWORD(wParam))
            {
                case IDC_DISPMODE1:
                {
                    CheckDlgButton(hW,IDC_DISPMODE2,FALSE);
                    return TRUE;
                }
                case IDC_DISPMODE2:
                {
                    CheckDlgButton(hW,IDC_DISPMODE1,FALSE);
                    return TRUE;
                }
                case IDC_DEF1:
                    OnCfgDef1(hW);
                    return TRUE;
                case IDC_SELFIX:
                    OnBugFixes(hW);
                    return TRUE;
                case IDC_KEYCONFIG:
                    OnKeyConfig(hW);
                    return TRUE;
                case IDC_SELDEV:
                    SelectDev(hW);
                    return TRUE;
                case IDCANCEL:
                    OnCfgCancel(hW);
                    return TRUE;
                case IDOK:
                    OnSoftOK(hW);
                    return TRUE;

                case IDC_RECORDING:
                    OnRecording(hW);
                    return TRUE;
            }
        }
    }
    return FALSE;
}

////////////////////////////////////////////////////////////////////////
// init dlg
////////////////////////////////////////////////////////////////////////

void ComboBoxAddRes(HWND hWC,char* cs)
{
    int i=ComboBox_FindString(hWC,-1,cs);
    if (i!=CB_ERR) return;
    ComboBox_AddString(hWC,cs);
}

// MOD2!!!
extern void BlitScreen15(unsigned char *, long, long);
extern void BlitScreen15_2x(unsigned char *, long, long);
extern void BlitScreen15_3x(unsigned char *, long, long);
extern void BlitScreen16(unsigned char *, long, long);
extern void BlitScreen16_2x(unsigned char *, long, long);
extern void BlitScreen16_3x(unsigned char *, long, long);
extern void BlitScreen32(unsigned char *, long, long);
extern void BlitScreen32_2x(unsigned char *, long, long);
extern void BlitScreen32_3x(unsigned char *, long, long);
extern void BlitScreen32_xbrz(unsigned char *, long, long);

extern void Std2xSaI_ex5(unsigned char *, DWORD, unsigned char *, int, int);
extern void Std2xSaI_ex6(unsigned char *, DWORD, unsigned char *, int, int);
extern void Std2xSaI_ex8(unsigned char *, DWORD, unsigned char *, int, int);
extern void Super2xSaI_ex5(unsigned char *, DWORD, unsigned char *, int, int);
extern void Super2xSaI_ex6(unsigned char *, DWORD, unsigned char *, int, int);
extern void Super2xSaI_ex8(unsigned char *, DWORD, unsigned char *, int, int);
extern void SuperEagle_ex5(unsigned char *, DWORD, unsigned char *, int, int);
extern void SuperEagle_ex6(unsigned char *, DWORD, unsigned char *, int, int);
extern void SuperEagle_ex8(unsigned char *, DWORD, unsigned char *, int, int);
extern void Scale2x_ex6_5(unsigned char *, DWORD, unsigned char *, int, int);
extern void Scale2x_ex8(unsigned char *, DWORD, unsigned char *, int, int);
extern void Scale3x_ex6_5(unsigned char *, DWORD, unsigned char *, int, int);
extern void Scale3x_ex8(unsigned char *, DWORD, unsigned char *, int, int);
extern void scale2xbrz(unsigned char *, DWORD, unsigned char *, int, int);
extern void scale3xbrz(unsigned char *, DWORD, unsigned char *, int, int);
extern void scale4xbrz(unsigned char *, DWORD, unsigned char *, int, int);
extern void scale5xbrz(unsigned char *, DWORD, unsigned char *, int, int);
extern void scale6xbrz(unsigned char *, DWORD, unsigned char *, int, int);

#include "blitters15.h"
#include "blitters16.h"
#include "blitters32.h"
#include "screenModes.h"

LPCTSTR getScreenMode()
{
	return screenModes[iUseNoStretchBlt].description;
}

long getScreenScale()
{
	return screenModes[iUseNoStretchBlt].scale32;
}

long getScreenRealScale(int depth)
{
	if (depth == 32)
	{
		return screenModes[iUseNoStretchBlt].scale32;
	}
	else
	{
		return screenModes[iUseNoStretchBlt].scale16;
	}
}

int isXBRZ()
{
	return screenModes[iUseNoStretchBlt].isXBRZ;
}

struct Blitter getBlitter(int depth)
{
	if (depth == 15)
	{
		return blitters15[iUseNoStretchBlt];
	}
	if (depth == 16)
	{
		return blitters16[iUseNoStretchBlt];
	}
	return blitters32[iUseNoStretchBlt];
}
// MOD2!!!

BOOL OnInitSoftDialog(HWND hW)
{
	HWND hWC;
	char cs[256];
	int i;

	ReadConfig(); // read registry stuff

	if (szDevName[0])
		SetDlgItemText(hW, IDC_DEVICETXT, szDevName);


	// MOD2!!!
	SetDlgItemInt(hW, IDC_MONITORW, monitorW, FALSE);
	SetDlgItemInt(hW, IDC_MONITORH, monitorH, FALSE);
	SetDlgItemInt(hW, IDC_GAMEW, gameW, FALSE);
	SetDlgItemInt(hW, IDC_GAMEH, gameH, FALSE);
	SetDlgItemInt(hW, IDC_OVERSCAN0, overscan0, FALSE);
	SetDlgItemInt(hW, IDC_OVERSCAN1, overscan1, FALSE);
	SetDlgItemInt(hW, IDC_KEY_PRESS_AFTER, keyPressAfter, FALSE);
	SetDlgItemInt(hW, IDC_KEY_RELEASE_AFTER, keyReleaseAfter, FALSE);

	if (backbufferDepth == 32) CheckDlgButton(hW, IDC_BACKBUFFER_DEFAULT, BST_CHECKED);
	if (backbufferDepth == 16) CheckDlgButton(hW, IDC_BACKBUFFER_16, BST_CHECKED);
	if (backbufferDepth == 15) CheckDlgButton(hW, IDC_BACKBUFFER_15, BST_CHECKED);

	if (initialAspectRatio == 0) CheckDlgButton(hW, IDC_ASPECT_RATIO_CUSTOM, BST_CHECKED);
	if (initialAspectRatio == 1) CheckDlgButton(hW, IDC_ASPECT_RATIO_STRETCH, BST_CHECKED);
	if (initialAspectRatio == 2) CheckDlgButton(hW, IDC_ASPECT_RATIO_4_3, BST_CHECKED);

	for (int index = 0; index < SEND_KEYS_MAX; index++)
	{
		hWC = GetDlgItem(hW, IDC_SEND_KEY0 + index);

		ComboBoxAddRes(hWC, "off");
		for (int functionKey = 1; functionKey <= 12; functionKey++)
		{
			wsprintf(cs, "F%d", functionKey);
			ComboBoxAddRes(hWC, cs);
		}
		ComboBox_SetCurSel(hWC, sendKey[index]);
	}
	// MOD2!!!


	// MOD2!!!  IDC_RESOLUTION
	hWC = GetDlgItem(hW, IDC_RESOLUTION);
	initResolutions();
	struct DisplayResolution *first = displayResolutions.first;

	while (first)
	{
		if (first->refresh == 0)
		{
			wsprintf(cs, "%d x %d", first->width, first->height);
		}
		else
		{
			wsprintf(cs, "%d x %d  %d Hz", first->width, first->height, first->refresh);
		}
		ComboBoxAddRes(hWC, cs);
		first = first->next;
	}
	i = getIndexFromResolutionAndCorrectNotFound(&iResX, &iResY, &iRefreshRate);
	ComboBox_SetCurSel(hWC, i);
	// MOD2!!!


	// IDC_COLDEPTH
	hWC = GetDlgItem(hW, IDC_COLDEPTH);
	ComboBox_AddString(hWC, "16 Bit");
	ComboBox_AddString(hWC, "32 Bit");
	wsprintf(cs, "%d Bit", iColDepth); // resolution
	i = ComboBox_FindString(hWC, -1, cs);
	if (i == CB_ERR) i = 0;
	ComboBox_SetCurSel(hWC, i);


	SetDlgItemInt(hW, IDC_WINX, LOWORD(iWinSize), FALSE); // window size
	SetDlgItemInt(hW, IDC_WINY, HIWORD(iWinSize), FALSE);


	if (UseFrameLimit) CheckDlgButton(hW, IDC_USELIMIT, TRUE);

	if (UseFrameSkip) CheckDlgButton(hW, IDC_USESKIPPING, TRUE);

	if (iWindowMode)
		CheckRadioButton(hW, IDC_DISPMODE1, IDC_DISPMODE2, IDC_DISPMODE2);
	else
		CheckRadioButton(hW, IDC_DISPMODE1, IDC_DISPMODE2, IDC_DISPMODE1);

	if (iSysMemory) CheckDlgButton(hW, IDC_SYSMEMORY, TRUE);

	if (iStopSaver) CheckDlgButton(hW, IDC_STOPSAVER, TRUE);

	if (iUseFixes) CheckDlgButton(hW, IDC_GAMEFIX, TRUE);

	if (iShowFPS) CheckDlgButton(hW, IDC_SHOWFPS, TRUE);

	if (bVsync) CheckDlgButton(hW, IDC_VSYNC, TRUE);

	if (bTransparent) CheckDlgButton(hW, IDC_TRANSPARENT, TRUE);

	if (bSSSPSXLimit) CheckDlgButton(hW, IDC_SSSPSXLIMIT, TRUE);


	// IDC_NOSTRETCH
	hWC = GetDlgItem(hW, IDC_NOSTRETCH); // stretching

	for (int index = 0; index < SCREEN_MODES; index++)
	{
		ComboBox_AddString(hWC, screenModes[index].description);
	}
	ComboBox_SetCurSel(hWC, iUseNoStretchBlt);


	// IDC_DITHER
	hWC = GetDlgItem(hW, IDC_DITHER); // dithering
	ComboBox_AddString(hWC, "No dithering (fastest)");
	ComboBox_AddString(hWC, "Game dependend dithering (slow)");
	ComboBox_AddString(hWC, "Always dither g-shaded polygons (slowest)");
	ComboBox_SetCurSel(hWC, iUseDither);


	if (iFrameLimit == 2) // frame limit wrapper
		CheckDlgButton(hW, IDC_FRAMEAUTO, TRUE);
	else
		CheckDlgButton(hW, IDC_FRAMEMANUELL, TRUE);


	sprintf(cs, "%.2f", fFrameRate);
	SetDlgItemText(hW, IDC_FRAMELIM, cs); // set frame rate

	return TRUE;
}

////////////////////////////////////////////////////////////////////////
// on ok: take vals
////////////////////////////////////////////////////////////////////////

int clamp(HWND handle, int control, int min, int max)
{
	int value = GetDlgItemInt(handle, control, NULL, FALSE);

	if (value < min) return min;
	if (value > max) return max;

	return value;
}

void GetSettings(HWND hW)
{
	HWND hWC;
	char cs[256];
	int i, j;


	// IDC_RESOLUTION
	hWC = GetDlgItem(hW, IDC_RESOLUTION);
	i = ComboBox_GetCurSel(hWC);
	getResolutionFromIndex(i, &iResX, &iResY, &iRefreshRate); // MOD2!!!


	// MOD2!!!
	monitorW        = clamp(hW, IDC_MONITORW,           1, 100);
	monitorH        = clamp(hW, IDC_MONITORH,           1, 100);
	gameW           = clamp(hW, IDC_GAMEW,              1, 100);
	gameH           = clamp(hW, IDC_GAMEH,              1, 100);
	overscan0       = clamp(hW, IDC_OVERSCAN0,          0, 100);
	overscan1       = clamp(hW, IDC_OVERSCAN1,          0, 100);
	keyPressAfter   = clamp(hW, IDC_KEY_PRESS_AFTER,   50, 10000);
	keyReleaseAfter = clamp(hW, IDC_KEY_RELEASE_AFTER, 50, 10000);

	if (IsDlgButtonChecked(hW, IDC_BACKBUFFER_DEFAULT) == BST_CHECKED) backbufferDepth = 32;
	if (IsDlgButtonChecked(hW, IDC_BACKBUFFER_16)      == BST_CHECKED) backbufferDepth = 16;
	if (IsDlgButtonChecked(hW, IDC_BACKBUFFER_15)      == BST_CHECKED) backbufferDepth = 15;

	if (IsDlgButtonChecked(hW, IDC_ASPECT_RATIO_CUSTOM)  == BST_CHECKED) initialAspectRatio = 0;
	if (IsDlgButtonChecked(hW, IDC_ASPECT_RATIO_STRETCH) == BST_CHECKED) initialAspectRatio = 1;
	if (IsDlgButtonChecked(hW, IDC_ASPECT_RATIO_4_3)     == BST_CHECKED) initialAspectRatio = 2;

	for (int index = 0; index < SEND_KEYS_MAX; index++)
	{
		sendKey[index] = ComboBox_GetCurSel(GetDlgItem(hW, IDC_SEND_KEY0 + index));
	}
	// MOD2!!!


	hWC = GetDlgItem(hW, IDC_COLDEPTH); // get color depth
	i = ComboBox_GetCurSel(hWC);
	ComboBox_GetLBText(hWC, i, cs);
	iColDepth = atol(cs);


	i = GetDlgItemInt(hW, IDC_WINX, NULL, FALSE); // get win size
	if (i < 50) i = 50;
	if (i > 20000) i = 20000;
	j = GetDlgItemInt(hW, IDC_WINY, NULL, FALSE);
	if (j < 50) j = 50;
	if (j > 20000) j = 20000;
	iWinSize = MAKELONG(i, j);


	if (IsDlgButtonChecked(hW, IDC_DISPMODE2)) // win mode
		iWindowMode = 1;
	else
		iWindowMode = 0;

	if (IsDlgButtonChecked(hW, IDC_USELIMIT)) // fps limit
		UseFrameLimit = 1;
	else
		UseFrameLimit = 0;

	if (IsDlgButtonChecked(hW, IDC_USESKIPPING)) // fps skip
		UseFrameSkip = 1;
	else
		UseFrameSkip = 0;

	if (IsDlgButtonChecked(hW, IDC_GAMEFIX)) // game fix
		iUseFixes = 1;
	else
		iUseFixes = 0;

	if (IsDlgButtonChecked(hW, IDC_SYSMEMORY)) // use system memory
		iSysMemory = 1;
	else
		iSysMemory = 0;

	if (IsDlgButtonChecked(hW, IDC_STOPSAVER)) // stop screen saver
		iStopSaver = 1;
	else
		iStopSaver = 0;

	if (IsDlgButtonChecked(hW, IDC_VSYNC)) // wait VSYNC
		bVsync = bVsync_Key = TRUE;
	else
		bVsync = bVsync_Key = FALSE;

	if (IsDlgButtonChecked(hW, IDC_TRANSPARENT)) // transparent menu
		bTransparent = TRUE;
	else
		bTransparent = FALSE;

	if (IsDlgButtonChecked(hW, IDC_SSSPSXLIMIT)) // SSSPSX fps limit mode
		bSSSPSXLimit = TRUE;
	else
		bSSSPSXLimit = FALSE;

	if (IsDlgButtonChecked(hW, IDC_SHOWFPS)) // show fps
		iShowFPS = 1;
	else
		iShowFPS = 0;


	// IDC_NOSTRETCH
	hWC = GetDlgItem(hW, IDC_NOSTRETCH);
	iUseNoStretchBlt = ComboBox_GetCurSel(hWC);


	// IDC_DITHER
	hWC = GetDlgItem(hW, IDC_DITHER);
	iUseDither = ComboBox_GetCurSel(hWC);


	if (IsDlgButtonChecked(hW, IDC_FRAMEAUTO)) // frame rate
		iFrameLimit = 2;
	else
		iFrameLimit = 1;


	// IDC_FRAMELIM
	GetDlgItemText(hW, IDC_FRAMELIM, cs, 255);
	fFrameRate = (float)atof(cs);

	if (fFrameRate < 10.0f)   fFrameRate = 10.0f;
	if (fFrameRate > 1000.0f) fFrameRate = 1000.0f;
}

void OnSoftOK(HWND hW)
{
    GetSettings(hW);

    if (!iWindowMode && !bTestModes())                    // check fullscreen sets
    {
        MessageBox(hW,"Resolution/color depth not supported!","Error",MB_ICONERROR|MB_OK);
        return;
    }

    WriteConfig();                                        // write registry

    EndDialog(hW,TRUE);
}

////////////////////////////////////////////////////////////////////////
// Cancel
////////////////////////////////////////////////////////////////////////

void OnCfgCancel(HWND hW)
{
    EndDialog(hW,FALSE);
}

////////////////////////////////////////////////////////////////////////
// Bug fixes
////////////////////////////////////////////////////////////////////////

BOOL CALLBACK BugFixesDlgProc(HWND hW, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
    switch (uMsg)
    {
        case WM_INITDIALOG:
        {
            int i;

            for (i=0; i<32; i++)
            {
                if (dwCfgFixes&(1<<i))
                    CheckDlgButton(hW,IDC_FIX1+i,TRUE);
            }
        }

        case WM_COMMAND:
        {
            switch (LOWORD(wParam))
            {
                case IDCANCEL:
                    EndDialog(hW,FALSE);
                    return TRUE;

                case IDOK:
                {
                    int i;
                    dwCfgFixes=0;
                    for (i=0; i<32; i++)
                    {
                        if (IsDlgButtonChecked(hW,IDC_FIX1+i))
                            dwCfgFixes|=(1<<i);
                    }
                    EndDialog(hW,TRUE);
                    return TRUE;
                }
            }
        }
    }
    return FALSE;
}

void OnBugFixes(HWND hW)
{
    DialogBox(hInst,MAKEINTRESOURCE(IDD_FIXES),
              hW,(DLGPROC)BugFixesDlgProc);
}

////////////////////////////////////////////////////////////////////////
// Recording options
////////////////////////////////////////////////////////////////////////

void RefreshCodec(HWND hW)
{
    char buffer[255];
    union
    {
        char chFCC[5];
        DWORD dwFCC;
    } fcc;
    ICINFO icinfo;
    memset(&icinfo,0,sizeof(icinfo));
    icinfo.dwSize = sizeof(icinfo);
    strcpy(fcc.chFCC,"VIDC");
    RECORD_COMPRESSION1.hic = ICOpen(fcc.dwFCC,RECORD_COMPRESSION1.fccHandler,ICMODE_QUERY);
    if (RECORD_COMPRESSION1.hic)
    {
        ICGetInfo(RECORD_COMPRESSION1.hic,&icinfo,sizeof(icinfo));
        ICClose(RECORD_COMPRESSION1.hic);
        wsprintf(buffer,"16 bit Compression: %ws",icinfo.szDescription);
    }
    else
        wsprintf(buffer,"16 bit Compression: Full Frames (Uncompressed)");
    SetDlgItemText(hW,IDC_COMPRESSION1,buffer);

    memset(&icinfo,0,sizeof(icinfo));
    icinfo.dwSize = sizeof(icinfo);
    RECORD_COMPRESSION2.hic = ICOpen(fcc.dwFCC,RECORD_COMPRESSION2.fccHandler,ICMODE_QUERY);
    if (RECORD_COMPRESSION2.hic)
    {
        ICGetInfo(RECORD_COMPRESSION2.hic,&icinfo,sizeof(icinfo));
        ICClose(RECORD_COMPRESSION2.hic);
        wsprintf(buffer,"24 bit Compression: %ws",icinfo.szDescription);
    }
    else
        wsprintf(buffer,"24 bit Compression: Full Frames (Uncompressed)");
    SetDlgItemText(hW,IDC_COMPRESSION2,buffer);
}


BOOL CALLBACK RecordingDlgProc(HWND hW, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
    switch (uMsg)
    {
        case WM_INITDIALOG:
        {
            HWND hWC;
            CheckDlgButton(hW,IDC_REC_MODE1,RECORD_RECORDING_MODE==0);
            CheckDlgButton(hW,IDC_REC_MODE2,RECORD_RECORDING_MODE==1);
            hWC = GetDlgItem(hW,IDC_VIDEO_SIZE);
            ComboBox_ResetContent(hWC);
            ComboBox_AddString(hWC,"Full");
            ComboBox_AddString(hWC,"Half");
            ComboBox_AddString(hWC,"Quarter");
            ComboBox_SetCurSel(hWC,RECORD_VIDEO_SIZE);

            SetDlgItemInt(hW,IDC_REC_WIDTH,RECORD_RECORDING_WIDTH,FALSE);
            SetDlgItemInt(hW,IDC_REC_HEIGHT,RECORD_RECORDING_HEIGHT,FALSE);

            hWC = GetDlgItem(hW,IDC_FRAME_RATE);
            ComboBox_ResetContent(hWC);
            ComboBox_AddString(hWC,"1");
            ComboBox_AddString(hWC,"2");
            ComboBox_AddString(hWC,"3");
            ComboBox_AddString(hWC,"4");
            ComboBox_AddString(hWC,"5");
            ComboBox_AddString(hWC,"6");
            ComboBox_AddString(hWC,"7");
            ComboBox_AddString(hWC,"8");
            ComboBox_SetCurSel(hWC,RECORD_FRAME_RATE_SCALE);
            CheckDlgButton(hW,IDC_COMPRESSION1,RECORD_COMPRESSION_MODE==0);
            CheckDlgButton(hW,IDC_COMPRESSION2,RECORD_COMPRESSION_MODE==1);
            RefreshCodec(hW);
        }

        case WM_COMMAND:
        {
            switch (LOWORD(wParam))
            {
                case IDC_RECCFG:
                {
                    if (IsDlgButtonChecked(hW,IDC_COMPRESSION1))
                    {
                        BITMAPINFOHEADER bitmap = {40,640,480,1,16,0,640*480*2,2048,2048,0,0};
                        if (!ICCompressorChoose(hW,ICMF_CHOOSE_DATARATE|ICMF_CHOOSE_KEYFRAME,&bitmap,NULL,&RECORD_COMPRESSION1,"16 bit Compression")) return TRUE;
                        if (RECORD_COMPRESSION1.cbState>sizeof(RECORD_COMPRESSION_STATE1))
                        {
                            memset(&RECORD_COMPRESSION1,0,sizeof(RECORD_COMPRESSION1));
                            memset(&RECORD_COMPRESSION_STATE1,0,sizeof(RECORD_COMPRESSION_STATE1));
                            RECORD_COMPRESSION1.cbSize	= sizeof(RECORD_COMPRESSION1);
                        }
                        else
                        {
                            if (RECORD_COMPRESSION1.lpState!=RECORD_COMPRESSION_STATE1)
                                memcpy(RECORD_COMPRESSION_STATE1,RECORD_COMPRESSION1.lpState,RECORD_COMPRESSION1.cbState);
                        }
                        RECORD_COMPRESSION1.lpState = RECORD_COMPRESSION_STATE1;
                    }
                    else
                    {
                        BITMAPINFOHEADER bitmap = {40,640,480,1,24,0,640*480*3,2048,2048,0,0};
                        if (!ICCompressorChoose(hW,ICMF_CHOOSE_DATARATE|ICMF_CHOOSE_KEYFRAME,&bitmap,NULL,&RECORD_COMPRESSION2,"24 bit Compression")) return TRUE;
                        if (RECORD_COMPRESSION2.cbState>sizeof(RECORD_COMPRESSION_STATE2))
                        {
                            memset(&RECORD_COMPRESSION2,0,sizeof(RECORD_COMPRESSION2));
                            memset(&RECORD_COMPRESSION_STATE2,0,sizeof(RECORD_COMPRESSION_STATE2));
                            RECORD_COMPRESSION2.cbSize	= sizeof(RECORD_COMPRESSION2);
                        }
                        else
                        {
                            if (RECORD_COMPRESSION2.lpState!=RECORD_COMPRESSION_STATE2)
                                memcpy(RECORD_COMPRESSION_STATE2,RECORD_COMPRESSION2.lpState,RECORD_COMPRESSION2.cbState);
                        }
                        RECORD_COMPRESSION2.lpState = RECORD_COMPRESSION_STATE2;
                    }
                    RefreshCodec(hW);
                    return TRUE;
                }
                case IDCANCEL:
                    EndDialog(hW,FALSE);
                    return TRUE;

                case IDOK:
                {
                    HWND hWC;
                    if (IsDlgButtonChecked(hW,IDC_REC_MODE1))	RECORD_RECORDING_MODE = 0;
                    else										RECORD_RECORDING_MODE = 1;
                    hWC = GetDlgItem(hW,IDC_VIDEO_SIZE);
                    RECORD_VIDEO_SIZE = ComboBox_GetCurSel(hWC);
                    RECORD_RECORDING_WIDTH = GetDlgItemInt(hW,IDC_REC_WIDTH,NULL,FALSE);
                    RECORD_RECORDING_HEIGHT = GetDlgItemInt(hW,IDC_REC_HEIGHT,NULL,FALSE);
                    hWC = GetDlgItem(hW,IDC_FRAME_RATE);
                    RECORD_FRAME_RATE_SCALE = ComboBox_GetCurSel(hWC);
                    if (IsDlgButtonChecked(hW,IDC_COMPRESSION1))	RECORD_COMPRESSION_MODE = 0;
                    else										RECORD_COMPRESSION_MODE = 1;
                    EndDialog(hW,TRUE);
                    return TRUE;
                }
            }
        }
    }
    return FALSE;
}

void OnRecording(HWND hW)
{
    DialogBox(hInst,MAKEINTRESOURCE(IDD_RECORDING),
              hW,(DLGPROC)RecordingDlgProc);

}


////////////////////////////////////////////////////////////////////////
// default
////////////////////////////////////////////////////////////////////////

void OnCfgDef1(HWND hW)
{
	CheckDlgButton(hW, IDC_DISPMODE1, BST_UNCHECKED);
	CheckDlgButton(hW, IDC_DISPMODE2, BST_CHECKED);

	CheckDlgButton(hW, IDC_BACKBUFFER_DEFAULT, BST_CHECKED);
	CheckDlgButton(hW, IDC_BACKBUFFER_16, BST_UNCHECKED);
	CheckDlgButton(hW, IDC_BACKBUFFER_15, BST_UNCHECKED);

	CheckDlgButton(hW, IDC_ASPECT_RATIO_CUSTOM, BST_CHECKED);
	CheckDlgButton(hW, IDC_ASPECT_RATIO_STRETCH, BST_UNCHECKED);
	CheckDlgButton(hW, IDC_ASPECT_RATIO_4_3, BST_UNCHECKED);

	CheckDlgButton(hW, IDC_FRAMEAUTO, BST_CHECKED);
	CheckDlgButton(hW, IDC_FRAMEMANUELL, BST_UNCHECKED);

	ComboBox_SetCurSel(GetDlgItem(hW, IDC_RESOLUTION), 0);
	ComboBox_SetCurSel(GetDlgItem(hW, IDC_COLDEPTH), 1);
	ComboBox_SetCurSel(GetDlgItem(hW, IDC_NOSTRETCH), 0);
	ComboBox_SetCurSel(GetDlgItem(hW, IDC_DITHER), 0);

	SetDlgItemInt(hW, IDC_WINX, 640, FALSE);
	SetDlgItemInt(hW, IDC_WINY, 480, FALSE);

	SetDlgItemInt(hW, IDC_MONITORW, 16, FALSE);
	SetDlgItemInt(hW, IDC_MONITORH, 9, FALSE);

	SetDlgItemInt(hW, IDC_OVERSCAN0, 0, FALSE);
	SetDlgItemInt(hW, IDC_OVERSCAN1, 0, FALSE);

	SetDlgItemInt(hW, IDC_GAMEW, 42, FALSE);
	SetDlgItemInt(hW, IDC_GAMEH, 30, FALSE);

	SetDlgItemInt(hW, IDC_FRAMELIM, 60, FALSE);

	SetDlgItemInt(hW, IDC_KEY_PRESS_AFTER, 1000, FALSE);
	SetDlgItemInt(hW, IDC_KEY_RELEASE_AFTER, 500, FALSE);

	CheckDlgButton(hW, IDC_TRANSPARENT, BST_UNCHECKED);
	CheckDlgButton(hW, IDC_SSSPSXLIMIT, BST_UNCHECKED);
	CheckDlgButton(hW, IDC_SHOWFPS, BST_UNCHECKED);
	CheckDlgButton(hW, IDC_USELIMIT, BST_CHECKED);
	CheckDlgButton(hW, IDC_USESKIPPING, BST_UNCHECKED);

	CheckDlgButton(hW, IDC_SYSMEMORY, BST_UNCHECKED);
	CheckDlgButton(hW, IDC_VSYNC, BST_UNCHECKED);
	CheckDlgButton(hW, IDC_STOPSAVER, BST_UNCHECKED);
	CheckDlgButton(hW, IDC_GAMEFIX, BST_UNCHECKED);

	for (int index = 0; index < SEND_KEYS_MAX; index++)
	{
		ComboBox_SetCurSel(GetDlgItem(hW, IDC_SEND_KEY0 + index), 0);
	}
}

////////////////////////////////////////////////////////////////////////
// read registry
////////////////////////////////////////////////////////////////////////

void ReadConfig(void)
{
	HKEY myKey;
	DWORD temp;
	DWORD type;
	DWORD size;


	// MOD2!!!
	monitorW = 16;
	monitorH = 9;
	gameW = 42;
	gameH = 30;
	overscan0 = 0;
	overscan1 = 0;
	backbufferDepth = 32;
	initialAspectRatio = 0;
	keyPressAfter = 1000;
	keyReleaseAfter = 500;

	for (int index = 0; index < SEND_KEYS_MAX; index++)
	{
		sendKey[index] = 0;
	}

	iWindowMode = 1;
	iColDepth = 32;
	iWinSize = MAKELONG(640, 480);
	iUseNoStretchBlt = 0;
	iUseDither = 0;
	bTransparent = FALSE;
	iShowFPS = 0;
	UseFrameLimit = 1;
	UseFrameSkip = 0;
	bSSSPSXLimit = FALSE;
	iFrameLimit = 2;
	fFrameRate = 60.0f;
	iSysMemory = 0;
	bVsync = FALSE;
	iStopSaver = 0;
	iUseFixes = 0;
	dwCfgFixes = 0;

	getFirstResolution(&iResX, &iResY, &iRefreshRate);

	iUseGammaVal = 2048;

	lstrcpy(szGPUKeys, szKeyDefaults);
	// MOD2!!!


	memset(szDevName, 0, 128);
	memset(&guiDev, 0, sizeof(GUID));


	// zn Windows config file / standard Windows psx config (registry)
	if (pConfigFile)
		ReadConfigFile();
	else
	{
		if (RegOpenKeyEx(HKEY_CURRENT_USER, "Software\\Vision Thing\\PSEmu Pro\\GPU\\PeteSoftMOD2", 0, KEY_ALL_ACCESS, &myKey) == ERROR_SUCCESS) // MOD2!!!
		{
			// MOD2!!!
			size = 4; if (RegQueryValueEx(myKey, "MOD2_MonitorW",           0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) monitorW           = (int)temp;
			size = 4; if (RegQueryValueEx(myKey, "MOD2_MonitorH",           0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) monitorH           = (int)temp;
			size = 4; if (RegQueryValueEx(myKey, "MOD2_GameW",              0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) gameW              = (int)temp;
			size = 4; if (RegQueryValueEx(myKey, "MOD2_GameH",              0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) gameH              = (int)temp;
			size = 4; if (RegQueryValueEx(myKey, "MOD2_Overscan0",          0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) overscan0          = (int)temp;
			size = 4; if (RegQueryValueEx(myKey, "MOD2_Overscan1",          0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) overscan1          = (int)temp;
			size = 4; if (RegQueryValueEx(myKey, "MOD2_BackbufferDepth",    0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) backbufferDepth    = (int)temp;
			size = 4; if (RegQueryValueEx(myKey, "MOD2_InitialAspectRatio", 0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) initialAspectRatio = (int)temp;
			size = 4; if (RegQueryValueEx(myKey, "MOD2_KeyPressAfter",      0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) keyPressAfter      = (int)temp;
			size = 4; if (RegQueryValueEx(myKey, "MOD2_KeyReleaseAfter",    0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) keyReleaseAfter    = (int)temp;

			char sendKeyString[128];

			for (int index = 0; index < SEND_KEYS_MAX; index++)
			{
				sprintf(sendKeyString, "MOD2_SendKey%d", index);

				size = 4; if (RegQueryValueEx(myKey, sendKeyString, 0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) sendKey[index] = (int)temp;
			}
			// MOD2!!!

			size = 4; if (RegQueryValueEx(myKey, "ResX",           0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) iResX            = (int)temp;
			size = 4; if (RegQueryValueEx(myKey, "ResY",           0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) iResY            = (int)temp;
			size = 4; if (RegQueryValueEx(myKey, "RefreshRate",    0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) iRefreshRate     = (int)temp;
			size = 4; if (RegQueryValueEx(myKey, "WinSize",        0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) iWinSize         = (int)temp;
			size = 4; if (RegQueryValueEx(myKey, "WindowMode",     0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) iWindowMode      = (int)temp;
			size = 4; if (RegQueryValueEx(myKey, "ColDepth",       0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) iColDepth        = (int)temp;
			size = 4; if (RegQueryValueEx(myKey, "UseFrameLimit",  0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) UseFrameLimit    = (int)temp;
			size = 4; if (RegQueryValueEx(myKey, "UseFrameSkip",   0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) UseFrameSkip     = (int)temp;
			size = 4; if (RegQueryValueEx(myKey, "FrameLimit",     0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) iFrameLimit      = (int)temp;
			size = 4; if (RegQueryValueEx(myKey, "CfgFixes",       0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) dwCfgFixes       = (int)temp;
			size = 4; if (RegQueryValueEx(myKey, "UseFixes",       0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) iUseFixes        = (int)temp;
			size = 4; if (RegQueryValueEx(myKey, "ShowFPS",        0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) iShowFPS         = (int)temp;
			size = 4; if (RegQueryValueEx(myKey, "UseNoStrechBlt", 0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) iUseNoStretchBlt = (int)temp;
			size = 4; if (RegQueryValueEx(myKey, "UseDither",      0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) iUseDither       = (int)temp;
			size = 4; if (RegQueryValueEx(myKey, "UseGamma",       0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) iUseGammaVal     = (int)temp;


			getIndexFromResolutionAndCorrectNotFound(&iResX, &iResY, &iRefreshRate); // MOD2!!!


			if (!iFrameLimit)
			{
				UseFrameLimit = 1;
				UseFrameSkip = 0;
				iFrameLimit = 2;
			}


			// try to get the float framerate... if none: take int framerate
			fFrameRate = 0.0f;
			size = 4; if (RegQueryValueEx(myKey, "FrameRateFloat", 0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) fFrameRate = *((float*)(&temp));

			if (fFrameRate == 0.0f)
			{
				fFrameRate = 60.0f;
				size = 4; if (RegQueryValueEx(myKey, "FrameRate", 0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) fFrameRate = (float)temp;
			}


			size = 4; if (RegQueryValueEx(myKey, "UseSysMemory", 0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) iSysMemory   = (int)temp;
			size = 4; if (RegQueryValueEx(myKey, "StopSaver",    0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) iStopSaver   = (int)temp;
			size = 4; if (RegQueryValueEx(myKey, "WaitVSYNC",    0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) bVsync       = bVsync_Key = (BOOL)temp;
			size = 4; if (RegQueryValueEx(myKey, "Transparent",  0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) bTransparent = (BOOL)temp;
			size = 4; if (RegQueryValueEx(myKey, "SSSPSXLimit",  0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) bSSSPSXLimit = (BOOL)temp;


			size = 11;           RegQueryValueEx(myKey, "GPUKeys",    0, &type, (LPBYTE)&szGPUKeys, &size);
			size = 128;          RegQueryValueEx(myKey, "DeviceName", 0, &type, (LPBYTE)szDevName,  &size);
			size = sizeof(GUID); RegQueryValueEx(myKey, "GuiDev",     0, &type, (LPBYTE)&guiDev,    &size);


			//
			// Recording options
			//

#define GetDWORD(xa, xb) size = 4; if (RegQueryValueEx(myKey, xa, 0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) xb = (unsigned long)temp;

#define GetBINARY(xa, xb) size = sizeof(xb); RegQueryValueEx(myKey, xa, 0, &type, (LPBYTE)&xb, &size);

			GetDWORD("RecordingMode",            RECORD_RECORDING_MODE);
			GetDWORD("RecordingVideoSize",       RECORD_VIDEO_SIZE);
			GetDWORD("RecordingWidth",           RECORD_RECORDING_WIDTH);
			GetDWORD("RecordingHeight",          RECORD_RECORDING_HEIGHT);
			GetDWORD("RecordingFrameRateScale",  RECORD_FRAME_RATE_SCALE);
			GetDWORD("RecordingCompressionMode", RECORD_COMPRESSION_MODE);

			GetBINARY("RecordingCompression1",      RECORD_COMPRESSION1);
			GetBINARY("RecordingCompressionState1", RECORD_COMPRESSION_STATE1);
			GetBINARY("RecordingCompression2",      RECORD_COMPRESSION2);
			GetBINARY("RecordingCompressionState2", RECORD_COMPRESSION_STATE2);

			if (RECORD_RECORDING_WIDTH  > 1024) RECORD_RECORDING_WIDTH  = 1024;
			if (RECORD_RECORDING_HEIGHT > 768)  RECORD_RECORDING_HEIGHT = 768;
			if (RECORD_VIDEO_SIZE       > 2)    RECORD_VIDEO_SIZE       = 2;
			if (RECORD_FRAME_RATE_SCALE > 7)    RECORD_FRAME_RATE_SCALE = 7;

			if (RECORD_COMPRESSION1.cbSize != sizeof(RECORD_COMPRESSION1))
			{
				memset(&RECORD_COMPRESSION1, 0, sizeof(RECORD_COMPRESSION1));
				RECORD_COMPRESSION1.cbSize = sizeof(RECORD_COMPRESSION1);
			}

			RECORD_COMPRESSION1.lpState = RECORD_COMPRESSION_STATE1;
			if (RECORD_COMPRESSION2.cbSize != sizeof(RECORD_COMPRESSION2))
			{
				memset(&RECORD_COMPRESSION2, 0, sizeof(RECORD_COMPRESSION2));
				RECORD_COMPRESSION2.cbSize = sizeof(RECORD_COMPRESSION2);
			}

			RECORD_COMPRESSION2.lpState = RECORD_COMPRESSION_STATE2;

			//
			// end of recording options
			//

			RegCloseKey(myKey);
		}
	}

	if (!iColDepth) iColDepth  = 32;
	if (iUseFixes)  dwActFixes = dwCfgFixes;
	SetFixes();

	if ((iUseGammaVal < 0) || (iUseGammaVal > 1536)) iUseGammaVal = 2048;

	// MOD2!!!
	setCorrectedResolution(iResX, iResY);
	setCorrectedResolution2(iResX, iResY);
	setStretchedResolution(iResX, iResY);
	// MOD2!!!
}

// MOD2!!!
//
// Monitor AR: 16/9     (monitorW/monitorH)
// Resolution: 800 600  (iResX/iResY)
// Wanted AR:  4/3      (gameW/gameH)
//
// Corrected Monitor AR W = 9 / 3 * 4 = 12
//
// Corrected Monitor AR: 12/9
//
// Corrected Resolution W = 800 / 16 * 12 = 600
//
// Corrected Resolution: 600 600
//
//
// Corrected Resolution W =
//
// = 800 / 16 * 12
// = 800 / 16 * (9 / 3 * 4)
//
//   800   9 * 4
// = --- * -----
//    16     3
//
//   800 * 9 * 4
// = -----------
//      16 * 3
//
//   iResX * monitorH * gameW
// = ------------------------
//       monitorW * gameH
//
void setCorrectedResolution(LONG resolutionW, LONG resolutionH)
{
	LONG correctedResolutionW = (resolutionW * (LONG)monitorH * (LONG)gameW) / ((LONG)monitorW * (LONG)gameH);

	if (correctedResolutionW > resolutionW) correctedResolutionW = resolutionW;

	LONG xOffset = (resolutionW - correctedResolutionW) >> 1;

	correctedResolution.left = xOffset;
	correctedResolution.top = 0;
	correctedResolution.right = xOffset + correctedResolutionW;
	correctedResolution.bottom = resolutionH;
}

void setCorrectedResolution2(LONG resolutionW, LONG resolutionH)
{
	LONG correctedResolutionW = (resolutionW * (LONG)monitorH * 4) / ((LONG)monitorW * 3);

	if (correctedResolutionW > resolutionW) correctedResolutionW = resolutionW;

	LONG xOffset = (resolutionW - correctedResolutionW) >> 1;

	correctedResolution2.left = xOffset;
	correctedResolution2.top = 0;
	correctedResolution2.right = xOffset + correctedResolutionW;
	correctedResolution2.bottom = resolutionH;
}

void setStretchedResolution(LONG resolutionW, LONG resolutionH)
{
	stretchedResolution.left = 0;
	stretchedResolution.top = 0;
	stretchedResolution.right = resolutionW;
	stretchedResolution.bottom = resolutionH;
}
// MOD2!!!

////////////////////////////////////////////////////////////////////////

void ReadWinSizeConfig(void)
{
	HKEY myKey;
	DWORD temp;
	DWORD type;
	DWORD size;

	if (pConfigFile) return; // no need to read stuff in ZN config file mode

	iWinSize = MAKELONG(640, 480);

	getFirstResolution(&iResX, &iResY, &iRefreshRate); // MOD2!!!

	if (RegOpenKeyEx(HKEY_CURRENT_USER, "Software\\Vision Thing\\PSEmu Pro\\GPU\\PeteSoftMOD2", 0, KEY_ALL_ACCESS, &myKey) == ERROR_SUCCESS) // MOD2!!!
	{
		size = 4; if (RegQueryValueEx(myKey, "ResX",        0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) iResX        = (int)temp;
		size = 4; if (RegQueryValueEx(myKey, "ResY",        0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) iResY        = (int)temp;
		size = 4; if (RegQueryValueEx(myKey, "RefreshRate", 0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) iRefreshRate = (int)temp;
		size = 4; if (RegQueryValueEx(myKey, "WinSize",     0, &type, (LPBYTE)&temp, &size) == ERROR_SUCCESS) iWinSize     = (int)temp;

		getIndexFromResolutionAndCorrectNotFound(&iResX, &iResY, &iRefreshRate); // MOD2!!!

		RegCloseKey(myKey);
	}
}

////////////////////////////////////////////////////////////////////////
// write registry
////////////////////////////////////////////////////////////////////////

void WriteConfig(void)
{
	HKEY myKey;
	DWORD myDisp;
	DWORD temp;


	RegCreateKeyEx(HKEY_CURRENT_USER, "Software\\Vision Thing\\PSEmu Pro\\GPU\\PeteSoftMOD2", 0, NULL, REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &myKey, &myDisp); // MOD2!!!


	// MOD2!!!
	temp = monitorW;           RegSetValueEx(myKey, "MOD2_MonitorW",           0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = monitorH;           RegSetValueEx(myKey, "MOD2_MonitorH",           0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = gameW;              RegSetValueEx(myKey, "MOD2_GameW",              0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = gameH;              RegSetValueEx(myKey, "MOD2_GameH",              0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = overscan0;          RegSetValueEx(myKey, "MOD2_Overscan0",          0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = overscan1;          RegSetValueEx(myKey, "MOD2_Overscan1",          0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = backbufferDepth;    RegSetValueEx(myKey, "MOD2_BackbufferDepth",    0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = initialAspectRatio; RegSetValueEx(myKey, "MOD2_InitialAspectRatio", 0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = keyPressAfter;      RegSetValueEx(myKey, "MOD2_KeyPressAfter",      0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = keyReleaseAfter;    RegSetValueEx(myKey, "MOD2_KeyReleaseAfter",    0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));

	char sendKeyString[128];

	for (int index = 0; index < SEND_KEYS_MAX; index++)
	{
		sprintf(sendKeyString, "MOD2_SendKey%d", index);

		temp = sendKey[index]; RegSetValueEx(myKey, sendKeyString, 0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	}
	// MOD2!!!

	temp = iResX;                   RegSetValueEx(myKey, "ResX",           0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = iResY;                   RegSetValueEx(myKey, "ResY",           0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = iRefreshRate;            RegSetValueEx(myKey, "RefreshRate",    0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = iWinSize;                RegSetValueEx(myKey, "WinSize",        0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = iWindowMode;             RegSetValueEx(myKey, "WindowMode",     0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = iColDepth;               RegSetValueEx(myKey, "ColDepth",       0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = UseFrameLimit;           RegSetValueEx(myKey, "UseFrameLimit",  0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = UseFrameSkip;            RegSetValueEx(myKey, "UseFrameSkip",   0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = dwCfgFixes;              RegSetValueEx(myKey, "CfgFixes",       0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = iUseFixes;               RegSetValueEx(myKey, "UseFixes",       0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = iShowFPS;                RegSetValueEx(myKey, "ShowFPS",        0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = iUseNoStretchBlt;        RegSetValueEx(myKey, "UseNoStrechBlt", 0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = iUseDither;              RegSetValueEx(myKey, "UseDither",      0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = iFrameLimit;             RegSetValueEx(myKey, "FrameLimit",     0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = iUseGammaVal;            RegSetValueEx(myKey, "UseGamma",       0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = (DWORD)fFrameRate;       RegSetValueEx(myKey, "FrameRate",      0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = *((DWORD *)&fFrameRate); RegSetValueEx(myKey, "FrameRateFloat", 0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = bVsync;                  RegSetValueEx(myKey, "WaitVSYNC",      0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = bTransparent;            RegSetValueEx(myKey, "Transparent",    0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = bSSSPSXLimit;            RegSetValueEx(myKey, "SSSPSXLimit",    0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = iSysMemory;              RegSetValueEx(myKey, "UseSysMemory",   0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));
	temp = iStopSaver;              RegSetValueEx(myKey, "StopSaver",      0, REG_DWORD, (LPBYTE)&temp, sizeof(temp));


	RegSetValueEx(myKey, "GPUKeys",    0, REG_BINARY, (LPBYTE)szGPUKeys, 11);
	RegSetValueEx(myKey, "DeviceName", 0, REG_BINARY, (LPBYTE)szDevName, 128);
	RegSetValueEx(myKey, "GuiDev",     0, REG_BINARY, (LPBYTE)&guiDev,   sizeof(GUID));


	//
	// Recording options
	//

	if (RECORD_COMPRESSION1.cbState > sizeof(RECORD_COMPRESSION_STATE1) || RECORD_COMPRESSION1.lpState != RECORD_COMPRESSION_STATE1)
	{
		memset(&RECORD_COMPRESSION1,       0, sizeof(RECORD_COMPRESSION1));
		memset(&RECORD_COMPRESSION_STATE1, 0, sizeof(RECORD_COMPRESSION_STATE1));

		RECORD_COMPRESSION1.cbSize  = sizeof(RECORD_COMPRESSION1);
		RECORD_COMPRESSION1.lpState = RECORD_COMPRESSION_STATE1;
	}

	if (RECORD_COMPRESSION2.cbState > sizeof(RECORD_COMPRESSION_STATE2) || RECORD_COMPRESSION2.lpState != RECORD_COMPRESSION_STATE2)
	{
		memset(&RECORD_COMPRESSION2,       0, sizeof(RECORD_COMPRESSION2));
		memset(&RECORD_COMPRESSION_STATE2, 0, sizeof(RECORD_COMPRESSION_STATE2));

		RECORD_COMPRESSION2.cbSize  = sizeof(RECORD_COMPRESSION2);
		RECORD_COMPRESSION2.lpState = RECORD_COMPRESSION_STATE2;
	}


#define SetDWORD(xa, xb)  RegSetValueEx(myKey, xa, 0, REG_DWORD,  (LPBYTE)&xb, sizeof(xb));
#define SetBINARY(xa, xb) RegSetValueEx(myKey, xa, 0, REG_BINARY, (LPBYTE)&xb, sizeof(xb));

	SetDWORD("RecordingMode",            RECORD_RECORDING_MODE);
	SetDWORD("RecordingVideoSize",       RECORD_VIDEO_SIZE);
	SetDWORD("RecordingWidth",           RECORD_RECORDING_WIDTH);
	SetDWORD("RecordingHeight",          RECORD_RECORDING_HEIGHT);
	SetDWORD("RecordingFrameRateScale",  RECORD_FRAME_RATE_SCALE);
	SetDWORD("RecordingCompressionMode", RECORD_COMPRESSION_MODE);

	SetBINARY("RecordingCompression1",      RECORD_COMPRESSION1);
	SetBINARY("RecordingCompressionState1", RECORD_COMPRESSION_STATE1);
	SetBINARY("RecordingCompression2",      RECORD_COMPRESSION2);
	SetBINARY("RecordingCompressionState2", RECORD_COMPRESSION_STATE2);

	RegCloseKey(myKey);
}

////////////////////////////////////////////////////////////////////////

HWND gHWND;

static HRESULT WINAPI Enum3DDevicesCallback( GUID* pGUID, LPSTR strDesc,
                                             LPSTR strName, LPD3DDEVICEDESC pHALDesc,
                                             LPD3DDEVICEDESC pHELDesc, LPVOID pvContext )
{
    BOOL IsHardware;

    // Check params
    if ( NULL==pGUID || NULL==pHALDesc || NULL==pHELDesc)
        return D3DENUMRET_CANCEL;

    // Handle specific device GUIDs. NullDevice renders nothing
    if ( IsEqualGUID( pGUID, &IID_IDirect3DNullDevice ) )
        return D3DENUMRET_OK;

    IsHardware = ( 0 != pHALDesc->dwFlags );
    if (!IsHardware) return D3DENUMRET_OK;

    bDeviceOK=TRUE;

    return D3DENUMRET_OK;
}

static BOOL WINAPI DirectDrawEnumCallbackEx( GUID FAR* pGUID, LPSTR strDesc,
                                             LPSTR strName, VOID* pV,
                                             HMONITOR hMonitor )
{
    // Use the GUID to create the DirectDraw object, so that information
    // can be extracted from it.

    LPDIRECTDRAW pDD;
    LPDIRECTDRAW4 g_pDD;
    LPDIRECT3D3 pD3D;

    if ( FAILED( DirectDrawCreate( pGUID, &pDD, 0L ) ) )
    {
        return D3DENUMRET_OK;
    }

    // Query the DirectDraw driver for access to Direct3D.
    if ( FAILED(IDirectDraw_QueryInterface(pDD, &IID_IDirectDraw4, (VOID**)&g_pDD)))
    {
        IDirectDraw_Release(pDD);
        return D3DENUMRET_OK;
    }
    IDirectDraw_Release(pDD);

    // Query the DirectDraw driver for access to Direct3D.

    if ( FAILED( IDirectDraw4_QueryInterface(g_pDD,&IID_IDirect3D3, (VOID**)&pD3D)))
    {
        IDirectDraw4_Release(g_pDD);
        return D3DENUMRET_OK;
    }

    bDeviceOK=FALSE;

    // Now, enumerate all the 3D devices
    IDirect3D3_EnumDevices(pD3D,Enum3DDevicesCallback,NULL);

    if (bDeviceOK)
    {
        HWND hWC=GetDlgItem(gHWND,IDC_DEVICE);
        int i=ComboBox_AddString(hWC,strDesc);
        GUID* g=(GUID*)malloc(sizeof(GUID));
        if (NULL != pGUID) *g=*pGUID;
        else              memset(g,0,sizeof(GUID));
        ComboBox_SetItemData(hWC,i,g);
    }

    IDirect3D3_Release(pD3D);
    IDirectDraw4_Release(g_pDD);
    return DDENUMRET_OK;
}

//-----------------------------------------------------------------------------

static BOOL WINAPI DirectDrawEnumCallback( GUID FAR* pGUID, LPSTR strDesc,
                                           LPSTR strName, VOID* pV)
{
    return DirectDrawEnumCallbackEx( pGUID, strDesc, strName, NULL, NULL );
}

//-----------------------------------------------------------------------------

void DoDevEnum(HWND hW)
{
    LPDIRECTDRAWENUMERATEEX pDDrawEnumFn;

    HMODULE hDDrawDLL = GetModuleHandle("DDRAW.DLL");
    if (NULL == hDDrawDLL) return;

    gHWND=hW;

    pDDrawEnumFn = (LPDIRECTDRAWENUMERATEEX)
                   GetProcAddress( hDDrawDLL, "DirectDrawEnumerateExA" );

    if (pDDrawEnumFn)
        pDDrawEnumFn( DirectDrawEnumCallbackEx, NULL,
                      DDENUM_ATTACHEDSECONDARYDEVICES |
                      DDENUM_DETACHEDSECONDARYDEVICES |
                      DDENUM_NONDISPLAYDEVICES );
    else
        DirectDrawEnumerate( DirectDrawEnumCallback, NULL );
}

////////////////////////////////////////////////////////////////////////

void FreeGui(HWND hW)
{
    int i,iCnt;
    HWND hWC=GetDlgItem(hW,IDC_DEVICE);
    iCnt=ComboBox_GetCount(hWC);
    for (i=0; i<iCnt; i++)
    {
        free((GUID*)ComboBox_GetItemData(hWC,i));
    }
}

////////////////////////////////////////////////////////////////////////

BOOL CALLBACK DeviceDlgProc(HWND hW, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
    switch (uMsg)
    {
        case WM_INITDIALOG:
        {
            HWND hWC;
            int i;
            DoDevEnum(hW);
            hWC=GetDlgItem(hW,IDC_DEVICE);
            i=ComboBox_FindStringExact(hWC,-1,szDevName);
            if (i==CB_ERR) i=0;
            ComboBox_SetCurSel(hWC,i);
            hWC=GetDlgItem(hW,IDC_GAMMA);
            ScrollBar_SetRange(hWC,0,1024,FALSE);
            if (iUseGammaVal==2048) ScrollBar_SetPos(hWC,512,FALSE);
            else
            {
                ScrollBar_SetPos(hWC,iUseGammaVal,FALSE);
                CheckDlgButton(hW,IDC_USEGAMMA,TRUE);
            }
        }

        case WM_HSCROLL:
        {
            HWND hWC=GetDlgItem(hW,IDC_GAMMA);
            int pos=ScrollBar_GetPos(hWC);
            switch (LOWORD(wParam))
            {
                case SB_THUMBPOSITION:
                    pos=HIWORD(wParam);
                    break;
                case SB_LEFT:
                    pos=0;
                    break;
                case SB_RIGHT:
                    pos=1024;
                    break;
                case SB_LINELEFT:
                    pos-=16;
                    break;
                case SB_LINERIGHT:
                    pos+=16;
                    break;
                case SB_PAGELEFT:
                    pos-=128;
                    break;
                case SB_PAGERIGHT:
                    pos+=128;
                    break;

            }
            ScrollBar_SetPos(hWC,pos,TRUE);
            return TRUE;
        }

        case WM_COMMAND:
        {
            switch (LOWORD(wParam))
            {
                case IDCANCEL:
                    FreeGui(hW);
                    EndDialog(hW,FALSE);
                    return TRUE;
                case IDOK:
                {
                    HWND hWC=GetDlgItem(hW,IDC_DEVICE);
                    int i=ComboBox_GetCurSel(hWC);
                    if (i==CB_ERR) return TRUE;
                    guiDev=*((GUID*)ComboBox_GetItemData(hWC,i));
                    ComboBox_GetLBText(hWC,i,szDevName);
                    FreeGui(hW);

                    if (!IsDlgButtonChecked(hW,IDC_USEGAMMA))
                        iUseGammaVal=2048;
                    else
                        iUseGammaVal=ScrollBar_GetPos(GetDlgItem(hW,IDC_GAMMA));

                    EndDialog(hW,TRUE);
                    return TRUE;
                }
            }
        }
    }
    return FALSE;
}

////////////////////////////////////////////////////////////////////////

void SelectDev(HWND hW)
{
    if (DialogBox(hInst,MAKEINTRESOURCE(IDD_DEVICE),
                  hW,(DLGPROC)DeviceDlgProc)==IDOK)
    {
        SetDlgItemText(hW,IDC_DEVICETXT,szDevName);
    }
}


////////////////////////////////////////////////////////////////////////

static HRESULT WINAPI EnumDisplayModesCallback( DDSURFACEDESC2* pddsd,
                                                VOID* pvContext )
{
    if (NULL==pddsd) return DDENUMRET_CANCEL;

    if (pddsd->ddpfPixelFormat.dwRGBBitCount==(unsigned int)iColDepth &&
        pddsd->dwWidth==(unsigned int)iResX &&
        pddsd->dwHeight==(unsigned int)iResY)
    {
        bDeviceOK=TRUE;
        return DDENUMRET_CANCEL;
    }

    return DDENUMRET_OK;
}

////////////////////////////////////////////////////////////////////////

BOOL bTestModes(void)
{
    LPDIRECTDRAW pDD;
    LPDIRECTDRAW4 g_pDD;

    GUID FAR* guid=0;
    int i;
    unsigned char* c=(unsigned char*)&guiDev;
    for (i=0; i<sizeof(GUID); i++,c++)
    {if (*c) {guid=&guiDev; break;}}

    bDeviceOK=FALSE;

    if ( FAILED( DirectDrawCreate(guid, &pDD, 0L ) ) )
        return FALSE;

    if (FAILED(IDirectDraw_QueryInterface(pDD, &IID_IDirectDraw4, (VOID**)&g_pDD)))
    {
        IDirectDraw_Release(pDD);
        return FALSE;
    }
    IDirectDraw_Release(pDD);

    IDirectDraw4_EnumDisplayModes(g_pDD,0,NULL,NULL,EnumDisplayModesCallback);

    IDirectDraw4_Release(g_pDD);

    return bDeviceOK;
}

////////////////////////////////////////////////////////////////////////
// define key dialog
////////////////////////////////////////////////////////////////////////

typedef struct KEYSETSTAG
{
    char szName[10];
    char cCode;
}
KEYSETS;

KEYSETS tMKeys[]=
{
    {"SPACE",          0x20},
    {"PRIOR",          0x21},
    {"NEXT",           0x22},
    {"END",            0x23},
    {"HOME",           0x24},
    {"LEFT",           0x25},
    {"UP",             0x26},
    {"RIGHT",          0x27},
    {"DOWN",           0x28},
    {"SELECT",         0x29},
    {"PRINT",          0x2A},
    {"EXECUTE",        0x2B},
    {"SNAPSHOT",       0x2C},
    {"INSERT",         0x2D},
    {"DELETE",         0x2E},
    {"HELP",           0x2F},
    {"NUMPAD0",        0x60},
    {"NUMPAD1",        0x61},
    {"NUMPAD2",        0x62},
    {"NUMPAD3",        0x63},
    {"NUMPAD4",        0x64},
    {"NUMPAD5",        0x65},
    {"NUMPAD6",        0x66},
    {"NUMPAD7",        0x67},
    {"NUMPAD8",        0x68},
    {"NUMPAD9",        0x69},
    {"MULTIPLY",       0x6A},
    {"ADD",            0x6B},
    {"SEPARATOR",      0x6C},
    {"SUBTRACT",       0x6D},
    {"DECIMAL",        0x6E},
    {"DIVIDE",         0x6F},
    {"F9",             VK_F9},
    {"F10",            VK_F10},
    {"F11",            VK_F11},
    {"F12",            VK_F12},
    {"",               0x00}
};

void SetGPUKey(HWND hWC,char szKey)
{
    int i,iCnt=ComboBox_GetCount(hWC);
    for (i=0; i<iCnt; i++)
    {
        if (ComboBox_GetItemData(hWC,i)==szKey) break;
    }
    if (i!=iCnt) ComboBox_SetCurSel(hWC,i);
}

BOOL CALLBACK KeyDlgProc(HWND hW, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
    switch (uMsg)
    {
        case WM_INITDIALOG:
        {
            int i,j,k;
            char szB[2];
            HWND hWC;
            for (i=IDC_KEY1; i<=IDC_KEY10; i++)
            {
                hWC=GetDlgItem(hW,i);

                for (j=0; tMKeys[j].cCode!=0; j++)
                {
                    k=ComboBox_AddString(hWC,tMKeys[j].szName);
                    ComboBox_SetItemData(hWC,k,tMKeys[j].cCode);
                }
                for (j=0x30; j<=0x39; j++)
                {
                    wsprintf(szB,"%c",j);
                    k=ComboBox_AddString(hWC,szB);
                    ComboBox_SetItemData(hWC,k,j);
                }
                for (j=0x41; j<=0x5a; j++)
                {
                    wsprintf(szB,"%c",j);
                    k=ComboBox_AddString(hWC,szB);
                    ComboBox_SetItemData(hWC,k,j);
                }
                SetGPUKey(GetDlgItem(hW,i),szGPUKeys[i-IDC_KEY1]);
            }
        }
        return TRUE;

        case WM_COMMAND:
        {
            switch (LOWORD(wParam))
            {
                case IDC_DEFAULT:
                {
                    int i;
                    for (i=IDC_KEY1; i<=IDC_KEY10; i++)
                        SetGPUKey(GetDlgItem(hW,i),szKeyDefaults[i-IDC_KEY1]);
                }
                break;

                case IDCANCEL:
                    EndDialog(hW,FALSE);
                    return TRUE;
                case IDOK:
                {
                    HWND hWC;
                    int i;
                    for (i=IDC_KEY1; i<=IDC_KEY10; i++)
                    {
                        hWC=GetDlgItem(hW,i);
                        szGPUKeys[i-IDC_KEY1]=(char)ComboBox_GetItemData(hWC,ComboBox_GetCurSel(hWC));
                        if (szGPUKeys[i-IDC_KEY1]<0x20) szGPUKeys[i-IDC_KEY1]=0x20;
                    }
                    EndDialog(hW,TRUE);
                    return TRUE;
                }
            }
        }
    }
    return FALSE;
}

void OnKeyConfig(HWND hW)
{
    DialogBox(hInst,MAKEINTRESOURCE(IDD_KEYS),
              hW,(DLGPROC)KeyDlgProc);
}

#else

////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
// LINUX VERSION
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////
// gtk linux stuff
////////////////////////////////////////////////////////////////////////

void ExecCfg(char* arg)
{
    char cfg[256];
    struct stat buf;

    strcpy(cfg, "./cfgPeopsSoft");
    if (stat(cfg, &buf) != -1)
    {
        sprintf(cfg, "%s %s", cfg, arg);
        system(cfg);
        return;
    }

    strcpy(cfg, "./cfg/cfgPeopsSoft");
    if (stat(cfg, &buf) != -1)
    {
        sprintf(cfg, "%s %s", cfg, arg);
        system(cfg);
        return;
    }

    sprintf(cfg, "%s/cfgPeopsSoft", getenv("HOME"));
    if (stat(cfg, &buf) != -1)
    {
        sprintf(cfg, "%s %s", cfg, arg);
        system(cfg);
        return;
    }

    printf("cfgPeopsSoft file not found!\n");
}

void SoftDlgProc(void)
{
    ExecCfg("configure");
}
#ifndef _FPSE

extern unsigned char revision;
extern unsigned char build;
#define RELEASE_DATE "01.05.2008"

void AboutDlgProc(void)
{
    char args[256];

    sprintf(args, "about %d %d %s", revision, build, RELEASE_DATE);
    ExecCfg(args);
}


////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////

void ReadConfig(void)
{
    // defaults
    iResX=640;
    iResY=480;
    iWinSize=MAKELONG(iResX,iResY);
    iColDepth=16;
    iWindowMode=1;
    iUseScanLines=0;
    UseFrameLimit=0;
    UseFrameSkip=0;
    iFrameLimit=2;
    fFrameRate=200.0f;
    dwCfgFixes=0;
    iUseFixes=0;
    iUseNoStretchBlt=1;
    iUseDither=0;
    iShowFPS=0;
    bSSSPSXLimit=FALSE;

    // read sets
    ReadConfigFile();

    // additional checks
    if (!iColDepth)       iColDepth=32;
    if (iUseFixes)        dwActFixes=dwCfgFixes;
    SetFixes();
}

void WriteConfig(void)
{

    struct stat buf;
    FILE* out;
    char t[256];
    int len, size;
    char* pB, * p;
    char t1[8];

    if (pConfigFile)
        strcpy(t,pConfigFile);
    else
    {
        strcpy(t,"cfg/gpuPeopsSoftX.cfg");
        out = fopen(t,"rb");
        if (!out)
        {
            strcpy(t,"gpuPeopsSoftX.cfg");
            out = fopen(t,"rb");
            if (!out) sprintf(t,"%s/gpuPeopsSoftX.cfg",getenv("HOME"));
            else     fclose(out);
        }
        else     fclose(out);
    }

    if (stat(t, &buf) != -1) size = buf.st_size;
    else size = 0;

    out = fopen(t,"rb");
    if (!out)
    {
        // defaults
        iResX=640;
        iResY=480;
        iColDepth=16;
        iWindowMode=1;
        iUseScanLines=0;
        UseFrameLimit=0;
        UseFrameSkip=0;
        iFrameLimit=2;
        fFrameRate=200.0f;
        dwCfgFixes=0;
        iUseFixes=0;
        iUseNoStretchBlt=1;
        iUseDither=0;
        iShowFPS=0;
        bSSSPSXLimit=FALSE;

        size = 0;
        pB=(char*)malloc(4096);
        memset(pB,0,4096);
    }
    else
    {
        pB=(char*)malloc(size+4096);
        memset(pB,0,size+4096);

        len = fread(pB, 1, size, out);
        fclose(out);
    }

    SetValue("ResX", iResX);
    SetValue("ResY", iResY);
    SetValue("NoStretch", iUseNoStretchBlt);
    SetValue("Dithering", iUseDither);
    SetValue("FullScreen", !iWindowMode);
    SetValue("ShowFPS", iShowFPS);
    SetValue("SSSPSXLimit", bSSSPSXLimit);
    SetValue("ScanLines", iUseScanLines);
    SetValue("UseFrameLimit", UseFrameLimit);
    SetValue("UseFrameSkip", UseFrameSkip);
    SetValue("FPSDetection", iFrameLimit);
    SetFloatValue("FrameRate", fFrameRate);
    SetValue("CfgFixes", (unsigned int)dwCfgFixes);
    SetValue("UseFixes", iUseFixes);

    out = fopen(t,"wb");
    if (!out) return;

    len = fwrite(pB, 1, size, out);
    fclose(out);

    free(pB);

}
#endif
#endif // LINUX_VERSION
