package debugger;

// Java imports
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;

// Project imports
import debugger.*;
import util.*;
import vue.*;

// One line of hex editor output
class CPUHexLine extends JPanel {

    // Instance fields
    private int      address; // Address of first byte on line
    private Debugger parent;  // Debugger UI manager

    // UI components
    private JLabel   lblAddress; // Line address
    private JLabel[] lblValue;   // Byte values



    ///////////////////////////////////////////////////////////////////////////
    //                             Constructors                              //
    ///////////////////////////////////////////////////////////////////////////

    // Default constructor
    CPUHexLine(Debugger parent) {
        super(null);

        // Configure instance fields
        address     = 0;
        this.parent = parent;

        // Configure address label
        lblAddress = new JLabel();
        lblAddress.setForeground(Util.getColor("windowText"));
        lblAddress.setOpaque(false);
        add(lblAddress);

        // Configure value labels
        lblValue = new JLabel[16];
        for (int x = 0; x < 16; x++) {
            JLabel label = lblValue[x] = new JLabel();
            label.putClientProperty("index", x);
            label.setOpaque(false);
            add(label);
        }

        // Configure component
        setOpaque(false);
        setFonts();

        // Configure event handlers
        addMouseListener(new MouseListener() {
            public void mouseClicked (MouseEvent e) { }
            public void mouseEntered (MouseEvent e) { }
            public void mouseExited  (MouseEvent e) { }
            public void mousePressed (MouseEvent e) { onMouseDown(e); }
            public void mouseReleased(MouseEvent e) { }
        });
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Package Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Change appearance when the current cursor position
    void configure(int cursor) {
        boolean focus        = parent.hexEditorHasFocus();
        Color   windowText   = Util.getColor("windowText");
        Color   selected     = Util.getColor("selected");
        Color   selectedText = Util.getColor("selectedText");
        Color   selectedBlur = Util.getColor("selected", 0.4f);

        // Configure byte labels
        for (int x = 0; x < 16; x++) {
            JLabel label = lblValue[x];

            if (x == cursor) {
                label.setBackground(focus ? selected : selectedBlur);
                label.setForeground(focus ? selectedText : windowText);
                label.setOpaque(true);
            }

            else {
                label.setForeground(windowText);
                label.setOpaque(false);
            }
        }

    }

    // Determine whether a line contains a given address
    boolean contains(int address) {
        return this.address == (address & -16);
    }

    // Retrieve the address of the current line
    int getAddress() {
        return address;
    }

    // Configure label positions
    void layout(int spacing) {
        int hexWidth   = parent.getHexWidth() * 2;
        int lineHeight = parent.getLineHeight();

        // Position address label
        lblAddress.setBounds(0, 0, hexWidth * 4, lineHeight);

        // Position byte values
        int left = hexWidth * 4 + spacing;
        for (int x = 0; x < 16; x++) {
            if ((x & 7) == 0) left += spacing;
            lblValue[x].setBounds(left, 0, hexWidth, lineHeight);
            left += hexWidth + spacing;
        }

    }

    // Specify the address and data of the line
    void setData(int address, byte[] data, int offset) {
        boolean hexCase = parent.getHexCase();

        // Configure instance fields
        this.address = address;

        // Configure UI components
        lblAddress.setText(String.format(hexCase ? "%08X" : "%08x", address));
        for (int x = 0; x < lblValue.length; x++)
            lblValue[x].setText(String.format(hexCase ? "%02X" : "%02x",
                (int) data[offset + x] & 0xFF));
    }

    // Specify new fonts
    void setFonts() {

        // Fetch fonts
        Font dialog = parent.getDialogFont();
        Font hex    = parent.getHexFont();

        // Update componts
        lblAddress.setFont(hex);
        for (int x = 0; x < 16; x++)
            lblValue[x].setFont(hex);
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Event Handlers                             //
    ///////////////////////////////////////////////////////////////////////////

    // Mouse press
    private void onMouseDown(MouseEvent e) {
        int button = e.getButton();

        // Only consider left clicks
        if (button != MouseEvent.BUTTON1)
            return;

        // Give focus to the hex editor
        parent.getHexEditor().requestFocus();

        // Retrieve font properties
        int hexWidth = parent.getHexWidth();
        int size     = lblValue[1].getX() - lblValue[0].getX();
        if (size == 0) return;

        // Prepare working variables
        int spacing = size - hexWidth * 2;
        int width   = hexWidth * 16 + spacing * 8;
        int left    = hexWidth *  8 + spacing * 3 / 2;
        int right   = left + width + spacing;
        int x       = e.getX();

        // Select the nearest byte clicked on
        int index = -1;
        if (x >= left && x < left + width)        // Left column
            index = (x - left) / size;
        else if (x >= right && x < right + width) // Right column
            index = 8 + (x - right) / size;
        if (index != -1)
            parent.seekHexEditor(address + index);
    }

}
