package debugger;

// Java imports
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;

// Project imports
import debugger.*;
import util.*;
import vue.*;

// One register item in a register list
class CPURegister {

    // Instance fields
    private boolean  canExpand; // Expansion is allowed
    private int      index;     // The register's index
    private Debugger parent;    // Debugger UI manager
    private int      set;       // The register set containing the register

    // UI components
    private JLabel         btnExpand; // Expand button
    private JPanel         expansion; // Expansion controls
    private ActionListener expAction; // Event listener for expansion controls
    private FocusListener  expFocus;  // Event listener for expansion controls
    private JLabel         lblName;   // Register name
    private JPanel         panSpacer; // Spacing for layout purposes
    private JTextField     txtValue;  // Value text box
    private JComponent[]   controls;  // Expansion controls



    ///////////////////////////////////////////////////////////////////////////
    //                               Constants                               //
    ///////////////////////////////////////////////////////////////////////////

    // Special registers
    static final int PC = -1;

    // Expand button text
    static final String COLLAPSE = "-";
    static final String EXPAND   = "+";

    // Presentation formats
    private static final int HEX      = 0;
    private static final int SIGNED   = 1;
    private static final int UNSIGNED = 2;
    private static final int FLOAT    = 3;

    // Program register names
    private static final String[] REGNAMES = {
        "r0",  "r1",  "hp",  "sp",  "gp",  "tp",  "r6",  "r7",
        "r8",  "r9",  "r10", "r11", "r12", "r13", "r14", "r15",
        "r16", "r17", "r18", "r19", "r20", "r21", "r22", "r23",
        "r24", "r25", "r26", "r27", "r28", "r29", "r30", "lp"
    };

    // System register names
    private static final String[] SYSREGNAMES = {
        "EIPC", "EIPSW", "FEPC", "FEPSW", "ECR", "PSW", "PIR", "TKCW",
        null,   null,    null,   null,    null,   null, null,  null,
        null,   null,    null,   null,    null,   null, null,  null,
        "CHCW", "ADTRE", null,   null,    null,   "29", "30",  "31"
    };



    ///////////////////////////////////////////////////////////////////////////
    //                             Constructors                              //
    ///////////////////////////////////////////////////////////////////////////

    // Default constructor
    CPURegister(Debugger parent, int set, int index, JPanel client) {

        // Configure instance fields
        canExpand   = false;
        this.index  = index;
        this.parent = parent;
        this.set    = set;

        // Configure UI
        configUI(client);

        // Configure event handlers
        MouseListener mouse = new MouseListener() {
            public void mouseClicked (MouseEvent e) { }
            public void mouseEntered (MouseEvent e) { }
            public void mouseExited  (MouseEvent e) { }
            public void mousePressed (MouseEvent e) { onExpand(e); }
            public void mouseReleased(MouseEvent e) { }
        };
        btnExpand.addMouseListener(mouse);
        lblName.addMouseListener(mouse);
        txtValue.addActionListener(e->client.requestFocus());
        txtValue.addFocusListener(new FocusListener() {
            public void focusGained(FocusEvent e) { onFocus(e); }
            public void focusLost  (FocusEvent e) { onValue(); }
        });
        expAction = e->{ client.requestFocus(); onExpansion(); };
        expFocus  = new FocusListener() {
            public void focusGained(FocusEvent e) { onFocus(e); }
            public void focusLost  (FocusEvent e) { onExpansion(); }
        };

        // Configure expansion area
        if (set == CPURegisterList.SYSTEM) switch (index) {
            case VUE.PSW: case VUE.EIPSW: case VUE.FEPSW:
                           configPSW();  break;
            case VUE.CHCW: configCHCW(); break;
            case VUE.ECR:  configECR();  break;
            case VUE.PIR:  configPIR();  break;
            case VUE.TKCW: configTKCW(); break;
        } else configProgram();

        // Populate controls
        refresh();
    }

    // Configure UI components
    private void configUI(JComponent client) {
        GridBagConstraints c = new GridBagConstraints();

        // Configure expand button
        btnExpand = new JLabel();
        btnExpand.setForeground(Util.getColor("windowText"));
        btnExpand.setHorizontalAlignment(SwingConstants.CENTER);
        c.fill = GridBagConstraints.BOTH;
        client.add(btnExpand, c);

        // Configure name label
        lblName = new JLabel(index == PC ? "PC" :
            set == CPURegisterList.SYSTEM?SYSREGNAMES[index]:REGNAMES[index]);
        lblName.setForeground(Util.getColor("windowText"));
        c.insets    = new Insets(0, 4, 0, 0);
        c.weightx = 1;
        client.add(lblName, c);

        // Configure value text box
        txtValue = new JTextField();
        txtValue.putClientProperty("hex", 8);
        txtValue.setBorder(null);
        txtValue.setForeground(Util.getColor("windowText"));
        txtValue.setOpaque(false);
        c.fill      = GridBagConstraints.NONE;
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.weightx   = 0;
        client.add(txtValue, c);

        // Spacing panel
        panSpacer = new JPanel(null);
        //panSpacer.setBackground(Color.red);
        panSpacer.setPreferredSize(new Dimension(0, 0));
        panSpacer.setOpaque(false);
        panSpacer.setVisible(false);
        c.gridwidth = 1;
        c.insets    = new Insets(0, 0, 0, 0);
        c.fill      = GridBagConstraints.BOTH;
        client.add(panSpacer, c);

        // Configure expansion pane
        expansion = new JPanel(null);
        expansion.setOpaque(false);
        expansion.setVisible(false);
        c.fill      = GridBagConstraints.BOTH;
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.insets    = new Insets(0, 8, 0, 0);
        client.add(expansion, c);
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Package Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Update controls with the current register value
    void refresh() {
        int value = getValue();

        // update value text box
        int    format = getFormat();
        String text   = null;
        switch (format) {
            case SIGNED:   text = "" + value;                        break;
            case UNSIGNED: text = "" + ((long) value & 0xFFFFFFFFL); break;
            case FLOAT:    text = "" + Float.intBitsToFloat(value);  break;
            case HEX:      text = String.format(
                parent.getHexCase() ? "%08X" : "%08x", value);       break;
        }
        txtValue.setText(text);
        refreshTextBox(txtValue, false, 8);

        // Update expansion controls
        if (set == CPURegisterList.SYSTEM) switch (index) {
            case VUE.PSW: case VUE.EIPSW: case VUE.FEPSW:
                           parsePSW (value); break;
            case VUE.CHCW: parseCHCW(value); break;
            case VUE.ECR:  parseECR (value); break;
        }

    }

    // Specify whether the control is expanded
    void setExpanded(boolean expanded) {
        if (!canExpand) return;
        btnExpand.setText(expanded ? COLLAPSE : EXPAND);
        expansion.setVisible(expanded);
        panSpacer.setVisible(expanded);
    }

    // Specify new fonts
    void setFonts() {

        // Update main controls
        Font dlgFont = parent.getDialogFont();
        Font hexFont = parent.getHexFont();
        btnExpand.setFont(dlgFont);
        lblName.setFont(dlgFont);
        refreshTextBox(txtValue, true, 8);

        // Update expansion controls
        for (int x = 0; controls != null && x < controls.length; x++) {
            JComponent control = controls[x];
            Font font =
                set   == CPURegisterList.SYSTEM &&
                index == VUE.PIR                &&
                x     == TXTPT
            ? hexFont : dlgFont;
            if (control instanceof JTextField)
                refreshTextBox(control, true, 0);
            else refreshExpansion(control, font);
        }

    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Event Handlers                             //
    ///////////////////////////////////////////////////////////////////////////

    // Expand button
    private void onExpand(MouseEvent e) {
        expansion.getParent().requestFocus();

        // Determine whether to toggle expansion
        if (
            !canExpand                          ||
            e.getButton() != MouseEvent.BUTTON1 ||
            e.getY()      >= parent.getLineHeight()
        ) return;

        // Toggle the expanded state
        setExpanded(!expansion.isVisible());
    }

    // An expansion control was modified
    private void onExpansion() {

        // Processing for system registers
        if (set == CPURegisterList.SYSTEM) switch (index) {
            case VUE.PSW: case VUE.EIPSW: case VUE.FEPSW:
                           setValue(composePSW ()); return;
            case VUE.CHCW: setValue(composeCHCW()); return;
            case VUE.ECR:  setValue(composeECR ()); return;
            default: return;
        }

        // Processing for program registers
        if (getFormat() == HEX) {
            txtValue.putClientProperty("hex", 8);
            txtValue.setFont(parent.getHexFont());
        } else {
            txtValue.putClientProperty("hex", null);
            txtValue.setFont(parent.getDialogFont());
        }
        refresh();
    }

    // Text box receives focus
    private void onFocus(FocusEvent e) {
        //((JTextField) e.getSource()).selectAll();
    }

    // Value text box
    private void onValue() {
        int value = getValue();

        // Parse the value entered
        String text = txtValue.getText();
        try { switch (getFormat()) {
            case HEX:      value = (int) Long.parseLong(text, 16); break;
            case SIGNED:   value = Integer.parseInt(text);         break;
            case UNSIGNED: value = (int) Long.parseLong(text);     break;
            case FLOAT:
                value = Float.floatToIntBits(Float.parseFloat(text));
                break;
        }} catch (Exception e) { }

        setValue(value);
    }



    ///////////////////////////////////////////////////////////////////////////
    //                       Program Register Methods                        //
    ///////////////////////////////////////////////////////////////////////////

    // Control indexes
    private static final int OPTHEX      = 0;
    private static final int OPTSIGNED   = 1;
    private static final int OPTUNSIGNED = 2;
    private static final int OPTFLOAT    = 3;

    // Control names
    private static final String[] GR_NAMES =
        { "Hex", "Signed", "Unsigned", "Float" };

    // Constructor
    private void configProgram() {
        canExpand = true;
        btnExpand.setText(EXPAND);

        // Configure layout
        expansion.setLayout(new GridBagLayout());
        ButtonGroup group = new ButtonGroup();

        // Configure controls
        controls = new JComponent[4];
        for (int x = 0; x < controls.length; x++) {
            JRadioButton control = new JRadioButton(GR_NAMES[x]);
            control.addActionListener(expAction);
            controls[x] = control;
            group.add(control);
            addControl(x, null, true);
        }

        // Default selection
        ((JRadioButton) controls[OPTHEX]).setSelected(true);
    }



    ///////////////////////////////////////////////////////////////////////////
    //                             CHCW Methods                              //
    ///////////////////////////////////////////////////////////////////////////

    // Control indexes
    private static final int CHKICE = 0;

    // Constructor
    private void configCHCW() {
        canExpand = true;
        btnExpand.setText(EXPAND);
        controls  = new JComponent[1];

        // Configure layout
        expansion.setLayout(new GridBagLayout());

        // Configure controls
        controls[CHKICE] = new JCheckBox("ICE");
        ((JCheckBox) controls[CHKICE]).addActionListener(expAction);
        addControl(CHKICE, null, true);
    }

    // Compose register value from controls
    private int composeCHCW() {
        return getChecked(CHKICE) ? 2 : 0;
    }

    // Configure controls from value bits
    private void parseCHCW(int value) {
        setChecked(CHKICE, (value & 2) != 0);
    }



    ///////////////////////////////////////////////////////////////////////////
    //                              ECR Methods                              //
    ///////////////////////////////////////////////////////////////////////////

    // Control indexes
    private static final int LBLEICC = 0;
    private static final int TXTEICC = 1;
    private static final int LBLFECC = 2;
    private static final int TXTFECC = 3;

    // Control names
    private static final String[] ECR_NAMES = { "EICC", "FECC" };

    // Constructor
    private void configECR() {
        canExpand = true;
        btnExpand.setText(EXPAND);

        // Configure layout
        expansion.setLayout(new GridBagLayout());
        Insets s4 = new Insets(0, 0, 0, 4);

        // Configure controls
        controls  = new JComponent[4];
        for (int x = 0; x < controls.length; x++) {

            // Label
            if ((x & 1) == 0) {
                controls[x] = new JLabel(ECR_NAMES[x / 2]);
                addControl(x, s4, false);
            }

            // Text box
            else {
                controls[x] = new JTextField();
                configTextBox(x, 4, 0);
                addControl(x, null, true);
            }
        }

    }

    // Compose register value from controls
    private int composeECR() {
        VUE vue   = parent.getVUE();
        int value = vue.getSystemRegister(index);

        int eicc = value & 0xFFFF;
        try {
            eicc = Integer.parseInt(getText(TXTEICC), 16);
            if ((eicc & 0xFFFF) != eicc) throw new RuntimeException();
        } catch (Exception e) { }

        int fecc = value >> 16 & 0xFFFF;
        try {
            fecc = Integer.parseInt(getText(TXTFECC), 16);
            if ((fecc & 0xFFFF) != fecc) throw new RuntimeException();
        } catch (Exception e) { }

        value = fecc << 16 | eicc;
        return value;
    }

    // Configure controls from value bits
    private void parseECR(int value) {
        String code = parent.getHexCase() ? "%04X" : "%04x";
        setText(TXTEICC, String.format(code, value       & 0xFFFF));
        setText(TXTFECC, String.format(code, value >> 16 & 0xFFFF));
    }



    ///////////////////////////////////////////////////////////////////////////
    //                              PSW Methods                              //
    ///////////////////////////////////////////////////////////////////////////

    // Control indexes
    private static final int CHKZ   =  0;
    private static final int CHKFRO =  1;
    private static final int CHKS   =  2;
    private static final int CHKFIV =  3;
    private static final int CHKOV  =  4;
    private static final int CHKFZD =  5;
    private static final int CHKCY  =  6;
    private static final int CHKFOV =  7;
    private static final int CHKAE  =  8;
    private static final int CHKFUD =  9;
    private static final int CHKEP  = 10;
    private static final int CHKFPR = 11;
    private static final int CHKNP  = 12;
    private static final int CHKID  = 13;
    private static final int LBLI   = 14;
    private static final int TXTI   = 15;

    // Control names
    private static final String[] PSW_NAMES = {
        "Z",  "FRO", "S",  "FIV", "OV", "FZD", "CY", "FOV",
        "AE", "FUD", "EP", "FPR", "NP", "ID", "I"
    };

    // Constructor
    private void configPSW() {
        canExpand = true;
        btnExpand.setText(EXPAND);
        controls  = new JComponent[16];

        // Configure layout
        expansion.setLayout(new GridBagLayout());
        Insets s4 = new Insets(0, 0, 0, 4);
        Insets s8 = new Insets(0, 0, 0, 8);

        // Configure controls
        for (int x = 0; x < controls.length; x++) {

            // I label
            if (x == LBLI) {
                controls[x] = new JLabel(PSW_NAMES[x]);
                addControl(x, s4, false);
            }

            // I text box
            else if (x == TXTI) {
                controls[x] = new JTextField();
                configTextBox(x, 0, 1.5f);
                addControl(x, null, true);
            }

            // Check box
            else {
                controls[x] = new JCheckBox(PSW_NAMES[x]);
                ((JCheckBox) controls[x]).addActionListener(expAction);
                if (((x & 1) == 0 || x == CHKID) && x != CHKNP)
                    addControl(x, s8, false);
                else addControl(x, null, true);
            }
        }

    }

    // Compose register value from controls
    private int composePSW() {
        VUE vue = parent.getVUE();

        int i = vue.getSystemRegister(index) >> 16 & 0xF;
        try {
            i = Integer.parseInt(getText(TXTI));
            if ((i & 0xF) != i) throw new RuntimeException();
        } catch (Exception e) { }

        int value = i << 16;
        if (getChecked(CHKZ  )) value |= 0x00000001;
        if (getChecked(CHKS  )) value |= 0x00000002;
        if (getChecked(CHKOV )) value |= 0x00000004;
        if (getChecked(CHKCY )) value |= 0x00000008;
        if (getChecked(CHKFPR)) value |= 0x00000010;
        if (getChecked(CHKFUD)) value |= 0x00000020;
        if (getChecked(CHKFOV)) value |= 0x00000040;
        if (getChecked(CHKFZD)) value |= 0x00000080;
        if (getChecked(CHKFIV)) value |= 0x00000100;
        if (getChecked(CHKFRO)) value |= 0x00000200;
        if (getChecked(CHKID )) value |= 0x00001000;
        if (getChecked(CHKAE )) value |= 0x00002000;
        if (getChecked(CHKEP )) value |= 0x00004000;
        if (getChecked(CHKNP )) value |= 0x00008000;
        return value;
    }

    // Configure controls from value bits
    private void parsePSW(int value) {
        setText(TXTI, String.format("%d", value >> 16 & 0xF));
        setChecked(CHKZ,   (value & 0x00000001) != 0);
        setChecked(CHKS,   (value & 0x00000002) != 0);
        setChecked(CHKOV,  (value & 0x00000004) != 0);
        setChecked(CHKCY,  (value & 0x00000008) != 0);
        setChecked(CHKFPR, (value & 0x00000010) != 0);
        setChecked(CHKFUD, (value & 0x00000020) != 0);
        setChecked(CHKFOV, (value & 0x00000040) != 0);
        setChecked(CHKFZD, (value & 0x00000080) != 0);
        setChecked(CHKFIV, (value & 0x00000100) != 0);
        setChecked(CHKFRO, (value & 0x00000200) != 0);
        setChecked(CHKID,  (value & 0x00001000) != 0);
        setChecked(CHKAE,  (value & 0x00002000) != 0);
        setChecked(CHKEP,  (value & 0x00004000) != 0);
        setChecked(CHKNP,  (value & 0x00008000) != 0);
    }



    ///////////////////////////////////////////////////////////////////////////
    //                              PIR Methods                              //
    ///////////////////////////////////////////////////////////////////////////

    // Control indexes
    private static final int LBLPT = 0;
    private static final int TXTPT = 1;

    // Constructor
    private void configPIR() {
        canExpand = true;
        btnExpand.setText(EXPAND);
        controls  = new JComponent[2];

        // Configure layout
        expansion.setLayout(new GridBagLayout());
        Insets s4 = new Insets(0, 0, 0, 4);

        // Configure controls
        controls[LBLPT] = new JLabel("PT");
        addControl(LBLPT, s4, false);
        controls[TXTPT] = new JLabel("5346");
        addControl(TXTPT, null, true);
    }



    ///////////////////////////////////////////////////////////////////////////
    //                              TKCW Methods                             //
    ///////////////////////////////////////////////////////////////////////////

    // Control indexes
    private static final int CHKOTM = 0;
    private static final int CHKFVT = 1;
    private static final int CHKFIT = 2;
    private static final int CHKFUT = 3;
    private static final int CHKFZT = 4;
    private static final int CHKFPT = 5;
    private static final int CHKRDI = 6;
    private static final int LBLRD  = 7;
    private static final int TXTRD  = 8;

    // Control names
    private static final String[] TKCW_NAMES =
        { "OTM", "FVT", "FIT", "FUT", "FZT", "FPT", "RDI", "RD", "0" };

    // Constructor
    private void configTKCW() {
        canExpand = true;
        btnExpand.setText(EXPAND);
        controls  = new JComponent[9];

        // Configure layout
        expansion.setLayout(new GridBagLayout());
        Insets s4 = new Insets(0, 0, 0, 4);
        Insets s8 = new Insets(0, 0, 0, 8);

        // Configure controls
        for (int x = 0; x < controls.length; x++) {

            // RD label
            if (x == LBLRD) {
                controls[x] = new JLabel(TKCW_NAMES[x]);
                addControl(x, s4, false);
            }

            // RD "text box"
            else if (x == TXTRD) {
                controls[x] = new JLabel(TKCW_NAMES[x]);
                addControl(x, null, true);
            }

            // Check box
            else {
                controls[x] = new JCheckBox(TKCW_NAMES[x]);
                controls[x].setEnabled(false);
                if (x < LBLRD && (x & 1) == 0)
                    addControl(x, s8, false);
                else addControl(x, null, true);
            }
        }

        // Fixed flag values
        setChecked(CHKFIT, true);
        setChecked(CHKFZT, true);
        setChecked(CHKFVT, true);
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Private Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Add an expansion control to the list and expansion area
    private void addControl(int index, Insets insets, boolean endOfRow) {
        JComponent control = controls[index];

        // Configure the control with common settings
        control.setOpaque(false);
        control.setForeground(Util.getColor("windowText"));
        if (control instanceof JCheckBox || control instanceof JRadioButton) {
            control.setBorder(null);
            control.setFocusable(false);
        }

        // Add the control to the expansion area
        GridBagConstraints c = new GridBagConstraints();
        c.anchor    = GridBagConstraints.LINE_START;
        c.gridwidth = endOfRow ? GridBagConstraints.REMAINDER : 1;
        c.insets    = insets == null ? c.insets : insets;
        c.weightx   = endOfRow ? 1.0 : 0.0;
        expansion.add(control, c);
    }

    // Configure an expansion text box
    private void configTextBox(int index, int hex, float width) {
        JTextField control = (JTextField) controls[index];
        control.addActionListener(expAction);
        control.addFocusListener(expFocus);
        control.setBorder(null);
        if (hex != 0)   control.putClientProperty("hex",   hex);
        if (width != 0) control.putClientProperty("width", width);
    }

    // Determine whether an expansion check box control is checked
    private boolean getChecked(int index) {
        return ((JCheckBox) controls[index]).isSelected();
    }

    // Determine the numeric format of a program register
    private int getFormat() {
        if (set == CPURegisterList.PROGRAM) for (int x = 0; x < 4; x++)
            if (((JRadioButton) controls[x]).isSelected())
                return x;
        return HEX;
    }

    // Retrieve the text from an expansion text box control
    private String getText(int index) {
        return ((JTextField) controls[index]).getText();
    }

    // Retrieve the current value of the register
    private int getValue() {
        VUE vue = parent.getVUE();
        return
            index == PC                     ? vue.getProgramCounter ()      :
            set   == CPURegisterList.SYSTEM ? vue.getSystemRegister (index) :
                                              vue.getProgramRegister(index)
        ;
    }

    // Configure the font and size of an expansion control
    private void refreshExpansion(JComponent control, Font font) {
        control.setFont(font);
        Dimension size = control.getPreferredSize();
        size.height = parent.getLineHeight();
        control.setPreferredSize(size);
    }

    // Configure the font and size of a text box
    private void refreshTextBox(JComponent control, boolean updateFont,
        int minHex) {

        // Retrieve fonts
        Font dlgFont = parent.getDialogFont();
        Font hexFont = parent.getHexFont();
        if (dlgFont == null || hexFont == null) return;
        FontMetrics dlgMetrics = parent.getDialogMetrics();
        FontMetrics hexMetrics = parent.getHexMetrics();

        // Retrieve text box properties
        JTextField box   = (JTextField) control;
        Integer    hex   = (Integer)    control.getClientProperty("hex");
        Float      width = (Float)      control.getClientProperty("width");

        // Set the text box's font
        if (updateFont)
            box.setFont(hex == null ? dlgFont : hexFont);

        // Select the text box's size
        Dimension size = new Dimension();
        size.height    = parent.getLineHeight();

        // The number of hex digits is specified
        if (hex != null)
            size.width = parent.getHexWidth() * hex;

        // A width is specified
        else if (width != null)
            size.width = (int) Math.round(width * size.height);

        // No width is specified
        else size.width = dlgMetrics.stringWidth(box.getText());

        // Set the text box's size
        if (minHex != 0)
            size.width = Math.max(size.width, parent.getHexWidth() * minHex);
        size.width += 2;
        box.setPreferredSize(size);
        box.revalidate();
    }

    // Check or un-check an expansion check box control
    private void setChecked(int index, boolean checked) {
        ((JCheckBox) controls[index]).setSelected(checked);
    }

    // Specify new text for an expansion text box control
    private void setText(int index, String text) {
        ((JTextField) controls[index]).setText(text);
    }

    // Store a new value into the register
    private void setValue(int value) {
        VUE vue = parent.getVUE();

        // Write to the appropriate register
        if (index == PC)
            vue.setProgramCounter(value);
        else if (set == CPURegisterList.SYSTEM)
            vue.setSystemRegister(index, value);
        else vue.setProgramRegister(index, value);

        // Update controls
        parent.refresh(false);
    }

}
