package debugger;

// Java imports
import java.awt.*;
import java.util.*;
import javax.swing.*;

// Project imports
import util.*;
import vue.*;

// Editable list of CPU registers
class CPURegisterList extends JScrollPane {

    // Instance fields
    private Debugger parent; // Debugger UI manager
    private ArrayList<CPURegister> registers; // List elements

    // UI components
    private JPanel client; // Main content pane



    ///////////////////////////////////////////////////////////////////////////
    //                               Constants                               //
    ///////////////////////////////////////////////////////////////////////////

    // Register sets
    static final int PROGRAM = 0;
    static final int SYSTEM  = 1;

    // System register list
    static final int[] SYSREGS = {
        CPURegister.PC, VUE.PSW, VUE.EIPC, VUE.EIPSW, VUE.FEPC,  VUE.FEPSW,
        VUE.ECR,        VUE.PIR, VUE.TKCW, VUE.CHCW,  VUE.ADTRE, 29, 30, 31
    };



    ///////////////////////////////////////////////////////////////////////////
    //                             Constructors                              //
    ///////////////////////////////////////////////////////////////////////////

    // Default constructor
    CPURegisterList(Debugger parent, int set) {
        super(VERTICAL_SCROLLBAR_ALWAYS, HORIZONTAL_SCROLLBAR_AS_NEEDED);

        // Configure instance fields
        this.parent = parent;
        registers = new ArrayList<CPURegister>();

        // Configure client area
        client = new JPanel(new GridBagLayout());
        client.setBackground(Util.getColor("window"));
        client.setFocusable(true);
        setViewportView(client);

        // Configure register list
        if (set == SYSTEM) for (int x = 0; x < SYSREGS.length; x++)
            registers.add(new CPURegister(parent, set, SYSREGS[x], client));
        else for (int x = 0; x < 32; x++)
            registers.add(new CPURegister(parent, set, x, client));

        // Padding at the bottom of the register list
        JPanel spacer = new JPanel();
        spacer.setOpaque(false);
        spacer.setPreferredSize(new Dimension(0, 0));
        GridBagConstraints c = new GridBagConstraints();
        c.fill      = GridBagConstraints.BOTH;
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.weighty   = 1;
        client.add(spacer, c);
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Package Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Update UI components with the current emulation state
    void refresh() {
        for (int x = 0; x < registers.size(); x++)
            registers.get(x).refresh();
    }

    // Specify whether a given register is expanded
    void setExpanded(int index, boolean expanded) {
        if (index >= 0 && index < registers.size())
            registers.get(index).setExpanded(expanded);
        
    }

    // Specify new fonts and record the widest register name
    void setFonts() {

        // Update registers
        for (int x = 0; x < registers.size(); x++) {
            CPURegister reg = registers.get(x);
            reg.setFonts();
        }

        // Configure scroll bars
        FontMetrics dialog = parent.getDialogMetrics();
        FontMetrics hex    = parent.getHexMetrics();
        if (dialog != null && hex != null) {
            int lineHeight = Math.max(dialog.getHeight(), hex.getHeight());
            getHorizontalScrollBar().setUnitIncrement(lineHeight);
            getVerticalScrollBar().setUnitIncrement(lineHeight);
        }

        return;
    }

}
