package debugger;

// Java imports
import java.awt.*;

// Project imports
import debugger.*;
import vue.*;

// One character in VIP memory
class VIPCharacter {

    // Instance fields
    private int      index;  // Index within VIP memory
    private Debugger parent; // Debugger UI manager

    // VIP fields
    int[] pixels; // Pattern pixel values



    ///////////////////////////////////////////////////////////////////////////
    //                             Constructors                              //
    ///////////////////////////////////////////////////////////////////////////

    // Default constructor
    VIPCharacter(Debugger parent, int index) {
        this.index  = index;
        this.parent = parent;
        pixels      = new int[64];
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Package Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Draw the image into a graphics context
    void paint(Graphics g, int x, int y, boolean hFlip,
        boolean vFlip, int scale, int palette, boolean generic) {

        // Select the brightness levels for the given palette
        int[] pal = palette == VIPPalettes.GENERIC ? new int[] { 1, 2, 3 }:
            parent.getPalettes().palettes[palette];

        // Process all pixels
        for (int Y = 0; Y < 8; Y++)
        for (int X = 0; X < 8; X++) {
            int value = sample(X, Y, hFlip, vFlip);
            g.setColor(parent.getColor(
                value == 0 ? -1 : pal[value - 1], generic));
            g.fillRect(x + X * scale, y + Y * scale, scale, scale);
        }

    }

    // Update data fields with the current emulation state
    void refresh() {

        // Prepare working variables
        int    bits   = 1;
        int    offset = getOffset();
        byte[] vram   = parent.getVRAM();

        // Process all pixels
        for (int p = 0; p < 64; p++, bits >>= 2) {
            if (bits == 1)
                bits = 0x100 | vram[offset++] & 0xFF;
            pixels[p] = bits & 3;
        }

    }

    // Retrieve one pixel from this character
    int sample(int x, int y, boolean hFlip, boolean vFlip) {
        return pixels[(vFlip ? 7 - y : y) << 3 | (hFlip ? 7 - x : x)];
    }

    // Retrieve all pixels from this character
    int[] sample(int[] pixels, boolean hFlip, boolean vFlip) {

        // Prepare working variables
        int offset = index << 4;     // RAM buffer offset
        int xStart = hFlip ?  7 : 0; // Initial X value
        int xStep  = hFlip ? -1 : 1; // X increment direction
        int yStart = vFlip ?  7 : 0; // Initial Y value
        int yStep  = vFlip ? -1 : 1; // Y increment direction

        // Produce a new output buffer if none was given
        if (pixels == null)
            pixels = new int[64];

        // Process all pixels
        for (int y = yStart, p = 0; (y & 8) == 0; y += yStep)
        for (int x = xStart; (x & 8) == 0; x += xStep, p++)
            pixels[p] = this.pixels[y << 3 | x];

        return pixels;
    }

    // Specify a new color for an individual pixel
    boolean setPixel(int x, int y, int value) {

        // No change is being made
        if (pixels[y << 3 | x] == value)
            return false;

        // Calculate the new byte value
        int    offset = getOffset() | y << 1 | x >> 2;
        byte[] vram   = parent.getVRAM();
               x      = (x & 3) << 1;
               value  = vram[offset] & 0xFF & ~(3 << x) | value << x;

        // Change the pixel value in the emulation state
        parent.getVUE().write(offset, VUE.U8, value, true);
        return true;
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Private Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Calculate the VRAM offset of this character
    private int getOffset() {
        return 0x00006000 | (index & 0x600) << 6 | (index & 0x1FF) << 4;
    }

}
