package debugger;

// Java imports
import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;
import java.util.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;

// Project imports
import debugger.*;
import util.*;
import vue.*;

// VIP character viewer
final class VIPChrPane extends JPanel {

    // Instance fields
    private VIPCharacter[] characters; // Character patterns
    private int            color;      // Selected color index
    private boolean        dragging;   // Drawing into the pattern box
    private boolean        generic;    // Force the generic colors
    private int            index;      // Selected character index
    private int            palette;    // Selected display palette
    private Debugger       parent;     // Debugger UI manager
    private int            scale;      // Scaling for character list images
    private boolean        updating;   // Setting spinner programmatically

    // UI elements
    private JComboBox<String> cmbPalette;    // Palette combo box
    private JPanel            panCharacters; // Character content panel
    private JPanel[]          panColors;     // Palette color boxes
    private JPanel            panControls;   // Controls content panel
    private JPanel            panPattern;    // Pattern box
    private JScrollPane       scrCharacters; // Scroll pane for character list
    private JScrollPane       scrControls;   // Scroll pane for controls
    private JSlider           sldScale;      // Scale slider
    private JSpinner          spnIndex;      // Index spinner
    private JTextField        txtAddress;    // Address text box
    private JTextField        txtMirror;     // Mirror text box
    private ArrayList<JComponent> uiFont;    // UI elements depending on font



    ///////////////////////////////////////////////////////////////////////////
    //                               Constants                               //
    ///////////////////////////////////////////////////////////////////////////

    // Default scale
    private static final int DEFAULTSCALE = 3;

    // Color selection borders
    private static final Border DESELECTED =
        new LineBorder(Util.getColor("control"), 3);
    private static final Border SELECTED   = new CompoundBorder(
        new LineBorder(Util.getColor("selected"), 2),
        new LineBorder(Util.getColor("control"), 1)
    );



    ///////////////////////////////////////////////////////////////////////////
    //                             Constructors                              //
    ///////////////////////////////////////////////////////////////////////////

    // Default constructor
    VIPChrPane(Debugger parent) {
        super(new BorderLayout());

        // Configure instance fields
        characters  = new VIPCharacter[2048];
        color       = -1;
        dragging    = false;
        generic     = true;
        index       = 0;
        palette     = VIPPalettes.GENERIC;
        this.parent = parent;
        scale       = DEFAULTSCALE;
        updating    = false;

        // Configure controls
        uiFont = new ArrayList<JComponent>();
        initCharacters();
        initControls();

        // Configure component
        add(scrCharacters, BorderLayout.CENTER    );
        add(scrControls,   BorderLayout.LINE_START);
        setIndex(index);
        setScale(scale);
    }

    // Character pane constructor
    private void initCharacters() {

        // Configure content pane
        panCharacters = new JPanel(null) {
            public void paintComponent(Graphics g) {
                super.paintComponent(g);
                onPaintChars(g);
            }
        };
        panCharacters.setBackground(Util.getColor("control"));
        panCharacters.setFocusable(true);
        panCharacters.addMouseListener(Util.onMouse(
            e->onMouseDownChars(e), null));
        resizeCharacters();

        // Configure scroll pane
        scrCharacters = new JScrollPane(panCharacters);

        // Configure characters
        for (int x = 0; x < 2048; x++)
            characters[x] = new VIPCharacter(parent, x);
    }

    // Controls pane constructor
    private void initControls() {

        // Top-level container
        panControls = new JPanel(new GridBagLayout()) {
            public void paintComponent(Graphics g)
                { super.paintComponent(g); onPaintControls(); }
        };
        panControls.setFocusable(true);
        panControls.addMouseListener(Util.onMouse(
            e->panControls.requestFocus(), null));

        // Scroll pane for container
        scrControls = new JScrollPane(panControls,
            JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED,
            JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
        scrControls.setBorder(null);

        // Index spinner
        JLabel lblIndex = new JLabel("Index");
        spnIndex = new JSpinner(new SpinnerNumberModel(0, 0, 2047, 1));
        spnIndex.setEditor(new JSpinner.NumberEditor(spnIndex, "#"));
        spnIndex.addChangeListener(e->onIndex());
        add(panControls, lblIndex, 1, false, false, 2, 4, 0, 4);
        add(panControls, spnIndex, 0, false, true, 2, 0, 0, 4);
        uiFont.add(lblIndex);
        uiFont.add(spnIndex);

        // Pattern group box
        JPanel frmPattern = new JPanel(new GridBagLayout());
        frmPattern.setBorder(new TitledBorder("Pattern"));
        add(panControls, frmPattern, 0, false, true, 2, 0, 0, 0);
        uiFont.add(frmPattern);

        // Pattern panel
        panPattern = new JPanel(null) {
            public void paintComponent(Graphics g)
                { super.paintComponent(g); onPaintPattern(g); }
        };
        panPattern.setPreferredSize(new Dimension(128, 128));
        panPattern.addMouseListener(Util.onMouse(
            e->onMouseDownPattern(e), e->onMouseUpPattern(e)));
        panPattern.addMouseMotionListener(Util.onMouseMotion(
            null, e->onMouseDownPattern(e)));
        add(frmPattern, panPattern, 0, true, false, 0, 2, 0, 2);

        // Text box event handlers
        ActionListener action =
            e->{ panControls.requestFocus(); onTextBox(e); };
        FocusListener  focus  = Util.onFocus(e->onTextBox(e), null);

        // Address text box
        JLabel lblAddress = new JLabel("Address");
        txtAddress = new JTextField();
        txtAddress.addActionListener(action);
        txtAddress.addFocusListener(focus);
        add(frmPattern, lblAddress, 1, false, false, 2, 2, 0, 4);
        add(frmPattern, txtAddress, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblAddress);
        uiFont.add(txtAddress);

        // Mirror text box
        JLabel lblMirror = new JLabel("Mirror");
        txtMirror = new JTextField();
        txtMirror.addActionListener(action);
        txtMirror.addFocusListener(focus);
        add(frmPattern, lblMirror, 1, false, false, 2, 2, 0, 4);
        add(frmPattern, txtMirror, 0, false, true, 2, 0, 2, 2);
        uiFont.add(lblMirror);
        uiFont.add(txtMirror);

        // Options group box
        JPanel frmOptions = new JPanel(new GridBagLayout());
        frmOptions.setBorder(new TitledBorder("Options"));
        add(panControls, frmOptions, 0, false, true, 0, 0, 0, 0);
        uiFont.add(frmOptions);

        // Scale slider
        JLabel lblScale = new JLabel("Scale");
        sldScale = new JSlider(1, 10, DEFAULTSCALE);
        sldScale.setFocusable(false);
        sldScale.setLabelTable(null);
        sldScale.setPreferredSize(new Dimension(0,
            sldScale.getPreferredSize().height));
        sldScale.setSnapToTicks(true);
        sldScale.addChangeListener(e->onScale(e));
        add(frmOptions, lblScale, 1, false, false, 2, 2, 0, 4);
        add(frmOptions, sldScale, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblScale);

        // Palette combo box
        JLabel lblPalette = new JLabel("Palette");
        cmbPalette = new JComboBox<String>(new String[] { "Generic",
            "BG 0",  "BG 1",  "BG 2",  "BG 3",
            "OBJ 0", "OBJ 1", "OBJ 2", "OBJ 3"
        });
        add(frmOptions, lblPalette, 1, false, false, 2, 2, 0, 4);
        add(frmOptions, cmbPalette, 0, false, true, 2, 0, 0, 2);
        cmbPalette.addActionListener(e->onPalette());
        uiFont.add(lblPalette);
        uiFont.add(cmbPalette);

        // Event handler for color boxes
        MouseListener mouse = Util.onMouse(e->onMouseDownColor(e), null);

        // Color boxes
        JPanel box = new JPanel();
        ((FlowLayout) box.getLayout()).setHgap(0);
        panColors = new JPanel[4];
        for (int x = 0; x < 4; x++) {
            JPanel color = panColors[x] = new JPanel(null) {
                public void paintComponent(Graphics g)
                    { super.paintComponent(g); onPaintColor(g, this); }
            };
            color.putClientProperty("color", x);
            color.setPreferredSize(new Dimension(30, 30));
            color.setBorder(DESELECTED);
            color.addMouseListener(mouse);
            box.add(color);
        }
        add(frmOptions, box, 0, true, false, 0, 0, 0, 0);

        // Finalize layout
        finish(panControls, 0);
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Event Handlers                             //
    ///////////////////////////////////////////////////////////////////////////

    // Index spinner
    private void onIndex() {
        if (!updating)
            setIndex((Integer) spnIndex.getValue());
    }

    // Mouse Down in the characters pane
    private void onMouseDownChars(MouseEvent e) {
        int button = e.getButton();
        int x      = e.getX();
        int y      = e.getY();

        // Common processing
        panCharacters.requestFocus();

        // Only process left clicks
        if (button != MouseEvent.BUTTON1)
            return;

        // Select the character clicked on, if any
        x /= scale * 8 + 1;
        y /= scale * 8 + 1;
        if (x >= 0 && x <= 15 && y >= 0 && y <= 127)
            setIndex(y * 16 + x);
    }

    // Color boxes
    private void onMouseDownColor(MouseEvent e) {

        // Common processing
        panControls.requestFocus();

        // Only process left clicks
        if (e.getButton() != MouseEvent.BUTTON1)
            return;

        // Deselect the current color
        if (color != -1)
            panColors[color].setBorder(DESELECTED);

        // Select the new color
        int index = (Integer) ((JComponent) e.getSource()
            ).getClientProperty("color");
        if (index != color) {
            color = index;
            panColors[color].setBorder(SELECTED);
        }

        // The current color was deselected
        else color = -1;
    }

    // Mouse Down in the pattern box
    private void onMouseDownPattern(MouseEvent e) {
        int button = e.getButton();
        int x      = e.getX() / 16;
        int y      = e.getY() / 16;

        // Common processing
        if (e.getID() == MouseEvent.MOUSE_PRESSED)
            panControls.requestFocus();

        // Only process left clicks within range
        if (!dragging && button != MouseEvent.BUTTON1 ||
            color == -1 || x < 0 || x > 7 || y < 0 || y > 7
        ) return;

        // Update the selected pixel
        if (characters[index].setPixel(x, y, color))
            parent.refresh(false);
        dragging = true;
    }

    // Mouse Up on in the pattern box
    private void onMouseUpPattern(MouseEvent e) {
        if (e.getButton() == MouseEvent.BUTTON1)
            dragging = false;
    }

    // Palette combo box
    private void onPalette() {
        int index = cmbPalette.getSelectedIndex();
        generic = index == 0;
        palette = index - 1;
        refresh();
    }

    // Painting the characters pane
    private void onPaintChars(Graphics g) {
        int pad   = 1; //grid ? 1 : 0;
        int dim   = scale * 8 + pad;

        // Determine the range of characters visible in the pane
        JViewport viewport = scrCharacters.getViewport();
        Point     pos      = viewport.getViewPosition();
        Dimension size     = viewport.getExtentSize();
        int left   = Math.max(Math.min( pos.x                / dim,  15), 0);
        int top    = Math.max(Math.min( pos.y                / dim, 127), 0);
        int right  = Math.max(Math.min((pos.x + size.width ) / dim,  15), 0);
        int bottom = Math.max(Math.min((pos.y + size.height) / dim, 127), 0);

        // Draw each visible character
        for (int y = top;  y <= bottom; y++)
        for (int x = left; x <= right; x++) {

            // Draw character
            characters[y << 4 | x].paint(g, x * dim, y * dim, false, false,
                scale, palette, generic);

            // Draw selection color
            if (index != y * 16 + x)
                continue;
            g.setColor(Util.getColor("selected", 0.5f));
            g.fillRect(x * dim - 1, y * dim - 1, dim + 1, dim + 1);
        }
    }

    // Painting the color boxes
    private void onPaintColor(Graphics g, JPanel box) {
        g.setColor(parent.getColor(
            (Integer) box.getClientProperty("color"), generic));
        g.fillRect(0, 0, box.getWidth(), box.getHeight());
    }

    // Painting the controls scroll pane
    private void onPaintControls() {

        // Determine the size of the inner element and the visible area
        Dimension extent = scrControls.getViewport().getExtentSize();
        Dimension prefer = panControls.getPreferredSize();
        if (extent.width == prefer.width)
            return;

        // Resize the container to exactly contain the inner element
        prefer.width  = scrControls.getWidth() + prefer.width - extent.width;
        prefer.height = 0;
        scrControls.setPreferredSize(prefer);
        scrControls.revalidate();
    }

    // Painting the pattern box
    private void onPaintPattern(Graphics g) {
        characters[index].paint(g, 0, 0, false, false, 16, palette, generic);
    }

    // Scale slider changed
    private void onScale(ChangeEvent e) {
        if (!updating)
            setScale(sldScale.getValue());
    }

    // Text box updated
    private void onTextBox(AWTEvent e) {
        String text  = ((JTextField) e.getSource()).getText();
        int    entry = 0;

        // Process the text into either an address or a mirror address
        try { entry = (int) Long.parseLong(text, 16); }
        catch (Exception x) { }
        int address = mirrorToAddress(entry & 0x0007FFFF);
        int mirror  = addressToMirror(entry & 0x0007FFFF);

        // The number entered was invalid
        if ((entry >> 24 & 7) != 0 || (address | mirror) == 0) {
            setIndex(index);
            return;
        }

        // Set the new index
        if (mirror == 0)
            mirror = addressToMirror(address & 0x0007FFFF);
        else mirror &= 0x0007FFFF;
        setIndex(mirror - 0x00078000 >> 4);
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Package Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Retrieve the character at the given index
    VIPCharacter get(int index) {
        return characters[index];
    }

    // Update UI components with the current emulation state
    void refresh() {

        // Update state fields
        for (int x = 0; x < 2048; x++)
            characters[x].refresh();

        // Update UI elements
        panCharacters.repaint();
        panPattern.repaint();
        for (int x = 0; x < 4; x++)
            panColors[x].repaint();
    }

    // Retrieve one pixel from one character
    int sample(int index, int x, int y, boolean hFlip, boolean yFlip) {
        return characters[index].sample(x, y, hFlip, yFlip);
    }

    // Specify new fonts
    void setFonts() {
        Font dlgFont = parent.getDialogFont();
        Font hexFont = parent.getHexFont();

        // Update all appropriate UI controls
        for (int x = 0; x < uiFont.size(); x++) {
            JComponent control = uiFont.get(x);
            if (control == txtAddress || control == txtMirror)
                control.setFont(hexFont);
            else if (control instanceof JPanel)
                ((TitledBorder) control.getBorder()).setTitleFont(dlgFont);
            else control.setFont(dlgFont);
        }
    }

    // Specify the current selected character index
    void setIndex(int index) {
        this.index = index;
        int mirror = 0x00078000 + index * 16;

        // Update spinner
        updating = true;
        spnIndex.setValue(Math.min(Math.max(index, 0), 2047));
        updating = false;

        // Update controls
        txtAddress.setText(String.format("%08X", mirrorToAddress(mirror)));
        txtMirror .setText(String.format("%08X", mirror ));
        panCharacters.repaint();
        panPattern.repaint();
    }

    // Specify a new scaling factor for the character list
    void setScale(int scale) {
        this.scale = scale;

        // Update slider
        updating = true;
        sldScale.setValue(scale);
        updating = false;

        // Configure characters list
        resizeCharacters();
        panCharacters.revalidate();

        // Configure characters scroll pane
        scale = scale * 8 + 1;
        scrCharacters.getHorizontalScrollBar().setUnitIncrement(scale);
        scrCharacters.getVerticalScrollBar().setUnitIncrement(scale);
        scrCharacters.repaint();
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Private Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Add a control to a container
    private void add(JComponent container, JComponent control, int colSpan,
        boolean center, boolean stretch, int top, int left, int bottom,
        int right) {
        GridBagConstraints c = new GridBagConstraints();
        c.gridwidth = colSpan != 0 ? colSpan : GridBagConstraints.REMAINDER;
        c.insets    = new Insets(top, left, bottom, right);
        c.weightx   = colSpan != 0 ? 0 : 1;
        if (!center) c.anchor = GridBagConstraints.LINE_START;
        if (stretch) c.fill   = GridBagConstraints.HORIZONTAL;
        container.add(control, c);
    }

    // Finalize a container layout
    private void finish(JComponent container, int spacing) {
        JPanel spacer = new JPanel(null);
        spacer.setPreferredSize(new Dimension(0, 0));
        spacer.setOpaque(false);
        GridBagConstraints c = new GridBagConstraints();
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.insets    = new Insets(spacing, 0, 0, 0);
        c.weighty   = 1;
        container.add(spacer, c);
    }

    // Convert a character address to its mirror's address
    private int addressToMirror(int address) {
        if (address >= 0x00006000 && address <= 0x00007FFF)
            return address + 0x00072000;
        if (address >= 0x0000E000 && address <= 0x0000FFFF)
            return address + 0x0006A000;
        if (address >= 0x00016000 && address <= 0x00017FFF)
            return address + 0x00066000;
        if (address >= 0x0001E000 && address <= 0x0001FFFF)
            return address + 0x00060000;
        return 0;
    }

    // Convert a character's mirror address to its address
    private int mirrorToAddress(int mirror) {
        if (mirror >= 0x00078000 && mirror <= 0x00079FFF)
            return mirror - 0x00072000;
        if (mirror >= 0x0007A000 && mirror <= 0x0007BFFF)
            return mirror - 0x0006A000;
        if (mirror >= 0x0007C000 && mirror <= 0x0007DFFF)
            return mirror - 0x00066000;
        if (mirror >= 0x0007E000 && mirror <= 0x0007FFFF)
            return mirror - 0x00060000;
        return 0;
    }

    // Calculate the new size of the characters list
    private void resizeCharacters() {
        panCharacters.setPreferredSize(new Dimension(
            (scale * 8 + 1) *  16 - 1,
            (scale * 8 + 1) * 128 - 1
        ));
    }

}
