package debugger;

// Java imports
import java.awt.*;

// Project imports
import debugger.*;
import vue.*;

// Manages colors for VIP palettes
class VIPPalettes {

    // Instance fields
    private Debugger parent; // Debugger UI manager

    // VIP fields
    int     background; // Clear color index
    int[]   brightness; // Brightness levels
    int[][] palettes;   // Brightness level indexes per palette



    ///////////////////////////////////////////////////////////////////////////
    //                               Constants                               //
    ///////////////////////////////////////////////////////////////////////////

    // Palette indexes
    static final int GENERIC = -1;
    static final int BG0     =  0;
    static final int BG1     =  1;
    static final int BG2     =  2;
    static final int BG3     =  3;
    static final int OBJ0    =  4;
    static final int OBJ1    =  5;
    static final int OBJ2    =  6;
    static final int OBJ3    =  7;



    ///////////////////////////////////////////////////////////////////////////
    //                             Constructors                              //
    ///////////////////////////////////////////////////////////////////////////

    // Default constructor
    VIPPalettes(Debugger parent) {
        background  = 0;
        brightness  = new int[3];
        palettes    = new int[8][3];
        this.parent = parent;
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Package Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Update data with the current emulation state
    void refresh() {
        VUE vue = parent.getVUE();

        // Process background color
        background = vue.read(0x0005F870, VUE.U8, true);

        // Process brightness levels
        int brta      = vue.read(0x0005F824, VUE.U16, true) & 0x00FF;
        int brtb      = vue.read(0x0005F826, VUE.U16, true) & 0x00FF;
        int brtc      = vue.read(0x0005F828, VUE.U16, true) & 0x00FF;
        brightness[0] = Math.min(brta,               127);
        brightness[1] = Math.min(brtb,               127);
        brightness[2] = Math.min(brta + brtb + brtc, 127);

        // Process palettes
        for (int x = 0; x < 8; x++) {
            int p = vue.read(0x0005F860 | x << 1, VUE.U16, true);
            palettes[x][0] = p >> 2 & 3;
            palettes[x][1] = p >> 4 & 3;
            palettes[x][2] = p >> 6 & 3;
        }

    }

    // Scale a color according to the brightness levels
    int[] scaleColor(int[] colors, int argb) {
        return scaleColor(colors, argb, brightness);
    }

    // Scale a color according to given brightness levels
    static int[] scaleColor(int[] colors,int argb,int brt1,int brt2,int brt3) {
        return scaleColor(colors, argb, new int[] { brt1, brt2, brt3 });
    }

    // Scale a color according to given brightness levels
    static int[] scaleColor(int[] colors, int argb, int[] brightness) {

        // Produce a new output buffer if none was given
        if (colors == null)
            colors = new int[3];

        // Process all brightness levels
        for (int x = 0; x < 3; x++) {
            int brt = brightness[x] * 2;
            colors[x] = argb & 0xFF000000 |
                ((argb >> 16 & 0xFF) * brt + 1) / 254 << 16 |
                ((argb >>  8 & 0xFF) * brt + 1) / 254 <<  8 |
                ((argb       & 0xFF) * brt + 1) / 254;
        }

        return colors;
    }

}
