package debugger;

// Java imports
import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;
import java.util.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;

// Project imports
import debugger.*;
import util.*;
import vue.*;

// VIP world viewer
class VIPWorldPane extends JPanel {

    // Instance fields
    private boolean     generic;  // Force the generic colors
    private int         index;    // Selected world index
    private int         line;     // Selected parameter scanline
    private Debugger    parent;   // Debugger UI manager
    private int         scale;    // Scaling for object image
    private boolean     updating; // Setting values programmatically
    private VIPWorld[]  worlds;   // Worlds

    // UI components
    private JCheckBox         chkEnd;        // End check box
    private JCheckBox         chkGeneric;    // Generic colors check box
    private JCheckBox         chkLeft;       // Left check box
    private JCheckBox         chkOverplane;  // Overplane check box
    private JCheckBox         chkRight;      // Right check box
    private JComboBox<String> cmbMode;       // Type combo box
    private JPanel            frmBackground; // Background group box
    private JPanel            frmObjects;    // Objects group box
    private JPanel            frmParams;     // Parameters group box
    private BufferedImage     imgFrame;      // Frame buffer
    private JLabel            lblEnd;        // End label
    private JLabel            lblGroup;      // Group label
    private JLabel            lblStart;      // Start label
    private JPanel            panControls;   // Controls content panel
    private JPanel            panPreview;    // Preview content panel
    private JScrollPane       scrControls;   // Scroll pane for controls
    private JScrollPane       scrPreview;    // Scroll pane for preview
    private JSlider           sldScale;      // Scale slider
    private JSpinner          spnABGX;       // Affine BG X spinner
    private JSpinner          spnABGY;       // Affine BG Y spinner
    private JSpinner          spnAVectorX;   // Affine Vector X spinner
    private JSpinner          spnAVectorY;   // Affine Vector Y spinner
    private JSpinner          spnAParallax;  // Affine Parallax spinner
    private JSpinner          spnBGBase;     // BG Base spinner
    private JSpinner          spnBGHeight;   // BG Height spinner
    private JSpinner          spnBGParallax; // BG Parallax spinner
    private JSpinner          spnBGWidth;    // BG Width spinner
    private JSpinner          spnBGX;        // BG X spinner
    private JSpinner          spnBGY;        // BG Y spinner
    private JSpinner          spnHeight;     // Height spinner
    private JSpinner          spnHLeft;      // H-bias Left spinner
    private JSpinner          spnHRight;     // H-bias Right spinner
    private JSpinner          spnIndex;      // Index spinner
    private JSpinner          spnLine;       // Parameters Line spinner
    private JSpinner          spnOverchar;   // Overplane spinner
    private JSpinner          spnParallax;   // Parallax spinner
    private JSpinner          spnWidth;      // Width spinner
    private JSpinner          spnX;          // X spinner
    private JSpinner          spnY;          // Y spinner
    private JTextField        txtAddress;    // Address text box
    private JTextField        txtPAddress;   // Parameters Address text box
    private JTextField        txtParamAddr;  // Params text box
    private ArrayList<JComponent> uiAffine;  // Affine UI elements
    private ArrayList<JComponent> uiFont;    // UI elements depending on font
    private ArrayList<JComponent> uiHBias;   // H-bias UI elements



    ///////////////////////////////////////////////////////////////////////////
    //                               Constants                               //
    ///////////////////////////////////////////////////////////////////////////

    // Default scale
    private static final int DEFAULTSCALE = 1;



    ///////////////////////////////////////////////////////////////////////////
    //                             Constructors                              //
    ///////////////////////////////////////////////////////////////////////////

    // Default constructor
    VIPWorldPane(Debugger parent) {
        super(new BorderLayout());

        // Configure instance fields
        generic     = false;
        index       = 0;
        line        = 0;
        this.parent = parent;
        scale       = DEFAULTSCALE;
        updating    = false;
        worlds      = new VIPWorld[32];

        // Configure controls
        imgFrame = new BufferedImage(384, 224, BufferedImage.TYPE_INT_ARGB);
        uiAffine = new ArrayList<JComponent>();
        uiFont   = new ArrayList<JComponent>();
        uiHBias  = new ArrayList<JComponent>();
        initPreview();
        initControls();

        // Configure component
        add(scrPreview,  BorderLayout.CENTER    );
        add(scrControls, BorderLayout.LINE_START);
        setIndex(index);
        setScale(scale);
    }

    // Preview pane constructor
    private void initPreview() {

        // Configure content pane
        panPreview = new JPanel(null) {
            public void paintComponent(Graphics g) {
                super.paintComponent(g);
                onPaintPreview(g);
            }
        };
        panPreview.setBackground(Util.getColor("control"));
        panPreview.setFocusable(true);
        panPreview.addMouseListener(Util.onMouse(
            e->panPreview.requestFocus(), null));
        resizePreview();

        // Configure scroll pane
        scrPreview = new JScrollPane(panPreview);

        // Configure worlds
        for (int x = 0; x < 32; x++)
            worlds[x] = new VIPWorld(parent, x);
    }

    // Control pane constructor
    private void initControls() {

        // Top-level container
        panControls = new JPanel(new GridBagLayout()) {
            public void paintComponent(Graphics g)
                { super.paintComponent(g); onPaintControls(); }
        };
        panControls.setFocusable(true);
        panControls.addMouseListener(Util.onMouse(
            e->panControls.requestFocus(), null));

        // Scroll pane for container
        scrControls = new JScrollPane(panControls,
            JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED,
            JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
        scrControls.setBorder(null);

        // Index spinner
        JLabel lblIndex = new JLabel("Index");
        spnIndex = new JSpinner(new SpinnerNumberModel(0, 0, 31, 1));
        spnIndex.setEditor(new JSpinner.NumberEditor(spnIndex, "#"));
        spnIndex.addChangeListener(e->{ if (!updating) onIndex(); });
        add(panControls, lblIndex, 1, false, false, 2, 4, 0, 4);
        add(panControls, spnIndex, 0, false, true, 2, 0, 0, 4);
        uiFont.add(lblIndex);
        uiFont.add(spnIndex);

        // Control groups
        initWorld();
        initBackground();
        initParams();
        initObjects();
        initOptions();

        // Finalize layout
        finish(panControls, 0);
    }

    // World group box constructor
    private void initWorld() {
        JPanel frmWorld = new JPanel(new GridBagLayout());
        frmWorld.setBorder(new TitledBorder("World"));
        add(panControls, frmWorld, 0, false, true, 2, 0, 0, 0);
        uiFont.add(frmWorld);

        // Event handlers
        ActionListener onAction = e->{ if (!updating) onEdit(e); };
        ChangeListener onChange = e->{ if (!updating) onEdit(e); };

        // Address text box
        JLabel lblAddress = new JLabel("Address");
        txtAddress = new JTextField();
        txtAddress.addActionListener(
            e->{ panControls.requestFocus(); onAddress(); });
        txtAddress.addFocusListener(Util.onFocus(null, e->onAddress()));
        add(frmWorld, lblAddress, 1, false, false, 2, 2, 0, 4);
        add(frmWorld, txtAddress, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblAddress);
        uiFont.add(txtAddress);

        // Mode combo box
        JLabel lblMode = new JLabel("Mode");
        cmbMode = new JComboBox<String>(new String[]
            { "Normal", "H-Bias", "Affine", "Object" });
        cmbMode.putClientProperty("offset", 0);
        cmbMode.addActionListener(onAction);
        add(frmWorld, lblMode, 1, false, false, 2, 2, 0, 4);
        add(frmWorld, cmbMode, 0, false, true, 2, 0, 0, 2);

        // X spinner
        JLabel lblX = new JLabel("X");
        spnX = new JSpinner(new SpinnerNumberModel(0, -512, 511, 1));
        spnX.setEditor(new JSpinner.NumberEditor(spnX, "#"));
        spnX.putClientProperty("offset", 2);
        spnX.addChangeListener(onChange);
        add(frmWorld, lblX, 1, false, false, 2, 2, 0, 4);
        add(frmWorld, spnX, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblX);
        uiFont.add(spnX);

        // Y spinner
        JLabel lblY = new JLabel("Y");
        spnY = new JSpinner(new SpinnerNumberModel(0, -32768, 32767, 1));
        spnY.setEditor(new JSpinner.NumberEditor(spnY, "#"));
        spnY.putClientProperty("offset", 6);
        spnY.addChangeListener(onChange);
        add(frmWorld, lblY, 1, false, false, 2, 2, 0, 4);
        add(frmWorld, spnY, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblY);
        uiFont.add(spnY);

        // Parallax spinner
        JLabel lblParallax = new JLabel("Parallax");
        spnParallax = new JSpinner(new SpinnerNumberModel(0, -512, 511, 1));
        spnParallax.setEditor(new JSpinner.NumberEditor(spnParallax, "#"));
        spnParallax.putClientProperty("offset", 4);
        spnParallax.addChangeListener(onChange);
        add(frmWorld, lblParallax, 1, false, false, 2, 2, 0, 4);
        add(frmWorld, spnParallax, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblParallax);
        uiFont.add(spnParallax);

        // Width spinner
        JLabel lblWidth = new JLabel("Width");
        spnWidth = new JSpinner(new SpinnerNumberModel(0, -4095, 4096, 1));
        spnWidth.setEditor(new JSpinner.NumberEditor(spnWidth, "#"));
        spnWidth.putClientProperty("offset", 14);
        spnWidth.addChangeListener(onChange);
        add(frmWorld, lblWidth, 1, false, false, 2, 2, 0, 4);
        add(frmWorld, spnWidth, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblWidth);
        uiFont.add(spnWidth);

        // Height spinner
        JLabel lblHeight = new JLabel("Height");
        spnHeight = new JSpinner(new SpinnerNumberModel(0, -32767, 32768, 1));
        spnHeight.setEditor(new JSpinner.NumberEditor(spnHeight, "#"));
        spnHeight.putClientProperty("offset", 16);
        spnHeight.addChangeListener(onChange);
        add(frmWorld, lblHeight, 1, false, false, 2, 2, 0, 4);
        add(frmWorld, spnHeight, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblHeight);
        uiFont.add(spnHeight);

        // Params text box
        JLabel lblParamAddr = new JLabel("Params");
        txtParamAddr = new JTextField();
        txtParamAddr.putClientProperty("offset", 18);
        txtParamAddr.addActionListener(e->
            { panControls.requestFocus(); if (!updating) onEdit(e); });
        txtParamAddr.addFocusListener(Util.onFocus(null, e->
            { if (!updating) onEdit(e); }));
        add(frmWorld, lblParamAddr, 1, false, false, 2, 2, 0, 4);
        add(frmWorld, txtParamAddr, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblParamAddr);
        uiFont.add(txtParamAddr);

        // Overplane spinner
        JLabel lblOverchar = new JLabel("Overplane");
        spnOverchar = new JSpinner(new SpinnerNumberModel(0, 0, 2047, 1));
        spnOverchar.setEditor(new JSpinner.NumberEditor(spnOverchar, "#"));
        spnOverchar.putClientProperty("offset", 20);
        spnOverchar.addChangeListener(onChange);
        add(frmWorld, lblOverchar, 1, false, false, 2, 2, 0, 4);
        add(frmWorld, spnOverchar, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblOverchar);
        uiFont.add(spnOverchar);

        // Left and right check boxes
        chkLeft = new JCheckBox("Left");
        chkLeft.setBorder(null);
        chkLeft.putClientProperty("offset", 0);
        chkLeft.addActionListener(onAction);
        chkRight = new JCheckBox("Right");
        chkRight.setBorder(null);
        chkRight.putClientProperty("offset", 0);
        chkRight.addActionListener(onAction);
        add(frmWorld, chkLeft, 1, false, false, 2, 2, 0, 0);
        add(frmWorld, chkRight, 0, false, false, 2, 2, 0, 2);
        uiFont.add(chkLeft);
        uiFont.add(chkRight);

        // End and Overplane check boxes
        chkEnd = new JCheckBox("End");
        chkEnd.setBorder(null);
        chkEnd.putClientProperty("offset", 0);
        chkEnd.addActionListener(onAction);
        chkOverplane = new JCheckBox("Overplane");
        chkOverplane.setBorder(null);
        chkOverplane.putClientProperty("offset", 0);
        chkOverplane.addActionListener(onAction);
        add(frmWorld, chkEnd, 1, false, false, 2, 2, 0, 0);
        add(frmWorld, chkOverplane, 0, false, false, 2, 2, 0, 0);
        uiFont.add(chkEnd);
        uiFont.add(chkOverplane);
    }

    // Background group box constructor
    private void initBackground() {
        frmBackground = new JPanel(new GridBagLayout());
        frmBackground.setBorder(new TitledBorder("Background"));
        add(panControls, frmBackground, 0, false, true, 2, 0, 0, 0);
        uiFont.add(frmBackground);

        // Event handlers
        ActionListener onAction = e->{ if (!updating) onEdit(e); };
        ChangeListener onChange = e->{ if (!updating) onEdit(e); };

        // Base spinner
        JLabel lblBGBase = new JLabel("Base map");
        spnBGBase = new JSpinner(new SpinnerNumberModel(0, 0, 15, 1));
        spnBGBase.setEditor(new JSpinner.NumberEditor(spnBGBase, "#"));
        spnBGBase.putClientProperty("offset", 0);
        spnBGBase.addChangeListener(onChange);
        add(frmBackground, lblBGBase, 1, false, false, 2, 2, 0, 4);
        add(frmBackground, spnBGBase, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblBGBase);
        uiFont.add(spnBGBase);

        // BG X spinner
        JLabel lblBGX = new JLabel("X");
        spnBGX = new JSpinner(new SpinnerNumberModel(0, -4096, 4095, 1));
        spnBGX.setEditor(new JSpinner.NumberEditor(spnBGX, "#"));
        spnBGX.putClientProperty("offset", 8);
        spnBGX.addChangeListener(onChange);
        add(frmBackground, lblBGX, 1, false, false, 2, 2, 0, 4);
        add(frmBackground, spnBGX, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblBGX);
        uiFont.add(spnBGX);

        // BG Y spinner
        JLabel lblBGY = new JLabel("Y");
        spnBGY = new JSpinner(new SpinnerNumberModel(0, -4096, 4095, 1));
        spnBGY.setEditor(new JSpinner.NumberEditor(spnBGY, "#"));
        spnBGY.putClientProperty("offset", 12);
        spnBGY.addChangeListener(onChange);
        add(frmBackground, lblBGY, 1, false, false, 2, 2, 0, 4);
        add(frmBackground, spnBGY, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblBGY);
        uiFont.add(spnBGY);

        // BG Parallax spinner
        JLabel lblBGParallax = new JLabel("Parallax");
        spnBGParallax = new JSpinner(new SpinnerNumberModel(0,-16384,16383,1));
        spnBGParallax.setEditor(new JSpinner.NumberEditor(spnBGParallax, "#"));
        spnBGParallax.putClientProperty("offset", 10);
        spnBGParallax.addChangeListener(onChange);
        add(frmBackground, lblBGParallax, 1, false, false, 2, 2, 0, 4);
        add(frmBackground, spnBGParallax, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblBGParallax);
        uiFont.add(spnBGParallax);

        // BG Width spinner
        JLabel lblBGWidth = new JLabel("Width");
        spnBGWidth = new JSpinner(new SpinnerNumberModel(0, 0, 3, 1));
        spnBGWidth.setEditor(new JSpinner.NumberEditor(spnBGWidth, "#"));
        spnBGWidth.putClientProperty("offset", 0);
        spnBGWidth.addChangeListener(onChange);
        add(frmBackground, lblBGWidth, 1, false, false, 2, 2, 0, 4);
        add(frmBackground, spnBGWidth, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblBGWidth);
        uiFont.add(spnBGWidth);

        // BG Height spinner
        JLabel lblBGHeight = new JLabel("Height");
        spnBGHeight = new JSpinner(new SpinnerNumberModel(0, 0, 3, 1));
        spnBGHeight.setEditor(new JSpinner.NumberEditor(spnBGHeight, "#"));
        spnBGHeight.putClientProperty("offset", 0);
        spnBGHeight.addChangeListener(onChange);
        add(frmBackground, lblBGHeight, 1, false, false, 2, 2, 0, 4);
        add(frmBackground, spnBGHeight, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblBGHeight);
        uiFont.add(spnBGHeight);
    }

    // Parameters group box constructor
    private void initParams() {
        frmParams = new JPanel(new GridBagLayout());
        frmParams.setBorder(new TitledBorder("Parameters"));
        add(panControls, frmParams, 0, false, true, 2, 0, 0, 0);
        uiFont.add(frmParams);

        // Event handlers
        ActionListener onAction = e->{ if (!updating) onParams(e); };
        ChangeListener onChange = e->{ if (!updating) onParams(e); };

        // Line spinner
        JLabel lblLine = new JLabel("Line");
        spnLine = new JSpinner(new SpinnerNumberModel(0, 0, 224, 1));
        spnLine.setEditor(new JSpinner.NumberEditor(spnLine, "#"));
        spnLine.addChangeListener(e->onLine());
        add(frmParams, lblLine, 1, false, false, 2, 2, 0, 4);
        add(frmParams, spnLine, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblLine);
        uiFont.add(spnLine);

        // Address text box
        JLabel lblPAddress = new JLabel("Address");
        txtPAddress = new JTextField();
        txtPAddress.addActionListener(
            e->{ panControls.requestFocus(); onPAddress(); });
        txtPAddress.addFocusListener(Util.onFocus(null, e->onPAddress()));
        add(frmParams, lblPAddress, 1, false, false, 2, 2, 0, 4);
        add(frmParams, txtPAddress, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblPAddress);
        uiFont.add(txtPAddress);

        // H-Bias Left spinner
        JLabel lblHLeft = new JLabel("Left");
        spnHLeft = new JSpinner(new SpinnerNumberModel(0, -4096, 4095, 1));
        spnHLeft.setEditor(new JSpinner.NumberEditor(spnHLeft, "#"));
        spnHLeft.putClientProperty("offset", 0);
        spnHLeft.addChangeListener(onChange);
        add(frmParams, lblHLeft, 1, false, false, 2, 2, 0, 4);
        add(frmParams, spnHLeft, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblHLeft); uiHBias.add(lblHLeft);
        uiFont.add(spnHLeft); uiHBias.add(spnHLeft);

        // H-Bias Right spinner
        JLabel lblHRight = new JLabel("Right");
        spnHRight = new JSpinner(new SpinnerNumberModel(0, -4096, 4095, 1));
        spnHRight.setEditor(new JSpinner.NumberEditor(spnHRight, "#"));
        spnHRight.putClientProperty("offset", 2);
        spnHRight.addChangeListener(onChange);
        add(frmParams, lblHRight, 1, false, false, 2, 2, 0, 4);
        add(frmParams, spnHRight, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblHRight); uiHBias.add(lblHRight);
        uiFont.add(spnHRight); uiHBias.add(spnHRight);

        // Affine Source X spinner
        JLabel lblABGX = new JLabel("Source X");
        spnABGX = new JSpinner(new SpinnerNumberModel(0, -4096, 4095.875, 1));
        spnABGX.setEditor(new JSpinner.NumberEditor(spnABGX, "#0.0##"));
        spnABGX.putClientProperty("offset", 0);
        spnABGX.addChangeListener(onChange);
        add(frmParams, lblABGX, 1, false, false, 2, 2, 0, 4);
        add(frmParams, spnABGX, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblABGX); uiAffine.add(lblABGX);
        uiFont.add(spnABGX); uiAffine.add(spnABGX);

        // Affine Source Y spinner
        JLabel lblABGY = new JLabel("Source Y");
        spnABGY = new JSpinner(new SpinnerNumberModel(0, -4096, 4095.875, 1));
        spnABGY.setEditor(new JSpinner.NumberEditor(spnABGY, "#0.0##"));
        spnABGY.putClientProperty("offset", 4);
        spnABGY.addChangeListener(onChange);
        add(frmParams, lblABGY, 1, false, false, 2, 2, 0, 4);
        add(frmParams, spnABGY, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblABGY); uiAffine.add(lblABGY);
        uiFont.add(spnABGY); uiAffine.add(spnABGY);

        // Affine Parallax spinner
        JLabel lblAParallax = new JLabel("Parallax");
        spnAParallax = new JSpinner(new SpinnerNumberModel(0,-32768,32767,1));
        spnAParallax.setEditor(new JSpinner.NumberEditor(spnAParallax, "#"));
        spnAParallax.putClientProperty("offset", 2);
        spnAParallax.addChangeListener(onChange);
        add(frmParams, lblAParallax, 1, false, false, 2, 2, 0, 4);
        add(frmParams, spnAParallax, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblAParallax); uiAffine.add(lblAParallax);
        uiFont.add(spnAParallax); uiAffine.add(spnAParallax);

        // Affine Vector X spinner
        JLabel lblAVectorX = new JLabel("Vector X");
        spnAVectorX = new JSpinner(new SpinnerNumberModel(
            0, -64, 63.998046875, 1));
        spnAVectorX.setEditor(new JSpinner.NumberEditor(spnAVectorX,"#0.0##"));
        spnAVectorX.putClientProperty("offset", 6);
        spnAVectorX.addChangeListener(onChange);
        add(frmParams, lblAVectorX, 1, false, false, 2, 2, 0, 4);
        add(frmParams, spnAVectorX, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblAVectorX); uiAffine.add(lblAVectorX);
        uiFont.add(spnAVectorX); uiAffine.add(spnAVectorX);

        // Affine Vector Y spinner
        JLabel lblAVectorY = new JLabel("Vector Y");
        spnAVectorY = new JSpinner(new SpinnerNumberModel(
            0, -64, 63.998046875, 1));
        spnAVectorY.setEditor(new JSpinner.NumberEditor(spnAVectorY,"#0.0##"));
        spnAVectorY.putClientProperty("offset", 8);
        spnAVectorY.addChangeListener(onChange);
        add(frmParams, lblAVectorY, 1, false, false, 2, 2, 0, 4);
        add(frmParams, spnAVectorY, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblAVectorY); uiAffine.add(lblAVectorY);
        uiFont.add(spnAVectorY); uiAffine.add(spnAVectorY);
    }

    // Objects group box constructor
    private void initObjects() {
        frmObjects = new JPanel(new GridBagLayout());
        frmObjects.setBorder(new TitledBorder("Objects"));
        add(panControls, frmObjects, 0, false, true, 2, 0, 0, 0);
        uiFont.add(frmObjects);

        // Group label
        JLabel lblGroup = new JLabel("Group");
        this.lblGroup = new JLabel("0");
        add(frmObjects, lblGroup, 1, false, false, 2, 4, 0, 4);
        add(frmObjects, this.lblGroup, 0, false, true, 2, 0, 0, 4);
        uiFont.add(lblGroup);
        uiFont.add(this.lblGroup);

        // Start label
        JLabel lblStart = new JLabel("Start");
        this.lblStart  = new JLabel("0");
        add(frmObjects, lblStart, 1, false, false, 2, 4, 0, 4);
        add(frmObjects, this.lblStart, 0, false, true, 2, 0, 0, 4);
        uiFont.add(lblStart);
        uiFont.add(this.lblStart);

        // End label
        JLabel lblEnd = new JLabel("End");
        this.lblEnd  = new JLabel("0");
        add(frmObjects, lblEnd, 1, false, false, 2, 4, 0, 4);
        add(frmObjects, this.lblEnd, 0, false, true, 2, 0, 0, 4);
        uiFont.add(lblEnd);
        uiFont.add(this.lblEnd);
    }

    // Options group box constructor
    private void initOptions() {
        JPanel frmOptions = new JPanel(new GridBagLayout());
        frmOptions.setBorder(new TitledBorder("Options"));
        add(panControls, frmOptions, 0, false, true, 2, 0, 0, 0);
        uiFont.add(frmOptions);

        // Scale slider
        JLabel lblScale = new JLabel("Scale");
        sldScale = new JSlider(1, 10, DEFAULTSCALE);
        sldScale.setFocusable(false);
        sldScale.setLabelTable(null);
        sldScale.setPreferredSize(new Dimension(0,
            sldScale.getPreferredSize().height));
        sldScale.setSnapToTicks(true);
        sldScale.addChangeListener(e->{ if (!updating) onScale(); });
        add(frmOptions, lblScale, 1, false, false, 2, 4, 0, 4);
        add(frmOptions, sldScale, 0, false, true, 2, 0, 0, 4);
        uiFont.add(lblScale);

        // Generic check box
        chkGeneric = new JCheckBox("Generic colors");
        chkGeneric.setBorder(null);
        chkGeneric.addActionListener(e->onGeneric());
        add(frmOptions, chkGeneric, 0, false, false, 2, 2, 0, 2);
        uiFont.add(chkGeneric);
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Event Handlers                             //
    ///////////////////////////////////////////////////////////////////////////

    // Address text box
    private void onAddress() {
        String text  = txtAddress.getText();
        int    entry = 0;

        // Process the text into an address
        try { entry = (int) Long.parseLong(text, 16); }
        catch (Exception x) { }
        int address = entry & 0x0007FFFF;

        // Update the object index
        if ((entry >> 24 & 7) == 0 && address >= 0x0003D800 &&
            address < 0x0003DC00)
            index = address - 0x0003D800 >> 5;
        setIndex(index);
    }

    // World attribute modified
    private void onEdit(EventObject e) {
        int    offset = 0x0003D800 | index << 5 | (Integer)
            ((JComponent) e.getSource()).getClientProperty("offset");
        byte[] vram   = parent.getVRAM();
        int    orig   = vram[offset] & 0xFF | (vram[offset + 1] & 0xFF) << 8;
        int    value  = orig;

        // Compose value by offset
        switch (offset & 31) {
            case 0: value =
                (chkLeft.isSelected()      ? 0x8000 : 0) |
                (chkRight.isSelected()     ? 0x4000 : 0) |
                cmbMode.getSelectedIndex()       << 12   |
                (Integer) spnBGWidth.getValue()  << 10   |
                (Integer) spnBGHeight.getValue() <<  8   |
                (chkOverplane.isSelected() ? 0x0080 : 0) |
                (chkEnd.isSelected()       ? 0x0040 : 0) |
                (Integer) spnBGBase.getValue();
                break;
            case  2: value = (Integer)spnX.getValue()          & 0x03FF; break;
            case  4: value = (Integer)spnParallax.getValue()   & 0x03FF; break;
            case  6: value = (Integer)spnY.getValue()          & 0xFFFF; break;
            case  8: value = (Integer)spnBGX.getValue()        & 0x1FFF; break;
            case 10: value = (Integer)spnBGParallax.getValue() & 0x7FFF; break;
            case 12: value = (Integer)spnBGY.getValue()        & 0x1FFF; break;
            case 14: value = (Integer)spnWidth.getValue()  - 1 & 0x1FFF; break;
            case 16: value = (Integer)spnHeight.getValue() - 1 & 0xFFFF; break;
            case 18:
                try {
                    value = (int) Long.parseLong(txtParamAddr.getText(), 16);
                    int region = value >> 24 & 7;
                    value &= 0x0007FFFF;
                    if (region!=0 || value<0x00020000 || value>0x0003FFFF)
                        throw new RuntimeException();
                    value = value - 0x00020000 >> 1;
                } catch (Exception E) { value = orig; }
                break;
            case 20: value = (Integer)spnOverchar.getValue()   & 0xFFFF; break;
        }

        // No action if nothing changed
        if (value == orig)
            return;

        // Push the new value into the emulation state
        parent.getVUE().write(offset, VUE.U16, value, true);
        parent.refresh(false);
    }

    // Generic check box
    private void onGeneric() {
        generic = chkGeneric.isSelected();
        refreshImage();
        panPreview.repaint();
    }

    // Index spinner
    private void onIndex() {
        setIndex((Integer) spnIndex.getValue());
    }

    // Line spinner
    private void onLine() {
        line = (Integer) spnLine.getValue();
        updating = true;
        refreshParams();
        updating = false;
    }

    // Parameters Address text box
    private void onPAddress() {
        int    size  = worlds[index].mode == 1 ? 4 : 16;
        int    base  = 0x00020000 | worlds[index].paramAddr << 1;
        String text  = txtPAddress.getText();
        int    entry = 0;

        // Process the text into an address
        try { entry = (int) Long.parseLong(text, 16); }
        catch (Exception x) { }
        int address = entry & 0x0007FFFF;

        // Update the object index
        if ((entry >> 24 & 7) == 0 && address >= base &&
            address < 0x0003FFFF)
            line = Math.min((address - base) / size, 223);
        refreshParams();
    }

    // Painting the controls scroll pane
    private void onPaintControls() {
        // Determine the size of the inner element and the visible area
        Dimension extent = scrControls.getViewport().getExtentSize();
        Dimension prefer = panControls.getPreferredSize();
        if (extent.width == prefer.width)
            return;

        // Resize the container to exactly contain the inner element
        prefer.width  = scrControls.getWidth() + prefer.width - extent.width;
        prefer.height = 0;
        scrControls.setPreferredSize(prefer);
        scrControls.revalidate();
    }

    // Painting the preview pane
    private void onPaintPreview(Graphics g) {
        g.drawImage(imgFrame, 0, 0, 384 * scale, 224 * scale, null);
    }

    // Scanline parameter modified
    private void onParams(EventObject e) {
        int    size    = worlds[index].mode == 1 ? 4 : 16;
        int    offset  = 0x00020000 | worlds[index].paramAddr << 1;
               offset += size * line + (Integer)
            ((JComponent) e.getSource()).getClientProperty("offset");
        byte[] vram   = parent.getVRAM();
        int    orig   = vram[offset] & 0xFF | (vram[offset + 1] & 0xFF) << 8;
        int    value  = orig;

        // Compose H-bias value by offset
        if (size == 4) switch (offset & 3) {
            case 0: value = (Integer) spnHLeft.getValue();  break;
            case 2: value = (Integer) spnHRight.getValue(); break;
        }

        // Compose Affine value by offset
        else switch (offset & 15) {
            case 0:
                value = (int) Math.round((Double) spnABGX.getValue() * 8);
                break;
            case 2: value = (Integer) spnAParallax.getValue(); break;
            case 4:
                value = (int) Math.round((Double) spnABGY.getValue() * 8);
                break;
            case 6:
                value = (int) Math.round((Double) spnAVectorX.getValue()*512);
                break;
            case 8:
                value = (int) Math.round((Double) spnAVectorY.getValue()*512);
                break;
        }

        // No action if nothing changed
        if (value == orig)
            return;

        // Push the new value into the emulation state
        parent.getVUE().write(offset, VUE.U16, value, true);
        parent.refresh(false);
    }

    // Scale slider changed
    private void onScale() {
        setScale(sldScale.getValue());
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Package Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Update UI components with the current emulation state
    void refresh() {

        // Update state fields
        for (int x = 0; x < 32; x++)
            worlds[x].refresh();

        // Update UI elements
        panPreview.repaint();
        setIndex(index);
    }

    // Specify new fonts
    void setFonts() {
        Font      dlgFont = parent.getDialogFont();
        Font      hexFont = parent.getHexFont();
        Dimension size    = spnX.getPreferredSize();

        // Update all appropriate UI controls
        for (int x = 0; x < uiFont.size(); x++) {
            JComponent control = uiFont.get(x);
            if (control instanceof JTextField)
                control.setFont(hexFont);
            else if (control instanceof JPanel)
                ((TitledBorder) control.getBorder()).setTitleFont(dlgFont);
            else control.setFont(dlgFont);
            if (control instanceof JSpinner)
                control.setPreferredSize(size);
        }
    }

    // Specify the current selected world index
    void setIndex(int index) {
        this.index       = index;
        int      address = 0x0003D800 | index << 5;
        VIPWorld world   = worlds[index];
        int      params  = 0x00020000 | world.paramAddr << 1;

        // Update controls
        updating = true;
        spnIndex.setValue(index);
        txtAddress.setText(String.format("%08X", address));
        chkLeft.setSelected(world.left);
        chkRight.setSelected(world.right);
        cmbMode.setSelectedIndex(world.mode);
        spnBGWidth.setValue(world.bgWidth);
        spnBGHeight.setValue(world.bgHeight);
        chkOverplane.setSelected(world.overplane);
        chkEnd.setSelected(world.end);
        spnBGBase.setValue(world.bgBase);
        spnX.setValue(world.x);
        spnParallax.setValue(world.parallax);
        spnY.setValue(world.y);
        spnBGX.setValue(world.bgX);
        spnBGParallax.setValue(world.bgParallax);
        spnBGY.setValue(world.bgY);
        spnWidth.setValue(world.width + 1);
        spnHeight.setValue(world.height + 1);
        txtParamAddr.setText(String.format("%08X", params));
        spnOverchar.setValue(world.overchar);
        //frmBackground.setVisible(world.mode != 3);
        refreshParams();
        refreshObjects();
        updating = false;

        // Update display
        refreshImage();
        panPreview.repaint();
    }

    // Specify a new scaling factor for the preview
    void setScale(int scale) {
        this.scale = scale;

        // Update slider
        updating = true;
        sldScale.setValue(scale);
        updating = false;

        // Configure preview
        resizePreview();
        panPreview.revalidate();

        // Configure map scroll pane
        scale = scale * 8;
        scrPreview.getHorizontalScrollBar().setUnitIncrement(scale);
        scrPreview.getVerticalScrollBar().setUnitIncrement(scale);
        scrPreview.repaint();
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Private Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Add a control to a container
    private void add(JComponent container, JComponent control, int colSpan,
        boolean center, boolean stretch, int top, int left, int bottom,
        int right) {
        GridBagConstraints c = new GridBagConstraints();
        c.gridwidth = colSpan != 0 ? colSpan : GridBagConstraints.REMAINDER;
        c.insets    = new Insets(top, left, bottom, right);
        c.weightx   = colSpan != 0 ? 0 : 1;
        if (!center) c.anchor = GridBagConstraints.LINE_START;
        if (stretch) c.fill   = GridBagConstraints.HORIZONTAL;
        container.add(control, c);
    }

    // Finalize a container layout
    private void finish(JComponent container, int spacing) {
        JPanel spacer = new JPanel(null);
        spacer.setPreferredSize(new Dimension(0, 0));
        spacer.setOpaque(false);
        GridBagConstraints c = new GridBagConstraints();
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.insets    = new Insets(spacing, 0, 0, 0);
        c.weighty   = 1;
        container.add(spacer, c);
    }

    // Determine whether this world displays objects
    private boolean isObjectWorld(int x) {
        VIPWorld world = worlds[x];
        return !world.end && (world.left || world.right) && world.mode == 3;
    }

    // Render the world into the frame image
    private void refreshImage() {
        VIPWorld world = worlds[index];

        // Determine the world's object group
        int group = 3;
        for (int x = 31; x > index; x--) {
            if (worlds[x].end) {
                group = -1;
                break;
            }
            if (worlds[x].isObjectWorld())
                group = group - 1 & 3;
        }

        // Retrieve object group pointers
        int[] groups = new int[4];
        VUE   vue    = parent.getVUE();
        for (int x = 0; x < 4; x++)
            groups[x] = vue.read(0x0005F848 | x << 1, VUE.U16, true);

        // Draw the frame image
        int[] pix = new int[384 * 224];
        for (int y = 0, o = 0; y < 224; y++)
        for (int x = 0; x < 384; x++, o++) {
            int left  = world.sample(x, y, 0, group, groups);
            int right = world.sample(x, y, 1, group, groups);
            pix[o] = parent.getColor(left,  0, generic) |
                     parent.getColor(right, 1, generic);
        }
        imgFrame.setRGB(0, 0, 384, 224, pix, 0, 384);
    }

    // Update object controls
    private void refreshObjects() {

        // Not an object world
        if (!isObjectWorld(index)) {
            frmObjects.setVisible(false);
            return;
        }

        // Determine which object group displays in this world
        int group = 3;
        for (int x = 31; x > index; x--) {

            // Worlds have stopped drawing by this point
            if (worlds[x].end) {
                frmObjects.setVisible(false);
                return;
            }

            // Check whether the world is an object world
            if (isObjectWorld(x))
                group = group - 1 & 3;
        }

        // Retrieve the object group pointers from VIP memory
        VUE vue = parent.getVUE();
        int[] groups = new int[4];
        for (int x = 0; x < 4; x++)
            groups[x] = vue.read(0x0005F848 | x << 1, VUE.U16, true) & 1023;

        // Update controls
        frmObjects.setVisible(true);
        lblGroup.setText("" + group);
        lblStart.setText("" + (group == 0 ? 0 : groups[group - 1] + 1 & 1023));
        lblEnd.setText("" + groups[group]);
    }

    // Update parameter controls
    private void refreshParams() {
        VIPWorld world = worlds[index];

        // Configure control visibility
        boolean hbias  = world.mode == 1;
        boolean affine = world.mode == 2;
        frmParams.setVisible(hbias || affine);
        for (int x = 0; x < uiHBias.size(); x++)
            uiHBias.get(x).setVisible(hbias);
        for (int x = 0; x < uiAffine.size(); x++)
            uiAffine.get(x).setVisible(affine);
        panControls.revalidate();

        // No further action if no controls are visible
        if (!(hbias || affine) || world.height < 0)
            return;

        // Update controls
        int[] params = world.getParams(null, line);
        spnLine.setValue(line);
        txtPAddress.setText(String.format("%08X", params[5]));
        spnHLeft.setValue(params[0]);
        spnHRight.setValue(params[1]);
        spnABGX.setValue(params[0] / 8.0);
        spnABGY.setValue(params[2] / 8.0);
        spnAParallax.setValue(params[1]);
        spnAVectorX.setValue(params[3] / 512.0);
        spnAVectorY.setValue(params[4] / 512.0);
    }

    // Resize the contents of the preview pane
    private void resizePreview() {
        panPreview.setPreferredSize(new Dimension(384 * scale, 224 * scale));
    }

}
