package util;

// Java imports
import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;
import java.io.*;
import java.util.*;
import javax.imageio.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.plaf.basic.*;

// Miscellaneous utility methods
public final class Util {

    // Static fields
    private static Font       defaultFont; // Default UI dialog font
    private static String[]   fontNames;   // Full list of local font families
    private static Graphics2D G;           // Graphics context for font metrics
    private static HashMap<String, Color> colors; // Colors cache



    ///////////////////////////////////////////////////////////////////////////
    //                            Event Handlers                             //
    ///////////////////////////////////////////////////////////////////////////

    // Focus, Blur
    public interface FocusHandler {
        public void onEvent(FocusEvent e);
    }

    // Mouse Down, Mouse Up, Mouse Move, Mouse Drag
    public interface MouseHandler {
        public void onEvent(MouseEvent e);
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Static Methods                             //
    ///////////////////////////////////////////////////////////////////////////

    // Static constructor
    static {

        // Use the system look and feel if available
        try { UIManager.setLookAndFeel(
            UIManager.getSystemLookAndFeelClassName() ); }
        catch (Exception e) { }

        // Graphics context for font metrics
        G = (Graphics2D) (new BufferedImage(
            1, 1, BufferedImage.TYPE_INT_ARGB).getGraphics());

        // Configure fonts
        defaultFont = new JLabel().getFont();
        refreshFonts();

        // Global colors
        colors = new HashMap<String, Color>();
        colors.put("control",       SystemColor.control);
        colors.put("controlText",   SystemColor.controlText);
        colors.put("highlight",     SystemColor.controlHighlight);
        colors.put("selected",      SystemColor.textHighlight);
        colors.put("selectedText",  SystemColor.textHighlightText);
        colors.put("shadow",        SystemColor.controlShadow);
        colors.put("window",        SystemColor.window);
        colors.put("windowText",    SystemColor.windowText);
    }

    // Adjust JSplitPane borders and divider size
    public static void adjustSplitPane(JSplitPane split) {
        split.setDividerSize(2);
        split.setContinuousLayout(true);
        split.setUI(new BasicSplitPaneUI() {
            public BasicSplitPaneDivider createDefaultDivider() {
                return new BasicSplitPaneDivider(this) {
                    public void setBorder(Border b) { }
                };
            }
        });
        split.setBorder(null);
    }

    // Load a file from the local filesystem
    public static byte[] fileLoad(File file) {
        try { return load(new FileInputStream(file)); }
        catch (Exception e) { }
        return null;
    }

    // Load a file from the JAR or local filesystem
    public static byte[] fileLoad(String filename) {
        byte[] ret = load(Util.class.getResourceAsStream(filename));
        return ret != null ? ret : fileLoad(new File(filename));
    }

    // Determine whether a font is available
    public static boolean fontExists(String name) {
        return Arrays.binarySearch(fontNames, name,
            String.CASE_INSENSITIVE_ORDER) >= 0;
    }

    // Modify a color to have transparency
    public static Color getColor(Color ref, float alpha) {
        return new Color(
            (int) Math.round(alpha * 255) << 24 |
            ref.getRGB() & 0x00FFFFFF,
        true);
    }

    // Retrieve a color with alpha from the colors cache
    public static Color getColor(String name, float alpha) {

        // Retrieve the cached color
        String key = name + (alpha == 1.0f ? "" : alpha);
        Color  ret = colors.get(key);
        if (ret != null) return ret;

        // Retrieve the base color by name
        ret = colors.get(name);
        if (ret == null) return null;

        // Store the new modified color
        ret = getColor(ret, alpha);
        colors.put(key, ret);
        return ret;
    }

    // Retrieve a color with full opacity from the colors cache
    public static Color getColor(String name) {
        return getColor(name, 1.0f);
    }

    // Retrieve the default font used by window controls
    public static Font getDefaultFont() {
        return defaultFont;
    }

    // Retrieve a font for a particular size in pixels
    public static Font getFont(String name, int style, int height) {

        // Perform a binary search to find the best font size
        int min =   1;
        int max = 100;
        for (;;) {

            // Determine the value to compare
            int         mid     = min + (max - min) / 2;
            Font        ret     = new Font(name, style, mid);
            FontMetrics metrics = getFontMetrics(ret);
            int         check   = metrics.getHeight();

            // Compare the value with the specified height
            if (check < height) min = mid;
            if (check > height) max = mid;
            if (check == height || min >= max - 1)
                return ret;
        }

    }

    // Determine the metrics for a font
    public static FontMetrics getFontMetrics(Font font) {
        return G.getFontMetrics(font);
    }

    // Return a list of all supported font names (sorted case-insensitive)
    public static String[] getFonts() {
        return Arrays.copyOf(fontNames, fontNames.length);
    }

    // Load an image from a file
    public static BufferedImage imageLoad(File file) {
        try { return ImageIO.read(file); }
        catch (Exception e) { }
        return null;
    }

    // Placebo method for invoking the static sonstructor
    public static void initialize() { }

    // Produce a FocusListener using functional interfaces
    public static FocusListener onFocus(FocusHandler focus, FocusHandler blur){
        return new FocusListener() {
            public void focusGained(FocusEvent e)
                { if (focus != null) focus.onEvent(e); }
            public void focusLost  (FocusEvent e)
                { if (blur  != null) blur.onEvent(e); }
        };
    }

    // Produce a MouseListener using functional interfaces
    public static MouseListener onMouse(MouseHandler down,
        MouseHandler up) {
        return new MouseListener() {
            public void mouseClicked (MouseEvent e) { }
            public void mouseEntered (MouseEvent e) { }
            public void mouseExited  (MouseEvent e) { }
            public void mousePressed (MouseEvent e)
                { if (down != null) down.onEvent(e); }
            public void mouseReleased(MouseEvent e)
                { if (up   != null) down.onEvent(e); }
        };
    }

    // Produce a MouseMotionListener using functional interfaces
    public static MouseMotionListener onMouseMotion(MouseHandler move,
        MouseHandler drag) {
        return new MouseMotionListener() {
            public void mouseDragged(MouseEvent e)
                { if (drag != null) drag.onEvent(e); }
            public void mouseMoved  (MouseEvent e)
                { if (move != null) move.onEvent(e); }
        };
    }

    // Regenerate the list of font names
    public static void refreshFonts() {
        fontNames = GraphicsEnvironment.getLocalGraphicsEnvironment(
            ).getAvailableFontFamilyNames();
        Arrays.sort(fontNames, String.CASE_INSENSITIVE_ORDER);
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Private Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Load all data from a stream into a byte buffer
    private static byte[] load(InputStream stream) {
        ByteArrayOutputStream out    = new ByteArrayOutputStream();
        byte[]                buffer = new byte[64 * 1024];

        try {

            // Read from the stream until EOF is reached
            for (;;) {
                int x = stream.read(buffer, 0, buffer.length);
                if (x == -1) break;
                out.write(buffer, 0, x);
            }

            // Return the loaded contents
            stream.close();
            return out.toByteArray();
        } catch (Exception e) { }

        // An error occurred
        return null;
    }

}