package vue;

// Video processor
class JavaVIP {

    // Instance fields
    JavaVUE parent; // Parent emulation context
    short[] ram;    // VIP memory

    // I/O registers
    int[] brt;    // Brightness Control Registers and Rest Control Register
    int   bkcol;  // BG Color Palette Control Register
    int   cta;    // Column Table Read Start Address
    int   frmcyc; // Game Frame Control Register
    int[] gplt;   // BG Palette Control Register
    int   intenb; // Interrupt Enable
    int   intpnd; // Interrupt Pending
    int[] jplt;   // OBJ Palette Control Registers
    int[] spt;    // OBJ Control Registers

    // DPSTTS
    boolean dp_disp;  // Display is enabled
    boolean dp_lock;  // CTA is frozen
    boolean dp_re;    // Memory refresh is enabled
    boolean dp_synce; // Displays cannot be shown

    // XPSTTS
    boolean xp_xpen;  // Drawing is enabled



    ///////////////////////////////////////////////////////////////////////////
    //                               Constants                               //
    ///////////////////////////////////////////////////////////////////////////

    // Interrupt register flags
    private static final int SCANERR    = 0x0001;
    private static final int LFBEND     = 0x0002;
    private static final int RFBEND     = 0x0004;
    private static final int GAMESTART  = 0x0008;
    private static final int FRAMESTART = 0x0010;
    private static final int SBHIT      = 0x2000;
    private static final int XPEND      = 0x4000;
    private static final int TIMEERR    = 0x8000;

    // Display register flags
    private static final int DPRST   = 0x0001;
    private static final int DISP    = 0x0002;
    private static final int L0BSY   = 0x0004;
    private static final int R0BSY   = 0x0008;
    private static final int L1BSY   = 0x0010;
    private static final int R1BSY   = 0x0020;
    private static final int SCANRDY = 0x0040;
    private static final int FCLK    = 0x0080;
    private static final int RE      = 0x0100;
    private static final int SYNCE   = 0x0200;
    private static final int LOCK    = 0x0400;

    // Drawing register flags
    private static final int XPRST    = 0x0001;
    private static final int XPEN     = 0x0002;
    private static final int F0BSY    = 0x0004;
    private static final int F1BSY    = 0x0008;
    private static final int OVERTIME = 0x0010;
    private static final int SBOUT    = 0x8000;



    ///////////////////////////////////////////////////////////////////////////
    //                             Constructors                              //
    ///////////////////////////////////////////////////////////////////////////

    // Package constructor
    JavaVIP(JavaVUE parent) {
        brt         = new int[4];
        gplt        = new int[4];
        jplt        = new int[4];
        this.parent = parent;
        spt         = new int[4];
        ram         = new short[0x20000];
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Package Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Read a value from the VIP bus
    int read(int address, int type, boolean debug) {
        address &= 0x0007FFFF;

        // RAM
        if (address < 0x00040000)
            return readMemory(address, type, debug);

        // Unmapped
        if (address < 0x0005E000)
            return 0;

        // I/O registers
        if (address < 0x00060000) {
            int value = readRegister(address, debug);
            if (type == VUE.S32) value = value & 0x0000FFFF |
                readRegister(address + 2, debug) << 16;
            return value;
        }

        // Unmapped
        if (address < 0x00078000)
            return 0;

        // Character memory mirrors
             if (address < 0x0007A000) address -= 0x00072000;
        else if (address < 0x0007C000) address -= 0x0006C000;
        else if (address < 0x0007E000) address -= 0x00066000;
        else                           address -= 0x00060000;
        return readMemory(address, type, debug);
    }

    // Initialize VIP for system reset
    void reset() {

        // Reset memory to zeroes to aid with debugging
        // The hardware does not do this
        for (int x = 0; x < ram.length; x++)
            ram[x] = 0;

        // Initialize VIP state
        dp_synce = false;
        dp_re    = false;
        dp_disp  = false;
        intenb   = 0;
        xp_xpen  = false;

        // Reset all other state as well
        // The hardware does not do this
        bkcol   = 0;
        cta     = 0;
        frmcyc  = 0;
        brt [0] = brt [1] = brt [2] = brt [3] = 0;
        gplt[0] = gplt[1] = gplt[2] = gplt[3] = 0;
        jplt[0] = jplt[1] = jplt[2] = jplt[3] = 0;
        spt [0] = spt [1] = spt [2] = spt [3] = 0;
    }

    // Write a value to the VIP bus
    void write(int address, int type, int value, boolean debug) {
        address &= 0x0007FFFF;

        // RAM
        if (address < 0x00040000) {
            writeMemory(address, type, value, debug);
            return;
        }

        // Unmapped
        if (address < 0x0005E000)
            return;

        // I/O registers
        if (address < 0x00060000) {
            writeRegister(address, value, debug);
            if (type == VUE.S32)
                writeRegister(address + 2, value >> 16, debug);
            return;
        }

        // Unmapped
        if (address < 0x00078000)
            return;

        // Character memory mirrors
             if (address < 0x0007A000) address -= 0x00072000;
        else if (address < 0x0007C000) address -= 0x0006C000;
        else if (address < 0x0007E000) address -= 0x00066000;
        else                           address -= 0x00060000;
        writeMemory(address, type, value, debug);
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Private Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Simulate a read of DPSTTS
    private int readDPSTTS() {
        return
            (dp_lock  ? LOCK  : 0) |
            (dp_synce ? SYNCE : 0) |
            (dp_re    ? RE    : 0) |
            SCANRDY                |
            (dp_disp  ? DISP  : 0)
        ;
    }

    // Read a value from VIP RAM
    private int readMemory(int address, int type, boolean debug) {
        int value = ram[address >> 1];
        switch (type) {
            case VUE.S8:  case VUE.U8: return parent.tweakValue(
                (address & 1) == 0 ? value : value >> 8, type);
            case VUE.S16: case VUE.U16:
                return parent.tweakValue(value, type);
        }
        return value & 0xFFFF | ram[(address >> 1) + 1] << 16;
    }

    // Read an I/O register as a halfword
    private int readRegister(int address, boolean debug) {
        switch (address >> 1) {
            case 0x0005F800 >> 1: return intpnd;       // INTPND
            case 0x0005F802 >> 1: return intenb;       // INTENB
            case 0x0005F804 >> 1: return 0;            // INTCLR
            case 0x0005F820 >> 1: return readDPSTTS(); // DPSTTS
            case 0x0005F822 >> 1: return 0;            // DPCTRL
            case 0x0005F824 >> 1: return brt[0];       // BRTA
            case 0x0005F826 >> 1: return brt[1];       // BRTB
            case 0x0005F828 >> 1: return brt[2];       // BRTC
            case 0x0005F82A >> 1: return brt[3];       // REST
            case 0x0005F82E >> 1: return frmcyc;       // FRMCYC
            case 0x0005F830 >> 1: return cta;          // CTA
            case 0x0005F840 >> 1: return readXPSTTS(); // XPSTTS
            case 0x0005F842 >> 1: return 0;            // XPCTRL
            case 0x0005F844 >> 1: return 2;            // VER
            case 0x0005F848 >> 1: return spt[0];       // SPT0
            case 0x0005F84A >> 1: return spt[1];       // SPT1
            case 0x0005F84C >> 1: return spt[2];       // SPT2
            case 0x0005F84E >> 1: return spt[3];       // SPT3
            case 0x0005F860 >> 1: return gplt[0];      // GPLT0
            case 0x0005F862 >> 1: return gplt[1];      // GPLT1
            case 0x0005F864 >> 1: return gplt[2];      // GPLT2
            case 0x0005F866 >> 1: return gplt[3];      // GPLT3
            case 0x0005F868 >> 1: return jplt[0];      // JPLT0
            case 0x0005F86A >> 1: return jplt[1];      // JPLT1
            case 0x0005F86C >> 1: return jplt[2];      // JPLT2
            case 0x0005F86E >> 1: return jplt[3];      // JPLT3
            case 0x0005F870 >> 1: return bkcol;        // BKCOL

            // Unmapped -- Explicit case labels to encourage tableswitch
            case 0x0005F806 >> 1: case 0x0005F808 >> 1: case 0x0005F80A >> 1:
            case 0x0005F80C >> 1: case 0x0005F80E >> 1: case 0x0005F810 >> 1:
            case 0x0005F812 >> 1: case 0x0005F814 >> 1: case 0x0005F816 >> 1:
            case 0x0005F818 >> 1: case 0x0005F81A >> 1: case 0x0005F81C >> 1:
            case 0x0005F81E >> 1: case 0x0005F82C >> 1: case 0x0005F832 >> 1:
            case 0x0005F834 >> 1: case 0x0005F836 >> 1: case 0x0005F838 >> 1:
            case 0x0005F83A >> 1: case 0x0005F83C >> 1: case 0x0005F83E >> 1:
            case 0x0005F846 >> 1: case 0x0005F850 >> 1: case 0x0005F852 >> 1:
            case 0x0005F854 >> 1: case 0x0005F856 >> 1: case 0x0005F858 >> 1:
            case 0x0005F85A >> 1: case 0x0005F85C >> 1: case 0x0005F85E >> 1:
        }

        return 0;
    }

    // Simulate a read of XPSTTS
    private int readXPSTTS() {
        return
            28 << 8 | // SBCOUNT
            (xp_xpen ?  XPEN : 0)
        ;
    }

    // Simulate a write to DPCTRL
    private void writeDPCTRL(int value) {

        // Parse flags
        dp_lock  = (value & LOCK ) != 0;
        dp_synce = (value & SYNCE) != 0;
        dp_re    = (value & RE   ) != 0;
        dp_disp  = (value & DISP ) != 0;
        if ((value & DPRST) == 0) return;

        // Clear interrupts
        int mask = ~(TIMEERR|FRAMESTART|GAMESTART|RFBEND|LFBEND|SCANERR);
        intenb  &= mask;
        intpnd  &= mask;
    }

    // Write a value into VIP RAM
    private void writeMemory(int address, int type, int value, boolean debug) {
        switch (type) {

            // Byte
            case VUE.S8:  case VUE.U8:
                int prev = ram[address >> 1];
                ram[address >> 1] = (address & 1) == 0 ?
                    (short) (prev & 0xFF00 | value &  0x00FF) :
                    (short) (prev & 0x00FF | value << 8     );
                return;

            // Halfword
            case VUE.S16: case VUE.U16:
                ram[address >> 1] = (short) value;
                return;
        }

        // Word
        ram[ address >> 1     ] = (short) value;
        ram[(address >> 1) + 1] = (short) (value >> 16);
    }

    // Write an I/O register as a halfword
    private void writeRegister(int address, int value, boolean debug) {
        switch (address >> 1) {
            case 0x0005F800 >> 1:                                  
                       if (debug) intpnd = value & 0xE01F;  return; // INTPND
            case 0x0005F802 >> 1: intenb = value & 0xE01F;  return; // INTENB
            case 0x0005F804 >> 1: intpnd &= ~value;         return; // INTCLR
            case 0x0005F820 >> 1:                           return; // DPSTTS
            case 0x0005F822 >> 1: writeDPCTRL(value);       return; // DPCTRL
            case 0x0005F824 >> 1: brt[0] = value & 0x00FF;  return; // BRTA
            case 0x0005F826 >> 1: brt[1] = value & 0x00FF;  return; // BRTB
            case 0x0005F828 >> 1: brt[2] = value & 0x00FF;  return; // BRTC
            case 0x0005F82A >> 1: brt[3] = value & 0x00FF;  return; // REST
            case 0x0005F82E >> 1: frmcyc = value & 0x000F;  return; // FRMCYC
            case 0x0005F830 >> 1: if (debug) cta = value;   return; // CTA
            case 0x0005F840 >> 1:                           return; // XPSTTS
            case 0x0005F842 >> 1: writeXPCTRL(value);       return; // XPCTRL
            case 0x0005F844 >> 1:                           return; // VER
            case 0x0005F848 >> 1: spt[0]  = value & 0x03FF; return; // SPT0
            case 0x0005F84A >> 1: spt[1]  = value & 0x03FF; return; // SPT1
            case 0x0005F84C >> 1: spt[2]  = value & 0x03FF; return; // SPT2
            case 0x0005F84E >> 1: spt[3]  = value & 0x03FF; return; // SPT3
            case 0x0005F860 >> 1: gplt[0] = value & 0x00FC; return; // GPLT0
            case 0x0005F862 >> 1: gplt[1] = value & 0x00FC; return; // GPLT1
            case 0x0005F864 >> 1: gplt[2] = value & 0x00FC; return; // GPLT2
            case 0x0005F866 >> 1: gplt[3] = value & 0x00FC; return; // GPLT3
            case 0x0005F868 >> 1: jplt[0] = value & 0x00FC; return; // JPLT0
            case 0x0005F86A >> 1: jplt[1] = value & 0x00FC; return; // JPLT1
            case 0x0005F86C >> 1: jplt[2] = value & 0x00FC; return; // JPLT2
            case 0x0005F86E >> 1: jplt[3] = value & 0x00FC; return; // JPLT3
            case 0x0005F870 >> 1: bkcol   = value & 0x0003; return; // BKCOL

            // Unmapped -- Explicit case labels to encourage tableswitch
            case 0x0005F806 >> 1: case 0x0005F808 >> 1: case 0x0005F80A >> 1:
            case 0x0005F80C >> 1: case 0x0005F80E >> 1: case 0x0005F810 >> 1:
            case 0x0005F812 >> 1: case 0x0005F814 >> 1: case 0x0005F816 >> 1:
            case 0x0005F818 >> 1: case 0x0005F81A >> 1: case 0x0005F81C >> 1:
            case 0x0005F81E >> 1: case 0x0005F82C >> 1: case 0x0005F832 >> 1:
            case 0x0005F834 >> 1: case 0x0005F836 >> 1: case 0x0005F838 >> 1:
            case 0x0005F83A >> 1: case 0x0005F83C >> 1: case 0x0005F83E >> 1:
            case 0x0005F846 >> 1: case 0x0005F850 >> 1: case 0x0005F852 >> 1:
            case 0x0005F854 >> 1: case 0x0005F856 >> 1: case 0x0005F858 >> 1:
            case 0x0005F85A >> 1: case 0x0005F85C >> 1: case 0x0005F85E >> 1:
        }

    }

    // Simulate a write to XPCTRL
    private void writeXPCTRL(int value) {

        // Parse flags
        xp_xpen = (value & XPEN) != 0;
        if ((value & XPRST) == 0) return;

        // Clear interrupts
        int mask = ~(TIMEERR | XPEND | SBHIT);
        intenb  &= mask;
        intpnd  &= mask;
    }

}
