package vue;
import vue.*;

// Java implementation of emulation core
class JavaVUE implements VUE {

    // Emulation fields
    JavaCPU cpu; // CPU state
    JavaVIP vip; // Video processor

    // Memory
    byte[] rom;  // ROM buffer
    byte[] sram; // SRAM buffer
    byte[] wram; // WRAM buffer

    // Instance fields
    Integer breakCode; // Application-supplied break code



    ///////////////////////////////////////////////////////////////////////////
    //                             Constructors                              //
    ///////////////////////////////////////////////////////////////////////////

    // Default constructor
    JavaVUE() {

        // Configure emulation fields
        cpu = new JavaCPU(this);
        vip = new JavaVIP(this);

        // Configure memory
        rom  = null;
        sram = null;
        wram = new byte[0x10000];

        // Configure instance fields
        breakCode = null;

        // Startup tasks
        reset();
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Public Methods                             //
    ///////////////////////////////////////////////////////////////////////////

    // Process the simulation
    public int emulate(int cycles) {

        // Initialize state prior to simulation
        breakCode = null;

        // Process for at least as many cycles as specified
        while (cycles > 0) {

            // Get cycles until next interrupt/auto break
            if (cpu.halt == JavaCPU.HALT) {
                cpu.stage  = JavaCPU.INTERRUPT;
                cpu.cycles = cycles;
                // cpu.cycles = Math.min(cpu.cycles, vip.untilIRQ());
                // ...
            }

            // Process by pipeline stage
            else outer: for(;;) {
                switch (cpu.stage) {
                    case JavaCPU.FETCH16: cpu.fetch16(); break;
                    case JavaCPU.FETCH32: cpu.fetch32(); break;
                    case JavaCPU.EXECUTE: cpu.execute(); break;
                    default: break outer;
                }
                if (breakCode != null)
                    return cycles;
            }

            // Process subsystems
            if (cpu.cycles > 0) {
                // vip.emulate(cpu.cycles);
                // ...
                cycles     -= cpu.cycles;
                cpu.cycles  = 0;
            }

            // Check for interrupts and exceptions
            switch (cpu.stage) {
                case JavaCPU.INTERRUPT: cpu.interrupt(); // Fallthrough
                case JavaCPU.EXCEPTION: cpu.exception();
            }
            if (breakCode != null)
                return cycles;

        }

        // Automatic break
        return cycles;
    }

    // Retrieve the application-supplied break code
    public Integer getBreakCode() {
        return breakCode;
    }

    // Retrieve the implementation's name
    public String getImplementation() {
        return "Java";
    }

    // Retrieve the current program counter address
    public int getProgramCounter() {
        return cpu.pc;
    }

    // Retrieve the value of a program register
    public int getProgramRegister(int index) {
        return index < 0 || index > 31 ? 0 : cpu.registers[index];
    }

    // Retrieve the current ROM into a byte buffer
    // Returns true if successful
    public boolean getROM(byte[] buffer, int offset) {
        return getMemory(rom, buffer, offset);
    }

    // Retrieve the current ROM size in bytes
    public int getROMSize() {
        return rom == null ? 0 : rom.length;
    }

    // Retrieve the current SRAM into a byte buffer
    // Returns true if successful
    public boolean getSRAM(byte[] buffer, int offset) {
        return getMemory(sram, buffer, offset);
    }

    // Retrieve the current SRAM size in bytes
    public int getSRAMSize() {
        return sram == null ? 0 : sram.length;
    }

    // Retrieve the value of a system register
    public int getSystemRegister(int regID) {
        return cpu.getSystemRegister(regID);
    }

    // Determine whether the implementation is native
    public boolean isNative() {
        return false;
    }

    // Read a value from the bus
    public int read(int address, int type, boolean debug) {

        // Validate type parameter
        int size = typeSize(type);
        if (size == 0) return 0;

        // Select memory region and address
        int region  = address >> 24 & 7;
        address    &= 0x00FFFFFF & -size;

        // Process access by memory region
        switch (region) {
            case 0: return vip.read(address, type, debug); // VIP
            case 1: return 0; // VSU
            case 2: return 0; // Other components
            case 3: return 0; // Unmapped
            case 4: return 0; // Expansion
            case 5: return readMemory(                    // WRAM
                wram, address & wram.length - 1, type);
            case 6: return sram == null ? 0 : readMemory( // SRAM
                sram, address & sram.length - 1, type);
            case 7: return rom  == null ? 0 : readMemory( // ROM
                rom,  address & rom.length  - 1, type);
        }

        return 0; // Required by compiler
    }

    // Read values from the bus into a byte buffer
    public boolean read(int address, int type, byte[] data, int offset,
        int length, boolean debug) {

        // Select parameters by size of data type
        int size  = typeSize(type);
        length   &= -size;

        // Error checking
        if (
            data   == null ||
            size   == 0    ||
            offset <  0    ||
            length <  size ||
            offset + length > data.length
        ) return false;

        // Process elements
        for (int x = 0; x < length; x += size) {

            // Read the value from the bus
            int value = read(address, type, debug);

            // Encode the value into the byte buffer
            data[offset] = (byte) value;
            if (size != 1)
                data[offset + 1] = (byte) (value >> 8);
            if (size == 4) {
                data[offset + 2] = (byte) (value >> 16);
                data[offset + 3] = (byte) (value >> 24);
            }

            // Update positions
            offset  += size;
            address += size;
        }

        return true;
    }

    // Perform system initialization
    public void reset() {

        // Reset WRAM to zeroes to aid with debugging
        // The hardware does not do this
        for (int x = 0; x < wram.length; x++)
            wram[x] = 0;

        // Reset subsystems
        cpu.reset();
        vip.reset();
    }

    // Specify a new address for the program counter
    public int setProgramCounter(int address) {
        return cpu.setProgramCounter(address);
    }

    // Specify a new value for a program register
    // Returns the actual new value of the program register
    public int setProgramRegister(int index, int value) {
        return index < 1 || index > 31 ? 0 : (cpu.registers[index] = value);
    }

    // Specify a new ROM buffer
    // Returns true if successful
    public boolean setROM(byte[] buffer, int offset, int length) {

        // Error checking
        if (buffer == null || length == 0) {
            rom = null;
            return true;
        }

        // Process the buffer
        buffer = setMemory(buffer, offset, length, 1024);
        if (buffer == null) return false;

        // Update ROM
        rom = buffer;
        return true;
    }

    // Specify a new SRAM buffer
    // Returns true if successful
    public boolean setSRAM(byte[] buffer, int offset, int length) {

        // Error checking
        if (buffer == null || length == 0) {
            sram = null;
            return true;
        }

        // Process the buffer
        buffer = setMemory(buffer, offset, length, 1);
        if (buffer == null) return false;

        // Update SRAM
        sram = buffer;
        return true;
    }

    // Store a value into a system register
    // Returns the actual new value of the system register
    public int setSystemRegister(int regID, int value) {
        return cpu.setSystemRegister(regID, value, true);
    }

    // Write a value to the bus
    public void write(int address, int type, int value, boolean debug) {

        // Validate type parameter
        int size = typeSize(type);
        if (size == 0) return;

        // Select memory region and address
        int region  = address >> 24 & 7;
        address    &= 0x00FFFFFF & -size;

        // Process access by memory region
        switch (region) {
            case 0: vip.write(address, type, value, debug); return; // VIP
            case 1: return; // VSU
            case 2: return; // Other components
            case 3: return; // Unmapped
            case 4: return; // Expansion
            case 5:                           writeMemory( // WRAM
                wram, address & wram.length - 1, type, value);
                return;
            case 6: if (sram != null)         writeMemory( // SRAM
                sram, address & sram.length - 1, type, value);
                return;
            case 7: if (debug && rom != null) writeMemory( // ROM
                rom,  address & rom.length  - 1, type, value);
                return;
        }

    }

    // Write values to the bus from a byte buffer
    public boolean write(int address, int type, byte[] data, int offset,
        int length, boolean debug) {

        // Select parameters by size of data type
        int size  = typeSize(type);
        length   &= -size;

        // Error checking
        if (
            data   == null ||
            size   == 0    ||
            offset <  0    ||
            length <  size ||
            offset + length > data.length
        ) return false;

        // Process elements
        for (int x = 0; x < length; x += size) {

            // Parse the value from the byte buffer
            int value = (int) data[offset] & 0xFF;
            if (size != 1) value |=
                ((int) data[offset + 1] & 0xFF) << 8;
            if (size == 4) value |=
                ((int) data[offset + 2] & 0xFF) << 16 |
                 (int) data[offset + 3]         << 24;

            // Write the value to the bus
            write(address, type, value, debug);

            // Update positions
            offset  += size;
            address += size;
        }

        return true;
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Package Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Read a value from a memory buffer
    int readMemory(byte[] mem, int offset, int type) {
        int size = typeSize(type);

        // Process the first byte of the value
        int value = (int) mem[offset] & 0xFF;
        if (size == 1) {
            if (type == VUE.S8 && (value & 0x80) != 0)
                value |= 0xFFFFFF00; // Sign-extend
            return value;
        }

        // Process the second byte of the value
        value |= ((int) mem[offset + 1] & 0xFF) << 8;
        if (size == 2) {
            if (type == VUE.S16 && (value & 0x8000) != 0)
                value |= 0xFFFF0000; // Sign-extend
            return value;
        }

        // Process the third and fourth bytes of the value
        return value |
            ((int) mem[offset + 2] & 0xFF) << 16 |
             (int) mem[offset + 3]         << 24
        ;
    }

    // Adjust a value to conform to a given data type
    static int tweakValue(int value, int type) {
        switch (type) {
            case VUE.S8:  return value & 0x000000FF |
                ((value & 0x00000080) != 0 ? 0xFFFFFF00 : 0);
            case VUE.S16: return value & 0x0000FFFF |
                ((value & 0x00008000) != 0 ? 0xFFFF0000 : 0);
            case VUE.U8:  return value & 0x000000FF;
            case VUE.U16: return value & 0x0000FFFF;
        }
        return value;
    }

    // Write a value into a memory buffer
    void writeMemory(byte[] mem, int offset, int type, int value) {
        int size = typeSize(type);

        // Process the first byte of the value
        mem[offset] = (byte) value;
        if (size == 1) return;

        // Process the second byte of the value
        mem[offset + 1] = (byte) (value >> 8);
        if (size == 2) return;

        // Process the third and fourth bytes of the value
        mem[offset + 2] = (byte) (value >> 16);
        mem[offset + 3] = (byte) (value >> 24);
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Private Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Retrieve the current ROM or SRAM into a byte buffer
    private boolean getMemory(byte[] mem, byte[] data, int offset) {

        // Implicit success if memory is unmapped
        if (mem == null) return true;

        // Error checking
        if (
            data   == null ||
            offset <  0    ||
            offset + mem.length > data.length
        ) return false;

        // Copy the memory into the byte buffer
        System.arraycopy(mem, 0, data, offset, mem.length);
        return true;
    }

    // Specify a new memory buffer with a minimum length
    private byte[] setMemory(byte[] data, int offset, int length, int min) {

        // Error checking
        if (
            offset < 0 ||
            length < 0 ||
            offset + length > data.length
        ) return null;

        // Ensure the length is a power of two in the appropriate range
        for (int x = min; ; x <<= 1) {
            if (x == length)     break;
            if (x == 0x01000000) return null;
        }

        // Copy the memory from the byte b uffer
        byte[] mem = new byte[length];
        System.arraycopy(data, offset, mem, 0, length);
        return mem;
    }

    // Determine the size in bytes of a data type
    private int typeSize(int type) {
        switch (type) {
            case VUE.S8:  case VUE.U8:  return 1;
            case VUE.S16: case VUE.U16: return 2;
            case VUE.S32:               return 4;
        }
        return 0;
    }
    
}
