package app;

// Java imports
import java.awt.*;
import java.awt.event.*;
import java.io.*;
import javax.swing.*;
import javax.swing.filechooser.*;

// Project imports
import debugger.*;
import util.*;
import vue.*;

// Top-level application window
public class App extends JFrame {

    // Instance fields
    private Debugger debugger; // Debugger UI componets
    private VUE      vue;      // Emulation core context
    private boolean  gm3d;     // Green/Magenta anaglyph mode

    // UI elements
    private JDesktopPane   desktop;    // Child window container
    private JFileChooser   dlgFile;    // Load ROM dialog box
    private JMenuBar       menuBar;    // Main menu bar
    private JInternalFrame wndCPU;
    private JInternalFrame wndVIP;



    ///////////////////////////////////////////////////////////////////////////
    //                             Constructors                              //
    ///////////////////////////////////////////////////////////////////////////

    // Default constructor
    public App() {
        super("PVB Emulator");

        // Configure instance fields
        vue      = VUE.create();
        debugger = new Debugger(vue);
        gm3d     = false;

        // Configure child components
        desktop = new JDesktopPane();
        desktop.setBackground(SystemColor.controlShadow);
        initMenuBar();
        initFonts();
        initCPU();
        initVIP();

        // Configure Load ROM dialog box
        dlgFile = new JFileChooser(".");
        dlgFile.addChoosableFileFilter(new FileNameExtensionFilter(
            "Virtual Boy ROMs", "vb"));
        dlgFile.setAcceptAllFileFilterUsed(true);
        dlgFile.setDialogTitle("Select ROM file");
        dlgFile.setDialogType(JFileChooser.OPEN_DIALOG);
        dlgFile.setApproveButtonText("Load");

        // Configure component
        setDefaultCloseOperation(DISPOSE_ON_CLOSE);
        setContentPane(desktop);
        setSize(800, 600);
        setLocationRelativeTo(null);
        setVisible(true);

        // Configure global keyboard handler
        KeyboardFocusManager.getCurrentKeyboardFocusManager()
            .addKeyEventDispatcher(e->onKey(e));
    }

    // Font constructor
    private void initFonts() {
        Font dialog = Util.getDefaultFont();
        Font hex = Util.getFont(Util.fontExists("Consolas") ? "Consolas" :
            Font.MONOSPACED, Font.PLAIN, Math.max(16,
                Util.getFontMetrics(dialog).getHeight()));
        debugger.setFonts(dialog, hex);
    }

    // Menu bar constructor
    private void initMenuBar() {
        menuBar = new JMenuBar();
        menuBar.setBorder(null);

        // File
        JMenu menu = new JMenu("File");

            // Load ROM...
            JMenuItem item = new JMenuItem("Load ROM...");
            item.addActionListener(e->onLoadROM());
            menu.add(item);

            menu.addSeparator();

            // Exit
            item = new JMenuItem("Exit");
            item.addActionListener(e->onExit());
            menu.add(item);
        menuBar.add(menu);

        setJMenuBar(menuBar);
    }

    // CPU window constructor
    private void initCPU() {
        wndCPU = new JInternalFrame("CPU");
        wndCPU.setDefaultCloseOperation(JInternalFrame.HIDE_ON_CLOSE);
        wndCPU.setClosable(true);
        wndCPU.setResizable(true);
        wndCPU.setSize(640, 480);
        wndCPU.setLocation(16, 16);
        desktop.add(wndCPU);

        JSplitPane left = new JSplitPane(JSplitPane.VERTICAL_SPLIT,
            debugger.getDisassembler(), debugger.getHexEditor());
        Util.adjustSplitPane(left);
        left.setResizeWeight(1);

        JSplitPane right = new JSplitPane(JSplitPane.VERTICAL_SPLIT,
            debugger.getSystemRegisters(), debugger.getProgramRegisters());
        Util.adjustSplitPane(right);
        right.setResizeWeight(0);

        JSplitPane split = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT,
            left, right);
        Util.adjustSplitPane(split);
        split.setResizeWeight(1);
        split.resetToPreferredSizes();
        wndCPU.setContentPane(split);
    }

    // VIP window constructor
    private void initVIP() {
        wndVIP = new JInternalFrame("VIP");
        wndVIP.setDefaultCloseOperation(JInternalFrame.HIDE_ON_CLOSE);
        wndVIP.setClosable(true);
        wndVIP.setResizable(true);
        wndVIP.setSize(640, 480);
        wndVIP.setLocation(0, 0);
        desktop.add(wndVIP);

        JTabbedPane tabs = new JTabbedPane();
        tabs.add("Characters", debugger.getCharacters());
        tabs.add("BG Maps",    debugger.getBGMaps());
        tabs.add("Objects",    debugger.getObjects());
        tabs.add("Worlds",     debugger.getWorlds());
        tabs.setBackground(Util.getColor("control"));
        tabs.setOpaque(true);

        wndVIP.setContentPane(tabs);
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Event Handlers                             //
    ///////////////////////////////////////////////////////////////////////////

    // File -> Exit
    private void onExit() {
        close();
    }

    // Default handler for all keyboard events
    private boolean onKey(KeyEvent e) {
        int code = e.getKeyCode();
        int mods = e.getModifiersEx();

        // Only process key down events
        if (e.getID() != KeyEvent.KEY_PRESSED)
            return false;

        // Ctrl key is down
        if ((mods & KeyEvent.CTRL_DOWN_MASK) != 0) switch (code) {
            case KeyEvent.VK_3: toggleAnaglyph(); break;
            default: return false;
        }

        // No modifiers
        else switch (code) {
            case KeyEvent.VK_F10: debugger.stepOver();   break;
            case KeyEvent.VK_F11: debugger.singleStep(); break;
            default: return false;
        }

        // The event was
        return true;
    }

    // File -> Load ROM...
    private void onLoadROM() {

        // Prompt the user to select a file
        if (dlgFile.showOpenDialog(this) != JFileChooser.APPROVE_OPTION)
            return;
        File file = dlgFile.getSelectedFile();
        if (file == null) return;

        // Load the file into memory
        byte[] rom = Util.fileLoad(file);
        if (rom == null) {
            JOptionPane.showMessageDialog(this,
                "An error occurred while reading the file.",
                "Load ROM", JOptionPane.ERROR_MESSAGE);
            return;
        }

        // Set the file data as the emulation context's ROM
        if (!vue.setROM(rom, 0, rom.length)) {
            JOptionPane.showMessageDialog(this,
                "The selected file does not appear to be a Virtual Boy ROM.",
                "Load ROM", JOptionPane.ERROR_MESSAGE);
            return;
        }

        // Temporary initialization tasks
        setTitle(file.getName() + " - PVB Emulator");
        vue.reset();
        wndVIP.setVisible(true);
        wndCPU.setVisible(true);
        debugger.refresh(true);
        debugger.getDisassembler().requestFocus();

    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Private Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Toggle between red/cyan and green/magenta anaglyph modes
    private void toggleAnaglyph() {
        gm3d = !gm3d;
        debugger.setAnaglyph(
            gm3d ? 0x00FF00 : 0xFF0000,
            gm3d ? 0xE800E8 : 0x00C8C8);
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Public Methods                             //
    ///////////////////////////////////////////////////////////////////////////

    public void close() {
        dispose();
    }

}
