package debugger;

// Java imports
import java.awt.*;
import javax.swing.*;

// Project imports
import debugger.*;
import util.*;

// One line of disassembler output
class CPUDasmLine extends JPanel {

    // Instance fields
    private int      address;     // Address of instruction
    private boolean  amInterrupt; // Representing an interrupt
    private int      size;        // Size of instruction
    private Debugger parent;      // Debugger UI manager

    // UI components
    private JLabel[] labels; // All displayed text



    ///////////////////////////////////////////////////////////////////////////
    //                               Constants                               //
    ///////////////////////////////////////////////////////////////////////////

    // Label indexes
    private static final int ADDRESS  = 0;
    private static final int BYTES    = 1;
    private static final int MNEMONIC = 2;
    private static final int OPERANDS = 3;
    private static final int EXTRA    = 4;



    ///////////////////////////////////////////////////////////////////////////
    //                            Static Methods                             //
    ///////////////////////////////////////////////////////////////////////////

    // Initialize column widths
    public static void clearWidths(int[] widths, boolean full) {
        for (int x = full ? 0 : OPERANDS; x < 5; x++)
            widths[x] = 0;
    }

    // Determine the total width of all columns
    public static int getWidth(int[] widths, int spacing, boolean showBytes) {
        int ret = 0;

        // Process all columns
        for (int x = 0; x < 5; x++) {

            // Skip columns under specific conditions
            if (
                x == BYTES && !showBytes ||
                (widths[x] == 0 && (x == OPERANDS || x == EXTRA))
            ) continue;

            // Increment by column width
            if (ret > 0) ret += spacing;
            ret += widths[x];
        }

        return ret;
    }



    ///////////////////////////////////////////////////////////////////////////
    //                             Constructors                              //
    ///////////////////////////////////////////////////////////////////////////

    // Default constructor
    CPUDasmLine(Debugger parent) {
        super(null);

        // Configure instance fields
        amInterrupt = false;
        this.parent = parent;

        // Configure labels
        labels = new JLabel[5];
        for (int x = 0; x < labels.length; x++) {
            JLabel label = labels[x] = new JLabel();
            label.setOpaque(false);
            label.setVisible(x != EXTRA);
            add(label);
        }

        // Configure component
        setOpaque(false);
        setFonts();
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Package Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Change appearance when selected or active
    void configure(boolean active, boolean select) {
        boolean focus         = parent.disassemblerHasFocus();
        Color   windowText    = Util.getColor("windowText");
        Color   selected      = Util.getColor("selected");
        Color   selectedText  = Util.getColor("selectedText");
        Color   selectedBlur  = Util.getColor("selected", 0.4f);
        Color   selectedFaint = Util.getColor("selected", 0.3f);

        // Configure background
        Color color = active ? focus ? selected : selectedBlur :
            select ? focus ? selectedBlur : selectedFaint : null;
        setOpaque(color != null);
        setBackground(color);

        // Configure foreground
        color = active && focus ? selectedText : windowText;
        for (int x = 0; x < labels.length; x++)
            labels[x].setForeground(color);
    }

    // Determine whether a line contains a given address
    boolean contains(int address) {
        return !amInterrupt && (
            this.address == address ||
            size == 4 && this.address + 2 == address
        );
    }

    // Retrieve the address of the current instruction
    int getAddress() {
        return address;
    }

    // Determine whether the output is an interrupt
    boolean isInterrupt() {
        return amInterrupt;
    }

    // Configure label positions and visibility
    void layout(int[] widths, int spacing, boolean showBytes) {
        int lineHeight = parent.getLineHeight();

        // Interrupt processing
        if (amInterrupt) {

            // Configure label visibility
            for (int x = 0; x < labels.length; x++)
                labels[x].setVisible(x == EXTRA);

            // Configure label position
            labels[EXTRA].setBounds(
                widths[0] + spacing +
                (showBytes ? widths[1] + spacing : 0),
            0, widths[2] + widths[3] + spacing, lineHeight);
            return;
        }

        // Instruction processing
        int left = 0;
        for (int x = 0; x < labels.length; x++) {
            JLabel label = labels[x];

            // Skip columns under specific conditions
            if (
                x == BYTES && !showBytes ||
                (label.getText() == null && (x == OPERANDS || x == EXTRA))
            ) {
                label.setVisible(false);
                continue;
            }

            // Configure label position and visibility
            if (left > 0) left += spacing;
            label.setBounds(left, 0, widths[x], lineHeight);
            label.setVisible(true);
            left += widths[x];
        }

    }

    // Measure column widths
    void measure(int[] widths, int spacing) {
        FontMetrics dialog = parent.getDialogMetrics();
        FontMetrics hex    = parent.getHexMetrics();
        if (dialog == null || hex == null)
            return;

        // Processing for interrupt status
        if (amInterrupt) {
            widths[3] = Math.max(widths[3],
                dialog.stringWidth(labels[EXTRA].getText()) -
                widths[2] - spacing
            );
            return;
        }

        // Processing for instruction status
        for (int x = 0; x < labels.length; x++) {
            FontMetrics metrics = x == EXTRA ? dialog : hex;
            String text = labels[x].getText();
            widths[x] = Math.max(widths[x], text == null ? 0 :
                metrics.stringWidth(labels[x].getText()));
        }

    }

    // Specify new fonts
    void setFonts() {

        // Fetch fonts
        Font dialog = parent.getDialogFont();
        Font hex    = parent.getHexFont();

        // Update labels
        for (int x = 0; x < labels.length; x++)
            labels[x].setFont(x == EXTRA ? dialog : hex);
    }

    // Specify new instruction contents
    void setInstruction(int address, int size, String[] columns) {

        // Configure instance fields
        this.address = address;
        amInterrupt  = false;
        this.size    = size;

        // Configure UI components
        for (int x = 0; x < labels.length; x++)
            labels[x].setText(x == MNEMONIC && columns[x] == null ?
                "---" : columns[x]);

    }

    // Specify new interrupt contents
    void setInterrupt(String text) {
        amInterrupt = true;
        labels[EXTRA].setText(text);
    }

}
