package debugger;

// Java imports
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;

// Project imports
import debugger.*;
import util.*;
import vue.*;

// UI element for displaying disassembly
final class CPUDasmPane extends JScrollPane {

    // Instance fields
    private int      address;   // Address of first line of output
    private Debugger parent;    // Debugger UI manager
    private boolean  showBytes; // Display the bytes column
    private int[]    widths;    // Column widths, in pixels
    private ArrayList<CPUDasmLine> lines; // Lines of output

    // UI elements
    private JPanel client; // Main content pane



    ///////////////////////////////////////////////////////////////////////////
    //                               Constants                               //
    ///////////////////////////////////////////////////////////////////////////

    // I/O register names
    private static HashMap<Integer, String> REGISTERS;

    // Static constructor
    static {
        REGISTERS = new HashMap<Integer, String>();

        // VIP registers
        REGISTERS.put(0x0005F800, "INTPND");
        REGISTERS.put(0x0005F802, "INTENB");
        REGISTERS.put(0x0005F804, "INTCLR");
        REGISTERS.put(0x0005F820, "DPSTTS");
        REGISTERS.put(0x0005F822, "DPCTRL");
        REGISTERS.put(0x0005F824, "BRTA");
        REGISTERS.put(0x0005F826, "BRTB");
        REGISTERS.put(0x0005F828, "BRTC");
        REGISTERS.put(0x0005F82A, "REST");
        REGISTERS.put(0x0005F82E, "FRMCYC");
        REGISTERS.put(0x0005F830, "CTA");
        REGISTERS.put(0x0005F840, "XPSTTS");
        REGISTERS.put(0x0005F842, "XPCTRL");
        REGISTERS.put(0x0005F844, "VER");
        REGISTERS.put(0x0005F848, "SPT0");
        REGISTERS.put(0x0005F84A, "SPT1");
        REGISTERS.put(0x0005F84C, "SPT2");
        REGISTERS.put(0x0005F84E, "SPT3");
        REGISTERS.put(0x0005F860, "GPLT0");
        REGISTERS.put(0x0005F862, "GPLT1");
        REGISTERS.put(0x0005F864, "GPLT2");
        REGISTERS.put(0x0005F866, "GPLT3");
        REGISTERS.put(0x0005F868, "JPLT0");
        REGISTERS.put(0x0005F86A, "JPLT1");
        REGISTERS.put(0x0005F86C, "JPLT2");
        REGISTERS.put(0x0005F86E, "JPLT3");
        REGISTERS.put(0x0005F87E, "BKCOL");
    }



    ///////////////////////////////////////////////////////////////////////////
    //                             Constructors                              //
    ///////////////////////////////////////////////////////////////////////////

    // Default constructor
    CPUDasmPane(Debugger parent) {
        super(VERTICAL_SCROLLBAR_NEVER, HORIZONTAL_SCROLLBAR_AS_NEEDED);

        // Configure instance fields
        address     = 0;
        this.parent = parent;
        showBytes   = true;
        widths      = new int[5];
        lines       = new ArrayList<CPUDasmLine>();

        // Configure UI elements
        client = new JPanel(null);
        client.setBackground(Util.getColor("window"));
        client.setFocusable(true);
        setViewportView(client);

        // Configure event handlers
        addComponentListener(new ComponentListener() {
            public void componentHidden (ComponentEvent e) { }
            public void componentMoved  (ComponentEvent e) { }
            public void componentResized(ComponentEvent e) { onResize(); }
            public void componentShown  (ComponentEvent e) { }
        });
        client.addFocusListener(new FocusListener() {
            public void focusGained(FocusEvent e) { onFocus(true); }
            public void focusLost  (FocusEvent e) { onFocus(false); }
        });
        client.addKeyListener(new KeyListener() {
            public void keyPressed (KeyEvent e) { onKeyDown(e); }
            public void keyReleased(KeyEvent e) { }
            public void keyTyped   (KeyEvent e) { }
        });
        client.addMouseListener(new MouseListener() {
            public void mouseClicked (MouseEvent e) { }
            public void mouseEntered (MouseEvent e) { }
            public void mouseExited  (MouseEvent e) { }
            public void mousePressed (MouseEvent e) { onMouseDown(e); }
            public void mouseReleased(MouseEvent e) { }
        });
        client.addMouseWheelListener(e->onMouseWheel(e));

    }



    ///////////////////////////////////////////////////////////////////////////
    //                               Overrides                               //
    ///////////////////////////////////////////////////////////////////////////

    // Determine focus on the contents panel instead of the scroll pane
    public boolean isFocusOwner() {
        return client.isFocusOwner();
    }

    // Requsest focus on the contents panel instead of the scroll pane
    public void requestFocus() {
        client.requestFocus();
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Package Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Update UI components with the current emulation state
    void refresh(boolean seekToPC) {
        VUE vue = parent.getVUE();
        int pc  = vue.getProgramCounter();

        if (seekToPC)
            seek(pc, getDefaultPosition(pc));
        else disassemble();
    }

    // Position a particular address in the default position in the display
    void seek(int address) {
        seek(address, getDefaultPosition(address));
    }

    // Position a particular address in a particular position in the display
    void seek(int address, int position) {
        this.address = getSeekAddress(address, position);
        disassemble();
    }

    // Specify new fonts
    void setFonts() {

        // Update controls and regenerate output
        for (int x = 0; x < lines.size(); x++)
            lines.get(x).setFonts();
        CPUDasmLine.clearWidths(widths, true);
        disassemble();

        // Configure scroll bar
        getHorizontalScrollBar().setUnitIncrement(parent.getLineHeight());
    }

    // Show or hide the bytes column
    void showBytesColumn(boolean show) {
        int numLines = Math.min(countLines(false), lines.size());
        int spacing  = parent.getLineHeight() * 3 / 2;
        for (int x = 0; x < numLines; x++)
            lines.get(x).layout(widths, spacing, show);
        client.repaint();
        showBytes = show;
    }

    // Toggle the bytes column
    boolean toggleBytesColumn() {
        showBytesColumn(!showBytes);
        return showBytes;
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Event Handlers                             //
    ///////////////////////////////////////////////////////////////////////////

    // Focus changed
    private void onFocus(boolean focused) {
        refresh(false);
    }

    // Key press
    private void onKeyDown(KeyEvent e) {
        int code = e.getKeyCode();
        int mods = e.getModifiersEx();
        int page = countLines(true);

        // Ctrl key is down
        if ((mods & KeyEvent.CTRL_DOWN_MASK) != 0) switch (code) {
            case KeyEvent.VK_G:
                parent.seekDisassembler();
                break;
            case KeyEvent.VK_H:
                toggleBytesColumn();
                break;
            case KeyEvent.VK_P:
                seek(parent.getVUE().getProgramCounter());
                break;
        }

        // No modifiers
        else switch (code) {
            case KeyEvent.VK_DOWN:      seekRelative(    1); break;
            case KeyEvent.VK_PAGE_DOWN: seekRelative( page); break;
            case KeyEvent.VK_PAGE_UP:   seekRelative(-page); break;
            case KeyEvent.VK_UP:        seekRelative(   -1); break;
        }

    }

    // Mouse press
    private void onMouseDown(MouseEvent e) {
        int button = e.getButton();

        if (button == MouseEvent.BUTTON1)
            client.requestFocus();
    }

    // Mouse wheel
    private void onMouseWheel(MouseWheelEvent e) {
        int amount = e.getWheelRotation() * e.getScrollAmount();
        int mods   = e.getModifiersEx();

        seekRelative(amount);
    }

    // Resize
    private void onResize() {
        disassemble();
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Private Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Calculate the number of lines of text in the display
    private int countLines(boolean visible) {
        int lineHeight = parent.getLineHeight();
        return lineHeight == 0 ? 0 : (visible ?
            getViewport().getExtentSize().height : getHeight() + lineHeight - 1
        ) / lineHeight;
    }

    // Regenerate output
    private void disassemble() {

        // Prepare working variables
        Disassembler dasm       = parent.getDasm();
        Instruction  inst       = new Instruction();
        int          lineHeight = parent.getLineHeight();
        VUE          vue        = parent.getVUE();
        int          pc         = vue.getProgramCounter();
        int          numLines   = countLines(false);
        String[]     columns    = new String[5];

        // Read instruction bits from the bus
        byte[] bits = new byte[numLines * 4];
        vue.read(address, VUE.U16, bits, 0, bits.length, false);

        // Process each line of output
        CPUDasmLine.clearWidths(widths, false);
        int spacing = lineHeight * 3 / 2;
        for (int x = 0, offset = 0; x < numLines; x++) {

            // Retrieve or create the UI component for this line
            CPUDasmLine line = null;
            if (x == lines.size()) {
                line = new CPUDasmLine(parent);
                lines.add(line);
                client.add(line);
            } else line = lines.get(x);

            // Decode the instruction
            inst.decode(
                (int) bits[offset    ] & 0xFF | (int) bits[offset + 1] << 8,
                (int) bits[offset + 2] & 0xFF | (int) bits[offset + 3] << 8
            );
            dasm.disassemble(address + offset, inst, columns);

            // Get the text for the extra column
            columns[4] = getExtra(address + offset, pc, inst, vue);
            if (columns[4] != null)
                columns[4] = "; " + columns[4];

            // Configure UI
            line.setInstruction(address + offset, inst.size, columns);
            line.configure(address + offset == pc, false);
            line.measure(widths, spacing);

            // Advance to next instruction
            offset += address + offset + 2 == pc ? 2 : inst.size;
        }

        // Configure layout
        int width = CPUDasmLine.getWidth(widths, spacing, showBytes);
        client.setPreferredSize(new Dimension(width, 0));
            width = Math.max(width, getViewport().getExtentSize().width);

        // Layout for all visible lines
        for (int y = 0; y < numLines; y++) {
            CPUDasmLine line = lines.get(y);
            line.setBounds(0, y * lineHeight, width, lineHeight);
            line.layout(widths, spacing, showBytes);
            line.setVisible(true);
        }

        // Hide all invisible lines
        for (int y = numLines; y < lines.size(); y++)
            lines.get(y).setVisible(false);

        // Swing behaves oddly without doing this
        client.repaint();
    }

    // Calculate the default position of seeking to a new address
    private int getDefaultPosition(int address) {
        int position = getPosition(address);
        return position != -1 ? position : countLines(true) / 3;
    }

    // Determine the contents of the extra column
    private String getExtra(int address, int pc, Instruction inst, VUE vue) {

        // Prepare memory access fields
        boolean access = inst.format == 6;
        int     target = !access ? 0 : inst.operands[2] +
                           vue.getProgramRegister(inst.operands[1]);
        int     region = target >> 24 & 7;

        // VIP registers
        if (address == pc && access && region == 0) {
            target &= 0x0007FFFF;
            return REGISTERS.get(target);
        }

        // No extra text
        return null;
    }

    // Determine the position of an address in the visible part of the display
    private int getPosition(int address) {

        // Search for the address in the lines of output
        int numLines = Math.min(countLines(true), lines.size());
        for (int x = 0; x < numLines; x++)
            if (lines.get(x).contains(address))
                return x;

        // Address is not visible
        return -1;
    }

    // Determine the address of the first line of output
    private int getSeekAddress(int address, int position) {
        VUE vue  = parent.getVUE();
        int pc   = vue.getProgramCounter();
        address &= -2; // Restrict to instruction alignment

        // Read instruction bits from the bus
        byte[] bits       = new byte[Math.abs(position) * 4 + 64];
        int[]  addresses  = new int[Math.max(position, 0) + 1];
        int    target     = address;
               address   -= position < 0 ? 64: bits.length;
        vue.read(address, VUE.U16, bits, 0, bits.length, false);

        // Decode until the instruction containing the address is encountered
        int offset = 0; // Position in machine code bits buffer
        int x      = 0; // Position in circular buffer
        for (;;) {

            // Index instruction addresses in a circular buffer
            addresses[x] = address;
            x = x + 1 == addresses.length ? 0 : x + 1;

            // The current address is exactly the target
            if (address == target) {
                address = addresses[x];
                break;
            }

            // Check if the target address is in the middle of the instruction
            int size = address + 2 == pc ? 2 :
                Instruction.getSize((int) bits[offset + 1] << 8);
            if (size == 4 && address + 2 == target) {
                address = addresses[x];
                break;
            }

            // Advance to the next instruction
            address += size;
            offset  += size;
        }

        // Top line address is in the circular buffer
        if (position >= 0)
            return addresses[x];

        // Continue decoding instructions until the given position is reached
        for (x = 0; x > position; x--) {
            int size = address + 2 == pc ? 2 :
                Instruction.getSize((int) bits[offset + 1] << 8);
            address += size;
            offset  += size;
        }
        return address;
    }

    // Scroll the current address into a different position in the display
    private void seekRelative(int amount) {
        address = getSeekAddress(address, -amount);
        disassemble();
    }

}
