package debugger;

// Java imports
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;

// Project imports
import debugger.*;
import util.*;
import vue.*;

// UI element for displaying the hex editor
class CPUHexPane extends JScrollPane {

    // Instance fields
    private int      address;     // Address of first line of output
    private boolean  editActive;
    private int      editAddress;
    private int      editSize;
    private int      editValue;
    private Debugger parent;      // Debugger UI manager
    private ArrayList<CPUHexLine> lines; // Lines of output

    // UI elements
    private JPanel client; // Main content pane

    // Static fields
    private static HashMap<Integer, Integer> DIGITS;



    ///////////////////////////////////////////////////////////////////////////
    //                              Construtors                              //
    ///////////////////////////////////////////////////////////////////////////

    // Static constructor
    static {
        DIGITS = new HashMap<Integer, Integer>(16);
        DIGITS.put(KeyEvent.VK_0,  0);
        DIGITS.put(KeyEvent.VK_1,  1);
        DIGITS.put(KeyEvent.VK_2,  2);
        DIGITS.put(KeyEvent.VK_3,  3);
        DIGITS.put(KeyEvent.VK_4,  4);
        DIGITS.put(KeyEvent.VK_5,  5);
        DIGITS.put(KeyEvent.VK_6,  6);
        DIGITS.put(KeyEvent.VK_7,  7);
        DIGITS.put(KeyEvent.VK_8,  8);
        DIGITS.put(KeyEvent.VK_9,  9);
        DIGITS.put(KeyEvent.VK_A, 10);
        DIGITS.put(KeyEvent.VK_B, 11);
        DIGITS.put(KeyEvent.VK_C, 12);
        DIGITS.put(KeyEvent.VK_D, 13);
        DIGITS.put(KeyEvent.VK_E, 14);
        DIGITS.put(KeyEvent.VK_F, 15);
    }

    // Default constructor
    CPUHexPane(Debugger parent) {
        super(VERTICAL_SCROLLBAR_NEVER, HORIZONTAL_SCROLLBAR_AS_NEEDED);

        // Configure instance fields
        address     = 0;
        editActive  = false;
        editAddress = 0;
        editSize    = 0;
        editValue   = 0;
        this.parent = parent;
        lines       = new ArrayList<CPUHexLine>();

        // Configure UI elements
        client = new JPanel(null);
        client.setBackground(Util.getColor("window"));
        client.setFocusable(true);
        setViewportView(client);

        // Configure event handlers
        addComponentListener(new ComponentListener() {
            public void componentHidden (ComponentEvent e) { }
            public void componentMoved  (ComponentEvent e) { }
            public void componentResized(ComponentEvent e) { onResize(); }
            public void componentShown  (ComponentEvent e) { }
        });
        client.addFocusListener(new FocusListener() {
            public void focusGained(FocusEvent e) { onFocus(true); }
            public void focusLost  (FocusEvent e) { onFocus(false); }
        });
        client.addKeyListener(new KeyListener() {
            public void keyPressed (KeyEvent e) { onKeyDown(e); }
            public void keyReleased(KeyEvent e) { }
            public void keyTyped   (KeyEvent e) { onType(e); }
        });
        client.addMouseWheelListener(e->onMouseWheel(e));
    }



    ///////////////////////////////////////////////////////////////////////////
    //                               Overrides                               //
    ///////////////////////////////////////////////////////////////////////////

    // Determine focus on the contents panel instead of the scroll pane
    public boolean isFocusOwner() {
        return client.isFocusOwner();
    }

    // Requsest focus on the contents panel instead of the scroll pane
    public void requestFocus() {
        client.requestFocus();
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Package Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Update UI components with the current emulation state
    void refresh() {

        // Prepare working variables
        Disassembler dasm       = parent.getDasm();
        int          hexWidth   = parent.getHexWidth();
        int          lineHeight = parent.getLineHeight();
        VUE          vue        = parent.getVUE();
        int          numLines   = countLines(false);

        // Read data from the bus
        byte[] data = new byte[numLines * 16];
        vue.read(address, VUE.S32, data, 0, data.length, false);

        // Process each line of output
        int spacing = hexWidth;
        for (int x = 0, offset = 0; x < numLines; x++) {

            // Retrieve or create the UI component for this line
            CPUHexLine line;
            if (x == lines.size()) {
                line = new CPUHexLine(parent);
                lines.add(line);
                client.add(line);
            } else line = lines.get(x);

            // Configure UI
            int cursor = !editActive ? -1 : editAddress - address - offset;
            if (editSize > 0 && cursor == (cursor & 0xF))
                data[offset + cursor] = (byte) editValue;
            line.setData(address + offset, data, offset);
            line.configure(cursor);

            // Advance to next line
            offset += 16;
        }

        // Configure layout
        int width = hexWidth * 40 + spacing * 18;
        client.setPreferredSize(new Dimension(width, 0));
            width = Math.max(width, getViewport().getExtentSize().width);

        // Layout for all visible lines
        for (int y = 0; y < numLines; y++) {
            CPUHexLine line = lines.get(y);
            line.setBounds(0, y * lineHeight, width, lineHeight);
            line.layout(spacing);
            line.setVisible(true);
        }

        // Hide all invisible lines
        for (int y = numLines; y < lines.size(); y++)
            lines.get(y).setVisible(false);

        // Swing behaves oddly without doing this
        client.repaint();
    }

    // Show a particular address in the display
    void seek(int address, boolean setCursor, boolean showAtTop) {

        // Move the edit cursor to the given address
        if (setCursor) {
            client.requestFocus();
            commitEdit();
            editActive  = true;
            editAddress = address;
        }

        // Ensure the address is visisble
        if (getPosition(address) == -1) {

            // Determine the position of the line in the output
            if (!showAtTop && address - this.address > 0) {
                this.address = (countLines(true) - 1) * -16;
            } else this.address = 0;

            // Calculate the address of the first line of output
            this.address += address & -16;
        }

        // Regenerate output
        refresh();
    }

    // Specify new fonts
    void setFonts() {

        // Update controls and regenerate output
        for (int x = 0; x < lines.size(); x++)
            lines.get(x).setFonts();
        refresh();

        // Configure component
        setPreferredSize(new Dimension(0, parent.getLineHeight() * 8));
        getHorizontalScrollBar().setUnitIncrement(parent.getLineHeight());
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Event Handlers                             //
    ///////////////////////////////////////////////////////////////////////////

    // Focus changed
    private void onFocus(boolean focused) {
        refresh();
    }

    // Key press
    private void onKeyDown(KeyEvent e) {
        int code = e.getKeyCode();
        int mods = e.getModifiersEx();
        int page = countLines(true);

        // Ctrl key is down
        if ((mods & KeyEvent.CTRL_DOWN_MASK) != 0) switch (code) {
            case KeyEvent.VK_G:
                parent.seekHexEditor();
                break;
            case KeyEvent.VK_P:
                parent.seekHexEditor(parent.getVUE().getProgramCounter());
                break;
        }

        // No modifiers
        else switch (code) {

            // Nagivation
            case KeyEvent.VK_DOWN:      moveCursor  ( 16);   break;
            case KeyEvent.VK_ENTER:     if (editSize > 0) moveCursor(1); break;
            case KeyEvent.VK_ESCAPE:    editActive = false; refresh();   break;
            case KeyEvent.VK_LEFT:      moveCursor  ( -1);   break;
            case KeyEvent.VK_PAGE_DOWN: seekRelative( page); break;
            case KeyEvent.VK_PAGE_UP:   seekRelative(-page); break;
            case KeyEvent.VK_RIGHT:     moveCursor  (  1);   break;
            case KeyEvent.VK_UP:        moveCursor  (-16);   break;
        }

    }

    // Mouse wheel
    private void onMouseWheel(MouseWheelEvent e) {
        int amount = e.getWheelRotation() * e.getScrollAmount();
        int mods   = e.getModifiersEx();

        seekRelative(amount);
    }

    // Resize
    private void onResize() {
        refresh();
    }

    // Key type
    private void onType(KeyEvent e) {
        char code = e.getKeyChar();
        int  mods = e.getModifiersEx();

        // Ignore the input
        if (!editActive || (mods & KeyEvent.CTRL_DOWN_MASK) != 0)
            return;

        // Process the entered text as a hex digit
        int digit = 0;
        try { digit = Integer.parseInt(new String(new char[] { code }), 16); }
        catch (Exception x) { return; }

        // Update edit parameters
        editValue = editValue << 4 | digit;
        editSize++;

        // Follow-up tasks
        if (editSize == 2) {
            commitEdit();
            parent.seekHexEditor(editAddress + 1);
        } else refresh();

    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Private Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Commit any partial edit
    private void commitEdit() {
        if (!editActive || editSize == 0)
            return;
        parent.getVUE().write(editAddress, VUE.U8, editValue, true);
        editSize  = 0;
        editValue = 0;
        parent.refresh(false);
    }

    // Calculate the number of lines of text in the display
    private int countLines(boolean visible) {
        int lineHeight = parent.getLineHeight();
        return lineHeight == 0 ? 0 : (visible ?
            getViewport().getExtentSize().height : getHeight() + lineHeight - 1
        ) / lineHeight;
    }

    // Determine the position of an address in the visible part of the display
    private int getPosition(int address) {

        // Search for the address in the lines of output
        int numLines = Math.min(countLines(true), lines.size());
        for (int x = 0; x < numLines; x++)
            if (lines.get(x).contains(address))
                return x;

        // Address is not visible
        return -1;
    }

    // Reposition the edit cursor
    private void moveCursor(int amount) {

        // The cursor is active
        if (editActive) {
            seek(editAddress + amount, true, false);
            return;
        }

        // The cursor is not active
        if ((amount & 1) == 0) {
            address += amount;
            refresh();
        }

    }

    // Scroll the current address into a different position in the display
    private void seekRelative(int amount) {
        address += amount * 16;
        refresh();
    }

}
