package debugger;

// Java imports
import java.awt.*;
import javax.swing.*;

// Project imports
import debugger.*;
import util.*;
import vue.*;

// Debugger UI component manager
public final class Debugger {

    // Instance fields
    private int[]        anaglyph; // Anaglyph colors
    private int[][]      colEyes;  // Rendering colors
    private Color[][]    colRed;   // Rendering shades of red
    private Disassembler dasm;     // Machine code translator
    private VIPPalettes  palettes; // VIP palettes
    private byte[]       vram;     // VIP memory cache
    private VUE          vue;      // Reference to emulation context

    // Font fields
    private Font        dlgFont;    // Font for UI components
    private FontMetrics dlgMetrics; // Metrics for UI components
    private Font        hexFont;    // Font for hexadecimal figures
    private FontMetrics hexMetrics; // Metrics for hexadecimal figures
    private int         hexWidth;   // Width in pixels of one hex character
    private int         lineHeight; // Height in pixels of a line of text

    // UI components
    private CPUDasmPane     uiDasm;     // CPU Disassembler pane
    private CPUHexPane      hexEditor;  // CPU Hex editor pane
    private CPURegisterList regProgram; // CPU Program register list
    private CPURegisterList regSystem;  // CPU System register list
    private VIPBGMapPane    bgMaps;     // VIP BG Maps tab
    private VIPChrPane      chrs;       // VIP Characters tab
    private VIPObjPane      objs;       // VIP Objects tab
    private VIPWorldPane    worlds;     // VIP Worlds tab



    ///////////////////////////////////////////////////////////////////////////
    //                             Constructors                              //
    ///////////////////////////////////////////////////////////////////////////

    // Default constructor
    public Debugger(VUE vue) {

        // Configure instance fields
        anaglyph   = new int[] { 0xFFFF0000, 0xFF0000FF };
        colEyes    = new int[4][3];
        colRed     = new Color[2][3];
        dasm       = new Disassembler();
        hexWidth   = 0;
        lineHeight = 0;
        palettes   = new VIPPalettes(this);
        vram       = new byte[0x40000];
        this.vue   = vue;

        // Configure UI components
        uiDasm     = new CPUDasmPane(this);
        hexEditor  = new CPUHexPane(this);
        regProgram = new CPURegisterList(this, CPURegisterList.PROGRAM);
        regSystem  = new CPURegisterList(this, CPURegisterList.SYSTEM);
        bgMaps     = new VIPBGMapPane(this);
        chrs       = new VIPChrPane(this);
        objs       = new VIPObjPane(this);
        worlds     = new VIPWorldPane(this);
        regSystem.setExpanded(1, true);
        refresh(true);
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Public Methods                             //
    ///////////////////////////////////////////////////////////////////////////

    // Retrieve the BG maps UI component
    public JComponent getBGMaps() {
        return bgMaps;
    }

    // Retrieve the characters UI component
    public JComponent getCharacters() {
        return chrs;
    }

    // Retrieve the disassembler UI component
    public JComponent getDisassembler() {
        return uiDasm;
    }

    // Retrieve the hex editor UI component
    public JComponent getHexEditor() {
        return hexEditor;
    }

    // Retrieve the ojbects UI component
    public JComponent getObjects() {
        return objs;
    }

    // Retrieve the program register list UI component
    public JComponent getProgramRegisters() {
        return regProgram;
    }

    // Retrieve the system register list UI component
    public JComponent getSystemRegisters() {
        return regSystem;
    }

    // Retrieve the worlds UI component
    public JComponent getWorlds() {
        return worlds;
    }

    // Update UI components with the current emulation state
    public void refresh(boolean seekToPC) {

        // CPU
        uiDasm.refresh(seekToPC);
        hexEditor.refresh();
        regProgram.refresh();
        regSystem.refresh();

        // VIP
        vue.read(0x00000000, VUE.U8, vram, 0, vram.length, true);
        palettes.refresh();
        refreshColors();
        chrs.refresh();
        bgMaps.refresh();
        objs.refresh();
        worlds.refresh();
    }

    // Prompt the user to seek the disassembler to a new address
    public void seekDisassembler() {
        Integer address = promptAddress("Go to",
            "Enter an address to seek to:");
        if (address != null)
            uiDasm.seek(address);
    }

    // Seek the disassembler to a specific address
    public void seekDisassembler(int address) {
        uiDasm.seek(address);
    }

    // Prompt the user to seek the hex editor to a new address
    public void seekHexEditor() {
        Integer address = promptAddress("Go to",
            "Enter an address to seek to:");
        if (address != null)
            hexEditor.seek(address, true, true);
    }

    // Seek the hex editor to a specific address
    public void seekHexEditor(int address) {
        hexEditor.seek(address, true, true);
    }

    // Specify colors for anaglyph output
    public void setAnaglyph(int left, int right) {
        anaglyph[0] = 0xFF000000 | left;
        anaglyph[1] = 0xFF000000 | right;
        refresh(false);
    }

    // Specify new fonts
    public void setFonts(Font dialog, Font hex) {

        // Configure font fields
        dlgFont    = dialog;
        dlgMetrics = Util.getFontMetrics(dialog);
        hexFont    = hex;
        hexMetrics = Util.getFontMetrics(hex);

        // Determine the maximum width of a hex character
        hexWidth = 0;
        String code = getHexCase() ? "%X" : "%x";
        if (hexMetrics != null) for (int x = 0; x < 16; x++)
            hexWidth = Math.max(hexWidth,
                hexMetrics.stringWidth(String.format(code, x)));

        // Determine the height of a line of text
        lineHeight = dlgMetrics == null || hexMetrics == null ? 0 :
            Math.max(dlgMetrics.getHeight(), hexMetrics.getHeight());

        // Update UI elements
        uiDasm.setFonts();
        hexEditor.setFonts();
        regSystem.setFonts();
        regProgram.setFonts();
        regSystem.setFonts();
        regProgram.setFonts();
        chrs.setFonts();
        bgMaps.setFonts();
        objs.setFonts();
        worlds.setFonts();
    }

    // Execute the next instruction
    public Integer singleStep() {
        vue.emulate(1);
        refresh(true);
        return vue.getBreakCode();
    }

    // Attempt to execute until the following address
    public Integer stepOver() {
        Integer breakCode = null;
        int cycles        = 20000000; // Cycles in one second
        int pc            = vue.getProgramCounter();
        int target        = pc +
            Instruction.getSize(vue.read(pc, VUE.U16, false));
        while (pc != target && cycles >= 0 && breakCode == null) {
            cycles    += vue.emulate(1) - 1;
            pc         = vue.getProgramCounter();
            breakCode  = vue.getBreakCode();
        }
        refresh(true);
        return vue.getBreakCode();
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Package Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Retrieve the anaglyph colors
    int[] getAnaglyph() {
        return anaglyph;
    }

    // Determine whether the disassembler has focus
    boolean disassemblerHasFocus() {
        return uiDasm.isFocusOwner();
    }

    // Retrieve an ARGB color value for a given brightness level and eye
    int getColor(int level, int eye, boolean generic) {
        return level == -1 ? 0xD8000000 : level == 0 ? 0xFF000000 :
            colEyes[(generic ? 2 : 0) + eye][level - 1];
    }

    // Retrieve a shade of red for a given brightness level
    Color getColor(int level, boolean generic) {
        return level == -1 ? new Color(0xD8000000, true) :
            level == 0 ? Color.black : colRed[generic ? 1 : 0][level - 1];
    }

    // Retrieve the disassembler context
    Disassembler getDasm() {
        return dasm;
    }

    // Retrieve the Dialog font
    Font getDialogFont() {
        return dlgFont;
    }

    // Retrieve the Dialog font metrics
    FontMetrics getDialogMetrics() {
        return dlgMetrics;
    }

    // Determine the case of hexadecimal figures
    // Returns true if uppercase, false if lowercase
    boolean getHexCase() {
        return dasm.getHexCase();
    }

    // Retrieve the hexadecimal font
    Font getHexFont() {
        return hexFont;
    }

    // Retrieve the hexadecimal font metrics
    FontMetrics getHexMetrics() {
        return hexMetrics;
    }

    // Retrieve the maximum width of a single hex character
    int getHexWidth() {
        return hexWidth;
    }

    // Retrieve the height of a line of text
    int getLineHeight() {
        return lineHeight;
    }

    // Retrieve a palette of a given index
    VIPPalettes getPalettes() {
        return palettes;
    }

    // Retrieve the handle to VIP memory
    byte[] getVRAM() {
        return vram;
    }

    // Rertrieve the emulation state context
    VUE getVUE() {
        return vue;
    }

    // Determine whether the hex editor has focus
    boolean hexEditorHasFocus() {
        return hexEditor.isFocusOwner();
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Private Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Prompt the user for an address
    private Integer promptAddress(String title, String message) {

        // Prompt the user to enter an address
        String input = JOptionPane.showInputDialog(uiDasm,
            message, title, JOptionPane.PLAIN_MESSAGE);
        if (input == null) return null;

        // Process the input
        try { return (int) Long.parseLong(input, 16); }
        catch (Exception e) {
            JOptionPane.showMessageDialog(uiDasm,
                "Could not interpret the value entered.",
                title, JOptionPane.ERROR_MESSAGE);
            return null;
        }

    }

    // Regenerate VIP colors cache
    private void refreshColors() {
        int[]   generic = { 42, 85, 127 };
        int[][] red     = new int[2][3];

        // Shades of red for character and BG map displays
        palettes.scaleColor(red[0], 0xFFFF0000);
        palettes.scaleColor(red[1], 0xFFFF0000, generic);

        // Anaglyph colors for other displays
        for (int x = 0; x < 2; x++) {
            for (int y = 0; y < 3; y++) {
                palettes.scaleColor(colEyes[0 + x], anaglyph[x]);
                palettes.scaleColor(colEyes[2 + x], anaglyph[x], generic);
                colRed[x][y] = new Color(red[x][y], true);
            }
        }

    }

}
