package debugger;

// Java imports
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;

// Project imports
import debugger.*;
import util.*;
import vue.*;

// VIP background map viewer
class VIPBGMapPane extends JPanel {

    // Instance fields
    private VIPCell[] cells;    // BG map cells
    private Color[][] colors;   // Palette ARGB colors
    private boolean   generic;  // Force the generic colors
    private boolean   grid;     // Leave space between characters
    private int       index;    // Selected global BG map cell
    private Debugger  parent;   // Debugger UI manager
    private int       scale;    // Scaling for BG map image
    private boolean   updating; // Setting values programmatically

    // UI components
    private JCheckBox         chkGeneric;   // Generic colors check box
    private JCheckBox         chkGrid;      // Show grid check box
    private JCheckBox         chkHFlip;     // Horizontal flip check box
    private JCheckBox         chkVFlip;     // Vertical flip check box
    private JComboBox<String> cmbPalette;   // Palette combo box
    private JPanel            panControls;  // Controls content panel
    private JPanel            panMap;       // Map content panel
    private JPanel            panPattern;   // Pattern box
    private JScrollPane       scrControls;  // Scroll pane for controls
    private JScrollPane       scrMap;       // Scroll pane for map
    private JSlider           sldScale;     // Scale slider
    private JSpinner          spnIndex;     // Index spinner
    private JSpinner          spnCharacter; // Character spinner
    private JSpinner          spnMap;       // Map spinner
    private JTextField        txtAddress;   // Address text box
    private ArrayList<JComponent> uiFont;   // UI elements depending on font



    ///////////////////////////////////////////////////////////////////////////
    //                               Constants                               //
    ///////////////////////////////////////////////////////////////////////////

    // Default scale
    private static final int DEFAULTSCALE = 2;



    ///////////////////////////////////////////////////////////////////////////
    //                             Constructors                              //
    ///////////////////////////////////////////////////////////////////////////

    // Default constructor
    VIPBGMapPane(Debugger parent) {
        super(new BorderLayout());

        // Configure instance fields
        colors      = new Color[5][4];
        generic     = false;
        grid        = false;
        index       = 0;
        cells       = new VIPCell[65536];
        this.parent = parent;
        scale       = DEFAULTSCALE;
        updating    = false;

        // Configure controls
        uiFont = new ArrayList<JComponent>();
        initMap();
        initControls();

        // Configure component
        add(scrMap,      BorderLayout.CENTER    );
        add(scrControls, BorderLayout.LINE_START);
        setIndex(index);
        setScale(scale);
    }

    // Map pane constructor
    private void initMap() {

        // Configure content pane
        panMap = new JPanel(null) {
            public void paintComponent(Graphics g) {
                super.paintComponent(g);
                onPaintMap(g);
            }
        };
        panMap.setBackground(Util.getColor("control"));
        panMap.setFocusable(true);
        panMap.addMouseListener(Util.onMouse(
            e->onMouseDownMap(e), null));
        resizeMap();

        // Configure scroll pane
        scrMap = new JScrollPane(panMap);

        // Configure cells
        for (int x = 0; x < 65536; x++)
            cells[x] = new VIPCell(parent, x);
    }

    // Control pane constructor
    private void initControls() {

        // Event handlers
        ActionListener onAction = e->{ if (!updating) onEdit(); };

        // Top-level container
        panControls = new JPanel(new GridBagLayout()) {
            public void paintComponent(Graphics g)
                { super.paintComponent(g); onPaintControls(); }
        };
        panControls.setFocusable(true);
        panControls.addMouseListener(Util.onMouse(
            e->panControls.requestFocus(), null));

        // Scroll pane for container
        scrControls = new JScrollPane(panControls,
            JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED,
            JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
        scrControls.setBorder(null);

        // Map spinner
        JLabel lblMap = new JLabel("Map");
        spnMap = new JSpinner(new SpinnerNumberModel(0, 0, 15, 1));
        spnMap.setEditor(new JSpinner.NumberEditor(spnMap, "#"));
        spnMap.addChangeListener(e->{ if (!updating) onMap(); });
        add(panControls, lblMap, 1, false, false, 2, 4, 0, 4);
        add(panControls, spnMap, 0, false, true, 2, 0, 0, 4);
        uiFont.add(lblMap);
        uiFont.add(spnMap);

        // Cell group box
        JPanel frmCell = new JPanel(new GridBagLayout());
        frmCell.setBorder(new TitledBorder("Cell"));
        add(panControls, frmCell, 0, false, true, 2, 0, 0, 0);
        uiFont.add(frmCell);

        // Pattern panel
        panPattern = new JPanel(null) {
            public void paintComponent(Graphics g)
                { super.paintComponent(g); onPaintPattern(g); }
        };
        panPattern.setPreferredSize(new Dimension(128, 128));
        add(frmCell, panPattern, 0, true, false, 0, 2, 0, 2);

        // Index spinner
        JLabel lblIndex = new JLabel("Index");
        spnIndex = new JSpinner(new SpinnerNumberModel(0, 0, 65535, 1));
        spnIndex.setEditor(new JSpinner.NumberEditor(spnIndex, "#"));
        spnIndex.addChangeListener(e->{ if (!updating) onIndex(); });
        add(frmCell, lblIndex, 1, false, false, 2, 2, 0, 4);
        add(frmCell, spnIndex, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblIndex);
        uiFont.add(spnIndex);

        // Address text box
        JLabel lblAddress = new JLabel("Address");
        txtAddress = new JTextField();
        txtAddress.addActionListener(
            e->{ panControls.requestFocus(); onAddress(); });
        txtAddress.addFocusListener(Util.onFocus(
            null, e->onAddress()));
        add(frmCell, lblAddress, 1, false, false, 2, 2, 0, 4);
        add(frmCell, txtAddress, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblAddress);
        uiFont.add(txtAddress);

        // Character spinner
        JLabel lblCharacter = new JLabel("Character");
        spnCharacter = new JSpinner(new SpinnerNumberModel(0, 0, 2047, 1));
        spnCharacter.setEditor(new JSpinner.NumberEditor(spnCharacter, "#"));
        spnCharacter.addChangeListener(e->{ if (!updating) onEdit(); });
        add(frmCell, lblCharacter, 1, false, false, 2, 2, 0, 4);
        add(frmCell, spnCharacter, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblCharacter);
        uiFont.add(spnCharacter);

        // Palette combo box
        JLabel lblPalette = new JLabel("Palette");
        cmbPalette = new JComboBox<String>(new String[]
            { "BG 0",  "BG 1",  "BG 2",  "BG 3" });
        add(frmCell, lblPalette, 1, false, false, 2, 2, 0, 4);
        add(frmCell, cmbPalette, 0, false, true, 2, 0, 0, 2);
        cmbPalette.addActionListener(onAction);
        uiFont.add(lblPalette);
        uiFont.add(cmbPalette);

        // Flip check boxes
        chkHFlip = new JCheckBox("H-flip");
        chkHFlip.setBorder(null);
        chkHFlip.addActionListener(onAction);
        chkVFlip = new JCheckBox("V-flip");
        chkVFlip.setBorder(null);
        chkVFlip.addActionListener(onAction);
        add(frmCell, chkHFlip, 1, false, false, 2, 2, 0, 0);
        add(frmCell, chkVFlip, 0, false, false, 2, 2, 0, 2);
        uiFont.add(chkHFlip);
        uiFont.add(chkVFlip);

        // Options group box
        JPanel frmOptions = new JPanel(new GridBagLayout());
        frmOptions.setBorder(new TitledBorder("Options"));
        add(panControls, frmOptions, 0, false, true, 2, 0, 0, 0);
        uiFont.add(frmOptions);

        // Scale slider
        JLabel lblScale = new JLabel("Scale");
        sldScale = new JSlider(1, 10, DEFAULTSCALE);
        sldScale.setFocusable(false);
        sldScale.setLabelTable(null);
        sldScale.setPreferredSize(new Dimension(0,
            sldScale.getPreferredSize().height));
        sldScale.setSnapToTicks(true);
        sldScale.addChangeListener(e->{ if (!updating) onScale(); });
        add(frmOptions, lblScale, 1, false, false, 2, 4, 0, 4);
        add(frmOptions, sldScale, 0, false, true, 2, 0, 0, 4);
        uiFont.add(lblScale);

        // Generic and grid check boxes
        chkGeneric = new JCheckBox("Generic colors");
        chkGeneric.setBorder(null);
        chkGeneric.addActionListener(e->onGeneric());
        chkGrid = new JCheckBox("Show grid");
        chkGrid.setBorder(null);
        chkGrid.addActionListener(e->onGrid());
        add(frmOptions, chkGeneric, 0, false, false, 2, 2, 0, 2);
        add(frmOptions, chkGrid, 0, false, false, 2, 2, 0, 2);
        uiFont.add(chkGeneric);
        uiFont.add(chkGrid);

        // Finalize layout
        finish(panControls, 0);
    }




    ///////////////////////////////////////////////////////////////////////////
    //                            Event Handlers                             //
    ///////////////////////////////////////////////////////////////////////////

    // Address text box
    private void onAddress() {
        String text  = txtAddress.getText();
        int    entry = 0;

        // Process the text into an address
        try { entry = (int) Long.parseLong(text, 16); }
        catch (Exception x) { }
        int address = entry & 0x0007FFFF;

        // Update the cell index
        if ((entry >> 24 & 7) == 0 && address >= 0x00020000 &&
            address < 0x00040000)
            index = address - 0x00020000 >> 1;
        setIndex(index);
    }

    // Cell attribute modified
    private void onEdit() {
        int    offset = 0x00020000 | index << 1;
        byte[] vram   = parent.getVRAM();
        int    orig   = vram[offset] & 0xFF | vram[offset + 1] << 8;

        // Compose value
        int value =
            cmbPalette.getSelectedIndex() << 14  |
            (chkHFlip.isSelected() ? 0x2000 : 0) |
            (chkVFlip.isSelected() ? 0x1000 : 0) |
            (Integer) spnCharacter.getValue() & 0x07FF;

        // No action if nothing changed
        if (value == orig)
            return;

        // Push the new value into the emulation state
        parent.getVUE().write(offset, VUE.U16, value, true);
        parent.refresh(false);
    }

    // Generic check box
    private void onGeneric() {
        generic = chkGeneric.isSelected();
        panMap.repaint();
        panPattern.repaint();
    }

    // Grid check box
    private void onGrid() {
        grid = chkGrid.isSelected();
        setScale(scale);
    }

    // Index spinner
    private void onIndex() {
        setIndex((Integer) spnIndex.getValue());
    }

    // Map spinner
    private void onMap() {
        setIndex((Integer) spnMap.getValue() << 12 | index & 0x0FFF);
    }

    // Mouse Down in the map pane
    private void onMouseDownMap(MouseEvent e) {
        int button = e.getButton();
        int x      = e.getX();
        int y      = e.getY();

        // Common processing
        panMap.requestFocus();

        // Only process left clicks
        if (button != MouseEvent.BUTTON1)
            return;

        // Select the cell clicked on, if any
        int scale = this.scale * 8 + (grid ? 1 : 0);
        x /= scale;
        y /= scale;
        if (x >= 0 && x <= 63 && y >= 0 && y <= 63)
            setIndex((index & 0xF000) | y << 6 | x);
    }

    // Painting the controls scroll pane
    private void onPaintControls() {

        // Determine the size of the inner element and the visible area
        Dimension extent = scrControls.getViewport().getExtentSize();
        Dimension prefer = panControls.getPreferredSize();
        if (extent.width == prefer.width)
            return;

        // Resize the container to exactly contain the inner element
        prefer.width  = scrControls.getWidth() + prefer.width - extent.width;
        prefer.height = 0;
        scrControls.setPreferredSize(prefer);
        scrControls.revalidate();
    }

    // Painting the map pane
    private void onPaintMap(Graphics g) {
        int pad   = grid ? 1 : 0;
        int dim   = scale * 8 + pad;

        // Determine the range of characters visible in the pane
        JViewport viewport = scrMap.getViewport();
        Point     pos      = viewport.getViewPosition();
        Dimension size     = viewport.getExtentSize();
        int left   = Math.max(Math.min( pos.x                / dim, 63), 0);
        int top    = Math.max(Math.min( pos.y                / dim, 63), 0);
        int right  = Math.max(Math.min((pos.x + size.width ) / dim, 63), 0);
        int bottom = Math.max(Math.min((pos.y + size.height) / dim, 63), 0);

        // Draw each visible character
        int map = index & 0xF000;
        VIPChrPane chars = (VIPChrPane) parent.getCharacters();
        for (int y = top;  y <= bottom; y++)
        for (int x = left; x <= right;  x++) {

            // Draw character
            VIPCell cell = cells[map | y << 6 | x];
            chars.get(cell.character).paint(g, x * dim, y * dim,
                cell.hFlip, cell.vFlip, scale,
                VIPPalettes.BG0 + cell.palette, generic);

            // Draw selection color
            if ((index & 0x0FFF) != (y << 6 | x))
                continue;
            g.setColor(Util.getColor("selected", 0.5f));
            g.fillRect(x * dim - 1, y * dim - 1, dim + 1, dim + 1);
        }
    }

    // Painting the pattern box
    private void onPaintPattern(Graphics g) {
        cells[index].paint(g, 0, 0, 16, generic);
    }

    // Scale slider changed
    private void onScale() {
        setScale(sldScale.getValue());
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Package Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Retrieve a cell given its index
    VIPCell get(int index) {
        return cells[index];
    }

    // Update UI components with the current emulation state
    void refresh() {

        // Update state fields
        for (int x = 0; x < 65536; x++)
            cells[x].refresh();

        // Update UI elements
        setIndex(index);
    }

    // Retrieve a pixel from a character
    int sample(int index, int x, int y) {
        return cells[index].sample(x, y);
    }

    // Specify new fonts
    void setFonts() {
        Font dlgFont = parent.getDialogFont();
        Font hexFont = parent.getHexFont();

        // Update all appropriate UI controls
        for (int x = 0; x < uiFont.size(); x++) {
            JComponent control = uiFont.get(x);
            if (control == txtAddress)
                control.setFont(hexFont);
            else if (control instanceof JPanel)
                ((TitledBorder) control.getBorder()).setTitleFont(dlgFont);
            else control.setFont(dlgFont);
        }
    }

    // Specify the current selected cell index
    void setIndex(int index) {
        this.index   = index;
        int address  = 0x00020000 + index * 2;
        VIPCell cell = cells[index];

        // Update controls
        updating = true;
        chkHFlip.setSelected(cell.hFlip);
        chkVFlip.setSelected(cell.vFlip);
        cmbPalette.setSelectedIndex(cell.palette);
        spnCharacter.setValue(cell.character);
        spnIndex.setValue(Math.min(Math.max(index,       0), 65535));
        spnMap.setValue  (Math.min(Math.max(index >> 12, 0),    15));
        txtAddress.setText(String.format("%08X", address));
        updating = false;

        // Redraw display
        panMap.repaint();
        panPattern.repaint();
    }

    // Specify a new scaling factor for the map
    void setScale(int scale) {
        this.scale = scale;

        // Update slider
        updating = true;
        sldScale.setValue(scale);
        updating = false;

        // Configure map preview
        resizeMap();
        panMap.revalidate();

        // Configure map scroll pane
        scale = scale * 8 + (grid ? 1 : 0);
        scrMap.getHorizontalScrollBar().setUnitIncrement(scale);
        scrMap.getVerticalScrollBar().setUnitIncrement(scale);
        scrMap.repaint();
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Private Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Add a control to a container
    private void add(JComponent container, JComponent control, int colSpan,
        boolean center, boolean stretch, int top, int left, int bottom,
        int right) {
        GridBagConstraints c = new GridBagConstraints();
        c.gridwidth = colSpan != 0 ? colSpan : GridBagConstraints.REMAINDER;
        c.insets    = new Insets(top, left, bottom, right);
        c.weightx   = colSpan != 0 ? 0 : 1;
        if (!center) c.anchor = GridBagConstraints.LINE_START;
        if (stretch) c.fill   = GridBagConstraints.HORIZONTAL;
        container.add(control, c);
    }

    // Finalize a container layout
    private void finish(JComponent container, int spacing) {
        JPanel spacer = new JPanel(null);
        spacer.setPreferredSize(new Dimension(0, 0));
        spacer.setOpaque(false);
        GridBagConstraints c = new GridBagConstraints();
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.insets    = new Insets(spacing, 0, 0, 0);
        c.weighty   = 1;
        container.add(spacer, c);
    }

    // Resize the contents of the map pane
    private void resizeMap() {
        int scale = this.scale * 8;
        int pad   = grid ? 1 : 0;
        int size  = (scale + pad) * 64 - pad;
        panMap.setPreferredSize(new Dimension(size, size));
    }

}
