package debugger;

// Java imports
import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;
import java.util.*;

// Project imports
import debugger.*;
import util.*;
import vue.*;

// VIP object viewer
class VIPObjPane extends JPanel {

    // Instance fields
    private Color[][]   cColors;  // Pattern colors
    private boolean     generic;  // Force the generic colors
    private int         index;    // Selected object index
    private VIPObject[] objects;  // Objects
    private int[][]     oColors;  // Preview colors
    private Debugger    parent;   // Debugger UI manager
    private int         scale;    // Scaling for object image
    private boolean     updating; // Setting values programmatically

    // UI components
    private JCheckBox         chkGeneric;   // Generic colors check box
    private JCheckBox         chkHFlip;     // Horizontal flip check box
    private JCheckBox         chkLeft;      // Left check box
    private JCheckBox         chkRight;     // Right check box
    private JCheckBox         chkVFlip;     // Vertical flip check box
    private JComboBox<String> cmbPalette;   // Palette combo box
    private BufferedImage     imgFrame;     // Frame buffer
    private JPanel            panControls;  // Controls content panel
    private JPanel            panPattern;   // Pattern box
    private JPanel            panPreview;   // Preview content panel
    private JScrollPane       scrControls;  // Scroll pane for controls
    private JScrollPane       scrPreview;   // Scroll pane for preview
    private JSlider           sldScale;     // Scale slider
    private JSpinner          spnIndex;     // Index spinner
    private JSpinner          spnCharacter; // Character spinner
    private JSpinner          spnParallax;  // Parallax spinner
    private JSpinner          spnX;         // X spinner
    private JSpinner          spnY;         // Y spinner
    private JTextField        txtAddress;   // Address text box
    private ArrayList<JComponent> uiFont;   // UI elements depending on font



    ///////////////////////////////////////////////////////////////////////////
    //                               Constants                               //
    ///////////////////////////////////////////////////////////////////////////

    // Default scale
    private static final int DEFAULTSCALE = 1;



    ///////////////////////////////////////////////////////////////////////////
    //                             Constructors                              //
    ///////////////////////////////////////////////////////////////////////////

    // Default constructor
    VIPObjPane(Debugger parent) {
        super(new BorderLayout());

        // Configure instance fields
        cColors     = new Color[5][4];
        generic     = false;
        index       = 0;
        objects     = new VIPObject[1024];
        oColors     = new int[15][4];
        this.parent = parent;
        scale       = DEFAULTSCALE;
        updating    = false;

        // Configure controls
        imgFrame = new BufferedImage(384, 224, BufferedImage.TYPE_INT_ARGB);
        uiFont   = new ArrayList<JComponent>();
        initPreview();
        initControls();

        // Configure component
        add(scrPreview,  BorderLayout.CENTER    );
        add(scrControls, BorderLayout.LINE_START);
        setIndex(index);
        setScale(scale);
    }

    // Preview pane constructor
    private void initPreview() {

        // Configure content pane
        panPreview = new JPanel(null) {
            public void paintComponent(Graphics g) {
                super.paintComponent(g);
                onPaintPreview(g);
            }
        };
        panPreview.setBackground(Util.getColor("control"));
        panPreview.setFocusable(true);
        panPreview.addMouseListener(Util.onMouse(
            e->panPreview.requestFocus(), null));
        resizePreview();

        // Configure scroll pane
        scrPreview = new JScrollPane(panPreview);

        // Configure objects
        for (int x = 0; x < 1024; x++)
            objects[x] = new VIPObject(parent, x);
    }

    // Control pane constructor
    private void initControls() {

        // Event handlers
        ActionListener onAction = e->{ if (!updating) onEdit(e); };
        ChangeListener onChange = e->{ if (!updating) onEdit(e); };

        // Top-level container
        panControls = new JPanel(new GridBagLayout()) {
            public void paintComponent(Graphics g)
                { super.paintComponent(g); onPaintControls(); }
        };
        panControls.setFocusable(true);
        panControls.addMouseListener(Util.onMouse(
            e->panControls.requestFocus(), null));

        // Scroll pane for container
        scrControls = new JScrollPane(panControls,
            JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED,
            JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
        scrControls.setBorder(null);

        // Index spinner
        JLabel lblIndex = new JLabel("Index");
        spnIndex = new JSpinner(new SpinnerNumberModel(0, 0, 1023, 1));
        spnIndex.setEditor(new JSpinner.NumberEditor(spnIndex, "#"));
        spnIndex.addChangeListener(e->{ if (!updating) onIndex(); });
        add(panControls, lblIndex, 1, false, false, 2, 4, 0, 4);
        add(panControls, spnIndex, 0, false, true, 2, 0, 0, 4);
        uiFont.add(lblIndex);
        uiFont.add(spnIndex);

        // Object group box
        JPanel frmObject = new JPanel(new GridBagLayout());
        frmObject.setBorder(new TitledBorder("Object"));
        add(panControls, frmObject, 0, false, true, 2, 0, 0, 0);
        uiFont.add(frmObject);

        // Pattern panel
        panPattern = new JPanel(null) {
            public void paintComponent(Graphics g)
                { super.paintComponent(g); onPaintPattern(g); }
        };
        panPattern.setPreferredSize(new Dimension(128, 128));
        add(frmObject, panPattern, 0, true, false, 0, 2, 0, 2);

        // Address text box
        JLabel lblAddress = new JLabel("Address");
        txtAddress = new JTextField();
        txtAddress.addActionListener(
            e->{ panControls.requestFocus(); onAddress(); });
        txtAddress.addFocusListener(Util.onFocus(null, e->onAddress()));
        add(frmObject, lblAddress, 1, false, false, 2, 2, 0, 4);
        add(frmObject, txtAddress, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblAddress);
        uiFont.add(txtAddress);

        // X spinner
        JLabel lblX = new JLabel("X");
        spnX = new JSpinner(new SpinnerNumberModel(0, -512, 511, 1));
        spnX.setEditor(new JSpinner.NumberEditor(spnX, "#"));
        spnX.putClientProperty("offset", 0);
        spnX.addChangeListener(onChange);
        add(frmObject, lblX, 1, false, false, 2, 2, 0, 4);
        add(frmObject, spnX, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblX);
        uiFont.add(spnX);

        // Y spinner
        JLabel lblY = new JLabel("Y");
        spnY = new JSpinner(new SpinnerNumberModel(0, -8, 247, 1));
        spnY.setEditor(new JSpinner.NumberEditor(spnY, "#"));
        spnY.putClientProperty("offset", 4);
        spnY.addChangeListener(onChange);
        add(frmObject, lblY, 1, false, false, 2, 2, 0, 4);
        add(frmObject, spnY, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblY);
        uiFont.add(spnY);

        // Parallax spinner
        JLabel lblParallax = new JLabel("Parallax");
        spnParallax = new JSpinner(new SpinnerNumberModel(0, -512, 511, 1));
        spnParallax.setEditor(new JSpinner.NumberEditor(spnParallax, "#"));
        spnParallax.putClientProperty("offset", 2);
        spnParallax.addChangeListener(onChange);
        add(frmObject, lblParallax, 1, false, false, 2, 2, 0, 4);
        add(frmObject, spnParallax, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblParallax);
        uiFont.add(spnParallax);

        // Character spinner
        JLabel lblCharacter = new JLabel("Character");
        spnCharacter = new JSpinner(new SpinnerNumberModel(0, 0, 2047, 1));
        spnCharacter.setEditor(new JSpinner.NumberEditor(spnCharacter, "#"));
        spnCharacter.putClientProperty("offset", 6);
        spnCharacter.addChangeListener(onChange);
        add(frmObject, lblCharacter, 1, false, false, 2, 2, 0, 4);
        add(frmObject, spnCharacter, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblCharacter);
        uiFont.add(spnCharacter);

        // Palette combo box
        JLabel lblPalette = new JLabel("Palette");
        cmbPalette = new JComboBox<String>(new String[]
            { "OBJ 0",  "OBJ 1",  "OBJ 2",  "OBJ 3" });
        cmbPalette.putClientProperty("offset", 6);
        cmbPalette.addActionListener(onAction);
        add(frmObject, lblPalette, 1, false, false, 2, 2, 0, 4);
        add(frmObject, cmbPalette, 0, false, true, 2, 0, 0, 2);
        uiFont.add(lblPalette);
        uiFont.add(cmbPalette);

        // Flip check boxes
        chkHFlip = new JCheckBox("H-flip");
        chkHFlip.setBorder(null);
        chkHFlip.putClientProperty("offset", 6);
        chkHFlip.addActionListener(onAction);
        chkVFlip = new JCheckBox("V-flip");
        chkVFlip.setBorder(null);
        chkVFlip.putClientProperty("offset", 6);
        chkVFlip.addActionListener(onAction);
        add(frmObject, chkHFlip, 1, false, false, 2, 2, 0, 0);
        add(frmObject, chkVFlip, 0, false, false, 2, 2, 0, 2);
        uiFont.add(chkHFlip);
        uiFont.add(chkVFlip);

        // Left and right check boxes
        chkLeft = new JCheckBox("Left");
        chkLeft.setBorder(null);
        chkLeft.putClientProperty("offset", 2);
        chkLeft.addActionListener(onAction);
        chkRight = new JCheckBox("Right");
        chkRight.setBorder(null);
        chkRight.putClientProperty("offset", 2);
        chkRight.addActionListener(onAction);
        add(frmObject, chkLeft, 1, false, false, 2, 2, 0, 0);
        add(frmObject, chkRight, 0, false, false, 2, 2, 0, 2);
        uiFont.add(chkLeft);
        uiFont.add(chkRight);

        // Options group box
        JPanel frmOptions = new JPanel(new GridBagLayout());
        frmOptions.setBorder(new TitledBorder("Options"));
        add(panControls, frmOptions, 0, false, true, 2, 0, 0, 0);
        uiFont.add(frmOptions);

        // Scale slider
        JLabel lblScale = new JLabel("Scale");
        sldScale = new JSlider(1, 10, DEFAULTSCALE);
        sldScale.setFocusable(false);
        sldScale.setLabelTable(null);
        sldScale.setPreferredSize(new Dimension(0,
            sldScale.getPreferredSize().height));
        sldScale.setSnapToTicks(true);
        sldScale.addChangeListener(e->{ if (!updating) onScale(); });
        add(frmOptions, lblScale, 1, false, false, 2, 4, 0, 4);
        add(frmOptions, sldScale, 0, false, true, 2, 0, 0, 4);
        uiFont.add(lblScale);

        // Generic check box
        chkGeneric = new JCheckBox("Generic colors");
        chkGeneric.setBorder(null);
        chkGeneric.addActionListener(e->onGeneric());
        add(frmOptions, chkGeneric, 0, false, false, 2, 2, 0, 2);
        uiFont.add(chkGeneric);

        // Finalize layout
        finish(panControls, 0);
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Event Handlers                             //
    ///////////////////////////////////////////////////////////////////////////

    // Address text box
    private void onAddress() {
        String text  = txtAddress.getText();
        int    entry = 0;

        // Process the text into an address
        try { entry = (int) Long.parseLong(text, 16); }
        catch (Exception x) { }
        int address = entry & 0x0007FFFF;

        // Update the object index
        if ((entry >> 24 & 7) == 0 && address >= 0x0003E000 &&
            address < 0x00040000)
            index = address - 0x0003E000 >> 3;
        setIndex(index);
    }

    // Object attribute modified
    private void onEdit(EventObject e) {
        int    offset = 0x0003E000 | index << 3 | (Integer)
            ((JComponent) e.getSource()).getClientProperty("offset");
        byte[] vram   = parent.getVRAM();
        int    orig   = vram[offset] & 0xFF | vram[offset + 1] << 8;
        int    value  = orig;

        // Compose value by offset
        switch (offset & 7) {
            case 0: value = (Integer) spnX.getValue() & 0x03FF; break;
            case 2: value =
                (chkLeft.isSelected()  ? 0x8000 : 0) |
                (chkRight.isSelected() ? 0x4000 : 0) |
                (Integer) spnParallax.getValue() & 0x03FF;
                break;
            case 4: value = (Integer) spnY.getValue() & 0x00FF; break;
            case 6: value =
                cmbPalette.getSelectedIndex() << 14  |
                (chkHFlip.isSelected() ? 0x2000 : 0) |
                (chkVFlip.isSelected() ? 0x1000 : 0) |
                (Integer) spnCharacter.getValue() & 0x07FF;
                break;
        }

        // No action if nothing changed
        if (value == orig)
            return;

        // Push the new value into the emulation state
        parent.getVUE().write(offset, VUE.U16, value, true);
        parent.refresh(false);
    }

    // Generic check box
    private void onGeneric() {
        generic = chkGeneric.isSelected();
        refreshImage();
        panPattern.repaint();
        panPreview.repaint();
    }

    // Index spinner
    private void onIndex() {
        setIndex((Integer) spnIndex.getValue());
    }

    // Painting the controls scroll pane
    private void onPaintControls() {

        // Determine the size of the inner element and the visible area
        Dimension extent = scrControls.getViewport().getExtentSize();
        Dimension prefer = panControls.getPreferredSize();
        if (extent.width == prefer.width)
            return;

        // Resize the container to exactly contain the inner element
        prefer.width  = scrControls.getWidth() + prefer.width - extent.width;
        prefer.height = 0;
        scrControls.setPreferredSize(prefer);
        scrControls.revalidate();
    }

    // Painting the pattern box
    private void onPaintPattern(Graphics g) {
        objects[index].paint(g, 0, 0, 16, generic);
    }

    // Painting the preview pane
    private void onPaintPreview(Graphics g) {
        g.drawImage(imgFrame, 0, 0, 384 * scale, 224 * scale, null);
    }

    // Scale slider changed
    private void onScale() {
        setScale(sldScale.getValue());
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Package Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Retrieve an object given its index
    VIPObject get(int index) {
        return objects[index];
    }

    // Update UI components with the current emulation state
    void refresh() {

        // Update state fields
        for (int x = 0; x < 1024; x++)
            objects[x].refresh();

        // Update UI elements
        refreshImage();
        panPattern.repaint();
        panPreview.repaint();
        setIndex(index);
    }

    // Retrieve a pixel from an object
    int sample(int index, int x, int y, int eye) {
        return objects[index].sample(x, y, eye);
    }

    // Specify new fonts
    void setFonts() {
        Font dlgFont = parent.getDialogFont();
        Font hexFont = parent.getHexFont();

        // Update all appropriate UI controls
        for (int x = 0; x < uiFont.size(); x++) {
            JComponent control = uiFont.get(x);
            if (control == txtAddress)
                control.setFont(hexFont);
            else if (control instanceof JPanel)
                ((TitledBorder) control.getBorder()).setTitleFont(dlgFont);
            else control.setFont(dlgFont);
        }
    }

    // Specify the current selected object index
    void setIndex(int index) {
        this.index    = index;
        int address   = 0x0003E000 | index << 3;
        VIPObject obj = objects[index];

        // Update controls
        updating = true;
        chkHFlip.setSelected(obj.hFlip);
        chkVFlip.setSelected(obj.vFlip);
        chkLeft.setSelected(obj.left);
        chkRight.setSelected(obj.right);
        cmbPalette.setSelectedIndex(obj.palette);
        spnCharacter.setValue(obj.character);
        spnIndex.setValue(Math.min(Math.max(index, 0), 1023));
        spnParallax.setValue(obj.parallax);
        spnX.setValue(obj.x);
        spnY.setValue(obj.y);
        txtAddress.setText(String.format("%08X", address));
        updating = false;

        // Redraw display
        panPattern.repaint();
        panPreview.repaint();
    }

    // Specify a new scaling factor for the preview
    void setScale(int scale) {
        this.scale = scale;

        // Update slider
        updating = true;
        sldScale.setValue(scale);
        updating = false;

        // Configure preview
        resizePreview();
        panPreview.revalidate();

        // Configure map scroll pane
        scale = scale * 8;
        scrPreview.getHorizontalScrollBar().setUnitIncrement(scale);
        scrPreview.getVerticalScrollBar().setUnitIncrement(scale);
        scrPreview.repaint();
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Private Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Add a control to a container
    private void add(JComponent container, JComponent control, int colSpan,
        boolean center, boolean stretch, int top, int left, int bottom,
        int right) {
        GridBagConstraints c = new GridBagConstraints();
        c.gridwidth = colSpan != 0 ? colSpan : GridBagConstraints.REMAINDER;
        c.insets    = new Insets(top, left, bottom, right);
        c.weightx   = colSpan != 0 ? 0 : 1;
        if (!center) c.anchor = GridBagConstraints.LINE_START;
        if (stretch) c.fill   = GridBagConstraints.HORIZONTAL;
        container.add(control, c);
    }

    // Finalize a container layout
    private void finish(JComponent container, int spacing) {
        JPanel spacer = new JPanel(null);
        spacer.setPreferredSize(new Dimension(0, 0));
        spacer.setOpaque(false);
        GridBagConstraints c = new GridBagConstraints();
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.insets    = new Insets(spacing, 0, 0, 0);
        c.weighty   = 1;
        container.add(spacer, c);
    }

    private void refreshImage() {
        VIPObject obj = objects[index];
        int[]     pix = new int[384 * 224];

        // Draw the frame image
        for (int y = 0, o = 0; y < 224; y++)
        for (int x = 0; x < 384; x++, o++) {
            int left  = obj.sample(x, y, 0);
            int right = obj.sample(x, y, 1);
            pix[o] = parent.getColor(left,  0, generic) |
                     parent.getColor(right, 1, generic);
        }
        imgFrame.setRGB(0, 0, 384, 224, pix, 0, 384);
    }

    // Resize the contents of the preview pane
    private void resizePreview() {
        panPreview.setPreferredSize(new Dimension(384 * scale, 224 * scale));
    }

}
