package debugger;

// Java imports
import java.awt.*;

// Package imports
import debugger.*;

// One object in VIP memory
class VIPObject {

    // Instance fields
    private int      index;  // Index within VIP memory
    private Debugger parent; // Debugger UI manager

    // VIP fields
    int     character; // Index of character to display
    boolean hFlip;     // Character graphic is flipped horizontally
    boolean left;      // Object is drawn to left image
    int     palette;   // Object Palette index
    int     parallax;  // Per-eye horizontal offset
    boolean right;     // Object is drawn to right image
    boolean vFlip;     // Character graphic is flipped vertically
    int     x;         // Horizontal position
    int     y;         // Vertical position



    ///////////////////////////////////////////////////////////////////////////
    //                             Constructors                              //
    ///////////////////////////////////////////////////////////////////////////

    // Default constructor
    VIPObject(Debugger parent, int index) {
        this.index  = index;
        this.parent = parent;
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Package Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Draw the element onto a component
    void paint(Graphics g, int x, int y, int scale, boolean generic) {
        ((VIPChrPane) parent.getCharacters()).get(character).paint(
            g, x, y, hFlip, vFlip, scale, VIPPalettes.OBJ0 + palette, generic);
    }

    // Update data fields with the current emulation state
    void refresh() {
        int    o    = 0x0003E000 | index << 3;
        byte[] vram = parent.getVRAM();
        x           = (vram[o + 0] & 0xFF | vram[o + 1] << 8) << 22 >> 22;
        left        = (vram[o + 3] & 0x80) != 0;
        right       = (vram[o + 3] & 0x40) != 0;
        parallax    = (vram[o + 2] & 0xFF | vram[o + 3] << 8) << 22 >> 22;
        y           =  vram[o + 4]; if (y < -8) y &= 0x00FF;
        palette     =  vram[o + 7] >> 6 & 3;
        hFlip       = (vram[o + 7] & 0x20) != 0;
        vFlip       = (vram[o + 7] & 0x10) != 0;
        character   = (vram[o + 6] & 0xFF | vram[o + 7] << 8) & 0x07FF;
    }

    // Retrieve the brightness value at given screen coordinates
    int sample(int x, int y, int eye) {

        // The object is not drawn for the current eye
        if (eye == 0 && !left || eye == 1 && !right)
            return -1;

        // Check pixel coordinates
        int X = this.x + (eye == 0 ? -parallax : parallax);
        int Y = this.y;
        if (x < X || x >= X + 8 || y < Y || y >= Y + 8)
            return -1;

        // Sample from the character
        int value = ((VIPChrPane) parent.getCharacters()).sample(
            character, x - X, y - Y, hFlip, vFlip);
        return value == 0 ? -1 : parent.getPalettes().palettes
            [VIPPalettes.OBJ0 + palette][value - 1];
    }

}
