package debugger;

// Java imports
import java.awt.*;

// Package imports
import debugger.*;

// One world in VIP memory
class VIPWorld {

    // Instance fields
    private int      index;  // Index within VIP memory
    private Debugger parent; // Debugger UI manager

    // Instance fields
    int     bgBase;     // Base background map
    int     bgHeight;   // Height in maps of background
    int     bgParallax; // Background source parallax
    int     bgWidth;    // Width in maps of background
    int     bgX;        // Background horizontal source coordinate
    int     bgY;        // Background vertical source coordinate
    boolean end;        // Terminate drawing procedure
    int     height;     // Height in pixels
    boolean left;       // World is drawn to left image
    int     mode;       // Window background/object mode
    boolean overplane;  // Use overplane character
    int     overchar;   // Index of overplane character
    int     parallax;   // Per-eye horizontal offset
    int     paramAddr;  // Address of per-row parameters
    boolean right;      // World is drawn to right image
    int     width;      // Width in pixels
    int     x;          // Horizontal position
    int     y;          // Vertical position



    ///////////////////////////////////////////////////////////////////////////
    //                             Constructors                              //
    ///////////////////////////////////////////////////////////////////////////

    // Default constructor
    VIPWorld(Debugger parent, int index) {
        this.index  = index;
        this.parent = parent;
    }



    ///////////////////////////////////////////////////////////////////////////
    //                            Package Methods                            //
    ///////////////////////////////////////////////////////////////////////////

    // Decode window parameters for one scanline
    int[] getParams(int[] params, int line) {
        byte[] vram = parent.getVRAM();
        int    size = mode == 1 ? 4 : 16;
        int    o    = 0x00020000 | paramAddr << 1 | line * size;

        // Create a new parameter buffer if none was given
        if (params == null)
            params = new int[6];
        params[5] = o;

        // Retrieve H-bias parameters
        if (mode == 1) {
            params[0] = vram[o + 0    ] & 0xFF | vram[o + 1    ] << 27 >> 19;
            params[1] = vram[o + 0 | 2] & 0xFF | vram[o + 1 | 2] << 27 >> 19;
            return params;
        }

        // Retrieve affine parameters
        params[0] = vram[o + 0] & 0xFF | vram[o + 1] << 8;
        params[1] = vram[o + 2] & 0xFF | vram[o + 3] << 8;
        params[2] = vram[o + 4] & 0xFF | vram[o + 5] << 8;
        params[3] = vram[o + 6] & 0xFF | vram[o + 7] << 8;
        params[4] = vram[o + 8] & 0xFF | vram[o + 9] << 8;
        return params;
    }

    // Determine whether the world is an object world
    boolean isObjectWorld() {
        return !end && (left || right) && mode == 3;
    }

    // Retreive a pixel from the world given screen coordinates
    int sample(int x, int y, int eye, int group, int[] groups) {
        return mode != 3 ? sampleBG(x, y, eye) :
            sampleOBJ(x, y, eye, group, groups);
    }

    // Retrieve a pixel from an object world
    int sampleOBJ(int x, int y, int eye, int group, int[] groups) {

        // Drawing ended before this object world was encountered
        if (group == -1)
            return -1;

        // Sample from front to back until a pixel is found
        VIPObjPane objects = (VIPObjPane) parent.getObjects();
        int        start   = group == 0 ? 0 : groups[group - 1] + 1 & 1023;
        int        end     = groups[group];
        for (int o = start; ; o = o + 1 & 1023) {
            int value = objects.sample(o, x, y, eye);
            if (value != -1)
                return value;
            if (o == end) break;
        }

        // No object pixel at this location
        return -1;
    }

    // Retrieve a pixel from a background world
    int sampleBG(int x, int y, int eye) {

        // World is not visible for the current eye
        if (eye == 0 && !left || eye == 1 && !right)
            return -1;

        // Check window dimensions
        if (width < 0 || height < 0)
            return -1;
        int W = width  + 1;
        int H = height + 1;
        if (mode != 2 && H < 8)
            H = 8;

        // Check pixel coordinates
        int X = this.x + (eye == 0 ? -parallax : parallax);
        int Y = this.y;
        if (x < X || x >= X + W || y < Y || y >= Y + H)
            return -1;

        // Calculate normal or H-bias background coordinates
        if (mode != 2) {
            x = x - X + bgX + (eye == 0 ? -bgParallax : bgParallax);
            y = y - Y + bgY;
            if (mode == 1)
                x += getParams(null, y - bgY)[eye];
        }

        // Calculate affine background coordinates
        else {
            int[] params = getParams(null, y - Y);
            int   shift  = x - X;
            if (eye == 0 && params[1] < 0) shift -= params[1];
            if (eye == 1 && params[1] > 0) shift += params[1];
            x = (params[0] << 6) + params[3] * shift >> 9;
            y = (params[2] << 6) + params[4] * shift >> 9;
        }

        // Determine which background cell to draw
        int cell = overchar;
        if (
            !overplane ||
            x >= 0 && x < 512 << bgWidth &&
            y >= 0 && y < 512 << bgHeight
        ) {
            W        = 1 << bgWidth;
            H        = 1 << bgHeight;
            int n    = Math.min(8, 1 << bgWidth << bgHeight);
            int base = bgBase & -n;
            X        = x >> 9 & W - 1;
            Y        = y >> 9 & H - 1;
            int map  = base + (Y * W + X & 7);
            cell     = map << 12 | y << 3 & 0x0FC0 | x >> 3 & 63;
        }

        // Sample from the background cell
        return ((VIPBGMapPane) parent.getBGMaps()).sample(
            cell, x & 7, y & 7);
    }

    // Update data fields with the current emulation state
    void refresh() {
        int    o    = 0x0003D800 | index << 5;
        byte[] vram = parent.getVRAM();
        left        = (vram[o +  1] & 0x80) != 0;
        right       = (vram[o +  1] & 0x40) != 0;
        mode        =  vram[o +  1] >> 4 & 3;
        bgWidth     =  vram[o +  1] >> 2 & 3;
        bgHeight    =  vram[o +  1] & 3;
        overplane   = (vram[o +  0] & 0x80) != 0;
        end         = (vram[o +  0] & 0x40) != 0;
        bgBase      =  vram[o +  0] & 15;
        x           = (vram[o +  2] & 0xFF | vram[o +  3] << 8) << 22 >> 22;
        parallax    = (vram[o +  4] & 0xFF | vram[o +  5] << 8) << 22 >> 22;
        y           = (vram[o +  6] & 0xFF | vram[o +  7] << 8);
        bgX         = (vram[o +  8] & 0xFF | vram[o +  9] << 8) << 19 >> 19;
        bgParallax  = (vram[o + 10] & 0xFF | vram[o + 11] << 8) << 17 >> 17;
        bgY         = (vram[o + 12] & 0xFF | vram[o + 13] << 8) << 19 >> 19;
        width       = (vram[o + 14] & 0xFF | vram[o + 15] << 8) << 19 >> 19;
        height      = (vram[o + 16] & 0xFF | vram[o + 17] << 8);
        paramAddr   = (vram[o + 18] & 0xFF | vram[o + 19] << 8) & 0xFFFF;
        overchar    = (vram[o + 20] & 0xFF | vram[o + 21] << 8) & 0xFFFF;
    }

}