#include <malloc.h>
#include <dos.h>

struct ModeList {
  unsigned short Modes;
};

struct VBE_Info {
  long                  VESASignature;
  unsigned short        Version;
  char far              *OEMStringPTR;
  long                  Capabilities;
  struct ModeList far   *VideoModePTR;
  unsigned short        TotalMemory;
  char                  Reserved [235];
};

struct Mode_Info {
  unsigned short        ModeAttributes;
  unsigned char         WinAAttributes;
  unsigned char         WinBAttributes;
  unsigned short        WinGranularity;
  unsigned short        WinSize;
  unsigned short        WinASegment;
  unsigned short        WinBSegment;
  void far              *WinFuncPtr;
  unsigned short        BytesPerScanLine;
  unsigned short        XResolution;
  unsigned short        YResolution;
  unsigned char         XCharSize;
  unsigned char         YCharSize;
  unsigned char         NumberOfPlanes;
  unsigned char         BitsPerPixel;
  unsigned char         NumberOfBanks;
  unsigned char         MemoryModel;
  unsigned char         BankSize;
  unsigned char         NumberOfImagePages;
  unsigned char         Reserved [225];
};

  struct VBE_Info       *VBEInfo;
  struct Mode_Info      *ModeInfo;
  struct REGPACK        Regs;
  unsigned short        WinASegment, WinBSegment;
  char                  CurBank;
  unsigned short        BankAddr [15];
  unsigned short        BytesPerScanLine;
  char far              *Screen;
  //Provisions for reading the screen
  char                  CurReadBank;
  char far              *ReadScreen;
  unsigned short        ReadType;

char IsVBE ();
//function OEMString : String;
char SupportVideoMode (unsigned short Mode);
void VESAGetModeInfo (unsigned short Mode);
char VESASetMode (unsigned short Mode);
char VESASwitchBank (unsigned short Bank);
char VESASwitchReadBank (unsigned short Bank);
void VESAPutPixel (long X, long Y, unsigned char Color);
unsigned char VESAGetPixel (long X, long Y);

//This function checks whether a VESA BIOS is installed
char IsVBE ()
{
  //Assign memory if not yet assigned
  if (VBEInfo == NULL)
    #ifdef __cplusplus
      VBEInfo = (VBE_Info *) (256);
    #else
      VBEInfo = malloc (256);
    #endif

  //Function 0 - Initialize
  Regs.r_ax = 0x4F00;
  Regs.r_es = FP_SEG (VBEInfo);
  Regs.r_di = FP_OFF (VBEInfo);
  intr (0x10, &Regs);

  //Return success if AX = 4F
  return Regs.r_ax == 0x4F;
}

//This functions checks whether a specific video mode is supported
char SupportVideoMode (unsigned short Mode)
{
  unsigned short Counter;

  Counter = 0;
  do {
    //If the video mode is found in the list, return success
    if (VBEInfo->VideoModePTR [Counter].Modes == Mode) {
      return 1;
    }
    //Move on to the next entry in the list
    Counter++;
    //Until the end is reached
  } while (VBEInfo->VideoModePTR [Counter].Modes != 0xFFFF);
  //Return failure
  return 0;
}

void VESAGetModeInfo (unsigned short Mode)
{
  //Assign memory if not yet assigned
  if (ModeInfo == NULL)
    #ifdef __cplusplus
      ModeInfo = (Mode_Info *) (256);
    #else
      ModeInfo = malloc (256);
    #endif

  //Function 1 - Get graphics mode info
  Regs.r_ax = 0x4F01;
  Regs.r_cx = Mode;
  Regs.r_es = FP_SEG (ModeInfo);
  Regs.r_di = FP_OFF (ModeInfo);
  intr (0x10, &Regs);
}

char VESASetMode (unsigned short Mode)
{
  unsigned short Counter;
  char Success;

  //Function 2 - Set graphics mode
  Regs.r_ax = 0x4F02;
  Regs.r_bx = Mode;
  intr (0x10, &Regs);

  //Return success if AX = 4F
  Success = Regs.r_ax == 0x4F;

  //Get information on the graphics mode
  VESAGetModeInfo (Mode);

  //Get the addresses of the banks in linear display memory and store them
  if (ModeInfo->WinGranularity > 0)
  for (Counter = 0; Counter <= 15; Counter++)
    BankAddr [Counter] = (Counter * 64) / ModeInfo->WinGranularity;

  WinASegment      = ModeInfo->WinASegment;
  WinBSegment      = ModeInfo->WinBSegment;
  BytesPerScanLine = ModeInfo->BytesPerScanLine;
  Screen           = MK_FP (WinASegment, 0);
  if (ModeInfo->WinAAttributes & 2) {
    ReadScreen = Screen;
    ReadType = 0;
  } else {
    ReadScreen = MK_FP (WinBSegment, 0);
    ReadType = 1;
  }

  return Success;
}

char VESASwitchBank (unsigned short Bank)
{
  CurBank = Bank;

  //Function 5 - Set SuperVGA window
  Regs.r_ax = 0x4F05;
  Regs.r_bx = 0;
  Regs.r_dx = BankAddr [Bank];
  intr (0x10, &Regs);

  //Return success if AX = 4F
  return Regs.r_ax == 0x4F;
}

char VESASwitchReadBank (unsigned short Bank)
{
  CurReadBank = Bank;

  //Function 5 - Set SuperVGA window
  Regs.r_ax = 0x4F05;
  Regs.r_bx = ReadType;
  Regs.r_dx = BankAddr [Bank];
  intr (0x10, &Regs);

  //Return success if AX = 4F
  return Regs.r_ax == 0x4F;
}

void VESAPutPixel (long X, long Y, unsigned char Color)
{
  long                OffSet;
  unsigned short Bank, Pixel;

  //Only if on screen
  if ((unsigned short)X >= ModeInfo->XResolution || (unsigned short)Y >= ModeInfo->YResolution) return;
  //Calculate linear offset in display memory
  OffSet = (BytesPerScanLine * Y) + X;
  //Calculate offset in 64K segment
  Pixel = OffSet;
  //Calculate bank
  Bank = OffSet >> 16;

  //Switch banks if necessary
  if (CurBank != Bank)
    VESASwitchBank (Bank);

  //Set pixel
  Screen [Pixel] = Color;
}

unsigned char VESAGetPixel (long X, long Y)
{
  long                OffSet;
  unsigned short Bank, Pixel;

  //Only if on screen
  if ((unsigned short)X >= ModeInfo->XResolution || (unsigned short)Y >= ModeInfo->YResolution) return 0;
  //Calculate linear offset in display memory
  OffSet = (BytesPerScanLine * Y) + X;
  //Calculate offset in 64K segment
  Pixel = OffSet;
  //Calculate bank
  Bank = (OffSet >> 16);

  //Switch banks if necessary
  if (CurReadBank != Bank)
    VESASwitchReadBank (Bank);

  //Get pixel
  return ReadScreen [Pixel];
}