#!/bin/bash

# coreutils, awk, mtools
# test with WINE as "HDFMONKEY=hdfmonkey.exe ./stest.sh"
# pass test_xxx 'xxx' name as an argument to run this test only

set -euo pipefail

HDFMONKEY=${HDFMONKEY:-hdfmonkey}

COUNT=0
SUCCESS=0

# keep working files in RAM (most distros have /tmp in RAM now anyway)
if [[ -w /dev/shm ]]; then
    WDIR="/dev/shm"
else
    WDIR="${TMPDIR:-/tmp}"
fi
HDF="$WDIR/hm-st.hdf"
DST="$WDIR/rbld.hdf"
DAT="$WDIR/test.dat"
OFST=$((534+63*512))    # FIXME: for FAT32 only!!!

function _cleanup() {
    if [[ -f "$DAT" ]]; then
        rm -f "$DAT"
    fi
    if [[ -f "$DST" ]]; then
        rm -f "$DST"
    fi
    if [[ -f "$HDF" ]]; then
        rm -f "$HDF"
    fi
}

function _HDFMONKEY() {
    case "$HDFMONKEY" in
        *.exe)
            env WINEDEBUG=-all ${WINE:-wine64} "$HDFMONKEY" "$@"
            ;;
        *)
            "$HDFMONKEY" "$@"
            ;;
    esac
}

function _partition_label() { # FIXME: offset for FAT32 only!!!
    od --skip-bytes=32861 --read-bytes=11 -A n -t c "$1" \
        | tr -d '[0 ]'
}

function _fat_type() {  # FIXME: offset is totally wrong!!! FAT32 only!!!
    od --skip-bytes=32872 --read-bytes=5 -A n -t a "$1" \
        | tr -d '[0 ]'
}

function _setup() { # size-in-mb [opts]
    local OPTS=$(if [[ $# -gt 1 ]]; then echo "$2"; fi)
    _HDFMONKEY create $OPTS "$HDF" ${1}M TST
    test $(($1*1024*1024+534)) -eq $(stat -c %s "$HDF")
    mdir -i "$HDF@@$OFST" > /dev/null
}

function _setup_data_quick() { # size-in-mb [opts]
    local OPTS=$(if [[ $# -gt 1 ]]; then echo "$2"; fi)
    _HDFMONKEY create $OPTS "$HDF" ${1}M TST
    truncate -s 468k "$DAT" # FIXME: od --read-bytes=131072 /dev/urandom > "$DAT"
    _HDFMONKEY put "$HDF" "$DAT" /
}

function assert_fat() {
    test "$1" = "$(_fat_type $HDF)"
}

function _test() {
    while [[ $# -gt 0 ]]; do
        _test_single "$1"
        shift
    done
}

function _test_single() { # func
    set +euo pipefail
    echo -n "$1: ..."
    COUNT=$(($COUNT + 1))
    (
        set -euo pipefail
        "test_$1"
    )
    if [[ $? -eq 0 ]]; then
        SUCCESS=$(($SUCCESS + 1))
        echo -e "\b\b\bOK "
    else
        echo -e "\b\b\bFAILED"
    fi
    set -euo pipefail
    _cleanup
}

function _report() {
    if [[ $SUCCESS -ne $COUNT ]]; then
        printf "\tFAILED $(($COUNT-$SUCCESS)) test(s) out of $COUNT\n"
    else
        printf "\tSUCCESS $COUNT tests\n"
    fi
}

trap '_cleanup' EXIT HUP INT TERM

function test_create() {
    _HDFMONKEY create "$HDF" 1M
    test $((1024*1024+534)) -eq $(stat -c %s "$HDF")
}

function test_create_fat() {
    _setup 33 --fat32
    assert_fat 'FAT32'
}

function test_create_advanced() {
    _HDFMONKEY create --fat32 "$HDF" 44M LBLTST
    assert_fat 'FAT32'
    test 'LBLTST' = $(_partition_label "$HDF")
}

function test_put() {
    _setup 1
    truncate -s 468k "$DAT"
    _HDFMONKEY put "$HDF" "$DAT" /
    mdir -i "$HDF@@$OFST" $(basename "$DAT") > /dev/null
}

function test_free() {
    _setup 1
    _HDFMONKEY free "$HDF" | awk '$2==$5 { print "OK" }' | grep -q 'OK'
    test $? -eq 0
}

function test_free_used() {
    _setup_data_quick 1
    _HDFMONKEY free "$HDF" | awk 'int($2)>int($5) { print "OK" }' | grep -q 'OK'
}

function test_rm_multiple() {
    true
}

function test_rebuild() {
    _setup_data_quick 1
    _HDFMONKEY rebuild "$HDF" "$DST"
    test $(stat -c %s "$HDF") -eq $(stat -c %s "$DST")
    mdir -i "$DST@@$OFST" > /dev/null
    mdir -i "$DST@@$OFST" $(basename "$DAT") > /dev/null
}

function test_rebuild_smaller() {
    _setup_data_quick 2
    _HDFMONKEY rebuild "$HDF" "$DST" TEST2 1M
    test $(stat -c %s "$HDF") -gt $(stat -c %s "$DST")
    mdir -i "$DST@@$OFST" > /dev/null
    mdir -i "$DST@@$OFST" $(basename "$DAT") > /dev/null
}

function test_rebuild_larger() {
    _setup_data_quick 1
    _HDFMONKEY rebuild "$HDF" "$DST" TEST2 2M
    test $(stat -c %s "$HDF") -lt $(stat -c %s "$DST")
    mdir -i "$DST@@$OFST" > /dev/null
    mdir -i "$DST@@$OFST" $(basename "$DAT") > /dev/null
}

function test_rebuild_keep() {
    _setup_data_quick 48 --fat32
    assert_fat 'FAT32'
    _HDFMONKEY rebuild --keep "$HDF" "$DST" TEST2 42M
    test $(stat -c %s "$HDF") -gt $(stat -c %s "$DST")
    test 'FAT32' = "$(_fat_type $DST)"
    mdir -i "$DST@@$OFST" > /dev/null
    mdir -i "$DST@@$OFST" $(basename "$DAT") > /dev/null
}

printf '\thdfmonkey smoke test suite v0.0.1 :)\n'

if [[ $# -gt 0 ]]; then
    while [[ $# -gt 0 ]]; do
        _test "$1"
        shift
    done
    _report
    exit 0
fi

_test create create_fat create_advanced

_test put

_test free free_used

_test rm_multiple

_test rebuild rebuild_smaller rebuild_larger rebuild_keep

_report

# EOF vim: et:ai:ts=4:sw=4:
