/*
 * Decompiled with CFR 0.152.
 */
package com.igormaznitsa.z80;

import com.igormaznitsa.z80.CompileInstructionException;
import com.igormaznitsa.z80.MemoryAccessProvider;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public final class Z80Instruction {
    public static final int SPEC_INDEX = 256;
    public static final int SPEC_OFFSET = 257;
    public static final int SPEC_UNSIGNED_BYTE = 258;
    public static final int SPEC_UNSIGNED_WORD = 259;
    private static final Pattern CODE_PART_CHECKING = Pattern.compile("([0-9A-F]{2})+(\\s+(d|e|nn|n))*(\\s*[0-9A-F]{2}+)?");
    private static final Pattern CODE_PART_PARSING = Pattern.compile("[0-9A-F]{2}|\\s+(?:d|e|nn|n)");
    private static final List<Z80Instruction> INSTRUCTIONS;
    private final Pattern compilePattern;
    private final int[] compileGroupTypes;
    private final int[] instructionCodeTemplate;
    private final String instructionTextTemplate;
    private final int length;
    private final int fixedPartLength;
    private final boolean has_index;
    private final boolean has_offset;
    private final boolean has_byte;
    private final boolean has_word;

    Z80Instruction(String def) {
        String codePart = def.substring(0, 11).trim();
        String asmPart = def.substring(11).trim();
        this.instructionCodeTemplate = Z80Instruction.parseCode(codePart);
        this.instructionTextTemplate = this.checkAsmPart(asmPart);
        int len = 0;
        int fixLength = 0;
        boolean calcFixLength = true;
        boolean hasIndex = false;
        boolean hasOffset = false;
        boolean hasByte = false;
        boolean hasWord = false;
        for (int c : this.instructionCodeTemplate) {
            ++len;
            switch (c) {
                case 256: {
                    hasIndex = true;
                    break;
                }
                case 257: {
                    hasOffset = true;
                    break;
                }
                case 258: {
                    hasByte = true;
                    break;
                }
                case 259: {
                    hasWord = true;
                    ++len;
                    break;
                }
            }
            if (c < 256 && calcFixLength) {
                ++fixLength;
                continue;
            }
            calcFixLength = false;
        }
        this.length = len;
        this.fixedPartLength = fixLength;
        this.has_byte = hasByte;
        this.has_index = hasIndex;
        this.has_offset = hasOffset;
        this.has_word = hasWord;
        String preprocessed = asmPart.replace("+d", "<").replace("e", ">").replace("nn", "%").replace("n", "&");
        StringBuilder builder = new StringBuilder();
        int asmGroupIndex = 0;
        int[] asmGroups = new int[10];
        StringBuilder workBuffer = new StringBuilder();
        for (char c : this.instructionTextTemplate.toCharArray()) {
            if (c == ' ') break;
            workBuffer.append(c);
        }
        workBuffer.setLength(0);
        String group = "(\\S.*)";
        builder.append('^');
        int hexNumCntr = 0;
        block16: for (char c : preprocessed.toCharArray()) {
            switch (c) {
                case '#': {
                    hexNumCntr = 2;
                    continue block16;
                }
                case '<': {
                    builder.append("(\\S.*)");
                    asmGroups[asmGroupIndex++] = 256;
                    continue block16;
                }
                case '>': {
                    builder.append("(\\S.*)");
                    asmGroups[asmGroupIndex++] = 257;
                    continue block16;
                }
                case '%': {
                    builder.append("(\\S.*)");
                    asmGroups[asmGroupIndex++] = 259;
                    continue block16;
                }
                case '&': {
                    builder.append("(\\S.*)");
                    asmGroups[asmGroupIndex++] = 258;
                    continue block16;
                }
                case '(': 
                case ')': 
                case ',': {
                    builder.append("\\s*\\").append(c).append("\\s*");
                    continue block16;
                }
                default: {
                    if (hexNumCntr > 0) {
                        workBuffer.append(c);
                        if (--hexNumCntr != 0) continue block16;
                        asmGroups[asmGroupIndex++] = Integer.parseInt(workBuffer.toString(), 16);
                        workBuffer.setLength(0);
                        builder.append("(\\S.*)");
                        continue block16;
                    }
                    if (Character.isDigit(c)) {
                        asmGroups[asmGroupIndex++] = c - 48;
                        builder.append("(\\S.*)");
                        continue block16;
                    }
                    builder.append(c);
                }
            }
        }
        builder.append('$');
        this.compilePattern = Pattern.compile(builder.toString(), 2);
        this.compileGroupTypes = Arrays.copyOf(asmGroups, asmGroupIndex);
    }

    public static List<Z80Instruction> getInstructions() {
        return INSTRUCTIONS;
    }

    private static int[] parseCode(String codePart) {
        if (CODE_PART_CHECKING.matcher(codePart).matches()) {
            int[] lst = new int[16];
            int index = 0;
            Matcher m = CODE_PART_PARSING.matcher(codePart);
            while (m.find()) {
                int value;
                String str;
                switch (str = m.group().trim()) {
                    case "d": {
                        value = 256;
                        break;
                    }
                    case "e": {
                        value = 257;
                        break;
                    }
                    case "nn": {
                        value = 259;
                        break;
                    }
                    case "n": {
                        value = 258;
                        break;
                    }
                    default: {
                        value = Integer.parseInt(str, 16);
                    }
                }
                lst[index++] = value;
            }
            return Arrays.copyOf(lst, index);
        }
        throw new IllegalArgumentException("Can't recognize byte command description [" + codePart + "]");
    }

    public static String indexToHex(byte index) {
        Object num = Integer.toHexString(Math.abs(index)).toUpperCase(Locale.ENGLISH);
        if (((String)num).length() < 2) {
            num = "0" + (String)num;
        }
        if (index < 0) {
            return "-#" + (String)num;
        }
        return "+#" + (String)num;
    }

    private static String offsetToHex(byte offset, int fixPartLenghtOfCommand, int programCounter) {
        if (programCounter < 0) {
            int theoffset = fixPartLenghtOfCommand + 1 + offset;
            Object num = Integer.toHexString(Math.abs(theoffset)).toUpperCase(Locale.ENGLISH);
            if (((String)num).length() < 2) {
                num = "0" + (String)num;
            }
            if (theoffset < 0) {
                return "PC-#" + (String)num;
            }
            return "PC+#" + (String)num;
        }
        int address = programCounter + offset + fixPartLenghtOfCommand + 1;
        String addressAsHex = Integer.toHexString(Math.abs(address)).toUpperCase(Locale.ENGLISH);
        return "#" + (String)(addressAsHex.length() < 4 ? "0000".substring(0, 4 - addressAsHex.length()) + addressAsHex : addressAsHex);
    }

    private static String unsignedByteToHex(byte value) {
        String str = Integer.toHexString(value & 0xFF).toUpperCase(Locale.ENGLISH);
        return "#" + (String)(str.length() < 2 ? "0" + str : str);
    }

    private static String unsignedWordToHex(byte low, byte hi) {
        String str = Integer.toHexString((hi << 8 | low & 0xFF) & 0xFFFF).toUpperCase(Locale.ENGLISH);
        return "#" + (String)(str.length() < 4 ? "0000".substring(0, 4 - str.length()) + str : str);
    }

    public boolean matches(String asm) {
        return this.compilePattern.matcher(asm.trim()).matches();
    }

    public byte[] compile(String asm, ExpressionProcessor expressionCalc) {
        Matcher m = this.compilePattern.matcher(asm.trim());
        if (m.find()) {
            for (int i = 0; i < this.compileGroupTypes.length; ++i) {
                int result;
                int value = this.compileGroupTypes[i];
                if (value >= 256 || (result = expressionCalc.evalExpression(m.group(i + 1))) == value) continue;
                return null;
            }
            byte[] resultBuff = new byte[16];
            int resultIndex = 0;
            block7: for (int type : this.instructionCodeTemplate) {
                if (type < 256) {
                    resultBuff[resultIndex++] = (byte)type;
                    continue;
                }
                int groupIndex = -1;
                for (int j = 0; j < this.compileGroupTypes.length; ++j) {
                    if (this.compileGroupTypes[j] != type) continue;
                    groupIndex = j + 1;
                    break;
                }
                if (groupIndex < 0) {
                    throw new Error("Unexpected state, group nt found!");
                }
                int result = expressionCalc.evalExpression(m.group(groupIndex));
                switch (type) {
                    case 256: {
                        if (result < -128 || result > 127) {
                            throw new CompileInstructionException(this, "Wrong index value [" + result + "]");
                        }
                        resultBuff[resultIndex++] = (byte)result;
                        continue block7;
                    }
                    case 257: {
                        if (result < -128 || result > 127) {
                            throw new CompileInstructionException(this, "Wrong offset value [" + result + "]");
                        }
                        resultBuff[resultIndex++] = (byte)result;
                        continue block7;
                    }
                    case 258: {
                        resultBuff[resultIndex++] = (byte)result;
                        continue block7;
                    }
                    case 259: {
                        resultBuff[resultIndex++] = (byte)result;
                        resultBuff[resultIndex++] = (byte)(result >>> 8);
                        continue block7;
                    }
                    default: {
                        throw new Error("Unexpected type [" + type + "]");
                    }
                }
            }
            return Arrays.copyOf(resultBuff, resultIndex);
        }
        return null;
    }

    public boolean hasIndex() {
        return this.has_index;
    }

    public boolean hasOffset() {
        return this.has_offset;
    }

    public boolean hasByte() {
        return this.has_byte;
    }

    public boolean hasWord() {
        return this.has_word;
    }

    public int getFixedPartLength() {
        return this.fixedPartLength;
    }

    public int getLength() {
        return this.length;
    }

    public int[] getInstructionCodes() {
        return this.instructionCodeTemplate;
    }

    public boolean matches(MemoryAccessProvider memoryAccessProvider, int offset) {
        block4: for (int j : this.instructionCodeTemplate) {
            switch (j) {
                case 256: 
                case 257: 
                case 258: {
                    ++offset;
                    continue block4;
                }
                case 259: {
                    offset += 2;
                    continue block4;
                }
                default: {
                    if ((memoryAccessProvider.readAddress(offset++) & 0xFF) == j) continue block4;
                    return false;
                }
            }
        }
        return true;
    }

    private String checkAsmPart(String asmPart) {
        String replace = asmPart.replace("+d", "%").replace("e", "%").replace("nn", "%").replace("n", "%");
        for (char c : replace.toCharArray()) {
            switch (c) {
                case 'd': 
                case 'e': 
                case 'n': {
                    throw new IllegalArgumentException("Wrong pattern format detected [" + c + "] in '" + asmPart + "'");
                }
            }
        }
        return asmPart;
    }

    public String decode(MemoryAccessProvider memoryAccessProvider, int address, int pcCounter) {
        String sindex = null;
        String soffset = null;
        String sbyte = null;
        String sword = null;
        block6: for (int j : this.instructionCodeTemplate) {
            switch (j) {
                case 256: {
                    sindex = Z80Instruction.indexToHex(memoryAccessProvider.readAddress(address++));
                    continue block6;
                }
                case 257: {
                    soffset = Z80Instruction.offsetToHex(memoryAccessProvider.readAddress(address++), this.fixedPartLength, pcCounter);
                    continue block6;
                }
                case 258: {
                    sbyte = Z80Instruction.unsignedByteToHex(memoryAccessProvider.readAddress(address++));
                    continue block6;
                }
                case 259: {
                    sword = Z80Instruction.unsignedWordToHex(memoryAccessProvider.readAddress(address++), memoryAccessProvider.readAddress(address++));
                    continue block6;
                }
                default: {
                    if ((memoryAccessProvider.readAddress(address++) & 0xFF) == j) continue block6;
                    return null;
                }
            }
        }
        String result = this.instructionTextTemplate;
        if (sindex != null) {
            result = result.replace("+d", sindex);
        }
        if (soffset != null) {
            result = result.replace("e", soffset);
        }
        if (sbyte != null) {
            result = result.replace("n", sbyte);
        }
        if (sword != null) {
            result = result.replace("nn", sword);
        }
        return result;
    }

    public String toString() {
        return this.instructionTextTemplate;
    }

    static {
        ArrayList<Z80Instruction> list = new ArrayList<Z80Instruction>(1500);
        InputStream in = Z80Instruction.class.getClassLoader().getResourceAsStream("z80opcodes.lst");
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(in, StandardCharsets.UTF_8));){
            String line;
            while ((line = reader.readLine()) != null) {
                String trimmed = line.trim();
                if (trimmed.isEmpty() || trimmed.startsWith("#")) continue;
                list.add(new Z80Instruction(trimmed));
            }
            INSTRUCTIONS = Collections.unmodifiableList(list);
        }
        catch (IOException ex) {
            throw new Error("Can't load Z80 instruction list", ex);
        }
    }

    public static interface ExpressionProcessor {
        public int evalExpression(String var1);
    }
}

