
#include "shared.h"

int context;            /* Active context index */
t_cpu cpu[MAX_68000];   /* CPU structures */

/*--------------------------------------------------------------------------*/
/* Interface routines                                                       */
/*--------------------------------------------------------------------------*/


int cpu_init(void)
{
    int count;
    int size;

    /* Initialize Starscream */
    s68000init();
    size = s68000GetContextSize();
    
    /* Set up each context */
    for(count = 0; count < MAX_68000; count += 1)
    {
        /* Clear memory structures and indices */
        memset(&cpu[count], 0, sizeof(t_cpu));

        /* Make new context structure */
        cpu[count].context = (void *)malloc(size);
        if(!cpu[count].context) return (0);
        memset(cpu[count].context, 0, size);

        /* Point to memory region structures */
        s68000SetContext(&cpu[count]);
        s68000context.s_fetch     = s68000context.u_fetch     = cpu[count].fetch;
        s68000context.s_readbyte  = s68000context.u_readbyte  = cpu[count].readbyte;
        s68000context.s_readword  = s68000context.u_readword  = cpu[count].readword;
        s68000context.s_writebyte = s68000context.u_writebyte = cpu[count].writebyte;
        s68000context.s_writeword = s68000context.u_writeword = cpu[count].writeword;
        s68000GetContext(&cpu[count]);
    }

    /* Set up first context */
    context = 0;
    s68000SetContext(&cpu[context]);

    return (1);
}


int cpu_reset(int num)
{
    UPDATE_CONTEXT;
    return (s68000reset());
}


int cpu_execute(int num, int cycles)
{
    UPDATE_CONTEXT;
    return (s68000exec(cycles));
}


int cpu_cause_interrupt(int num, int level)
{
    UPDATE_CONTEXT;
    return (s68000interrupt(level, -1));
}


void cpu_flush_interrupts(int num)
{
    UPDATE_CONTEXT;
    s68000flushInterrupts();
}


int cpu_readpc(int num)
{
    UPDATE_CONTEXT;
    return (s68000readPC());
}


void cpu_stop(int num)
{
    UPDATE_CONTEXT;
    s68000releaseTimeslice();
}


/*--------------------------------------------------------------------------*/
/* Memory map routines                                                      */
/*--------------------------------------------------------------------------*/


void cpu_add_fetch(int num, unsigned lowaddr, unsigned highaddr, unsigned offset)
{
    struct STARSCREAM_PROGRAMREGION *p = NULL;

    /* Update current memory region */
    p = &cpu[num].fetch[(cpu[num].index[0])];
    p->lowaddr  = lowaddr;
    p->highaddr = highaddr;
    p->offset   = offset - lowaddr;

    /* Point to next region */
    cpu[num].index[0] += 1;
    
    /* Set up end marker */
    p = &cpu[num].fetch[(cpu[num].index[0])];
    p->lowaddr  = -1;
    p->highaddr = -1;
    p->offset   = NULL;
}


void cpu_add_read_byte(int num, int lowaddr, int highaddr, void *memorycall, void *userdata)
{
    struct STARSCREAM_DATAREGION *p = NULL;

    /* Update current memory region */
    p = &cpu[num].readbyte[(cpu[num].index[1])];
    p->lowaddr    = lowaddr;
    p->highaddr   = highaddr;
    p->memorycall = memorycall;
    p->userdata   = userdata;

    /* Point to next region */
    cpu[num].index[1] += 1;
    
    /* Set up end marker */
    p = &cpu[num].readbyte[(cpu[num].index[1])];
    p->lowaddr    = -1;
    p->highaddr   = -1;
    p->memorycall = NULL;
    p->userdata   = NULL;
}


void cpu_add_read_word(int num, int lowaddr, int highaddr, void *memorycall, void *userdata)
{
    struct STARSCREAM_DATAREGION *p = NULL;

    /* Update current memory region */
    p = &cpu[num].readword[(cpu[num].index[2])];
    p->lowaddr    = lowaddr;
    p->highaddr   = highaddr;
    p->memorycall = memorycall;
    p->userdata   = userdata;

    /* Point to next region */
    cpu[num].index[2] += 1;
    
    /* Set up end marker */
    p = &cpu[num].readword[(cpu[num].index[2])];
    p->lowaddr    = -1;
    p->highaddr   = -1;
    p->memorycall = NULL;
    p->userdata   = NULL;
}


void cpu_add_write_byte(int num, int lowaddr, int highaddr, void *memorycall, void *userdata)
{
    struct STARSCREAM_DATAREGION *p = NULL;

    /* Update current memory region */
    p = &cpu[num].writebyte[(cpu[num].index[3])];
    p->lowaddr    = lowaddr;
    p->highaddr   = highaddr;
    p->memorycall = memorycall;
    p->userdata   = userdata;

    /* Point to next region */
    cpu[num].index[3] += 1;
    
    /* Set up end marker */
    p = &cpu[num].writebyte[(cpu[num].index[3])];
    p->lowaddr    = -1;
    p->highaddr   = -1;
    p->memorycall = NULL;
    p->userdata   = NULL;
}


void cpu_add_write_word(int num, int lowaddr, int highaddr, void *memorycall, void *userdata)
{
    struct STARSCREAM_DATAREGION *p = NULL;

    /* Update current memory region */
    p = &cpu[num].writeword[(cpu[num].index[4])];
    p->lowaddr    = lowaddr;
    p->highaddr   = highaddr;
    p->memorycall = memorycall;
    p->userdata   = userdata;

    /* Point to next region */
    cpu[num].index[4] += 1;
    
    /* Set up end marker */
    p = &cpu[num].writeword[(cpu[num].index[4])];
    p->lowaddr    = -1;
    p->highaddr   = -1;
    p->memorycall = NULL;
    p->userdata   = NULL;
}


void cpu_add_write(int num, int lowaddr, int highaddr, void *memorycall, void *userdata)
{
    cpu_add_write_byte(num, lowaddr, highaddr, memorycall, userdata);
    cpu_add_write_word(num, lowaddr, highaddr, memorycall, userdata);
}


void cpu_add_read(int num, int lowaddr, int highaddr, void *memorycall, void *userdata)
{
    cpu_add_read_byte(num, lowaddr, highaddr, memorycall, userdata);
    cpu_add_read_word(num, lowaddr, highaddr, memorycall, userdata);
}


void cpu_add_readwrite(int num, int lowaddr, int highaddr, void *memorycall, void *userdata)
{
    cpu_add_write_byte(num, lowaddr, highaddr, memorycall, userdata);
    cpu_add_write_word(num, lowaddr, highaddr, memorycall, userdata);
    cpu_add_read_byte(num, lowaddr, highaddr, memorycall, userdata);
    cpu_add_read_word(num, lowaddr, highaddr, memorycall, userdata);
}
