
#include "shared.h"

#define LOG_ERROR       (0)     /* Log error messages to 'error.log' */
#define LOG_CYM         (0)     /* Log YM2151 output to .CYM file */

/* Rendering data */
byte obj_cache[0x100000];       /* Unpacked 4-bpp object graphics (dynamic) */
byte bg_cache[0x2000][8][8];    /* 3-bpp tiles converted to 8-bpp tiles (dynamic) */
byte bg_tile_info[0x2000];      /* Transparency state of each tile (dynamic) */

/* Pixel conversion tables */
word pixel[PALETTE_SIZE];       /* Palette -> 5:6:5 pixel lookup */

/* Video hardware registers */
byte tile_bank[2];              /* Tile banks */
word fg_page[4];                /* Foreground page addresses */
word bg_page[4];                /* Background page addresses */
word fg_scrollx, fg_scrolly;    /* Foreground scroll regs. */
word bg_scrollx, bg_scrolly;    /* Background scroll regs. */

/* System 16 data */
byte rom[0x40000];              /* 256k program ROM */
byte ram[0x4000];               /* 16k RAM */
byte tileram[0x10000];          /* 64k tile RAM */
byte textram[0x1000];           /* 4k text RAM */
byte objram[0x1000];            /* 4k object RAM */
byte palram[0x1000];            /* 4k color RAM */

/* Sound emulation data */
byte z80_ram[0x800];            /* 2k RAM */
byte z80_rom[0x8000];           /* 32k program ROM */
byte sound_command;             /* Latch for 68000 -> Z80 comms. */
byte ym2151_reg[0x100];         /* YM2151 registers */
byte ym2151_latch = 0;          /* YM2151 register latch */
byte ym2151_toggle = 0;         /* Dummy value for the YM251 status port */

/*--------------------------------------------------------------------------*/

void init_system(void)
{
    /* Clear memory */
    memset(obj_cache, 0, sizeof(obj_cache));
    memset(bg_cache, 0, sizeof(bg_cache));
    memset(rom, 0, sizeof(rom));
    memset(ram, 0, sizeof(ram));
    memset(z80_rom, 0, sizeof(z80_rom));
    memset(z80_ram, 0, sizeof(z80_ram));
    memset(tileram, 0, sizeof(tileram));
    memset(textram, 0, sizeof(textram));
    memset(objram, 0, sizeof(objram));
    memset(palram, 0, sizeof(palram));

    tile_bank[0] = 0x00;
    tile_bank[1] = 0x01;
    fg_page[0]   = 0x0000;
    fg_page[1]   = 0x0000;
    fg_page[2]   = 0x0000;
    fg_page[3]   = 0x0000;
    bg_page[0]   = 0x0000;
    bg_page[1]   = 0x0000;
    bg_page[2]   = 0x0000;
    bg_page[3]   = 0x0000;
    fg_scrollx = fg_scrolly = 0;
    bg_scrollx = bg_scrolly = 0;

    /* Load the 68000 program ROM */
    load_file_even("rom/shinobi.a1", rom + 0x00000, 0x10000);
    load_file_odd ("rom/shinobi.a4", rom + 0x00000, 0x10000);
    load_file_even("rom/shinobi.a2", rom + 0x20000, 0x10000);
    load_file_odd ("rom/shinobi.a5", rom + 0x20000, 0x10000);

    /* Load the Z80 program ROM */
    load_file("rom/shinobi.a7", z80_rom, 0x8000);

    /* Load and convert the object and background graphics ROMs */
    cache_bg_tiles();
    cache_obj_tiles();

    /* Set up Z80 emulation */
    z80_init_memmap();
    z80_map_fetch(0x0000, 0x7FFF, z80_rom);
    z80_map_read (0x0000, 0x7FFF, z80_rom);
    z80_map_fetch(0xF800, 0xFFFF, z80_ram);
    z80_map_read (0xF800, 0xFFFF, z80_ram);
    z80_map_write(0xF800, 0xFFFF, z80_ram);
    z80_set_in(z80_read_port);
    z80_set_out(z80_write_port);
    z80_end_memmap();
    z80_reset();

    /* Set up 68000 emulation */
    cpu_init();
    cpu_add_fetch     (0, 0x000000, 0x03FFFF, (unsigned)rom);
    cpu_add_fetch     (0, 0xFFFC00, 0xFFFFFF, (unsigned)ram);
    cpu_add_read      (0, 0x000000, 0x03FFFF, NULL, rom);
    cpu_add_readwrite (0, 0x400000, 0x40FFFF, NULL, tileram);
    cpu_add_readwrite (0, 0x410000, 0x410FFF, NULL, textram);
    cpu_add_readwrite (0, 0x440000, 0x440FFF, NULL, objram);
    cpu_add_readwrite (0, 0x840000, 0x840FFF, NULL, palram);
    cpu_add_readwrite (0, 0xFFC000, 0xFFFFFF, NULL, ram);
    cpu_add_write_byte(0, 0x000000, 0xFFFFFF, m68k_write_memory_8, NULL);
    cpu_add_write_word(0, 0x000000, 0xFFFFFF, m68k_write_memory_16, NULL);
    cpu_add_read_byte (0, 0x000000, 0xFFFFFF, m68k_read_memory_8, NULL);
    cpu_add_read_word (0, 0x000000, 0xFFFFFF, m68k_read_memory_16, NULL);
    cpu_reset(0);
}

/*--------------------------------------------------------------------------*/
/* ROM file loading                                                         */
/*--------------------------------------------------------------------------*/

void cache_bg_tiles(void)
{
    char *file[] = {"rom/shinobi.b9","rom/shinobi.b10","rom/shinobi.b11"};
    byte *plane[3];
    int name, x, y, c, i, j, count;

    /* Load up the bitplane files */
    for(j = 0; j < 3; j += 1)
    {
        plane[j] = malloc(0x10000);
        load_file(file[j], plane[j], 0x10000);
    }

    /* Decode 3-bit pixel data into 8-bit pixels */
    for(name = 0; name < 0x2000; name += 1)
    {
        bg_tile_info[name] = count = 0;

        /* Tile rows */
        for(y = 0; y < 8; y += 1)
        {
            /* Byte offset into bitplanes */
            i = (name << 3) | (y);

            /* Tile columns */
            for(x = 0; x < 8; x += 1)
            {
                /* Assemble pixel from bitplanes */
                c  = ((plane[0][i] >> (7 - x)) & 1) << 0;
                c |= ((plane[1][i] >> (7 - x)) & 1) << 1;
                c |= ((plane[2][i] >> (7 - x)) & 1) << 2;
                c &= 0x07;

                if(c == 0x00) count += 1;

                /* Save pixel in tile cache */
                bg_cache[name][y][x] = c;
            }
        }

        /* Record tile attributes */
        if(count == 0x40)
            bg_tile_info[name] |= TILE_TRANSPARENT;
        else if(count == 0x00)
            bg_tile_info[name] |= TILE_OPAQUE;
        else if(count != 0x00)
            bg_tile_info[name] |= TILE_SEMITRANSPARENT;
    }

    /* Free bitplane memory */
    for(j = 0; j < 3; j += 1) free (plane[j]);
}

void cache_obj_tiles(void)
{
    char *file[] = {"rom/shinobi.b1","rom/shinobi.b5",
                    "rom/shinobi.b2","rom/shinobi.b6",
                    "rom/shinobi.b3","rom/shinobi.b7",
                    "rom/shinobi.b4","rom/shinobi.b8"};
    int x, c;
    byte *buf = NULL;

    /* Load object pattern data */
    buf = malloc(0x80000);
    load_file_even(file[1], buf + 0x00000, 0x10000);
    load_file_odd (file[0], buf + 0x00000, 0x10000);
    load_file_even(file[3], buf + 0x20000, 0x10000);
    load_file_odd (file[2], buf + 0x20000, 0x10000);
    load_file_even(file[5], buf + 0x40000, 0x10000);
    load_file_odd (file[4], buf + 0x40000, 0x10000);
    load_file_even(file[7], buf + 0x60000, 0x10000);
    load_file_odd (file[6], buf + 0x60000, 0x10000);

    for(x = 0; x < 0x80000; x += 1)
    {
        c = (buf[x] >> 4) & 0x0F;
        if(c == 0x0F) c = 0x00;
        obj_cache[(x << 1) | (0)] = c;

        c = (buf[x] >> 0) & 0x0F;
        if(c == 0x0F) c = 0x00;
        obj_cache[(x << 1) | (1)] = c;
    }

    /* Free 512k sprite ROM buffer */
    if(buf) free (buf);
}

/*--------------------------------------------------------------------------*/
/* Z80 Memory handlers                                                      */
/*--------------------------------------------------------------------------*/

UBYTE z80_read_port(UWORD port)
{
    switch(port & 0xFF)
    {
        case 0x01: /* YM2151 status */
            ym2151_toggle ^= 0xFF;
            return (ym2151_toggle);

        case 0xC0: /* Sound command latch */
            return (sound_command);
    }

    error("z80 read port %02X (%04X)\n", port & 0xFF, z80_get_reg(Z80_REG_PC));
    return (0xFF);
}

void z80_write_port(UWORD port, UBYTE data)
{
    switch(port & 0xFF)
    {
        case 0x00: /* YM2151 register latch */
            ym2151_latch = (data & 0xFF);            
            return;

        case 0x01: /* YM2151 data port */
            ym2151_reg[ym2151_latch] = data;
#if LOG_CYM
            fputc(ym2151_latch, cym_log);
            fputc(ym2151_reg[ym2151_latch], cym_log);
#endif
            return;

        case 0x40: /* UPD7759 */
            return;

        case 0x80: /* UPD7759 */
            return;
    }

    error("z80 write port %04X = %02X (%04X)\n", port & 0xFF, data, z80_get_reg(Z80_REG_PC));
}

/*--------------------------------------------------------------------------*/
/* 68000 Memory handlers                                                    */
/*--------------------------------------------------------------------------*/

void m68k_write_memory_8(int address, int data)
{
    switch(address)
    {
        case 0xFE0007: /* Z80 comms. */
            sound_command = (data & 0xFF);
            z80_raise_IRQ(0xFF);
            z80_emulate(32); /* Ensure interrupt is taken */
            z80_lower_IRQ();
            return;

        case 0xC43001: /* ? */
            return;

        case 0xC40001: /* ? */
            return;
    }

    error("write byte %02X to %08X (%08X)\n", data, address, cpu_readpc(0));
}


void m68k_write_memory_16(int address, int data)
{
    error("write word %04X to %08X (%08X)\n", data, address, cpu_readpc(0));
}


int m68k_read_memory_8(int address)
{
    byte temp = 0xFF;

    switch(address)
    {
        case 0xC41001: /* General controls */
            if(key[KEY_1])      temp &= ~0x01; /* Coin 1 */
            if(key[KEY_2])      temp &= ~0x02; /* Coin 2 */
            if(key[KEY_3])      temp &= ~0x10; /* 1P Start */
            if(key[KEY_4])      temp &= ~0x20; /* 2P Start */
            if(key[KEY_5])      temp &= ~0x04; /* Test */
            if(key[KEY_6])      temp &= ~0x08; /* Service */
            if(key[KEY_7])      temp &= ~0x40; /* Unused */
            if(key[KEY_8])      temp &= ~0x80; /* Unused */
            return (temp);
        
        case 0xC41003: /* Player 1 controls */
            if(key[KEY_LEFT])   temp &= ~0x80; /* Left */
            if(key[KEY_RIGHT])  temp &= ~0x40; /* Right */
            if(key[KEY_UP])     temp &= ~0x20; /* Up */
            if(key[KEY_DOWN])   temp &= ~0x10; /* Down */
            if(key[KEY_F])      temp &= ~0x08; /* Unused */
            if(key[KEY_D])      temp &= ~0x04; /* Jump */
            if(key[KEY_S])      temp &= ~0x02; /* Attack */
            if(key[KEY_A])      temp &= ~0x01; /* Magic */
            return (temp);

        case 0xC41007: /* Player 2 controls */
            return (temp);

        case 0xC42001: /* DIP switch #1 */
            return (temp);

        case 0xC42003: /* DIP switch #2 */
            return (temp);
    }

    error("read byte from %08X (%08X)\n", address, cpu_readpc(0));
    return (temp);
}


int m68k_read_memory_16(int address)
{
    error("read word from %08X (%08X)\n", address, cpu_readpc(0));
    return (0);
}

/*--------------------------------------------------------------------------*/
/* Palette compression stuff                                                */
/*--------------------------------------------------------------------------*/

void compress_palette(void)
{
    int index;
    word temp;

    for(index = 0; index < (PALETTE_SIZE); index += 1)
    {
        int r, g, b;
        temp = palram[(index << 1) | (1)] << 8 | palram[(index << 1)];

        r = (temp & 0x000F) << 1;
        g = (temp & 0x00F0) >> 2;
        b = (temp & 0x0F00) >> 7;
        if(temp & 0x1000) r |= 1;
        if(temp & 0x4000) g |= 2;
        if(temp & 0x8000) g |= 1;
        if(temp & 0x2000) b |= 1;

        pixel[index] = makecol16(r << 3, g << 2, b << 3);
    }
}

