
#include "shared.h"

FILE *errorlog = NULL;          /* File handle for error message logging */
FILE *cym_log = NULL;           /* File handle for CYM logging */
BITMAP *bmp = NULL;             /* Output bitmap for showing display */
int layer_enable = -1;          /* Layer enable flags */
int running = 1;                /* Emulator running switch */
int frame_count = 0;            /* Frame count */
int frame_skip = 2;             /* Frame skip level */
int scanlines = 0;


int main (int argc, char **argv)
{
    if(stricmp(argv[1], "-about") == 0)
    {
        printf("\nSega System 16 Emulator\n");
        printf("(C) 2000  Charles Mac Donald\n");
        printf("Version 0.1, build date: %s, %s\n", __DATE__, __TIME__);
        printf("Usage: sys <gamename> [-options]\n");
        printf("Type 'sys -help' for a summary of the available options\n");
        exit(1);
    }

    if(stricmp(argv[1], "-scanlines") == 0)
    {
        scanlines = 1;
    }

    /* Initialize the emulation */
    init_machine();
    init_system();

    /* Main loop */
    while(running)
    {
        dword ret;

        /* Frame events */
        frame_count += 1;
#if LOG_CYM
        fputc(0x00, cym_log);
#endif

        /* Handle user input */
        if(key[KEY_ESC] || key[KEY_END]) running = 0;
        if(check_key(KEY_TAB))
        {
            cpu_reset(0);
            z80_reset();
        }

        /* Run CPU's for a frame */
        ret = cpu_execute(0, 636 * 262);
        if(ret != 0x80000000) {
            error("starscream error: %08X\n", ret);
            running = 0;
        }
        cpu_cause_interrupt(0, 4);
        z80_emulate(227 * 262);

        if(check_key(KEY_F1)) frame_skip = 1; 
        if(check_key(KEY_F2)) frame_skip = 2;
        if(check_key(KEY_F3)) frame_skip = 3; 
        if(check_key(KEY_F4)) frame_skip = 4; 

        if(check_key(KEY_Q)) layer_enable ^= 0x01;
        if(check_key(KEY_W)) layer_enable ^= 0x02;
        if(check_key(KEY_E)) layer_enable ^= 0x04;
        if(check_key(KEY_R)) layer_enable ^= 0x08;
        if(check_key(KEY_T)) layer_enable = -1;

        /* Update video */
        if(frame_count % frame_skip == 0)
        {
            compress_palette();
            update_video();
            if(scanlines)
            {
                int y;
                for(y = 0; y < 224; y += 1)
                blit(bmp, screen, MARGIN, y, CENTER_X, 16+(y<<1), 320, 1);
            }
            else
            {
                blit(bmp, screen, MARGIN, 0, CENTER_X, CENTER_Y, 320, 224);
            }
        }
    }

    trash_machine();
    return (0);
}


void init_machine(void)
{
    int ret;

    /* Set up Allegro */
    allegro_init();
    set_color_depth(16);
    ret = set_gfx_mode(GFX_AUTODETECT, 320, (scanlines?480:240), 0, 0);
    if(ret != 0) {
        printf("Error setting graphics mode.\nAllegro says: `%s'\n", allegro_error);
        exit(1);
    }

    install_keyboard();
    bmp = create_bitmap(512, 256);
    clear(bmp);

#if LOG_ERROR
    /* Open the error log file */
    errorlog = fopen("error.log", "w");
#endif

#if LOG_CYM
    /* Open the CYM log file */
    cym_log = fopen("shinobi.cym","wb");
#endif

}

void trash_machine(void)
{
    set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
    error("PC %08X\n", cpu_readpc(0));

#if LOG_CYM
    if(cym_log) fclose(cym_log);
#endif

#if LOG_ERROR
    if(errorlog) fclose(errorlog);
#endif
}


int check_key(int code)
{
    static char lastbuf[0x100] = {0};

    if((!key[code]) && (lastbuf[code] == 1))
        lastbuf[code] = 0;

    if((key[code]) && (lastbuf[code] == 0))
    {
        lastbuf[code] = 1;
        return (1);
    }                                                                    

    return (0);
}

void load_file(char *filename, char *mem, int size)
{
    FILE *fd = NULL;
    fd = fopen(filename, "rb");
    if(!fd) exit(1);
    fread(mem, size, 1, fd);
    fclose(fd);
}

void load_file_even(char *filename, char *mem, int size)
{
    FILE *fd = NULL;
    char *buf = NULL;
    int count;
    fd = fopen(filename, "rb");
    if(!fd) exit(1);
    buf = malloc(size);
    fread(buf, size, 1, fd);
    fclose(fd);
    for(count = 0; count < size; count += 1)
        mem[(count << 1) | (0)] = buf[count];
    free(buf);
}

void load_file_odd(char *filename, char *mem, int size)
{
    FILE *fd = NULL;
    char *buf = NULL;
    int count;
    fd = fopen(filename, "rb");
    if(!fd) exit(1);
    buf = malloc(size);
    fread(buf, size, 1, fd);
    fclose(fd);
    for(count = 0; count < size; count += 1)
        mem[(count << 1) | (1)] = buf[count];
    free(buf);
}

void save_file(char *filename, char *mem, int size)
{
    FILE *fd = NULL;
    fd = fopen(filename, "wb");
    if(!fd) exit(1);
    fwrite(mem, size, 1, fd);
    fclose(fd);
}

void error(char *fmt, ...)
{
    va_list ap;
    va_start(ap, fmt);
    if(errorlog) vfprintf(errorlog, fmt, ap);
    va_end(ap);
}


