#define _USE_MATH_DEFINES
#include "Butterworth.h"
#include <stdlib.h>
#include <math.h>

Butterworth::Butterworth(int _n, double s, double f):
	w2(nullptr),
	w1(nullptr),
	w0(nullptr),
	d2(nullptr),
	d1(nullptr),
	A(nullptr) {
	recalc(_n, s, f);
}

Butterworth::~Butterworth() {
	flush();
}

void Butterworth::flush() {
	if (w2) free(w2);
	if (w1) free(w1);
	if (w0) free(w0);
	if (d2) free(d2);
	if (d1) free(d1);
	if (A) free(A);
	w2 = w1 = w0 = d2 = d1 = A = nullptr;
}

double Butterworth::process(double x) {
	for(int i =0; i <n; ++i) {
		w0[i] = d1[i]*w1[i] + d2[i]*w2[i] + x;
		x = A[i]*(w0[i] + 2.0*w1[i] + w2[i]);
		w2[i] = w1[i];
		w1[i] = w0[i];
	}
	return x;
}

void Butterworth::recalc(int _n, double s, double f) {
	flush();
	n = _n/2;
	A = (double *)malloc(n*sizeof(double));
	d1 = (double *)malloc(n*sizeof(double));
	d2 = (double *)malloc(n*sizeof(double));
	w0 = (double *)calloc(n, sizeof(double));
	w1 = (double *)calloc(n, sizeof(double));
	w2 = (double *)calloc(n, sizeof(double));
	double a = tan(M_PI*f/s);
	double a2 = a*a;
	double r;
	for(int i =0; i <n; ++i) {
		r = sin(M_PI*(2.0*i+1.0)/(4.0*n));
		s = a2 + 2.0*a*r + 1.0;
		A[i] = a2/s;
		d1[i] = 2.0*(1-a2)/s;
		d2[i] = -(a2 - 2.0*a*r + 1.0)/s;
	}
}

ChebyshevII::ChebyshevII(int _n, double s, double f):
	w2(nullptr),
	w1(nullptr),
	w0(nullptr),
	d2(nullptr),
	d1(nullptr),
	A(nullptr) {
	recalc(_n, s, f);
}

ChebyshevII::~ChebyshevII() {
	flush();
}

void ChebyshevII::flush() {
	if (w2) free(w2);
	if (w1) free(w1);
	if (w0) free(w0);
	if (d2) free(d2);
	if (d1) free(d1);
	if (A) free(A);
	w2 = w1 = w0 = d2 = d1 = A = nullptr;
}

double ChebyshevII::process(double x) {
	for(int i =0; i <n; ++i) {
		w0[i] = d1[i]*w1[i] + d2[i]*w2[i] + x;
		x = A[i]*(w0[i] + 2.0*w1[i] + w2[i]);
		w2[i] = w1[i];
		w1[i] = w0[i];
	}
	return x;
}

void ChebyshevII::recalc(int _n, double s, double f) {
	flush();
	n = _n / 2; // Number of second-order sections
	if (n <= 0) return; // Prevent division by zero
	A = (double *)malloc(n * sizeof(double));
	d1 = (double *)malloc(n * sizeof(double));
	d2 = (double *)malloc(n * sizeof(double));
	w0 = (double *)calloc(n, sizeof(double));
	w1 = (double *)calloc(n, sizeof(double));
	w2 = (double *)calloc(n, sizeof(double));

	// Pre-warp the frequency
	double fs = s; // Sampling frequency
	double fc = f; // Stopband edge frequency
	double a = tan(M_PI * fc / fs);
	//if (a <= 0 || !std::isfinite(a)) return; // Prevent invalid pre-warping
	double a2 = a * a;

	// Hardcode stopband ripple (40 dB)
	double ripple_dB = 40.0;
	double As = pow(10.0, ripple_dB / 20.0);
	double epsilon = sqrt(1.0 / (As * As) - 1.0);
	//if (epsilon <= 0 || !std::isfinite(epsilon)) return; // Prevent invalid epsilon
	double v0 = asinh(1.0 / epsilon) / n;
	//if (!std::isfinite(v0)) return; // Prevent invalid v0

	for (int i = 0; i < n; ++i) {
		// Pole angle
		double theta = M_PI * (2.0 * i + 1.0) / (2.0 * n);
		double sigma = sinh(v0) * cos(theta);
		double omega = cosh(v0) * sin(theta);

		// Zero (on imaginary axis in s-plane)
		double cos_term = cos(M_PI * (2.0 * i + 1.0) / (2.0 * n));
		if (fabs(cos_term) < 1e-10) cos_term = 1e-10; // Prevent division by zero
		double zi = 1.0 / cos_term;

		// Bilinear transform for poles
		double p_re = -sigma;
		double p_im = omega;
		double p_mag = sqrt(p_re * p_re + p_im * p_im);
		double den_p = 1.0 + 2.0 * p_re + p_mag * p_mag;
		if (fabs(den_p) < 1e-10) den_p = 1e-10; // Prevent division by zero
		double p_re_bilinear = (1.0 - p_mag * p_mag) / den_p;
		double p_im_bilinear = 2.0 * p_im / den_p;

		// Bilinear transform for zeros
		double z_re = 0.0;
		double z_im = zi;
		double z_mag = zi;
		double den_z = 1.0 + z_mag * z_mag; // z_re = 0 simplifies this
		if (fabs(den_z) < 1e-10) den_z = 1e-10; // Prevent division by zero
		double z_re_bilinear = (1.0 - z_mag * z_mag) / den_z;
		double z_im_bilinear = 2.0 * z_im / den_z;

		// Denominator coefficients
		double den = 1.0 + 2.0 * p_re_bilinear + p_re_bilinear * p_re_bilinear + p_im_bilinear * p_im_bilinear;
		if (fabs(den) < 1e-10) den = 1e-10; // Prevent division by zero
		d1[i] = 2.0 * (1.0 - p_re_bilinear * p_re_bilinear - p_im_bilinear * p_im_bilinear) / den;
		d2[i] = -(1.0 - 2.0 * p_re_bilinear + p_re_bilinear * p_re_bilinear + p_im_bilinear * p_im_bilinear) / den;

		// Numerator coefficients
		double num = 1.0 + 2.0 * z_re_bilinear + z_re_bilinear * z_re_bilinear + z_im_bilinear * z_im_bilinear;
		A[i] = num / den;

		// Normalize gain for unity at DC (z=1)
		double num_dc = 1.0 + 2.0 * z_re_bilinear + z_re_bilinear * z_re_bilinear + z_im_bilinear * z_im_bilinear;
		double den_dc = 1.0 + 2.0 * p_re_bilinear + p_re_bilinear * p_re_bilinear + p_im_bilinear * p_im_bilinear;
		if (fabs(num_dc) < 1e-10) num_dc = 1e-10; // Prevent division by zero
		A[i] *= 1.0 / num_dc; // Normalize to unity gain at DC
	}
}

LPF_RC::LPF_RC():
	a0(1.0),
	b1(0.0),
	z1(0.0) {
}

LPF_RC::LPF_RC(double Fc):
	z1(0.0) {
	setFc(Fc);
}

void LPF_RC::setFc(double Fc) {
	b1 = exp(-2.0 * M_PI * Fc);
	a0 = 1.0 - b1;
}

double LPF_RC::process(double in) {
	return z1 = in * a0 + z1 * b1;
}

HPF_RC::HPF_RC():
	LPF_RC() {
}

HPF_RC::HPF_RC(double Fc):
	LPF_RC(Fc) {
}

double HPF_RC::process(double in) {
	return in -LPF_RC::process(in);
}
